/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* Program to dump a compiled Malaga rule file with intermediate code. */

/* includes =================================================================*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "files.h"
#include "rule_type.h"
#include "rules.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL

/* variables ================================================================*/

LOCAL string_t program_name; /* this is set to argv[0] by main */

LOCAL rule_sys_t *rule_sys;

/* functions ================================================================*/

LOCAL void error_handler (string_t format, ...) NO_RETURN;
LOCAL void error_handler (string_t format, ...)
/* Print an error message and exit. */
{
  va_list arg;
  
  fflush (stdout);

  fprintf (stderr, "%s: ", program_name);

  va_start (arg, format);
  vfprintf (stderr, format, arg);
  va_end (arg);

  fprintf (stderr, "\n");
  exit (1);
}

/*---------------------------------------------------------------------------*/

LOCAL void print_instruction (int_t pc)
/* Print the instruction at <pc>. */
{
  int_t opcode = OPCODE (rule_sys->instrs[pc]);
  int_t info = INSTR_INFO (rule_sys->instrs[pc]);

  switch (opcode) 
  {
  case INS_ERROR:
    if (info == ASSERTION_ERROR || info == NO_RETURN_ERROR)
      printf ("error               %ld", info);
    else
      printf ("error               \"%s\"", rule_sys->strings + info);
    break;
  case INS_TERMINATE:
    printf ("terminate");
    break;
  case INS_NOP:
    printf ("nop");
    break;
  case INS_TERMINATE_IF_NULL:
    printf ("terminate_if_null");
    break;
  case INS_ADD_END_STATE:
    printf ("add_end_state");
    break;
  case INS_ADD_STATE:
  {
    string_t rule_string = rule_set_readable (rule_sys, info);

    printf ("add_state           %s", rule_string);
    free_mem (&rule_string);
    break;
  }
  case INS_ADD_ALLO:
    printf ("add_allo");
    break;
  case INS_ACCEPT:
    printf ("accept");
    break;
  case INS_PUSH_NULL:
    printf ("push_null           %ld", info);
    break;
  case INS_PUSH_VAR:
    printf ("push_var            %ld", info);
    break;
  case INS_PUSH_CONST:
  {
    string_t value_string = value_to_readable (rule_sys->values + info, TRUE);

    printf ("push_const          %s", value_string);
    free_mem (&value_string);
    break;
  }
  case INS_PUSH_SYMBOL:
    printf ("push_symbol         %s", get_symbol_name (info));
    break;
  case INS_PUSH_PATTERN_VAR:
    printf ("push_pattern_var    %ld", info);
    break;
  case INS_POP:
    printf ("pop                 %ld", info);
    break;
  case INS_BUILD_LIST:
    printf ("build_list          %ld", info);
    break;
  case INS_BUILD_RECORD:
    printf ("build_record        %ld", info);
    break;
  case INS_BUILD_PATH:
    printf ("build_path          %ld", info);
    break;
  case INS_DOT_OPERATION:
    printf ("dot_operation");
    break;
  case INS_PLUS_OPERATION:
    printf ("plus_operation");
    break;
  case INS_MINUS_OPERATION:
    printf ("minus_operation");
    break;
  case INS_ASTERISK_OPERATION:
    printf ("asterisk_operation");
    break;
  case INS_SLASH_OPERATION:
    printf ("slash_operation");
    break;
  case INS_UNARY_MINUS_OP:
    printf ("unary_minus_op");
    break;
  case INS_GET_ATTRIBUTE:
    printf ("get_attribute       %s", get_symbol_name (info));
    break;
  case INS_REMOVE_ATTRIBUTE:
    printf ("remove_attribute    %s", get_symbol_name (info));
    break;
  case INS_STD_FUNCTION:
  {
    string_t name;

    switch (info)
    {
    case FUNC_TO_ATOMS: name = "to_atoms"; break;
    case FUNC_IS_CAPITAL: name = "is_capital"; break;
    case FUNC_GET_LENGTH: name = "get_length"; break;
    case FUNC_TO_MULTI: name = "to_multi"; break;
    case FUNC_TO_SET: name = "to_set"; break;
    case FUNC_GET_SWITCH: name = "get_switch"; break;
    case FUNC_GET_VALUE_STRING: name = "get_value_string"; break;
    case FUNC_GET_VALUE_TYPE: name = "get_value_type"; break;
    case FUNC_TRANSMIT: name = "transmit"; break;
    case FUNC_FLOOR: name = "floor"; break;
    default:
      error ("internal (unknown standard function)");
    }

    printf ("std_function        %s", name);
    break;
  }
  case INS_MATCH:
  {
    string_t string;
    
    string = new_string_readable (rule_sys->strings + info, NULL);
    printf ("match               %s", string);
    free_mem (&string);
    break;
  }
  case INS_SET_VAR:
    printf ("set_var             %ld", info);
    break;
  case INS_PLUS_VAR:
    printf ("plus_var            %ld", info);
    break;
  case INS_MINUS_VAR:
    printf ("minus_var           %ld", info);
    break;
  case INS_ASTERISK_VAR:
    printf ("asterisk_var        %ld", info);
    break;
  case INS_SLASH_VAR:
    printf ("slash_var           %ld", info);
    break;
  case INS_SET_VAR_PATH:
    printf ("set_var_path        %ld", info);
    break;
  case INS_PLUS_VAR_PATH:
    printf ("plus_var_path       %ld", info);
    break;
  case INS_MINUS_VAR_PATH:
    printf ("minus_var_path      %ld", info);
    break;
  case INS_ASTERISK_VAR_PATH:
    printf ("asterisk_var_path   %ld", info);
    break;
  case INS_SLASH_VAR_PATH:
    printf ("slash_var_path      %ld", info);
    break;
  case INS_GET_1ST_ELEMENT:
    printf ("get_1st_element");
    break;
  case INS_ITERATE:
    printf ("iterate             %ld", info);
    break;
  case INS_JUMP:
    printf ("jump                %ld", info);
    break;
  case INS_JUMP_IF_EQUAL:
    printf ("jump_if_equal       %ld", info);
    break;
  case INS_JUMP_IF_NOT_EQUAL:
    printf ("jump_if_not_equal   %ld", info);
    break;
  case INS_JUMP_IF_CONGR:
    printf ("jump_if_congr       %ld", info);
    break;
  case INS_JUMP_IF_NOT_CONGR:
    printf ("jump_if_not_congr   %ld", info);
    break;
  case INS_JUMP_IF_IN:
    printf ("jump_if_in          %ld", info);
    break;
  case INS_JUMP_IF_NOT_IN:
    printf ("jump_if_not_in      %ld", info);
    break;
  case INS_JUMP_IF_LESS:
    printf ("jump_if_less        %ld", info);
    break;
  case INS_JUMP_IF_NOT_LESS:
    printf ("jump_if_not_less    %ld", info);
    break;
  case INS_JUMP_IF_GREATER:
    printf ("jump_if_greater     %ld", info);
    break;
  case INS_JUMP_IF_NOT_GREATER:
    printf ("jump_if_not_greater %ld", info);
    break;
  case INS_JUMP_IF_NULL:
    printf ("jump_if_null        %ld", info);
    break;
  case INS_JUMP_IF_NOT_NULL:
    printf ("jump_if_not_null    %ld", info);
    break;
  case INS_JUMP_IF_YES:
    printf ("jump_if_yes         %ld", info);
    break;
  case INS_JUMP_IF_NO:
    printf ("jump_if_no          %ld", info);
    break;
  case INS_JUMP_NOW:
    printf ("jump_now            %ld", info);
    break;
  case INS_JUMP_LATER:
    printf ("jump_later          %ld", info);
    break;
  case INS_JUMP_SUBRULE:
    printf ("jump_subrule        %s", 
	    rule_sys->strings + rule_sys->rules[info].name);
    break;
  case INS_RETURN:
    printf ("return              %ld", info);
    break;
  default:
    printf ("ILLEGAL INSTRUCTION %ld", opcode);
    break;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void print_rules (string_t source_name)
/* Print all rules of <source_name>. */
{
  int_t i;
  string_t instr_source_name;
  int_t source_line, instr_source_line;
  FILE *stream;
  int_t c;

  source_line = 0;
  if (source_name != NULL)
    stream = open_stream (source_name, "r");
  else
    stream = NULL;

  for (i = 0; i < rule_sys->instrs_size; i++) 
  {
    /* Find the source code. */
    source_of_instr (rule_sys, i, NULL, &instr_source_line, 
		     &instr_source_name, NULL);
	  
    /* Print the source if it is part of the current source file. */
    if (instr_source_name != NULL && source_name != NULL
	&& strcmp (name_in_path (instr_source_name), 
		   name_in_path (source_name)) == 0) 
    {
      while (source_line < instr_source_line || instr_source_line == -1) 
      {
	c = getc (stream);
	if (c == EOF)
	  break;
	else if (c == '\n')
	  source_line++;
	
	putchar (c);
      }
    }
    
    printf ("|%6ld:  ", i);
    print_instruction (i);
    printf ("\n");
  }

  if (source_name != NULL)
  {
    /* Print remainder of source file. */
    while ((c = getc (stream)) != EOF)
      putchar (c);

    close_stream (&stream, source_name);
  }
}

/* top level functions ======================================================*/

LOCAL void program_message (void)
/* Print some information about the program. */
{
  printf ("%s (%s) - Copyright (C) 1995-1999 Bjoern Beutel\n",
	  program_name, MALAGA_VERSION);
  printf ("It is part of Malaga, a system for Natural Language Analysis.\n");
  printf ("This program comes with ABSOLUTELY NO WARRANTY.\n");
  printf ("This is free software which you may redistribute "
	  "under certain conditions.\n");
  printf ("For details, refer to the GNU General Public License.\n");
}

/*---------------------------------------------------------------------------*/

GLOBAL int main (int argc, string_t argv[])
/* The main function of "maldump". */
{
  string_t rule_file, symbol_file, source_file;
  int_t i;

  error = error_handler;
  program_name = argv[0];

#ifdef HANGUL
  init_hangul ();
#endif

  /* Parse arguments. */
  
  if (argc == 2 && (strcmp_no_case (argv[1], "-version") == 0
		    || strcmp_no_case (argv[1], "-v") == 0))
  {
    program_message ();
    exit (0);
  }

  rule_file = symbol_file = source_file = NULL;
  for (i = 1; i < argc; i++) 
  {
    string_t argument;

    if (*argv[i] == '-')
      argument = new_string (argv[i], NULL);
    else
      argument = absolute_path (argv[i], NULL);
    
    if (has_extension (argument, "all_c"))
      set_file_name (&rule_file, argument, NULL);
    else if (has_extension (argument, "mor_c"))
      set_file_name (&rule_file, argument, NULL);
    else if (has_extension (argument, "syn_c"))
      set_file_name (&rule_file, argument, NULL);
    else if (has_extension (argument, "sym_c"))
      set_file_name (&symbol_file, argument, NULL);
    else if (has_extension (argument, "esym_c"))
      set_file_name (&symbol_file, argument, NULL);
    else
      set_file_name (&source_file, argument, NULL);

    free_mem (&argument);
  }
  
  if (rule_file == NULL)
    error ("missing rule file name");
  
  if (symbol_file == NULL)
    error ("missing symbol file name");

  init_values ();

  init_symbols (symbol_file);
  rule_sys = read_rule_sys (rule_file);

  print_rules (source_file);
    
  free_rule_sys (&rule_sys);
  terminate_symbols ();
  free_mem (&symbol_file);
  free_mem (&rule_file);
  free_mem (&source_file);

  terminate_values ();

#ifdef HANGUL
  terminate_hangul ();
#endif
  
  return 0;
}

/* end of file ==============================================================*/
