/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* Invoke the display process from Malaga. */

/* includes =================================================================*/

#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED 1
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <setjmp.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>
#include <signal.h>
#include <pwd.h>
#include <fcntl.h>
#include "basic.h"
#include "files.h"
#include "input.h"
#include "commands.h"

#undef GLOBAL
#define GLOBAL

#include "display.h"

/* variables ================================================================*/

LOCAL pid_t process_id; /* ID of display process */
LOCAL string_t display_command_line; /* command line for display output */

/* functions ================================================================*/

GLOBAL void stop_display_process (void)
/* Stop the Malaga display process. */
{
  close_stream (&display_stream, NULL);
  if (process_id != 0)
  {
    kill (process_id, SIGTERM);
    process_id = 0;
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void start_display_process (void)
/* Start the Malaga display process by executing <display_command_line>
 * if it is not already running. */
{
  int pipe_fd[2];

  if (process_id != 0)
  {
    if (waitpid (process_id, 0, WNOHANG) == 0)
      return;

    process_id = 0;
    close_stream (&display_stream, NULL);
  }

  if (pipe (pipe_fd) == -1)
    error ("can't create pipe to display process: %s", strerror (errno));
    
  if (display_command_line == NULL) 
    error ("missing display command line");

  switch (process_id = fork ())
  {
  case -1:
    error ("can't create display process: %s", strerror (errno));
    break;
    
  case 0:
  {
    string_t arguments, argument;
    string_t *args;
    int_t num_args, i;

    dup2 (pipe_fd[0], STDIN_FILENO);
    close (pipe_fd[0]);
    close (pipe_fd[1]);

    /* Count arguments. */
    num_args = 0;
    arguments = display_command_line;
    while (*arguments != EOS)
    {
      argument = parse_word (&arguments);
      free_mem (&argument);
      num_args++;
    }

    /* Create argument vector. */
    args = new_vector (sizeof (string_t), num_args + 1);
    arguments = display_command_line;
    for (i = 0; i < num_args; i++)
      args[i] = parse_word (&arguments);
    args[i] = NULL;

    execvp (args[0], args);
    fprintf (stderr, "can't start display process \"%s\": %s\n", 
	     args[0], strerror (errno));
    exit (1);
  }
  default:
    close (pipe_fd[0]);
    display_stream = fdopen (pipe_fd[1], "w");
    if (display_stream == NULL)
      error ("can't open data stream: %s", strerror (errno));
#ifdef HANGUL
    fprintf (display_stream, "hangul\n");
#else
    fprintf (display_stream, "latin1\n");
#endif
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void do_display_option (string_t arguments)
/* Set the command line to start the display process. */
{
  if (*arguments == EOS)
    printf ("display: \"%s\"\n", 
	    (display_command_line == NULL) ? "" : display_command_line);
  else
  {
    stop_display_process ();
    free_mem (&display_command_line);
    display_command_line = parse_word (&arguments);
  }

  parse_end (arguments);
}

GLOBAL command_t display_option =
{
  "display", do_display_option,
  "Usage: set display \"<display_command_line>\"\n"
  "Set the command line that is used to start the display process.\n"
};

/* end of file ==============================================================*/
