#
# Copyright (c) 2010 Canonical
#
# Written by Gustavo Niemeyer <gustavo@niemeyer.net>
#
# This file is part of the Xpresser GUI automation library.
#
# Xpresser is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3,
# as published by the Free Software Foundation.
#
# Xpresser is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import threading
import time

import gtk

from xpresser import Xpresser, ImageNotFound
from xpresser.image import Image
from xpresser.imagematch import ImageMatch

from xpresser.tests.images import get_image_path
from xpresser.tests.test_xutils import XUtilsTestBase


SLEEP_DELAY = 1.5


class XpresserTestBase(XUtilsTestBase):

    def setUp(self):
        super(XpresserTestBase, self).setUp()
        self.xp = Xpresser()
        self.xp.load_images(get_image_path())


class XpresserTest(XpresserTestBase):

    def test_load_images_and_get_image(self):
        image = self.xp.get_image("red-circle")
        self.assertEquals(type(image), Image)
        self.assertEquals(image.name, "red-circle")

    def test_type(self):
        entry = gtk.Entry()
        window = self.create_window(entry)
        try:
            window.present()
            entry.grab_focus()
            self.flush_gtk()
            self.xp.type("Hello there!")
            self.flush_gtk()
            self.assertEquals(entry.get_text(), "Hello there!")
        finally:
            window.destroy()


class XpresserButtonTest(XpresserTestBase):

    def setUp(self):
        super(XpresserButtonTest, self).setUp()
        self.window = self.create_button_window()
        self.button = self.window.get_child()
        self.button_clicked = False
        self.button_rclicked = False
        self.button_hovered = False

        def clicked(widget, event):
            if event.button == 1:
                self.button_clicked = True
            elif event.button == 3:
                self.button_rclicked = True
            self.window.destroy()

        def entered(widget):
            self.button_hovered = True

        self.button.connect("button_press_event", clicked)
        self.button.connect("enter", entered)

        self.flush_gtk()

    def tearDown(self):
        self.window.destroy()

    def get_button_center(self):
        button_x, button_y = self.button.window.get_position()
        button_width, button_height = self.button.window.get_size()
        return (button_x + button_width//2, button_y + button_height//2)

    def test_find_image_name(self):
        match = self.xp.find("red-square")
        self.assertEquals(type(match), ImageMatch)
        self.assertEquals(match.focus_point, self.get_button_center())

    def test_find_image(self):
        image = self.xp.get_image("red-square")
        match = self.xp.find(image)
        self.assertEquals(type(match), ImageMatch)
        self.assertEquals(match.focus_point, self.get_button_center())

    def test_find_with_delay(self):
        self.window.hide()
        self.flush_gtk()
        def show_window():
            time.sleep(SLEEP_DELAY)
            self.window.show()
            self.flush_gtk()
        thread = threading.Thread(target=show_window)
        thread.start()
        match = self.xp.find("red-square")
        self.assertEquals(type(match), ImageMatch)
        self.assertEquals(match.focus_point, self.get_button_center())

    def test_find_failed(self):
        started = time.time()
        self.assertRaises(ImageNotFound,
                          self.xp.find, "blue-square", timeout=SLEEP_DELAY)
        self.assertTrue(time.time() - started > SLEEP_DELAY)

    def test_wait(self):
        self.window.hide()
        self.flush_gtk()
        def show_window():
            time.sleep(SLEEP_DELAY)
            self.window.show()
            self.flush_gtk()
        thread = threading.Thread(target=show_window)
        started = time.time()
        thread.start()
        self.xp.wait("red-square")
        self.assertTrue(time.time() - started > SLEEP_DELAY)

    def test_wait_failed(self):
        started = time.time()
        self.assertRaises(ImageNotFound,
                          self.xp.wait, "blue-square", timeout=SLEEP_DELAY)
        self.assertTrue(time.time() - started > SLEEP_DELAY)

    def test_click_position(self):
        self.xp.click(*self.get_button_center())
        self.flush_gtk()
        self.assertTrue(self.button_clicked)

    def test_hover_position(self):
        self.xp.hover(*self.get_button_center())
        self.flush_gtk()
        self.assertTrue(self.button_hovered)
        self.assertFalse(self.button_clicked)

    def test_click_image_name(self):
        self.xp.click("red-square")
        self.flush_gtk()
        self.assertTrue(self.button_clicked)

    def test_right_click_image_name(self):
        self.xp.right_click("red-square")
        self.flush_gtk()
        self.assertTrue(self.button_rclicked)

    def test_hover_image_name(self):
        self.xp.hover("red-square")
        self.flush_gtk()
        self.assertTrue(self.button_hovered)
        self.assertFalse(self.button_clicked)

    def test_click_image_match(self):
        match = self.xp.find("red-square")
        self.xp.click(match)
        self.flush_gtk()
        self.assertTrue(self.button_clicked)

    def test_right_click_image_match(self):
        match = self.xp.find("red-square")
        self.xp.right_click(match)
        self.flush_gtk()
        self.assertTrue(self.button_rclicked)

    def test_hover_image_match(self):
        match = self.xp.find("red-square")
        self.xp.hover(match)
        self.flush_gtk()
        self.assertTrue(self.button_hovered)
        self.assertFalse(self.button_clicked)

    def test_click_waits(self):
        self.window.hide()
        self.flush_gtk()
        def show_window():
            time.sleep(SLEEP_DELAY)
            self.window.show()
            self.flush_gtk()
        thread = threading.Thread(target=show_window)
        started = time.time()
        thread.start()
        self.xp.click("red-square")
        self.assertTrue(time.time() - started > SLEEP_DELAY)
        self.flush_gtk()
        self.assertTrue(self.button_clicked)

    def test_right_click_waits(self):
        self.window.hide()
        self.flush_gtk()
        def show_window():
            time.sleep(SLEEP_DELAY)
            self.window.show()
            self.flush_gtk()
        thread = threading.Thread(target=show_window)
        started = time.time()
        thread.start()
        self.xp.right_click("red-square")
        self.assertTrue(time.time() - started > SLEEP_DELAY)
        self.flush_gtk()
        self.assertTrue(self.button_rclicked)

    def test_hover_waits(self):
        self.window.hide()
        self.flush_gtk()
        def show_window():
            time.sleep(SLEEP_DELAY)
            self.window.show()
            self.flush_gtk()
        thread = threading.Thread(target=show_window)
        started = time.time()
        thread.start()
        self.xp.hover("red-square")
        self.assertTrue(time.time() - started > SLEEP_DELAY)
        self.flush_gtk()
        self.assertTrue(self.button_hovered)
