from .main import Application
from ..utils import show_desktop, get_ldtp_version
from time import time, sleep
import tempfile
import pynotify
import ldtp, ldtputils
import os
import gtk, glib
from distutils import version

try:
    import indicate
except ImportError:
    indicate = None

class IndicatorApplet(Application):
    """
    Indicator Applet manages the new indicator messages applet
    """
    IA_TOPLEVEL = "embindicator-applet"
    def __init__(self):
        Application.__init__(self, 'indicator-applet')
        self.indicators = []
        self.server = None

    def open(self):
        pass
    
    def add_server(self, desktop_file):
        """
        Add a new server to the indicator applet.

        @type desktop_file: string
        @param desktop_file: The path to the file describing the server.
            Example of format:

            [Desktop Entry]
            Encoding=UTF-8
            Name=Phony Internet Messenger
            GenericName=Internet Messenger
            Comment=Send instant messages over phony protocols
            Exec=phony
            StartupNotify=true
            Terminal=false
            Type=Application
            Categories=Network;InstantMessaging;

        """

        if not self.server:
            try:
                self.server = indicate.indicate_server_ref_default()
            except AttributeError:
                raise Exception, \
                    "no libindicate Python bindings, install python-indicate."
        self.server.set_type("message.im")
        self.server.set_desktop_file(desktop_file)
        self.server.show()
        while gtk.events_pending():
            gtk.main_iteration()

    def show_indicator(self, sender):
        """
        It shows a basic indicator without needing to provide a desktop file

        @type sender: string
        @param sender: The name of the indicator to be shown in the applet.
        """
        def _timeout_cb():
            gtk.main_quit()
            return False

        indicator = indicate.IndicatorMessage()
        indicator.set_property("subtype", "im")
        indicator.set_property("sender", sender)
        indicator.set_property_time("time", time())
        pixbuf = gtk.gdk.pixbuf_new_from_file(
            "/usr/share/icons/hicolor/22x22/apps/gnome-freecell.png")
        indicator.set_property_icon("icon", pixbuf)
        indicator.show()
        self.indicators.append(indicator)
        glib.timeout_add_seconds(1, _timeout_cb)
        gtk.main()

    def capture_applet_icon(self):
        """
        It captures a screenshot of the indicator applet icon

        @return: The path of the file containing the screenshot
        """
        x, y, w, h = ldtp.getobjectsize(self.TOP_PANEL, self.IA_TOPLEVEL)
        
        ldtp_one_six = version.StrictVersion('1.6.0')
        ldtp_current = version.StrictVersion(get_ldtp_version())

        if ldtp_current < ldtp_one_six:
            screeny = ldtputils.imagecapture(
                outFile=tempfile.mktemp('.png', 'ia_'),
                x=x, y=y, resolution1=w, resolution2=h)        
        else:
            screeny = ldtputils.imagecapture(
                outFile=tempfile.mktemp('.png', 'ia_'),
                x=x, y=y, width=w, height=h)        

        return screeny

    def is_server_shown(self, sender, already_shown=True):
        """
        It says if a server is being shown or not

        @type sender: string
        @param sender: The name of the Indicator server to check

        @type already_shown: boolean
        @param already_shown: We need a way to distinguish between the normal menu and the indicator-applet menu
            Workaround in the mean time:
            Set already_shown as True, if there is already a menu with the same name
            (in that case we will look for mnuServer1
            or to False, if there is no already a menu with the same name

        @return: True, if the server is being shown; False, otherwise.
        """
        if already_shown:
            return ldtp.objectexist(self.TOP_PANEL, 
                    'mnu' + sender.replace(' ','') + '1')
        else:
            return ldtp.objectexist(self.TOP_PANEL, 
                    'mnu' + sender.replace(' ',''))
            
    def is_indicator_shown(self, sender):
        """
        It says if an indicator is being shown or not

        @type sender: string
        @param sender: The name of the indicator to check
        
        @return: True, if the indicator is being shown; False, otherwise
        """
        return ldtp.objectexist(self.TOP_PANEL, 
                    'mnu' + sender.replace(' ',''))

    def select_indicator(self, sender):
        """
        It selects the menu item of an indicator 

        @type sender: string
        @param sender: The name of the indicator to select 
        """
        ldtp.selectmenuitem(self.TOP_PANEL, 'mnu' + sender.replace(' ',''))

    def select_server(self, sender, already_shown=True):
        """
        It selects the menu item of a server indicator 

        @type sender: string
        @param sender: The name of the Indicator server to select 

        @type already_shown: boolean
        @param already_shown: We need a way to distinguish between the normal menu and the indicator-applet menu
            Workaround in the mean time:
            Set already_shown as True, if there is already a menu with the same name
            (in that case we will look for mnuServer1
            or to False, if there is no already a menu with the same name
        """

        if already_shown:
            ldtp.selectmenuitem(self.TOP_PANEL, 'mnu' + sender.replace(' ',''), + '1')
        else:
            ldtp.selectmenuitem(self.TOP_PANEL, 'mnu' + sender.replace(' ',''))

    def wait_for_indicator_display(self, sender, timeout=5):
        handlers = []
        displayed = [False]

        def _display_cb(indicator):
            indicator.hide() # This is just normal behavior, so why not?
            displayed[0] = True
            gtk.main_quit()
            
        def _timeout_cb():
            gtk.main_quit()
            return False

        for indicator in self.indicators:
            if sender == indicator.get_property("sender"):
                handler = indicator.connect("user-display", _display_cb)
                handlers.append((handler, indicator))

        glib.timeout_add_seconds(timeout, _timeout_cb)

        gtk.main()

        for handler, indicator in handlers:
            indicator.disconnect(handler)

        return displayed[0]

    def wait_for_server_display(self, timeout=5):
        displayed = [False]
        handler = 0

        def _display_cb(indicator):
            indicator.hide() # This is just normal behavior, so why not?
            displayed[0] = True
            gtk.main_quit()
            
        def _timeout_cb():
            gtk.main_quit()
            return False

        handler = self.server.connect("server-display", _display_cb)

        glib.timeout_add_seconds(timeout, _timeout_cb)

        gtk.main()

        self.server.disconnect(handler)

        return displayed[0]

    def close(self):
        for indicator in self.indicators:
            indicator.hide()
        # BUG: 351537
        # self.server.hide()
        sleep(1)

class NotifyOSD(Application):
    """
    NotifyOSD class manages the notifications produced by notify-osd
    """

    def __init__(self):
        self.focus_desktop = False
        self.screenshots = []

        if not pynotify.init('notify-osd-test'):
            raise ldtp.LdtpExecutionError, \
                "Failed to initialize notification connection."

        info = pynotify.get_server_info()
        if info.get('name', None) != 'notify-osd':
            raise ldtp.LdtpExecutionError, \
                "The notify service is '%s', expected 'notify-osd'" % \
                info.get('name', None)

    def open(self, focus_desktop=True):
        self.focus_desktop = focus_desktop

        if self.focus_desktop:
            show_desktop(True)

    def close(self):
        if self.focus_desktop:
            show_desktop(False)
        for screenshot in self.screenshots:
            if os.path.exists(screenshot):
                os.remove(screenshot)

    def notify(self, summary, body="", icon=None):
        """
        Giving a summary, body and icon, it creates a notification bubble

        @type summary: string
        @param summary: The header of the notification

        @type body: string
        @param body: The text to show as body of the notification

        @type icon: string
        @param icon: The name of the icon to show
        """
        n = pynotify.Notification (summary, body, icon)
        n.show ()

    def notify_synchronous(self, summary, body="", icon=None, value=-1):
        """
        Giving a summary, body, icon and value it creates a confirmation bubble

        @type summary: string
        @param summary: The header of the notification

        @type body: string
        @param body: The text to show as body of the notification

        @type icon: string
        @param icon: The name of the icon to show

        @type value: int
        @param value: The value of the quantity of the confirmation bubble (i.e. volume)
        """
        n = pynotify.Notification (summary, body, icon)
        n.set_hint("synchronous", "volume")
        n.set_hint("value", value)
        n.show ()

    def grab_image_and_wait(self, summary, timeOut=30):
        """
        It waits for a notification to appear and grabs a screenshot

        @type summary: string
        @param summary: The summary of the notification to look for

        @type timeOut: int
        @param timeOut: The number of seconds to wait for the notification to appear

        @return: List with the time elapsed and the path to the screenshot
        """
        ldtp.waittillguiexist(summary, guiTimeOut=timeOut)
        start_time = time()
        sleep(1)
        x, y, w, h = ldtp.getwindowsize(summary)

        ldtp_one_six = version.StrictVersion('1.6.0')
        ldtp_current = version.StrictVersion(get_ldtp_version())

        if ldtp_current < ldtp_one_six:
            screenshot = \
                ldtputils.imagecapture(out_file=tempfile.mktemp('.png', 'nosd_'),
                                   x=x+3, y=y+3, 
                                   resolution1=w-6, 
                                   resolution2=h-6)
        else:
            screenshot = \
                ldtputils.imagecapture(out_file=tempfile.mktemp('.png', 'nosd_'),
                                   x=x+3, y=y+3, 
                                   width=w-6, 
                                   height=h-6)
        
        ldtp.waittillguinotexist(summary)
        end_time = time() - start_time
        self.screenshots.append(screenshot)
        return (end_time, screenshot)

    def get_extents(self, summary, wait=False):
        """
        It gets the limits of the bubble

        @type summary: string
        @param summary: The summary of the bubble to get the size

        @return: (x, y, width, height)
        """
        if wait:
            exists = ldtp.waittillguiexist(summary)            
        else:
            exists = ldtp.guiexist(summary)            
            
        if exists:
            return ldtp.getwindowsize(summary)
        else:
            return -1, -1, -1, -1
            
if __name__ == "__main__":
    from time import sleep
    test = IndicatorApplet()
    test.open()
    test.add_server('/usr/share/applications/transmission.desktop')
    test.show_indicator('Elmer Fud')
    print 'sleeping'
    sleep(20)
    #print test.wait_for_indicator_display('Elmer Fud', 20)
