# logger.py
#
# Author: Facundo Batista <facundo@taniquetil.com.ar>
#
# Copyright 2010 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Logging set up."""


import logging
import os
import sys
import traceback

from logging.handlers import RotatingFileHandler

import xdg.BaseDirectory


class CustomRotatingFH(RotatingFileHandler):
    """Rotating handler that starts a new file for every run."""

    def __init__(self, *args, **kwargs):
        RotatingFileHandler.__init__(self, *args, **kwargs)
        self.doRollover()


def exception_handler(exc_type, exc_value, tb):
    """Handle an unhandled exception."""
    # stderr
    exception = traceback.format_exception(exc_type, exc_value, tb)
    exception = "".join(exception)
    print >> sys.stderr, exception

    # log
    logger = logging.getLogger('magicicada')
    logger.error("Unhandled exception!\n%s", exception)


def set_up():
    """Set up the logging."""

    # choose the folder to store the logs
    cache = xdg.BaseDirectory.xdg_cache_home
    logfolder = os.path.join(cache, 'magicicada')
    if not os.path.exists(logfolder):
        os.makedirs(logfolder)
    logfile = os.path.join(logfolder, 'magicicada.log')

    logger = logging.getLogger('magicicada')
    handler = CustomRotatingFH(logfile, maxBytes=1e6, backupCount=10)
    logger.addHandler(handler)
    formatter = logging.Formatter("%(asctime)s %(name)-23s"
                                  "%(levelname)-8s %(message)s",
                                  '%Y-%m-%d %H:%M:%S')
    handler.setFormatter(formatter)
    logger.setLevel(logging.DEBUG)

    # hook the exception handler
    sys.excepthook = exception_handler
