/* mdump.c -- Dump text image
   Copyright (C) 2003, 2004
     National Institute of Advanced Industrial Science and Technology (AIST)
     Registration Number H15PRO112

   This file is part of the m17n library.

   The m17n library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The m17n library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the m17n library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

/***en
    @page mdump dump text image

    @section mdump-synopsis SYNOPSIS

    mdump [ OPTION ... ] [ FILE ]

    @section mdump-description DESCRIPTION

    Dump a text as a Netpbm image.

    The Netpbm image is written to a file created in the current
    directory with the name "BASE.pbm" where BASE is the basename of
    FILE.  If FILE is omitted, text is read from standard input, and
    the image is dumped into the file "output.pbm".

    The following OPTIONs are available.

    <ul>

    <li> -s SIZE

    SIZE is the font size in point.  The default font size is 12 point.

    <li> -d DPI

    DPI is the resolution in dots per inch.  The default resolution is
    300 dpi.

    <li> -p PAPER

    PAPER is the paper size: a4, a4r, a5, a5r, b5, b5r, letter, or
    WxH.  In the last case, W and H are the width and height in
    millimeter.  If this option is specified, PAPER limits the image
    size.  If FILE is too large for a single page, multiple files with
    the names "BASE.01.pbm", "BASE.02.pbm", etc. are created.

    <li> -m MARGIN

    MARGIN is the horizontal and vertical margin in millimeter.  The
    default margin is 20 mm.  It is ignored when PAPER is not
    specified.

    <li> -c POS

    POS is the character position of cursor to draw.  By default,
    cursor is not drawn.

    <li> -x

    FILE is assumed to be an XML file generated by the serialize
    facility of the m17n library, and FILE is deserialized before the
    image is created.

    <li> -w

    Each line is broken at word boundary.

    <li> -f FILTER

    FILTER is a string containing a shell command line.  If this
    option is specified, the Netpbm image is not written info a
    file but is given to FILTER as standard input.  If FILTER
    contains "%s", that part is replaced by a basename of FILE.
    So, the default behaviour is the same as specifying "cat >
    %s.pbm" as FILTER.

    <li> -q

    Quiet mode.  Don't print any messages.

    <li> --version

    Print the version number.

    <li> -h, --help

    Print this message.

    </ul>
*/

#ifndef FOR_DOXYGEN

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>

#include <X11/Xlib.h>

#include <m17n-gui.h>
#include <m17n-misc.h>

#define VERSION "1.0"

/* Enumuration of the supported paper types.  */
enum paper_type
  {
    PAPER_A4,
    PAPER_A4R,
    PAPER_A5,
    PAPER_A5R,
    PAPER_B5,
    PAPER_B5R,
    PAPER_LETTER,
    PAPER_USER,
    PAPER_NOLIMIT
  };

/* Array of paper sizes for the supported paper types.  */
struct
{
  int width, height;		/* in millimeter */
} paper_size[PAPER_NOLIMIT] = {
  { 210, 297 },			/* a4 */
  { 297, 210 },			/* a4r */
  { 148, 210 },			/* a5 */
  { 210, 148 },			/* a5r */
  { 250, 176 },			/* b5 */
  { 176, 250 },			/* b5r */
  { 216, 279 },			/* letter */
};


/* Print the usage of this program (the name is PROG), and exit with
   EXIT_CODE.  */

void
help_exit (char *prog, int exit_code)
{
  char *p = prog;

  while (*p)
    if (*p++ == '/')
      prog = p;

  printf ("Usage: %s [ OPTION ...] [ FILE ]\n", prog);
  printf ("Dump a text as a Netpbm image into a PBM file.\n");
  printf ("  The PBM file is created in the current directory\n");
  printf ("    with the name \"BASE.pbm\" where BASE is the basename of FILE.\n");
  printf ("  If FILE is omitted, text is read from standard input, and\n");
  printf ("    dumped into the file \"output.pbm\".\n");
  printf ("The following OPTIONs are available.\n");
  printf ("  %-13s %s", "-s SIZE",
	  "Font size in point (default 12).\n");
  printf ("  %-13s %s", "-d DPI",
	  "Resolution in dots per inch (defualt 300).\n");
  printf ("  %-13s %s", "-p PAPER",
	  "Paper size; a4, a4r, a5, a5r, b5, b5r, letter, or WxH.\n");
  printf ("  %-13s %s", "-m MARGIN",
	  "Marginal space in millimeter (default 20).\n");
  printf ("  %-13s %s", "-c POS",
	  "Character position of cursor to draw (default no cursor)\n");
  printf ("  %-13s %s", "-x",
	  "FILE is assumed to be an XML file.\n");
  printf ("  %-13s %s", "-f FILTER",
	  "String containing a shell command line to be used as a filter.\n");
  printf ("  %-13s %s", "-w", "Each line is broken at word boundary.\n");
  printf ("  %-13s %s", "-q", "Quiet mode.  Don't print any messages.\n");
  printf ("  %-13s %s", "--version", "Print the version number.\n");
  printf ("  %-13s %s", "-h, --help", "Print this message.\n");
  exit (exit_code);
}


/* Format MSG by FMT and print the result to the stderr, and exit.  */

#define FATAL_ERROR(fmt, arg)	\
  do {				\
    fprintf (stderr, fmt, arg);	\
    exit (1);			\
  } while (0)


/* Move POS to the next line head in M-text MT whose length is LEN.
   If POS is already on the last line, set POS to LEN.  */

#define NEXTLINE(pos, len)			\
  do {						\
    pos = mtext_character (mt, pos, len, '\n');	\
    if (pos < 0)				\
      pos = len;				\
    else					\
      pos++;					\
  } while (0)


/* Find the range of M-text MT that fits in one page of height HEIGHT
   when drawn from the character position POS.  Set RECT->y to the
   Y-offset of the first baseline.  */

int
find_page_end (MFrame *frame, int height, MText *mt, int pos,
	       MDrawControl *control, MDrawMetric *rect)
{
  int len = mtext_len (mt);
  int to = pos;
  int y = 0, yoff;

  while (to < len)
    {
      int next = to;

      NEXTLINE (next, len);
      mdraw_text_extents (frame, mt, to, next, control, NULL, NULL, rect);
      if (to == pos)
	yoff = rect->y;
      if (y + rect->height > height)
	{
	  MDrawGlyphInfo info;

	  while (to < next)
	    {
	      mdraw_glyph_info (frame, mt, to, to, control, &info);
	      if (y + info.this.height > height)
		break;
	      y += info.this.height;
	      to = info.line_to;
	    }
	  break;
	}
      y += rect->height;
      to = next;
    }

  rect->y = yoff;
  return to;
}


/* Table to convert a byte of LSBFirst to MSBFirst. */
char reverse_bit_order[256];

/* Initialize the above table.  */

void
init_reverse_bit_order ()
{
  int i;

  for (i = 0; i < 256; i++)
    reverse_bit_order[i]
      = (((i & 1) << 7) | ((i & 2) << 5) | ((i & 4) << 3) | ((i & 8) << 1)
	 | ((i & 16) >> 1) | ((i & 32) >> 3)
	 | ((i & 64) >> 5) | ((i & 128) >> 7));
}


/* Dump the image in IMAGE into a file whose name is generated from
   FILENAME and PAGE_INDEX (if it is not zero).  */

void
dump_image (XImage *image, char *filename, char *filter,
	    int white_is_zero, int page_index, int quiet_mode)
{
  FILE *fp;
  int pbm_bytes_per_line;
  char *data = image->data;
  int x, y;

  if (page_index)
    {
      char *name = alloca (strlen (filename) + 8);

      sprintf (name, "%s.%02d", filename, page_index);
      filename = name;
    }

  if (filter)
    {
      char *command = alloca (strlen (filename) + strlen (filter) + 1);

      sprintf (command, filter, filename);
      fp = popen (command, "w");
      if (! fp)
	FATAL_ERROR ("Can't run the command \"%s\"\n", command);
      if (! quiet_mode)
	printf ("Running \"%s\" ... ", command);
    }
  else
    {
      char *fullname = alloca (strlen (filename) + 5);

      sprintf (fullname, "%s.pbm", filename);
      fp = fopen (fullname, "w");
      if (! fp)
	FATAL_ERROR ("Can't write to \"%s\"\n", fullname);
      if (! quiet_mode)
	printf ("Writing %s ... ", fullname);
    }

  if (image->bitmap_bit_order != MSBFirst
      || (image->byte_order != MSBFirst
	  && image->bitmap_unit != 8))
    {
      /* We must adjust image->data for PBM.  */
      int bytes_per_unit = image->bitmap_unit / 8;

      for (y = 0; y < image->height; y++, data += image->bytes_per_line)
	{
	  char b;

	  if (image->byte_order != MSBFirst)
	    {
	      if (reverse_bit_order[0] == 0)
		init_reverse_bit_order ();
	      if (bytes_per_unit == 2)
		for (x = 0; x < image->bytes_per_line; x += 2)
		  b = data[x], data[x] = data[x + 1], data[x + 1] = b;
	      else if (bytes_per_unit == 6)
		for (x = 0; x < image->bytes_per_line; x += 3)
		  {
		    b = data[x], data[x] = data[x + 3], data[x + 3] = b;
		    x++;
		    b = data[x], data[x] = data[x + 1], data[x + 1] = b;
		  }
	    }

	  if (image->bitmap_bit_order != MSBFirst)
	    for (x = 0; x < image->bytes_per_line; x++)
	      data[x] = reverse_bit_order[(unsigned char) data[x]];
	  if (! white_is_zero)
	    for (x = 0; x < image->bytes_per_line; x++)
	      data[x] = ~data[x];
	}
      /* Reset DATA.  */
      data = image->data;
    }

  /* Generate PBM (Portable Bitmap File Format) of P4 format.  */
  fprintf (fp, "P4\n%d %d\n", image->width, image->height);
  pbm_bytes_per_line = (image->width + 7) / 8;
  for (y = 0; y < image->height; y++, data += image->bytes_per_line)
    fwrite (data, 1, pbm_bytes_per_line, fp);

  fclose (fp);
  if (! quiet_mode)
    printf (" done (%dx%d)\n", image->width, image->height);
}

extern int line_break (MText *mt, int pos, int from, int to, int line, int y);

int
main (int argc, char **argv)
{
  Display *display;
  int screen;
  GC gc;
  Pixmap pixmap;
  XImage *image;
  int fontsize = 120;
  int paper = PAPER_NOLIMIT;
  int dpi = 300;
  int margin = 20;
  int xml = 0;
  FILE *fp = stdin;
  int cursor_pos = -1;
  int quiet_mode = 0;
  int break_by_word = 0;
  char *filter = NULL;
  int i;
  int paper_width, paper_height;
  int page_index;

  MFrame *frame;
  MText *mt;
  MDrawControl control;
  MDrawMetric rect;
  char *filename = "output";
  int len, from;

  /* Parse the command line arguments.  */
  for (i = 1; i < argc; i++)
    {
      if (! strcmp (argv[i], "--help")
	       || ! strcmp (argv[i], "-h")
	       || ! strcmp (argv[i], "-?"))
	help_exit (argv[0], 0);
      else if (! strcmp (argv[i], "--version"))
	{
	  printf ("mdump (m17n library) %s\n", VERSION);
	  printf ("Copyright (C) 2003, 2004 AIST, JAPAN\n");
	  exit (0);
	}
      else if (! strcmp (argv[i], "-s") && i + 1< argc)
	{
	  fontsize = atoi (argv[++i]);
	  if (! fontsize)
	    FATAL_ERROR ("Invalid font size: %s\n", argv[i]);
	}
      else if (! strcmp (argv[i], "-p") && i + 1< argc)
	{
	  int w, h;

	  i++;
	  if (! strcmp (argv[i], "a4"))
	    paper = PAPER_A4;
	  else if (! strcmp (argv[i], "a4r"))
	    paper = PAPER_A4R;
	  else if (! strcmp (argv[i], "a5"))
	    paper = PAPER_A5;
	  else if (! strcmp (argv[i], "a5r"))
	    paper = PAPER_A5R;
	  else if (! strcmp (argv[i], "b5"))
	    paper = PAPER_B5;
	  else if (! strcmp (argv[i], "b5r"))
	    paper = PAPER_B5R;
	  else if (! strcmp (argv[i], "letter"))
	    paper = PAPER_LETTER;
	  else if (sscanf (argv[i], "%dx%d", &w, &h) == 2
		   && w > 0 && h > 0)
	    {
	      paper = PAPER_USER;
	      paper_size[paper].width = w;
	      paper_size[paper].height = h;
	    }
	  else
	    FATAL_ERROR ("Invalid paper type: %s\n", argv[i]);
	}
      else if (! strcmp (argv[i], "-d") && i + 1< argc)
	{
	  dpi = atoi (argv[++i]);
	  if (! dpi)
	    FATAL_ERROR ("Invalid resolution: %s\n", argv[i]);
	}
      else if (! strcmp (argv[i], "-m") && i + 1< argc)
	{
	  margin = atoi (argv[++i]);
	  if (margin < 0)
	    FATAL_ERROR ("Invalid margin: %s\n", argv[i]);
	}
      else if (! strcmp (argv[i], "-c") && i + 1< argc)
	{
	  cursor_pos = atoi (argv[++i]);
	  if (cursor_pos < 0)
	    FATAL_ERROR ("Invalid cursor position: %s\n", argv[i]);
	}
      else if (! strcmp (argv[i], "-f") && i + 1< argc)
	{
	  filter = argv[++i];
	}
      else if (! strcmp (argv[i], "-x"))
	{
	  xml = 1;
	}
      else if (! strcmp (argv[i], "-w"))
	{
	  break_by_word = 1;
	}
      else if (! strcmp (argv[i], "-q"))
	{
	  quiet_mode = 1;
	}
      else if (argv[i][0] != '-')
	{
	  fp = fopen (argv[i], "r");
	  if (! fp)
	    FATAL_ERROR ("Fail to open the file %s!\n", argv[i]);
	  filename = basename (argv[i]);
	}
      else
	{
	  fprintf (stderr, "Unknown or invalid option: %s\n", argv[i]);
	  help_exit (argv[0], 1);
	}
    }

  /* Initialize the m17n library.  */
  M17N_INIT ();
  if (merror_code != MERROR_NONE)
    FATAL_ERROR ("%s\n", "Fail to initialize the m17n library.");

  mt = mconv_decode_stream (Mcoding_utf_8, fp);
  fclose (fp);
  if (xml)
    mt = mtext_deserialize (mt);
  if (! mt)
    FATAL_ERROR ("%s\n", "Fail to decode the input file or stream!");

  len = mtext_len (mt);

  if (paper == PAPER_NOLIMIT)
    paper_width = paper_height = margin = 0;
  else
    {
      paper_width = paper_size[paper].width * dpi / 25.4;
      paper_height = paper_size[paper].height * dpi / 25.4;
      margin = margin * dpi / 25.4;
    }

  display = XOpenDisplay (NULL);
  screen = DefaultScreen (display);

  {
    MPlist *plist = mplist (), *p;
    MFontset *fontset = mfontset ("truetype");
    MFace *face = mface ();

    mface_put_prop (face, Mfontset, fontset);
    mface_put_prop (face, Msize, (void *) (fontsize * dpi / 100));
    p = mplist_add (plist, msymbol ("display"), display);
    p = mplist_add (p, msymbol ("depth"), (void *) 1);
    p = mplist_add (p, Mface, face);
    m17n_object_unref (face);
    frame = mframe (plist);
    m17n_object_unref (plist);
    if (! frame)
      FATAL_ERROR ("%s\n", "Can't open a frame (perhaps no font avairable)!");
  }

  memset (&control, 0, sizeof control);
  control.as_image = 1;
  control.two_dimensional = 1;
  control.enable_bidi = 1;
  if (cursor_pos >= 0)
    {
      control.with_cursor = 1;
      if (cursor_pos > len)
	cursor_pos = len;
      control.cursor_pos = cursor_pos;
      control.cursor_width = -1;
    }
  else
    control.ignore_formatting_char = 1;
  if (break_by_word)
    control.line_break = line_break;

  if (paper == PAPER_NOLIMIT)
    {
      control.max_line_width = 0;
      mdraw_text_extents (frame, mt, 0, len, &control, NULL, NULL, &rect);
      paper_width = rect.width;
      paper_height = rect.height;
    }
  else
    control.max_line_width = paper_width - margin * 2;

  pixmap = XCreatePixmap (display, RootWindow (display, screen),
			  paper_width, paper_height, 1);
  gc = XCreateGC (display, pixmap, 0L, NULL);

  from = 0;
  page_index = 1;
  while (from < len)
    {
      int to;

      if (paper == PAPER_NOLIMIT)
	to = len;
      else
	to = find_page_end (frame, paper_height - margin * 2, mt, from,
			    &control, &rect);

      XSetForeground (display, gc, WhitePixel (display, screen));
      XFillRectangle (display, pixmap, gc, 0, 0, paper_width, paper_height);
      mdraw_text_with_control (frame, (MDrawWindow) pixmap,
			       margin, margin - rect.y, mt, from, to,
			       &control);
      XSetForeground (display, gc, BlackPixel (display, screen));
#if 0
      XDrawRectangle (display, pixmap, gc, margin, margin,
		      paper_width - margin * 2 - 1,
		      paper_height - margin * 2 - 1);
#endif
      image = XGetImage (display, pixmap, 0, 0, paper_width, paper_height,
			 AllPlanes, XYPixmap);
      XInitImage (image);
      dump_image (image, filename, filter, !WhitePixel (display, screen),
		  ((from > 0 || to < len) ? page_index : 0),
		  quiet_mode);

      from = to;
      page_index++;
    }

  m17n_object_unref (frame);
  m17n_object_unref (mt);
  M17N_FINI ();
  XCloseDisplay (display);
  exit (0);
}
#endif /* not FOR_DOXYGEN */
