/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: zobject.c,v 1.14 2004/06/21 14:10:47 sasa Exp $
 *
 ***************************************************************************/

#include <zorp/zobject.h>
#include <zorp/log.h>

typedef struct _ZClassFuncs
{ 
  gint method_count;
  void (*methods[1])(void);
} ZClassFuncs;

/**
 * z_object_free_method:
 * @s: ZObject instance
 * 
 * This function is the free method of the ZObject class. It currently does
 * nothing as ZObject has no real data structure.
 **/
void
z_object_free_method(ZObject *s G_GNUC_UNUSED)
{
  /* empty */
}

/* dummy ZClass class descriptor */
ZClass ZClass__class =
{
  Z_CLASS_HEADER,
  NULL,
  "ZClass",
  sizeof(ZClass),
  NULL
};

static ZObjectFuncs z_object_funcs = 
{
  Z_FUNCS_COUNT(ZObject),
  z_object_free_method
};

/* root class */
ZClass ZObject__class =
{
  Z_CLASS_HEADER,
  NULL,
  "ZObject",
  sizeof(ZObject),
  &z_object_funcs
};

/**
 * z_object_ref:
 * @self: ZObject instance
 *
 * Increment the reference count of @self.
 **/
ZObject *
z_object_ref(ZObject *self)
{
  if (self)
    {
      g_static_mutex_lock(&self->ref_lock);
      z_incref(&self->ref_cnt);
      g_static_mutex_unlock(&self->ref_lock);
    }
  return self;
}

/**
 * z_object_unref:
 * @self: ZObject instance
 *
 * Decrement the reference count of @self and free it if the reference count
 * goes down to zero.
 **/
void
z_object_unref(ZObject *self)
{
  if (self)
    {
      g_static_mutex_lock(&self->ref_lock);
      if (z_decref(&self->ref_cnt) == 0)
        {
          g_static_mutex_unlock(&self->ref_lock);
          z_object_free(self);
          g_free(self);
          return;
        }
      g_static_mutex_unlock(&self->ref_lock);
    }
}

/**
 * z_object_is_subclass:
 * @class: parent class
 * @subclass: child class
 *
 * This function checks whether @subclass is derived from @class if
 * debug is enabled.
 **/
gboolean
z_object_is_subclass(ZClass *class G_GNUC_UNUSED, ZClass *subclass G_GNUC_UNUSED)
{
  ZClass *p;
  
  p = subclass;
  while (p && p != class)
    p = p->super_class;
  return !!p;
}

/**
 * z_object_is_compatible:
 * @self: ZObject instance
 * @class: class descriptor
 *
 * This function checks whether @self is compatible of @class, e.g. whether
 * it is derived from @class.
 **/
gboolean
z_object_is_compatible(ZObject *self, ZClass *class)
{
  if (!self)
    return FALSE;
  else
    return z_object_is_subclass(class, self->isa);
}

/**
 * z_object_is_instance:
 * @self: ZObject instance
 * @class: class descriptor
 *
 * This function returns whether @self is an instance of @class.
 **/
gboolean
z_object_is_instance(ZObject *self, ZClass *class)
{
  if (!self)
    return FALSE;
  else
    return self->isa == class;
}

/**
 * z_object_resolve_funcs:
 * @class: class descriptor
 *
 * This function is called once for each class instantiated. It resolves
 * NULL methods to be used from parent classes.
 **/
static void
z_object_resolve_funcs(ZClass *class)
{
  gint i;
  
  if (class->funcs_resolved)
    return;
  if (class->super_class)
    {
      z_object_resolve_funcs(class->super_class);
      for (i = 0; i < class->super_class->funcs->method_count; i++)
        {
          /* inherit from parent */
          if (((ZClassFuncs *) class->funcs)->methods[i] == NULL)
            ((ZClassFuncs *) class->funcs)->methods[i] = ((ZClassFuncs *) class->super_class->funcs)->methods[i];
        }
    }
  class->funcs_resolved = TRUE;
}

/**
 * z_object_new:
 * @class: class descriptor
 *
 * This function takes a class descriptor and creates an instance of @class
 * by allocating an appropriate memory block, initializing the reference
 * counter and ZObject specific fields.
 **/
ZObject *
z_object_new(ZClass *class)
{
  ZObject *inst;
  
  if (!class->funcs_resolved)
    z_object_resolve_funcs(class);
  
  inst = g_malloc0(class->size);
  inst->isa = class;
  inst->ref_cnt = 1;
  return inst;
}

/**
 * z_object_new_compatible:
 * @class: class decriptor
 * @compat: class descriptor
 *
 * This function checks whether @class is compatible with @compat, and if it
 * is, it returns a new instance of @class.
 **/
ZObject *
z_object_new_compatible(ZClass *class, ZClass *compat)
{
  if (z_object_is_subclass(compat, class))
    {
      return z_object_new(class);
    }
  else
    {
      g_assert(0 && "Requested class is not compatible");
      return NULL;
    }
}
