/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: log.c,v 1.76 2004/09/28 17:14:48 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor : kisza
 * Last audited version: 1.7
 * Notes:
 *
 ***************************************************************************/

#include <zorp/log.h>
#include <zorp/thread.h>
#include <zorp/error.h>

#include <stdarg.h>
#include <stdio.h>
#include <fcntl.h>
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/types.h>
#ifdef HAVE_SYS_TIME_H
  #include <sys/time.h>
#endif
#include <time.h>

#ifdef G_OS_WIN32
#  include <windef.h>
#  include <winbase.h>
#  include <io.h>
#  include <sys/stat.h>
#  define close _close
#  define open _open
#else
#  include <sys/socket.h>
#  include <sys/un.h>
#endif

#undef G_LOG_DOMAIN
#define G_LOG_DOMAIN "Zorp"

/* per-thread GHashTable based logtag cache */
typedef struct _ZLogTagCache
{
  gboolean empty_hash;
  gboolean used;
  GHashTable *tag_hash;
} ZLogTagCache;

/* parsed item of a ZLogSpec, pattern and verbosity_level */
typedef struct _ZLogSpecItem
{
  gchar *pattern;
  gint verbose_level;
} ZLogSpecItem;

/* a complete logspec with a list of ZLogSpecItems */
typedef struct _ZLogSpec
{
  GSList *items;
  gint verbose_level;
} ZLogSpec;

/* protects logtag_caches array when grabbing and releasing a thread specific log cache */
static GStaticMutex logtag_cache_lock = G_STATIC_MUTEX_INIT;
static GStaticPrivate current_logtag_cache = G_STATIC_PRIVATE_INIT;
static GPtrArray *logtag_caches;

/* protects log_spec/log_spec_str */
static GStaticMutex log_spec_lock = G_STATIC_MUTEX_INIT;
static ZLogSpec log_spec;
static gchar *log_spec_str;
static gboolean log_escape_nonprintable_chars = FALSE;

static ZLogMapTagFunc log_map_tag;
static gint log_mapped_tags_count;
static guchar *log_mapped_tags_verb;

static gboolean log_tags;
static gboolean stderr_syslog = FALSE;



gchar fake_session_id[256] = "noname/nosession";

static GMainContext *log_context = NULL;

#ifndef G_OS_WIN32
/*
 * This is a private reimplementation of syslog() as that one had a
 * mysterious bug in it and my bug reports were ignored.
 */

#if HAVE_BUGGY_SYSLOG_IN_LIBC
 
#define SYSLOG_SOCKET "/dev/log"

const gchar *syslog_tag = NULL;
int syslog_fd = -1;


/**
 * z_open_syslog:
 * @tag: program name used in log messages
 *
 * Analogous to openlog(), open the syslog() connection to local syslogd. 
 *
 * Returns: TRUE when the operation succeeds
 **/
gboolean
z_open_syslog(const gchar *tag)
{
  struct sockaddr_un s_un;
  
  syslog_tag = tag;
  syslog_fd = socket(AF_UNIX, SOCK_STREAM, 0);
  
  if (syslog_fd == -1)
    {
      return FALSE;
    }
  
  s_un.sun_family = AF_UNIX;
  g_strlcpy(s_un.sun_path, SYSLOG_SOCKET, sizeof(s_un.sun_path));
  if (connect(syslog_fd, (struct sockaddr *) &s_un, sizeof(s_un)) == -1)
    {
      close(syslog_fd);
      syslog_fd = socket(AF_UNIX, SOCK_DGRAM, 0);
      if (connect(syslog_fd, (struct sockaddr *) &s_un, sizeof(s_un)) == -1)
        {
          close(syslog_fd);
          syslog_fd = -1;
          return FALSE;
        }
    }
  return TRUE;
}

/**
 * z_close_syslog_internal:
 * @fd: syslog connection to close
 *
 * Close the connection to the local syslogd. The syslog connection is
 * specified by the @fd argument to avoid races when reopening
 * connections.
 * 
 * Returns: whether the operation was succesful
 **/
gboolean
z_close_syslog_internal(int fd)
{
  if (fd != -1)
    {
      close(fd);
      return TRUE;
    }
  return FALSE;
}

/**
 * z_close_syslog:
 *
 * Close the connection to the global syslogd.
 *
 * Returns TRUE to indicate success.
 **/
gboolean
z_close_syslog(void)
{
  return z_close_syslog_internal(syslog_fd);
}

/**
 * z_send_syslog:
 * @pri: syslog priority 
 * @msg: syslog message
 *
 * Send the specified message to syslog.
 **/
gboolean
z_send_syslog(gint pri, const gchar *msg)
{
  /*
   * Log message max line length is 1024 (g_log max line length)
   * + 128 for timestamp, process name and pid
   */
  gchar buf[2048];
  const guchar *p;
  gchar timestamp[32];
  time_t now;
  struct tm t;
  gint len, rc = 0, attempt = 0;
  int sfd = syslog_fd;
  static GStaticMutex lock = G_STATIC_MUTEX_INIT;
  
  now = time(NULL);
  localtime_r(&now, &t);
  
  strftime(timestamp, sizeof(timestamp), "%h %e %H:%M:%S", &t);
  
  g_snprintf(buf, sizeof(buf), "<%d>%s %s[%d]: ", pri, timestamp, syslog_tag, (int) getpid());
  if (log_escape_nonprintable_chars)
    {
      len = strlen(buf);
      for (p = msg; *p && len < sizeof(buf) - 5; p++)
        {
          if (*p >= 0x20 && *p <= 0x7F)
            {
              buf[len++] = *p;
            }
          else
            {
              g_snprintf(&buf[len], 5, "<%02X>", (guchar) *p);
              len += 4;
            }
        }
    }
  else
    {
      g_strlcat(buf, msg, sizeof(buf) - 1);
      len = strlen(buf);
    }
  buf[len++] = '\n';
  buf[len] = 0;
  do
    {
      attempt++;
      if (sfd != -1)
        rc = write(sfd, buf, len);
      if (sfd == -1 || (rc == -1 && errno != EINTR && errno != EAGAIN))
        {
          g_static_mutex_lock(&lock);
          if (sfd == syslog_fd)
            {
              z_open_syslog(syslog_tag);
              z_close_syslog_internal(sfd);
            }
            
          sfd = syslog_fd;
          g_static_mutex_unlock(&lock);
        }
    }    
  while (rc == -1 && attempt <= 1);
  return TRUE;
}

#else

/* use the syslog implementation in libc */

/**
 * z_open_syslog:
 * @tag: program name to display in log messages
 *
 * Open a connection to the local system logging process. This function uses
 * the native syslog() implementation in libc to offer syslog functions.
 **/
gboolean
z_open_syslog(const gchar *tag)
{
  openlog(tag, LOG_NDELAY | LOG_PID, 0);
  return TRUE;
}

/**
 * z_close_syslog:
 *
 * Close the connection to the local syslog process. 
 **/
gboolean
z_close_syslog(void)
{
  closelog();
  return TRUE;
}

/**
 * z_send_syslog:
 * @pri: priority of the message 
 * @msg: message itself
 * 
 * Send a message to syslogd. It uses the syslog() function in libc.
 **/
gboolean
z_send_syslog(gint pri, const gchar *msg)
{
  syslog(pri, "%s", msg);
  return TRUE;
}

#endif

#else

int syslog_fd = -1;

/**
 * z_open_syslog:
 * @tag: program name used in log messages
 *
 * Analogous to openlog(), open the syslog() connection to local syslogd. 
 * Windows implementation.
 *
 * Returns: TRUE when the operation succeeds
 **/
gboolean
z_open_syslog(const gchar *tag)
{  
  char fn[256];

  g_strlcpy(fn, getenv("windir"), sizeof(fn));
  g_strlcat(fn, "\\debug\\", sizeof(fn));
  g_strlcat(fn, tag, sizeof(fn)); 

  if((syslog_fd = open(fn, _O_APPEND | _O_RDWR | _O_CREAT, _S_IREAD | _S_IWRITE  )) == -1)
    {
      close(syslog_fd);
      syslog_fd = -1;
      return FALSE;
    }
  return TRUE;
}

/**
 * z_close_syslog:
 *
 * Close the connection to the local syslog process. 
 **/
gboolean
z_close_syslog(void)
{
  if (syslog_fd != -1)
    {
      close(syslog_fd);
      return TRUE;
    }
  return FALSE;
}

#endif

/* logspec parsing and evaluation handling */

/**
 * z_log_glob_match:
 * @glob: pattern to match @tag against
 * @tag: message tag
 *
 * Check when the pattern in a logspec matches the specified tag.
 **/
static gboolean
z_log_spec_glob_match(const gchar *glob, const gchar *tag)
{
  gchar *p1, *p2;
  gint len1, len2;

  p1 = strchr(glob, '.');
  p2 = strchr(tag, '.');

  while (p1 && p2)
    {
      len1 = p1 - glob;
      len2 = p2 - tag;
      if (((len1 != 1) || (memcmp(glob, "*", 1) != 0)) &&
          ((len1 != len2) || memcmp(glob, tag, len1) != 0))
        return FALSE;
        
      glob = p1 + 1;
      tag = p2 + 1;

      p1 = strchr(glob, '.');
      p2 = strchr(tag, '.');
    }
  if (p1)
    len1 = p1 - glob;
  else
    len1 = strlen(glob);
  if (p2)
    len2 = p2 - tag;
  else
    len2 = strlen(tag);
  if (((len1 != 1) || (memcmp(glob, "*", 1) != 0)) &&
      ((len1 != len2) || memcmp(glob, tag, len1) != 0))
    return FALSE;
  glob += len1;
  tag += len2;
  if (strlen(glob) > strlen(tag))
    return FALSE;
  return TRUE;
}

/**
 * z_log_spec_eval:
 * @self: ZLogSpec structure
 * @tag: message to return verbosity for
 *
 * Evaluate the currently parsed logspec in @self and return the verbosity level
 * associated with @tag.
 **/
static gint
z_log_spec_eval(ZLogSpec *self, const gchar *tag)
{
  GSList *l;
  ZLogSpecItem *lsi;
  
  l = self->items;
  while (l)
    {
      lsi = (ZLogSpecItem *) l->data;
      if (z_log_spec_glob_match(lsi->pattern, tag))
        {
          return lsi->verbose_level;
        }
      l = g_slist_next(l);
    }
  return self->verbose_level;
}

/**
 * z_log_spec_destroy:
 * @self: ZLogSpec instance
 *
 * Free all resources associated with @self. Does not free @self as it is assummed
 * to be allocated statically.
 **/
static void
z_log_spec_destroy(ZLogSpec *self)
{
  GSList *l, *l_next;
  ZLogSpecItem *lsi;
  
  l = self->items;
  while (l)
    {
      l_next = g_slist_next(l);
      lsi = (ZLogSpecItem *) l->data;
      
      g_free(lsi->pattern);
      g_free(lsi);
      g_slist_free_1(l);
      l = l_next;
    }
  self->items = NULL;
}

/**
 * z_log_spec_init:
 * @self: ZLogSpec instance
 * @logspec_str: logspec specification
 * @default_verbosity: global verbosity level
 *
 * Parse a user-specified logspec into @self.
 **/
static gboolean
z_log_spec_init(ZLogSpec *self, const gchar *logspec_str, gint default_verbosity)
{
  ZLogSpecItem *item;
  gchar *tmp = g_strdup(logspec_str ? logspec_str : ""), *src;
  gint new_level;

  src = tmp;
  self->items = NULL;
  self->verbose_level = default_verbosity;
  
  while (*src)
    {
      const gchar *glob, *num;
      gchar *end;

      while (*src == ',' || *src == ' ')
        src++;

      glob = src;
      while (isalnum((guchar) (*src)) || *src == '.' || *src == '*')
        src++;

      if (*src != ':')
        {
          /* invalid log spec */
          goto invalid_logspec;
        }
      *src = 0;
      src++;
      num = src;

      new_level = strtoul(num, &end, 10);
      
      item = g_new(ZLogSpecItem, 1);
      item->pattern = g_strdup(glob);
      item->verbose_level = new_level;
      self->items = g_slist_prepend(self->items, item);
      
      src = end;
      while (*src && *src != ',')
        src++;
    }
  self->items = g_slist_reverse(self->items);
  g_free(tmp);
  return TRUE;
  
 invalid_logspec:
  z_log_spec_destroy(self);
  g_free(tmp);
  return FALSE;
}


/* log tag cache 
 *
 * Each thread has its own dedicated logtag cache to avoid locking. Caches are shared
 * which means that once a thread terminates it releases its cache which can be reused
 * by an independent thread.
 * */

/**
 * z_log_clear_caches:
 *
 * Clear all thread specific caches. It is called after changing the
 * verbosity level or the logspec.
 * 
 **/
void
z_log_clear_caches(void)
{
  guint i;
  
  g_static_mutex_lock(&logtag_cache_lock);
  for (i = 0; i < logtag_caches->len; i++)
    {
      ZLogTagCache *lc = g_ptr_array_index(logtag_caches, i);
      
      lc->empty_hash = TRUE;
    }
  g_static_mutex_unlock(&logtag_cache_lock);
  if (log_mapped_tags_verb)
    {
      memset(log_mapped_tags_verb, 0, log_mapped_tags_count * sizeof(log_mapped_tags_verb[0]));
    }
}

/**
 * z_log_grab_cache:
 *
 * Grab a thread specific log-tag cache.
 **/
void
z_log_grab_cache(void)
{
  guint i;
  ZLogTagCache *lc = NULL;
  
  g_static_mutex_lock(&logtag_cache_lock);
  
  for (i = 0; i < logtag_caches->len; i++)
    {
      lc = g_ptr_array_index(logtag_caches, i);
      
      if (!lc->used)
        break;
      else
        lc = NULL;
    }
  
  if (!lc)
    {
      lc = g_new0(ZLogTagCache, 1);
      lc->tag_hash = g_hash_table_new(g_str_hash, g_str_equal);
      g_ptr_array_add(logtag_caches, lc);
    }
  lc->used = 1;
  g_static_private_set(&current_logtag_cache, lc, NULL);
  
  g_static_mutex_unlock(&logtag_cache_lock);
}

/**
 * z_log_grab_cache:
 *
 * Release a thread specific log-tag cache to make it usable by other threads.
 **/
void
z_log_release_cache(void)
{
  ZLogTagCache *lc;
  
  g_static_mutex_lock(&logtag_cache_lock);
  
  lc = g_static_private_get(&current_logtag_cache);
  if (lc)
    lc->used = 0;
  
  g_static_mutex_unlock(&logtag_cache_lock);
}

/**
 * z_log_thread_started:
 * @thread:
 * @user_data:
 *
 * Thread startup function called at thread startup time to allocate a logtag cache.
 **/
static void
z_log_thread_started(ZThread *thread G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
  z_log_grab_cache();
}

/**
 * z_log_thread_started:
 * @thread:
 * @user_data:
 *
 * Thread startup function called at thread shutdown time to release the logtag cache.
 **/
static void
z_log_thread_stopped(ZThread *thread G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
  z_log_release_cache();
}

/* global log state manipulation */

/**
 * z_log_change_verbose_level:
 * @direction: specifies the change direction (-1 decrease, 0 set, 1 increase)
 * @value: change with this value
 *
 * This function changes the global verbosity level as specified by the
 * @direction and @value parameters.
 **/
gboolean
z_log_change_verbose_level(gint direction, gint value, gint *new_value)
{
  gint old_verbose_level = log_spec.verbose_level;
  
  g_static_mutex_lock(&log_spec_lock);
  if (direction < 0)
    log_spec.verbose_level -= value;
  else if (direction == 0)
    log_spec.verbose_level = value;
  else
    log_spec.verbose_level += value;
  if (log_spec.verbose_level < 0)
    log_spec.verbose_level = 0;
  if (log_spec.verbose_level > 10)
    log_spec.verbose_level = 10;
  g_static_mutex_unlock(&log_spec_lock);
  
  if (old_verbose_level != log_spec.verbose_level)
    {  
      z_log_clear_caches();
      /*LOG
        This message reports that Zorp changed its verbosity level.
       */
      z_log(NULL, CORE_INFO, 0, "Changing verbosity level; verbose_level='%d'", log_spec.verbose_level);
    }
  *new_value = log_spec.verbose_level;
  return TRUE;
}

/**
 * z_log_change_logspec:
 * @newspec: change the logspec to this value, leave it unchanged if set to NULL
 *
 * Change the logspec value and return the new setting.
 **/
gboolean
z_log_change_logspec(const gchar *new_log_spec_str, const gchar **new_value)
{
  if (new_log_spec_str)
    {
      ZLogSpec new_spec;
      
      if (z_log_spec_init(&new_spec, new_log_spec_str, log_spec.verbose_level))
        {
          g_static_mutex_lock(&log_spec_lock);
          z_log_spec_destroy(&log_spec);
          log_spec = new_spec;
          
          if (log_spec_str)
            g_free(log_spec_str);
            
          log_spec_str = g_strdup(new_log_spec_str);
          g_static_mutex_unlock(&log_spec_lock);
          z_log_clear_caches();
          /*LOG
            This message reports that Zorp changed its logspec.
           */
          z_log(NULL, CORE_INFO, 0, "Changing logspec; verbose_level='%d', logspec='%s'", log_spec.verbose_level, new_log_spec_str);
        }
      else
        {
          z_log(NULL, CORE_ERROR, 0, "Invalid logspec, reverting to old logspec; new_logspec='%s'", new_log_spec_str);
          return FALSE;
        }
    }
    
  *new_value = log_spec_str;
  return TRUE;
}

/**
 * z_log_enable_tag_map_cache:
 * @map_tags: function to map message tags to IDs
 * @max_tag: maximum ID value assigned to tags
 *
 * This function enables the "tag_map cache" which makes tag caching very
 * efficient by using an array based lookup instead of GHashTable. 
 * NOTE: this function can only be called once, at startup time.
 **/
void
z_log_enable_tag_map_cache(ZLogMapTagFunc map_tags, gint max_tag)
{
  g_assert(!log_map_tag);
  
  log_map_tag = map_tags;
  log_mapped_tags_count = max_tag;
  log_mapped_tags_verb = g_new0(guchar, max_tag);
}

/**
 * z_log_enabled_len:
 * @tag: log message tag
 * @tag_len: length of @tag
 * @level: log message level
 *
 * Checks if a message with a given class/level combination would actually
 * be written to the log. It can be used prior to constructing complex log
 * messages to decide whether the messages need to be constucted at all.
 * All results are cached, thus the second invocation will not parse the 
 * log specifications again.
 *
 * Returns: TRUE if the log would be written, FALSE otherwise
 **/
gboolean
z_log_enabled_len(gchar *tag, gsize tag_len, int level)
{
  gint verbose;
  ZLogTagCache *lc;
  GHashTable *tag_hash;
  
  if (G_LIKELY(!log_spec.items))
    {
      /* fastpath, no logspec, decision is simple */
      return level <= log_spec.verbose_level;
    }
  if (G_LIKELY(log_map_tag))
    {
      /* somewhat less fast path, map_tag is defined, use it for caching */
      gint tag_ndx = log_map_tag(tag, tag_len);
      if (G_LIKELY(tag_ndx != -1))
        {
          /* known keyword, use id indexed array to lookup tag specific verbosity */
          verbose = log_mapped_tags_verb[tag_ndx];
          if (G_LIKELY(verbose))
            verbose--;
          else
            {
              g_static_mutex_lock(&log_spec_lock);
              verbose = z_log_spec_eval(&log_spec, tag);
              log_mapped_tags_verb[tag_ndx] = verbose + 1;
              g_static_mutex_unlock(&log_spec_lock);
            }
          return level <= verbose;
        }
    }
  /* check slow ghashtable based cache */
  lc = ((ZLogTagCache *) g_static_private_get(&current_logtag_cache));
  if (!lc)
    {
      return level <= log_spec.verbose_level;
    }
  if (lc->empty_hash)
    {
      g_hash_table_destroy(lc->tag_hash);
      lc->tag_hash = g_hash_table_new(g_str_hash, g_str_equal);
      lc->empty_hash = FALSE;
    }
  tag_hash = lc->tag_hash;
  verbose = GPOINTER_TO_UINT(g_hash_table_lookup(tag_hash, tag));
  if (!verbose)
    {
      /* slooooww path, evaluate logspec */
      g_static_mutex_lock(&log_spec_lock);
      verbose = z_log_spec_eval(&log_spec, tag);
      g_static_mutex_unlock(&log_spec_lock);
      g_hash_table_insert(tag_hash, tag, GUINT_TO_POINTER(verbose + 1));
    }
  else
    verbose--;
  
  return (level <= verbose);
}

/* Main entry points for logging */

/**
 * z_log_session_id:
 * @session_id: default session_id
 *
 * This helper function is used by the z_log() macro to get the current
 * session id. If the argument is NULL the session_id assigned to the
 * current thread is used, otherwise the value of the argument is returned.
 **/
const gchar *
z_log_session_id(const gchar *session_id)
{
  if (session_id == NULL || session_id[0] == 0)
    {
      ZThread *thread = z_thread_self();
      if (thread == NULL)
        return fake_session_id;
      else
        return thread->name;
    }
  return session_id;
}


/**
 * z_logv2:
 * @class: log message class
 * @level: log message verbosity level
 * @format: log message format specified in printf form
 * @ap: format arguments va_list
 *
 * This function sends a message formatted as printf format string and
 * arguments to the syslog. The associated class/level pair is checked
 * whether the message really needs to be written.
 **/
void
z_logv2(gchar *class, int level, gchar *format, va_list ap)
{
  int saved_errno = errno;
  
  if (log_tags)
    {
      /*
       * g_log cut it in 1024. byte,
       * so eonught to write this amount of data.
       */
      gchar msgbuf[1024];
      g_vsnprintf(msgbuf, sizeof(msgbuf), format, ap);

#if ZORPLIB_ENABLE_TRACE
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%p -> %s(%d): %s", g_thread_self(), class, level, msgbuf);
#else
      g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s(%d): %s", class, level, msgbuf);
#endif
    }
  else
    {
      g_logv(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, format, ap);
    }
  errno = saved_errno;
}

/**
 * z_logv:
 * @class: log message class
 * @level: log message verbosity level
 * @format: log message format specified in printf form
 * @ap: format arguments va_list
 *
 * This function sends a message formatted as printf format string and
 * arguments to the syslog. The associated class/level pair is checked
 * whether the message really needs to be written.
 **/
void
z_logv(gchar *class, int level, gchar *format, va_list ap)
{
  if (z_log_enabled(class, level))
    z_logv2(class, level, format, ap);
}


/**
 * z_llog:
 * @class: log message class
 * @level: log message verbosity level
 * @format: log message format specified in printf form
 *
 * This message is the same as z_logv() but format string and arguments
 * are specified directly.
 **/
void
z_llog(gchar *class, int level, gchar *format, ...)
{
  va_list l;

  va_start(l, format);
  z_logv(class, level, format, l);
  va_end(l);
}

/**
 * z_llog2:
 * @class: log message class
 * @level: log message verbosity level
 * @format: log message format specified in printf form
 *
 * This message is the same as z_logv() but format string and arguments
 * are specified directly.
 **/
void
z_llog2(gchar *class, int level, gchar *format, ...)
{
  va_list l;

  va_start(l, format);
  z_logv2(class, level, format, l);
  va_end(l);
}

#ifdef G_OS_WIN32

void 
z_log(gchar *session_id, gchar *class, int level, gchar *format, ...)
{
  va_list l;
  gchar msgbuf[2048];


  if (!z_log_enabled(class, level))
    return;

  va_start(l, format);

  g_vsnprintf(msgbuf, sizeof(msgbuf), format, l);

  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%p -> %s(%d): %s", g_thread_self(), class, level, msgbuf);
  va_end(l);
}

#endif /* G_OS_WIN32 */

#ifndef G_OS_WIN32
/**
 * z_log_func:
 * @log_domain: GLIB log domain
 * @log_flags: GLIB log flags
 * @message: message
 * @user_data: not used 
 *
 * This function is registered as a GLib log handler and sends all GLIB
 * messages to messages to syslog. Zorp itself does not use GLIB logging
 * it calls z_log() directly.
 **/
static void
z_log_func(const gchar *log_domain G_GNUC_UNUSED,
	   GLogLevelFlags log_flags,
	   const gchar *message,
	   gpointer user_data G_GNUC_UNUSED)
{
  int pri = LOG_INFO;
  if (log_flags & G_LOG_LEVEL_DEBUG)
    pri = LOG_DEBUG;
  else if (log_flags & G_LOG_LEVEL_WARNING)
    pri = LOG_WARNING;
  else if (log_flags & G_LOG_LEVEL_ERROR)
    pri = LOG_ERR;

  z_send_syslog(pri | ZORP_SYSLOG_FACILITY, message);
}

#else

/*+ log handler function to send Win32 debug message +*/
static void
z_log_win32_debugmsg(const gchar *log_domain,
                 GLogLevelFlags  log_flags,
                    const gchar *message,
                       gpointer  user_data)
{
  OutputDebugString(message);
  OutputDebugString("\n");
}

static void
z_log_win32_syslogmsg(const gchar *log_domain,
                 GLogLevelFlags  log_flags,
                    const gchar *message,
                       gpointer  user_data)
{
  write(syslog_fd, message, strlen(message));
  write(syslog_fd, "\n", 1);
}

#endif

/**
 * z_fetch_stderr:
 * @channel: the read end of the STDERR pipe
 * @condition: the I/O condition triggering this callback
 * @arg: not used
 *
 * This function is registered as a read callback of the STDERR pipe to 
 * fetch messages sent to stderr. It fetches messages line-by-line and
 * uses z_log() to send messages to log.
 *
 * Returns: TRUE to indicate further reading is needed, FALSE otherwise
 **/
gboolean
z_fetch_stderr(GIOChannel *channel, GIOCondition condition G_GNUC_UNUSED, gpointer arg G_GNUC_UNUSED)
{
  gchar *line = NULL;
  GIOStatus status = G_IO_STATUS_NORMAL;
  GError *err = NULL;

  status = g_io_channel_read_line(channel, &line, NULL, NULL, &err);
  
  switch (status)
    {
    case G_IO_STATUS_NORMAL:
      /*NOLOG*/
      z_log(NULL, CORE_STDERR, 3, "%s", line);
      break;
    case G_IO_STATUS_AGAIN:
      break;
    case G_IO_STATUS_EOF:
      /*LOG
        This message indicates that the program closed its stderr. No further
        message on the stderr will be logged.
       */
      z_log(NULL, CORE_STDERR, 4, "The program closed its stderr. No further stderr logging is possible.");
      return FALSE;
    default:
      /*LOG
        This message indicates that an error occurred while reading from stderr.
       */
      z_log(NULL, CORE_STDERR, 3, "Can not read from stderr; result='%s'", (err != NULL) ? ((err)->message) : ("Unknown error"));
      return FALSE;
    }
  g_free(line);
  return TRUE;
}

/**
 * z_log_source_new:
 * @fd: read side of the stderr pipe
 *
 * Creates the source watching the stderr pipe.
 **/
void
z_log_source_new(gint fd)
{
  GIOChannel *channel;
  GSource *source;
  
  channel = g_io_channel_unix_new(fd);
  g_io_channel_set_encoding(channel, NULL, NULL);
  g_io_channel_set_flags(channel, G_IO_FLAG_NONBLOCK, NULL);
  source = g_io_create_watch(channel, G_IO_IN);
  g_source_set_callback(source, (GSourceFunc) z_fetch_stderr, NULL, NULL);
  g_source_attach(source, log_context);
  g_source_unref(source);
}

/**
 * z_log_run:
 * @user_data: thread data pointer, assumed to point to the stderr fd
 *
 * STDERR reading thread function.
 **/
void *
z_log_run(gpointer user_data)
{
  GMainContext *c, *old_context;
  gint *fd = (gint *)user_data;
  
  old_context = log_context = g_main_context_new();
  g_main_context_acquire(log_context);
  z_log_source_new(*fd);

  do
    {
      c = log_context;
  
      if (c)
        g_main_context_iteration(c, TRUE);
    }
  while (c);

  g_main_context_release(old_context);
  g_main_context_unref(old_context);
  return NULL;
}

/**
 * z_log_enable_syslog:
 * @syslog_name: syslog program name
 * 
 * This function can be called after z_log_init() to redirect internal
 * messages to syslog.
 **/
void
z_log_enable_syslog(const gchar *syslog_name)
{
  z_open_syslog(syslog_name);
#ifndef G_OS_WIN32
  g_log_set_handler(G_LOG_DOMAIN, 0xff, z_log_func, NULL);
#else
  g_log_set_handler(G_LOG_DOMAIN, 0xff, z_log_win32_syslogmsg, NULL);
#endif
}

/**
 * z_log_enable_stderr_redirect:
 * @threaded: specifies whether to use a separate thread for stderr reading
 * 
 * This function can be called after z_log_init() to redirect stderr
 * messages messages to the system log.
 **/
void
z_log_enable_stderr_redirect(gboolean threaded)
{
#ifndef G_OS_WIN32
  static int grab[2];

  if (pipe(grab) < 0)
    {
      /*LOG
        This message is indicates that pipe creation failed. It is likely
	that the system runs out of free fds.
       */
      z_log(NULL, CORE_ERROR, 3, "Error creating stderr-syslog pipe;");
      return;
    }
  stderr_syslog = TRUE;
  dup2(grab[1], 1);
  dup2(grab[1], 2);
  if (grab[1] != 2 && grab[1] != 1)
    close(grab[1]);

  if (threaded)
    {
      if (!z_thread_new("Log thread", z_log_run, &grab[0]))
        threaded = FALSE;
    }

  if (!threaded)
    {
      log_context = g_main_context_default();
      if (!g_main_context_acquire(log_context))
        {
          log_context = g_main_context_new();
          g_main_context_acquire(log_context);
        }
      g_main_context_ref(log_context);
      z_log_source_new(grab[0]);
    }
#endif
}


/**
 * z_log_init:
 * @ls: log specification
 * @verbose_level: global verbose level
 * @syslog_name: the program name to appear in syslogs
 * @flags: log flags (ZLF_* macros)
 *
 * Initialize the logging subsystem according to the options
 * in specified in the @flags parameter.
 **/
gboolean
z_log_init(const gchar *ls, gint verbose_level, const gchar *syslog_name, guint flags)
{
  if (!z_log_spec_init(&log_spec, ls, verbose_level))
    {
      z_log(NULL, CORE_ERROR, 0, "Invalid logspec; logspec='%s'", ls);
      return FALSE;
    }
  log_spec_str = ls ? g_strdup(ls) : NULL;
    
  log_tags = !!(flags & ZLF_TAGS);
  
  logtag_caches = g_ptr_array_new();
  
  z_log_grab_cache();
  z_thread_register_start_callback((GFunc) z_log_thread_started, NULL);
  z_thread_register_stop_callback((GFunc) z_log_thread_stopped, NULL);

  if (flags & ZLF_SYSLOG)
    {
      z_log_enable_syslog(syslog_name);

#ifndef G_OS_WIN32
      if (flags & ZLF_STDERR)
        z_log_enable_stderr_redirect(flags & ZLF_THREAD);
#endif
    }
#ifdef G_OS_WIN32
  else if (flags & ZLF_WINDEBUG)
    {
      g_log_set_handler(G_LOG_DOMAIN, 0xff, z_log_win32_debugmsg, NULL);
    }
#endif  

  if (flags & ZLF_ESCAPE)
    log_escape_nonprintable_chars = TRUE;
  return TRUE;
}

/**
 * z_log_destroy:
 *
 * Deinitialize the logging subsystem.
 **/
void
z_log_destroy(void)
{
  GMainContext *c;
#ifndef G_OS_WIN32  
  if (stderr_syslog)
    {
      close(1);
      close(2);
    }
#endif
  z_close_syslog();
  
  c = log_context;
  log_context = NULL;
  g_main_context_wakeup(c);
}
