/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: io.c,v 1.18 2004/05/22 14:04:16 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/io.h>
#include <zorp/log.h>
#include <zorp/cap.h>

#include <sys/types.h>
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <fcntl.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <netinet/tcp.h>
#  include <netinet/in.h>
#endif

/**
 * z_fd_set_nonblock:
 * @fd: file descriptor to change
 * @enable: specifies whether to enable or disable O_NONBLOCK
 *
 * This function enables or disables the non-blocking mode of operation on
 * the given fd.
 *
 * Returns: whether the operation was successful
 **/
gboolean 
z_fd_set_nonblock(int fd, gboolean enable)
{
#ifdef G_OS_WIN32

  /* Note: this assumes the fd is a socket. */
  unsigned long argp;
  argp = enable;
  if (ioctlsocket(fd, FIONBIO, &argp) == SOCKET_ERROR)
#else
  int flags;

  if ((flags = fcntl(fd, F_GETFL)) == -1)
    return FALSE;

  if (enable)
    flags |= O_NONBLOCK;
  else
    flags &= ~O_NONBLOCK;
    
  if (fcntl(fd, F_SETFL, flags) < 0)
#endif
    {
      /*LOG
        This message indicates that changing the blocking state on the given fd
        failed for the given reason. Please report this event to the
        Zorp QA team.
       */
      z_log(NULL, CORE_ERROR, 3, "Changing blocking mode failed; fd='%d', enable='%d', error='%s'", fd, enable, g_strerror(errno));
      return FALSE;
    }
  return TRUE;
}

/**
 * z_fd_set_keepalive:
 * @fd: file descriptor of a socket
 * @enable: whether to enable or disable TCP keepalive
 *
 * This function enables or disables the TCP keepalive feature for socket
 * specified by @fd.
 *
 * Returns: whether the operation was successful
 **/
gboolean
z_fd_set_keepalive(int fd, gboolean enable)
{
  if (setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, (char *)(&enable), sizeof(enable)) == -1)
    {
      /*LOG
        This message indicates that changing the keep-alive state on the given fd
        failed for the given reason. Please report this event to the
        Zorp QA team.
       */
      z_log(NULL, CORE_ERROR, 4, "setsockopt(SOL_SOCKET, SO_KEEPALIVE) failed; fd='%d', enable='%d', error='%s'", fd, enable, g_strerror(errno));
      return FALSE;
    }
  return TRUE;
}

/**
 * z_fd_set_keepalive:
 * @fd: file descriptor of a socket
 * @enable: whether to enable or disable TCP SO_OOBINLINE
 *
 * This function enables or disables the TCP SO_OOBINLINE feature for socket
 * specified by @fd.
 *
 * Returns: whether the operation was successful
 **/
gboolean
z_fd_set_oobinline(int fd, gboolean enable)
{
  if (setsockopt(fd, SOL_SOCKET, SO_OOBINLINE, (char *)(&enable), sizeof(enable)) == -1)
    {
      /*LOG
        This message indicates that changing the OOBINLINE state on the given fd
        failed for the given reason. Please report this event to the 
        Zorp QA team.
       */
      z_log(NULL, CORE_ERROR, 4, "setsockopt(SOL_SOCKET, SO_OOBINLINE) failed; fd='%d', enable='%d', error='%s'", fd, enable, g_strerror(errno));
      return FALSE;
    }
  return TRUE;
}

#if ZORPLIB_ENABLE_TOS

gboolean
z_fd_get_peer_tos(gint fd, gint *tos)
{
  gint tmp;
  guchar tos_value;
  gchar buf[256];
  gsize buflen, len;
    
  *tos = -1;
  tmp = 1;
  if (setsockopt(fd, SOL_IP, IP_RECVTOS, &tmp, sizeof(tmp)) < 0)
    {
      z_log(NULL, CORE_ERROR, 8, "Error in setsockopt(SOL_IP, IP_RECVTOS); fd='%d', error='%s'", fd, g_strerror(errno));
    }

  buflen = sizeof(buf);
  len = sizeof(tos_value);
  if (getsockopt(fd, SOL_IP, IP_PKTOPTIONS, &buf, &buflen) >= 0)
    {
      struct msghdr msg;
      struct cmsghdr *cmsg;
      
      msg.msg_controllen = buflen;
      msg.msg_control = buf;
      for (cmsg = CMSG_FIRSTHDR(&msg); cmsg; cmsg = CMSG_NXTHDR(&msg, cmsg))
        {
          if (cmsg->cmsg_level == SOL_IP && cmsg->cmsg_type == IP_TOS)
            {
              *tos = *(guchar *) CMSG_DATA(cmsg);
              break;
            }
        }
    }
  if (*tos == -1)
    {
      if (getsockopt(fd, SOL_IP, IP_TOS, &tos_value, &len) >= 0)
        {
          *tos = tos_value;
        }
      else
        {
          z_log(NULL, CORE_ERROR, 2, "Error in getsockopt(SOL_IP, IP_PKTOPTIONS) || getsockopt(SOL_IP, IP_TOS); fd='%d', error='%s'", fd, g_strerror(errno));
          *tos = -1;
          return FALSE;
        }
    }
  return TRUE;
}

gboolean
z_fd_set_our_tos(gint fd, gint tos)
{
  gboolean res = TRUE;
  
  if (tos != -1)
    {
      guchar tos_value = tos & 0xFF;
      gsize len;
      cap_t saved_caps;
      
      saved_caps = cap_save();
      len = sizeof(tos_value);
      cap_enable(CAP_NET_ADMIN);
      if (setsockopt(fd, SOL_IP, IP_TOS, &tos_value, len) < 0)
        {
          z_log(NULL, CORE_ERROR, 3, "Error setting ToS value on socket; fd='%d', tos='%d', error='%s'", fd, tos, g_strerror(errno));
          res = FALSE;
        }
      else
        {
          z_log(NULL, CORE_DEBUG, 6, "Setting socket ToS value; fd='%d', tos='%d'", fd, tos);
        }
      cap_restore(saved_caps);
    }
  return res;
}

gboolean
z_fd_get_our_tos(gint fd, gint *tos)
{
  guchar tos_value;
  gsize len;
  
  len = sizeof(tos_value);
  if (getsockopt(fd, SOL_IP, IP_TOS, &tos_value, &len) < 0)
    {
      z_log(NULL, CORE_ERROR, 2, "Error in getsockopt(SOL_IP, IP_TOS); fd='%d', error='%s'", fd, g_strerror(errno));
      *tos = -1;
      return FALSE;
    }
  *tos = tos_value;
  return TRUE;
}

#endif
