/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: stream.c,v 1.57.2.2 2003/09/16 19:40:28 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/stream.h>
#include <zorp/log.h>
#include <zorp/error.h>

#include <string.h>
#include <sys/types.h>

/*
 * Innentol vannak a nyilvanos source fuggvenyek. Ezek csak a
 * stream-hez rendelt source-okat hivjak.
 */


gboolean
z_stream_source_prepare(GSource *s, gint *timeout)
{
  ZStreamSource *self = (ZStreamSource *)s;
  gboolean ret = FALSE;
  
  z_enter();
  ret = self->stream->funcs->watch_prepare(self->stream, s, timeout);
  z_leave();
  return ret;
}

gboolean
z_stream_source_check (GSource *s)
{
  ZStreamSource *self = (ZStreamSource *)s;
  gboolean ret = FALSE;

  z_enter();
  ret = self->stream->funcs->watch_check(self->stream, s);
  z_leave();
  return ret;
}

gboolean
z_stream_source_dispatch (GSource *s,
                     GSourceFunc  callback G_GNUC_UNUSED,
                       gpointer   user_data G_GNUC_UNUSED)
{
  ZStreamSource *self = (ZStreamSource *)s;
  gboolean ret = FALSE;
      
  z_enter();
  ret = self->stream->funcs->watch_dispatch(self->stream, s);
  z_leave();
  return ret;
}

void
z_stream_source_finalize(GSource *s)
{
  ZStreamSource *self = (ZStreamSource *)s;
  ZStream *stream;
  
  z_enter();

  if (self->stream->funcs->watch_finalize != NULL)
    self->stream->funcs->watch_finalize(self->stream, s);

  if (self->stream->user_data_read && self->stream->user_data_read_notify)
    self->stream->user_data_read_notify(self->stream->user_data_read);
  if (self->stream->user_data_write && self->stream->user_data_write_notify)
    self->stream->user_data_write_notify(self->stream->user_data_write);
  if (self->stream->user_data_pri && self->stream->user_data_pri_notify)
    self->stream->user_data_pri_notify(self->stream->user_data_pri);

  stream = self->stream;
  self->stream = NULL;

  z_stream_unref(stream);
  z_leave();
}

static GSourceFuncs z_stream_source_funcs = 
{
  z_stream_source_prepare,
  z_stream_source_check,   
  z_stream_source_dispatch,
  z_stream_source_finalize
};

GSource *
z_stream_source_new(ZStream *stream)
{
  ZStreamSource *self = (ZStreamSource *) g_source_new(&z_stream_source_funcs, sizeof(ZStreamSource));
  
  z_enter();
  z_stream_ref(stream);
  self->stream = stream;

  z_leave();
  return &self->super;
}

/* general functions to be used as ZStream virtual functions */

gboolean
z_stream_ctrl_method(ZStream *s, guint function, gpointer value, guint vlen)
{
  gboolean res = FALSE;
  
  z_enter();
  switch (ZST_CTRL_MSG(function))
    {
    case ZST_CTRL_GET_COND_READ:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = s->want_read;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_COND_READ:
      if (vlen == sizeof(gboolean))
        {
          s->want_read =  *((gboolean *)value);
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_COND_WRITE:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = s->want_write;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_COND_WRITE:
      if (vlen == sizeof(gboolean))
        {
          s->want_write = *((gboolean *)value);
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_COND_PRI:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = s->want_pri;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_COND_PRI:
      if (vlen == sizeof(gboolean))
        {
          s->want_pri = *((gboolean *)value);
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_CALLBACK_READ:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          cbv->cb = s->read_cb;
          cbv->cb_data = s->user_data_read;
          cbv->cb_notify = s->user_data_read_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_CALLBACK_READ:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          s->read_cb = cbv->cb;
          s->user_data_read = cbv->cb_data;
          s->user_data_read_notify = cbv->cb_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_CALLBACK_WRITE:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          cbv->cb = s->write_cb;
          cbv->cb_data = s->user_data_write;
          cbv->cb_notify = s->user_data_write_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_CALLBACK_WRITE:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          s->write_cb = cbv->cb;
          s->user_data_write = cbv->cb_data;
          s->user_data_write_notify = cbv->cb_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_CALLBACK_PRI:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          cbv->cb = s->pri_cb;
          cbv->cb_data = s->user_data_pri;
          cbv->cb_notify = s->user_data_pri_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_CALLBACK_PRI:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          s->pri_cb = cbv->cb;
          s->user_data_pri = cbv->cb_data;
          s->user_data_pri_notify = cbv->cb_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_TIMEOUT_BLOCK:
      if (vlen == sizeof(gint))
        {
          s->timeout = *((gint *)value);
          res = TRUE;
        }
      break;
    default:
      if (s->parent)
        {
          res = z_stream_ctrl(s->parent, function, value, vlen);
          z_leave();
          return res;
        }
      break;
    }

  if (res && (function & ZST_CTRL_MSG_FORWARD) && s->parent)
    res = z_stream_ctrl(s->parent, function, value, vlen);
  
  z_leave();
  return res;
}

void
z_stream_set_timeout_forward(ZStream *s, gint timeout_type, gint timeout_value)
{
  z_stream_set_timeout(s->parent, timeout_type, timeout_value);
}

void
z_stream_init(ZStream *s, gint type, ZStreamFuncs *funcs, gchar *name)
{
  s->ref = 1;
  s->type = type;
  s->funcs = funcs;
  s->timeout = -2;
  s->time_open = time(NULL);
  g_strlcpy(s->name, name, sizeof(s->name));
}

void
z_stream_ref(ZStream *stream)
{
  if (stream)
    {
      g_static_mutex_lock(&stream->lock);
      stream->ref++;
      g_static_mutex_unlock(&stream->lock);
    }
}

void
z_stream_unref(ZStream *stream)
{
  if (!stream)
    return;
  
  g_static_mutex_lock(&stream->lock);
  if (--stream->ref == 0)
    {
      g_static_mutex_unlock(&stream->lock);
      stream->funcs->freefn(stream);
    }
  else
    g_static_mutex_unlock(&stream->lock);
}
