/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: source.c,v 1.23.2.3 2003/05/27 17:18:56 sasa Exp $
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/source.h>
#include <zorp/log.h>

#include <glib.h>

/* miscelaneous source objects for use throughout the ZMS */

typedef struct _ZThresholdSource
{
  GSource super;
  guint idle_threshold;
  guint busy_threshold;
  time_t last_call;
  time_t start_time;
} ZThresholdSource;

/**
 * z_threshold_source_prepare:
 * @s: source object
 * @timeout: returns the calculated timeout here
 *
 * prepare() function for the threshold source
 **/
static gboolean
z_threshold_source_prepare(GSource *s, gint *timeout)
{
  ZThresholdSource *self = (ZThresholdSource *)s;
  time_t now;
  
  now = time(NULL);
  self->start_time = now;
  
  *timeout = MIN(self->idle_threshold,
                 self->busy_threshold + self->last_call - now) * 1000;
  
  return FALSE;
}

/**
 * z_threshold_source_check:
 * @s: source object
 *
 * check() function for the threshold source
 **/
static gboolean
z_threshold_source_check(GSource *s)
{
  ZThresholdSource *self = (ZThresholdSource *) s;
  time_t now;
  gboolean ret;
  
  z_enter();
  
  now = time(NULL);
  
  ret = ((time_t)(self->start_time + self->idle_threshold) <= now);
  ret = ret || ((time_t)(self->last_call + self->busy_threshold) <= now);
  
  z_leave();
  return ret;
}

/**
 * z_threshold_source_dispatch:
 * @s: source object
 * @callback: callback function associated with the source
 * @user_data: pointer to be passed to the callback function
 *
 * dispatch() function for the threshold source
 **/
static gboolean
z_threshold_source_dispatch(GSource     *s,
                            GSourceFunc  callback,
                            gpointer     user_data)
{
  ZThresholdSource *self = (ZThresholdSource *)s;
  gboolean rc = FALSE;

  z_enter();
  
  if (callback != NULL)
    {
      rc =  (*callback) (user_data);
      self->last_call = time(NULL);
    }
  else
    {
      /*LOG
        This message indicate an internal error, when
        callback does'nt link to the source.
       */
      z_log(NULL, CORE_ERROR, 4, "Callback function not set!");
    }
  
  z_leave();
  return rc;
}

static GSourceFuncs 
z_threshold_source_funcs = 
{
  z_threshold_source_prepare,
  z_threshold_source_check,
  z_threshold_source_dispatch,
  NULL
};

/**
 * z_threshold_source_new:
 * @idle_threshold:
 * @busy_threshold:
 * 
 *
 **/
GSource *
z_threshold_source_new(guint idle_threshold, guint busy_threshold)
{
  ZThresholdSource *self;
  
  self = (ZThresholdSource *) g_source_new(&z_threshold_source_funcs, sizeof(ZThresholdSource));
  self->idle_threshold = idle_threshold;
  self->busy_threshold = busy_threshold;
  return &self->super;
}

/**
 * z_threshold_source_set_threshold:
 * @source: the threshold source instance
 * @idle_threshold: new idle threshold
 * @busy_threshold: new busy threshold
 * 
 * This function changes the thresholds associated with the threshold source.
 **/
void
z_threshold_source_set_threshold(GSource *source, guint idle_threshold, guint busy_threshold)
{
  ZThresholdSource *self = (ZThresholdSource *) source;
  
  self->idle_threshold = idle_threshold;
  self->busy_threshold = busy_threshold;
}

typedef struct _ZTimeoutSource
{
  GSource super;
  time_t timeout_target;
} ZTimeoutSource;

/**
 * z_timeout_source_prepare:
 * @s: source object
 * @timeout: returns the calculated timeout here
 *
 * prepare() function for the timeout source
 **/
static gboolean
z_timeout_source_prepare(GSource *s, gint *timeout)
{
  ZTimeoutSource *self = (ZTimeoutSource *)s;
  GTimeVal now;
  
  g_source_get_current_time(s, &now);
  if (self->timeout_target != -1 && self->timeout_target <= now.tv_sec)
    {
      return TRUE;
    }
  else if (self->timeout_target != -1)
    {
      *timeout = (self->timeout_target - now.tv_sec) * 1000;
    }
  
  return FALSE;
}

/**
 * z_timeout_source_check:
 * @s: source object
 *
 * check() function for the timeout source
 **/
static gboolean
z_timeout_source_check(GSource *s)
{
  ZTimeoutSource *self = (ZTimeoutSource *) s;
  GTimeVal now;
  
  g_source_get_current_time(s, &now);
  if (self->timeout_target != -1 && self->timeout_target <= now.tv_sec)
    {	
      return TRUE;
    }
  return FALSE;
}

/**
 * z_timeout_source_dispatch:
 * @s: source object
 * @callback: callback function associated with the source
 * @user_data: pointer to be passed to the callback function
 *
 * dispatch() function for the timeout source
 **/
static gboolean
z_timeout_source_dispatch(GSource     *s,
                          GSourceFunc  callback,
                          gpointer     user_data)
{
  return (*callback)(user_data);
}

static GSourceFuncs z_timeout_source_funcs = 
{
  z_timeout_source_prepare,
  z_timeout_source_check,
  z_timeout_source_dispatch,
  NULL
};

/**
 * z_timeout_source_set_timeout:
 * @source: a reference to the timeout source
 * @new_timeout: the new timeout value 
 * 
 * This function changes the timeout associated to the timeout source
 * pointed to by @source.
 **/
void
z_timeout_source_set_timeout(GSource *source, gulong new_timeout)
{
  ZTimeoutSource *self = (ZTimeoutSource *) source;
  
  self->timeout_target = time(NULL) + new_timeout / 1000;
}

/**
 * z_timeout_source_new:
 * @initial_timeout: the initial timeout value
 *
 * This function creates and initializes a source which issues a callback
 * when a given timeout elapses. It does not matter how many times the
 * poll() loop runs as the time is saved as the target time instead of an
 * interval which would start each time the poll loop is run.
 **/
GSource *
z_timeout_source_new(gulong initial_timeout)
{
  ZTimeoutSource *self;
  
  self = (ZTimeoutSource *) g_source_new(&z_timeout_source_funcs, sizeof(ZTimeoutSource));
  self->timeout_target = time(NULL) + initial_timeout / 1000;
  return &self->super;
}

