/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      PicoUsbDevice.h 
 * \brief     Defines the PicoUsbDevice class
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: PicoUsbDevice.h,v 1.6 2008/02/13 20:15:20 douglas Exp $
 *
 *************************************************************************** */

// Avoid including this header more than once
#ifndef PICOUSBDEVICE_H
#define PICOUSBDEVICE_H

#include "PicoStatus.h" // PICO_RETURNS and PICODEVICE_STATES types

// ////////////////////////////////////////////////////////////////////////////
/// The PicoUsbDevice class defines a platform-independent interface through
/// which product-specific drivers can access the Pico USB hardware. It 
/// comprises two main parts:
/// <ul><li> A collection of static member functions for listing devices
/// available on the system. The implementation of these functions is 
/// platform-specific and may be found in the files PicoLinuxUsbDevice.cpp and 
/// PicoMacOSXUsbDevice.cpp.
/// <li> A set of member functions which must be implemented by platform-specific 
/// derived classes (the "PicoUsbDevice interface"). Note that there is no 
/// constructor defined: it is platform-specific as the constructor takes
/// the OS interface to the USB device as an argument. The constructor
/// is only called by the platform-specific implementation of the static 
/// functions and is never called by user code.</ul>
// ////////////////////////////////////////////////////////////////////////////
class PicoUsbDevice {
public:
	/// Status of a USB pipe
	typedef enum {
		Stalled=-3,            ///< A pipe is stalled when...
		ErrorGettingStatus=-2, ///< Error occurred while determining the pipe status.
		ErrorGettingInfo=-1,   ///< Error occurred getting the information for the PipeInformation struct.
		StatusOK=0             ///< Pipe is available.
	} PipeStatus;


	/// To contain the information returned by GetPipeInformation
	typedef struct {
		unsigned char direction;
		unsigned char number;
		unsigned char transferType;
		unsigned short maxPacketSize;
		unsigned char interval;
		PipeStatus status;
	} PipeInformation;
	
	// Static functions
	public:
		static int Init();
		static int Count(unsigned short product);
		static int Enumerate(PicoUsbDevice **list, unsigned int length, unsigned short product);
		
	// Virtual public functions
	public:
		// Opening and closing
		virtual PICODEVICE_STATES Open (void) = 0;
		virtual void              Close(void) = 0;
		virtual ~PicoUsbDevice(){}
		
		// Getting device information
		virtual int GetPID(void)   = 0;
		virtual int GetDID(void)   = 0;
		virtual int GetDeviceClass(void)   = 0;
		virtual int GetPipes(void) = 0;
		virtual short GetHandle(void) = 0;
		virtual const char *      GetSerialString(void) = 0;
		virtual PICODEVICE_STATES GetDeviceState (void) = 0;
		virtual PicoUsbDevice::PipeInformation GetPipeInformation(int pipeNumber) = 0;
		
		// Data transfer
		virtual PICO_RETURNS ReadPipe (int pipeNumber,void *buf,unsigned int *size) = 0;
		virtual PICO_RETURNS WritePipe(int pipeNumber,void *buf,unsigned int size) = 0;
		virtual PICO_RETURNS ResetPipe(int pipeNumber) = 0;
};

#endif // PICOUSBDEVICE_H
