/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/


#include "cursor.h"
#include "sdo_oci.h"
#include "TeUtils.h"


//Given a string, remove all blanks and tabs (rigth)
char* StrClean(char *name)
{
    int j;
	
	if ((name == NULL)) /*SGError.Handler(NULLPOINTER,FATAL);*/ 
		return 0;

	for (j = strlen(name)-1; j>=0; j--)
	{
		if ( !((name[j]==' ') || (name[j]=='\t') || (name[j]== '\0')) ) 
			break;
	}
	
	name[j+1] = '\0';

	return name;
}


OCICursor::OCICursor(OCIConnection* connec)
{
	int i;
	conn = connec;
	stmthpToQuery = NULL;
	dschp = NULL;
	fieldValue = "";
	ordinates = 0;

	isOpen_ = false;
	for(i=0;i<MAX_ROWS;i++)
	{
		global_geom_obj[i] = NULL;
		global_geom_ind[i] = NULL;
		lobp[i] = NULL; 
	}
	
	row_Index = -1;
	rows_Fetched = 0;
	rows_Mem = 0;
	row_Cur = -1;
	last_Row = false;
	errorMessage = "";
	numColls_ = -1;
}

bool OCICursor::Open()  
{
	
	sword status;
	if(isOpen_)
		this->Close();
	
	isOpen_ = false;

	// Initialize statement handle 
	status = SDO_OCIHandleAlloc((dvoid*)conn->envhp, (dvoid**)&stmthpToQuery, 
								  (ub4)OCI_HTYPE_STMT, (size_t)0, (dvoid**)0);
	if(!CheckError(status))
		return false;
	
	// describe spatial object types (OCIDescribe)
	status = SDO_OCIHandleAlloc((dvoid*)conn->envhp, (dvoid**)&dschp, 
								(ub4)OCI_HTYPE_DESCRIBE, (size_t)0, (dvoid **)0);
	if(!CheckError(status))
		return false;

	status = SDO_OCIObjectNew(conn->envhp, conn->errhp, conn->svchp, OCI_TYPECODE_VARRAY, 
					conn->tdo_ordinates, (dvoid*)NULL, OCI_DURATION_SESSION, TRUE, 
					(dvoid**)&ordinates);

	if(!CheckError(status))
		return false;

	isOpen_ = true;
	return true;
}

void OCICursor::Close()
{
	// Finalize stmthpToQuery and dschp
	if(isOpen_)
	{
		SDO_OCIHandleFree((dvoid *)stmthpToQuery, (ub4)OCI_HTYPE_STMT);
		SDO_OCIHandleFree((dvoid*)dschp, (ub4)OCI_HTYPE_DESCRIBE);
		FreeResult();
		isOpen_ = false;
	}
	numColls_ = -1;
	stmthpToQuery = NULL;
	dschp = NULL;
}

void OCICursor::FreeResult()
{	
	int i;
	if(global_geom_obj)
	{
		for(i=0;i<rows_Mem;i++)  // O que est em memria ou max rows??????
		{
		/// free the spatial object instance 
			CheckError(OCIObjectFree(conn->envhp, conn->errhp, (dvoid *)global_geom_obj[i], 
				   (ub2)OCI_OBJECTFREE_FORCE));

			global_geom_obj[i] = NULL;
			global_geom_ind[i] = NULL;
		}
	}

	if (lobp)
	{
		for(i=0;i<MAX_ROWS;i++)
		{
			CheckError(OCIDescriptorFree((dvoid *) lobp, (ub4) OCI_DTYPE_LOB));
			lobp[i] = NULL;
		}
	}

	OCIObjectFree(conn->envhp, conn->errhp, (dvoid *)ordinates, (ub2)OCI_OBJECTFREE_FORCE);
	ordinates = 0;

	vector<void *>::iterator itb = buffers.begin();
	for (itb; itb!=buffers.end(); itb++)
		delete(*(itb));
	
	buffers.clear();
	defines.clear(); // ver se o freestmtp desaloca os OCIDefines 
	ind.clear();
	colType_.clear();
	colSize_.clear();
	colScale_.clear();
	colName_.clear();
	numColls_ = -1;
}


void OCICursor::DefineByPos(int pos, void* value, int size, void* indicator, 
						 int type)
{
	OCIDefine *defnp = NULL;

	CheckError(SDO_OCIDefineByPos(stmthpToQuery, &defnp, conn->errhp, (ub4)pos, 
								  (dvoid *)value, (sb4)size, type, 
								  (dvoid *)indicator, (ub2 *)0, (ub2 *)0,
								  (ub4)OCI_DEFAULT));
}


bool OCICursor::Fetch(int rows)
{
	sword status;
	status = OCIStmtFetch(stmthpToQuery, conn->errhp, (ub4) rows, (ub4) OCI_FETCH_NEXT,
                               (ub4) OCI_DEFAULT);

	CheckError(status);

	if (status == OCI_SUCCESS || status == OCI_SUCCESS_WITH_INFO)
		return true;
	else
		return false;

}

bool OCICursor::AppendOrdinates(const double& val)
{
	
	OCINumber	oci_number;
	sword       status;

	if(!ordinates)
	{
		status = SDO_OCIObjectNew(conn->envhp, conn->errhp, conn->svchp, OCI_TYPECODE_VARRAY, 
					conn->tdo_ordinates, (dvoid*)NULL, OCI_DURATION_SESSION, TRUE, 
					(dvoid**)&ordinates);

		if(!CheckError(status))
			return false;
	}
	
	status = OCINumberFromReal(conn->errhp, (dvoid *)&(val), 
		(uword)sizeof(double),&oci_number);

	status = OCICollAppend(conn->envhp, conn->errhp, 
		(dvoid *) &oci_number,
		(dvoid *)0, (OCIColl *)ordinates);

	if(!CheckError(status))
		return false;

	return true;
}

bool OCICursor::BindOrdinates()
{
	OCIBind		*bnd1p = NULL;
	sword		status; 

	if(!ordinates)
	{
		status = SDO_OCIObjectNew(conn->envhp, conn->errhp, conn->svchp, OCI_TYPECODE_VARRAY, 
					conn->tdo_ordinates, (dvoid*)NULL, OCI_DURATION_SESSION, TRUE, 
					(dvoid**)&ordinates);

		if(!CheckError(status))
			return false;
	}
	
	/* bind coordinate varray object */
	status = OCIBindByName(stmthpToQuery, &bnd1p, conn->errhp, 
	    (text *)":ordinates", (sb4)-1, (dvoid *)0, (sb4)0, SQLT_NTY, (dvoid *)0, 
		(ub2 *)0, (ub2 *)0, (ub4)0, (ub4 *)0, (ub4)OCI_DEFAULT);
	if(!CheckError(status))
		return false;

	status = OCIBindObject(bnd1p, conn->errhp, conn->tdo_ordinates, (dvoid **)&ordinates, (ub4 *)0, 
	    (dvoid **)0, (ub4 *)0);
	if(!CheckError(status))
		return false;

	return true;
}

bool OCICursor::MoveFirst()
{
	// this is for scroolable cursor
	sword status = OCIStmtFetch2(stmthpToQuery, conn->errhp, (ub4)MAX_ROWS, OCI_FETCH_FIRST, (sb4) 0, OCI_DEFAULT);

	if (status == OCI_SUCCESS || status == OCI_SUCCESS_WITH_INFO  || status == OCI_NO_DATA)
	{
		OCIAttrGet((dvoid *)stmthpToQuery, (ub4)OCI_HTYPE_STMT,
			     (dvoid *)&rows_Fetched, (ub4 *)0, 
			     (ub4)OCI_ATTR_ROW_COUNT, conn->errhp);

		if(!rows_Fetched)  
			return false;

		if(status == OCI_NO_DATA)
			last_Row = true;

		row_Index = 0;

		if(rows_Fetched < MAX_ROWS)  
			rows_Mem = rows_Fetched;
		else
			rows_Mem = MAX_ROWS;

		row_Cur = 0;
		return true;
	}

	return false;
}

bool OCICursor::MoveNext()
{
	sword status;
	
	if((row_Index+1) >= rows_Mem) 
	{
		if(last_Row)
			return false;
		
		// this is for scroolable cursor
		status = OCIStmtFetch2(stmthpToQuery, conn->errhp, (ub4) MAX_ROWS, /*OCI_DEFAULT*/ OCI_FETCH_NEXT, (sb4) 0, OCI_DEFAULT);

		if (status == OCI_SUCCESS || status == OCI_SUCCESS_WITH_INFO || status == OCI_NO_DATA)
		{
		
			OCIAttrGet((dvoid *)stmthpToQuery, (ub4)OCI_HTYPE_STMT,
					 (dvoid *)&rows_Fetched, (ub4 *)0, 
					 (ub4)OCI_ATTR_ROWS_FETCHED, conn->errhp);

			if(!rows_Fetched)
				return false;

			if(status == OCI_NO_DATA)
			{
				if(last_Row)
					return false;
				else
					last_Row = true;
			}

			row_Index = 0;

			if(rows_Fetched < MAX_ROWS)
				rows_Mem = rows_Fetched;
			else
				rows_Mem = MAX_ROWS;

			row_Cur++;
			return true;
		}
		else
			return false;
	}
	else
		row_Index++;

	row_Cur++;
	return true;
}

bool OCICursor::MoveLast()  
{

	sword status = OCIStmtFetch2(stmthpToQuery, conn->errhp, (ub4) 1, 
                               OCI_FETCH_LAST, (sb4) 0, OCI_DEFAULT);
	//podemos pegar informacao da ultima linha

	if (status == OCI_SUCCESS || status == OCI_SUCCESS_WITH_INFO)
	{
		last_Row = true;
		return true;
	}
	else
		return false;

}

bool OCICursor::MoveTo(int pos) //begin in 0
{
	
	sword status;
	int auxPos = MAX_ROWS*int(pos/MAX_ROWS);

	// row_Cur = absolute current row 
	// row_Index = relative current row - client side
	if(int(row_Cur/MAX_ROWS) == int(pos/MAX_ROWS)) 
	{
		row_Index = pos - (MAX_ROWS*int(pos/MAX_ROWS));
		row_Cur = pos;
		return true;
	}

	if(pos<row_Cur)
	{
		if(!MoveFirst())
			return false;
	}

	status = OCIStmtFetch2(stmthpToQuery, conn->errhp, (ub4) MAX_ROWS, OCI_FETCH_ABSOLUTE, (sb4)auxPos+1, OCI_DEFAULT); 
	row_Index = pos-auxPos;
	row_Cur = pos;

	CheckError(status);

	if (status == OCI_SUCCESS || status == OCI_SUCCESS_WITH_INFO || status == OCI_NO_DATA)
	{
		OCIAttrGet((dvoid *)stmthpToQuery, (ub4)OCI_HTYPE_STMT,
				 (dvoid *)&rows_Fetched, (ub4 *)0, 
				 (ub4)OCI_ATTR_ROWS_FETCHED, conn->errhp);

		if(!rows_Fetched)
			return false;
		
		if(status == OCI_NO_DATA)
			last_Row = true;
		return true;
	}

	return false;

	/*  // funciona mas fica lento
	sword status;
	// this part is for scroolable cursor
	if(pos >= 0)
	{
		status = OCIStmtFetch2(stmthpToQuery, conn->errhp, (ub4) MAX_ROWS, 
                      OCI_FETCH_ABSOLUTE, (sb4) pos+1, OCI_DEFAULT);
	}
	else
	{
		status = OCIStmtFetch2(stmthpToQuery, conn->errhp, (ub4) MAX_ROWS, 
                      OCI_FETCH_RELATIVE, (sb4) pos+1, OCI_DEFAULT);
	}

	*/
}

// Throw CCursorException if OCI error found
bool OCICursor::CheckError(sword status)
{
	sb4 errcode = 0;
	char message[256];
	bool returnedVal = false;

	if (status == OCI_ERROR)
	{
		SDO_OCIErrorGet((dvoid*)conn->errhp, (ub4)1, (text*)NULL, &errcode, 
						(text*)message, (ub4)256, OCI_HTYPE_ERROR);

		errorMessage = message;
		return false;
	}
	
	switch (status)
	{ 
		case OCI_SUCCESS:
			errorMessage = "Success!";
			returnedVal = true;
			break;

		case OCI_SUCCESS_WITH_INFO:
			errorMessage = "Success with information!";
			returnedVal = true;
			break;

		case OCI_NEED_DATA:
			errorMessage = "Need data!";
			break;
		
		case OCI_NO_DATA:
			errorMessage = "No data!";
			break;

		//An invalid handle was passed as a parameter or a user callback is passed an
		//invalid handle or invalid context. No further diagnostics are available.
		case OCI_INVALID_HANDLE:
			errorMessage = "Invalid handle!";
			break;

		case OCI_STILL_EXECUTING:
			errorMessage = "Still executing!";
			break;

		case OCI_CONTINUE:
			errorMessage = "Continue!";
			break;
		default:
			break;
	}

	return returnedVal;

}


bool OCICursor::Prepare(const string& stmt)
{	
	ub4 size = stmt.size();
	sword status = SDO_OCIStmtPrepare(conn->svchp, (OCIStmt *)stmthpToQuery, conn->errhp, (text*)stmt.c_str(), (ub4)size,  
		(text*)0, (ub4)0, (ub4)OCI_NTV_SYNTAX, (ub4)OCI_DEFAULT);
	if(!CheckError(status))
		return false;

	return true;
}


int OCICursor::QueryType()  
{	
	ub2 querytype_;
		
	sword status = OCIAttrGet((dvoid *)stmthpToQuery, (ub4)OCI_HTYPE_STMT, (ub2 *)&querytype_,
	(ub4*)NULL, (ub4)OCI_ATTR_STMT_TYPE, (OCIError *)conn->errhp);
	if(!CheckError(status))
		return -1;
	return (querytype_);
}

bool OCICursor::Query(const string& query)
{
	if (!isOpen_)
		Open();
			
	row_Index = -1;
	rows_Fetched = 0;
	rows_Mem = 0;
	row_Cur = -1;
	last_Row = false;

	if(!Prepare(query))
		return false;	
	
	// query type equal to 1 = OCI_STMT_SELECT
	if(QueryType()!=1) 
		return false;
		
	//iters equal to zero because the defines (OCIDefines) have not located yet 
	sword status = OCIStmtExecute(conn->svchp, stmthpToQuery, conn->errhp, (ub4)0, (ub4)0, (OCISnapshot *)NULL, 
								  (OCISnapshot *)NULL, OCI_STMT_SCROLLABLE_READONLY);
	if (!CheckError(status))
		return false;

	loadCollDescription(); // load columns description
	if(!AllocateCursor())
		return false;
	
	return true; 
}


bool OCICursor::QuerySDO(const string& query)
{
	if (!isOpen_)
		Open();

	row_Index = -1;
	rows_Fetched = 0;
	rows_Mem = 0;
	row_Cur = -1;
	last_Row = false;

	if(!Prepare(query))
		return false;

	if(!BindOrdinates())
		return false;	

	// query type equal to 1 = OCI_STMT_SELECT
	if (QueryType() != 1) //must be executed by OCIConnection->execute()
		return false;
		
	//iters equal to zero because the defines (OCIDefines) have not located yet 
	sword status = OCIStmtExecute(conn->svchp, stmthpToQuery, conn->errhp, (ub4)0, (ub4)0, (OCISnapshot *)NULL, 
								  (OCISnapshot *)NULL, OCI_STMT_SCROLLABLE_READONLY);
	
	if (!CheckError(status))
		return false;
	
	loadCollDescription(); // load columns description
	if(!AllocateCursor())
		return false;

	return true; 
}


int OCICursor::NumCol()  
{	
	if(numColls_>=0)
		return numColls_;
	
	int numcols_;

	/* Get the number of columns in the query */
	CheckError(OCIAttrGet(stmthpToQuery, OCI_HTYPE_STMT, &numcols_,
	0, OCI_ATTR_PARAM_COUNT, conn->errhp));
	return (numcols_);
}


int OCICursor::NumRows()
{
	int numrows_ = 0;
	
	/* Get the number of rows in the query */
	CheckError(SDO_OCIAttrGet((dvoid*)stmthpToQuery, (ub4)OCI_HTYPE_STMT, (dvoid*)&numrows_,
	(ub4*)0, (ub4)OCI_ATTR_ROW_COUNT, (OCIError*)conn->errhp));
	return (numrows_);

}


void OCICursor::loadCollDescription() 
{
	OCIParam* colhd=NULL;
	//int		dtype;
	ub2	dtype = 0;
	string	colname;
	//int		colsize;
	ub2  colsize = (ub2) 0;
	//int		colscale;
	sb1  colscale = (sb1) 0;
	
	numColls_ = NumCol();

	int i;
	for(i=1; i<=numColls_; ++i)
	{

		// get parameter for i-th column
		CheckError(SDO_OCIParamGet((dvoid*)stmthpToQuery, (ub4)OCI_HTYPE_STMT, (OCIError *)conn->errhp, (dvoid**)&colhd, (ub4)i));

		// get data type 
		CheckError(SDO_OCIAttrGet((dvoid *)colhd, (ub4)OCI_DTYPE_PARAM, (dvoid *)&dtype, (ub4*)0, (ub4)OCI_ATTR_DATA_TYPE, (OCIError *)conn->errhp));

		// get coll name 
		text *colname_ = NULL;
		ub4	colnamesz_;
				
		CheckError(OCIAttrGet((dvoid *)colhd, (ub4)OCI_DTYPE_PARAM, (dvoid **)&colname_, (ub4*)&colnamesz_, (ub4)OCI_ATTR_NAME, (OCIError *)conn->errhp ));

		char temp[100];
		for(i=0;i<(int)colnamesz_;i++)
			temp[i] = colname_[i];
		temp[colnamesz_] = '\0';
		colname = temp;

		// retrieve the column size attribute
		CheckError(SDO_OCIAttrGet((dvoid *)colhd, (ub4)OCI_DTYPE_PARAM, (dvoid *)&colsize, (ub4*)0, (ub4)OCI_ATTR_DATA_SIZE, (OCIError *)conn->errhp ));

		// retrieve the column scale attribute
		CheckError(SDO_OCIAttrGet((dvoid *)colhd, (ub4)OCI_DTYPE_PARAM, (dvoid *)&colscale, (ub4*)0, (ub4)OCI_ATTR_SCALE, (OCIError *)conn->errhp ));

		colName_.push_back(colname);
		colType_.push_back((int)dtype);
		colSize_.push_back((int)colsize);
		colScale_.push_back((int)colscale);
	}
}


int OCICursor::ColType (int colnumber)   
{
	//first coll number is 1
	if((colnumber==0) || (colnumber>(int)colType_.size()))
		return 0;

	return colType_[colnumber-1];
}


string OCICursor::ColName (int colnumber) 
{	
	//first coll number is 1
	if((colnumber==0) || (colnumber>(int)colName_.size()))
		return "";

	return colName_[colnumber-1];
}


int OCICursor::ColSize (int colnumber) 
{
	//first coll number is 1
	if((colnumber==0) || (colnumber>(int)colSize_.size()))
		return 0;

	return colSize_[colnumber-1];
}


int OCICursor::ColScale (int colnumber) 
{
	//first coll number is 1
	if((colnumber==0) || (colnumber>(int)colScale_.size()))
		return 0;

	return colScale_[colnumber-1];
}


 bool OCICursor::AllocateCursor()
 {
	
	int			size;
	int			coltype=0;
	int			colsize=0;
	int			colscale=0;
		
	try
	{
		ind.resize(numColls_);

		for(int nc=0; nc<numColls_; ++nc)
		{
			defines.push_back (0);
			buffers.push_back (0);
		}

		for(int i=1; i<=numColls_;i++)
		{
			coltype = colType_[i-1];
			colsize = colSize_[i-1];
			colscale = colScale_[i-1];

			switch (coltype)
			{
				case 3: //INTEGER
			
					buffers[i-1] = new signed int[MAX_ROWS];
					colsize = sizeof(signed int);

					if(!CheckError(SDO_OCIDefineByPos(stmthpToQuery, &(defines[i-1]), conn->errhp, (ub4)i, 
							(dvoid *)(signed int*)buffers[i-1], (sb4)colsize, coltype, 
							(dvoid *)&ind[i-1], (ub2 *)0, (ub2 *)0,(ub4)OCI_DEFAULT)))
							return false; 

					break;

			
				case 2: //NUMBER
										
					buffers[i-1] = (OCINumber *) new OCINumber[MAX_ROWS]; //();								
					if(!CheckError(OCIDefineByPos(stmthpToQuery, &(defines[i-1]), conn->errhp, (ub4)i, 
							(dvoid *)buffers[i-1], sizeof(OCINumber), SQLT_VNU, 
							(dvoid *)&ind[i-1], (ub2 *)0, (ub2 *)0,(ub4)OCI_DEFAULT)))
							return false; 
			
					break;
							

				case 4: //FLOAT DOUBLE
					
					buffers[i-1] = new double[MAX_ROWS];
					colsize = sizeof(double);
					coltype = 4;
				
					if(!CheckError(SDO_OCIDefineByPos(stmthpToQuery, &(defines[i-1]), conn->errhp, (ub4)i, 
							(dvoid *)(double*)buffers[i-1], (sb4)colsize, coltype, 
							(dvoid *)&ind[i-1], (ub2 *)0, (ub2 *)0, (ub4)OCI_DEFAULT)))
							return false; 
				
				
					break;

				case 96: //CHAR
				case 9: //VARCHAR:
				case 1: //VARCHAR2:	
					buffers[i-1] = (char *) new char[MAX_ROWS*(colsize+1)];
			
					if(!CheckError(SDO_OCIDefineByPos(stmthpToQuery, &(defines[i-1]), conn->errhp, (ub4)i, 
							(dvoid *)buffers[i-1], (sb4)(colsize+1), SQLT_STR, 
							(dvoid *)&ind[i-1], (ub2 *)0, (ub2 *)0,(ub4)OCI_DEFAULT)))
							return false; 
				
					break;

				case 12: //Date
					buffers[i-1] = (OCIDate *) new OCIDate[MAX_ROWS]; //();								
					if(!CheckError(OCIDefineByPos(stmthpToQuery, &(defines[i-1]), conn->errhp, (ub4)i, 
							(dvoid *)buffers[i-1], sizeof(OCIDate), SQLT_ODT, 
							(dvoid *)&ind[i-1], (ub2 *)0, (ub2 *)0,(ub4)OCI_DEFAULT)))
							return false; 
				
					break;


				case 108: //OBJECT SDO_GEOMETRY
				
					if(!CheckError(SDO_OCIObjectNew(conn->envhp, conn->errhp, conn->svchp,
							OCI_TYPECODE_OBJECT, conn->tdo_geometry, (dvoid*)NULL, 
							OCI_DURATION_SESSION, TRUE, 
							(dvoid**)global_geom_obj)))
							return false;
					
					defines[i-1] = NULL;
								
					if(!CheckError(SDO_OCIDefineByPos(stmthpToQuery, &(defines[i-1]), conn->errhp, (ub4)i, 
							(dvoid *)0, (sb4)0, SQLT_NTY, (dvoid *)&ind[i-1],
							(ub2 *)0, (ub2 *)0, (ub4)OCI_DEFAULT)))
							return false;
					
					if(!CheckError(OCIDefineObject(defines[i-1], conn->errhp, conn->tdo_geometry, 
							(dvoid **)global_geom_obj, (ub4 *)0, 
							(dvoid **)global_geom_ind, (ub4 *)0)))
							return false;

					break;

				case 113 :	//SQLT_BLOB

					//  Allocate lob descriptors.
					for(size=0;size<MAX_ROWS;size++)
					{
						if(!CheckError(OCIDescriptorAlloc((dvoid *) conn->envhp, (dvoid **) &lobp[size],
							(ub4) OCI_DTYPE_LOB,
							(size_t) 0, (dvoid **) 0)))
							return false;
					}
					if(!CheckError(OCIDefineByPos (stmthpToQuery, &(defines[i-1]), conn->errhp, (ub4)i,
							(dvoid *)lobp, 0 , SQLT_BLOB,
							(dvoid *)&ind[i-1], (ub2 *)0, (ub2 *)0, OCI_DEFAULT)))
							
							return false;
					break;

				default:
					break;
			
			} //switch
	
		} //for
	}//try

	catch(...) 
	{
		return false;
	}

	return true;
}	

char* OCICursor::GetFieldValue(int i)  //inicia em 1
{ 
	
	int				tempInt;
	double			tempDouble;
	char			str[30];
	unsigned int	size; 
	string			tempDate;
		
	if(i < 1)
		return (char*)0;

	int coltype = colType_[i-1]; 
	int colsize = colSize_[i-1]; 
	
	//OBS:	When the SQL use grouping functions (MAX, MIN, etc), the returned value always has
	//		colscale equal to zero, even when it is double
//	int colscale = colScale_[i-1]; 

	int indica = ind[i-1].sbind[row_Index];

	switch (coltype)
		{
			case 3: //INTEGER

				if(indica == -1)
					return "0";

				tempInt = *((int*)buffers[i-1]+row_Index);
				fieldValue = Te2String(tempInt);
				return ((char*)fieldValue.c_str());
				break;

			case 2: //NUMBER
				
				if(indica == -1)
					return "0";
				
				if(!CheckError(OCINumberToReal(conn->errhp, ((OCINumber *)buffers[i-1]+row_Index), 
					(uword)sizeof(double), (dvoid *)&tempDouble)))
					return "0";
				
				fieldValue = Te2String(tempDouble);
				return ((char*)fieldValue.c_str());
				break;
				

			case 4: //FLOAT DOUBLE
					
				if(indica == -1)
					return "0";
				
				tempDouble = *((const double*)buffers[i-1]+row_Index);
				fieldValue = Te2String(tempDouble);
				return ((char*)fieldValue.c_str());
				break;

			case 96: //CHAR
			case 9: //VARCHAR:
			case 1: //VARCHAR2:
				
				if(indica == -1)
					return "";

				fieldValue = ((char*)buffers[i-1]+((colsize+1)*row_Index));
				fieldValue = StrClean((char*)fieldValue.c_str());
				return ((char*)fieldValue.c_str());
				break;

			case 12: //Date
				
				if(indica == -1)
					return "";

				fieldValue = "";
				size = sizeof(OCIDate);
																	
				if(!CheckError(OCIDateToText(conn->errhp, ((OCIDate *)buffers[i-1]+row_Index), 
					NULL, 0, NULL, 0, &size, (unsigned char*)tempDate.c_str())))
					return "";
				memcpy(str, tempDate.c_str(), size);
				strncat((char*)fieldValue.c_str(), str, size);

				return ((char*)fieldValue.c_str());
				break;

			case 108: //OBJECT SDO_GEOMETRY
			default:
				break;
								
		} //switch
		
	return (char*)0;
}

int OCICursor::GetDimArraySize()
{
	int ndim=-1;

	// Get the size of the elem info array
    CheckError(OCICollSize(conn->envhp, conn->errhp, 
				(OCIColl *)(global_geom_obj[row_Index]->sdo_elem_info), 
				&ndim));

	return(ndim);
}

bool OCICursor::GetDimElement(int i,int &elem)
{
	int				exists;
	OCINumber		*oci_number;
	double			el;


	OCICollGetElem(conn->envhp, conn->errhp, 
	   (OCIColl *)(global_geom_obj[row_Index]->sdo_elem_info), 
	   (sb4)i-1, (int *)&exists, 
	   (dvoid **)&oci_number, (dvoid **)0);

	OCINumberToReal(conn->errhp, oci_number, (uword)sizeof(double),
		(dvoid *)&el);
	
	elem = (int)el;
	return true;
}


int OCICursor::GetNumberOrdinates()
{
	int nOrds=-1;

	/* Get the size of the ordinates array */
    CheckError(OCICollSize(conn->envhp, conn->errhp, 
				(OCIColl *)(global_geom_obj[row_Index]->sdo_ordinates), 
				&nOrds));

	return(nOrds);
}

bool OCICursor::GetCoordinates(int i, TeCoord2D& coord)
{
	int				exists;
	OCINumber		*oci_number;
	double			coor_x;
	double			coor_y;

	int pos = i;

	OCICollGetElem(conn->envhp, conn->errhp, 
	   (OCIColl *)(global_geom_obj[row_Index]->sdo_ordinates), 
	   (sb4)(pos-1), (int *)&exists, 
	   (dvoid **)&oci_number, (dvoid **)0);

	OCINumberToReal(conn->errhp, oci_number, (uword)sizeof(double),
		(dvoid *)&coor_x);
	
	pos++;

	OCICollGetElem(conn->envhp, conn->errhp, 
	   (OCIColl *)(global_geom_obj[row_Index]->sdo_ordinates),  
	   (sb4)(pos-1), (int *)&exists, 
	   (dvoid **)&oci_number, (dvoid **)0);

	OCINumberToReal(conn->errhp, oci_number, (uword)sizeof(double),
		(dvoid *)&coor_y);

	coord.x(coor_x);
	coord.y(coor_y);

	return true;
}


bool OCICursor::GetCoordinates(vector<TeCoord2D>& result)
{
	
	OCIIter		*iterator;
	dvoid		*elem;
	OCIInd		*elemind ;
	double		ordinate1, ordinate2; 
	OCINumber	*aux1;
	OCINumber	*aux2;
	boolean		eoc;

	sword status = OCIIterCreate(conn->envhp, conn->errhp, 
		(OCIArray *)(global_geom_obj[row_Index]->sdo_ordinates), &iterator);
	if (status != OCI_SUCCESS)
		return false;
	
	/* Get the first and second element of the clients varray */
	status = OCIIterNext(conn->envhp, conn->errhp, iterator, &elem,
		(dvoid **) &elemind, &eoc);
	if (status != OCI_SUCCESS)
	{
		OCIIterDelete(conn->envhp, conn->errhp, &iterator);
		return false;
	}

	aux1 = (OCINumber *)elem;
	OCINumberToReal(conn->errhp, (OCINumber *)aux1, (uword)sizeof(double),
					(dvoid *)&ordinate1);

	status = OCIIterNext(conn->envhp, conn->errhp, iterator, &elem,
				(dvoid **)&elemind, &eoc);
	if (status != OCI_SUCCESS)
	{
		OCIIterDelete(conn->envhp, conn->errhp, &iterator);
		return false;
	}

	aux2 = (OCINumber *)elem;
	OCINumberToReal(conn->errhp, (OCINumber *) aux2, (uword)sizeof(double),
					(dvoid *)&ordinate2);

	TeCoord2D coord(ordinate1, ordinate2);
	result.push_back (coord);
	
	while (!eoc && (status == OCI_SUCCESS))
	{
		status = OCIIterNext(conn->envhp, conn->errhp, iterator, &elem,
				(dvoid **)&elemind, &eoc);
		if (status != OCI_SUCCESS)
		{
			OCIIterDelete(conn->envhp, conn->errhp, &iterator);
			return false;
		}

		aux1 = (OCINumber *)elem;
		OCINumberToReal(conn->errhp, (OCINumber *)aux1, (uword)sizeof(double),
					(dvoid *)&ordinate1);

		status = OCIIterNext(conn->envhp, conn->errhp, iterator, &elem,
				(dvoid **)&elemind, &eoc);
		if (status != OCI_SUCCESS)
		{
			OCIIterDelete(conn->envhp, conn->errhp, &iterator);
			return false;
		}

		aux2 = (OCINumber *)elem;
		OCINumberToReal(conn->errhp, (OCINumber *) aux2, (uword)sizeof(double),
					(dvoid *)&ordinate2);

		TeCoord2D coord(ordinate1, ordinate2);
		result.push_back (coord);
	}

	/* destroy the iterator */
	status = OCIIterDelete(conn->envhp, conn->errhp, &iterator);
	return true;
}


int
OCICursor::GetGeometryType()
{
	int gtype=-1;

	CheckError(OCINumberToInt(conn->errhp, &(global_geom_obj[row_Index]->sdo_gtype),
				 (uword)sizeof(int), OCI_NUMBER_SIGNED,
				 (dvoid *)&gtype));

	return gtype;
}

int
OCICursor::GetSpatialReferenceId()
{
	int srid=-1;

	CheckError(OCINumberToInt(conn->errhp, &(global_geom_obj[row_Index]->sdo_srid),
				 (uword)sizeof(int), OCI_NUMBER_SIGNED,
				 (dvoid *)&srid));

	return srid;
}

bool
OCICursor::GetXYZcoord(double& x, double& y)
{
	if (global_geom_ind[row_Index]->sdo_point._atomic == OCI_IND_NOTNULL)
	{
		if (global_geom_ind[row_Index]->sdo_point.x == OCI_IND_NOTNULL)
		{
			if(!CheckError(OCINumberToReal(conn->errhp, &(global_geom_obj[row_Index]->sdo_point.x), 
										 (uword)sizeof(double),(dvoid *)&x)))

				return false;
		}

		if (global_geom_ind[row_Index]->sdo_point.y == OCI_IND_NOTNULL)
		{
			if(!CheckError(OCINumberToReal(conn->errhp, &(global_geom_obj[row_Index]->sdo_point.y), 
										 (uword)sizeof(double),(dvoid *)&y)))
				return false;
		}

		/*
		if (z && (global_geom_ind[row_Index]->sdo_point.z == OCI_IND_NOTNULL))
		{
			if(!CheckError(OCINumberToReal(conn->errhp, &(global_geom_obj[row_Index]->sdo_point.z), 
										 (uword)sizeof(double),(dvoid *)&z)))
				return false;
		}*/

		return true;
	}
	else
		return false;
}


bool
OCICursor::WriteBlob(const unsigned char* buffer, unsigned int bufsize)
{
	
	int		offset = 1;
	sword	status;
	
	status = SDO_OCILobWrite(conn->svchp, conn->errhp, *lobp, &bufsize, offset,
        (dvoid *) buffer, (ub4) bufsize, OCI_ONE_PIECE,
        (dvoid *)0, (sb4 (*)(dvoid *, dvoid *, ub4 *, ub1 *)) 0,
		(ub2) 0, (ub1) SQLCS_IMPLICIT);
	
	CheckError(status);
	if (status == OCI_SUCCESS)
		return true;

	return false;
}

int 
OCICursor::SizeBlob()
{
	ub4 lenp=0;
	CheckError(OCILobGetLength(conn->svchp, conn->errhp, lobp[row_Index], &lenp));
	return lenp;
}


bool
OCICursor::ReadBlob(unsigned char* buffer, unsigned int bufsize)
{
	sword	status;
	int		offset = 1;
			
	status = SDO_OCILobRead(conn->svchp, conn->errhp, lobp[row_Index], &bufsize, offset,
              (dvoid *) buffer, (ub4)bufsize , (dvoid *) 0, 0, (ub2) 0, 
			   (ub1) SQLCS_IMPLICIT);
    
	if (status == OCI_SUCCESS )
      return true;

	return false;
}



