#ifndef TAGCOLL_REVERSER_H
#define TAGCOLL_REVERSER_H

/*
 * "Reverse" a tagged collection, associating items to tags instead of tags to
 * items
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma interface

#include <TagcollConsumer.h>

#include <string>
#include <map>

// TagcollConsumer that builds a tagged collection 
template<class ITEM = std::string, class TAG = std::string>
class TagcollReverser : public TagcollConsumer<ITEM, TAG>
{
protected:
	std::map< TAG, OpSet<ITEM> > tagged;
	OpSet<ITEM> untagged;
	ITEM untagged_name;

public:
	virtual ~TagcollReverser() throw () {}

	// Set the item name to use when reversing untagged items.  If not set,
	// untagged items won't be included in the reversed collection
	void setUntaggedItemName(const ITEM& item) throw ()
	{
		untagged_name = item;
	}

	virtual void consume(const ITEM& item) throw ()
	{
		untagged += item;
	}
	
	virtual void consume(const ITEM& item, const OpSet<TAG>& tags) throw ()
	{
		for (typename OpSet<ITEM>::const_iterator i = tags.begin();
				i != tags.end(); i++)
			tagged[*i] += item;
	}

	// Send the reversed data to a consumer
	void output(TagcollConsumer<TAG, ITEM>& consumer) throw ()
	{
		if (untagged_name.size() > 0)
			if (untagged.empty())
				consumer.consume(untagged_name);
			else
				consumer.consume(untagged_name, untagged);
		
		for (typename std::map< TAG, OpSet<ITEM> >::const_iterator i = tagged.begin();
				i != tagged.end(); i++)
			if (i->second.empty())
				consumer.consume(i->first);
			else
				consumer.consume(i->first, i->second);
	}
};

// vim:set ts=4 sw=4:
#endif
