/*
 * Merge tags of items appearing multiple times in a stream of tagged items
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma implementation

#include "InputMerger.h"

using namespace std;

template class InputMerger<string>;
template class InputMerger<int>;

namespace Debtags { class Package; }
template class InputMerger<Debtags::Package*>;

template<class T>
void InputMerger<T>::consume(const T& item) throw ()
{
	typename map< T, OpSet<string> >::iterator i = coll.find(item);
	if (i == coll.end())
		coll.insert(make_pair(item, OpSet<string>()));
}

template<class T>
void InputMerger<T>::consume(const T& item, const OpSet<string>& tags) throw ()
{
	typename map< T, OpSet<string> >::iterator i = coll.find(item);
	if (i == coll.end())
		coll.insert(make_pair(item, tags));
	else
		i->second += tags;
}

template<class T>
OpSet<string> InputMerger<T>::getTagsetForItem(const T& item) const throw ()
{
	typename map< T, OpSet<string> >::const_iterator i = coll.find(item);
	
	if (i == coll.end())
		return OpSet<string>();
	else
		return i->second;
}
	

template<class T>
void InputMerger<T>::output(TagcollConsumer<T>& consumer) throw ()
{
	for (typename map< T, OpSet<string> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		if (i->second.size() == 0)
			consumer.consume(i->first);
		else
			consumer.consume(i->first, i->second);
}


template<class T>
TagcollChange<T, std::string> InputMerger<T>::applyChange(
		const TagcollChange<T, std::string>& change) throw ()
{
	TagcollChange<T, std::string> rev;

	for (typename TagcollChange<T, std::string>::const_iterator i = change.begin(); i != change.end(); i++)
	{
		typename map< T, OpSet<string> >::iterator it = coll.find(i->first);
		rev.insert(make_pair(it->first, it->second));
		it->second = i->second;
	}

	return rev;
}

template<class T>
OpSet<std::string> InputMerger<T>::getAllTags() const throw ()
{
	OpSet<std::string> tags;

	for (typename map< T, OpSet<string> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		tags += i->second;
	
	return tags;
}

template<class T>
OpSet<std::string> InputMerger<T>::getCompanionTags(const OpSet<std::string>& ts) const throw ()
{
	OpSet<std::string> tags;

	for (typename map< T, OpSet<string> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		if (i->second.contains(ts))
			tags += i->second - ts;
	
	return tags;
}

template<class T>
OpSet<T> InputMerger<T>::getRelatedItems(const T& item, int maxdistance) const throw ()
{
	OpSet<T> res;

	typename map< T, OpSet<string> >::const_iterator base = coll.find(item);
	if (base == coll.end())
		return res;
	OpSet<std::string> tags = base->second;

	for (typename map< T, OpSet<string> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
	{
		if (i == base)
			continue;
		int dist = tags.distance(i->second);
		if (dist >= 0 && dist <= maxdistance)
			res += i->first;
	}
	
	return res;
}

template<class T>
OpSet<T> InputMerger<T>::getItemsContaining(const OpSet<std::string>& ts) const throw ()
{
	OpSet<T> res;

	for (typename map< T, OpSet<string> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		if (i->second.contains(ts))
			res += i->first;
	
	return res;
}


// vim:set ts=4 sw=4:
