/*
 * Collection of tag implications and a a TagcollFilter to apply or compress
 * them
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma implementation

#include "Implications.h"

#include <algorithm>

using namespace std;

//#define VERB_PACK 1
//#define VERB_DEST 1
//#define VERB_CONS 1

#if VERB_PACK || VERB_DEST || VERB_CONS
#include <stdio.h>
#include "stringf.h"
using namespace stringf;

void print_tagset(FILE* out, const OpSet<string>& tags)
{
	for (OpSet<string>::const_iterator i = tags.begin();
			i != tags.end(); i++)
		if (i == tags.begin())
			fprintf(out, "%.*s", PFSTR(*i));
		else
			fprintf(out, ", %.*s", PFSTR(*i));
}

void indent(FILE* out, const OpSet<string>& tags)
{
	fprintf(out, "%*s", tags.size(), "");
}
#endif


OpSet<string> ImplicationList::getDestinations(const string& tag, const OpSet<string>& seen) const throw ()
{
	impl_t::const_iterator i = implications.find(tag);
	if (i == implications.end())
		return OpSet<string>();

	// res <- the union of all the destinations of the tag
	OpSet<string> res;
	for (OpSet<string>::const_iterator t = i->second.begin();
			t != i->second.end(); t++)
	{
		// If the tag is already in res, then also all his destinations are
		// already there
		if (!res.contains(*t) && !seen.contains(*t))
		{
			res += *t;
			res += getDestinations(*t, seen + tag);
		}
	}

#ifdef VERB_DEST
	indent(stderr, seen);
	fprintf(stderr, "%.*s dests: ", PFSTR(tag));
	print_tagset(stderr, res);
	fprintf(stderr, "\n");
#endif

	return res;
}

bool ImplicationList::reaches(const string& tag1, const string& tag2, const OpSet<string>& seen) const throw ()
{
	// Check if we've reached the target
	if (tag1 == tag2)
		return true;
	
	// No: see if we have other paths to follow
	impl_t::const_iterator i = implications.find(tag1);
	if (i == implications.end())
		return false;

	// Try all paths
	for (OpSet<string>::const_iterator t = i->second.begin();
			t != i->second.end(); t++)
		if (!seen.contains(*t) && reaches(*t, tag2, seen + tag1))
			return true;

	// Nothing has been found
	return false;
}


void ImplicationList::consume(const string& item, const OpSet<string>& tags) throw ()
{
#ifdef VERB_CONS
	fprintf(stderr, "New implication: %.*s -> ", PFSTR(item));
	print_tagset(stderr, tags);
	fprintf(stderr, "\n");
#endif

	implications.insert(make_pair(item, tags));
}

// Expand a single tag
OpSet<string> ImplicationList::expand(const string& tag) throw ()
{
	return getDestinations(tag) + tag;
}

// Expand a full tagset
OpSet<string> ImplicationList::expand(const OpSet<string>& tags) throw ()
{
	OpSet<string> res = tags;
	
	for (OpSet<string>::const_iterator t = tags.begin();
			t != tags.end(); t++)
		res += expand(*t);

	return res;
}

// Remove unnecessary arcs from the dag
void ImplicationList::pack() throw ()
{
#ifdef VERB_PACK
	fprintf(stderr, "ImplicationList::pack\n");
#endif
	// For every tag
	for (map< string, OpSet<string> >::iterator i = implications.begin();
			i != implications.end(); i++)
	{
		OpSet<string> redundant;

		// For every couple of parents A and B, if A -> B but not B -> A, then B is redundant
		// I need to check every combination; however, I can ignore in the
		// search items that have already been found as redundant

		OpSet<string> candidates = i->second;
#ifdef VERB_PACK
		fprintf(stderr, "  %.*s candidates:", PFSTR(i->first));
		print_tagset(stderr, i->second);
		fprintf(stderr, "\n");
#endif
		while (candidates.size() > 1)
		{
			OpSet<string>::const_iterator a = candidates.begin();
			OpSet<string>::const_iterator b = a;
			OpSet<string> got;
			for (++b; b != candidates.end(); b++)
			{
				bool ab = reaches(*a, *b);
				bool ba = reaches(*b, *a);
#ifdef VERB_PACK
				fprintf(stderr, "    checking %.*s and %.*s (%s%s)\n", PFSTR(*a), PFSTR(*b),
						ab ? " -> " : "", ba ? " <- " : "");
#endif
				if (ab && !ba)
				{
#ifdef VERB_PACK
					fprintf(stderr, "      %.*s is redundant\n", PFSTR(*b));
#endif
					got += *b;
					break;
				} else if (ba && !ab) {
#ifdef VERB_PACK
					fprintf(stderr, "    %.*s is redundant\n", PFSTR(*a));
#endif
					got += *a;
					break;
				}
			}
			candidates -= got;
			redundant += got;
			if (!candidates.empty())
				candidates.erase(candidates.begin());
		}

		i->second -= redundant;
#ifdef VERB_PACK
		fprintf(stderr, "   %.*s result:", PFSTR(i->first));
		print_tagset(stderr, i->second);
		fprintf(stderr, "\n");
#endif
	}
}


// Output the fully expanded implication dag to a TagcollConsumer
void ImplicationList::outputFull(TagcollConsumer<std::string>& consumer) throw ()
{
	for (map<string, OpSet<string> >::const_iterator i = implications.begin();
			i != implications.end(); i++)
	{
		OpSet<string> destinations = getDestinations(i->first);

		if (destinations.empty())
			consumer.consume(i->first);
		else
			consumer.consume(i->first, destinations);
	}
}

// Output the implication dag to a TagcollConsumer
void ImplicationList::output(TagcollConsumer<std::string>& consumer) throw ()
{
	for (map<string, OpSet<string> >::const_iterator i = implications.begin();
			i != implications.end(); i++)
		if (i->second.empty())
			consumer.consume(i->first);
		else
			consumer.consume(i->first, i->second);
}

void CompressImplications::consume(const string& item, const OpSet<string>& tags) throw ()
{
	// Create the union of the expansion sets of each single tag, without the tag
	// tags = tags - this union
	OpSet<string> redundant;
	for (OpSet<string>::const_iterator t = tags.begin();
			t != tags.end(); t++)
	{
		OpSet<string> expanded = implications.expand(*t);
		for (OpSet<string>::const_iterator i = expanded.begin();
				i != expanded.end(); i++)
			if (*i != *t)
				redundant.insert(*i);
	}
	
	OpSet<string> compressed;
	
	set_difference(tags.begin(), tags.end(),
					redundant.begin(), redundant.end(),
					inserter(compressed, compressed.begin()));
	
	consumer->consume(item, compressed);
}


// vim:set ts=4 sw=4:
