/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef SA_DATA_STORAGE
#warning You should not include this file directly from your program.
#endif

/**
 * \defgroup data_storage Data storage device interface
 * @{
 */
#ifndef SA_COMMON_DATA_STORAGE_H_
#define SA_COMMON_DATA_STORAGE_H_

/** \struct sa_data_storage data_storage.h
 * This structure gathers the details about the activity of one data storage device.
 */
SA_EXPORT struct sa_data_storage {
#ifdef SA_DATA_STORAGE_NAME
	char name[SA_DATA_STORAGE_NAME]; //!< Device's name. It's used as the unique identifier of the device.
#endif
#ifdef SA_DATA_STORAGE_READS
	uint64_t reads; //!< Number of reads.
#endif
#ifdef SA_DATA_STORAGE_READS_MERGED
	uint64_t reads_merged; //!< Number of reads which are adjacent to each other and have been merged for efficiency
#endif
#ifdef SA_DATA_STORAGE_SECTORS_READ
	uint64_t sectors_read; //!< Total number of sectors read
#endif
#ifdef SA_DATA_STORAGE_TIME_SPENT_READING
	uint64_t time_spent_reading; //!< Number of seconds spent performing read operations
#endif
#ifdef SA_DATA_STORAGE_WRITES
	uint64_t writes; //!< Number of writes.
#endif
#ifdef SA_DATA_STORAGE_SECTORS_WRITTEN
	uint64_t sectors_written; //!< Total number of sectors written
#endif
#ifdef SA_DATA_STORAGE_TIME_SPENT_WRITING
	uint64_t time_spent_writing; //!< Number of seconds spent performing write operations
#endif
#ifdef SA_DATA_STORAGE_BYTES_READ
	uint64_t bytes_read; //!< Total number of bytes read
#endif
#ifdef SA_DATA_STORAGE_BYTES_WRITTEN
	uint64_t bytes_written; //!< Total number of bytes written
#endif
};

#ifdef SA_OPEN_DATA_STORAGE
/**
 * Prepares the resources needed for retrieving data storage statistics. This function exists (and is needed) only when SA_OPEN_DATA_STORAGE is defined.
 * @return If successful 0 is returned, otherwise an error code is returned. If the operative system is not supported the return value will be ENOTSUP.
 * @see sa_close_data_storage()
 */
SA_EXPORT int sa_open_data_storage(void);
#endif

#ifdef SA_CLOSE_DATA_STORAGE
/**
 * This function closes the resources used for retrieving data storage statistics. You should call it even when there was a previous error in another function of this API. This function exists (and is needed) only when SA_CLOSE_DATA_STORAGE is defined.
 * @return If successful 0 is returned, otherwise an error code is returned.
 * @see sa_open_data_storage()
 */
SA_EXPORT int sa_close_data_storage(void);
#endif

/**
 * Gives the total number of data storage devices.
 * @param number The number will be stored here
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_count_data_storages(uint16_t* number);

/**
 * Retrieves statistics from a data storage device identified by its name.
 * @param name The name of the data storage device.
 * @param dst Where the statistics will be stored.
 * @return If successful 0 is returned, otherwise an error code is returned. ENODEV is returned when the requested device was not found.
 */
SA_EXPORT int sa_get_data_storage(char* name, struct sa_data_storage* dst);

/**
 * Retrieves statistics about all the storage devices' activity.
 * @param dst A buffer where the statistics will be stored.
 * @param dst_size The number of devices that fits in the dst buffer. If it's not big enough dst will be filled but ENOMEM will be returned.
 * @param written The amount of device statistics written.
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_get_data_storages(struct sa_data_storage* dst, uint16_t dst_size, uint16_t* written);

/*@}*/
#endif /* SA_COMMON_DATA_STORAGE_H_ */
