/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

#ifdef ENABLE_WBXML
#include <libsyncml/parser/sml_wbxml_internals.h>
#endif

#include <libsyncml/sml_command_internals.h>
#include <libsyncml/sml_elements_internals.h>

static SmlParser *start_wbxml_parser(const char *data)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlParser *parser = smlParserNew(SML_MIMETYPE_WBXML, 0, &error);
	fail_unless(parser != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	
#ifdef ENABLE_WBXML
	char *buffer = NULL;
	unsigned int size = 0;
	fail_unless(smlWbxmlConvertTo(NULL, data, &buffer, &size, &error), NULL);

	if (!smlParserStart(parser, buffer, size, &error)) {
		fail_unless(error != NULL, NULL);
		
		smlParserFree(parser);
		
		return NULL;
	}
#endif
	
	return parser;
}

START_TEST (check_parser_syncml_empty)
{
	char *data = "<SyncML></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser == NULL, NULL);
}
END_TEST

START_TEST (check_parser_no_syncbody)
{
	char *data = "<SyncML><SyncHdr></SyncHdr></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_no_synchdr)
{
	char *data = "<SyncML><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser == NULL, NULL);
}
END_TEST

START_TEST (check_parser_synchdr_empty)
{
	char *data = "<SyncML><SyncHdr></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_verproto)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_verdtd)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_msgid)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_sessionid)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_target)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_source)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	fail_unless(header->target != NULL, NULL);
	fail_unless(header->source != NULL, NULL);
	fail_unless(header->protocol == SML_PROTOCOL_SYNCML, NULL);
	fail_unless(header->version == SML_VERSION_11, NULL);
	fail_unless(!strcmp(header->sessionID, "1"), NULL);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_source2)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>asd</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	fail_unless(header->target != NULL, NULL);
	fail_unless(header->source != NULL, NULL);
	fail_unless(header->protocol == SML_PROTOCOL_SYNCML, NULL);
	fail_unless(header->version == SML_VERSION_11, NULL);
	fail_unless(!strcmp(header->sessionID, "asd"), NULL);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_synchdr_additional)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source><test></test></SyncHdr><SyncBody></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(!smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header == NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(!smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status_empty)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status></Status></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(!smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status_cmdid)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID></Status></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(!smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);

	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status_msgref)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef></Status></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(!smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status_cmdref)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>1</CmdRef></Status></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(!smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserEnd(parser, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_status2)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = TRUE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == FALSE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_final)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = FALSE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == TRUE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_final2)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Status><CmdID>1</CmdID><MsgRef>1</MsgRef><CmdRef>0</CmdRef><Cmd>Sync</Cmd></Status><Final/></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = FALSE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == TRUE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_cmd)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Alert><CmdID>1</CmdID><Item><Target><LocURI>/test</LocURI></Target><Source><LocURI>/test</LocURI></Source><Meta><Anchor xmlns=\"syncml:metinf\"><Next>Next</Next><Last>last</Last></Anchor></Meta></Item><Data>200</Data></Alert></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 1, NULL);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = TRUE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == FALSE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

START_TEST (check_parser_cmd_final)
{
	char *data = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Alert><CmdID>1</CmdID><Item><Target><LocURI>/test</LocURI></Target><Source><LocURI>/test</LocURI></Source><Meta><Anchor xmlns=\"syncml:metinf\"><Next>Next</Next><Last>last</Last></Anchor></Meta></Item><Data>200</Data></Alert><Final></Final></SyncBody></SyncML>";
	SmlParser *parser = start_wbxml_parser(data);
	fail_unless(parser != NULL, NULL);
	
	SmlError *error = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	
	fail_unless(smlParserGetHeader(parser, &header, &cred, &error), NULL);
	fail_unless(header != NULL, NULL);
	fail_unless(cred == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(header->messageID == 1, NULL);
	
	SmlStatus *status = NULL;
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlParserGetStatus(parser, &status, &error), NULL);
	fail_unless(status == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = NULL;
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(cmd->cmdID == 1, NULL);
	
	fail_unless(smlParserGetCommand(parser, &cmd, &error), NULL);
	fail_unless(cmd == NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlBool final = FALSE;
	SmlBool end = TRUE;
	fail_unless(smlParserEnd(parser, &final, &end, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(final == TRUE, NULL);
	fail_unless(end == FALSE, NULL);
	
	smlParserFree(parser);
}
END_TEST

Suite *parser_suite(void)
{
	Suite *s = suite_create("Parser");
	//Suite *s2 = suite_create("Parser");
	
	create_case(s, "check_parser_syncml_empty", check_parser_syncml_empty);
	create_case(s, "check_parser_no_syncbody", check_parser_no_syncbody);
	create_case(s, "check_parser_no_synchdr", check_parser_no_synchdr);
	create_case(s, "check_parser_synchdr_empty", check_parser_synchdr_empty);
	create_case(s, "check_parser_synchdr_verproto", check_parser_synchdr_verproto);
	create_case(s, "check_parser_synchdr_verdtd", check_parser_synchdr_verdtd);
	create_case(s, "check_parser_synchdr_msgid", check_parser_synchdr_msgid);
	create_case(s, "check_parser_synchdr_sessionid", check_parser_synchdr_sessionid);
	create_case(s, "check_parser_synchdr_additional", check_parser_synchdr_additional);
	create_case(s, "check_parser_synchdr_target", check_parser_synchdr_target);
	create_case(s, "check_parser_synchdr_source", check_parser_synchdr_source);
	create_case(s, "check_parser_synchdr_source2", check_parser_synchdr_source2);
	create_case(s, "check_parser_status_empty", check_parser_status_empty);
	create_case(s, "check_parser_status_cmdid", check_parser_status_cmdid);
	create_case(s, "check_parser_status_msgref", check_parser_status_msgref);
	create_case(s, "check_parser_status_cmdref", check_parser_status_cmdref);
	create_case(s, "check_parser_status", check_parser_status);
	create_case(s, "check_parser_status2", check_parser_status2);
	create_case(s, "check_parser_final", check_parser_final);
	create_case(s, "check_parser_final2", check_parser_final2);
	create_case(s, "check_parser_cmd", check_parser_cmd);
	create_case(s, "check_parser_cmd_final", check_parser_cmd_final);
	
	return s;
}

int main(void)
{
	int nf;

	Suite *s = parser_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_VERBOSE);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
