/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

#include <libsyncml/sml_session_internals.h>

#include <libsyncml/transports/http_client.h>
#include <libsyncml/transports/http_server.h>

#define NUM_SESSIONS 30

typedef struct managerTracker {
	SmlManager *manager;
	SmlSession *sessions[NUM_SESSIONS];
	int num_sessions;
} managerTracker;

START_TEST (manager_new)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	
	SmlManager *manager = smlManagerNew(server, &error);
	fail_unless(manager != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlManagerFree(manager);
	
	smlTransportFree(server);
}
END_TEST

START_TEST (manager_run)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	
	SmlManager *manager = smlManagerNew(server, &error);
	fail_unless(manager != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlManagerStart(manager, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlManagerStop(manager);
	
	smlManagerFree(manager);
	
	smlTransportFree(server);
}
END_TEST

unsigned int transport_errors = 0;
unsigned int num_sessions = 0;
unsigned int num_finals = 0;
unsigned int num_end = 0;
unsigned int session_errors = 0;

unsigned int num_replies = 0;
unsigned int num_alerts = 0;
unsigned int num_syncs = 0;
unsigned int num_changes = 0;

static void _manager_event(SmlManager *manager, SmlManagerEventType type, SmlSession *session, SmlError *error, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p, %p)", __func__, manager, type, session, error, userdata);
	managerTracker *tracker = userdata;
	smlAssert(manager);
	smlAssert(userdata);
	
	switch (type) {
		case SML_MANAGER_SESSION_FLUSH:
			break;
		case SML_MANAGER_CONNECT_DONE:
		case SML_MANAGER_DISCONNECT_DONE:
			fail(NULL);
			break;
		case SML_MANAGER_TRANSPORT_ERROR:
			transport_errors++;
			break;
		case SML_MANAGER_SESSION_NEW:
			smlAssert(session);
			tracker->sessions[tracker->num_sessions] = session;
			tracker->num_sessions++;
			num_sessions++;
			smlSessionRef(session);
			break;
		case SML_MANAGER_SESSION_FINAL:
			num_finals++;
			break;
		case SML_MANAGER_SESSION_END:
			num_end++;
			break;
		case SML_MANAGER_SESSION_ERROR:
		case SML_MANAGER_SESSION_WARNING:
			session_errors++;
			break;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

START_TEST (manager_receive)
{
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;
	
	num_replies = 0;
	num_alerts = 0;
	num_syncs = 0;
	num_changes = 0;

	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12000;
	clientConfig.url = "http://127.0.0.1:12000";
	clientConfig.proxy = NULL;
	clientConfig.username = NULL;
	clientConfig.password = NULL;
	clientConfig.cafile = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12000;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	char *datastr = "<SyncML></SyncML>";
	SmlTransportData *data = smlTransportDataNew(datastr, strlen(datastr), SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportDataDeref(data);
	
	while (session_errors != 1) {
		smlManagerDispatch(servermanager);
		usleep(100);
	}
	
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 0, NULL);
	fail_unless(num_finals == 0, NULL);
	fail_unless(num_end == 0, NULL);
	fail_unless(session_errors == 1, NULL);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (manager_receive_session)
{
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;
	
	num_replies = 0;
	num_alerts = 0;
	num_syncs = 0;
	num_changes = 0;
	
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12001;
	clientConfig.url = "http://127.0.0.1:12001";
	clientConfig.proxy = NULL;
	clientConfig.username = NULL;
	clientConfig.password = NULL;
	clientConfig.cafile = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12001;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	char *datastr = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Alert><CmdID>1</CmdID><Item><Target><LocURI>/test</LocURI></Target><Source><LocURI>/test</LocURI></Source><Meta><Anchor xmlns=\"syncml:metinf\"><Next>Next</Next><Last>last</Last></Anchor></Meta></Item><Data>200</Data></Alert><Final></Final></SyncBody></SyncML>";
	SmlTransportData *data = smlTransportDataNew(datastr, strlen(datastr), SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportDataDeref(data);
	
	while (num_finals != 1 || session_errors != 1) {
		smlManagerDispatch(servermanager);
		usleep(100);
	}
	
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 1, NULL);
	fail_unless(num_finals == 1, NULL);
	fail_unless(num_end == 0, NULL);
	fail_unless(session_errors == 1, NULL);
	
	fail_unless(servertracker->sessions[0] != NULL, NULL);
	smlSessionUnref(servertracker->sessions[0]);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (manager_end_session)
{
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;
	
	num_replies = 0;
	num_alerts = 0;
	num_syncs = 0;
	num_changes = 0;
	
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12002;
	clientConfig.url = "http://127.0.0.1:12002";
	clientConfig.proxy = NULL;
	clientConfig.username = NULL;
	clientConfig.password = NULL;
	clientConfig.cafile = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12002;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	char *datastr = "<SyncML><SyncHdr><VerProto>SyncML/1.1</VerProto><VerDTD>1.1</VerDTD><MsgID>1</MsgID><SessionID>1</SessionID><Target><LocURI>test</LocURI></Target><Source><LocURI>test</LocURI></Source></SyncHdr><SyncBody><Alert><CmdID>1</CmdID><Item><Target><LocURI>/test</LocURI></Target><Source><LocURI>/test</LocURI></Source><Meta><Anchor xmlns=\"syncml:metinf\"><Next>test1</Next><Last>test2</Last></Anchor></Meta></Item><Data>200</Data></Alert><Final></Final></SyncBody></SyncML>";
	SmlTransportData *data = smlTransportDataNew(datastr, strlen(datastr), SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportDataDeref(data);
	
	while (num_finals != 1 || session_errors != 1) {
		smlManagerDispatch(servermanager);
		usleep(100);
	}
	
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 1, NULL);
	fail_unless(num_finals == 1, NULL);
	fail_unless(num_end == 0, NULL);
	fail_unless(session_errors == 1, NULL);
	
	fail_unless(servertracker->sessions[0] != NULL, NULL);
	fail_unless(smlSessionEnd(servertracker->sessions[0], &error), NULL);
	fail_unless(error == NULL, NULL);
	smlSessionUnref(servertracker->sessions[0]);
	
	while (num_end != 1) {
		smlManagerDispatch(servermanager);
		usleep(100);
	}
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

static void _recv_alert(SmlSession *session, SmlCommand *cmd, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, cmd, userdata);
	SmlError *error = NULL;
	
	num_alerts++;
	
	SmlStatus *reply = smlCommandNewReply(cmd, SML_NO_ERROR, &error);
	if (!reply)
		goto error;
	
	if (!smlSessionSendReply(session, reply, &error)) {
		smlStatusUnref(reply);
		goto error;
	}
	
	smlStatusUnref(reply);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	if (error)
		smlErrorDeref(&error);
}

static void _status_reply(SmlSession *session, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, userdata);
	
	smlAssert(session != NULL);
	smlAssert(status != NULL);
	
	if (GPOINTER_TO_INT(userdata) == 2) {
		smlAssert(smlStatusGetCode(status) >= 500);
	} else if (GPOINTER_TO_INT(userdata) == 1) {
		smlAssert(smlStatusGetClass(status) == SML_ERRORCLASS_SUCCESS);
	} else
		abort();
	
	num_replies++;

	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _header_callback(SmlSession *session, SmlHeader *header, SmlCred *cred, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, session, header, cred, userdata);
	smlAssert(session);
	
	SmlStatus *reply = smlStatusNew(SML_AUTH_ACCEPTED, 0, session->lastReceivedMessageID, session->source, session->target, SML_COMMAND_TYPE_HEADER, NULL);
	if (!reply)
		goto error;
	
	if (!smlSessionSendReply(session, reply, NULL))
		goto error;
	
	smlStatusUnref(reply);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s", __func__);
	return;
}

START_TEST (manager_start_session)
{
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;
	
	num_replies = 0;
	num_alerts = 0;
	num_syncs = 0;
	num_changes = 0;
	
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12004;
	clientConfig.url = "http://127.0.0.1:12004";
	clientConfig.proxy = NULL;
	clientConfig.username = NULL;
	clientConfig.password = NULL;
	clientConfig.cafile = NULL;

	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12004;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlManagerObjectRegister(servermanager, SML_COMMAND_TYPE_ALERT, NULL, loc, NULL, NULL, _recv_alert, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlManagerRegisterHeaderHandler(servermanager, _header_callback, NULL, NULL);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_CLIENT, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, NULL, 0, &error);
	
	fail_unless(smlManagerSessionAdd(clientmanager, session, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlCommand *cmd = smlCommandNewAlert(SML_ALERT_TWO_WAY, loc, loc, "last", "next", NULL, &error);
	fail_unless(cmd != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	fail_unless(smlSessionSendCommand(session, cmd, NULL, _status_reply, GINT_TO_POINTER(1), &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlCommandUnref(cmd);
	
	fail_unless(smlSessionFlush(session, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlSessionUnref(session);
	
	while (num_finals != 1 || num_alerts != 1 || num_sessions != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 1, NULL);
	fail_unless(num_end == 0, NULL);
	fail_unless(session_errors == 0, NULL);
	
	fail_unless(servertracker->sessions[0] != NULL, NULL);
	fail_unless(smlSessionEnd(servertracker->sessions[0], &error), NULL);
	fail_unless(error == NULL, NULL);
	smlSessionUnref(servertracker->sessions[0]);
	
	while (num_end != 2 || num_replies != 1 || num_finals != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 2, NULL);
	fail_unless(num_end == 2, NULL);
	fail_unless(session_errors == 0, NULL);
	fail_unless(num_replies == 1, NULL);
	fail_unless(num_alerts == 1, NULL);
	fail_unless(num_syncs == 0, NULL);
	fail_unless(num_changes == 0, NULL);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (manager_register)
{
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;
	
	num_replies = 0;
	num_alerts = 0;
	num_syncs = 0;
	num_changes = 0;
	
	setup_testbed(NULL);
	SmlError *error = NULL;
	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12005;
	clientConfig.url = "http://127.0.0.1:12005";
	clientConfig.proxy = NULL;
	clientConfig.username = NULL;
	clientConfig.password = NULL;
	clientConfig.cafile = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12005;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlManagerObjectRegister(clientmanager, SML_COMMAND_TYPE_ALERT, NULL, loc, NULL, NULL, _recv_alert, NULL, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlLocationUnref(loc);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

Suite *manager_suite(void)
{
	Suite *s = suite_create("Manager");
	//Suite *s2 = suite_create("Manager");
	
	create_case(s, "manager_new", manager_new);
	create_case(s, "manager_run", manager_run);
	create_case(s, "manager_receive", manager_receive);
	create_case(s, "manager_receive_session", manager_receive_session);
	create_case(s, "manager_end_session", manager_end_session);
	create_case(s, "manager_register", manager_register);
	create_case(s, "manager_start_session", manager_start_session);
	
	return s;
}

int main(void)
{
	int nf;

	Suite *s = manager_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_VERBOSE);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
