/* SimpleDialog.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */
package be.ugent.caagt.swirl.dialogs;

import be.ugent.caagt.swirl.StandardButtons;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.HeadlessException;
import java.awt.MouseInfo;
import java.awt.Point;
import java.awt.PointerInfo;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

/**
 * Simple dialog window consisting of three different components:
 *
 * <ul>
 * <li>
 * A central panel, provided by the client.
 * </li>
 * <li>
 * An <code>OK</code> button and a <code>CANCEL</code> button, placed at the
 * bottom right of the window.
 * </li>
 * <li>
 * An optional icon, placed to the left of the central panel.
 * </li>
 * </ul>
 *
 * <p>
 * To use a dialog object of this type, you should first construct it, then add
 * the central component to it, using {@link #setCentralComponent} and finally
 * initialize it by calling <code>init</code>. Once initialized, you can call
 * {@link #launch} several times and inspect the user action using {@link
 * #getReturnValue}. <p>
 * Alternatively, consider extending the class {@link DialogPanel}
 * and using one of its {@code createDialog} methods.<p>
 * An icon can be installed, changed or removed
 * using {@link #setIcon} before the dialog is launched.
 * </p>
 */
public class SimpleDialog extends JDialog {
    
    //
    private final Component parent;
    
    /**
     * Construct a simple dialog with the given parent.
     */
    private SimpleDialog (Dialog window, Component parent, String title, boolean modal)
    throws HeadlessException {
        super (window, title, modal);
        this.parent = parent;
        this.label = new JLabel ();
    }
    
    /**
     * Construct a simple dialog with the given parent.
     */
    private SimpleDialog (Frame window, Component parent, String title, boolean modal)
    throws HeadlessException {
        super (window, title, modal);
        this.parent = parent;
        this.label = new JLabel ();
    }
    
    /**
     * Creates a new simple dialog with the given component as a parent.
     */
    public static SimpleDialog createSimpleDialog (Component parent,
            String title, boolean modal) {
        if (parent == null)
            return new SimpleDialog ((Frame)null, null, title, modal);
        
        Window window;
        if (parent instanceof Window)
            window = (Window)parent;
        else
            window = SwingUtilities.getWindowAncestor (parent);
        
        if (window instanceof Frame)
            return new SimpleDialog ((Frame)window, parent, title, modal);
        else
            return new SimpleDialog ((Dialog)window, parent, title, modal);
    }
    
    /** Return value if cancel is chosen (or window is simply closed). */
    public static final int CANCEL_OPTION = 1;
    
    /** Return value if approve (yes, ok) is chosen. */
    public static final int APPROVE_OPTION = 0;
    
    /** Return value if an error occured. */
    public static final int ERROR_OPTION = -1;
    
    /** Label which displays the icon */
    private final JLabel label;
    
    /** Central component. */
    private JComponent central;
    
    /**
     * Add the central component to the dialog.
     */
    public void setCentralComponent (JComponent component) {
        this.central = component;
    }
    
    /**
     * Add the icon to the dialog (or remove when null}.
     */
    public void setIcon (Icon icon) {
        label.setIcon (icon);
    }
    
    /** Main panel. */
    private JPanel panel;
    
    
    /** Button panel. */
    protected JPanel buttonPanel;
    
    /**
     * Creates the button panel and makes the <code>OK</code>-button the
     * default button.
     */
    protected void initButtonPanel () {
        
        // buttons
        buttonPanel = new JPanel ();
        buttonPanel.setLayout (new ButtonAreaLayout ());
        // OK Button
        JButton okButton = StandardButtons.createOKButton ();
        okButton.addActionListener (new ActionListener () {
            public void actionPerformed (ActionEvent e) {
                returnValue = APPROVE_OPTION;
                setVisible (false);
            }
        });
        getRootPane ().setDefaultButton (okButton);
        buttonPanel.add (okButton);
        
        // Cancel Button
        JButton cancelButton = StandardButtons.createCancelButton ();
        cancelButton.addActionListener (new ActionListener () {
            public void actionPerformed (ActionEvent e) {
                returnValue = CANCEL_OPTION;
                setVisible (false);
            }
        });
        buttonPanel.add (cancelButton);
    }
    
    /**
     * Adds margins to the label if the icon is not null.
     */
    private void initLabel () {
        if (label.getIcon () == null)
            panel.remove (label);
        else
            panel.add (label, BorderLayout.WEST);
    }
    
    /**
     * Return the button panel for this dialog. Can be used
     * to add further buttons.
     */
    public JPanel getButtonPanel () {
        return buttonPanel;
    }
    
    /** Last value returned by {@link #launch}. */
    private int returnValue = ERROR_OPTION;
    
    /**
     * Get the last value returned by {@link #launch}. One of
     * <code>CANCEL_OPTION</code>, <code>APPROVE_OPTION</code> or
     * <code>ERROR_OPTION</code>.
     */
    public int getReturnValue () {
        return returnValue;
    }
    
    /**
     * Show this dialog and wait for the user to respond to it. Displays the dialog
     * centered with respect to the parent component.
     *
     * @return an indication of the users action, either
     *         <code>CANCEL_OPTION</code>, <code>APPROVE_OPTION</code> or
     *         <code>ERROR_OPTION</code>.
     */
    public int launch () {
        if (parent == null)
            return launch (0,0);
        else
            return launch ((parent.getWidth ()- getWidth ())/ 2, (parent.getHeight () - getHeight ())/2);
    }
    
    /**
     * Show this dialog and wait for the user to respond to it. Displays the dialog
     * with the top left corner of its root pane at the current pointer position.
     *
     * @return an indication of the users action, either
     *         <code>CANCEL_OPTION</code>, <code>APPROVE_OPTION</code> or
     *         <code>ERROR_OPTION</code>.
     */
    public int launchAtPointerPosition () {
        try {
            PointerInfo pi = MouseInfo.getPointerInfo ();
            Point location = pi.getLocation ();
            if (location == null)
                return launch ();
            else {
                // TODO: extract common code with launch (int,int)
                initLabel ();
                pack ();
                Point myLocation = getRootPane ().getLocation ();
                setLocation (location.x - myLocation.x, location.y - myLocation.y);
                returnValue = ERROR_OPTION;
                setVisible (true);
                return returnValue;
            }
        } catch (HeadlessException ex) {
            return launch ();
        }
        
    }
    
    /**
     * Show this dialog and wait for the user to respond to it. Displays the
     * dialog window with its top left corner at the given relative position
     * with respect to the parent component.
     *
     * @return an indication of the users action, either
     *         <code>CANCEL_OPTION</code>, <code>APPROVE_OPTION</code> or
     *         <code>ERROR_OPTION</code>.
     */
    public int launch (int x, int y) {
        initLabel ();
        pack ();
        if (parent != null) {
            Point point = parent.getLocationOnScreen ();
            setLocation (point.x + x, point.y + y);
        }
        
        returnValue = ERROR_OPTION;
        setVisible (true);
        return returnValue;
    }
    
    
    /**
     * Show a simple modal dialog with the given panel at its center and wait
     * for the user to respond to it.
     *
     * @param panel Central component to be displayed
     * @param title Title of the dialog window
     *
     * @return an indication of the users action, either
     *         <code>CANCEL_OPTION</code>, <code>APPROVE_OPTION</code> or
     *         <code>ERROR_OPTION</code>.
     */
    public static int showSimpleDialog (Component parent, String title,
            JComponent panel) {
        return showSimpleDialog (parent, title, panel, null);
    }
    
    /**
     * Show a simple modal dialog with the given panel at its center and wait
     * for the user to respond to it.
     *
     * @param panel Central component to be displayed
     * @param title Title of the dialog window
     * @param icon  Icon to be displayed to the left of the panel
     *
     * @return an indication of the users action, either
     *         <code>CANCEL_OPTION</code>, <code>APPROVE_OPTION</code> or
     *         <code>ERROR_OPTION</code>.
     */
    public static int showSimpleDialog (Component parent, String title,
            JComponent panel, Icon icon) {
        SimpleDialog dialog = createSimpleDialog (parent, title,
                true);
        dialog.setCentralComponent (panel);
        dialog.init ();
        dialog.setIcon (icon);
        int result = dialog.launch ();
        dialog.dispose ();
        return result;
    }
    
    /**
     * Initializes and populates the dialog. Must be called once before the
     * first call to {@link #launch}.
     */
    public void init () {
        panel = new JPanel (new BorderLayout (12, 11));
        panel.setBorder (BorderFactory.createEmptyBorder (12, 12, 11, 11));
        setContentPane (panel);
        initButtonPanel ();
        
        panel.add (buttonPanel, BorderLayout.SOUTH);
        if (central != null)
            panel.add (central, BorderLayout.CENTER);
        
        addWindowListener (new WindowAdapter () {
            public void windowClosing (WindowEvent e) {
                returnValue = CANCEL_OPTION;
            }
        });
        setDefaultCloseOperation (HIDE_ON_CLOSE);
    }
}
