/* libspf - Sender Policy Framework library
*
*  ANSI C implementation of spf-draft-200405.txt
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*  Author: Sean Comeau   <scomeau@obscurity.org>
*
*  FILE: spf.h
*  DESC: main library header file
*
*  License:
*
*  The libspf Software License, Version 1.0
*
*  Copyright (c) 2004 James Couzens & Sean Comeau  All rights
*  reserved.
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*  1. Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*
*  2. Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*
*  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
*  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
*  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
*  DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS MAKING USE OF THIS LICESEN
*  OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
*  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
*  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
*  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
*  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
*  SUCH DAMAGE.
*
*/


#ifndef _SPF_H
#define _SPF_H 1

#include <sys/types.h>    /* typedefs */
#include <netinet/in.h>   /* in_addr struct fBSD */
#include <arpa/inet.h>    /* in_addr struct */

__BEGIN_DECLS


/* spf protocol version we support */
#define SPF_VERSION 1

/*
*  For reference purposes commented out are the constants based on
*  RFC 883, RFC 1034, RFC 1035.  Because we're working with IN_TXT
*  records we will use a larger packet size at 65536 bytes which
*  is likey to cover most circumstances.
*
*  #define PACKETSZ  512   max response packet size
*  #define MAXDNAME  1025  max uncompressed IN_TXT record
*  #define MAXCDNAME 255   max compressed IN_TXT record
*
*/

#define MAX_CNAME     5     /* we follow up max CNAMEs */

#define MAX_DEBUG_LEN 1024

#define MAX_MACRO_LEN 1024  /* max length of an expanded macro */
#define MAX_PLEN      1024  /* buffer to make UTIL_printf thread safe */
#define LOCAL_PART    256   /* local-part, text before @ in email addy */
#define ENV_SENDER    512   /* entire FROM: string passed by MTA */
#define CUR_DOM       256   /* text after @ in email addy for cur query */
#define UTC_TIME      22    /* time since epoch */
#define IP_ADDR       17    /* ip of remote peer - DON'T CHANGE FROM 17!! */
#define IP_VER        8     /* ip protocol version */
#define ENV_HELO      512   /* entire HELO string passed by MTA */

#define MAX_HNAME     256   /* hostname of MTA */
#define MAX_RESULT    64    /* human readable SPF result */
#define MAX_ERROR     96    /* human readable error reason */
#define EXPLAIN_S     256   /* change to EXPLAIN when cleaning up */
#define ENV_RECV      512   /* Received-SPF: header string */
#define MAX_RES_STR   12    /* maximum legnth of a res str eg: "pass" */

#define MAX_MECHANISM 256   /* maximum length of a mechanism */
#define MAX_HEADER    512   /* maximum length of header for prepend */
#define MAX_SMTP_RES  256   /* maximum length of smtp resonse string */

/* human readable string equivalents of spf responses */
#define HR_PASS     "pass"
#define HR_NONE     "none"
#define HR_S_FAIL   "softfail"
#define HR_H_FAIL   "fail"
#define HR_ERROR    "error"
#define HR_NEUTRAL  "neutral"
#define HR_UNKNOWN  "unknown"
#define HR_UNMECH   "unknown mechanism"

/* default explanation */
#define SPF_EXPLAIN "See http://spf.pobox.com/why.html?sender=%{S}&"      \
                    "ip=%{I}&receiver=%{xR}"

/* default best guess */
#define SPF_GUESS   "v=spf1 a/24 mx/24 ptr "

/* trusted forwarder */
#define SPF_TRUSTED "v=spf1 include:spf.trusted-forwarder.org "

/* This helps out Solaris and other places lacking the definitions
 * used in sys/types.h
#ifndef u_int32_t
#define u_int32_t unsigned int
#endif

#ifndef u_int16_t
#define u_int16_t unsigned short int
#endif

#ifndef u_int8_t
#define u_int8_t unsigned char
#endif

#ifndef int32_t
#define int32_t int
#endif

#ifndef int16_t
#define in16_t short int
#endif

#ifndef int8_t
#define int8_t char
#endif
*/


/* SPF_BOOL
*
*  Our own internal boolean enumeration, simple true or false.
*
*  Sendmail has issues because it makes use of FALSE and TRUE
*  Simple way around it iis to simply check and see if they are
*  defined or not since we're declaring this enumeration
*  globally.
*
*  A more graceful fix has been implemented in libspf 3.0 where
*  we now make use of SPF_TRUE and SPF_FALSE to steer clear of
*  problems such as this elsewhere.
*
*/

#ifdef TRUE
#undef TRUE
#endif

#ifdef FALSE
#undef FALSE
#endif

typedef enum SPF_BOOL
{
  FALSE = 0,
  TRUE
} SPF_BOOL;


/* SPF_RESULT
*
*  Error codes representing the result of an SPF policy examination
*
*  sucessful parse (some match was made) (+all)
*  not participating (no SPF/TXT records)
*  ~all
*  failed parse (no match made) (-all)
*  dns problem / error
*  ?all
*  permanent parsing error during record examination
*
*/
typedef enum SPF_RESULT
{
  SPF_PASS = 0,     /* + */
  SPF_NONE,
  SPF_S_FAIL,       /* ~ */
  SPF_H_FAIL,       /* - */
  SPF_ERROR,
  SPF_NEUTRAL,      /* ? */
  SPF_UNKNOWN,
  SPF_UNMECH        /* unknown mechanism */
} SPF_RESULT;


/* SPF_ACTION
*
*  Error codes representing the the action to be taken as a result
*  of the response the library was able to obtain whilst trying to
*  obtain or examin an SPF policy
*
*/
typedef enum SPF_ACTION
{
  DEFER = 0,
  TARPIT,
  ALLOW,
  REJECT
} SPF_ACTION;


/* SPF_MECHANISM
*
*  Error codes representing the various mechanism types employed
*  as defined in the RFC
*
*/
#undef VERSION /* autoconf */
typedef enum SPF_MECHANISM
{
  NO_POLICY = 0,
  VERSION,
  ALL,
  INCLUDE,
  A,
  MX,
  PTR,
  IP4,
  IP6,
  EXISTS,
  REDIRECT,
  EXPLAIN,
  DEFAULT, /* this is OLD school for early adopters = ~,?,+,- */
  UNMECH
} SPF_MECHANISM;


/* spf_result_t
*
*  Storage container used to store the result of an SPF parse.
*
*/
typedef struct spf_result_t
{
  size_t                  sl;      /* spf result string length */
  char                    s[32];   /* spf result type string */
  SPF_RESULT              i;       /* spf result type */
  size_t                  hl;      /* length of header string */
  char                    h[512];  /* Received-SPF: header string */
  char                    p;       /* prefix identifier */
} spf_result_t;


/* policy_addr_t
*
*  Storage container used to store parsed out ip addresses in their
*  binary format (in_addr struct) and an unsigned integer containing
*  the netmask
*
*/
typedef struct policy_addr_s
{
  struct in_addr    addr;     /* in_addr struct (unsigned long) */
  int8_t            cidr;     /* address cidr length */
  SPF_RESULT        prefix;   /* spf prefix (-,+,~,?) */
} policy_addr_t;


/* spf_config_t
*
* Global config structure
*
*/
typedef struct spf_config_s
{
  int               level;          /* w:  */
} spf_config_t;


/* split_str_node_t
*
*  This structure is used to store where the head and tail are when
*  creating a list of split_str_node_t structures.
*
*/
typedef struct strbuf_node_s
{
  size_t                 len;    /* length of string */
  char                   *s;     /* expanded string macro */
  struct strbuf_node_s   *next;  /* pointer to next node */
} strbuf_node_t;


/* strbuf_t
*
*  This structure is used exclusively by marco.c functions and is used
*  to store macros during parsing.
*
*/
typedef struct strbuf_s
{
  strbuf_node_t   *head;      /* head node */
  u_int8_t        elements;   /* number of nodes in list */
} strbuf_t;


/* split_str_node_t
*
*  This structure is used to store where the head and tail are when
*  creating a list of split_str_node_t structures.
*
*/
typedef struct split_str_node_s
{
  size_t                   len;    /* length of string */
  char                     *s;     /* expanded string macro */
  struct split_str_node_s  *next;  /* pointer to next node */
} split_str_node_t;


/* split_str_t
*
*  This structure is used exclusively by the UTIL_reverse function and is
*  used to reverse a string using a semi-arbitrary delimiter (see
*  UTIL_is_spf_delim for valid delimiters, or the SPF RFC)
*/
typedef struct split_str_s
{
  split_str_node_t  *head;      /* head node */
  split_str_node_t  *tail;      /* tail node */
  int               elements;   /* number of nodes in list */
} split_str_t;


/* peer_info_t
*
*  Used to store information about the connected peer.  Only one of
*  SMTP protocol specific three strings will be necessarily be
*  populated in the following order of precedence: FROM, EHLO, HELO.
*
*  The ip_ver string will contain 'in-addr' if the connecting peer
*  is using IPv4, or 'ip6' if the connect
*
*  Various political and technical pressures have recently led to
*  the deprecation of the IP6.INT name space in favour of IP6.ARPA.
*  This makes IPv6 PTR data management difficult, since interim
*  clients will search IP6.INT while standard clients will search
*  IP6.ARPA. We present a simple method based on DNAME RR's
*  (see [RFC2672]) and ISC BIND9 whereby zone information can be
*  managed in a single location and then made visible in two
*  namespaces.  (http://www.isc.org/tn/isc-tn-2002-1.html)
*
*  RFC 937 (POP) states: The maximum length of a command line is 512
*  characters (including the command word and the CRLF).
*  POLICY_MATCH = TRUE
*
*  Note: from can be removed and just work on local_part@cur_dom
*
*/
typedef struct peer_info_s
{
  u_int8_t        use_trust;        /* if 1 then enabled trustedfwder */
  u_int8_t        use_guess;        /* if 1 then enabled best guess */

  spf_result_t    *spf_result;      /* table of str, see spf_result_t */
  SPF_RESULT      RES;              /* SPF error codes for result */
  SPF_RESULT      RES_P;            /* prefix behaviour */
  char            *rs;              /* ptr str result of SPF query */

  SPF_BOOL        ALL;              /* Was all mechanism parsed */
  char            *p;               /* prefix from all mechanism */

  u_int8_t        spf_ver;          /* version of SPF */

  char            *helo;            /* HELO string */
  char            *ehlo;            /* pointer to HELO string */
  char            *from;            /* FROM string */

  char            *explain;         /* Result of an explain query */
  char            *guess;           /* Query if result is TF fails */
  char            *trusted;         /* Query if primary result is none */
  char            *ptr_mhost;       /* validate against during ptr mech */
  char            *cur_dom;         /* @domain of the current query */
  char            *cur_eaddr;       /* current email address, updated during
                                     * to be accurate during INCLUDE and
                                     * REDIRECT mechanisms
                                     */
  char            *mta_hname;       /* ptr to MTA hname eg: mail.foo.org */

  char            ip_ver[IP_VER];   /* IP Protocol Version */
  struct in_addr  addr;             /* IP of the remote host (peer) */
  char            *r_ip;            /* pointer to remote ip from MTA */
  char            *r_vhname;        /* validated hostname of remotehost */

  char local_part[LOCAL_PART];      /* local part of address (user) */
  char utc_time[UTC_TIME];          /* The number of seconds since the Epoch */
  char last_m[MAX_MECHANISM];       /* last mechanism parsed */
  char error[MAX_ERROR];            /* error (if any) that caused failure */

} peer_info_t;


extern spf_config_t   confg;
extern u_int8_t       spf_rlevel;

/*  Main library functions (main.c) */
extern peer_info_t  *SPF_init(const char *local, const char *rip, const char
*expl,
                      const char *tf, const char *guess, u_int32_t use_trust,
                      u_int32_t use_guess);
extern peer_info_t  *SPF_close(peer_info_t *peer_info);
extern SPF_RESULT   SPF_policy_main(peer_info_t *peer_info);
extern SPF_RESULT   SPF_policy_main_rec(peer_info_t *peer_info, int rec);
extern SPF_BOOL     SPF_parse_policy(peer_info_t *peer_info, const char *policy);
extern char         *SPF_result(peer_info_t *peer_info);
extern SPF_BOOL     SPF_smtp_from(peer_info_t *peer_info, const char *s);
extern SPF_BOOL     SPF_smtp_helo(peer_info_t *peer_info, const char *s);

/* Functions that alter headers (header.c) */
extern char         *SPF_build_header(peer_info_t *peer_info);
extern char         *SPF_get_explain(peer_info_t *peer_info);



__END_DECLS /* _SPF_H */

#endif /* spf.h */
