/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * vim:expandtab:autoindent:tabstop=4:shiftwidth=4:filetype=c:cindent:textwidth=0:
 *
 * Copyright (C) 2005 Dell Inc.
 *  by Michael Brown <Michael_E_Brown@dell.com>
 * Licensed under the Open Software License version 2.1
 *
 * Alternatively, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 */

// compat header should always be first header if including system headers
#define LIBSMBIOS_SOURCE
#include "smbios/compat.h"

#include <iomanip>

#include "SmbiosImpl.h"
// message.h should be included last.
#include "smbios/message.h"

using namespace smbiosLowlevel;
using namespace std;

#define NOT_IMPLEMENTED { throw NotImplementedImpl(); }

namespace smbios
{
    ISmbiosItem::~ISmbiosItem()
    {}

    ISmbiosItem::ISmbiosItem()
    {}

    //
    // COPY CONSTRUCTOR
    //
    SmbiosItem::SmbiosItem (const SmbiosItem & source)
            : ISmbiosItem(), header (source.header), header_size(source.header_size)
    {
        // only one allocation here. If it fails, there is
        // nothing to rollback, so we are exception safe.
        // if we add another allocation, we need to restructure this.
        u8 *newSmbiosItem = new u8[ header_size ];
        memcpy (newSmbiosItem, source.header, header_size);
        header = reinterpret_cast<const smbios_structure_header *>(newSmbiosItem);
    }

    //
    // REGULAR CONSTRUCTOR
    //
    SmbiosItem::SmbiosItem (const smbios_structure_header *init_header)
            : ISmbiosItem(), header(init_header), header_size(0)
    {
        // we need to copy all of our data out of the SmbiosTable
        // so that we have our own copy. This effectively lets
        // smbiosItem have a separate lifetime from it's containing
        // table.
        //
        // we do all of the stuff below, up to the "new", to figure
        // out the size of the item.

        // hop over to the next struct using smbios parsing rules
        // see smbiostable code for more details
        const u8 *nextStruct = reinterpret_cast<const u8 *>(header)  + header->length ;

        // skip over the strings in the string table. It ends with a double null
        while (*nextStruct || nextStruct[1])
            nextStruct++;

        // skip over the actual double null
        nextStruct += 2;

        // now we are at the next struct, we know the size
        // of the struct we are supposed to be pointing at.
        header_size = nextStruct - reinterpret_cast<const u8 *>(header);

        // only one allocation here. If it fails, there is
        // nothing to rollback, so we are safe.
        // if we add another allocation, we need to restructure this.
        u8 *newSmbiosItem = new u8[header_size];
        memcpy (newSmbiosItem, header, header_size);
        header = reinterpret_cast<const smbios_structure_header *>(newSmbiosItem);
    }

    SmbiosItem::~SmbiosItem ()
    {
        // its ugly because microsoft vc++ 6 sucks so much.
        delete [] const_cast<u8 *>(reinterpret_cast<const u8 *>(header));
        header = 0;
    }

    // gcc workaround. overprotective git. #!#$J@*(&$%
    // This is only used to format informational output stuff.
    // it loses precision, so don't do it except for display
    static u32 force_u64_to_u32(u64 orig)
    {
        // only gives correct results for little endian (I think)
        // but shouldn't matter, as it is for information purposes only.
        u32 *temp32 = reinterpret_cast<u32 *>(&orig);
        return *temp32;
    }

    // Lifetime of returned char* is the same as the SmbiosItem
    // FIXME: This function needs to make sure it doesn't run past the end of the table for
    // malformed strings.
    //
    // Invariant: will always return a valid const char * or throw an exception
    const char *SmbiosItem::getStringByStringNumber (u8 which) const
    {
        const char *string_pointer = reinterpret_cast<const char *>(header);

        //  either user is an idiot and should go read the spec,
        //  _or_ there is a '\0' in the table, indicating
        //  the string does not exist. :-(
        //  In either case, we throw an exception.
        if (!which)     //strings are numbered beginning with 1
        {
            throw StringUnavailableImpl(_("String does not exist."));
        }

        // don't deref header if it isn't valid
        // the world is not right if this happens.
        if (!header)
        {
            InternalErrorImpl internalError;
            internalError.setMessageString(_("Not a valid header. header is zero."));
            throw internalError;
        }

        // start out at the end of the header. This is where
        // the first string starts
        string_pointer += header->length;

        for (; which > 1; which--)
        {
            string_pointer += strlen (string_pointer);
            string_pointer++;  // skip past '\0'

            // check that we don't overflow this item
            //  additionally, split test into temp vars outside if() to work
            //  around astyle formatting bug where it will break code.
            const u8 *cur_loc = reinterpret_cast<const u8 *>(string_pointer);
            const u8 *base_loc =  reinterpret_cast<const u8 *>(header);
            if( cur_loc >= base_loc + header_size)
            {
                ParseExceptionImpl parseException;
                parseException.setMessageString(_("Overflow while getting byte data at location: cur_loc >= base_loc + header_size\n cur_loc : %(cur_loc)i\n base_loc : %(base_loc)i\n header_size : %(header_size)i "));
                parseException.setParameter("cur_loc",    force_u64_to_u32(reinterpret_cast<u64>(cur_loc)));
                parseException.setParameter("base_loc",   force_u64_to_u32(reinterpret_cast<u64>(base_loc)));
                parseException.setParameter("header_size",static_cast<u32>(header_size));
                throw parseException;
            }

            // if it is still '\0', that means we are
            // at the end of this item and should stop.
            // user gave us a bad index
            if( ! *string_pointer )
            {
                throw StringUnavailableImpl(_("The string does not exist. Bad index caused this error"));
            }
        }

        return string_pointer;
    }

    const char *SmbiosItem::getString(unsigned int header_offset) const
    {
        DataOutOfBoundsImpl dataOutOfBounds;
        // apply quick sanity check.
        if( header_offset >= getLength() )
        {
            dataOutOfBounds.setMessageString(_("Attempt to access string outside the length of header. offset : %(header_offset)i, header_length : %(header_length)i"));
            dataOutOfBounds.setParameter("header_offset",static_cast<int>(header_offset));
            dataOutOfBounds.setParameter("header_length",static_cast<int>(getLength()));
            throw dataOutOfBounds;
        }

        // Another world-gone-mad check
        if (!header)
        {
            InternalErrorImpl().setMessageString(_("Not a valid header. header is zero."));
            throw InternalErrorImpl();
        }

        return getStringByStringNumber ( (reinterpret_cast<const u8 *>(header))[header_offset]);
    }

    std::auto_ptr<const ISmbiosItem> SmbiosItem::clone() const
    {
        return auto_ptr<const ISmbiosItem>(new SmbiosItem (*this));
    }

    std::auto_ptr<ISmbiosItem> SmbiosItem::clone()
    {
        return auto_ptr<ISmbiosItem>(new SmbiosItem (*this));
    }

    u8 SmbiosItem::getType () const
    {
        return header->type;
    }

    u8 SmbiosItem::getLength () const
    {
        return header->length;
    }

    u16 SmbiosItem::getHandle () const
    {
        return header->handle;
    }

    // We do not implement the string passing interfaces
    // because we have no way of converting from string to int.
    u8          SmbiosItem::getU8 ( const std::string ) const   NOT_IMPLEMENTED;
    u16         SmbiosItem::getU16( const std::string ) const   NOT_IMPLEMENTED;
    u32         SmbiosItem::getU32( const std::string ) const   NOT_IMPLEMENTED;
    u64         SmbiosItem::getU64( const std::string ) const   NOT_IMPLEMENTED;
    const char *SmbiosItem::getString(const std::string ) const NOT_IMPLEMENTED;
    u32         SmbiosItem::getBitfield( const std::string, const std::string ) const   NOT_IMPLEMENTED;

    void checkItemBounds( size_t total_size, size_t length, size_t offset, size_t size)
    {
        DataOutOfBoundsImpl dataOutOfBounds;
        dataOutOfBounds.setParameter("offset",static_cast<int>(offset));
        dataOutOfBounds.setParameter("header_length",static_cast<int>(total_size));

        // tricky.  Need all three tests here in this order to avoid security hole
        if( offset > length )
        {
            dataOutOfBounds.setMessageString(_("Attempt to access data outside the length of header. offset : %(offset)i, header_length : %(header_length)i"));
            throw dataOutOfBounds;
        }

        if( offset + size < offset )
        {
            dataOutOfBounds.setMessageString(_("Attempt to access data outside the length of header. offset : %(offset)i, header_length : %(header_length)i"));
            throw dataOutOfBounds;
        }

        if( offset + size > length )
        {
            dataOutOfBounds.setMessageString(_("Attempt to access data outside the length of header. offset : %(offset)i, header_length : %(header_length)i"));
            throw dataOutOfBounds;
        }

        if( offset >= total_size ) // world gone mad check.
            // data is inside what the header says is
            // the length, but outside the range of the
            // buffer we are using to hold the header.
            // Impossible?
        {
            dataOutOfBounds.setMessageString(_("Attempt to access data outside header buffer. Impossible situation! offset : %(offset)i, header_length : %(header_length)i"));
            throw dataOutOfBounds;
        }

    }

    void getData(const ISmbiosItem &item, u8 *data, unsigned int offset, unsigned int size)
    {
        for (unsigned int i=0; i< size; i++)
        {
            data[i] = item.getU8(offset +i);
        }
    }

    u8 SmbiosItem::getU8 ( unsigned int offset ) const
    {
        checkItemBounds( header_size, header->length, offset, 1U );
        u8 byte1 = static_cast<u8>(reinterpret_cast<const u8 *>(header)[ offset + 0 ]);
        return byte1;
    }

    u16 SmbiosItem::getU16 ( unsigned int offset ) const
    {
        u16 retval = 0;
        checkItemBounds(header_size, header->length, offset, sizeof(retval));
        getData(*this, reinterpret_cast<u8 *>(&retval), offset, sizeof(retval));
        return retval;
    }


    u32 SmbiosItem::getU32 ( unsigned int offset ) const
    {
        u32 retval = 0;
        checkItemBounds(header_size, header->length, offset, sizeof(retval));
        getData(*this, reinterpret_cast<u8 *>(&retval), offset, sizeof(retval));
        return retval;
    }

    u64 SmbiosItem::getU64 ( unsigned int offset ) const
    {
        u64 retval = 0;
        checkItemBounds(header_size, header->length, offset, sizeof(retval));
        getData(*this, reinterpret_cast<u8 *>(&retval), offset, sizeof(retval));
        return retval;
    }

    void checkBitfieldBounds( unsigned int bound, unsigned int lsb, unsigned int msb, DataOutOfBoundsImpl &doob )
    {
        if(lsb > bound || msb > bound)
        {
            doob.setMessageString(_("The total length of bit field is out of bounds. The largest accessible bit is %(bound)i. lsb: %(lsb)i , msb: %(msb)i"));
            doob.setParameter( "bound", bound );
            throw doob;
        }
    }

    // Hope this is right too! ;-)
    u32 SmbiosItem::getBitfield ( unsigned int offset, unsigned int fieldLen, unsigned int lsb, unsigned int msb ) const
    {
        u64 bitfield = 0;
        DataOutOfBoundsImpl dataOutOfBounds;
        // reduce code duplication by setting these up now
        dataOutOfBounds.setParameter("lsb",lsb);
        dataOutOfBounds.setParameter("msb",msb);
        dataOutOfBounds.setParameter("offset",static_cast<int>(offset));
        dataOutOfBounds.setParameter("header_length",static_cast<int>(header->length));

        checkItemBounds( header_size, header->length, offset, fieldLen );

        //If msb is less/equal to lsb, they are only requesting a single bit
        if(msb <= lsb)
        {
            msb=lsb;
        }

        //Determine how many bits they want and make the mask
        u64 bitlen = (msb-lsb) + 1;
        u64 mask = 0;
        for(u64 i=0;i<bitlen;i++)
        {
            mask = (mask << 1);
            mask |= 1;
        }

        switch(fieldLen)
        {
        case FIELD_LEN_BYTE:
            checkBitfieldBounds( 7, lsb, msb, dataOutOfBounds);
            bitfield = getU8(offset);
            break;
        case FIELD_LEN_WORD:
            checkBitfieldBounds( 15, lsb, msb, dataOutOfBounds);
            bitfield = getU16(offset);
            break;
        case FIELD_LEN_DWORD:
            checkBitfieldBounds( 31, lsb, msb, dataOutOfBounds);
            bitfield = getU32(offset);
            break;
        case FIELD_LEN_QWORD:
            checkBitfieldBounds( 63, lsb, msb, dataOutOfBounds);
            bitfield = getU64(offset);
            break;
        default:
            dataOutOfBounds.setMessageString(_("Unsupported field length"));
            throw dataOutOfBounds;
        }

        return static_cast<u32>((bitfield >> lsb) & mask);
    }

    const u8 *SmbiosItem::getBufferCopy(size_t &size) const
    {
        size = header_size;

        const u8 *newBuffer = new u8[ size ];
        memcpy (const_cast<u8 *>(newBuffer), header, size);
        return newBuffer;
    }

    const size_t SmbiosItem::getBufferSize() const
    {
        return header_size;
    }

    void SmbiosItem::fixup( const SmbiosWorkaroundTable *workaround ) const
    {
        u8 *buffer = const_cast<u8 *>(reinterpret_cast<const u8 *>(header));
        workaround->fixupItem( this, buffer, header_size );
    }

    ostream & operator << (ostream & cout, const ISmbiosItem & item)
    {
        return item.streamify (cout);
    }

    ostream & SmbiosItem::streamify (ostream & cout) const
    {
        if (header == 0)  // violates class invariant, should never happen
            cout << "operator << on an uninitialized SmbiosItem!";
        else
        {
            std::ios::fmtflags old_opts = cout.flags ();
            cout << "Handle 0x" << hex << setfill ('0') <<
            setw (4) << getHandle () << endl;
            cout << "\tDMI type 0x" << static_cast<int>(getType()) << dec <<
            ", " << static_cast<int>(getLength()) << " bytes." <<
            endl;
            cout.flags (old_opts);
        }
        return cout;
    }
}
