////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Graphics/Sprite.hpp>
#include <SFML/Graphics/Image.hpp>
#include <SFML/Graphics/OpenGL.hpp>


namespace sf
{
////////////////////////////////////////////////////////////
/// Default constructor
////////////////////////////////////////////////////////////
Sprite::Sprite() :
myImage     (NULL),
mySubRect   (0, 0, 1, 1),
myIsFlippedX(false),
myIsFlippedY(false)
{

}


////////////////////////////////////////////////////////////
/// Construct the sprite from a source image
////////////////////////////////////////////////////////////
Sprite::Sprite(const Image& Img, float Left, float Top, float ScaleX, float ScaleY, float Rotation, const Color& Col) :
Drawable    (Left, Top, ScaleX, ScaleY, Rotation, Col),
myImage     (&Img),
mySubRect   (0, 0, Img.GetWidth(), Img.GetHeight()),
myIsFlippedX(false),
myIsFlippedY(false)
{

}


////////////////////////////////////////////////////////////
/// Set the image of the sprite
////////////////////////////////////////////////////////////
void Sprite::SetImage(const Image& Img)
{
    // If there was no source image before, adjust the rectangle
    if (!myImage)
        SetSubRect(IntRect(0, 0, Img.GetWidth(), Img.GetHeight()));

    // Assign the new image
    myImage = &Img;
}


////////////////////////////////////////////////////////////
/// Set the sub-rectangle of the sprite inside the source image
////////////////////////////////////////////////////////////
void Sprite::SetSubRect(const IntRect& SubRect)
{
    mySubRect = SubRect;
}


////////////////////////////////////////////////////////////
/// Resize the object (by changing its scale factors)
////////////////////////////////////////////////////////////
void Sprite::Resize(float Width, float Height)
{
    if ((mySubRect.GetWidth() > 0) && (mySubRect.GetHeight() > 0))
    {
        if (Width  > 0.f) SetScaleX(Width  / mySubRect.GetWidth());
        if (Height > 0.f) SetScaleY(Height / mySubRect.GetHeight());
    }
}


////////////////////////////////////////////////////////////
/// Flip the sprite horizontally
////////////////////////////////////////////////////////////
void Sprite::FlipX(bool Flipped)
{
    myIsFlippedX = Flipped;
}


////////////////////////////////////////////////////////////
/// Flip the sprite vertically
////////////////////////////////////////////////////////////
void Sprite::FlipY(bool Flipped)
{
    myIsFlippedY = Flipped;
}


////////////////////////////////////////////////////////////
/// Get the source image of the sprite
////////////////////////////////////////////////////////////
const Image* Sprite::GetImage() const
{
    return myImage;
}


////////////////////////////////////////////////////////////
/// Get the sub-rectangle of the sprite inside the source image
////////////////////////////////////////////////////////////
const IntRect& Sprite::GetSubRect() const
{
    return mySubRect;
}


////////////////////////////////////////////////////////////
/// Get the sprite width
////////////////////////////////////////////////////////////
float Sprite::GetWidth() const
{
    return mySubRect.GetWidth() * GetScaleX();
}


////////////////////////////////////////////////////////////
/// Get the sprite height
////////////////////////////////////////////////////////////
float Sprite::GetHeight() const
{
    return mySubRect.GetHeight() * GetScaleY();
}


////////////////////////////////////////////////////////////
/// Get the color of a given pixel in the sprite
////////////////////////////////////////////////////////////
Color Sprite::GetPixel(unsigned int X, unsigned int Y) const
{
    if (myImage)
    {
        unsigned int ImageX = mySubRect.Left + static_cast<unsigned int>(X / GetScaleX());
        unsigned int ImageY = mySubRect.Top  + static_cast<unsigned int>(Y / GetScaleY());

        return myImage->GetPixel(ImageX, ImageY) * GetColor();
    }
    else
    {
        return GetColor();
    }
}


////////////////////////////////////////////////////////////
/// /see sfDrawable::Render
////////////////////////////////////////////////////////////
void Sprite::Render(const RenderWindow&) const
{
    // Get the sprite size
    float Width  = static_cast<float>(mySubRect.GetWidth());
    float Height = static_cast<float>(mySubRect.GetHeight());

    // Check if the image is valid
    if (myImage)
    {
        // Calculate the texture coordinates
        FloatRect TexCoords = myImage->GetTexCoords(mySubRect);

        // Set the texture
        myImage->Bind();

        // Calculate the texture coordinates
        FloatRect Rect(myIsFlippedX ? TexCoords.Right  : TexCoords.Left,
                       myIsFlippedY ? TexCoords.Bottom : TexCoords.Top,
                       myIsFlippedX ? TexCoords.Left   : TexCoords.Right,
                       myIsFlippedY ? TexCoords.Top    : TexCoords.Bottom);

        // Draw the textured sprite
        glBegin(GL_QUADS);
            glTexCoord2f(Rect.Left,  Rect.Top);    glVertex2f(0,     0);
            glTexCoord2f(Rect.Left,  Rect.Bottom); glVertex2f(0,     Height);
            glTexCoord2f(Rect.Right, Rect.Bottom); glVertex2f(Width, Height);
            glTexCoord2f(Rect.Right, Rect.Top);    glVertex2f(Width, 0);
        glEnd();
    }
    else
    {
        // Disable texturing
        glDisable(GL_TEXTURE_2D);

        // Draw the colored rectangle
        glBegin(GL_QUADS);
            glVertex2f(0,     0);
            glVertex2f(0,     Height);
            glVertex2f(Width, Height);
            glVertex2f(Width, 0);
        glEnd();
    }
}

} // namespace sf
