////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

#ifndef SFML_RENDERWINDOW_HPP
#define SFML_RENDERWINDOW_HPP

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Graphics/Color.hpp>
#include <SFML/Graphics/Image.hpp>
#include <SFML/Graphics/View.hpp>
#include <SFML/Graphics/Rect.hpp>
#include <SFML/Window/Window.hpp>
#include <string>


namespace sf
{
class Drawable;

////////////////////////////////////////////////////////////
/// Simple wrapper for sf::Window that allows easy
/// 2D rendering
////////////////////////////////////////////////////////////
class SFML_API RenderWindow : public Window
{
public :

    ////////////////////////////////////////////////////////////
    /// Default constructor
    ///
    ////////////////////////////////////////////////////////////
    RenderWindow();

    ////////////////////////////////////////////////////////////
    /// Construct the window
    ///
    /// \param Mode :              Video mode to use
    /// \param Title :             Title of the window
    /// \param WindowStyle :       Window style (Resize | Close by default)
    /// \param AntialiasingLevel : Level of antialiasing (0 by default, disabled)
    ///
    ////////////////////////////////////////////////////////////
    RenderWindow(VideoMode Mode, const std::string& Title, unsigned long WindowStyle = Style::Resize | Style::Close, int AntialiasingLevel = 0);

    ////////////////////////////////////////////////////////////
    /// Construct the window from an existing control
    ///
    /// \param Handle :            Platform-specific handle of the control
    /// \param AntialiasingLevel : Level of antialiasing (0 by default, disabled)
    ///
    ////////////////////////////////////////////////////////////
    RenderWindow(WindowHandle Handle, int AntialiasingLevel = 0);

    ////////////////////////////////////////////////////////////
    /// Destructor
    ///
    ////////////////////////////////////////////////////////////
    virtual ~RenderWindow();

    ////////////////////////////////////////////////////////////
    /// Draw something on the window
    ///
    /// \param Object : Object to draw
    ///
    ////////////////////////////////////////////////////////////
    void Draw(const Drawable& Object) const;

    ////////////////////////////////////////////////////////////
    /// Save the content of the window to an image
    ///
    /// \return Image instance containing the contents of the screen
    ///
    ////////////////////////////////////////////////////////////
    Image Capture() const;

    ////////////////////////////////////////////////////////////
    /// Change the background color of the window.
    /// The default color is black
    ///
    /// \param Col : New background color
    ///
    ////////////////////////////////////////////////////////////
    void SetBackgroundColor(const Color& Col);

    ////////////////////////////////////////////////////////////
    /// Change the current active view.
    /// The current view is defined with the initial size of the window
    ///
    /// \param NewView : Pointer to the new view (pass NULL to set the default view)
    ///
    ////////////////////////////////////////////////////////////
    void SetView(const View* NewView);

    ////////////////////////////////////////////////////////////
    /// Get the current view rectangle
    ///
    /// \return Current view rectangle, in global coordinates
    ///
    ////////////////////////////////////////////////////////////
    const FloatRect& GetViewRect() const;

    ////////////////////////////////////////////////////////////
    /// Tell SFML to optimize its calls to the graphics driver,
    /// in case the user is not doing custom OpenGL calls.
    /// This parameter is false by default
    ///
    /// \param Optimize : True to enable internal states optimizations, false to go back to safe mode
    ///
    ////////////////////////////////////////////////////////////
    void OptimizeForNonOpenGL(bool Optimize);

private :

    ////////////////////////////////////////////////////////////
    /// /see Window::OnCreate
    ///
    ////////////////////////////////////////////////////////////
    virtual void OnCreate();

    ////////////////////////////////////////////////////////////
    /// /see Window::OnDisplay
    ///
    ////////////////////////////////////////////////////////////
    virtual void OnDisplay();

    ////////////////////////////////////////////////////////////
    /// /see Window::OnEvent
    ///
    ////////////////////////////////////////////////////////////
    virtual void OnEventReceived(const Event& EventReceived);

    ////////////////////////////////////////////////////////////
    /// Set the OpenGL render states needed for the SFML rendering
    ///
    ////////////////////////////////////////////////////////////
    void SetRenderStates() const;

    ////////////////////////////////////////////////////////////
    // Member data
    ////////////////////////////////////////////////////////////
    Color     myBackgroundColor;       ///< Background color
    View      myDefaultView;           ///< Default view
    FloatRect myCurrentRect;           ///< Rectangle corresponding to the current view
    float     myCurrentProjection[16]; ///< Projection matrix currently used by the window
    bool      myOptimizeStates;        ///< This flag is a hint, to save many calls to the graphics driver when not mixing with OpenGL
};

} // namespace sf


#endif // SFML_RENDERWINDOW_HPP
