%javamethodmodifiers getLibSBMLVersion "
/**
 * Returns the version number of this copy of libSBML as an integer.
 * <p>
 * @return the libSBML version as an integer; version 1.2.3 becomes 10203.
 */ public
";


%javamethodmodifiers getLibSBMLDottedVersion "
/**
 * Returns the version number of this copy of libSBML as a string.
 * <p>
 * @return the libSBML version as a string; version 1.2.3 becomes
 * \'1.2.3\'.
 * <p>
 * @see #getLibSBMLVersionString()
 */ public
";


%javamethodmodifiers getLibSBMLVersionString "
/**
 * Returns the version number of this copy of libSBML as a string without
 * periods.
 * <p>
 * @return the libSBML version as a string: version 1.2.3 becomes \'10203\'.
 * <p>
 * @see #getLibSBMLDottedVersion()
 */ public
";


%typemap(javaimports) SBMLReader "
/** 
 * Methods for reading SBML from files and text strings.
 * <p>
 * <p style=\'color: #777; font-style: italic\'>
This class of objects is defined by libSBML only and has no direct
equivalent in terms of SBML components.  This class is not prescribed by
the SBML specifications, although it is used to implement features
defined in SBML.
</p>

 * <p>
 * The {@link SBMLReader} class provides the main interface for reading SBML
 * content from files and strings.  The methods for reading SBML all return
 * an {@link SBMLDocument} object representing the results.
 * <p>
 * In the case of failures (such as if the SBML contains errors or a file
 * cannot be read), the errors will be recorded with the {@link SBMLErrorLog}
 * object kept in the {@link SBMLDocument} returned by {@link SBMLReader}.  Consequently,
 * immediately after calling a method on {@link SBMLReader}, callers should always
 * check for errors and warnings using the methods for this purpose
 * provided by {@link SBMLDocument}.
 * <p>
 * For convenience as well as easy access from other languages besides C++,
 * this file also defines two global functions,
 * {@link libsbml#readSBML(String filename)}
 * and {@link libsbml#readSBMLFromString(String xml)}.
 * They are equivalent to creating an {@link SBMLReader}
 * object and then calling the
 * {@link SBMLReader#readSBML(String filename)} or
 * {@link SBMLReader#readSBMLFromString(String xml)}
 * methods, respectively.
 * <p>
 * <h2>Support for reading compressed files</h2>
 * <p>
 * LibSBML provides support for reading (as well as writing) compressed
 * SBML files.  The process is transparent to the calling
 * application&mdash;the application does not need to do anything
 * deliberate to invoke the functionality.  If a given SBML filename ends
 * with an extension for the <em>gzip</em>, <em>zip</em> or <em>bzip2</em> compression
 * formats (respectively, <code>.gz</code>, <code>.zip</code>, or <code>.bz2</code>), then the methods
 * {@link SBMLReader#readSBML(String filename)} and
 * {@link SBMLWriter#writeSBML(SBMLDocument d, String filename)}
 * will automatically decompress and compress the file while writing and
 * reading it.  If the filename has no such extension, it
 * will be read and written uncompressed as normal.
 * <p>
 * The compression feature requires that the <em>zlib</em> (for <em>gzip</em> and 
 * <em>zip</em> formats) and/or <em>bzip2</em> (for <em>bzip2</em> format) be available on the
 * system running libSBML, and that libSBML was configured with their
 * support compiled-in.  Please see the libSBML <a href=\'../../../{@link libsbml}-installation.html\'>installation instructions</a>  for more information about this.  The methods
 * {@link SBMLReader#hasZlib()} and
 * {@link SBMLReader#hasBzip2()}
 * can be used by an application to query at run-time whether support
 * for the compression libraries is available in the present copy of
 * libSBML.
 * <p>
 * Support for compression is not mandated by the SBML standard, but
 * applications may find it helpful, particularly when large SBML models
 * are being communicated across data links of limited bandwidth.
 */
"


%javamethodmodifiers SBMLReader::SBMLReader "
/**
   * Creates a new {@link SBMLReader} and returns it. 
   * <p>
   * The libSBML {@link SBMLReader} objects offer methods for reading SBML in
   * XML form from files and text strings.
   */ public
";


%javamethodmodifiers SBMLReader::readSBML(const std::string& filename) "
/**
   * Reads an SBML document from a file.
   * <p>
   * This method is identical to {@link SBMLReader#readSBMLFromFile(String filename)}.
   * <p>
   * If the file named <code>filename</code> does not exist or its content is not
   * valid SBML, one or more errors will be logged with the {@link SBMLDocument}
   * object returned by this method.  Callers can use the methods on
   * {@link SBMLDocument} such as {@link SBMLDocument#getNumErrors()} and
   * {@link SBMLDocument#getError(long n)} to get the errors.  The object returned by
   * {@link SBMLDocument#getError(long n)} is an {@link SBMLError} object, and it has methods to
   * get the error code, category, and severity level of the problem, as
   * well as a textual description of the problem.  The possible severity
   * levels range from informational messages to fatal errors; see the
   * documentation for {@link SBMLError} for more information.
   * <p>
   * If the file <code>filename</code> could not be read, the file-reading error will
   * appear first.  The error code  can provide a clue about what happened.  For example,
   * a file might be unreadable (either because it does not actually exist
   * or because the user does not have the necessary access priviledges to
   * read it) or some sort of file operation error may have been reported
   * by the underlying operating system.  Callers can check for these
   * situations using a program fragment such as the following:
   * <div class=\'fragment\'><pre>
 {@link SBMLReader} reader = new {@link SBMLReader}();
 {@link SBMLDocument} doc  = reader.readSBMLFromFile(filename);
 
 if (doc.getNumErrors() &gt; 0)
 {
     if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileUnreadable)
     {
         // Handle case of unreadable file here.
     } 
     else if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileOperationError)
     {
         // Handle case of other file operation error here.
     }
     else
     {
         // Handle other error cases.
     }
 }
 </pre></div>
 <p>
   * <p>
   * If the given filename ends with the suffix <code>\'.gz\'</code> (for example, 
   * <code>\'myfile.xml.gz\'</code>), the file is assumed to be compressed in <em>gzip</em>
   * format and will be automatically decompressed upon reading.
   * Similarly, if the given filename ends with <code>\'.zip\'</code> or <code>\'.bz2\'</code>, the
   * file is assumed to be compressed in <em>zip</em> or <em>bzip2</em> format
   * (respectively).  Files whose names lack these suffixes will be read
   * uncompressed.  Note that if the file is in <em>zip</em> format but the
   * archive contains more than one file, only the first file in the
   * archive will be read and the rest ignored.
   * <p>
   * To read a gzip/zip file, libSBML needs to be configured and linked with the
<a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
time.  It also needs to be linked with the <a target=\'_blank\'
href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
these are the default configurations for libSBML.)  Errors about unreadable
files will be logged if a compressed filename is given and libSBML was
<em>not</em> linked with the corresponding required library.

   * <p>
   * @param filename the name or full pathname of the file to be read.
   * <p>
   * @return a pointer to the {@link SBMLDocument} created from the SBML content.
   * <p>
   * @note LibSBML versions 2.x and later versions behave differently in
   * error handling in several respects.  One difference is how early some
   * errors are caught and whether libSBML continues processing a file in
   * the face of some early errors.  In general, libSBML versions after 2.x
   * stop parsing SBML inputs sooner than libSBML version 2.x in the face
   * of XML errors, because the errors may invalidate any further SBML
   * content.  For example, a missing XML declaration at the beginning of
   * the file was ignored by libSBML 2.x but in version 3.x and later, it
   * will cause libSBML to stop parsing the rest of the input altogether.
   * While this behavior may seem more severe and intolerant, it was
   * necessary in order to provide uniform behavior regardless of which
   * underlying XML parser (Expat, Xerces, libxml2) is being used by
   * libSBML.  The XML parsers themselves behave differently in their error
   * reporting, and sometimes libSBML has to resort to the lowest common
   * denominator.
   * <p>
   * @see SBMLError
   */ public
";


%javamethodmodifiers SBMLReader::readSBMLFromFile(const std::string& filename) "
/**
   * Reads an SBML document from a file.
   * <p>
   * This method is identical to {@link SBMLReader#readSBML(String filename)}.
   * <p>
   * If the file named <code>filename</code> does not exist or its content is not
   * valid SBML, one or more errors will be logged with the {@link SBMLDocument}
   * object returned by this method.  Callers can use the methods on
   * {@link SBMLDocument} such as {@link SBMLDocument#getNumErrors()} and
   * {@link SBMLDocument#getError(long n)} to get the errors.  The object returned by
   * {@link SBMLDocument#getError(long n)} is an {@link SBMLError} object, and it has methods to
   * get the error code, category, and severity level of the problem, as
   * well as a textual description of the problem.  The possible severity
   * levels range from informational messages to fatal errors; see the
   * documentation for {@link SBMLError} for more information.
   * <p>
   * If the file <code>filename</code> could not be read, the file-reading error will
   * appear first.  The error code  can provide a clue about what happened.  For example,
   * a file might be unreadable (either because it does not actually exist
   * or because the user does not have the necessary access priviledges to
   * read it) or some sort of file operation error may have been reported
   * by the underlying operating system.  Callers can check for these
   * situations using a program fragment such as the following:
   * <div class=\'fragment\'><pre>
 {@link SBMLReader} reader = new {@link SBMLReader}();
 {@link SBMLDocument} doc  = reader.readSBMLFromFile(filename);
 
 if (doc.getNumErrors() &gt; 0)
 {
     if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileUnreadable)
     {
         // Handle case of unreadable file here.
     } 
     else if (doc.getError(0).getErrorId() == libsbmlConstants.XMLFileOperationError)
     {
         // Handle case of other file operation error here.
     }
     else
     {
         // Handle other error cases.
     }
 }
 </pre></div>
 <p>
   * <p>
   * If the given filename ends with the suffix <code>\'.gz\'</code> (for example, 
   * <code>\'myfile.xml.gz\'</code>), the file is assumed to be compressed in <em>gzip</em>
   * format and will be automatically decompressed upon reading.
   * Similarly, if the given filename ends with <code>\'.zip\'</code> or <code>\'.bz2\'</code>, the
   * file is assumed to be compressed in <em>zip</em> or <em>bzip2</em> format
   * (respectively).  Files whose names lack these suffixes will be read
   * uncompressed.  Note that if the file is in <em>zip</em> format but the
   * archive contains more than one file, only the first file in the
   * archive will be read and the rest ignored.
   * <p>
   * To read a gzip/zip file, libSBML needs to be configured and linked with the
<a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
time.  It also needs to be linked with the <a target=\'_blank\'
href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
these are the default configurations for libSBML.)  Errors about unreadable
files will be logged if a compressed filename is given and libSBML was
<em>not</em> linked with the corresponding required library.

   * <p>
   * @param filename the name or full pathname of the file to be read.
   * <p>
   * @return a pointer to the {@link SBMLDocument} created from the SBML content.
   * <p>
   * @note LibSBML versions 2.x and later versions behave differently in
   * error handling in several respects.  One difference is how early some
   * errors are caught and whether libSBML continues processing a file in
   * the face of some early errors.  In general, libSBML versions after 2.x
   * stop parsing SBML inputs sooner than libSBML version 2.x in the face
   * of XML errors, because the errors may invalidate any further SBML
   * content.  For example, a missing XML declaration at the beginning of
   * the file was ignored by libSBML 2.x but in version 3.x and later, it
   * will cause libSBML to stop parsing the rest of the input altogether.
   * While this behavior may seem more severe and intolerant, it was
   * necessary in order to provide uniform behavior regardless of which
   * underlying XML parser (Expat, Xerces, libxml2) is being used by
   * libSBML.  The XML parsers themselves behave differently in their error
   * reporting, and sometimes libSBML has to resort to the lowest common
   * denominator.
   * <p>
   * @see SBMLError
   * @see SBMLDocument
   */ public
";


%javamethodmodifiers SBMLReader::readSBMLFromString(const std::string& xml) "
/**
   * Reads an SBML document from the given XML string.
   * <p>
   * This method is flexible with respect to the presence of an XML
   * declaration at the beginning of the string.  In particular, if the
   * string in <code>xml</code> does not begin with the XML declaration
   * <code>&lt;?xml version=\'1.0\' encoding=\'UTF-8\'?&gt;</code>, then this
   * method will automatically prepend the declaration to <code>xml</code>.
   * <p>
   * This method will log a fatal error if the content given in the
   * parameter <code>xml</code> is not SBML.  See the method documentation for
   * {@link SBMLReader#readSBML(String filename)}
   * for an example of code for testing the returned error code.
   * <p>
   * @param xml a string containing a full SBML model
   * <p>
   * @return a pointer to the {@link SBMLDocument} created from the SBML content.
   * <p>
   * @see SBMLReader#readSBML(String filename)
   */ public
";


%javamethodmodifiers SBMLReader::hasZlib "
/**
   * Static method; returns <code>true</code> if this copy of libSBML supports
   * <i>gzip</I> and <i>zip</i> format compression.
   * <p>
   * @return <code>true</code> if libSBML has been linked with the <i>zlib</i>
   * library, <code>false</code> otherwise.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link SBMLReader}), and the other
   * will be a standalone top-level function with the name
   * SBMLReader_hasZlib(). They are functionally identical. 
   * <p>
   * @see SBMLReader#hasBzip2() 
   */ public
";


%javamethodmodifiers SBMLReader::hasBzip2 "
/**
   * Static method; returns <code>true</code> if this copy of libSBML supports
   * <i>bzip2</i> format compression.
   * <p>
   * @return <code>true</code> if libSBML is linked with the <i>bzip2</i>
   * libraries, <code>false</code> otherwise.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link SBMLReader}), and the other
   * will be a standalone top-level function with the name
   * SBMLReader_hasBzip2(). They are functionally identical. 
   * <p>
   * @see SBMLReader#hasZlib() 
   */ public
";


%javamethodmodifiers SBMLReader::readInternal(const char* content, bool isFile = true) "
/**
   * Used by readSBML() and readSBMLFromString().
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   * @internal
   */ public
";


%javamethodmodifiers readSBML(const char *filename) "
/**
 * Reads an SBML document from the given file <code>filename</code>.
 * <p>
 * If <code>filename</code> does not exist, or it is not an SBML file, an error will
 * be logged in the error log of the {@link SBMLDocument} object returned by this
 * method.  Calling programs can inspect this error log to determine
 * the nature of the problem.  Please refer to the definition of
 * {@link SBMLDocument} for more information about the error reporting mechanism.
 * <p>
 * @return a pointer to the {@link SBMLDocument} read.
 */ public
";


%javamethodmodifiers readSBMLFromFile(const char *filename) "
/**
 * Reads an SBML document from the given file <code>filename</code>.
 * <p>
 * If <code>filename</code> does not exist, or it is not an SBML file, an error will
 * be logged in the error log of the {@link SBMLDocument} object returned by this
 * method.  Calling programs can inspect this error log to determine
 * the nature of the problem.  Please refer to the definition of
 * {@link SBMLDocument} for more information about the error reporting mechanism.
 * <p>
 * @return a pointer to the {@link SBMLDocument} read.
 */ public
";


%javamethodmodifiers readSBMLFromString(const char *xml) "
/**
 * Reads an SBML document from a string assumed to be in XML format.
 * <p>
 * If the string does not begin with XML declaration,
 *<div class=\'fragment\'><pre>
&lt;?xml version=\'1.0\' encoding=\'UTF-8\'?&gt;
</pre></div>
 * <p>
 * an XML declaration string will be prepended.
 * <p>
 * This method will report an error if the given string <code>xml</code> is not SBML.
 * The error will be logged in the error log of the {@link SBMLDocument} object
 * returned by this method.  Calling programs can inspect this error log to
 * determine the nature of the problem.  Please refer to the definition of
 * {@link SBMLDocument} for more information about the error reporting mechanism.
 * <p>
 * @return a pointer to the {@link SBMLDocument} read.
 */ public
";


%typemap(javaimports) SBMLWriter "
/** 
 * Methods for writing SBML to files and text strings.
 * <p>
 * <p style=\'color: #777; font-style: italic\'>
This class of objects is defined by libSBML only and has no direct
equivalent in terms of SBML components.  This class is not prescribed by
the SBML specifications, although it is used to implement features
defined in SBML.
</p>

 * <p>
 * The {@link SBMLWriter} class is the converse of {@link SBMLReader}, and provides the
 * main interface for serializing SBML models into XML and writing the
 * result to files and text strings.  The methods for writing SBML all take
 * an {@link SBMLDocument} object and a destination.  They return a boolean or
 * integer value to indicate success or failure.
 * <p>
 * <h2>Support for writing compressed files</h2>
 * <p>
 * LibSBML provides support for writing (as well as reading) compressed
 * SBML files.  The process is transparent to the calling
 * application&mdash;the application does not need to do anything
 * deliberate to invoke the functionality.  If a given SBML filename ends
 * with an extension for the <em>gzip</em>, <em>zip</em> or <em>bzip2</em> compression
 * formats (respectively, <code>&quot;.gz&quot;</code>,
 * <code>&quot;.zip&quot;</code>, or <code>&quot;.bz2&quot;</code>),
 * then the methods
 * {@link SBMLWriter#writeSBML(SBMLDocument d, String filename)}
 * and {@link SBMLReader#readSBML(String filename)}
 * will automatically compress and decompress the file while writing and
 * reading it.  If the filename has no such extension, it
 * will be written and read uncompressed as normal.
 * <p>
 * The compression feature requires that the <em>zlib</em> (for <em>gzip</em> and 
 * <em>zip</em> formats) and/or <em>bzip2</em> (for <em>bzip2</em> format) be available on the
 * system running libSBML, and that libSBML was configured with their
 * support compiled-in.  Please see the libSBML <a href=\'../../../{@link libsbml}-installation.html\'>installation instructions</a> for 
 * more information about this.  The methods
 * {@link SBMLWriter#hasZlib()} and
 * {@link SBMLWriter#hasBzip2()}
 * can be used by an application to query at run-time whether support
 * for the compression libraries is available in the present copy of
 * libSBML.
 * <p>
 * Support for compression is not mandated by the SBML standard, but
 * applications may find it helpful, particularly when large SBML models
 * are being communicated across data links of limited bandwidth.
 */
"


%javamethodmodifiers SBMLWriter::SBMLWriter "
/**
   * Creates a new {@link SBMLWriter}.
   * <p>
   * The libSBML {@link SBMLWriter} objects offer methods for writing SBML in
   * XML form to files and text strings.
   */ public
";


%javamethodmodifiers SBMLWriter::setProgramName(const std::string& name) "
/**
   * Sets the name of this program, i.e., the program that is about to
   * write out the {@link SBMLDocument}.
   * <p>
   * If the program name and version are set (see
   * {@link SBMLWriter#setProgramVersion(String version)}), the
   * following XML comment, intended for human consumption, will be written
   * at the beginning of the XML document:
   * <div class=\'fragment\'><pre>
 &lt;!-- Created by &lt;program name&gt; version &lt;program version&gt;
 on yyyy-MM-dd HH:mm with libSBML version &lt;{@link libsbml} version&gt;. --&gt;
 </pre></div>
   * <p>
   * If the program name and version are not set at some point before
   * calling the writeSBML() methods, no such comment is written out.
   * <p>
   * @param name the name of this program (where \'this program\' refers to
   * program in which libSBML is embedded, not libSBML itself!)
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @see #setProgramVersion(String version)
   */ public
";


%javamethodmodifiers SBMLWriter::setProgramVersion(const std::string& version) "
/**
   * Sets the version of this program, i.e., the program that is about to
   * write out the {@link SBMLDocument}.
   * <p>
   * If the program version and name are set (see
   * {@link SBMLWriter#setProgramName(String name)}), the
   * following XML comment, intended for human consumption, will be written
   * at the beginning of the document:
   * <div class=\'fragment\'><pre>
 &lt;!-- Created by &lt;program name&gt; version &lt;program version&gt;
 on yyyy-MM-dd HH:mm with libSBML version &lt;{@link libsbml} version&gt;. --&gt;
 </pre></div>
   * <p>
   * If the program version and name are not set at some point before
   * calling the writeSBML() methods, no such comment is written out.
   * <p>
   * @param version the version of this program (where \'this program\'
   * refers to program in which libSBML is embedded, not libSBML itself!)
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @see #setProgramName(String name)
   */ public
";


%javamethodmodifiers SBMLWriter::writeSBML(const SBMLDocument* d, const std::string& filename) "
/**
   * Writes the given SBML document to filename.
   * <p>
   * If the given filename ends with the suffix <code>&quot;.gz&quot;</code>
(for example, <code>&quot;myfile.xml.gz&quot;</code>), libSBML assumes the
caller wants the file to be written compressed in <em>gzip</em> format.
Similarly, if the given filename ends with <code>&quot;.zip&quot;</code> or
<code>&quot;.bz2&quot;</code>, libSBML assumes the caller wants the file to
be compressed in <em>zip</em> or <em>bzip2</em> format (respectively).
Files whose names lack these suffixes will be written uncompressed.
<em>Special considerations for the zip format</em>: If the given filename
ends with <code>&quot;.zip&quot;</code>, the file placed in the zip archive
will have the suffix <code>&quot;.xml&quot;</code> or
<code>&quot;.sbml&quot;</code>.  For example, the file in the zip archive
will be named <code>&quot;test.xml&quot;</code> if the given filename is
<code>&quot;test.xml.zip&quot;</code> or <code>&quot;test.zip&quot;</code>.
Similarly, the filename in the archive will be
<code>&quot;test.sbml&quot;</code> if the given filename is
<code>&quot;test.sbml.zip&quot;</code>.

   * <p>
   * @param d the SBML document to be written
   * <p>
   * @param filename the name or full pathname of the file where the SBML
   * is to be written. 
   * <p>
   * @return <code>true</code> on success and <code>false</code> if the filename could not be
   * opened for writing.
   * <p>
   * @note To write a gzip/zip file, libSBML needs to be configured and linked with
the <a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at
compile time.  It also needs to be linked with the <a target=\'_blank\'
href=\'\'>bzip2</a> library to write files in <em>bzip2</em> format.  (Both
of these are the default configurations for libSBML.)  Errors about
unreadable files will be logged and this method will return
<code>false</code> if a compressed filename is given and libSBML was
<em>not</em> linked with the corresponding required library.

   * <p>
   * @see #setProgramVersion(String version)
   * @see #setProgramName(String name)
   */ public
";


%javamethodmodifiers SBMLWriter::writeSBML(const SBMLDocument* d, std::ostream& stream) "
/**
   * Writes the given SBML document to the output stream.
   * <p>
   * @param d the SBML document to be written
   * <p>
   * @param stream the stream object where the SBML is to be written.
   * <p>
   * @return <code>true</code> on success and <code>false</code> if one of the underlying
   * parser components fail (rare).
   * <p>
   * @see #setProgramVersion(String version)
   * @see #setProgramName(String name)
   */ public
";


%javamethodmodifiers SBMLWriter::writeToString(const SBMLDocument* d) "
/**
   * Writes the given SBML document to an in-memory string and returns a
   * pointer to it.
   * <p>
   * The string is owned by the caller and should be freed (with <code>free</code>())
   * when no longer needed.
   * <p>
   * @param d the SBML document to be written
   * <p>
   * @return the string on success and <code>0</code> if one of the underlying parser
   * components fail.
   * <p>
   * @see #setProgramVersion(String version)
   * @see #setProgramName(String name)
   * @internal
   */ public
";


%javamethodmodifiers SBMLWriter::writeSBMLToFile(const SBMLDocument* d, const std::string& filename) "
/**
   * Writes the given SBML document to filename.
   * <p>
   * If the given filename ends with the suffix <code>&quot;.gz&quot;</code>
(for example, <code>&quot;myfile.xml.gz&quot;</code>), libSBML assumes the
caller wants the file to be written compressed in <em>gzip</em> format.
Similarly, if the given filename ends with <code>&quot;.zip&quot;</code> or
<code>&quot;.bz2&quot;</code>, libSBML assumes the caller wants the file to
be compressed in <em>zip</em> or <em>bzip2</em> format (respectively).
Files whose names lack these suffixes will be written uncompressed.
<em>Special considerations for the zip format</em>: If the given filename
ends with <code>&quot;.zip&quot;</code>, the file placed in the zip archive
will have the suffix <code>&quot;.xml&quot;</code> or
<code>&quot;.sbml&quot;</code>.  For example, the file in the zip archive
will be named <code>&quot;test.xml&quot;</code> if the given filename is
<code>&quot;test.xml.zip&quot;</code> or <code>&quot;test.zip&quot;</code>.
Similarly, the filename in the archive will be
<code>&quot;test.sbml&quot;</code> if the given filename is
<code>&quot;test.sbml.zip&quot;</code>.

   * <p>
   * @param d the SBML document to be written
   * <p>
   * @param filename the name or full pathname of the file where the SBML
   * is to be written. 
   * <p>
   * @return <code>true</code> on success and <code>false</code> if the filename could not be
   * opened for writing.
   * <p>
   * @note To write a gzip/zip file, libSBML needs to be configured and linked with
the <a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at
compile time.  It also needs to be linked with the <a target=\'_blank\'
href=\'\'>bzip2</a> library to write files in <em>bzip2</em> format.  (Both
of these are the default configurations for libSBML.)  Errors about
unreadable files will be logged and this method will return
<code>false</code> if a compressed filename is given and libSBML was
<em>not</em> linked with the corresponding required library.

   * <p>
   * @see #setProgramVersion(String version)
   * @see #setProgramName(String name)
   */ public
";


%javamethodmodifiers SBMLWriter::writeSBMLToString(const SBMLDocument* d) "
/**
   * Writes the given SBML document to an in-memory string and returns a
   * pointer to it.
   * <p>
   * The string is owned by the caller and should be freed (with <code>free</code>())
   * when no longer needed.
   * <p>
   * @param d the SBML document to be written
   * <p>
   * @return the string on success and <code>0</code> if one of the underlying parser
   * components fail.
   * <p>
   * @see #setProgramVersion(String version)
   * @see #setProgramName(String name)
   */ public
";


%javamethodmodifiers SBMLWriter::hasZlib "
/**
   * Predicate returning <code>true</code> if this copy of libSBML has been linked
   * with the <em>zlib</em> library.
   * <p>
   * LibSBML supports reading and writing files compressed with either
   * bzip2 or zip/gzip compression.  The facility depends on libSBML having
   * been compiled with the necessary support libraries.  This method
   * allows a calling program to inquire whether that is the case for the
   * copy of libSBML it is using.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link SBMLWriter}), and the other
   * will be a standalone top-level function with the name
   * SBMLWriter_hasZlib(). They are functionally identical. 
   * <p>
   * @return <code>true</code> if libSBML is linked with zlib, <code>false</code> otherwise.
   * <p>
   * @see SBMLWriter#hasBzip2() 
   */ public
";


%javamethodmodifiers SBMLWriter::hasBzip2 "
/**
   * Predicate returning <code>true</code> if this copy of libSBML has been linked
   * with the <em>bzip2</em> library.
   * <p>
   * LibSBML supports reading and writing files compressed with either
   * bzip2 or zip/gzip compression.  The facility depends on libSBML having
   * been compiled with the necessary support libraries.  This method
   * allows a calling program to inquire whether that is the case for the
   * copy of libSBML it is using.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link SBMLWriter}), and the other
   * will be a standalone top-level function with the name
   * SBMLWriter_hasZlib(). They are functionally identical. 
   * <p>
   * @return <code>true</code> if libSBML is linked with bzip2, <code>false</code> otherwise.
   * <p>
   * @see SBMLWriter#hasZlib() 
   */ public
";


%javamethodmodifiers writeSBML(const SBMLDocument_t *d, const char *filename) "
/**
 * Writes the given SBML document <code>d</code> to the file named by <code>filename</code>.
 * <p>
 * This function is identical to <a href=\'#writeSBMLToFile(org.sbml.libsbml.SBMLDocument, java.lang.String)\'><code>writeSBMLToFile(SBMLDocument d, String filename)</code></a>.
 * <p>
 * If the given filename ends with the suffix <code>&quot;.gz&quot;</code>
(for example, <code>&quot;myfile.xml.gz&quot;</code>), libSBML assumes the
caller wants the file to be written compressed in <em>gzip</em> format.
Similarly, if the given filename ends with <code>&quot;.zip&quot;</code> or
<code>&quot;.bz2&quot;</code>, libSBML assumes the caller wants the file to
be compressed in <em>zip</em> or <em>bzip2</em> format (respectively).
Files whose names lack these suffixes will be written uncompressed.
<em>Special considerations for the zip format</em>: If the given filename
ends with <code>&quot;.zip&quot;</code>, the file placed in the zip archive
will have the suffix <code>&quot;.xml&quot;</code> or
<code>&quot;.sbml&quot;</code>.  For example, the file in the zip archive
will be named <code>&quot;test.xml&quot;</code> if the given filename is
<code>&quot;test.xml.zip&quot;</code> or <code>&quot;test.zip&quot;</code>.
Similarly, the filename in the archive will be
<code>&quot;test.sbml&quot;</code> if the given filename is
<code>&quot;test.sbml.zip&quot;</code>.

 * <p>
 * @param d the {@link SBMLDocument} object to be written out in XML format
 * <p>
 * @param filename a string giving the path to a file where the XML
 * content is to be written.
 * <p>
 * @return <code>1</code> on success and <code>0</code> (zero) if <code>filename</code> could not be
 * written.  Some possible reasons for failure include (a) being unable to
 * open the file, and (b) using a filename that indicates a compressed SBML
 * file (i.e., a filename ending in <code>&quot;.zip&quot;</code> or
 * similar) when the compression functionality has not been enabled in
 * the underlying copy of libSBML.
 * <p>
 * <p>
 * <p>
 * @see SBMLWriter#hasZlib()
 * @see SBMLWriter#hasBzip2()
 */ public
";


%javamethodmodifiers writeSBMLToString(const SBMLDocument_t *d) "
/**
 * Writes the given SBML document <code>d</code> to an in-memory string and
 * returns the string.
 * <p>
 * This is a convenience function that uses the
 * {@link SBMLWriter#writeSBMLToString(SBMLDocument d)} method internally,
 * yet does not require the caller to create an {@link SBMLWriter} object first.
 * <p>
 * @param d an {@link SBMLDocument} object to be written out in XML format
 * <p>
 * @return the string on success and <code>null</code> if one of the underlying parser
 * components fail.
 * <p>
 * <p>
 */ public
";


%javamethodmodifiers writeSBMLToFile(const SBMLDocument_t *d, const char *filename) "
/**
 * Writes the given SBML document <code>d</code> to the file <code>filename</code>.
 * <p>
 * This is a convenience function that uses the
 * {@link SBMLWriter#writeSBMLToFile(SBMLDocument d, String
 * filename)} method internally, yet does not require the caller to create
 * an {@link SBMLWriter} object first.
 * <p>
 * If the given filename ends with the suffix <code>&quot;.gz&quot;</code>
(for example, <code>&quot;myfile.xml.gz&quot;</code>), libSBML assumes the
caller wants the file to be written compressed in <em>gzip</em> format.
Similarly, if the given filename ends with <code>&quot;.zip&quot;</code> or
<code>&quot;.bz2&quot;</code>, libSBML assumes the caller wants the file to
be compressed in <em>zip</em> or <em>bzip2</em> format (respectively).
Files whose names lack these suffixes will be written uncompressed.
<em>Special considerations for the zip format</em>: If the given filename
ends with <code>&quot;.zip&quot;</code>, the file placed in the zip archive
will have the suffix <code>&quot;.xml&quot;</code> or
<code>&quot;.sbml&quot;</code>.  For example, the file in the zip archive
will be named <code>&quot;test.xml&quot;</code> if the given filename is
<code>&quot;test.xml.zip&quot;</code> or <code>&quot;test.zip&quot;</code>.
Similarly, the filename in the archive will be
<code>&quot;test.sbml&quot;</code> if the given filename is
<code>&quot;test.sbml.zip&quot;</code>.

 * <p>
 * @param d an {@link SBMLDocument} object to be written out in XML format
 * <p>
 * @param filename a string giving the path to a file where the XML
 * content is to be written.
 * <p>
 * @return <code>1</code> on success and <code>0</code> (zero) if <code>filename</code> could not be
 * written.  Some possible reasons for failure include (a) being unable to
 * open the file, and (b) using a filename that indicates a compressed SBML
 * file (i.e., a filename ending in <code>&quot;.zip&quot;</code> or
 * similar) when the compression functionality has not been enabled in
 * the underlying copy of libSBML.
 * <p>
 * <p>
 * <p>
 * @see SBMLWriter#hasZlib()
 * @see SBMLWriter#hasBzip2()
 */ public
";


%javamethodmodifiers SBMLTypeCode_toString(int tc, const char* pkgName) "
/**
 * This method takes an SBML type code and returns a string representing
 * the code.
 * <p>
 * LibSBML attaches an identifying code to every
 * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
 * other languages, the set of type codes is stored in an enumeration; in
 * the Java language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class {@link
 * libsbmlConstants}.  The names of the type codes all begin with the
 * characters <code>SBML_.</code> 
 * <p>
 * @return a human readable name for the given
 * SBML type code.
 * <p>
 * @note The caller does not own the returned string and is therefore not
 * allowed to modify it.
 */ public
";


%typemap(javaimports) SBase "
/** 
 * LibSBML implementation of SBase, the base class of most SBML objects.
 * <p>
 * Most components in SBML are derived from a single abstract base type,
 * {@link SBase}.  In addition to serving as the parent class for most other
 * classes of objects in SBML, this base type is designed to allow a
 * modeler or a software package to attach arbitrary information to each
 * major element or list in an SBML model.
 * <p>
 * {@link SBase} has an optional subelement called \'notes\'.  It is intended to
 * serve as a place for storing optional information intended to be seen by
 * humans.  An example use of the \'notes\' element would be to contain
 * formatted user comments about the model element in which the \'notes\'
 * element is enclosed.  There are certain conditions on the XHTML content
 * permitted inside the \'notes\' element; please consult the <a
 * target=\'_blank\' href=\'http://sbml.org/Documents/Specifications\'>SBML
 * specification document</a> corresponding to the SBML Level and Version
 * of your model for more information about the requirements for \'notes\'
 * content.
 * <p>
 * {@link SBase} has another optional subelement called \'annotation\'.  Whereas the
 * \'notes\' element described above is a container for content to be shown
 * directly to humans, the \'annotation\' element is a container for optional
 * software-generated content <em>not</em> meant to be shown to humans.  The
 * element\'s content type is <a target=\'_blank\'
 * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
 * \'any\'</a>, allowing essentially arbitrary data content.  SBML places
 * only a few restrictions on the organization of the content; these are
 * intended to help software tools read and write the data as well as help
 * reduce conflicts between annotations added by different tools.  As is
 * the case with \'notes\', it is important to refer to the <a
 * target=\'_blank\' href=\'http://sbml.org/Documents/Specifications\'>SBML
 * specification document</a> corresponding to the SBML Level and Version
 * of your model for more information about the requirements for
 * \'annotation\' content.
 * <p>
 * It is worth pointing out that the \'annotation\' element in the definition
 * of {@link SBase} exists in order that software developers may attach optional
 * application-specific data to the elements in an SBML model.  However, it
 * is important that this facility not be misused.  In particular, it is
 * <em>critical</em> that data essential to a model definition or that can
 * be encoded in existing SBML elements is <em>not</em> stored in
 * \'annotation\'. {@link Parameter} values, functional dependencies between model
 * elements, etc., should not be recorded as annotations.  It is crucial to
 * keep in mind the fact that data placed in annotations can be freely
 * ignored by software applications.  If such data affects the
 * interpretation of a model, then software interoperability is greatly
 * impeded.
 * <p>
 * SBML Level 2 introduced an optional {@link SBase} attribute named \'metaid\' for
 * supporting metadata annotations using RDF (<a target=\'_blank\'
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>). The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML identifier
 * type, which means each \'metaid\' value must be globally unique within an
 * SBML file.  (Importantly, this uniqueness criterion applies across any
 * attribute with type <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML
 * ID</a>, not just the \'metaid\' attribute used by SBML&mdash;something to
 * be aware of if your application-specific XML content inside the
 * \'annotation\' subelement happens to use <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)  The \'metaid\' value
 * serves to identify a model component for purposes such as referencing
 * that component from metadata placed within \'annotation\' subelements.
 * <p>
 * Beginning with SBML Level 2 Version 3, {@link SBase} also has an optional
 * attribute named \'sboTerm\' for supporting the use of the Systems Biology
 * Ontology.  In SBML proper, the data type of the attribute is a string of
 * the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a seven digit integer number;
 * libSBML simplifies the representation by only storing the \'NNNNNNN\'
 * integer portion.  Thus, in libSBML, the \'sboTerm\' attribute on {@link SBase} has
 * data type <code>int</code>, and {@link SBO} identifiers are stored simply as integers.
 * (For convenience, {@link SBase} offers methods for returning both the integer
 * form and a text-string form of the {@link SBO} identifier.)  {@link SBO} terms are a
 * type of optional annotation, and each different class of SBML object
 * derived from {@link SBase} imposes its own requirements about the values
 * permitted for \'sboTerm\'.  Please consult the SBML Level&nbsp;2
 * Version&nbsp;4 specification for more information about the use of {@link SBO}
 * and the \'sboTerm\' attribute.
 * <p>
 * Finally, note that, in the list of methods on {@link SBase}, there is no public
 * constructor because {@link SBase} is an abstract class.  The constructors reside
 * in the subclasses derived from {@link SBase}.
 * <p>
 * <p>
 * <h2>Standard format for annotations linking data resources</h2>
 * <p>
 * SBML Level 2 Versions 2, 3 and 4, and Level&nbsp;3, define a proposed
 * regular format for encoding two particular categories of annotations:
 * (a) references to controlled vocabulary terms and database identifiers
 * which define and describe biological and biochemical entities in a
 * model; and (b) descriptions of the provenance of a model, including its
 * author(s) and modification history.
 */
"


%javamethodmodifiers SBase::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this {@link SBase} object.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */ public
";


%javamethodmodifiers SBase::clone() const "
/**
   * Creates and returns a deep copy of this {@link SBase} object.
   * <p>
   * @return a (deep) copy of this {@link SBase} object.
   */ public
";


%javamethodmodifiers SBase::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given <code>id</code> in the
   * model-wide <code>SId</code> namespace, or <code>null</code> if no such object is found.
   * <p>
   * @param id string representing the \'id\' attribute value of the object
   * to find.
   * <p>
   * @return pointer to the first element found with the given identifier.
   */ public
";


%javamethodmodifiers SBase::getElementByMetaId(std::string metaid) "
/**
   * Returns the first child element it can find with a specific \'metaid\'
   * attribute value, or <code>null</code> if no such object is found.
   * <p>
   * @param metaid string representing the \'metaid\' attribute value of the
   * object to find.
   * <p>
   * @return pointer to the first element found with the given meta-identifier.
   */ public
";


%javamethodmodifiers SBase::getAllElements "
/**
   * Returns a {@link List} of all child {@link SBase} objects, including those nested to
   * an arbitrary depth.
   * <p>
   * @return a pointer to a {@link List} of pointers to all children objects.
   */ public
";


%javamethodmodifiers SBase::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the <code>SIdRef</code> attributes on this element, including any
   * found in MathML content (if such exists).
   * <p>
   * This method works by looking at all attributes and (if appropriate)
   * mathematical formulas, comparing the identifiers to the value of 
   * <code>oldid</code>.  If any matches are found, the matching identifiers are replaced
   * with <code>newid</code>.  The method does <em>not</em> descend into child elements.
   * <p>
   * @param oldid the old identifier
   * @param newid the new identifier
   */ public
";


%javamethodmodifiers SBase::renameMetaIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the <code>MetaIdRef</code> attributes on this element.
   * <p>
   * This method works by looking at all meta-attribute values, comparing
   * the identifiers to the value of <code>oldid</code>.  If any matches are found,
   * the matching identifiers are replaced with <code>newid</code>.  The method does
   * <em>not</em> descend into child elements.
   * <p>
   * @param oldid the old identifier
   * @param newid the new identifier
   */ public
";


%javamethodmodifiers SBase::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the <code>UnitSIdRef</code> attributes on this element.
   * <p>
   * This method works by looking at all unit identifier attribute values
   * (including, if appropriate, inside mathematical formulas), comparing the
   * unit identifiers to the value of <code>oldid</code>.  If any matches are found,
   * the matching identifiers are replaced with <code>newid</code>.  The method does
   * <em>not</em> descend into child elements.
   * <p>
   * @param oldid the old identifier
   * @param newid the new identifier
   */ public
";


%javamethodmodifiers SBase::replaceSIDWithFunction(const std::string& id, const ASTNode* function) "
/**
   * If this object has a child \'math\' object (or anything with ASTNodes in
   * general), replace all nodes with the name \'id\' with the provided
   * function.
   * <p>
   * @note This function does nothing itself&mdash;subclasses with {@link ASTNode}
   * subelements must override this function.
   * @internal
   */ public
";


%javamethodmodifiers SBase::divideAssignmentsToSIdByFunction(const std::string& id, const ASTNode* function) "
/**
   * If the function of this object is to assign a value has a child \'math\'
   * object (or anything with ASTNodes in general), replace the \'math\'
   * object with the function (existing/function).
   * <p>
   * @note This function does nothing itself&mdash;subclasses with {@link ASTNode}
   * subelements must override this function.
   * @internal
   */ public
";


%javamethodmodifiers SBase::multiplyAssignmentsToSIdByFunction(const std::string& id, const ASTNode* function) "
/**
   * If this assignment assigns a value to the \'id\' element, replace the \'math\' object with the function (existing*function). 
   * @internal 
   */ public
";


%javamethodmodifiers SBase::getElementFromPluginsBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the
   * model-wide SId namespace from all plug-ins associated with this
   * element, or <code>null</code> if no such object is found.
   * <p>
   * @param id string representing the id of objects to find
   * <p>
   * @return pointer to the first element found with the given id.
   * @internal
   */ public
";


%javamethodmodifiers SBase::getElementFromPluginsByMetaId(std::string metaid) "
/**
   * Returns the first child element it can find with the given metaid from
   * all plug-ins associated with this element, or <code>null</code> if no such object
   * is found.
   * <p>
   * @param metaid string representing the metaid of objects to find
   * <p>
   * @return pointer to the first element found with the given metaid.
   * @internal
   */ public
";


%javamethodmodifiers SBase::hasNonstandardIdentifierBeginningWith(const std::string& prefix) "
/**
   * Check to see if the given prefix is used by any of the IDs defined by
   * extension elements *excluding* \'id\' and \'metaid\' attributes (as, for
   * example, the spatial id attributes \'spid\').
   * @internal
   */ public
";


%javamethodmodifiers SBase::prependStringToAllIdentifiers(const std::string& prefix) "
/**
   * Add the given string to all identifiers in the object.  If the string
   * is added to anything other than an id or a metaid, this code is
   * responsible for tracking down and renaming all *idRefs in the package
   * extention that identifier comes from.
   * @internal
   */ public
";


%javamethodmodifiers SBase::getAllElementsFromPlugins "
/**
   * Returns a {@link List} of all child {@link SBase} objects contained in SBML package
   * plug-ins.
   * <p>
   * This method walks down the list of all packages used by the enclosing
   * {@link Model} and returns all objects contained in them.
   * <p>
   * @return a pointer to a {@link List} of pointers to all children objects from
   * plug-ins.
   */ public
";


%javamethodmodifiers SBase::getMetaId() const "
/**
   * Returns the value of the \'metaid\' attribute of this object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * @return the meta-identifier of this SBML object.
   * <p>
   * @see #isSetMetaId()
   * @see #setMetaId(String metaid)
   */ public
";


%javamethodmodifiers SBase::getMetaId "
/**
   * Returns the value of the \'metaid\' attribute of this object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format).  The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * @return the meta-identifier of this SBML object, as a string.
   * <p>
   * @see #isSetMetaId()
   * @see #setMetaId(String metaid)
   */ public
";


%javamethodmodifiers SBase::getId() const "
/**
   * Returns the value of the \'metaid\' attribute of this object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format).  The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * @return the meta-identifier of this SBML object, as a string.
   * <p>
   * @see #isSetMetaId()
   * @see #setMetaId(String metaid)
   * @internal
   */ public
";


%javamethodmodifiers SBase::getName() const "
/**
   * Returns the value of the \'metaid\' attribute of this object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format).  The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model
   * component for purposes such as referencing that component from
   * metadata placed within \'annotation\' subelements.
   * <p>
   * @return the meta-identifier of this SBML object, as a string.
   * <p>
   * @see #isSetMetaId()
   * @see #setMetaId(String metaid)
   * @internal
   */ public
";


%javamethodmodifiers SBase::getNotes "
/**
   * Returns the content of the \'notes\' subelement of this object as
   * a tree of {@link XMLNode} objects.
   * <p>
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type {@link SBase} can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   * <p>
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method {@link SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)}; however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and &nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * <p>
   * The \'notes\' element content returned by this method will be in XML
   * form, but libSBML does not provide an object model specifically for
   * the content of notes.  Callers will need to traverse the XML tree
   * structure using the facilities available on {@link XMLNode} and related
   * objects.  For an alternative method of accessing the notes, see
   * getNotesString().
   * <p>
   * @return the content of the \'notes\' subelement of this SBML object as a
   * tree structure composed of {@link XMLNode} objects.
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   * @see SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)
   */ public
";


%javamethodmodifiers SBase::getNotesString "
/**
   * Returns the content of the \'notes\' subelement of this object as a
   * string.
   * <p>
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type {@link SBase} can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   * <p>
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method {@link SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)}; however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and &nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * <p>
   * For an alternative method of accessing the notes, see getNotes(),
   * which returns the content as an {@link XMLNode} tree structure.  Depending on
   * an application\'s needs, one or the other method may be more
   * convenient.
   * <p>
   * @return the content of the \'notes\' subelement of this SBML object as a
   * string.
   * <p>
   * @see #getNotes()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   * @see SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)
   */ public
";


%javamethodmodifiers SBase::getAnnotation "
/**
   * Returns the content of the \'annotation\' subelement of this object as
   * a tree of {@link XMLNode} objects.
   * <p>
   * Whereas the SBML \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * The annotations returned by this method will be in XML form.  LibSBML
   * provides an object model and related interfaces for certain specific
   * kinds of annotations, namely model history information and RDF
   * content.  See the {@link ModelHistory}, {@link CVTerm} and {@link RDFAnnotationParser} classes
   * for more information about the facilities available.
   * <p>
   * @return the annotation of this SBML object as a tree of {@link XMLNode} objects.
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */ public
";


%javamethodmodifiers SBase::getAnnotationString "
/**
   * Returns the content of the \'annotation\' subelement of this object as a
   * character string.
   * <p>
   * Whereas the SBML \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * The annotations returned by this method will be in string form.
   * <p>
   * @return the annotation of this SBML object as a character string.
   * <p>
   * @see #getAnnotation()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */ public
";


%javamethodmodifiers SBase::getNamespaces() const "
/**
   * Returns a list of the XML Namespaces declared on this SBML document.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in SBML Level&nbsp;3) packages used in addition to SBML Level&nbsp;3
   * Core.
   * <p>
   * @return the XML Namespaces associated with this SBML object
   * <p>
   * @see #getLevel()
   * @see #getVersion()
   */ public
";


%javamethodmodifiers SBase::getSBMLDocument() const "
/**
   * Returns the {@link SBMLDocument} object containing <em>this</em> object instance.
   * <p>
   * LibSBML uses the class {@link SBMLDocument} as a top-level container for
   * storing SBML content and data associated with it (such as warnings and
   * error messages).  An SBML model in libSBML is contained inside an
   * {@link SBMLDocument} object.  {@link SBMLDocument} corresponds roughly to the class
   * <i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
   * specifications, but it does not have a direct correspondence in SBML
   * Level&nbsp;1.  (But, it is created by libSBML no matter whether the
   * model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)
   * <p>
   * This method allows the caller to obtain the {@link SBMLDocument} for the
   * current object.
   * <p>
   * @return the parent {@link SBMLDocument} object of this SBML object.
   * <p>
   * @see #getParentSBMLObject()
   * @see #getModel()
   */ public
";


%javamethodmodifiers SBase::getSBMLDocument "
/**
   * Returns the {@link SBMLDocument} object containing <em>this</em> object instance.
   * <p>
   * LibSBML uses the class {@link SBMLDocument} as a top-level container for
   * storing SBML content and data associated with it (such as warnings and
   * error messages).  An SBML model in libSBML is contained inside an
   * {@link SBMLDocument} object.  {@link SBMLDocument} corresponds roughly to the class
   * <i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
   * specifications, but it does not have a direct correspondence in SBML
   * Level&nbsp;1.  (But, it is created by libSBML no matter whether the
   * model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)
   * <p>
   * This method allows the caller to obtain the {@link SBMLDocument} for the
   * current object.
   * <p>
   * @return the parent {@link SBMLDocument} object of this SBML object.
   * <p>
   * @see #getParentSBMLObject()
   * @see #getModel()
   */ public
";


%javamethodmodifiers SBase::getParentSBMLObject "
/**
   * Returns the parent SBML object containing this object.
   * <p>
   * This returns the immediately-containing object.  This method is
   * convenient when holding an object nested inside other objects in an
   * SBML model.  
   * <p>
   * @return the parent SBML object of this SBML object.
   * <p>
   * @see #getSBMLDocument()
   * @see #getModel()
   */ public
";


%javamethodmodifiers SBase::getParentSBMLObject() const "
/**
   * Returns the parent SBML object containing this object.
   * <p>
   * This returns the immediately-containing object.  This method is
   * convenient when holding an object nested inside other objects in an
   * SBML model.  
   * <p>
   * @return the parent SBML object of this SBML object.
   * <p>
   * @see #getSBMLDocument()
   * @see #getModel()
   */ public
";


%javamethodmodifiers SBase::getAncestorOfType(int type, const std::string pkgName = "core") "
/**
   * Returns the first ancestor object that has the given SBML type code.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * This method searches the tree of objects that are parents of this
   * object, and returns the first one that has the given SBML type code.
   * If the optional argument <code>pkgName</code> is given, it will cause the search
   * to be limited to the SBML Level&nbsp;3 package given.
   * <p>
   * @param type the SBML type code of the object sought
   * <p>
   * @param pkgName (optional) the short name of an SBML Level&nbsp;3
   * package to which the sought-after object must belong
   * <p>
   * @return the ancestor SBML object of this SBML object that corresponds
   * to the given SBML object type
   * code, or <code>null</code> if no ancestor exists.
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   */ public
";


%javamethodmodifiers SBase::getAncestorOfType(int type, const std::string pkgName = "core") const "
/**
   * Returns the first ancestor object that has the given SBML type code.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * This method searches the tree of objects that are parents of this
   * object, and returns the first one that has the given SBML type code.
   * If the optional argument <code>pkgName</code> is given, it will cause the search
   * to be limited to the SBML Level&nbsp;3 package given.
   * <p>
   * @param type the SBML type code of the object sought
   * <p>
   * @param pkgName (optional) the short name of an SBML Level&nbsp;3
   * package to which the sought-after object must belong
   * <p>
   * @return the ancestor SBML object of this SBML object that corresponds
   * to the given SBML object type
   * code, or <code>null</code> if no ancestor exists.
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   */ public
";


%javamethodmodifiers SBase::getSBOTerm() const "
/**
   * Returns the integer portion of the value of the \'sboTerm\' attribute of
   * this object.
   * <p>
   * Beginning with SBML Level 2 Version 3, objects derived from {@link SBase} have
   * an optional attribute named \'sboTerm\' for supporting the use of the
   * Systems Biology Ontology.  In SBML proper, the data type of the
   * attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
   * seven digit integer number; libSBML simplifies the representation by
   * only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
   * \'sboTerm\' attribute on {@link SBase} has data type <code>int</code>, and {@link SBO} identifiers
   * are stored simply as integers.  (For convenience, libSBML offers
   * methods for returning both the integer form and a text-string form of
   * the {@link SBO} identifier.)
   * <p>
   * {@link SBO} terms are a type of optional annotation, and each different class
   * of SBML object derived from {@link SBase} imposes its own requirements about
   * the values permitted for \'sboTerm\'.  Please consult the SBML
   * Level&nbsp;2 Version&nbsp;4 specification for more information about
   * the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * @return the value of the \'sboTerm\' attribute as an integer, or <code>-1</code>
   * if the value is not set.
   */ public
";


%javamethodmodifiers SBase::getSBOTermID() const "
/**
   * Returns the string representation of the \'sboTerm\' attribute of
   * this object.
   * <p>
   * Beginning with SBML Level 2 Version 3, objects derived from {@link SBase} have
   * an optional attribute named \'sboTerm\' for supporting the use of the
   * Systems Biology Ontology.  In SBML proper, the data type of the
   * attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
   * seven digit integer number; libSBML simplifies the representation by
   * only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
   * \'sboTerm\' attribute on {@link SBase} has data type <code>int</code>, and {@link SBO} identifiers
   * are stored simply as integers.  This method returns the entire {@link SBO}
   * identifier as a text string in the form \'SBO:NNNNNNN\'.
   * <p>
   * {@link SBO} terms are a type of optional annotation, and each different class
   * of SBML object derived from {@link SBase} imposes its own requirements about
   * the values permitted for \'sboTerm\'.  Please consult the SBML
   * Level&nbsp;2 Version&nbsp;4 specification for more information about
   * the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * @return the value of the \'sboTerm\' attribute as a string (its value
   * will be of the form \'SBO:NNNNNNN\'), or an empty string if
   * the value is not set.
   */ public
";


%javamethodmodifiers SBase::getSBOTermAsURL() const "
/**
   * Returns the identifiers.org URL representation of the \'sboTerm\' attribute of
   * this object.
   * <p>
   * This method returns the entire {@link SBO}
   * identifier as a text string in the form 
   * \'http://identifiers.org/biomodels.sbo/SBO:NNNNNNN\'.
   * <p>
   * {@link SBO} terms are a type of optional annotation, and each different class
   * of SBML object derived from {@link SBase} imposes its own requirements about
   * the values permitted for \'sboTerm\'.  Please consult the SBML
   * Level&nbsp;2 Version&nbsp;4 specification for more information about
   * the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * @return the value of the \'sboTerm\' attribute as an identifiers.org URL
   * (its value will be of the form 
   * \'http://identifiers.org/biomodels.sbo/SBO:NNNNNNN\'), or an empty string if
   * the value is not set.
   */ public
";


%javamethodmodifiers SBase::getLine() const "
/**
   * Returns the line number on which this object first appears in the XML
   * representation of the SBML document.
   * <p>
   * @return the line number of this SBML object.
   * <p>
   * @note The line number for each construct in an SBML model is set upon
   * reading the model.  The accuracy of the line number depends on the
   * correctness of the XML representation of the model, and on the
   * particular XML parser library being used.  The former limitation
   * relates to the following problem: if the model is actually invalid
   * XML, then the parser may not be able to interpret the data correctly
   * and consequently may not be able to establish the real line number.
   * The latter limitation is simply that different parsers seem to have
   * their own accuracy limitations, and out of all the parsers supported
   * by libSBML, none have been 100% accurate in all situations. (At this
   * time, libSBML supports the use of <a target=\'_blank\'
   * href=\'http://xmlsoft.org\'>libxml2</a>, <a target=\'_blank\'
   * href=\'http://expat.sourceforge.net/\'>Expat</a> and <a target=\'_blank\'
   * href=\'http://xerces.apache.org/xerces-c/\'>Xerces</a>.)
   * <p>
   * @see #getColumn()
   */ public
";


%javamethodmodifiers SBase::getColumn() const "
/**
   * Returns the column number on which this object first appears in the XML
   * representation of the SBML document.
   * <p>
   * @return the column number of this SBML object.
   * <p>
   * @note The column number for each construct in an SBML model is set
   * upon reading the model.  The accuracy of the column number depends on
   * the correctness of the XML representation of the model, and on the
   * particular XML parser library being used.  The former limitation
   * relates to the following problem: if the model is actually invalid
   * XML, then the parser may not be able to interpret the data correctly
   * and consequently may not be able to establish the real column number.
   * The latter limitation is simply that different parsers seem to have
   * their own accuracy limitations, and out of all the parsers supported
   * by libSBML, none have been 100% accurate in all situations. (At this
   * time, libSBML supports the use of <a target=\'_blank\'
   * href=\'http://xmlsoft.org\'>libxml2</a>, <a target=\'_blank\'
   * href=\'http://expat.sourceforge.net/\'>Expat</a> and <a target=\'_blank\'
   * href=\'http://xerces.apache.org/xerces-c/\'>Xerces</a>.)
   * <p>
   * @see #getLine()
   */ public
";


%javamethodmodifiers SBase::getModelHistory() const "
/**
   * Returns the {@link ModelHistory} object, if any, attached to this object.
   * <p>
   * @return the {@link ModelHistory} object attached to this object, or <code>null</code> if
   * none exist.
   * <p>
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the {@link Model} element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from {@link SBase}.
   */ public
";


%javamethodmodifiers SBase::getModelHistory "
/**
   * Returns the {@link ModelHistory} object, if any, attached to this object.
   * <p>
   * @return the {@link ModelHistory} object attached to this object, or <code>null</code> if
   * none exist.
   * <p>
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the {@link Model} element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from {@link SBase}.
   */ public
";


%javamethodmodifiers SBase::isSetMetaId() const "
/**
   * Predicate returning <code>true</code> if this object\'s \'metaid\' attribute is set.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model component for purposes
   * such as referencing that component from metadata placed within
   * \'annotation\' subelements.
   * <p>
   * @return <code>true</code> if the \'metaid\' attribute of this SBML object is
   * set, <code>false</code> otherwise.
   * <p>
   * @see #getMetaId()
   * @see #setMetaId(String metaid)
   */ public
";


%javamethodmodifiers SBase::isSetId() const "
/**
   * Predicate returning <code>true</code> if this object\'s \'metaid\' attribute is set.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model component for purposes
   * such as referencing that component from metadata placed within
   * \'annotation\' subelements.
   * <p>
   * @return <code>true</code> if the \'metaid\' attribute of this SBML object is
   * set, <code>false</code> otherwise.
   * <p>
   * @see #getMetaId()
   * @see #setMetaId(String metaid)
   * @internal
   */ public
";


%javamethodmodifiers SBase::isSetName() const "
/**
   * Predicate returning <code>true</code> if this object\'s \'metaid\' attribute is set.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model component for purposes
   * such as referencing that component from metadata placed within
   * \'annotation\' subelements.
   * <p>
   * @return <code>true</code> if the \'metaid\' attribute of this SBML object is
   * set, <code>false</code> otherwise.
   * <p>
   * @see #getMetaId()
   * @see #setMetaId(String metaid)
   * @internal
   */ public
";


%javamethodmodifiers SBase::isSetNotes() const "
/**
   * Predicate returning <code>true</code> if this
   * object\'s \'notes\' subelement exists and has content.
   * <p>
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type {@link SBase} can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   * <p>
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method {@link SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)}; however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and &nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * <p>
   * @return <code>true</code> if a \'notes\' subelement exists, <code>false</code> otherwise.
   * <p>
   * @see #getNotes()
   * @see #getNotesString()
   * @see #setNotes(XMLNode notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   * @see SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)
   */ public
";


%javamethodmodifiers SBase::isSetAnnotation() const "
/**
   * Predicate returning <code>true</code> if this
   * object\'s \'annotation\' subelement exists and has content.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * @return <code>true</code> if a \'annotation\' subelement exists, <code>false</code>
   * otherwise.
   * <p>
   * @see #getAnnotation()
   * @see #getAnnotationString()
   * @see #setAnnotation(XMLNode annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */ public
";


%javamethodmodifiers SBase::isSetSBOTerm() const "
/**
   * Predicate returning <code>true</code> if this
   * object\'s \'sboTerm\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'sboTerm\' attribute of this SBML object is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers SBase::setMetaId(const std::string& metaid) "
/**
   * Sets the value of the \'metaid\' attribute of this object.
   * <p>
   * The string <code>metaid</code> is copied.  The value of <code>metaid</code> must be an
   * identifier conforming to the syntax defined by the XML 1.0 data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>ID</a>.  Among other
   * things, this type requires that a value is unique among all the values
   * of type XML ID in an {@link SBMLDocument}.  Although SBML only uses <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a> for the \'metaid\'
   * attribute, callers should be careful if they use
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>\'s in XML
   * portions of a model that are not defined by SBML, such as in the
   * application-specific content of the \'annotation\' subelement.
   * <p>
   * @param metaid the identifier string to use as the value of the
   * \'metaid\' attribute
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @see #getMetaId()
   * @see #isSetMetaId()
   */ public
";


%javamethodmodifiers SBase::isSetModelHistory "
/**
   * Predicate returning <code>true</code> if this
   * object has a {@link ModelHistory} object attached to it.
   * <p>
   * @return <code>true</code> if the {@link ModelHistory} of this object is set, 
   * <code>false</code> otherwise.
   * <p>
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the {@link Model} element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from {@link SBase}.
   */ public
";


%javamethodmodifiers SBase::setId(const std::string& sid) "
/**
   * Predicate returning <code>true</code> if this
   * object has a {@link ModelHistory} object attached to it.
   * <p>
   * @return <code>true</code> if the {@link ModelHistory} of this object is set, 
   * <code>false</code> otherwise.
   * <p>
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the {@link Model} element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from {@link SBase}.
   * @internal
   */ public
";


%javamethodmodifiers SBase::setName(const std::string& name) "
/**
   * Predicate returning <code>true</code> if this
   * object has a {@link ModelHistory} object attached to it.
   * <p>
   * @return <code>true</code> if the {@link ModelHistory} of this object is set, 
   * <code>false</code> otherwise.
   * <p>
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the {@link Model} element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from {@link SBase}.
   * @internal
   */ public
";


%javamethodmodifiers SBase::setAnnotation(const XMLNode* annotation) "
/**
   * Sets the value of the \'annotation\' subelement of this SBML object.
   * <p>
   * The content of <code>annotation</code> is copied, and any previous content of
   * this object\'s \'annotation\' subelement is deleted.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Call this method will result in any existing content of the
   * \'annotation\' subelement to be discarded.  Unless you have taken steps
   * to first copy and reconstitute any existing annotations into the 
   * <code>annotation</code> that is about to be assigned, it is likely that performing
   * such wholesale replacement is unfriendly towards other software
   * applications whose annotations are discarded.  An alternative may be
   * to use {@link SBase#appendAnnotation(XMLNode annotation)} or
   * {@link SBase#appendAnnotation(String annotation)}.
   * <p>
   * @param annotation an XML structure that is to be used as the new content
   * of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */ public
";


%javamethodmodifiers SBase::setAnnotation(const std::string& annotation) "
/**
   * Sets the value of the \'annotation\' subelement of this SBML object.
   * <p>
   * The content of <code>annotation</code> is copied, and any previous content of
   * this object\'s \'annotation\' subelement is deleted.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Call this method will result in any existing content of the
   * \'annotation\' subelement to be discarded.  Unless you have taken steps
   * to first copy and reconstitute any existing annotations into the 
   * <code>annotation</code> that is about to be assigned, it is likely that performing
   * such wholesale replacement is unfriendly towards other software
   * applications whose annotations are discarded.  An alternative may be
   * to use {@link SBase#appendAnnotation(XMLNode annotation)} or
   * {@link SBase#appendAnnotation(String annotation)}.
   * <p>
   * @param annotation an XML string that is to be used as the content
   * of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode annotation)
   * @see #appendAnnotation(XMLNode annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */ public
";


%javamethodmodifiers SBase::appendAnnotation(const XMLNode* annotation) "
/**
   * Appends the given <code>annotation</code> to the \'annotation\' subelement of this
   * object.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is <a
   * target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type \'any\'</a>,
   * allowing essentially arbitrary well-formed XML data content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Unlike {@link SBase#setAnnotation(XMLNode annotation)} or
   * {@link SBase#setAnnotation(String annotation)}, this method
   * allows other annotations to be preserved when an application adds its
   * own data.
   * <p>
   * @param annotation an XML structure that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(String annotation)
   * @see #unsetAnnotation()
   */ public
";


%javamethodmodifiers SBase::appendAnnotation(const std::string& annotation) "
/**
   * Appends the given <code>annotation</code> to the \'annotation\' subelement of this
   * object.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is <a
   * target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type \'any\'</a>,
   * allowing essentially arbitrary well-formed XML data content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Unlike {@link SBase#setAnnotation(XMLNode annotation)} or
   * {@link SBase#setAnnotation(String annotation)}, this method
   * allows other annotations to be preserved when an application adds its
   * own data.
   * <p>
   * @param annotation an XML string that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode annotation)
   * @see #unsetAnnotation()
   */ public
";


%javamethodmodifiers SBase::removeTopLevelAnnotationElement(const std::string elementName, const std::string elementURI = "") "
/**
   * Removes the top-level element within the \'annotation\' subelement of this
   * SBML object with the given name and optional URI.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * Calling this method allows a particular annotation element to be removed
   * whilst the remaining annotations remain intact.
   * <p>
   * @param elementName a string representing the name of the top level
   * annotation element that is to be removed
   * @param elementURI an optional string that is used to check both the name
   * and URI of the top level element to be removed
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * <li> {@link  libsbmlConstants#LIBSBML_ANNOTATION_NAME_NOT_FOUND LIBSBML_ANNOTATION_NAME_NOT_FOUND }
   * <li> {@link  libsbmlConstants#LIBSBML_ANNOTATION_NS_NOT_FOUND LIBSBML_ANNOTATION_NS_NOT_FOUND }
   * </ul>
   * <p>
   * @see #replaceTopLevelAnnotationElement(XMLNode )
   * @see #replaceTopLevelAnnotationElement(String)
   */ public
";


%javamethodmodifiers SBase::replaceTopLevelAnnotationElement(const XMLNode* annotation) "
/**
   * Replaces the given top-level element within the \'annotation\' 
   * subelement of this SBML object and with the annotation element supplied.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * This method determines the name of the element to be replaced from the
   * annotation argument. Functionally it is equivalent to calling
   * <code> removeTopLevelAnnotationElement(name); appendAnnotation(annotation_with_name);
   * </code> with the exception that the placement of the annotation element remains
   * the same.
   * <p>
   * @param annotation {@link XMLNode} representing the replacement top level annotation 
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   * <p>
   * @see #removeTopLevelAnnotationElement(String elementName, String elementURI)
   * @see #replaceTopLevelAnnotationElement(String)
   */ public
";


%javamethodmodifiers SBase::replaceTopLevelAnnotationElement(const std::string& annotation) "
/**
   * Replaces the given top-level element within the \'annotation\' 
   * subelement of this SBML object and with the annotation element supplied.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * This method determines the name of the element to be replaced from the
   * annotation argument. Functionally it is equivalent to calling
   * <code> removeTopLevelAnnotationElement(name); appendAnnotation(annotation_with_name);
   * </code> with the exception that the placement of the annotation element remains
   * the same.
   * <p>
   * @param annotation string representing the replacement top level annotation 
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   * <p>
   * @see #removeTopLevelAnnotationElement(String elementName, String elementURI)
   * @see #replaceTopLevelAnnotationElement(XMLNode)
   */ public
";


%javamethodmodifiers SBase::setNotes(const XMLNode* notes) "
/**
   * Sets the value of the \'notes\' subelement of this SBML object.
   * <p>
   * The content of <code>notes</code> is copied, and any existing content of this
   * object\'s \'notes\' subelement is deleted.
   * <p>
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type {@link SBase} can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   * <p>
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method {@link SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)}; however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and &nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * <p>
   * @param notes an XML structure that is to be used as the content of the
   * \'notes\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   * @see SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)
   */ public
";


%javamethodmodifiers SBase::setNotes(const std::string& notes) "
/**
   * Sets the value of the \'notes\' subelement of this SBML object to a copy
   * of the string <code>notes</code>.
   * <p>
   * The content of <code>notes</code> is copied, and any existing content of this
   * object\'s \'notes\' subelement is deleted.
   * <p>
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type {@link SBase} can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   * <p>
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method {@link SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)}; however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and &nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * <p>
   * The following code illustrates a very simple way of setting the notes
   * using this method.  Here, the object being annotated is the whole SBML
   * document, but that is for illustration purposes only; you could of
   * course use this same approach to annotate any other SBML component.
   * <div class=\'fragment\'><pre>
{@link SBMLDocument} s = new {@link SBMLDocument}(3, 1);
s.setNotes(\'&lt;body xmlns=\'http://www.w3.org/1999/xhtml\'&gt;here is my note&lt;/p&gt;&lt;/body&gt;\');
</pre></div>
   * <p>
   * <p>
   * @param notes an XML string that is to be used as the content of the
   * \'notes\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode notes)
   * @see #appendNotes(XMLNode notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   * @see SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)
   */ public
";


%javamethodmodifiers SBase::appendNotes(const XMLNode* notes) "
/**
   * Appends the given <code>notes</code> to the \'notes\' subelement of this object.
   * <p>
   * The content of <code>notes</code> is copied.
   * <p>
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type {@link SBase} can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   * <p>
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method {@link SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)}; however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and &nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * <p>
   * @param notes an XML node structure that is to appended to the content
   * of the \'notes\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(String notes)
   * @see #unsetNotes()
   * @see SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)
   */ public
";


%javamethodmodifiers SBase::appendNotes(const std::string& notes) "
/**
   * Appends the given <code>notes</code> to the \'notes\' subelement of this object.
   * <p>
   * The content of the parameter <code>notes</code> is copied.
   * <p>
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type {@link SBase} can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   * <p>
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method {@link SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)}; however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and &nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * <p>
   * @param notes an XML string that is to appended to the content of
   * the \'notes\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode notes)
   * @see #unsetNotes()
   * @see SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)
   */ public
";


%javamethodmodifiers SBase::setModelHistory(ModelHistory * history) "
/**
   * Sets the {@link ModelHistory} of this object.
   * <p>
   * The content of <code>history</code> is copied, and this object\'s existing model
   * history content is deleted.
   * <p>
   * @param history {@link ModelHistory} of this object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   * <p>
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the {@link Model} element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from {@link SBase}.
   */ public
";


%javamethodmodifiers SBase::setSBMLDocument(SBMLDocument* d) "
/**
   * Sets the parent {@link SBMLDocument} of this SBML object.
   * <p>
   * @param d the {@link SBMLDocument} object to use
   * <p>
   * @see #connectToChild()
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers SBase::connectToParent(SBase* parent) "
/**
   * Sets the parent SBML object of this SBML object.
   * (Creates a child-parent relationship by the child)
   * This function is called when a child element is
   * set/added/created by its parent element (e.g. by setXXX,
   * addXXX, createXXX, and connectToChild functions of the
   * parent element).
   * <p>
   * @param parent the SBML object to use
   * @internal
   */ public
";


%javamethodmodifiers SBase::connectToChild "
/**
   * Sets this SBML object to child SBML objects (if any).
   * (Creates a child-parent relationship by the parent)
   * <p>
   * Subclasses must override this function if they define
   * one ore more child elements.
   * Basically, this function needs to be called in
   * constructor, copy constructor, assignment operator.
   * <p>
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers SBase::setSBOTerm(int value) "
/**
   * Sets the value of the \'sboTerm\' attribute.
   * <p>
   * Beginning with SBML Level 2 Version 3, objects derived from {@link SBase} have
   * an optional attribute named \'sboTerm\' for supporting the use of the
   * Systems Biology Ontology.  In SBML proper, the data type of the
   * attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
   * seven digit integer number; libSBML simplifies the representation by
   * only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
   * \'sboTerm\' attribute on {@link SBase} has data type <code>int</code>, and {@link SBO} identifiers
   * are stored simply as integers. 
   * <p>
   * {@link SBO} terms are a type of optional annotation, and each different class
   * of SBML object derived from {@link SBase} imposes its own requirements about
   * the values permitted for \'sboTerm\'.  Please consult the SBML
   * Level&nbsp;2 Version&nbsp;4 specification for more information about
   * the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * @param value the NNNNNNN integer portion of the {@link SBO} identifier
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @see #setSBOTerm(String sbo_id)
   */ public
";


%javamethodmodifiers SBase::setSBOTerm(const std::string &sboid) "
/**
   * Sets the value of the \'sboTerm\' attribute by string.
   * <p>
   * Beginning with SBML Level 2 Version 3, objects derived from {@link SBase} have
   * an optional attribute named \'sboTerm\' for supporting the use of the
   * Systems Biology Ontology.  In SBML proper, the data type of the
   * attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
   * seven digit integer number; libSBML simplifies the representation by
   * only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
   * \'sboTerm\' attribute on {@link SBase} has data type <code>int</code>, and {@link SBO} identifiers
   * are stored simply as integers.  This method lets you set the value of
   * \'sboTerm\' as a complete string of the form \'SBO:NNNNNNN\', whereas
   * setSBOTerm(int value) allows you to set it using the integer form.
   * <p>
   * {@link SBO} terms are a type of optional annotation, and each different class
   * of SBML object derived from {@link SBase} imposes its own requirements about
   * the values permitted for \'sboTerm\'.  Please consult the SBML
   * Level&nbsp;2 Version&nbsp;4 specification for more information about
   * the use of {@link SBO} and the \'sboTerm\' attribute.
   * <p>
   * @param sboid the {@link SBO} identifier string of the form \'SBO:NNNNNNN\'
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @see #setSBOTerm(int value)
   */ public
";


%javamethodmodifiers SBase::setNamespaces(XMLNamespaces* xmlns) "
/**
   * Sets the namespaces relevant of this SBML object.
   * <p>
   * The content of <code>xmlns</code> is copied, and this object\'s existing
   * namespace content is deleted.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * <p>
   * @param xmlns the namespaces to set
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::unsetMetaId "
/**
   * Unsets the value of the \'metaid\' attribute of this SBML object.
   * <p>
   * The optional attribute named \'metaid\', present on every major SBML
   * component type, is for supporting metadata annotations using RDF
   * (Resource Description Format). The attribute value has the data type
   * <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML
   * identifier type, which means each \'metaid\' value must be globally
   * unique within an SBML file.  (Importantly, this uniqueness criterion
   * applies across any attribute with type <a
   * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, not just the
   * \'metaid\' attribute used by SBML&mdash;something to be aware of if your
   * application-specific XML content inside the \'annotation\' subelement
   * happens to use <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)
   * The \'metaid\' value serves to identify a model component for purposes
   * such as referencing that component from metadata placed within
   * \'annotation\' subelements.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::unsetId "
/**
   * Unsets the value of the \'id\' attribute of this SBML object.
   * <p>
   * Most (but not all) objects in SBML include two common attributes: \'id\'
   * and \'name\'.  The identifier given by an object\'s \'id\' attribute value
   * is used to identify the object within the SBML model definition.
   * Other objects can refer to the component using this identifier.  The
   * data type of \'id\' is always either <code>Sid</code> or
   * <code>UnitSId</code>, depending on the object in question.  Both
   * data types are defined as follows:
   * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
   *   letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *   digit  .= \'0\'..\'9\'
   *   idChar .= letter | digit | \'_\'
   *   SId    .= ( letter | \'_\' ) idChar*
   * </pre>
   * <p>
   * The equality of <code>SId</code> and <code>UnitSId</code> type values
   * in SBML is determined by an exact character sequence match; i.e.,
   * comparisons of these identifiers must be performed in a case-sensitive
   * manner.  This applies to all uses of <code>SId</code> and
   * <code>UnitSId</code>.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this SBML object.
   * <p>
   * Most (but not all) objects in SBML include two common attributes: \'id\'
   * and \'name\'.  In contrast to the \'id\' attribute, the \'name\' attribute is
   * optional and is not intended to be used for cross-referencing purposes
   * within a model.  Its purpose instead is to provide a human-readable
   * label for the component.  The data type of \'name\' is the type
   * <code>string</code> defined in XML Schema.  SBML imposes no
   * restrictions as to the content of \'name\' attributes beyond those
   * restrictions defined by the <code>string</code> type in XML Schema.
   * <p>
   * The recommended practice for handling \'name\' is as follows.  If a
   * software tool has the capability for displaying the content of \'name\'
   * attributes, it should display this content to the user as a
   * component\'s label instead of the component\'s \'id\'.  If the user
   * interface does not have this capability (e.g., because it cannot
   * display or use special characters in symbol names), or if the \'name\'
   * attribute is missing on a given component, then the user interface
   * should display the value of the \'id\' attribute instead.  (Script
   * language interpreters are especially likely to display \'id\' instead of
   * \'name\'.)
   * <p>
   * As a consequence of the above, authors of systems that automatically
   * generate the values of \'id\' attributes should be aware some systems
   * may display the \'id\'\'s to the user.  Authors therefore may wish to
   * take some care to have their software create \'id\' values that are: (a)
   * reasonably easy for humans to type and read; and (b) likely to be
   * meaningful, for example by making the \'id\' attribute be an abbreviated
   * form of the name attribute value.
   * <p>
   * An additional point worth mentioning is although there are
   * restrictions on the uniqueness of \'id\' values, there are no
   * restrictions on the uniqueness of \'name\' values in a model.  This
   * allows software applications leeway in assigning component identifiers.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::unsetNotes "
/**
   * Unsets the value of the \'notes\' subelement of this SBML object.
   * <p>
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type {@link SBase} can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   * <p>
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method {@link SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)}; however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and &nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @see #getNotesString()
   * @see #isSetNotes()
   * @see #setNotes(XMLNode notes)
   * @see #setNotes(String notes)
   * @see #appendNotes(XMLNode notes)
   * @see #appendNotes(String notes)
   * @see SyntaxChecker#hasExpectedXHTMLSyntax(XMLNode xhtml)
   */ public
";


%javamethodmodifiers SBase::unsetAnnotation "
/**
   * Unsets the value of the \'annotation\' subelement of this SBML object.
   * <p>
   * Whereas the {@link SBase} \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content <em>not</em> meant to be shown to
   * humans.  Every object derived from {@link SBase} can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   * <p>
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @see #getAnnotation()
   * @see #getAnnotationString()
   * @see #isSetAnnotation()
   * @see #setAnnotation(XMLNode annotation)
   * @see #setAnnotation(String annotation)
   * @see #appendAnnotation(XMLNode annotation)
   * @see #appendAnnotation(String annotation)
   */ public
";


%javamethodmodifiers SBase::unsetSBOTerm "
/**
   * Unsets the value of the \'sboTerm\' attribute of this SBML object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::addCVTerm(CVTerm * term, bool newBag = false) "
/**
   * Adds a copy of the given {@link CVTerm} object to this SBML object.
   * <p>
   * @param term the {@link CVTerm} to assign.
   * <p>
   * @param newBag if <code>true</code>, creates a new RDF bag with the same identifier
   * as a previous bag, and if <code>false</code>, adds the term to an existing
   * RDF bag with the same type of qualifier as the term being added.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }, if
   * this object lacks a \'metaid\' attribute
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   * <p>
   * @note Since the CV Term uses the \'metaid\' attribute of the object as a
   * reference, if the object has no \'metaid\' attribute value set, then the
   * {@link CVTerm} will not be added.
   * <p>
   * @warning The fact that this method <em>copies</em> the object passed to it
   * means that the caller will be left holding a physically different
   * object instance than the one contained in <em>this</em> object.  Changes
   * made to the original object instance (such as resetting attribute
   * values) will <em>not affect the instance added here</em>.  In
   * addition, the caller should make sure to free the original object if
   * it is no longer being used, or else a memory leak will result.
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   */ public
";


%javamethodmodifiers SBase::getCVTerms "
/**
   * Returns a list of {@link CVTerm} objects in the annotations of this SBML
   * object.
   * <p>
   * @return the list of CVTerms for this SBML object.
   */ public
";


%javamethodmodifiers SBase::getCVTerms() const "
/**
   * Returns a list of {@link CVTerm} objects in the annotations of this SBML
   * object.
   * <p>
   * @return the list of CVTerms for this SBML object.
   */ public
";


%javamethodmodifiers SBase::getNumCVTerms "
/**
   * Returns the number of {@link CVTerm} objects in the annotations of this SBML
   * object.
   * <p>
   * @return the number of CVTerms for this SBML object.
   */ public
";


%javamethodmodifiers SBase::getCVTerm(unsigned int n) "
/**
   * Returns the nth {@link CVTerm} in the list of CVTerms of this SBML
   * object.
   * <p>
   * @param n long the index of the {@link CVTerm} to retrieve
   * <p>
   * @return the nth {@link CVTerm} in the list of CVTerms for this SBML object.
   */ public
";


%javamethodmodifiers SBase::unsetCVTerms "
/**
   * Clears the list of {@link CVTerm} objects attached to this SBML object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::unsetModelHistory "
/**
   * Unsets the {@link ModelHistory} object attached to this object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the {@link Model} element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from {@link SBase}.
   */ public
";


%javamethodmodifiers SBase::getResourceBiologicalQualifier(std::string resource) "
/**
   * Returns the MIRIAM <em>biological qualifier</em> associated with the
   * given resource.
   * <p>
   * In <a target=\'_blank\' href=\'http://biomodels.net/miriam\'>MIRIAM</a>,
   * qualifiers are an optional means of indicating the relationship
   * between a model component and its annotations.  There are two broad
   * kinds of annotations: <em>model</em> and <em>biological</em>.  The
   * latter kind is used to qualify the relationship between a model
   * component and a biological entity which it represents.  Examples of
   * relationships include \'is\' and \'has part\', but many others are
   * possible.  MIRIAM defines <a target=\'_blank\'
   * href=\'http://www.ebi.ac.uk/miriam/main/qualifiers/\'>numerous
   * relationship qualifiers</a> to enable different software tools to
   * qualify biological annotations in the same standardized way.  In
   * libSBML, the MIRIAM controlled-vocabulary annotations on an SBML model
   * element are represented using lists of {@link CVTerm} objects, and the
   * the MIRIAM biological qualifiers are represented using
   * values whose
   * names begin with <code>BQB_</code> in the interface class
   * {@link libsbmlConstants}.
   * <p>
   * This method searches the controlled-vocabulary annotations
   * (i.e., the list of {@link CVTerm} objects) on the present object, then out of
   * those that have biological qualifiers, looks for an annotation to the
   * given <code>resource</code>.  If such an annotation is found, it returns the
   * type of biological qualifier associated with that resource as a 
   * value whose name
   * begins with <code>BQB_</code> from the interface
   * class {@link libsbmlConstants}.
   * <p>
   * @param resource string representing the resource; e.g.,
   * <code>\'http://www.geneontology.org/#GO:0005892\'</code>.
   * <p>
   * @return the qualifier associated with the resource,
   * or {@link  libsbmlConstants#BQB_UNKNOWN BQB_UNKNOWN} if the
   * resource does not exist.
   * <p>
   * @note The set of MIRIAM biological qualifiers grows over
   * time, although relatively slowly.  The values are up to date with
   * MIRIAM at the time of a given libSBML release.  The set of values in
   * list of <code>BQB_</code> constants defined in {@link libsbmlConstants}
   * may be expanded in later libSBML releases, to match
   * the values defined by MIRIAM at that later time.
   * <p>
   */ public
";


%javamethodmodifiers SBase::getResourceModelQualifier(std::string resource) "
/**
   * Returns the MIRIAM <em>model qualifier</em> associated with the
   * given resource.
   * <p>
   * In <a target=\'_blank\' href=\'http://biomodels.net/miriam\'>MIRIAM</a>,
   * qualifiers are an optional means of indicating the relationship
   * between a model component and its annotations.  There are two broad
   * kinds of annotations: <em>model</em> and <em>biological</em>.  The
   * former kind is used to qualify the relationship between a model
   * component and another modeling object.  An example qualifier is
   * \'isDerivedFrom\', to indicate that a given component of the model is
   * derived from the modeling object represented by the referenced
   * resource.  MIRIAM defines <a target=\'_blank\'
   * href=\'http://www.ebi.ac.uk/miriam/main/qualifiers/\'>numerous
   * relationship qualifiers</a> to enable different software tools to
   * qualify model annotations in the same standardized way.  In libSBML,
   * the MIRIAM controlled-vocabulary annotations on an SBML model element
   * are represented using lists of {@link CVTerm} objects, and the 
   * the MIRIAM model qualifiers are represented using
   * values whose
   * names begin with <code>BQM_</code> in the interface class
   * {@link libsbmlConstants}.
   * <p>
   * This method method searches the controlled-vocabulary annotations
   * (i.e., the list of {@link CVTerm} objects) on the present object, then out of
   * those that have model qualifiers, looks for an annotation to the given
   * <code>resource</code>.  If such an annotation is found, it returns the type of
   * type of model qualifier associated with that resource as a 
   * value whose name
   * begins with <code>BQM_</code> from the interface
   * class {@link libsbmlConstants}.
   * <p>
   * @param resource string representing the resource; e.g.,
   * <code>\'http://www.geneontology.org/#GO:0005892\'</code>.
   * <p>
   * @return the model qualifier
   * type associated with the resource, or {@link 
   * libsbmlConstants#BQM_UNKNOWN BQM_UNKNOWN} if the resource
   * does not exist.
   * <p>
   * @note The set of MIRIAM model qualifiers grows over
   * time, although relatively slowly.  The values are up to date with
   * MIRIAM at the time of a given libSBML release.  The set of values in
   * list of <code>BQM_</code> constants defined in {@link libsbmlConstants}
   * may be expanded in later libSBML releases, to match
   * the values defined by MIRIAM at that later time.
   * <p>
   */ public
";


%javamethodmodifiers SBase::getModel() const "
/**
   * Returns the {@link Model} object in which the current object is located.
   * <p>
   * @return the parent {@link Model} of this SBML object.
   * <p>
   * @see #getParentSBMLObject()
   * @see #getSBMLDocument()
   */ public
";


%javamethodmodifiers SBase::getLevel() const "
/**
   * Returns the SBML Level of the {@link SBMLDocument} object containing this
   * object.
   * <p>
   * @return the SBML level of this SBML object.
   * <p>
   * @see #getVersion()
   * @see #getNamespaces()
   * @see #getPackageVersion()
   */ public
";


%javamethodmodifiers SBase::getVersion() const "
/**
   * Returns the Version within the SBML Level of the {@link SBMLDocument} object
   * containing this object.
   * <p>
   * @return the SBML version of this SBML object.
   * <p>
   * @see #getLevel()
   * @see #getNamespaces()
   */ public
";


%javamethodmodifiers SBase::getPackageVersion() const "
/**
   * Returns the Version of the SBML Level&nbsp;3 package to which this
   * element belongs to.
   * <p>
   * @return the version of the SBML Level&nbsp;3 package to which this
   * element belongs. The value <code>0</code> will be returned if this element
   * belongs to the SBML Level&nbsp;3 Core package.
   * <p>
   * @see #getLevel()
   * @see #getVersion()
   */ public
";


%javamethodmodifiers SBase::getPackageName() const "
/**
   * Returns the name of the SBML Level&nbsp;3 package in which this
   * element is defined.
   * <p>
   * @return the name of the SBML package in which this element is defined.
   * The string <code>&quot;core&quot;</code> will be returned if this
   * element is defined in SBML Level&nbsp;3 Core. The string
   * <code>&quot;unknown&quot;</code> will be returned if this element is
   * not defined in any SBML package.
   */ public
";


%javamethodmodifiers SBase::getTypeCode() const "
/**
   * Returns the libSBML type code for this object.
   * <p>
   * This method may return the type code of this SBML object, or it may
   * return {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN}.  This
   * is because subclasses of {@link SBase} are not required to implement this
   * method to return a type code.  This method is meant primarily for the
   * LibSBML C interface, in which class and subclass information is not
   * readily available.
   * <p>
   * @return the SBML object type code
   * of this SBML object or
   * {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (the default).
   * <p>
   * @see #getElementName()
   * @see #getPackageName()
   */ public
";


%javamethodmodifiers SBase::hasValidLevelVersionNamespaceCombination "
/**
   * Predicate returning <code>true</code> if this
   * object\'s level/version and namespace values correspond to a valid
   * SBML specification.
   * <p>
   * The valid combinations of SBML Level, Version and Namespace as of this
   * release of libSBML are the following:
   * <ul>
   * <li> Level&nbsp;1 Version&nbsp;2: <code>\'http://www.sbml.org/sbml/level1\'</code>
   * <li> Level&nbsp;2 Version&nbsp;1: <code>\'http://www.sbml.org/sbml/level2\'</code>
   * <li> Level&nbsp;2 Version&nbsp;2: <code>\'http://www.sbml.org/sbml/level2/version2\'</code>
   * <li> Level&nbsp;2 Version&nbsp;3: <code>\'http://www.sbml.org/sbml/level2/version3\'</code>
   * <li> Level&nbsp;2 Version&nbsp;4: <code>\'http://www.sbml.org/sbml/level2/version4\'</code>
   * <li> Level&nbsp;3 Version&nbsp;1 Core: <code>\'http://www.sbml.org/sbml/level3/version1/core\'</code>
   * </ul>
   * <p>
   * @return <code>true</code> if the level, version and namespace values of this 
   * SBML object correspond to a valid set of values, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers SBase::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * This is overridden by subclasses to return a string appropriate to the
   * SBML component.  For example, {@link Model} defines it as returning 
   * <code>\'model\'</code>, {@link CompartmentType} defines it as returning <code>\'compartmentType\'</code>,
   * and so on.
   */ public
";


%javamethodmodifiers SBase::toSBML "
/**
   * Returns a string consisting of a partial SBML corresponding to just
   * this object.
   * <p>
   * @return the partial SBML that describes this SBML object.
   * <p>
   * @warning This is primarily provided for testing and debugging
   * purposes.  It may be removed in a future version of libSBML.
   */ public
";


%javamethodmodifiers SBase::getPlugin(const std::string& package) "
/**
   * Returns a plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension with the given package name or URI.
   * <p>
   * @param package the name or URI of the package
   * <p>
   * @return the plug-in object (the libSBML extension interface) of
   * a package extension with the given package name or URI.
   */ public
";


%javamethodmodifiers SBase::getPlugin(const std::string& package) const "
/**
   * Returns a plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension with the given package name or URI.
   * <p>
   * @param package the name or URI of the package
   * <p>
   * @return the plug-in object (the libSBML extension interface) of a
   * package extension with the given package name or URI.
   */ public
";


%javamethodmodifiers SBase::getPlugin(unsigned int n) "
/**
   * Returns the nth plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension.
   * <p>
   * @param n the index of the plug-in to return
   * <p>
   * @return the plug-in object (the libSBML extension interface) of
   * a package extension with the given package name or URI.
   */ public
";


%javamethodmodifiers SBase::getPlugin "
/**
   * Returns the nth plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension.
   * <p>
   * @param n the index of the plug-in to return
   * <p>
   * @return the plug-in object (the libSBML extension interface) of a
   * package extension with the given package name or URI.
   */ public
";


%javamethodmodifiers SBase::getNumPlugins() const "
/**
   * Returns the number of plug-in objects (extenstion interfaces) for SBML
   * Level&nbsp;3 package extensions known.
   * <p>
   * @return the number of plug-in objects (extension interfaces) of
   * package extensions known by this instance of libSBML.
   */ public
";


%javamethodmodifiers SBase::enablePackage(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/**
   * Enables or disables the given SBML Level&nbsp;3 package
   * <p>
   * This method enables or disables the specified package on this object
   * and other objects connected by child-parent links in the same
   * {@link SBMLDocument} object.
   * <p>
   * @param pkgURI the URI of the package
   * <p>
   * @param pkgPrefix the XML prefix of the package
   * <p>
   * @param flag whether to enable (<code>true</code>) or disable (<code>false</code>) the package
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_UNKNOWN LIBSBML_PKG_UNKNOWN }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_VERSION_MISMATCH LIBSBML_PKG_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_CONFLICTED_VERSION LIBSBML_PKG_CONFLICTED_VERSION }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::disablePackage(const std::string& pkgURI, const std::string& pkgPrefix) "
/**
   * Disables the given SBML Level&nbsp;3 package
   * <p>
   * This method enables or disables the specified package on this object
   * and other objects connected by child-parent links in the same
   * {@link SBMLDocument} object.
   * <p>
   * @param pkgURI the URI of the package
   * <p>
   * @param pkgPrefix the XML prefix of the package
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_UNKNOWN LIBSBML_PKG_UNKNOWN }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_VERSION_MISMATCH LIBSBML_PKG_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_CONFLICTED_VERSION LIBSBML_PKG_CONFLICTED_VERSION }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::enablePackageInternal(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/**
   * Enables/Disables the given package with this element and child
   * elements (if any).
   * (This is an internal implementation for enablePackage function)
   * <p>
   * @note Subclasses in which one or more child elements are defined 
   * must override this function.
   * <p>
   * <p>
   * @see #connectToChild()
   * @internal
   */ public
";


%javamethodmodifiers SBase::isPackageURIEnabled(const std::string& pkgURI) const "
/**
   * Predicate returning <code>true</code> if an SBML Level&nbsp;3 package with the
   * given URI is enabled with this object.
   * <p>
   * @param pkgURI the URI of the package
   * <p>
   * @return <code>true</code> if the given package is enabled within this object, @c 
   * false otherwise.
   * <p>
   * @see #isPackageEnabled(String pkgName)
   */ public
";


%javamethodmodifiers SBase::isPackageEnabled(const std::string& pkgName) const "
/**
   * Predicate returning <code>true</code> if the given SBML Level&nbsp;3 package is
   * enabled with this object.
   * <p>
   * The search ignores the package version.
   * <p>
   * @param pkgName the name of the package
   * <p>
   * @return <code>true</code> if the given package is enabled within this object, 
   * <code>false</code> otherwise.
   * <p>
   * @see #isPackageURIEnabled(String pkgURI)
   */ public
";


%javamethodmodifiers SBase::isPkgURIEnabled(const std::string& pkgURI) const "
/**
   * Predicate returning <code>true</code> if an SBML Level&nbsp;3 package with the
   * given URI is enabled with this object.
   * <p>
   * @param pkgURI the URI of the package
   * <p>
   * @return <code>true</code> if the given package is enabled within this object, @c 
   * false otherwise.
   * <p>
   * @see #isPkgEnabled(String pkgName)
   * <p>
   * @deprecated <div class=\'deprecated\'>Replaced in libSBML 5.2.0 by
   * isPackageURIEnabled(String pkgURI)
   </div>
   * */ public
";


%javamethodmodifiers SBase::isPkgEnabled(const std::string& pkgName) const "
/**
   * Predicate returning <code>true</code> if the given SBML Level&nbsp;3 package is
   * enabled with this object.
   * <p>
   * The search ignores the package version.
   * <p>
   * @param pkgName the name of the package
   * <p>
   * @return <code>true</code> if the given package is enabled within this object, 
   * <code>false</code> otherwise.
   * <p>
   * @see #isPkgURIEnabled(String pkgURI)
   * <p>
   * @deprecated <div class=\'deprecated\'>Replaced in libSBML 5.2.0 by
   * isPackageEnabled(String pkgName)
   </div>
   * */ public
";


%javamethodmodifiers SBase::writeExtensionElements(XMLOutputStream& stream) const "
/**
   * Writes out contained SBML objects of package extensions (if any)
   * as XML elements.
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers SBase::read(XMLInputStream& stream) "
/**
   * Reads (initializes) this SBML object by reading from XMLInputStream.
   * @internal
   */ public
";


%javamethodmodifiers SBase::write(XMLOutputStream& stream) const "
/**
   * Writes (serializes) this SBML object by writing it to XMLOutputStream.
   * @internal
   */ public
";


%javamethodmodifiers SBase::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * @internal
   */ public
";


%javamethodmodifiers SBase::hasRequiredAttributes() const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * @internal
   */ public
";


%javamethodmodifiers SBase::hasRequiredElements() const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * @internal
   */ public
";


%javamethodmodifiers SBase::setSBMLNamespaces(SBMLNamespaces * sbmlns) "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * @internal
   */ public
";


%javamethodmodifiers SBase::setSBMLNamespacesAndOwn(SBMLNamespaces * sbmlns) "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * @internal
   */ public
";


%javamethodmodifiers SBase::getSBMLNamespaces() const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * @internal
   */ public
";


%javamethodmodifiers SBase::removeDuplicateAnnotations "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * @internal
   */ public
";


%javamethodmodifiers SBase::checkMathMLNamespace(const XMLToken elem) "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.  For example:
   * @internal
   */ public
";


%javamethodmodifiers SBase::removeFromParentAndDelete "
/**
   * Removes itself from its parent.  If the parent was storing it as a
   * pointer, it is deleted.  If not, it is simply cleared (as in {@link ListOf}
   * objects).  Pure virutal, as every {@link SBase} element has different parents,
   * and therefore different methods of removing itself.  Will fail (and
   * not delete itself) if it has no parent object.  This function is
   * designed to be overridden, but for all objects whose parent is of the
   * class {@link ListOf}, the default implementation will work.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::matchesSBMLNamespaces(const SBase * sb) "
/**
   * Returns <code>true</code> if this object\'s set of XML namespaces are the same
   * as the given object\'s XML namespaces.
   * <p>
   * @param sb an object to compare with respect to namespaces
   * <p>
   * @return boolean, <code>true</code> if this object\'s collection of namespaces is
   * the same as <code>sb\'s</code>, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers SBase::matchesRequiredSBMLNamespacesForAddition(const SBase * sb) "
/**
   * Returns <code>true</code> if this object\'s set of XML namespaces are a subset
   * of the given object\'s XML namespaces.
   * <p>
   * @param sb an object to compare with respect to namespaces
   * <p>
   * @return boolean, <code>true</code> if this object\'s collection of namespaces is
   * a subset of <code>sb\'s</code>, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers SBase::setUserData(void *userData) "
/**
   * Sets the user data of this element. This can be used by the application
   * developer to attach custom information to the node. In case of a deep
   * copy this attribute will passed as it is. The attribute will be never
   * interpreted by this class.
   * <p>
   * @param userData specifies the new user data. 
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers SBase::*getUserData() const "
/**
   * Returns the user data that has been previously set via setUserData().
   * <p>
   * @return the user data of this node, or <code>null</code> if no user data has been set.
   * <p>
   * <p>
   */ public
";


%javamethodmodifiers SBase::setElementText(const std::string &text) "
/** 
   * When overridden allows {@link SBase} elements to use the text included in between
   * the elements tags. The default implementation does nothing.
   * <p>
   * @param text the text string found between the element tags.
   */ public
";


%javamethodmodifiers SBase::matchesCoreSBMLNamespace(const SBase * sb) "
/** 
   * When overridden allows {@link SBase} elements to use the text included in between
   * the elements tags. The default implementation does nothing.
   * <p>
   * @param text the text string found between the element tags.
   * @internal
   */ public
";


%javamethodmodifiers SBase::SBase(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link SBase} object with the given SBML level, version.
   * @internal
   */ public
";


%javamethodmodifiers SBase::SBase(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link SBase} object with the given {@link SBMLNamespaces}.
   * Only subclasses may create {@link SBase} objects.
   * @internal
   */ public
";


%javamethodmodifiers SBase::SBase(const SBase& orig) "
/**
  * Copy constructor. Creates a copy of this {@link SBase} object.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
  * @internal
  */ public
";


%javamethodmodifiers SBase::createObject(XMLInputStream& stream) "
/**
   * Subclasses should override this method to create, store, and then
   * return an SBML object corresponding to the next {@link XMLToken} in the
   * XMLInputStream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%javamethodmodifiers SBase::hasValidLevelVersionNamespaceCombination(int typecode, XMLNamespaces *xmlns) "
/**
   * Predicate returning <code>true</code> if this
   * object\'s level/version and namespace values correspond to a valid
   * SBML specification.
   * <p>
   * The valid combinations of SBML Level, Version and Namespace as of this
   * release of libSBML are the following:
   * <ul>
   * <li> Level&nbsp;1 Version&nbsp;2: <code>\'http://www.sbml.org/sbml/level1\'</code>
   * <li> Level&nbsp;2 Version&nbsp;1: <code>\'http://www.sbml.org/sbml/level2\'</code>
   * <li> Level&nbsp;2 Version&nbsp;2: <code>\'http://www.sbml.org/sbml/level2/version2\'</code>
   * <li> Level&nbsp;2 Version&nbsp;3: <code>\'http://www.sbml.org/sbml/level2/version3\'</code>
   * <li> Level&nbsp;2 Version&nbsp;4: <code>\'http://www.sbml.org/sbml/level2/version4\'</code>
   * <li> Level&nbsp;3 Version&nbsp;1 Core: <code>\'http://www.sbml.org/sbml/level3/version1/core\'</code>
   * </ul>
   * <p>
   * @param typecode the typecode for this element
   * @param xmlns the namespaces used by this element.
   * <p>
   * @note  This function is provided as convenience method to be called from constructors. This 
   *        allows to use it in scenarios where the namespaces or typecode have not yet been initialized. 
   * <p>
   * @return <code>true</code> if the level, version and namespace values of this 
   * SBML object correspond to a valid set of values, <code>false</code> otherwise.
   * @internal
   */ public
";


%javamethodmodifiers SBase::readOtherXML(XMLInputStream& stream) "
/**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   * @internal
   */ public
";


%javamethodmodifiers SBase::getElementPosition() const "
/**
   * The SBML XML Schema is written such that the order of child elements
   * is significant.  LibSBML can read elements out of order.  If you
   * override this method to indicate the ordinal position of element with
   * respect to its siblings, libSBML will log an error if the element is
   * read out of order.
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings or <code>-1</code> (the default) to indicate the position is not
   * significant.
   * @internal
   */ public
";


%javamethodmodifiers SBase::getErrorLog "
/**
   * @return the {@link SBMLErrorLog} used to log errors during while reading and
   * validating SBML.
   * @internal
   */ public
";


%javamethodmodifiers SBase::logError(  unsigned int       id , const unsigned int level   = 2 , const unsigned int version = 3 , const std::string& details = "" ) "
/**
   * Convenience method for easily logging problems from within method
   * implementations.
   * <p>
   * This is essentially a short form of getErrorLog()->logError(...)
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   * @internal
   */ public
";


%javamethodmodifiers SBase::logUnknownAttribute( const std::string& attribute, const unsigned int level, const unsigned int version, const std::string& element ) "
/**
   * Helper to log a common type of error.
   * @internal
   */ public
";


%javamethodmodifiers SBase::logUnknownElement( const std::string& element, const unsigned int level, const unsigned int version ) "
/**
   * Helper to log a common type of error.
   * @internal
   */ public
";


%javamethodmodifiers SBase::logEmptyString( const std::string& attribute, const unsigned int level, const unsigned int version, const std::string& element) "
/**
   * Helper to log a common type of error.
   * @internal
   */ public
";


%javamethodmodifiers SBase::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to add the list of
   * expected attributes. Be sure to call your parents implementation 
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers SBase::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers SBase::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.  For example:
   * <p>
   *   {@link SBase#writeAttributes(stream)};
   *   stream.writeAttribute( \'id\'  , mId   );
   *   stream.writeAttribute( \'name\', mName );
   *   ...
   * <p>
   * (NOTICE) this function doesn\'t write xmlns attributes.
   * Be sure to implement wirteXMLNS() function to write xmlns attributes.
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers SBase::writeXMLNS(XMLOutputStream& stream) const "
/**
   * <p>
   * Subclasses should override this method to write their xmlns attriubutes
   * (if any) to the XMLOutputStream. 
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers SBase::syncAnnotation "
/**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode mAnnotation) is synchronized with the 
   * current {@link CVTerm} objects (List mCVTerm).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   * @internal
   */ public
";


%javamethodmodifiers SBase::reconstructRDFAnnotation "
/**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode mAnnotation) is synchronized with the 
   * current {@link CVTerm} objects (List mCVTerm).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   * @internal
   */ public
";


%javamethodmodifiers SBase::checkOrderAndLogError(SBase* object, int expected) "
/**
   * Checks that the SBML element appears in the expected order.
   * <p>
   * If <code>object</code> is not in the expected position, an error is logged.
   * @internal
   */ public
";


%javamethodmodifiers SBase::checkListOfPopulated(SBase* object) "
/**
   * Checks that an SBML {@link ListOf} element is populated.  
   * If a listOf element has been declared with no elements, 
   * an error is logged.
   * @internal
   */ public
";


%javamethodmodifiers SBase::checkUnitSyntax(unsigned int flag = 0) "
/**
   * Checks the syntax of the unit attribute.
   * The syntax of an unit is of type UnitSId which is defined as:
   * <p>
   *  - letter .= \'a\'..\'z\',\'A\'..\'Z\'
   *  - digit  .= \'0\'..\'9\'
   *  - idChar .= letter | digit | \'_\'
   *  - UnitSId    .= ( letter | \'_\' ) idChar*
   * <p>
   * If the syntax of the unit attribute of this object is incorrect, 
   * an error is logged
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   * @internal
   */ public
";


%javamethodmodifiers SBase::checkDefaultNamespace(const XMLNamespaces* xmlns, const std::string& elementName, const std::string& prefix = "") "
/**
   * Checks that the given default namespace in the given element is valid.
   * If the given default namespace is not valid, an error is logged.
   * @internal
   */ public
";


%javamethodmodifiers SBase::checkAnnotation "
/**
   * Checks the annotation does not declare an sbml namespace.
   * If the annotation declares an sbml namespace an error is logged.
   * @internal
   */ public
";


%javamethodmodifiers SBase::checkXHTML(const XMLNode *) "
/**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   * @internal
   */ public
";


%javamethodmodifiers SBase::loadPlugins(SBMLNamespaces *sbmlns) "
/**
   * Checks that the XHTML is valid.
   * If the xhtml does not conform to the specification of valid xhtml within
   * an sbml document, an error is logged.
   * @internal
   */ public
";


%javamethodmodifiers SBase::createExtensionObject(XMLInputStream& stream) "
/**
   * Create, store, and then return an SBML object of package extensions
   * corresponding to the next {@link XMLToken} in the XMLInputStream.
   * <p>
   * @return the SBML object of package extensions corresponding to next 
   * {@link XMLToken} in the XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%javamethodmodifiers SBase::setElementNamespace(const std::string &uri) "
/**
   * Sets the XML namespace to which this element belongs to.
   * For example, all elements that belong to SBML Level 3 Version 1 Core
   * must set the namespace to \'http://www.sbml.org/sbml/level3/version1/core\'; 
   * all elements that belong to Layout Extension Version 1 for SBML Level 3
   * Version 1 Core must set the namespace to 
   * \'http://www.sbml.org/sbml/level3/version1/layout/version1/\'
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * @internal
   */ public
";


%javamethodmodifiers SBase::getElementNamespace() const "
/**
   * Gets the XML namespace (URI) to which this element belongs to.
   * @internal
   */ public
";


%javamethodmodifiers SBase::getURI() const "
/**
   * Gets the URI to which this element belongs to.
   * For example, all elements that belong to SBML Level 3 Version 1 Core
   * must would have the URI \'http://www.sbml.org/sbml/level3/version1/core\'; 
   * all elements that belong to Layout Extension Version 1 for SBML Level 3
   * Version 1 Core must would have the URI
   * \'http://www.sbml.org/sbml/level3/version1/layout/version1/\'
   * <p>
   * Unlike getElementNamespace, this function first returns the URI for this 
   * element by looking into the {@link SBMLNamespaces} object of the document with 
   * the its package name. if not found it will return the result of 
   * getElementNamespace
   * <p>
   * @return the URI this elements  
   * <p>
   * @see getPackageName
   * @see getElementNamespace
   * @see getSBMLNamespaces
   * @see getSBMLDocument
   * @internal
   */ public
";


%javamethodmodifiers SBase::readExtensionAttributes(const XMLAttributes& attributes) "
/**
   * Read attributes of package extensions from the given {@link XMLAttributes} 
   * set into their specific fields. 
   * <p>
   * Be sure to call your parents implementation of this function as well.  
   * For example:
   * <div class=\'fragment\'><pre>
{@link SBase}.readExtensionAttributes(attributes);
</pre></div>
   * <div class=\'fragment\'><pre>
{@link SBase}.readExtensionAttributes(attributes);
</pre></div>
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers SBase::writeExtensionAttributes(XMLOutputStream& stream) const "
/**
   * Write attributes of package extensions to the XMLOutputStream.  
   * <p>
   * Be sure to call your parents implementation of this function as well.  
   * For example:
   * <p>
   *   {@link SBase#writeExtensionAttributes(stream)};
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers SBase::storeUnknownExtAttribute(const std::string& element, const XMLAttributes& xattr, unsigned int index) "
/**
   * Stores the given attribute to the list of ignored attributes if
   * the given attribute belongs to some unknown package extension.
   * Unknown attribute error will be logged if the \'required\' attribute
   * of the package is \'true\' in {@link SBMLDocument} element.
   * <p>
   * The stored attributes will be written out as-is when writing the 
   * {@link SBMLDocument} to a string or a file (i.e. Attributes and elements of 
   * unknown package extensions will not be lost when reading/writing 
   * a file/sting containing them.)
   * <p>
   * @param element the string of element which contains the given attribute
   * @param xattr the {@link XMLAttributes} object which is contained in the given 
   * element
   * @param index the index of the target attribute in the given {@link XMLAttributes}
   * object.
   * <p>
   * @return true will be returned if the given attribute belongs
   * to some unknown package extension, otherwise false will be returned.
   * @internal
   */ public
";


%javamethodmodifiers SBase::storeUnknownExtElement(XMLInputStream &stream) "
/**
   * Stores the element of next token if the element belongs to some 
   * unknown package extension. Unknown element error will be logged if
   * the \'required\' attribute of the package is \'true\' in {@link SBMLDocument} 
   * element.
   * <p>
   * The stored elements will be written out as-is when writing the 
   * {@link SBMLDocument} to a string or a file (i.e. Attributes and elements of 
   * unknown package extensions will not be lost when reading/writing 
   * a file/sting containing them.)
   * <p>
   * @return true will be returned if the element of next token belongs
   * to some unknown package extension, otherwise false will be returned.
   * @internal
   */ public
";


%javamethodmodifiers SBase::getPrefix() const "
/**
   * Return the prefix of this element.
   * @internal
   */ public
";


%javamethodmodifiers SBase::getSBMLPrefix() const "
/**
   * Return the SBML prefix of this element. This will be the same as getPrefix()
   * unless the element in question is an element of an SBML extension class.
   * @internal
   */ public
";


%javamethodmodifiers SBase::getRootElement "
/**
   * Returns the root element of the node tree to which this element is connected.
   * <p>
   * @note The root element may not be an {@link SBMLDocument} element. For example,
   * this element is the root element if this element doesn\'t have a parent
   * SBML object (i.e. mParentSBMLObject is null)
   * <p>
   * @see enablePackageInternal
   * @internal
   */ public
";


%javamethodmodifiers SBase::getHasBeenDeleted() const "
/**
   * Returns the root element of the node tree to which this element is connected.
   * <p>
   * @note The root element may not be an {@link SBMLDocument} element. For example,
   * this element is the root element if this element doesn\'t have a parent
   * SBML object (i.e. mParentSBMLObject is null)
   * <p>
   * @see enablePackageInternal
   * @internal
   */ public
";


%javamethodmodifiers SBase::setSBaseFields(const XMLToken& element) "
/**
   * Stores the location (line and column) and any XML namespaces (for
   * roundtripping) declared on this SBML (XML) element.
   * @internal
   */ public
";


%javamethodmodifiers SBase::readAnnotation(XMLInputStream& stream) "
/**
   * @return true if read an <annotation> element from the stream
   * @internal
   */ public
";


%javamethodmodifiers SBase::removeDuplicatedResources(CVTerm *term, QualifierType_t type) "
/**
   * removes resources from the term object that alread exist on this object 
   * @internal 
   */ public
";


%javamethodmodifiers SBase::addTermToExistingBag(CVTerm *term, QualifierType_t type) "
/**
  * adds the given term to an existing bag. Returns 1 if added, 0 otherwise. 
  * @internal 
  */ public
";


%javamethodmodifiers SBase::readNotes(XMLInputStream& stream) "
/**
   * @return true if read a <notes> element from the stream
   * @internal
   */ public
";


%typemap(javaimports) ListOf "
/** 
 * Parent class for the various SBML \'ListOfXYZ\' classes.
 * <p>
 * <p style=\'color: #777; font-style: italic\'>
This class of objects is defined by libSBML only and has no direct
equivalent in terms of SBML components.  This class is not prescribed by
the SBML specifications, although it is used to implement features
defined in SBML.
</p>

 * <p>
 * SBML defines various ListOf___ classes that are containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the attributes and subelements of
 * {@link SBase}, such as \'metaid\' as and \'annotation\'.  The ListOf___ classes do
 * not add any attributes of their own.
 * <p>
 * The {@link ListOf} class in libSBML is a utility class that serves as the parent
 * class for implementing the ListOf__ classes.  It provides methods for
 * working generically with the various SBML lists of objects in a program.
 * LibSBML uses this separate list class rather than ordinary
 * Java lists,
 * so that it can provide the methods and features associated with {@link SBase}.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers ListOf::ListOf(unsigned int level   = SBML_DEFAULT_LEVEL, unsigned int version = SBML_DEFAULT_VERSION) "
/**
   * Creates a new {@link ListOf} object.
   * <p>
   * @param level the SBML Level; if not assigned, defaults to the
   * value of SBML_DEFAULT_LEVEL.
   * <p>
   * @param version the Version within the SBML Level; if not assigned,
   * defaults to the value of SBML_DEFAULT_VERSION.
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   */ public
";


%javamethodmodifiers ListOf::ListOf(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOf} with {@link SBMLNamespaces} object.
   * <p>
   * @param sbmlns the set of namespaces that this {@link ListOf} should contain.
   */ public
";


%javamethodmodifiers ListOf::ListOf(const ListOf& orig) "
/**
   * Copy constructor;  creates a copy of this {@link ListOf}.
   * <p>
   * @param orig the {@link ListOf} instance to copy.
   */ public
";


%javamethodmodifiers ListOf::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next item in the
   * list.
   */ public
";


%javamethodmodifiers ListOf::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOf}.
   * <p>
   * @return a (deep) copy of this {@link ListOf}.
   */ public
";


%javamethodmodifiers ListOf::append(const SBase* item) "
/**
   * Adds item to the end of this {@link ListOf}.
   * <p>
   * This variant of the method makes a clone of the <code>item</code> handed to it.
   * This means that when the {@link ListOf} is destroyed, the original items will
   * not be destroyed.
   * <p>
   * @param item the item to be added to the list.
   * <p>
   * @see #appendAndOwn(SBase item)
   */ public
";


%javamethodmodifiers ListOf::appendAndOwn(SBase* item) "
/**
   * Adds item to the end of this {@link ListOf}.
   * <p>
   * This variant of the method does not clone the <code>item</code> handed to it;
   * instead, it assumes ownership of it.  This means that when the {@link ListOf}
   * is destroyed, the item will be destroyed along with it.
   * <p>
   * @param item the item to be added to the list.
   * <p>
   * @see #append(SBase item)
   */ public
";


%javamethodmodifiers ListOf::appendFrom(const ListOf* list) "
/**
   * Adds a clone of all items in the provided {@link ListOf} to this object.  This means that when this {@link ListOf} is destroyed, the original items will not be destroyed.
   * <p>
   * @param list A list of items to be added.
   * <p>
   * @see #append(SBase item)
   */ public
";


%javamethodmodifiers ListOf::insert(int location, const SBase* item) "
/** 
   * Inserts the item at the given position of this {@link ListOf}
   * <p>
   * This variant of the method makes a clone of the <code>item</code> handet to it. 
   * This means that when the {@link ListOf} is destroyed, the original items will
   * not be destroyed. 
   * <p>
   * @param location the location where to insert the item
   * @param item the item to be inserted to the list
   * <p>
   * @see #insertAndOwn(int location, SBase item)
   */ public
";


%javamethodmodifiers ListOf::insertAndOwn(int location, SBase* item) "
/** 
   * Inserts the item at the given position of this {@link ListOf}
   * <p>
   * This variant of the method makes a clone of the <code>item</code> handet to it. 
   * This means that when the {@link ListOf} is destroyed, the original items will
   * not be destroyed. 
   * <p>
   * @param location the location where to insert the item
   * @param item the item to be inserted to the list
   * <p>
   * @see #insert(int location, SBase item)
   */ public
";


%javamethodmodifiers ListOf::get "
/**
   * Get an item from the list.
   * <p>
   * @param n the index number of the item to get.
   * <p>
   * @return the nth item in this {@link ListOf} items.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOf::get(unsigned int n) "
/**
   * Get an item from the list.
   * <p>
   * @param n the index number of the item to get.
   * <p>
   * @return the nth item in this {@link ListOf} items.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOf::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.
   * <p>
   * @param id string representing the id of objects to find
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers ListOf::getElementByMetaId(std::string metaid) "
/**
   * Returns the first child element it can find with the given metaid, or null if no such object is found.
   * <p>
   * @param metaid string representing the metaid of objects to find
   * <p>
   * @return pointer to the first element found with the given metaid.
   */ public
";


%javamethodmodifiers ListOf::getAllElements "
/**
   * Returns a {@link List} of all child {@link SBase} objects, including those nested to an arbitrary depth
   * <p>
   * @return a {@link List} of pointers to all children objects.
   */ public
";


%javamethodmodifiers ListOf::get(const std::string& sid) const "
/**
   * Get an item from the list based on its identifier.
   * <p>
   * @param sid a string representing the the identifier of the item to get.
   * <p>
   * @return item in this {@link ListOf} items with the given id or <code>null</code> if no such
   * item exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOf::get(const std::string& sid) "
/**
   * Get an item from the list based on its identifier.
   * <p>
   * @param sid a string representing the the identifier of the item to get.
   * <p>
   * @return item in this {@link ListOf} items with the given id or <code>null</code> if no such
   * item exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOf::clear(bool doDelete = true) "
/**
   * Removes all items in this {@link ListOf} object.
   * <p>
   * If parameter <code>doDelete</code> is <code>true</code> (default), all items in this {@link ListOf}
   * object are deleted and cleared, and thus the caller doesn\'t have to
   * delete those items.  Otherwise, all items are just cleared from this
   * {@link ListOf} object and the caller is responsible for deleting all items.  (In
   * that case, pointers to all items should be stored elsewhere before
   * calling this function.)
   * <p>
   * @param doDelete if <code>true</code> (default), all items are deleted and cleared.
   * Otherwise, all items are just cleared and not deleted.
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   */ public
";


%javamethodmodifiers ListOf::removeFromParentAndDelete "
/**
   * Because {@link ListOf} objects typically live as object children of their parent object and not as pointer children, this function clears itself, but does not attempt to do anything else.  If a particular {@link ListOf} subclass does indeed exist as a pointer only, this function will need to be overridden.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers ListOf::remove(unsigned int n) "
/**
   * Removes the <em>n</em>th item from this {@link ListOf} items and returns a
   * pointer to it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOf::remove(const std::string& sid) "
/**
   * Removes item in this {@link ListOf} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOf::size() const "
/**
   * Get the size of this {@link ListOf}.
   * <p>
   * @return the number of items in this {@link ListOf} items.
   */ public
";


%javamethodmodifiers ListOf::setSBMLDocument(SBMLDocument* d) "
/**
   * Sets the parent {@link SBMLDocument} of this SBML object.
   * <p>
   * @param d the {@link SBMLDocument} that should become the parent of this
   * {@link ListOf}.
   * @internal
   */ public
";


%javamethodmodifiers ListOf::connectToChild "
/**
   * Sets this SBML object to child SBML objects (if any).
   * (Creates a child-parent relationship by the parent)
   * <p>
   * Subclasses must override this function if they define
   * one ore more child elements.
   * Basically, this function needs to be called in
   * constructor, copy constructor and assignment operator.
   * <p>
   * @see setSBMLDocument
   * @see enablePackageInternal
   * @internal
   */ public
";


%javamethodmodifiers ListOf::getTypeCode() const "
/**
   * Returns the libSBML type code for this object, namely, 
   * <code>SBML_LIST_OF.</code>
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOf::getItemTypeCode() const "
/**
   * Get the type code of the objects contained in this {@link ListOf}.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   */ public
";


%javamethodmodifiers ListOf::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link ListOf}, is
   * always <code>\'listOf\'.</code>
   * <p>
   * @return the XML name of this element.
   */ public
";


%javamethodmodifiers ListOf::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOf::enablePackageInternal(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/**
   * Enables/Disables the given package with this element and child
   * elements (if any).
   * (This is an internal implementation for enablePackage function)
   * <p>
   * @note Subclasses of the SBML Core package in which one or more child
   * elements are defined must override this function.
   * @internal
   */ public
";


%javamethodmodifiers ListOf::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers ListOf::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOf::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.  For example:
   * <p>
   *   {@link SBase#writeAttributes(stream)};
   *   stream.writeAttribute( \'id\'  , mId   );
   *   stream.writeAttribute( \'name\', mName );
   *   ...
   * @internal
   */ public
";


%javamethodmodifiers ListOf::isValidTypeForList(SBase * item) "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.  For example:
   * <p>
   *   {@link SBase#writeAttributes(stream)};
   *   stream.writeAttribute( \'id\'  , mId   );
   *   stream.writeAttribute( \'name\', mName );
   *   ...
   * @internal
   */ public
";


%typemap(javaimports) Model "
/** 
 * LibSBML implementation of SBML\'s Model construct.
 * <p>
 * In an SBML model definition, a single object of class {@link Model} serves as
 * the overall container for the lists of the various model components.
 * All of the lists are optional, but if a given list container is present
 * within the model, the list must not be empty; that is, it must have
 * length one or more.  The following are the components and lists
 * permitted in different Levels and Versions of SBML in
 * version 5.6.0
 * of libSBML:
 * <ul>
 * <li> In SBML Level 1, the components are: {@link UnitDefinition}, {@link Compartment},
 * {@link Species}, {@link Parameter}, {@link Rule}, and {@link Reaction}.  Instances of the classes are
 * placed inside instances of classes {@link ListOfUnitDefinitions},
 * {@link ListOfCompartments}, {@link ListOfSpecies}, {@link ListOfParameters}, {@link ListOfRules}, and
 * {@link ListOfReactions}.
 * <p>
 * <li> In SBML Level 2 Version 1, the components are: {@link FunctionDefinition},
 * {@link UnitDefinition}, {@link Compartment}, {@link Species}, {@link Parameter}, {@link Rule}, {@link Reaction} and
 * {@link Event}.  Instances of the classes are placed inside instances of classes
 * {@link ListOfFunctionDefinitions}, {@link ListOfUnitDefinitions}, {@link ListOfCompartments},
 * {@link ListOfSpecies}, {@link ListOfParameters}, {@link ListOfRules}, {@link ListOfReactions}, and
 * {@link ListOfEvents}.
 * <p>
 * <li> In SBML Level 2 Versions 2, 3 and 4, the components are:
 * {@link FunctionDefinition}, {@link UnitDefinition}, {@link CompartmentType}, {@link SpeciesType},
 * {@link Compartment}, {@link Species}, {@link Parameter}, {@link InitialAssignment}, {@link Rule}, {@link Constraint},
 * {@link Reaction} and {@link Event}.  Instances of the classes are placed inside
 * instances of classes {@link ListOfFunctionDefinitions}, {@link ListOfUnitDefinitions},
 * {@link ListOfCompartmentTypes}, {@link ListOfSpeciesTypes}, {@link ListOfCompartments},
 * {@link ListOfSpecies}, {@link ListOfParameters}, {@link ListOfInitialAssignments}, {@link ListOfRules},
 * {@link ListOfConstraints}, {@link ListOfReactions}, and {@link ListOfEvents}.
 * <p>
 * <li> In SBML Level 3 Version 1, the components are: {@link FunctionDefinition},
 * {@link UnitDefinition}, {@link Compartment}, {@link Species}, {@link Parameter}, {@link InitialAssignment},
 * {@link Rule}, {@link Constraint}, {@link Reaction} and {@link Event}.  Instances of the classes are
 * placed inside instances of classes {@link ListOfFunctionDefinitions},
 * {@link ListOfUnitDefinitions}, {@link ListOfCompartments}, {@link ListOfSpecies},
 * {@link ListOfParameters}, {@link ListOfInitialAssignments}, {@link ListOfRules},
 * {@link ListOfConstraints}, {@link ListOfReactions}, and {@link ListOfEvents}.  
 * </ul>
 * <p>
 * Although all the lists are optional, there are dependencies between SBML
 * components such that defining some components requires defining others.
 * An example is that defining a species requires defining a compartment,
 * and defining a reaction requires defining a species.  The dependencies
 * are explained in more detail in the SBML specifications.
 * <p>
 * In addition to the above lists and attributes, the {@link Model} class in both
 * SBML Level&nbsp;2 and Level&nbsp;3 has the usual two attributes of \'id\'
 * and \'name\', and both are optional.  As is the case for other SBML
 * components with \'id\' and \'name\' attributes, they must be used according
 * to the guidelines described in the SBML specifications.  (Within the
 * frameworks of SBML Level&nbsp;2 and Level&nbsp;3 Version&nbsp;1 Core, a
 * {@link Model} object identifier has no assigned meaning, but extension packages
 * planned for SBML Level&nbsp;3 are likely to make use of this
 * identifier.)
 * <p>
 * Finally, SBML Level&nbsp;3 has introduced a number of additional {@link Model}
 * attributes.  They are discussed in a separate section below.
 * <p>
 * <p>
 * <h2>Approaches to creating objects using the libSBML API</h2>
 * <p>
 * LibSBML provides two main mechanisms for creating objects: class
 * constructors
 * (e.g., <a href=\'org/sbml/{@link libsbml}/Species.html\'>Species()</a> ), 
 * and <code>create<span class=\'placeholder\'><em>Object</em></span>()</code>
 * methods (such as {@link Model#createSpecies()}) provided by certain <span
 * class=\'placeholder\'><em>Object</em></span> classes such as {@link Model}.  These
 * multiple mechanisms are provided by libSBML for flexibility and to
 * support different use-cases, but they also have different implications
 * for the overall model structure.
 * <p>
 * In general, the recommended approach is to use the <code>create<span
 * class=\'placeholder\'><em>Object</em></span>()</code> methods.  These
 * methods both create an object <em>and</em> link it to the parent in one step.
 * Here is an example:<div class=\'fragment\'><pre>
// Create an {@link SBMLDocument} object in Level 3 Version 1 format:

{@link SBMLDocument} sbmlDoc = new {@link SBMLDocument}(3, 1);

// Create a {@link Model} object inside the {@link SBMLDocument} object and set
// its identifier.  The call returns a pointer to the {@link Model} object
// created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).  Note that
// the call to setId() returns a status code, and a real program
// should check this status code to make sure everything went okay.

{@link Model} model = sbmlDoc.createModel();
model.setId(&#34;BestModelEver&#34;);

// Create a {@link Species} object inside the {@link Model} and set its identifier.
// Similar to the lines above, this call returns a pointer to the {@link Species}
// object created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).  Note that, like
// with {@link Model}, the call to setId() returns a status code, and a real program
// should check this status code to make sure everything went okay.

{@link Species} sp = model.createSpecies();
sp.setId(&#34;BestSpeciesEver&#34;);
</pre></div>
 * <p>
 * <p>
 * The <code>create<span
 * class=\'placeholder\'><em>Object</em></span>()</code> methods return a
 * pointer to the object created, but they also add the object to the
 * relevant list of object instances contained in the parent.  (These lists
 * become the <code>&lt;listOf<span
 * class=\'placeholder\'><em>Object</em></span>s&gt;</code> elements in the
 * finished XML rendition of SBML.)  In the example above,
 * {@link Model#createSpecies()} adds the created species directly to the
 * <code>&lt;listOfSpecies<i></i>&gt;</code> list in the model.  Subsequently,
 * methods called on the species change the species in the model (which is
 * what is expected in most situations).
 * <p>
 * <h2>Consistency and adherence to SBML specifications</h2>
 * <p>
 * To make it easier for applications to do whatever they need,
 * libSBML version 5.6.0
 * is relatively lax when it comes to enforcing correctness and
 * completeness of models <em>during</em> model construction and editing.
 * Essentially, libSBML <em>will</em> <em>not</em> in most cases check automatically
 * that a model\'s components have valid attribute values, or that the
 * overall model is consistent and free of errors&mdash;even obvious errors
 * such as duplication of identifiers.  This allows applications great
 * leeway in how they build their models, but it means that software
 * authors must take deliberate steps to ensure that the model will be, in
 * the end, valid SBML.  These steps include such things as keeping track
 * of the identifiers used in a model, manually performing updates in
 * certain situations where an entity is referenced in more than one place
 * (e.g., a species that is referenced by multiple {@link SpeciesReference}
 * objects), and so on.
 * <p>
 * That said, libSBML does provide powerful features for deliberately
 * performing validation of SBML when an application decides it is time to
 * do so.  The interfaces to these facilities are on the {@link SBMLDocument}
 * class, in the form of {@link SBMLDocument#checkInternalConsistency()} and
 * {@link SBMLDocument#checkConsistency()}.  Please refer to the documentation for
 * {@link SBMLDocument} for more information about this.
 * <p>
 * While applications may play fast and loose and live like free spirits
 * during the construction and editing of SBML models, they should always
 * make sure to call {@link SBMLDocument#checkInternalConsistency()} and/or
 * {@link SBMLDocument#checkConsistency()} before writing out the final version of
 * an SBML model.
 * <p>
 * <p>
 * <h2>Model attributes introduced in SBML Level&nbsp;3</h2>
 * <p>
 * As mentioned above, the {@link Model} class has a number of optional attributes
 * in SBML Level&nbsp;3 Version&nbsp;1 Core.  These are \'substanceUnits\',
 * \'timeUnits\', \'volumeUnits\', \'areaUnits\', \'lengthUnits\', \'extentUnits\',
 * and \'conversionFactor.  The following provide more information about
 * them.
 * <p>
 * <h3>The \'substanceUnits\' attribute</h3>
 * <p>
 * The \'substanceUnits\' attribute is used to specify the unit of
 * measurement associated with substance quantities of {@link Species} objects that
 * do not specify units explicitly.  If a given {@link Species} object definition
 * does not specify its unit of substance quantity via the \'substanceUnits\'
 * attribute on the {@link Species} object instance, then that species inherits the
 * value of the {@link Model} \'substanceUnits\' attribute.  If the {@link Model} does not
 * define a value for this attribute, then there is no unit to inherit, and
 * all species that do not specify individual \'substanceUnits\' attribute
 * values then have <em>no</em> declared units for their quantities.  The
 * SBML Level&nbsp;3 Version&nbsp;1 Core specification provides more
 * details.
 * <p>
 * Note that when the identifier of a species appears in a model\'s
 * mathematical expressions, the unit of measurement associated with that
 * identifier is <em>not solely determined</em> by setting \'substanceUnits\'
 * on {@link Model} or {@link Species}.  Please see the discussion about units given in
 * the documentation for the {@link Species} class.
 * <p>
 * <p>
 * <h3>The \'timeUnits\' attribute</h3>
 * <p>
 * The \'timeUnits\' attribute on SBML Level&nbsp;3\'s {@link Model} object is used to
 * specify the unit in which time is measured in the model.  This attribute
 * on {@link Model} is the <em>only</em> way to specify a unit for time in a model.
 * It is a global attribute; time is measured in the model everywhere in
 * the same way.  This is particularly relevant to {@link Reaction} and {@link RateRule}
 * objects in a model: all {@link Reaction} and {@link RateRule} objects in SBML define
 * per-time values, and the unit of time is given by the \'timeUnits\'
 * attribute on the {@link Model} object instance.  If the {@link Model} \'timeUnits\'
 * attribute has no value, it means that the unit of time is not defined
 * for the model\'s reactions and rate rules.  Leaving it unspecified in an
 * SBML model does not result in an invalid model in SBML Level&nbsp;3;
 * however, as a matter of best practice, we strongly recommend that all
 * models specify units of measurement for time.
 * <p>
 * <p>
 * <h3>The \'volumeUnits\', \'areaUnits\', and \'lengthUnits\' attributes</h3>
 * <p>
 * The attributes \'volumeUnits\', \'areaUnits\' and \'lengthUnits\' together are
 * used to set the units of measurements for the sizes of {@link Compartment}
 * objects in an SBML Level&nbsp;3 model when those objects do not
 * otherwise specify units.  The three attributes correspond to the most
 * common cases of compartment dimensions: \'volumeUnits\' for compartments
 * having a \'spatialDimensions\' attribute value of <code>\'3\'</code>, \'areaUnits\' for
 * compartments having a \'spatialDimensions\' attribute value of <code>\'2\'</code>, and
 * \'lengthUnits\' for compartments having a \'spatialDimensions\' attribute
 * value of <code>\'1\'.</code>  The attributes are not applicable to compartments
 * whose \'spatialDimensions\' attribute values are <em>not</em> one of <code>\'1\'</code>, 
 * <code>\'2\'</code> or <code>\'3\'.</code>
 * <p>
 * If a given {@link Compartment} object instance does not provide a value for its
 * \'units\' attribute, then the unit of measurement of that compartment\'s
 * size is inherited from the value specified by the {@link Model} \'volumeUnits\',
 * \'areaUnits\' or \'lengthUnits\' attribute, as appropriate based on the
 * {@link Compartment} object\'s \'spatialDimensions\' attribute value.  If the {@link Model}
 * object does not define the relevant attribute, then there are no units
 * to inherit, and all {@link Compartment} objects that do not set a value for
 * their \'units\' attribute then have <em>no</em> units associated with
 * their compartment sizes.
 * <p>
 * The use of three separate attributes is a carry-over from SBML
 * Level&nbsp;2.  Note that it is entirely possible for a model to define a
 * value for two or more of the attributes \'volumeUnits\', \'areaUnits\' and
 * \'lengthUnits\' simultaneously, because SBML models may contain
 * compartments with different numbers of dimensions.
 * <p>
 * <p>
 * <h3>The \'extentUnits\' attribute</h3>
 * <p>
 * Reactions are processes that occur over time.  These processes involve
 * events of some sort, where a single ``reaction event\'\' is one in which
 * some set of entities (known as reactants, products and modifiers in
 * SBML) interact, once.  The <em>extent</em> of a reaction is a measure of
 * how many times the reaction has occurred, while the time derivative of
 * the extent gives the instantaneous rate at which the reaction is
 * occurring.  Thus, what is colloquially referred to as the \'rate of the
 * reaction\' is in fact equal to the rate of change of reaction extent.
 * <p>
 * In SBML Level&nbsp;3, the combination of \'extentUnits\' and \'timeUnits\'
 * defines the units of kinetic laws in SBML and establishes how the
 * numerical value of each {@link KineticLaw} object\'s mathematical formula is
 * meant to be interpreted in a model.  The units of the kinetic laws are
 * taken to be \'extentUnits\' divided by \'timeUnits\'.
 * <p>
 * Note that this embodies an important principle in SBML Level&nbsp;3
 * models: <em>all reactions in an SBML model must have the same units</em>
 * for the rate of change of extent.  In other words, the units of all
 * reaction rates in the model <em>must be the same</em>.  There is only
 * one global value for \'extentUnits\' and one global value for \'timeUnits\'.
 * <p>
 * <p>
 * <h3>The \'conversionFactor\' attribute</h3>
 * <p>
 * The attribute \'conversionFactor\' in SBML Level&nbsp;3\'s {@link Model} object
 * defines a global value inherited by all {@link Species} object instances that do
 * not define separate values for their \'conversionFactor\' attributes.  The
 * value of this attribute must refer to a {@link Parameter} object instance
 * defined in the model.  The {@link Parameter} object in question must be a
 * constant; ie it must have its \'constant\' attribute value set to 
 * <code>\'true\'.</code>
 * <p>
 * If a given {@link Species} object definition does not specify a conversion
 * factor via the \'conversionFactor\' attribute on {@link Species}, then the species
 * inherits the conversion factor specified by the {@link Model} \'conversionFactor\'
 * attribute.  If the {@link Model} does not define a value for this attribute,
 * then there is no conversion factor to inherit.  More information about
 * conversion factors is provided in the SBML Level&nbsp;3 Version&nbsp;1
 * specification.
 */
"


%javamethodmodifiers Model::Model(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link Model} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Model}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Model}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Model} object to an {@link SBMLDocument}
   * (e.g., using {@link SBMLDocument#setModel(Model m)}), the SBML Level, SBML Version
   * and XML namespace of the document <em>override</em> the values used
   * when creating the {@link Model} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link Model} is an important aid to producing valid SBML.  Knowledge
   * of the intented SBML Level and Version determine whether it is valid
   * to assign a particular value to an attribute, or whether it is valid
   * to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Model::Model(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link Model} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Model} object to an {@link SBMLDocument} (e.g.,
   * using {@link SBMLDocument#setModel(Model m)}), the SBML XML namespace of the document 
   * <em>overrides</em> the value used when creating the {@link Model} object via this
   * constructor.  This is necessary to ensure that an SBML document is a
   * consistent structure.  Nevertheless, the ability to supply the values
   * at the time of creation of a {@link Model} is an important aid to producing
   * valid SBML.  Knowledge of the intented SBML Level and Version
   * determine whether it is valid to assign a particular value to an
   * attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Model::Model(const Model& orig) "
/**
   * Copy constructor; creates a (deep) copy of the given {@link Model} object.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers Model::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Constraint}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */ public
";


%javamethodmodifiers Model::clone() const "
/**
   * Creates and returns a deep copy of this {@link Model} object.
   * <p>
   * @return a (deep) copy of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.
   * <p>
   * @param id string representing the id of objects to find.
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers Model::getElementByMetaId(std::string metaid) "
/**
   * Returns the first child element it can find with the given metaid, or null if no such object is found.
   * <p>
   * @param metaid string representing the metaid of objects to find
   * <p>
   * @return pointer to the first element found with the given metaid.
   */ public
";


%javamethodmodifiers Model::getAllElements "
/**
   * Returns a {@link List} of all child {@link SBase} objects, including those nested to an arbitrary depth
   * <p>
   * @return a {@link List} of pointers to all children objects.
   */ public
";


%javamethodmodifiers Model::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link Model}.
   * <p>
   * @return the id of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link Model}.
   * <p>
   * @return the name of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getSubstanceUnits() const "
/**
   * Returns the value of the \'substanceUnits\' attribute of this {@link Model}.
   * <p>
   * @return the substanceUnits of this {@link Model}.
   * <p>
   * @note The \'substanceUnits\' attribute is available in
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::getTimeUnits() const "
/**
   * Returns the value of the \'timeUnits\' attribute of this {@link Model}.
   * <p>
   * @return the timeUnits of this {@link Model}.
   * <p>
   * @note The \'timeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::getVolumeUnits() const "
/**
   * Returns the value of the \'volumeUnits\' attribute of this {@link Model}.
   * <p>
   * @return the volumeUnits of this {@link Model}.
   * <p>
   * @note The \'volumeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::getAreaUnits() const "
/**
   * Returns the value of the \'areaUnits\' attribute of this {@link Model}.
   * <p>
   * @return the areaUnits of this {@link Model}.
   * <p>
   * @note The \'areaUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::getLengthUnits() const "
/**
   * Returns the value of the \'lengthUnits\' attribute of this {@link Model}.
   * <p>
   * @return the lengthUnits of this {@link Model}.
   * <p>
   * @note The \'lengthUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::getExtentUnits() const "
/**
   * Returns the value of the \'extentUnits\' attribute of this {@link Model}.
   * <p>
   * @return the extentUnits of this {@link Model}.
   * <p>
   * @note The \'extentUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::getConversionFactor() const "
/**
   * Returns the value of the \'conversionFactor\' attribute of this {@link Model}.
   * <p>
   * @return the conversionFactor of this {@link Model}.
   * <p>
   * @note The \'conversionFactor\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Model::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Model::isSetSubstanceUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'substanceUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'substanceUnits\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'substanceUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::isSetTimeUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'timeUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'timeUnits\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'substanceUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::isSetVolumeUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'volumeUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'volumeUnits\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'volumeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::isSetAreaUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'areaUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'areaUnits\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'areaUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::isSetLengthUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'lengthUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'lengthUnits\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'lengthUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::isSetExtentUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'extentUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'extentUnits\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'extentUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::isSetConversionFactor() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Model}\'s \'conversionFactor\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'conversionFactor\' attribute of this {@link Model} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'conversionFactor\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link Model}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Model::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link Model}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Model::setSubstanceUnits(const std::string& units) "
/**
   * Sets the value of the \'substanceUnits\' attribute of this {@link Model}.
   * <p>
   * The string in <code>units</code> is copied.
   * <p>
   * @param units the new substanceUnits for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'substanceUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::setTimeUnits(const std::string& units) "
/**
   * Sets the value of the \'timeUnits\' attribute of this {@link Model}.
   * <p>
   * The string in <code>units</code> is copied.
   * <p>
   * @param units the new timeUnits for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'timeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::setVolumeUnits(const std::string& units) "
/**
   * Sets the value of the \'volumeUnits\' attribute of this {@link Model}.
   * <p>
   * The string in <code>units</code> is copied.
   * <p>
   * @param units the new volumeUnits for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'volumeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::setAreaUnits(const std::string& units) "
/**
   * Sets the value of the \'areaUnits\' attribute of this {@link Model}.
   * <p>
   * The string in <code>units</code> is copied.
   * <p>
   * @param units the new areaUnits for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'areaUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::setLengthUnits(const std::string& units) "
/**
   * Sets the value of the \'lengthUnits\' attribute of this {@link Model}.
   * <p>
   * The string in <code>units</code> is copied.
   * <p>
   * @param units the new lengthUnits for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'lengthUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::setExtentUnits(const std::string& units) "
/**
   * Sets the value of the \'extentUnits\' attribute of this {@link Model}.
   * <p>
   * The string in <code>units</code> is copied.
   * <p>
   * @param units the new extentUnits for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'extentUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::setConversionFactor(const std::string& units) "
/**
   * Sets the value of the \'conversionFactor\' attribute of this {@link Model}.
   * <p>
   * The string in <code>units</code> is copied.
   * <p>
   * @param units the new conversionFactor for the {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'conversionFactor\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::unsetId "
/**
   * Unsets the value of the \'id\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Model::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Model::unsetSubstanceUnits "
/**
   * Unsets the value of the \'substanceUnits\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'substanceUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::unsetTimeUnits "
/**
   * Unsets the value of the \'timeUnits\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'timeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::unsetVolumeUnits "
/**
   * Unsets the value of the \'volumeUnits\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'volumeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::unsetAreaUnits "
/**
   * Unsets the value of the \'areaUnits\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'areaUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::unsetLengthUnits "
/**
   * Unsets the value of the \'lengthUnits\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'lengthUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::unsetExtentUnits "
/**
   * Unsets the value of the \'extentUnits\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'extentUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::unsetConversionFactor "
/**
   * Unsets the value of the \'conversionFactor\' attribute of this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'conversionFactor\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on {@link Model} in lower Levels of SBML.
   */ public
";


%javamethodmodifiers Model::addFunctionDefinition(const FunctionDefinition* fd) "
/**
   * Adds a copy of the given {@link FunctionDefinition} object to this {@link Model}.
   * <p>
   * @param fd the {@link FunctionDefinition} to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createFunctionDefinition()}
   * for a method that does not lead to these issues.
   * <p>
   * @see #createFunctionDefinition()
   */ public
";


%javamethodmodifiers Model::addUnitDefinition(const UnitDefinition* ud) "
/**
   * Adds a copy of the given {@link UnitDefinition} object to this {@link Model}.
   * <p>
   * @param ud the {@link UnitDefinition} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createUnitDefinition()} for
   * a method that does not lead to these issues.
   * <p>
   * @see #createUnitDefinition()
   */ public
";


%javamethodmodifiers Model::addCompartmentType(const CompartmentType* ct) "
/**
   * Adds a copy of the given {@link CompartmentType} object to this {@link Model}.
   * <p>
   * @param ct the {@link CompartmentType} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createCompartmentType()}
   * for a method that does not lead to these issues.
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   * <p>
   * @see #createCompartmentType()
   */ public
";


%javamethodmodifiers Model::addSpeciesType(const SpeciesType* st) "
/**
   * Adds a copy of the given {@link SpeciesType} object to this {@link Model}.
   * <p>
   * @param st the {@link SpeciesType} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createSpeciesType()} for a
   * method that does not lead to these issues.
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   * <p>
   * @see #createSpeciesType()
   */ public
";


%javamethodmodifiers Model::addCompartment(const Compartment* c) "
/**
   * Adds a copy of the given {@link Compartment} object to this {@link Model}.
   * <p>
   * @param c the {@link Compartment} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createCompartment()} for a
   * method that does not lead to these issues.
   * <p>
   * @see #createCompartment()
   */ public
";


%javamethodmodifiers Model::addSpecies(const Species* s) "
/**
   * Adds a copy of the given {@link Species} object to this {@link Model}.
   * <p>
   * @param s the {@link Species} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createSpecies()} for a
   * method that does not lead to these issues.
   * <p>
   * @see #createSpecies()
   */ public
";


%javamethodmodifiers Model::addParameter(const Parameter* p) "
/**
   * Adds a copy of the given {@link Parameter} object to this {@link Model}.
   * <p>
   * @param p the {@link Parameter} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createParameter()} for a
   * method that does not lead to these issues.
   * <p>
   * @see #createParameter()
   */ public
";


%javamethodmodifiers Model::addInitialAssignment(const InitialAssignment* ia) "
/**
   * Adds a copy of the given {@link InitialAssignment} object to this {@link Model}.
   * <p>
   * @param ia the {@link InitialAssignment} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createInitialAssignment()}
   * for a method that does not lead to these issues.
   * <p>
   * @see #createInitialAssignment()
   */ public
";


%javamethodmodifiers Model::addRule(const Rule* r) "
/**
   * Adds a copy of the given {@link Rule} object to this {@link Model}.
   * <p>
   * @param r the {@link Rule} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see the methods
   * {@link Model#createAlgebraicRule()}, {@link Model#createAssignmentRule()} and
   * {@link Model#createRateRule()} for methods that do not lead to these issues.
   * <p>
   * @see #createAlgebraicRule()
   * @see #createAssignmentRule()
   * @see #createRateRule()
   */ public
";


%javamethodmodifiers Model::addConstraint(const Constraint* c) "
/**
   * Adds a copy of the given {@link Constraint} object to this {@link Model}.
   * <p>
   * @param c the {@link Constraint} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createConstraint()} for a
   * method that does not lead to these issues.
   * <p>
   * @see #createConstraint()
   */ public
";


%javamethodmodifiers Model::addReaction(const Reaction* r) "
/**
   * Adds a copy of the given {@link Reaction} object to this {@link Model}.
   * <p>
   * @param r the {@link Reaction} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createReaction()} for a
   * method that does not lead to these issues.
   * <p>
   * @see #createReaction()
   */ public
";


%javamethodmodifiers Model::addEvent(const Event* e) "
/**
   * Adds a copy of the given {@link Event} object to this {@link Model}.
   * <p>
   * @param e the {@link Event} object to add
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Model}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Model}</em>.  In addition, the caller should make sure
   * to free the original object if it is no longer being used, or else a
   * memory leak will result.  Please see {@link Model#createEvent()} for a method
   * that does not lead to these issues.
   * <p>
   * @see #createEvent()
   */ public
";


%javamethodmodifiers Model::createFunctionDefinition "
/**
   * Creates a new {@link FunctionDefinition} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link FunctionDefinition} object created
   * <p>
   * @see #addFunctionDefinition(FunctionDefinition fd)
   */ public
";


%javamethodmodifiers Model::createUnitDefinition "
/**
   * Creates a new {@link UnitDefinition} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link UnitDefinition} object created
   * <p>
   * @see #addUnitDefinition(UnitDefinition ud)
   */ public
";


%javamethodmodifiers Model::createUnit "
/**
   * Creates a new {@link Unit} object within the last {@link UnitDefinition} object
   * created in this model and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The mechanism by which the {@link UnitDefinition} was created is not
   * significant.  If a {@link UnitDefinition} object does not exist in this model,
   * a new {@link Unit} is <em>not</em> created and <code>null</code> is returned instead.
   * <p>
   * @return the {@link Unit} object created
   * <p>
   * @see #addUnitDefinition(UnitDefinition ud)
   */ public
";


%javamethodmodifiers Model::createCompartmentType "
/**
   * Creates a new {@link CompartmentType} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link CompartmentType} object created
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   * <p>
   * @see #addCompartmentType(CompartmentType ct)
   */ public
";


%javamethodmodifiers Model::createSpeciesType "
/**
   * Creates a new {@link SpeciesType} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link SpeciesType} object created
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   * <p>
   * @see #addSpeciesType(SpeciesType st)
   */ public
";


%javamethodmodifiers Model::createCompartment "
/**
   * Creates a new {@link Compartment} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link Compartment} object created
   * <p>
   * @see #addCompartment(Compartment c)
   */ public
";


%javamethodmodifiers Model::createSpecies "
/**
   * Creates a new {@link Species} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link Species} object created
   * <p>
   * @see #addSpecies(Species s)
   */ public
";


%javamethodmodifiers Model::createParameter "
/**
   * Creates a new {@link Parameter} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link Parameter} object created
   * <p>
   * @see #addParameter(Parameter p)
   */ public
";


%javamethodmodifiers Model::createInitialAssignment "
/**
   * Creates a new {@link InitialAssignment} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link InitialAssignment} object created
   * <p>
   * @see #addInitialAssignment(InitialAssignment ia)
   */ public
";


%javamethodmodifiers Model::createAlgebraicRule "
/**
   * Creates a new {@link AlgebraicRule} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link AlgebraicRule} object created
   * <p>
   * @see #addRule(Rule r)
   */ public
";


%javamethodmodifiers Model::createAssignmentRule "
/**
   * Creates a new {@link AssignmentRule} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link AssignmentRule} object created
   * <p>
   * @see #addRule(Rule r)
   */ public
";


%javamethodmodifiers Model::createRateRule "
/**
   * Creates a new {@link RateRule} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link RateRule} object created
   * <p>
   * @see #addRule(Rule r)
   */ public
";


%javamethodmodifiers Model::createConstraint "
/**
   * Creates a new {@link Constraint} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link Constraint} object created
   * <p>
   * @see #addConstraint(Constraint c)
   */ public
";


%javamethodmodifiers Model::createReaction "
/**
   * Creates a new {@link Reaction} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link Reaction} object created
   * <p>
   * @see #addReaction(Reaction r)
   */ public
";


%javamethodmodifiers Model::createReactant "
/**
   * Creates a new {@link SpeciesReference} object for a reactant inside the last
   * {@link Reaction} object in this {@link Model}, and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The mechanism by which the last {@link Reaction} object was created and added
   * to this {@link Model} is not significant.  It could have been created in a
   * variety of ways, for example using createReaction().  If a {@link Reaction}
   * does not exist for this model, a new {@link SpeciesReference} is <em>not</em>
   * created and <code>null</code> is returned instead.
   * <p>
   * @return the {@link SpeciesReference} object created
   */ public
";


%javamethodmodifiers Model::createProduct "
/**
   * Creates a new {@link SpeciesReference} object for a product inside the last
   * {@link Reaction} object in this {@link Model}, and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The mechanism by which the last {@link Reaction} object was created and added
   * to this {@link Model} is not significant.  It could have been created in a
   * variety of ways, for example using createReaction().  If a {@link Reaction}
   * does not exist for this model, a new {@link SpeciesReference} is <em>not</em>
   * created and <code>null</code> is returned instead.
   * <p>
   * @return the {@link SpeciesReference} object created
   */ public
";


%javamethodmodifiers Model::createModifier "
/**
   * Creates a new {@link ModifierSpeciesReference} object for a modifier species
   * inside the last {@link Reaction} object in this {@link Model}, and returns a pointer
   * to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The mechanism by which the last {@link Reaction} object was created and added
   * to this {@link Model} is not significant.  It could have been created in a
   * variety of ways, for example using createReaction().  If a {@link Reaction}
   * does not exist for this model, a new {@link ModifierSpeciesReference} is 
   * <em>not</em> created and <code>null</code> is returned instead.
   * <p>
   * @return the {@link SpeciesReference} object created
   */ public
";


%javamethodmodifiers Model::createKineticLaw "
/**
   * Creates a new {@link KineticLaw} inside the last {@link Reaction} object created in
   * this {@link Model}, and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The mechanism by which the last {@link Reaction} object was created and added
   * to this {@link Model} is not significant.  It could have been created in a
   * variety of ways, for example using createReaction().  If a {@link Reaction}
   * does not exist for this model, or a {@link Reaction} exists but already has a
   * {@link KineticLaw}, a new {@link KineticLaw} is <em>not</em> created and <code>null</code> is returned
   * instead.
   * <p>
   * @return the {@link KineticLaw} object created
   */ public
";


%javamethodmodifiers Model::createKineticLawParameter "
/**
   * Creates a new local {@link Parameter} inside the {@link KineticLaw} object of the last
   * {@link Reaction} created inside this {@link Model}, and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The last {@link KineticLaw} object in this {@link Model} could have been created in a
   * variety of ways.  For example, it could have been added using
   * createKineticLaw(), or it could be the result of using
   * {@link Reaction#createKineticLaw()} on the {@link Reaction} object created by a
   * createReaction().  If a {@link Reaction} does not exist for this model, or the
   * last {@link Reaction} does not contain a {@link KineticLaw} object, a new {@link Parameter} is
   * <em>not</em> created and <code>null</code> is returned instead.
   * <p>
   * @return the {@link Parameter} object created
   */ public
";


%javamethodmodifiers Model::createKineticLawLocalParameter "
/**
   * Creates a new {@link LocalParameter} inside the {@link KineticLaw} object of the last
   * {@link Reaction} created inside this {@link Model}, and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The last {@link KineticLaw} object in this {@link Model} could have been created in a
   * variety of ways.  For example, it could have been added using
   * createKineticLaw(), or it could be the result of using
   * {@link Reaction#createKineticLaw()} on the {@link Reaction} object created by a
   * createReaction().  If a {@link Reaction} does not exist for this model, or the
   * last {@link Reaction} does not contain a {@link KineticLaw} object, a new {@link Parameter} is
   * <em>not</em> created and <code>null</code> is returned instead.
   * <p>
   * @return the {@link Parameter} object created
   */ public
";


%javamethodmodifiers Model::createEvent "
/**
   * Creates a new {@link Event} inside this {@link Model} and returns it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * @return the {@link Event} object created
   */ public
";


%javamethodmodifiers Model::createEventAssignment "
/**
   * Creates a new {@link EventAssignment} inside the last {@link Event} object created in
   * this {@link Model}, and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The mechanism by which the last {@link Event} object in this model was created
   * is not significant.  It could have been created in a variety of ways,
   * for example by using createEvent().  If no {@link Event} object exists in this
   * {@link Model} object, a new {@link EventAssignment} is <em>not</em> created and <code>null</code> is
   * returned instead.
   * <p>
   * @return the {@link EventAssignment} object created
   */ public
";


%javamethodmodifiers Model::createTrigger "
/**
   * Creates a new {@link Trigger} inside the last {@link Event} object created in
   * this {@link Model}, and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The mechanism by which the last {@link Event} object in this model was created
   * is not significant.  It could have been created in a variety of ways,
   * for example by using createEvent().  If no {@link Event} object exists in this
   * {@link Model} object, a new {@link Trigger} is <em>not</em> created and <code>null</code> is
   * returned instead.
   * <p>
   * @return the {@link Trigger} object created
   */ public
";


%javamethodmodifiers Model::createDelay "
/**
   * Creates a new {@link Delay} inside the last {@link Event} object created in
   * this {@link Model}, and returns a pointer to it.
   * <p>
   * The SBML Level and Version of the enclosing {@link Model} object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   * <p>
   * The mechanism by which the last {@link Event} object in this model was created
   * is not significant.  It could have been created in a variety of ways,
   * for example by using createEvent().  If no {@link Event} object exists in this
   * {@link Model} object, a new {@link Delay} is <em>not</em> created and <code>null</code> is
   * returned instead.
   * <p>
   * @return the {@link Delay} object created
   */ public
";


%javamethodmodifiers Model::setAnnotation(const XMLNode* annotation) "
/**
   * Sets the value of the \'annotation\' subelement of this SBML object to a
   * copy of <code>annotation</code>.
   * <p>
   * Any existing content of the \'annotation\' subelement is discarded.
   * Unless you have taken steps to first copy and reconstitute any
   * existing annotations into the <code>annotation</code> that is about to be
   * assigned, it is likely that performing such wholesale replacement is
   * unfriendly towards other software applications whose annotations are
   * discarded.  An alternative may be to use appendAnnotation().
   * <p>
   * @param annotation an XML structure that is to be used as the content
   * of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @see #appendAnnotation(XMLNode annotation)
   */ public
";


%javamethodmodifiers Model::setAnnotation(const std::string& annotation) "
/**
   * Sets the value of the \'annotation\' subelement of this SBML object to a
   * copy of <code>annotation</code>.
   * <p>
   * Any existing content of the \'annotation\' subelement is discarded.
   * Unless you have taken steps to first copy and reconstitute any
   * existing annotations into the <code>annotation</code> that is about to be
   * assigned, it is likely that performing such wholesale replacement is
   * unfriendly towards other software applications whose annotations are
   * discarded.  An alternative may be to use appendAnnotation().
   * <p>
   * @param annotation an XML string that is to be used as the content
   * of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #appendAnnotation(String annotation)
   */ public
";


%javamethodmodifiers Model::appendAnnotation(const XMLNode* annotation) "
/**
   * Appends annotation content to any existing content in the \'annotation\'
   * subelement of this object.
   * <p>
   * The content in <code>annotation</code> is copied.  Unlike setAnnotation(), this
   * method allows other annotations to be preserved when an application
   * adds its own data.
   * <p>
   * @param annotation an XML structure that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #setAnnotation(XMLNode annotation)
   */ public
";


%javamethodmodifiers Model::appendAnnotation(const std::string& annotation) "
/**
   * Appends annotation content to any existing content in the \'annotation\'
   * subelement of this object.
   * <p>
   * The content in <code>annotation</code> is copied.  Unlike setAnnotation(), this 
   * method allows other annotations to be preserved when an application
   * adds its own data.
   * <p>
   * @param annotation an XML string that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @see #setAnnotation(String annotation)
   */ public
";


%javamethodmodifiers Model::getListOfFunctionDefinitions() const "
/**
   * Get the {@link ListOfFunctionDefinitions} object in this {@link Model}.
   * <p>
   * @return the list of FunctionDefinitions for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfFunctionDefinitions "
/**
   * Get the {@link ListOfFunctionDefinitions} object in this {@link Model}.
   * <p>
   * @return the list of FunctionDefinitions for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfUnitDefinitions() const "
/**
   * Get the {@link ListOfUnitDefinitions} object in this {@link Model}.
   * <p>
   * @return the list of UnitDefinitions for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfUnitDefinitions "
/**
   * Get the {@link ListOfUnitDefinitions} object in this {@link Model}.
   * <p>
   * @return the list of UnitDefinitions for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfCompartmentTypes() const "
/**
   * Get the {@link ListOfCompartmentTypes} object in this {@link Model}.
   * <p>
   * @return the list of CompartmentTypes for this {@link Model}.
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getListOfCompartmentTypes "
/**
   * Get the {@link ListOfCompartmentTypes} object in this {@link Model}.
   * <p>
   * @return the list of CompartmentTypes for this {@link Model}.
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getListOfSpeciesTypes() const "
/**
   * Get the {@link ListOfSpeciesTypes} object in this {@link Model}.
   * <p>
   * @return the list of SpeciesTypes for this {@link Model}.
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getListOfSpeciesTypes "
/**
   * Get the {@link ListOfSpeciesTypes} object in this {@link Model}.
   * <p>
   * @return the list of SpeciesTypes for this {@link Model}.
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getListOfCompartments() const "
/**
   * Get the {@link ListOfCompartments} object in this {@link Model}.
   * <p>
   * @return the list of Compartments for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfCompartments "
/**
   * Get the {@link ListOfCompartments} object in this {@link Model}.
   * <p>
   * @return the list of Compartments for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfSpecies() const "
/**
   * Get the {@link ListOfSpecies} object in this {@link Model}.
   * <p>
   * @return the list of {@link Species} for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfSpecies "
/**
   * Get the {@link ListOfSpecies} object in this {@link Model}.
   * <p>
   * @return the list of {@link Species} for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfParameters() const "
/**
   * Get the {@link ListOfParameters} object in this {@link Model}.
   * <p>
   * @return the list of Parameters for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfParameters "
/**
   * Get the {@link ListOfParameters} object in this {@link Model}.
   * <p>
   * @return the list of Parameters for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfInitialAssignments() const "
/**
   * Get the {@link ListOfInitialAssignments} object in this {@link Model}.
   * <p>
   * @return the list of InitialAssignments for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfInitialAssignments "
/**
   * Get the {@link ListOfInitialAssignments} object in this {@link Model}.
   * <p>
   * @return the list of {@link InitialAssignment} for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfRules() const "
/**
   * Get the {@link ListOfRules} object in this {@link Model}.
   * <p>
   * @return the list of Rules for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfRules "
/**
   * Get the {@link ListOfRules} object in this {@link Model}.
   * <p>
   * @return the list of Rules for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfConstraints() const "
/**
   * Get the {@link ListOfConstraints} object in this {@link Model}.
   * <p>
   * @return the list of Constraints for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfConstraints "
/**
   * Get the {@link ListOfConstraints} object in this {@link Model}.
   * <p>
   * @return the list of Constraints for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfReactions() const "
/**
   * Get the {@link ListOfReactions} object in this {@link Model}.
   * <p>
   * @return the list of Reactions for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfReactions "
/**
   * Get the {@link ListOfReactions} object in this {@link Model}.
   * <p>
   * @return the list of Reactions for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfEvents() const "
/**
   * Get the {@link ListOfEvents} object in this {@link Model}.
   * <p>
   * @return the list of Events for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getListOfEvents "
/**
   * Get the {@link ListOfEvents} object in this {@link Model}.
   * <p>
   * @return the list of Events for this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getFunctionDefinition "
/**
   * Get the nth FunctionDefinitions object in this {@link Model}.
   * <p>
   * @return the nth {@link FunctionDefinition} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getFunctionDefinition(unsigned int n) "
/**
   * Get the nth FunctionDefinitions object in this {@link Model}.
   * <p>
   * @return the nth {@link FunctionDefinition} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getFunctionDefinition(const std::string& sid) const "
/**
   * Get a {@link FunctionDefinition} object based on its identifier.
   * <p>
   * @return the {@link FunctionDefinition} in this {@link Model} with the identifier
   * <code>sid</code> or <code>null</code> if no such {@link FunctionDefinition} exists.
   */ public
";


%javamethodmodifiers Model::getFunctionDefinition(const std::string& sid) "
/**
   * Get a {@link FunctionDefinition} object based on its identifier.
   * <p>
   * @return the {@link FunctionDefinition} in this {@link Model} with the identifier
   * <code>sid</code> or <code>null</code> if no such {@link FunctionDefinition} exists.
   */ public
";


%javamethodmodifiers Model::getUnitDefinition "
/**
   * Get the nth {@link UnitDefinition} object in this {@link Model}.
   * <p>
   * @return the nth {@link UnitDefinition} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getUnitDefinition(unsigned int n) "
/**
   * Get the nth {@link UnitDefinition} object in this {@link Model}.
   * <p>
   * @return the nth {@link UnitDefinition} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getUnitDefinition(const std::string& sid) const "
/**
   * Get a {@link UnitDefinition} based on its identifier.
   * <p>
   * @return the {@link UnitDefinition} in this {@link Model} with the identifier <code>sid</code> or
   * <code>null</code> if no such {@link UnitDefinition} exists.
   */ public
";


%javamethodmodifiers Model::getUnitDefinition(const std::string& sid) "
/**
   * Get a {@link UnitDefinition} based on its identifier.
   * <p>
   * @return the {@link UnitDefinition} in this {@link Model} with the identifier <code>sid</code> or
   * <code>null</code> if no such {@link UnitDefinition} exists.
   */ public
";


%javamethodmodifiers Model::getCompartmentType "
/**
   * Get the nth {@link CompartmentType} object in this {@link Model}.
   * <p>
   * @return the nth {@link CompartmentType} of this {@link Model}.
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getCompartmentType(unsigned int n) "
/**
   * Get the nth {@link CompartmentType} object in this {@link Model}.
   * <p>
   * @return the nth {@link CompartmentType} of this {@link Model}.
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getCompartmentType(const std::string& sid) const "
/**
   * Get a {@link CompartmentType} object based on its identifier.
   * <p>
   * @return the {@link CompartmentType} in this {@link Model} with the identifier <code>sid</code>
   * or <code>null</code> if no such {@link CompartmentType} exists.
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getCompartmentType(const std::string& sid) "
/**
   * Get a {@link CompartmentType} object based on its identifier.
   * <p>
   * @return the {@link CompartmentType} in this {@link Model} with the identifier <code>sid</code>
   * or <code>null</code> if no such {@link CompartmentType} exists.
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getSpeciesType "
/**
   * Get the nth {@link SpeciesType} object in this {@link Model}.
   * <p>
   * @return the nth {@link SpeciesType} of this {@link Model}.
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getSpeciesType(unsigned int n) "
/**
   * Get the nth {@link SpeciesType} object in this {@link Model}.
   * <p>
   * @return the nth {@link SpeciesType} of this {@link Model}.
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getSpeciesType(const std::string& sid) const "
/**
   * Get a {@link SpeciesType} object based on its identifier.
   * <p>
   * @return the {@link SpeciesType} in this {@link Model} with the identifier <code>sid</code> or
   * <code>null</code> if no such {@link SpeciesType} exists.
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getSpeciesType(const std::string& sid) "
/**
   * Get a {@link SpeciesType} object based on its identifier.
   * <p>
   * @return the {@link SpeciesType} in this {@link Model} with the identifier <code>sid</code> or
   * <code>null</code> if no such {@link SpeciesType} exists.
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getCompartment "
/**
   * Get the nth {@link Compartment} object in this {@link Model}.
   * <p>
   * @return the nth {@link Compartment} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getCompartment(unsigned int n) "
/**
   * Get the nth {@link Compartment} object in this {@link Model}.
   * <p>
   * @return the nth {@link Compartment} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getCompartment(const std::string& sid) const "
/**
   * Get a {@link Compartment} object based on its identifier.
   * <p>
   * @return the {@link Compartment} in this {@link Model} with the identifier <code>sid</code> or
   * <code>null</code> if no such {@link Compartment} exists.
   */ public
";


%javamethodmodifiers Model::getCompartment(const std::string& sid) "
/**
   * Get a {@link Compartment} object based on its identifier.
   * <p>
   * @return the {@link Compartment} in this {@link Model} with the identifier <code>sid</code> or
   * <code>null</code> if no such {@link Compartment} exists.
   */ public
";


%javamethodmodifiers Model::getSpecies "
/**
   * Get the nth {@link Species} object in this {@link Model}.
   * <p>
   * @return the nth {@link Species} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getSpecies(unsigned int n) "
/**
   * Get the nth {@link Species} object in this {@link Model}.
   * <p>
   * @return the nth {@link Species} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getSpecies(const std::string& sid) const "
/**
   * Get a {@link Species} object based on its identifier.
   * <p>
   * @return the {@link Species} in this {@link Model} with the identifier <code>sid</code> or <code>null</code>
   * if no such {@link Species} exists.
   */ public
";


%javamethodmodifiers Model::getSpecies(const std::string& sid) "
/**
   * Get a {@link Species} object based on its identifier.
   * <p>
   * @return the {@link Species} in this {@link Model} with the identifier <code>sid</code> or <code>null</code>
   * if no such {@link Species} exists.
   */ public
";


%javamethodmodifiers Model::getParameter "
/**
   * Get the nth {@link Parameter} object in this {@link Model}.
   * <p>
   * @return the nth {@link Parameter} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getParameter(unsigned int n) "
/**
   * Get the nth {@link Parameter} object in this {@link Model}.
   * <p>
   * @return the nth {@link Parameter} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getParameter(const std::string& sid) const "
/**
   * Get a {@link Parameter} object based on its identifier.
   * <p>
   * @return the {@link Parameter} in this {@link Model} with the identifier <code>sid</code> or <code>null</code>
   * if no such {@link Parameter} exists.
   */ public
";


%javamethodmodifiers Model::getParameter(const std::string& sid) "
/**
   * Get a {@link Parameter} object based on its identifier.
   * <p>
   * @return the {@link Parameter} in this {@link Model} with the identifier <code>sid</code> or <code>null</code>
   * if no such {@link Parameter} exists.
   */ public
";


%javamethodmodifiers Model::getInitialAssignment "
/**
   * Get the nth {@link InitialAssignment} object in this {@link Model}.
   * <p>
   * @return the nth {@link InitialAssignment} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getInitialAssignment(unsigned int n) "
/**
   * Get the nth {@link InitialAssignment} object in this {@link Model}.
   * <p>
   * @return the nth {@link InitialAssignment} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getInitialAssignment(const std::string& symbol) const "
/**
   * Get an {@link InitialAssignment} object based on the symbol to which it
   * assigns a value.
   * <p>
   * @return the {@link InitialAssignment} in this {@link Model} with the given \'symbol\'
   * attribute value or <code>null</code> if no such {@link InitialAssignment} exists.
   */ public
";


%javamethodmodifiers Model::getInitialAssignment(const std::string& symbol) "
/**
   * Get an {@link InitialAssignment} object based on the symbol to which it
   * assigns a value.
   * <p>
   * @return the {@link InitialAssignment} in this {@link Model} with the given \'symbol\'
   * attribute value or <code>null</code> if no such {@link InitialAssignment} exists.
   */ public
";


%javamethodmodifiers Model::getRule "
/**
   * Get the nth {@link Rule} object in this {@link Model}.
   * <p>
   * @return the nth {@link Rule} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getRule(unsigned int n) "
/**
   * Get the nth {@link Rule} object in this {@link Model}.
   * <p>
   * @return the nth {@link Rule} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getRule(const std::string& variable) const "
/**
   * Get a {@link Rule} object based on the variable to which it assigns a value.
   * <p>
   * @return the {@link Rule} in this {@link Model} with the given \'variable\' attribute
   * value or <code>null</code> if no such {@link Rule} exists.
   */ public
";


%javamethodmodifiers Model::getRule(const std::string& variable) "
/**
   * Get a {@link Rule} object based on the variable to which it assigns a value.
   * <p>
   * @return the {@link Rule} in this {@link Model} with the given \'variable\' attribute
   * value or <code>null</code> if no such {@link Rule} exists.
   */ public
";


%javamethodmodifiers Model::getConstraint "
/**
   * Get the nth {@link Constraint} object in this {@link Model}.
   * <p>
   * @return the nth {@link Constraint} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getConstraint(unsigned int n) "
/**
   * Get the nth {@link Constraint} object in this {@link Model}.
   * <p>
   * @return the nth {@link Constraint} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getReaction "
/**
   * Get the nth {@link Reaction} object in this {@link Model}.
   * <p>
   * @return the nth {@link Reaction} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getReaction(unsigned int n) "
/**
   * Get the nth {@link Reaction} object in this {@link Model}.
   * <p>
   * @return the nth {@link Reaction} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getReaction(const std::string& sid) const "
/**
   * Get a {@link Reaction} object based on its identifier.
   * <p>
   * @return the {@link Reaction} in this {@link Model} with the identifier <code>sid</code> or <code>null</code>
   * if no such {@link Reaction} exists.
   */ public
";


%javamethodmodifiers Model::getReaction(const std::string& sid) "
/**
   * Get a {@link Reaction} object based on its identifier.
   * <p>
   * @return the {@link Reaction} in this {@link Model} with the identifier <code>sid</code> or <code>null</code>
   * if no such {@link Reaction} exists.
   */ public
";


%javamethodmodifiers Model::getSpeciesReference(const std::string& sid) "
/**
   * Get a {@link SpeciesReference} object based on its identifier.
   * <p>
   * @return the {@link SpeciesReference} in this {@link Model} with the identifier <code>sid</code> or <code>null</code>
   * if no such {@link SpeciesReference} exists.
   */ public
";


%javamethodmodifiers Model::getSpeciesReference(const std::string& sid) const "
/**
   * Get a {@link SpeciesReference} object based on its identifier.
   * <p>
   * @return the {@link SpeciesReference} in this {@link Model} with the identifier <code>sid</code> or <code>null</code>
   * if no such {@link SpeciesReference} exists.
   */ public
";


%javamethodmodifiers Model::getEvent "
/**
   * Get the nth {@link Event} object in this {@link Model}.
   * <p>
   * @return the nth {@link Event} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getEvent(unsigned int n) "
/**
   * Get the nth {@link Event} object in this {@link Model}.
   * <p>
   * @return the nth {@link Event} of this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getEvent(const std::string& sid) const "
/**
   * Get an {@link Event} object based on its identifier.
   * <p>
   * @return the {@link Event} in this {@link Model} with the identifier <code>sid</code> or <code>null</code> if
   * no such {@link Event} exists.
   */ public
";


%javamethodmodifiers Model::getEvent(const std::string& sid) "
/**
   * Get an {@link Event} object based on its identifier.
   * <p>
   * @return the {@link Event} in this {@link Model} with the identifier <code>sid</code> or <code>null</code> if
   * no such {@link Event} exists.
   */ public
";


%javamethodmodifiers Model::getNumFunctionDefinitions() const "
/**
   * Get the number of {@link FunctionDefinition} objects in this {@link Model}.
   * <p>
   * @return the number of FunctionDefinitions in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getNumUnitDefinitions() const "
/**
   * Get the number of {@link UnitDefinition} objects in this {@link Model}.
   * <p>
   * @return the number of UnitDefinitions in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getNumCompartmentTypes() const "
/**
   * Get the number of {@link CompartmentType} objects in this {@link Model}.
   * <p>
   * @return the number of CompartmentTypes in this {@link Model}.
   * <p>
   * @note The {@link CompartmentType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getNumSpeciesTypes() const "
/**
   * Get the number of {@link SpeciesType} objects in this {@link Model}.
   * <p>
   * @return the number of SpeciesTypes in this {@link Model}.
   * <p>
   * @note The {@link SpeciesType} object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%javamethodmodifiers Model::getNumCompartments() const "
/**
   * Get the number of {@link Compartment} objects in this {@link Model}.
   * <p>
   * @return the number of Compartments in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getNumSpecies() const "
/**
   * Get the number of Specie objects in this {@link Model}.
   * <p>
   * @return the number of {@link Species} in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getNumSpeciesWithBoundaryCondition() const "
/**
   * Get the number of {@link Species} in this {@link Model} having their
   * \'boundaryCondition\' attribute value set to <code>true.</code>
   * <p>
   * @return the number of {@link Species} in this {@link Model} with boundaryCondition set
   * to true.
   */ public
";


%javamethodmodifiers Model::getNumParameters() const "
/**
   * Get the number of {@link Parameter} objects in this {@link Model}.
   * <p>
   * @return the number of Parameters in this {@link Model}.  Parameters defined in
   * KineticLaws are not included.
   */ public
";


%javamethodmodifiers Model::getNumInitialAssignments() const "
/**
   * Get the number of {@link InitialAssignment} objects in this {@link Model}.
   * <p>
   * @return the number of InitialAssignments in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getNumRules() const "
/**
   * Get the number of {@link Rule} objects in this {@link Model}.
   * <p>
   * @return the number of Rules in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getNumConstraints() const "
/**
   * Get the number of {@link Constraint} objects in this {@link Model}.
   * <p>
   * @return the number of Constraints in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getNumReactions() const "
/**
   * Get the number of {@link Reaction} objects in this {@link Model}.
   * <p>
   * @return the number of Reactions in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::getNumEvents() const "
/**
   * Get the number of {@link Event} objects in this {@link Model}.
   * <p>
   * @return the number of Events in this {@link Model}.
   */ public
";


%javamethodmodifiers Model::removeFromParentAndDelete "
/**
   * Finds this {@link Model}\'s parent {@link SBMLDocument} and calls setModel(null) on it, indirectly deleting itself.  Overridden from the {@link SBase} function since the parent is not a {@link ListOf}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Model::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%javamethodmodifiers Model::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers Model::isBoolean(const ASTNode* node) const "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertL1ToL2 "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertL1ToL3 "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertL2ToL3 "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertL2ToL1(bool strict = false) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertL3ToL1 "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertL3ToL2(bool strict = false) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::addModifiers "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::addConstantAttribute "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::setSpatialDimensions(double dims = 3.0) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::addDefinitionsForDefaultUnits "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertParametersToLocals(unsigned int level, unsigned int version) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::setSpeciesReferenceConstantValueAndStoichiometry "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::removeMetaId "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::removeSBOTerms(bool strict) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::removeHasOnlySubstanceUnits "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::removeSBOTermsNotInL2V2(bool strict) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::removeDuplicateTopLevelAnnotations "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::removeParameterRuleUnits(bool strict) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertStoichiometryMath "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::assignRequiredValues "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::dealWithModelUnits "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::dealWithStoichiometry "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::dealWithEvents(bool strict) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::convertToL2Strict "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::setSBMLDocument(SBMLDocument* d) "
/**
   * Predicate returning <code>true</code> if the
   * given {@link ASTNode} is a boolean.
   * <p>
   * Often times, this question can be answered with the {@link ASTNode}\'s own
   * isBoolean() method, but if the AST is an expression that calls a
   * function defined in the {@link Model}\'s {@link ListOfFunctionDefinitions}, the model
   * is needed for lookup context.
   * <p>
   * @return true if the given {@link ASTNode} is a boolean.
   * @internal
   */ public
";


%javamethodmodifiers Model::connectToChild "
/**
   * Sets this SBML object to child SBML objects (if any).
   * (Creates a child-parent relationship by the parent)
   * <p>
   * Subclasses must override this function if they define
   * one ore more child elements.
   * Basically, this function needs to be called in
   * constructor, copy constructor and assignment operator.
   * <p>
   * @see setSBMLDocument
   * @see enablePackageInternal
   * @internal
   */ public
";


%javamethodmodifiers Model::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers Model::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link Model}, is
   * always <code>\'model\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'model\'.</code>
   */ public
";


%javamethodmodifiers Model::getElementPosition() const "
/**
   * @return the ordinal position of the element with respect to its
   * siblings or -1 (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers Model::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Model::populateListFormulaUnitsData "
/**
   * Populates the list of FormulaDataUnits with the units derived 
   * for the model. The list contains elements of class
   * FormulaUnitsData. 
   * <p>
   * The first element of the list refers to the default units
   * of \'substance per time\' derived from the model and has the
   * unitReferenceId \'subs_per_time\'. This facilitates the comparison of units
   * derived from mathematical formula with the expected units.
   * <p>
   * The next elements of the list record the units of the 
   * compartments and species established from either explicitly
   * declared or default units.
   * <p>
   * The next elements record the units of any parameters.
   * <p>
   * Subsequent elements of the list record the units derived for
   * each mathematical expression encountered within the model.
   * <p>
   * @note This function is utilised by the {@link Unit} Consistency Validator.
   * The list is populated prior to running the validation and thus
   * the consistency of units can be checked by accessing the members
   * of the list and comparing the appropriate data.
   */ public
";


%javamethodmodifiers Model::isPopulatedListFormulaUnitsData "
/**
   * Predicate returning <code>true</code> if 
   * the list of FormulaUnitsData is populated.
   * <p>
   * @return <code>true</code> if the list of FormulaUnitsData is populated, 
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Model::addFormulaUnitsData(const FormulaUnitsData* fud) "
/**
   * Adds a copy of the given FormulaUnitsData object to this {@link Model}.
   * <p>
   * @param fud the FormulaUnitsData to add
   * @internal
   */ public
";


%javamethodmodifiers Model::createFormulaUnitsData "
/**
   * Creates a new FormulaUnitsData inside this {@link Model} and returns it.
   * <p>
   * @return the FormulaUnitsData object created
   * @internal
   */ public
";


%javamethodmodifiers Model::getFormulaUnitsData "
/**
   * Get the nth FormulaUnitsData object in this {@link Model}.
   * <p>
   * @return the nth FormulaUnitsData of this {@link Model}.
   * @internal
   */ public
";


%javamethodmodifiers Model::getFormulaUnitsData(unsigned int n) "
/**
   * Get the nth FormulaUnitsData object in this {@link Model}.
   * <p>
   * @return the nth FormulaUnitsData of this {@link Model}.
   * @internal
   */ public
";


%javamethodmodifiers Model::getFormulaUnitsData(const std::string& sid, int typecode) const "
/**
   * Get a FormulaUnitsData object based on its unitReferenceId and typecode.
   * <p>
   * @return the FormulaUnitsData in this {@link Model} with the unitReferenceId <code>sid</code> 
   * and the typecode (int) <code>typecode</code> or <code>null</code>
   * if no such FormulaUnitsData exists.
   * <p>
   * @note The typecode (int) parameter is necessary as the unitReferenceId
   * of the FormulaUnitsData need not be unique. For example if a {@link Species}
   * with id \'s\' is assigned by an {@link AssignmentRule} there will be two 
   * elements of the FormulaUnitsData list with the unitReferenceId \'s\'; 
   * one with
   * typecode \'SBML_SPECIES\' referring to the units related to the species, 
   * the other with typecode \'SBML_ASSIGNMENT_RULE\' referring to the units
   * derived from the math element of the {@link AssignmentRule}.
   * @internal
   */ public
";


%javamethodmodifiers Model::getFormulaUnitsData(const std::string& sid, int) "
/**
   * Get a FormulaUnitsData object based on its unitReferenceId and typecode.
   * <p>
   * @return the FormulaUnitsData in this {@link Model} with the unitReferenceId <code>sid</code> 
   * and the typecode (int) <code>typecode</code> or <code>null</code>
   * if no such FormulaUnitsData exists.
   * <p>
   * @note The typecode (int) parameter is necessary as the unitReferenceId
   * of the FormulaUnitsData need not be unique. For example if a {@link Species}
   * with id \'s\' is assigned by an {@link AssignmentRule} there will be two 
   * elements of the FormulaUnitsData list with the unitReferenceId \'s\'; 
   * one with
   * typecode \'SBML_SPECIES\' referring to the units related to the species, 
   * the other with typecode \'SBML_ASSIGNMENT_RULE\' referring to the units
   * derived from the math element of the {@link AssignmentRule}.
   * @internal
   */ public
";


%javamethodmodifiers Model::getNumFormulaUnitsData() const "
/**
   * Get the number of FormulaUnitsData objects in this {@link Model}.
   * <p>
   * @return the number of FormulaUnitsData in this {@link Model}.
   * @internal
   */ public
";


%javamethodmodifiers Model::getListFormulaUnitsData "
/**
   * Get the list of FormulaUnitsData object in this {@link Model}.
   * <p>
   * @return the list of FormulaUnitsData for this {@link Model}.
   * @internal
   */ public
";


%javamethodmodifiers Model::getListFormulaUnitsData() const "
/**
   * Get the list of FormulaUnitsData object in this {@link Model}.
   * <p>
   * @return the list of FormulaUnitsData for this {@link Model}.
   * @internal
   */ public
";


%javamethodmodifiers Model::hasRequiredElements() const "
/**
   * Predicate returning <code>true</code> if
   * all the required elements for this {@link Model} object
   * have been set.
   * <p>
   * @note The required elements for a {@link Model} object are:
   * listOfCompartments (L1 only); listOfSpecies (L1V1 only);
   * listOfReactions(L1V1 only)
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers Model::removeFunctionDefinition(unsigned int n) "
/**
   * Removes the nth {@link FunctionDefinition} object from this {@link Model} object and 
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link FunctionDefinition} object to remove
   * <p>
   * @return the {@link FunctionDefinition} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeFunctionDefinition(const std::string& sid) "
/**
   * Removes the {@link FunctionDefinition} object with the given identifier from this {@link Model} 
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link FunctionDefinition} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the {@link FunctionDefinition} object to remove
   * <p>
   * @return the {@link FunctionDefinition} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link FunctionDefinition}
   * object with the identifier exists in this {@link Model} object.
   */ public
";


%javamethodmodifiers Model::removeUnitDefinition(unsigned int n) "
/**
   * Removes the nth {@link UnitDefinition} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link UnitDefinition} object to remove
   * <p>
   * @return the {@link UnitDefinition} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeUnitDefinition(const std::string& sid) "
/**
   * Removes the {@link UnitDefinition} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link UnitDefinition} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the {@link UnitDefinition} object to remove
   * <p>
   * @return the {@link UnitDefinition} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link UnitDefinition}
   * object with the identifier exists in this {@link Model} object.
   */ public
";


%javamethodmodifiers Model::removeCompartmentType(unsigned int n) "
/**
   * Removes the nth {@link CompartmentType} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link CompartmentType} object to remove
   * <p>
   * @return the ComapartmentType object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeCompartmentType(const std::string& sid) "
/**
   * Removes the {@link CompartmentType} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link CompartmentType} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the object to remove
   * <p>
   * @return the {@link CompartmentType} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link CompartmentType}
   * object with the identifier exists in this {@link Model} object.
   */ public
";


%javamethodmodifiers Model::removeSpeciesType(unsigned int n) "
/**
   * Removes the nth {@link SpeciesType} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link SpeciesType} object to remove
   * <p>
   * @return the {@link SpeciesType} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeSpeciesType(const std::string& sid) "
/**
   * Removes the {@link SpeciesType} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link SpeciesType} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the {@link SpeciesType} object to remove
   * <p>
   * @return the {@link SpeciesType} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link SpeciesType}
   * object with the identifier exists in this {@link Model} object.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeCompartment(unsigned int n) "
/**
   * Removes the nth {@link Compartment} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Compartment} object to remove
   * <p>
   * @return the {@link Compartment} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeCompartment(const std::string& sid) "
/**
   * Removes the {@link Compartment} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Compartment} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Compartment} object to remove
   * <p>
   * @return the {@link Compartment} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link Compartment}
   * object with the identifier exists in this {@link Model} object.
   */ public
";


%javamethodmodifiers Model::removeSpecies(unsigned int n) "
/**
   * Removes the nth {@link Species} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Species} object to remove
   * <p>
   * @return the {@link Species} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeSpecies(const std::string& sid) "
/**
   * Removes the {@link Species} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Species} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Species} object to remove
   * <p>
   * @return the {@link Species} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link Species}
   * object with the identifier exists in this {@link Model} object.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeParameter(unsigned int n) "
/**
   * Removes the nth {@link Parameter} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Parameter} object to remove
   * <p>
   * @return the {@link Parameter} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeParameter(const std::string& sid) "
/**
   * Removes the {@link Parameter} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Parameter} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Parameter} object to remove
   * <p>
   * @return the {@link Parameter} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link Parameter}
   * object with the identifier exists in this {@link Model} object.
   */ public
";


%javamethodmodifiers Model::removeInitialAssignment(unsigned int n) "
/**
   * Removes the nth {@link InitialAssignment} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link InitialAssignment} object to remove
   * <p>
   * @return the {@link InitialAssignment} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeInitialAssignment(const std::string& symbol) "
/**
   * Removes the {@link InitialAssignment} object with the given \'symbol\' attribute 
   * from this {@link Model} object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link InitialAssignment} objects in this {@link Model} object have the
   * \'symbol\' attribute <code>symbol</code>, then <code>null</code> is returned.
   * <p>
   * @param symbol the \'symbol\' attribute of the {@link InitialAssignment} object to remove
   * <p>
   * @return the {@link InitialAssignment} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link InitialAssignment}
   * object with the \'symbol\' attribute exists in this {@link Model} object.
   */ public
";


%javamethodmodifiers Model::removeRule(unsigned int n) "
/**
   * Removes the nth {@link Rule} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Rule} object to remove
   * <p>
   * @return the {@link Rule} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeRule(const std::string& variable) "
/**
   * Removes the {@link Rule} object with the given \'variable\' attribute from this {@link Model} 
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Rule} objects in this {@link Model} object have the \'variable\' attribute
   * <code>variable</code>, then <code>null</code> is returned.
   * <p>
   * @param variable the \'variable\' attribute of the {@link Rule} object to remove
   * <p>
   * @return the {@link Rule} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link Rule}
   * object with the \'variable\' attribute exists in this {@link Model} object.
   */ public
";


%javamethodmodifiers Model::removeConstraint(unsigned int n) "
/**
   * Removes the nth {@link Constraint} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Constraint} object to remove
   * <p>
   * @return the {@link Constraint} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeReaction(unsigned int n) "
/**
   * Removes the nth {@link Reaction} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Reaction} object to remove
   * <p>
   * @return the {@link Reaction} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeReaction(const std::string& sid) "
/**
   * Removes the {@link Reaction} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Reaction} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Reaction} object to remove
   * <p>
   * @return the {@link Reaction} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link Reaction}
   * object with the identifier exists in this {@link Model} object.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeEvent(unsigned int n) "
/**
   * Removes the nth {@link Event} object from this {@link Model} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Event} object to remove
   * <p>
   * @return the {@link Event} object removed.  As mentioned above, 
   * the caller owns the returned item. <code>null</code> is returned if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers Model::removeEvent(const std::string& sid) "
/**
   * Removes the {@link Event} object with the given identifier from this {@link Model}
   * object and returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * If none of the {@link Event} objects in this {@link Model} object have the identifier 
   * <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the {@link Event} object to remove
   * <p>
   * @return the {@link Event} object removed.  As mentioned above, the 
   * caller owns the returned object. <code>null</code> is returned if no {@link Event}
   * object with the identifier exists in this {@link Model} object.
   * <p>
   */ public
";


%javamethodmodifiers Model::appendFrom(const Model* model) "
/**
   * Takes the contents of the passed-in {@link Model}, makes copies of everything, and appends those copies to the appropriate places in this {@link Model}.  Also calls \'appendFrom\' on all plugin objects.
   * <p>
   * @param model the {@link Model} to merge with this one.
   * <p>
   */ public
";


%javamethodmodifiers Model::enablePackageInternal(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/**
   * Enables/Disables the given package with this element and child
   * elements (if any).
   * (This is an internal implementation for enablePackage function)
   * <p>
   * @note Subclasses of the SBML Core package in which one or more child
   * elements are defined must override this function.
   * @internal
   */ public
";


%javamethodmodifiers Model::readOtherXML(XMLInputStream& stream) "
/**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   * @internal
   */ public
";


%javamethodmodifiers Model::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%javamethodmodifiers Model::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers Model::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Model::readL1Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Model::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Model::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Model::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Model::syncAnnotation "
/**
   * Synchronizes the annotation of this SBML object.
   * <p>
   * Annotation element (XMLNode mAnnotation) is synchronized with the
   * current {@link CVTerm} objects (List mCVTerm), {@link ModelHistory} object 
   * (ModelHistory mHistory) and ListOfLayouts object (ListOfLayouts mLayouts).
   * Currently, this method is called in getAnnotation, isSetAnnotation,
   * and writeElements methods.
   * @internal
   */ public
";


%typemap(javaimports) SBMLDocument "
/** 
 * Container for an SBML document and interface for global operations
 * on SBML documents.
 * <p>
 * LibSBML uses the class {@link SBMLDocument} as a top-level
 * container for storing SBML content and data associated with it (such as
 * warnings and error messages).  The two primary means of reading an SBML
 * model, {@link SBMLReader#readSBML(String filename)} and
 * {@link SBMLReader#readSBMLFromString(String xml)}, both return an {@link SBMLDocument}
 * object.  From there, callers can inquire about any errors encountered
 * (e.g., using {@link SBMLDocument#getNumErrors()}), access the {@link Model} object, and
 * perform other actions such as consistency-checking and model
 * translation.
 * <p>
 * <p>
 * When creating fresh models programmatically, the starting point is
 * typically the creation of an {@link SBMLDocument} object instance.  The
 * {@link SBMLDocument} constructor accepts arguments for the SBML Level and
 * Version of the model to be created.  After creating the {@link SBMLDocument}
 * object, calling programs then typically call {@link SBMLDocument#createModel()}
 * almost immediately, and then proceed to call the methods on the {@link Model}
 * object to fill out the model\'s contents.
 * <p>
 * {@link SBMLDocument} corresponds roughly to the class <i>Sbml</i> defined in the
 * SBML Level&nbsp;2 specification and <i>SBML</i> in the Level&nbsp;3
 * specification.  It does not have a direct correspondence in SBML
 * Level&nbsp;1.  (However, to make matters simpler for applications,
 * libSBML creates an {@link SBMLDocument} no matter whether the model is
 * Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)  In its barest form, when written out in
 * XML format for (e.g.) SBML Level&nbsp;2 Version&nbsp;4, the corresponding
 * structure is the following:
 * <div class=\'fragment\'><pre>
&lt;sbml xmlns=\'http://www.sbml.org/sbml/level2/version4\' level=\'2\' version=\'4\'&gt;
  ...
&lt;/sbml&gt;</pre></div>
 * <p>
 * {@link SBMLDocument} is derived from {@link SBase}, and therefore contains the usual {@link SBase}
 * attributes (in SBML Level&nbsp;2 and Level&nbsp;3) of \'metaid\' and \'sboTerm\', as
 * well as the subelements \'notes\' and \'annotation\'.  It also contains the
 * attributes \'level\' and \'version\' indicating the Level and Version of the
 * SBML data structure.  These can be accessed using the methods defined by
 * the {@link SBase} class for that purpose.
 * <p>
 * <h2>Checking consistency and adherence to SBML specifications</h2>
 * <p>
 * One of the most important features of libSBML is its ability to perform
 * SBML validation to ensure that a model adheres to the SBML specification
 * for whatever Level+Version combination the model uses.  {@link SBMLDocument}
 * provides the methods for running consistency-checking and validation
 * rules on the SBML content.
 * <p>
 * First, a brief explanation of the rationale is in order.  In libSBML
 * versions up to and including the version&nbsp;3.3.x series, the
 * individual methods for creating and setting attributes and other
 * components were quite lenient, and allowed a caller to compose SBML
 * entities that might not, in the end, represent valid SBML.  This allowed
 * applications the freedom to do things such as save incomplete models
 * (which is useful when models are being developed over long periods of
 * time).  In the version&nbsp;4.x series, libSBML is somewhat stricter,
 * but still permits structures to be created independently and the results
 * to be combined in a separate step.  In all these cases, it means that a
 * separate validation step is necessary when a calling program finally
 * wants to finish a complete SBML document.
 * <p>
 * The primary interface to this validation facility is {@link SBMLDocument}\'s
 * {@link SBMLDocument#checkInternalConsistency()} and
 * {@link SBMLDocument#checkConsistency()}.  The former verifies the basic
 * internal consistency and syntax of an SBML document, and the latter
 * implements more elaborate validation rules (both those defined by the
 * SBML specifications, as well as additional rules offered by libSBML).
 * <p>
 * The checks performed by {@link SBMLDocument#checkInternalConsistency()} are
 * hardwired and cannot be changed by calling programs, but the validation
 * performed by {@link SBMLDocument#checkConsistency()} is under program control
 * using the method {@link SBMLDocument#setConsistencyChecks(int categ, boolean
 * onoff)}.  Applications can selectively disable specific kinds of checks
 * that they may not be interested by calling
 * {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)} with
 * appropriate parameters.
 * <p>
 * <p>
 * These methods have slightly different relevance depending on whether a
 * model is created programmaticaly from scratch, or whether it is read in
 * from a file or data stream.  The following list summarizes the possible
 * scenarios.
 * <p>
 * <em>Scenario 1: Creating a model from scratch</em>.  Before writing out
 * the model, 
 * <p>
 * <ul>
 * <li> Call {@link SBMLDocument#checkInternalConsistency()}, then inquire about
 * the results by calling {@link SBMLDocument#getNumErrors()}
 * </ul>
 * <p>
 * <ul>
 * <li> Call {@link SBMLDocument#setConsistencyChecks(int categ, boolean
 * onoff)}  to configure
 * which checks will be performed by {@link SBMLDocument#checkConsistency()}
 * </ul>
 * <p>
 * <ul>
 * <li> Call {@link SBMLDocument#checkConsistency()}, then inquire about the results by
 * calling {@link SBMLDocument#getNumErrors()}
 * </ul>
 * <p>
 * <em>Scenario 2: Reading a model from a file or data stream.</em> After
 * reading the model,
 * <p>
 * <ul>
 * <li> Basic consistency checks will have been performed automatically by
 * libSBML upon reading the content, so you only need to inquire about the
 * results by using {@link SBMLDocument#getNumErrors()}
 * </ul>
 * <p>
 * <ul>
 * <li> Call {@link SBMLDocument#setConsistencyChecks(int categ, boolean
 * onoff)}  to configure
 * which checks are performed by {@link SBMLDocument#checkConsistency()}
 * </ul>
 * <p>
 * <ul>
 * <li> Call {@link SBMLDocument#checkConsistency()}, then inquire about the results
 * by calling {@link SBMLDocument#getNumErrors()}
 * </ul>
 * <p>
 * <p>
 * <p>
 * <h2>Converting documents between Levels and Versions of SBML</h2>
 * <p>
 * LibSBML provides facilities for limited translation of SBML between
 * Levels and Versions of the SBML specifications.  The method for doing is
 * is {@link SBMLDocument#setLevelAndVersion(long lev, long ver, boolean strict)} .  In 
 * general, models can be converted upward without difficulty (e.g., from
 * SBML Level&nbsp;1 to Level&nbsp;2, or from an earlier Version of
 * Level&nbsp;2 to the latest Version of Level&nbsp;2).  Sometimes models
 * can be translated downward as well, if they do not use constructs
 * specific to more advanced Levels of SBML.
 * <p>
 * Calling {@link SBMLDocument#setLevelAndVersion(long lev, long ver, boolean strict)}  will not <em>necessarily</em> lead
 * to a successful conversion.  The method will return a boolean value
 * to indicate success or failure.  Callers must check the error log (see 
 * next section) attached to the {@link SBMLDocument} object after calling
 * {@link SBMLDocument#setLevelAndVersion(long lev, long ver)}  in order to assess whether any
 * problems arose.
 * <p>
 * If an application is interested in translating to a lower Level and/or
 * Version of SBML within a Level, the following methods allow for prior
 * assessment of whether there is sufficient compatibility to make a
 * translation possible:
 * <ul>
 * <li> {@link SBMLDocument#checkL1Compatibility()},
 * <li> {@link SBMLDocument#checkL2v1Compatibility()},
 * <li> {@link SBMLDocument#checkL2v2Compatibility()},
 * <li> {@link SBMLDocument#checkL2v3Compatibility()}, 
 * <li> {@link SBMLDocument#checkL2v4Compatibility()}, and
 * <li> {@link SBMLDocument#checkL3v1Compatibility()}.
 * </ul>
 * <p>
 * Some changes between Versions of SBML Level&nbsp;2 may lead to
 * unexpected behaviors when attempting conversions in either direction.
 * For example, SBML Level&nbsp;2 Version&nbsp;4 relaxed the requirement
 * for consistency in units of measurement between expressions annd
 * quantities in a model.  As a result, a model written in Version&nbsp;4,
 * if converted to Version&nbsp;3 with no other changes, may fail
 * validation as a Version&nbsp;3 model because Version&nbsp;3 imposed
 * stricter requirements on unit consistency.
 * <p>
 * Other changes between SBML Level 2 and Level 3 make downward conversions
 * challenging.  In some cases, it means that a model converted to
 * Level&nbsp;2 from Level&nbsp;3 will contain attributes that were not
 * explicitly given in the Level&nbsp;3 model, because in Level&nbsp;2
 * these attributes may have been optional or have default values.
 * <p>
 * <h2>Error handling</h2>
 * <p>
 * Upon reading a model, {@link SBMLDocument} logs any problems encountered while
 * reading the model from the file or data stream.  The log contains
 * objects that record diagnostic information about any notable issues that
 * arose.  Whether the problems are warnings or errors, they are both
 * reported through a single common interface involving the object class
 * {@link SBMLError}.
 * <p>
 * The methods {@link SBMLDocument#getNumErrors()}, {@link SBMLDocument#getError(long n)}  and
 * {@link SBMLDocument#printErrors()} allow callers to interact with the warnings
 * or errors logged.  Alternatively, callers may retrieve the entire log as
 * an {@link SBMLErrorLog} object using the method {@link SBMLDocument#getErrorLog()}.
 * The {@link SBMLErrorLog} object provides some alternative methods for
 * interacting with the set of errors and warnings.  In either case,
 * applications typically should first call {@link SBMLDocument#getNumErrors()} to
 * find out if any issues have been logged after specific libSBML
 * operations such as the ones discussed in the sections above.  If they
 * have, then an application will should proceed to inspect the individual
 * reports using either the direct interfaces on {@link SBMLDocument} or using the
 * methods on the {@link SBMLErrorLog} object.
 * <p>
 * <p>
 * <p>
 */
"


%javamethodmodifiers SBMLDocument::getDefaultLevel "
/**
   * The default SBML Level of new {@link SBMLDocument} objects.
   * <p>
   * This \'default Level\' corresponds to the most recent SBML specification
   * Level available at the time libSBML version 5.6.0
   * was released.  For this copy of libSBML, the value is <code>3</code>.
   * The default Level is used by {@link SBMLDocument} if no Level is explicitly
   * specified at the time of the construction of an {@link SBMLDocument} instance.
   * <p>
   * @return an integer indicating the most recent SBML specification Level
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link SBMLDocument}), and the
   * other will be a standalone top-level function with the name
   * SBMLDocument_getDefaultLevel(). They are functionally
   * identical. 
   * <p>
   * @see SBMLDocument#getDefaultVersion() 
   */ public
";


%javamethodmodifiers SBMLDocument::getDefaultVersion "
/**
   * The default Version of new {@link SBMLDocument} objects.
   * <p>
   * This \'default Version\' corresponds to the most recent Version
   * within the most recent Level of SBML available at the time libSBML
   * version 5.6.0
   * was released.  For this copy of libSBML, the value is <code>1</code>
   * (where the default Level of SBML is <code>3</code>).  The default
   * Version is used by {@link SBMLDocument} if no Version is explicitly specified
   * at the time of the construction of an {@link SBMLDocument} instance.
   * <p>
   * @return an integer indicating the most recent SBML specification
   * Version
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link SBMLDocument}), and the
   * other will be a standalone top-level function with the name
   * SBMLDocument_getDefaultVersion(). They are functionally
   * identical. 
   * <p>
   * @see SBMLDocument#getDefaultLevel() 
   */ public
";


%javamethodmodifiers SBMLDocument::SBMLDocument(unsigned int level = 0, unsigned int version = 0) "
/**
   * Creates a new {@link SBMLDocument}, optionally with given values for the SBML
   * Level and Version.
   * <p>
   * If <em>both</em> the SBML Level and Version attributes are not
   * specified, the SBML document is treated as having the latest Level and
   * Version of SBML as determined by {@link SBMLDocument#getDefaultLevel()} and
   * {@link SBMLDocument#getDefaultVersion()}; <em>however</em>, the {@link SBMLDocument}
   * object is otherwise left blank.  In particular, the blank {@link SBMLDocument}
   * object has no associated XML attributes, including (but not limited
   * to) an XML namespace declaration.  The XML namespace declaration is
   * not added until the model is written out, <em>or</em> the method
   * {@link SBMLDocument#setLevelAndVersion(long lev, long ver, boolean strict)}
   * is called.  This may be important to keep in mind
   * if an application needs to add additional XML namespace declarations
   * on the <code>&lt;sbml&gt;</code> element.  Application writers should
   * either provide values for <code>level</code> and <code>version</code> on the call to this
   * constructor, or else call
   * {@link SBMLDocument#setLevelAndVersion(long lev, long ver, boolean strict)}
   * shortly after creating the {@link SBMLDocument} object.
   * <p>
   * @param level an integer for the SBML Level
   * <p>
   * @param version an integer for the Version within the SBML Level
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   * <p>
   * @see SBMLDocument#setLevelAndVersion(long lev, long ver, boolean strict)
   * @see #getDefaultLevel()
   * @see #getDefaultVersion()
   */ public
";


%javamethodmodifiers SBMLDocument::SBMLDocument(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link SBMLDocument} using the given {@link SBMLNamespaces} object 
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   */ public
";


%javamethodmodifiers SBMLDocument::SBMLDocument(const SBMLDocument& orig) "
/**
   * Copy constructor; creates a copy of this {@link SBMLDocument}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers SBMLDocument::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link SBMLDocument}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */ public
";


%javamethodmodifiers SBMLDocument::clone() const "
/**
   * Creates and returns a deep copy of this {@link SBMLDocument}.
   * <p>
   * @return a (deep) copy of this {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers SBMLDocument::getModel() const "
/**
   * Returns the {@link Model} object stored in this {@link SBMLDocument}.
   * <p>
   * It is important to note that this method <em>does not create</em> a
   * {@link Model} instance.  The model in the {@link SBMLDocument} must have been created
   * at some prior time, for example using {@link SBMLDocument#createModel()} 
   * or {@link SBMLDocument#setModel(Model m)}.
   * This method returns <code>null</code> if a model does not yet exist.
   * <p>
   * @return the {@link Model} contained in this {@link SBMLDocument}.
   * <p>
   * @see #createModel()
   */ public
";


%javamethodmodifiers SBMLDocument::getModel "
/**
   * Returns the {@link Model} object stored in this {@link SBMLDocument}.
   * <p>
   * It is important to note that this method <em>does not create</em> a
   * {@link Model} instance.  The model in the {@link SBMLDocument} must have been created
   * at some prior time, for example using {@link SBMLDocument#createModel()} 
   * or {@link SBMLDocument#setModel(Model m)}.
   * This method returns <code>null</code> if a model does not yet exist.
   * <p>
   * @return the {@link Model} contained in this {@link SBMLDocument}.
   * <p>
   * @see #createModel()
   */ public
";


%javamethodmodifiers SBMLDocument::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.
   * <p>
   * @param id string representing the id of objects to find
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers SBMLDocument::getElementByMetaId(std::string metaid) "
/**
   * Returns the first child element it can find with the given metaid, or itself if it has the given metaid, or null if no such object is found.
   * <p>
   * @param metaid string representing the metaid of objects to find
   * <p>
   * @return pointer to the first element found with the given metaid.
   */ public
";


%javamethodmodifiers SBMLDocument::getAllElements "
/**
   * Returns a {@link List} of all child {@link SBase} objects, including those nested to an arbitrary depth
   * <p>
   * @return a {@link List} of pointers to all children objects.
   */ public
";


%javamethodmodifiers SBMLDocument::expandFunctionDefinitions "
/**
   * Removes {@link FunctionDefinition} constructs from the document and expands
   * any instances of their use within <code>&lt;math&gt;</code> elements.
   * <p>
   * For example, suppose a {@link Model} contains a {@link FunctionDefinition} with
   * identifier <code>\'f\'</code> representing the math expression: <em>f(x, y) = x *
   * y</em>.  Suppose further that there is a reaction in which the
   * <code>&lt;math&gt;</code> element of the {@link KineticLaw} object contains
   * <code>f(s, p)</code>, where <code>s</code> and <code>p</code> are other identifiers
   * defined in the model.  The outcome of invoking this method is that the
   * <code>&lt;math&gt;</code> of the {@link KineticLaw} now represents the
   * expression <em>s * p</em> and the model no longer contains any
   * {@link FunctionDefinition} objects.
   * <p>
   * @return bool <code>true</code> if the transformation was successful, 
   * <code>false</code>, otherwise.
   * <p>
   * @note This function will check the consistency of a model before
   * attemptimg the transformation.  If the model is not valid SBML, the
   * transformation will not be performed and the function will return 
   * <code>false.</code>
   */ public
";


%javamethodmodifiers SBMLDocument::expandInitialAssignments "
/**
   * Removes {@link InitialAssignment} constructs from the document and
   * replaces them with appropriate values.
   * <p>
   * For example, suppose a {@link Model} contains a {@link InitialAssignment} to a symbol
   * <code>\'k\'</code> where <code>\'k\'</code> is the identifier of a {@link Parameter}.  The outcome of
   * invoking this method is that the \'value\' attribute of the {@link Parameter}
   * definition is set to the result calculated using the {@link InitialAssignment}
   * object\'s <code>&lt;math&gt;</code> formula, and the corresponding
   * {@link InitialAssignment} is then removed from the {@link Model}.
   * <p>
   * @return bool <code>true</code> if the transformation was successful, 
   * <code>false</code>, otherwise.
   * <p>
   * @note This function will check the consistency of a model before
   * attemptimg the transformation.  If the model is not valid SBML, the
   * transformation will not be performed and the function will return 
   * <code>false.</code>  As part of that process, this method will check that it has
   * values for any components referred to by the <code>&lt;math&gt;</code>
   * elements of {@link InitialAssignment} objects.  In cases where not all of the
   * values have been declared (e.g., if the mathematical expression refers
   * to model entities that have no declared values), the {@link InitialAssignment}
   * in question will <em>not</em> be removed and this method will return 
   * <code>false.</code>
   */ public
";


%javamethodmodifiers SBMLDocument::setLevelAndVersion(unsigned int level, unsigned int version, bool strict = true) "
/**
   * Sets the SBML Level and Version of this {@link SBMLDocument} instance,
   * attempting to convert the model as needed.
   * <p>
   * This method is the principal way in libSBML to convert models between
   * Levels and Versions of SBML.  Generally, models can be converted
   * upward without difficulty (e.g., from SBML Level&nbsp;1 to
   * Level&nbsp;2, or from an earlier Version of Level&nbsp;2 to the latest
   * Version of Level&nbsp;2).  Sometimes models can be translated downward
   * as well, if they do not use constructs specific to more advanced
   * Levels of SBML.
   * <p>
   * Before calling this method, callers may check compatibility directly
   * using the methods {@link SBMLDocument#checkL1Compatibility()},
   * {@link SBMLDocument#checkL2v1Compatibility()},
   * {@link SBMLDocument#checkL2v2Compatibility()},
   * {@link SBMLDocument#checkL2v3Compatibility()},
   * {@link SBMLDocument#checkL2v4Compatibility()}, and
   * {@link SBMLDocument#checkL3v1Compatibility()}.
   * <p>
   * The valid combinations of SBML Level and Version as of this release
   * of libSBML are the following: 
   * <ul>
   * <li> Level&nbsp;1 Version&nbsp;2
   * <li> Level&nbsp;2 Version&nbsp;1
   * <li> Level&nbsp;2 Version&nbsp;2
   * <li> Level&nbsp;2 Version&nbsp;3
   * <li> Level&nbsp;2 Version&nbsp;4
   * <li> Level&nbsp;3 Version&nbsp;1
   * </ul>
   * <p>
   * Strict conversion applies the additional criteria that both the
   * source and the target model must be consistent SBML.  Users can
   * control the consistency checks that are applied using the
   * {@link SBMLDocument#setConsistencyChecksForConversion(int categ, boolean onoff)} method.  If either
   * the source or the potential target model have validation errors, the
   * conversion is not performed.  When a strict conversion is successful,
   * the underlying SBML object model is altered to reflect the new level
   * and version.  Thus, information that cannot be converted
   * (e.g. sboTerms) will be lost.
   * <p>
   * @param level the desired SBML Level
   * <p>
   * @param version the desired Version within the SBML Level
   * <p>
   * @param strict boolean indicating whether to check consistency
   * of both the source and target model when performing
   * conversion (defaults to <code> true </code>)
   * <p>
   * @return <code>true</code> if the level and version of the document were
   * successfully set to the requested values (which may have required
   * conversion of the model), <code>false</code> otherwise.
   * <p>
   * @note Calling this method will not <em>necessarily</em> lead to a successful
   * conversion.  If the conversion fails, it will be logged in the error
   * list associated with this {@link SBMLDocument}.  Callers should consult
   * getNumErrors() to find out if the conversion succeeded without
   * problems.  For conversions from Level&nbsp;2 to Level&nbsp;1, callers
   * can also check the Level of the model after calling this method to
   * find out whether it is Level&nbsp;1.  (If the conversion to
   * Level&nbsp;1 failed, the Level of this model will be left unchanged.)
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   * <p>
   * @see #checkL1Compatibility()
   * @see #checkL2v1Compatibility()
   * @see #checkL2v2Compatibility()
   * @see #checkL2v3Compatibility()
   * @see #checkL2v4Compatibility()
   * @see #checkL3v1Compatibility()
   * @see #checkL3v1Compatibility()
   */ public
";


%javamethodmodifiers SBMLDocument::updateSBMLNamespace(const std::string& package, unsigned int level, unsigned int version) "
/**
   * @param package
   * @param level
   * @param version
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::setModel(const Model* m) "
/**
   * Sets the {@link Model} for this {@link SBMLDocument} to a copy of the given {@link Model}.
   * <p>
   * @param m the new {@link Model} to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * </ul>
   * <p>
   * @see #createModel()
   * @see #getModel()
   */ public
";


%javamethodmodifiers SBMLDocument::createModel(const std::string& sid = "") "
/**
   * Creates a new {@link Model} inside this {@link SBMLDocument}, and returns a pointer to
   * it.
   * <p>
   * In SBML Level&nbsp;2, the use of an identifier on a {@link Model} object is
   * optional.  This method takes an optional argument, <code>sid</code>, for setting
   * the identifier.  If not supplied, the identifier attribute on the
   * {@link Model} instance is not set.
   * <p>
   * @param sid the identifier of the new {@link Model} to create.
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   * <p>
   * @see #getModel()
   * @see SBMLDocument#setModel(Model m)
   */ public
";


%javamethodmodifiers SBMLDocument::setLocationURI(const std::string& uri) "
/**
   * Sets the location of this {@link SBMLDocument}.  Called automatically when
   * readSBMLFromFile is used, but may be set manually as well.
   */ public
";


%javamethodmodifiers SBMLDocument::getLocationURI "
/**
   * Get the location of this {@link SBMLDocument}.  If this document was 
   * read from a file or had its location set manually, that filename
   * or set location will be returned, otherwise, an empty string is returned.
   */ public
";


%javamethodmodifiers SBMLDocument::setConsistencyChecks(SBMLErrorCategory_t category, bool apply) "
/**
   * Controls the consistency checks that are performed when
   * {@link SBMLDocument#checkConsistency()} is called.
   * <p>
   * This method works by adding or subtracting consistency checks from the
   * set of all possible checks that {@link SBMLDocument#checkConsistency()} knows
   * how to perform.  This method may need to be called multiple times in
   * order to achieve the desired combination of checks.  The first
   * argument (<code>category</code>) in a call to this method indicates the category
   * of consistency/error checks that are to be turned on or off, and the
   * second argument (<code>apply</code>, a boolean) indicates whether to turn it on
   * (value of <code>true</code>) or off (value of <code>false</code>).
   * <p>
   * * The possible categories (values to the argument <code>category</code>) are the
   * set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
   * in the interface class {@link libsbmlConstants}.
   * The following are the possible choices:
   * <p>
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_GENERAL_CONSISTENCY
   * LIBSBML_CAT_GENERAL_CONSISTENCY}: Correctness and consistency
   * of specific SBML language constructs.  Performing this set of checks
   * is highly recommended.  With respect to the SBML specification, these
   * concern failures in applying the validation rules numbered 2xxxx in
   * the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3 Version&nbsp;1
   * specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_IDENTIFIER_CONSISTENCY
   * LIBSBML_CAT_IDENTIFIER_CONSISTENCY}: Correctness and
   * consistency of identifiers used for model entities.  An example of
   * inconsistency would be using a species identifier in a reaction rate
   * formula without first having declared the species.  With respect to
   * the SBML specification, these concern failures in applying the
   * validation rules numbered 103xx in the Level&nbsp;2 Versions&nbsp;2&ndash;4
   * and Level&nbsp;3 Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_UNITS_CONSISTENCY
   * LIBSBML_CAT_UNITS_CONSISTENCY}: Consistency of measurement
   * units associated with quantities in a model.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 105xx in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3
   * Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_MATHML_CONSISTENCY
   * LIBSBML_CAT_MATHML_CONSISTENCY}: Syntax of MathML constructs.
   * With respect to the SBML specification, these concern failures in
   * applying the validation rules numbered 102xx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_SBO_CONSISTENCY
   * LIBSBML_CAT_SBO_CONSISTENCY}: Consistency and validity of SBO
   * identifiers (if any) used in the model.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 107xx in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3
   * Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_OVERDETERMINED_MODEL
   * LIBSBML_CAT_OVERDETERMINED_MODEL}: Static analysis of whether
   * the system of equations implied by a model is mathematically
   * overdetermined.  With respect to the SBML specification, this is
   * validation rule #10601 in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and
   * Level&nbsp;3 Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_MODELING_PRACTICE
   * LIBSBML_CAT_MODELING_PRACTICE}: Additional checks for
   * recommended good modeling practice. (These are tests performed by
   * libSBML and do not have equivalent SBML validation rules.)
   * </ul>
   * <p>
   * <em>By default, all validation checks are applied</em> to the model in
   * an {@link SBMLDocument} object <em>unless</em>
   * {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}
   * is called to indicate that only a subset should be applied.  Further,
   * this default (i.e., performing all checks) applies separately to
   * <em>each new {@link SBMLDocument} object</em> created.  In other words, each
   * time a model is read using {@link SBMLReader#readSBML(String filename)},
   * {@link SBMLReader#readSBMLFromString(String xml)},
   * or the global functions readSBML() and readSBMLFromString(), a new
   * {@link SBMLDocument} is created and for that document, a call to
   * {@link SBMLDocument#checkConsistency()} will default to applying all possible checks.
   * Calling programs must invoke
   * {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}
   * for each such new model if they wish to change the consistency checks
   * applied.
   * <p>
   * @param category a value drawn from * the set of SBML error categories indicating the
   * consistency checking/validation to be turned on or off.
   * <p>
   * @param apply a boolean indicating whether the checks indicated by
   * <code>category</code> should be applied or not.
   * <p>
   * @see SBMLDocument#checkConsistency()
   */ public
";


%javamethodmodifiers SBMLDocument::setConsistencyChecksForConversion(SBMLErrorCategory_t category, bool apply) "
/**
   * Controls the consistency checks that are performed when
   * {@link SBMLDocument#setLevelAndVersion(long lev, long ver, boolean strict)} is called.
   * <p>
   * This method works by adding or subtracting consistency checks from the
   * set of all possible checks that may be performed to avoid conversion
   * to or from an invalid document.  This method may need to be called 
   * multiple times in
   * order to achieve the desired combination of checks.  The first
   * argument (<code>category</code>) in a call to this method indicates the category
   * of consistency/error checks that are to be turned on or off, and the
   * second argument (<code>apply</code>, a boolean) indicates whether to turn it on
   * (value of <code>true</code>) or off (value of <code>false</code>).
   * <p>
   * * The possible categories (values to the argument <code>category</code>) are the
   * set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
   * in the interface class {@link libsbmlConstants}.
   * The following are the possible choices:
   * <p>
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_GENERAL_CONSISTENCY
   * LIBSBML_CAT_GENERAL_CONSISTENCY}: Correctness and consistency
   * of specific SBML language constructs.  Performing this set of checks
   * is highly recommended.  With respect to the SBML specification, these
   * concern failures in applying the validation rules numbered 2xxxx in
   * the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3 Version&nbsp;1
   * specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_IDENTIFIER_CONSISTENCY
   * LIBSBML_CAT_IDENTIFIER_CONSISTENCY}: Correctness and
   * consistency of identifiers used for model entities.  An example of
   * inconsistency would be using a species identifier in a reaction rate
   * formula without first having declared the species.  With respect to
   * the SBML specification, these concern failures in applying the
   * validation rules numbered 103xx in the Level&nbsp;2 Versions&nbsp;2&ndash;4
   * and Level&nbsp;3 Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_UNITS_CONSISTENCY
   * LIBSBML_CAT_UNITS_CONSISTENCY}: Consistency of measurement
   * units associated with quantities in a model.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 105xx in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3
   * Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_MATHML_CONSISTENCY
   * LIBSBML_CAT_MATHML_CONSISTENCY}: Syntax of MathML constructs.
   * With respect to the SBML specification, these concern failures in
   * applying the validation rules numbered 102xx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_SBO_CONSISTENCY
   * LIBSBML_CAT_SBO_CONSISTENCY}: Consistency and validity of SBO
   * identifiers (if any) used in the model.  With respect to the SBML
   * specification, these concern failures in applying the validation rules
   * numbered 107xx in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3
   * Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_OVERDETERMINED_MODEL
   * LIBSBML_CAT_OVERDETERMINED_MODEL}: Static analysis of whether
   * the system of equations implied by a model is mathematically
   * overdetermined.  With respect to the SBML specification, this is
   * validation rule #10601 in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and
   * Level&nbsp;3 Version&nbsp;1 specifications.
   * <p>
   * <li> {@link  libsbmlConstants#LIBSBML_CAT_MODELING_PRACTICE
   * LIBSBML_CAT_MODELING_PRACTICE}: Additional checks for
   * recommended good modeling practice. (These are tests performed by
   * libSBML and do not have equivalent SBML validation rules.)
   * </ul>
   * <p>
   * <em>By default, all validation checks are applied</em> to the model in
   * an {@link SBMLDocument} object <em>unless</em>
   * {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}
   * is called to indicate that only a subset should be applied.  Further,
   * this default (i.e., performing all checks) applies separately to
   * <em>each new {@link SBMLDocument} object</em> created.  In other words, each
   * time a model is read using {@link SBMLReader#readSBML(String filename)},
   * {@link SBMLReader#readSBMLFromString(String xml)},
   * or the global functions readSBML() and readSBMLFromString(), a new
   * {@link SBMLDocument} is created and for that document, a call to
   * {@link SBMLDocument#checkConsistency()} will default to applying all possible checks.
   * Calling programs must invoke
   * {@link SBMLDocument#setConsistencyChecks(int categ, boolean onoff)}
   * for each such new model if they wish to change the consistency checks
   * applied.
   * <p>
   * @param category a value drawn from * the set of SBML error categories indicating the consistency
   * checking/validation to be turned on or off.
   * <p>
   * @param apply a boolean indicating whether the checks indicated by
   * <code>category</code> should be applied or not.
   * <p>
   * @see SBMLDocument#setLevelAndVersion(long lev, long ver, boolean strict)
   */ public
";


%javamethodmodifiers SBMLDocument::checkConsistency "
/**
   * Performs consistency checking and validation on this SBML document.
   * <p>
   * If this method returns a nonzero value (meaning, one or more
   * consistency checks have failed for SBML document), the failures may be
   * due to warnings <em>or</em> errors.  Callers should inspect the severity
   * flag in the individual {@link SBMLError} objects returned by
   * {@link SBMLDocument#getError(long n)} to determine the nature of the failures.
   * <p>
   * @return the number of failed checks (errors) encountered.
   * <p>
   * @see SBMLDocument#checkInternalConsistency()
   */ public
";


%javamethodmodifiers SBMLDocument::validateSBML "
/**
   * Performs consistency checking and validation on this SBML document.
   * <p>
   * If this method returns a nonzero value (meaning, one or more
   * consistency checks have failed for SBML document), the failures may be
   * due to warnings <em>or</em> errors.  Callers should inspect the severity
   * flag in the individual {@link SBMLError} objects returned by
   * {@link SBMLDocument#getError(long n)} to determine the nature of the failures.
   * <p>
   * @note unlike checkConsistency this method will write the document
   *       in order to determine all errors for the document. This will 
   *       also clear the error log. 
   * <p>
   * @return the number of failed checks (errors) encountered.
   * <p>
   * @see SBMLDocument#checkConsistency()
   */ public
";


%javamethodmodifiers SBMLDocument::checkInternalConsistency "
/**
   * Performs consistency checking on libSBML\'s internal representation of 
   * an SBML {@link Model}.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)}.
   * <p>
   * @return the number of failed checks (errors) encountered.
   * <p>
   * The distinction between this method and
   * {@link SBMLDocument#checkConsistency()} is that this method reports on
   * fundamental syntactic and structural errors that violate the XML
   * Schema for SBML; by contrast, {@link SBMLDocument#checkConsistency()}
   * performs more elaborate model verifications and also validation
   * according to the validation rules written in the appendices of the
   * SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 specification documents.
   * <p>
   * @see SBMLDocument#checkConsistency()
   */ public
";


%javamethodmodifiers SBMLDocument::checkL1Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;1 and can be converted
   * to Level&nbsp;1.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)}.
   * <p>
   * @return the number of failed checks (errors) encountered.
   */ public
";


%javamethodmodifiers SBMLDocument::checkL2v1Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;1 and can
   * be converted to Level&nbsp;2 Version&nbsp;1.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)}.
   * <p>
   * @return the number of failed checks (errors) encountered.
   */ public
";


%javamethodmodifiers SBMLDocument::checkL2v2Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;2 and can
   * be converted to Level&nbsp;2 Version&nbsp;2.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)}.
   * <p>
   * @return the number of failed checks (errors) encountered.
   */ public
";


%javamethodmodifiers SBMLDocument::checkL2v3Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;3 and can
   * be converted to Level&nbsp;2 Version&nbsp;3.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)}.
   * <p>
   * @return the number of failed checks (errors) encountered.
   */ public
";


%javamethodmodifiers SBMLDocument::checkL2v4Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;4 and can
   * be converted to Level&nbsp;2 Version&nbsp;4.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)}.
   * <p>
   * @return the number of failed checks (errors) encountered.
   */ public
";


%javamethodmodifiers SBMLDocument::checkL3v1Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;3 Version&nbsp;1 and can
   * be converted to Level&nbsp;3 Version&nbsp;1.
   * <p>
   * Callers should query the results of the consistency check by calling
   * {@link SBMLDocument#getError(long n)}.
   * <p>
   * @return the number of failed checks (errors) encountered.
   */ public
";


%javamethodmodifiers SBMLDocument::getError "
/**
   * Returns the nth error or warning encountered during parsing,
   * consistency checking, or attempted translation of this model.
   * <p>
   * Callers can use method {@link XMLError#getSeverity()} on the result to assess
   * the severity of the problem.  The possible severity levels range from
   * informational messages to fatal errors.
   * <p>
   * @return the error or warning indexed by integer <code>n</code>, or return 
   * <code>null</code> if <code>n &gt; (getNumErrors() - 1)</code>.
   * <p>
   * @param n the integer index of the error sought.
   * <p>
   * @see SBMLDocument#getNumErrors()
   */ public
";


%javamethodmodifiers SBMLDocument::getNumErrors() const "
/**
   * Returns the number of errors or warnings encountered during parsing,
   * consistency checking, or attempted translation of this model.
   * <p>
   * @return the number of errors or warnings encountered
   * <p>
   * @see SBMLDocument#getError(long n)
   */ public
";


%javamethodmodifiers SBMLDocument::printErrors(std::ostream& stream = std::cerr) const "
/**
   * Prints all the errors or warnings encountered trying to parse,
   * check, or translate this SBML document.
   * <p>
   * It prints the text to the stream given by the optional parameter 
   * <code>stream</code>.  If no parameter is given, it prints the output to the
   * standard error stream.
   * <p>
   * If no errors have occurred, i.e., <code>getNumErrors() == 0</code>, no
   * output will be sent to the stream.
   * <p>
   * The format of the output is:
   * <div class=\'fragment\'><pre>
   N error(s):
     line NNN: (id) message
 </pre></div>
   * <p>
   * @param stream the ostream or ostringstream object indicating where
   * the output should be printed.
   * <p>
   * <!-- Don\'t remove the leading </dl> below. It\'s a hack for javadoc. -->
</dl><dl class=\'docnote\'><dt><b>Documentation note:</b></dt><dd>
The native C++ implementation of this method defines a default argument
value. In the documentation generated for different libSBML language
bindings, you may or may not see corresponding arguments in the method
declarations. For example, in Java and C#, a default argument is handled by
declaring two separate methods, with one of them having the argument and
the other one lacking the argument. However, the libSBML documentation will
be <em>identical</em> for both methods. Consequently, if you are reading
this and do not see an argument even though one is described, please look
for descriptions of other variants of this method near where this one
appears in the documentation.
</dd></dl>
 
   * <p>
   * @see #getNumErrors()
   * @see #getErrorLog()
   * @see SBMLDocument#getError(long n)
   */ public
";


%javamethodmodifiers SBMLDocument::setSBMLDocument(SBMLDocument* d) "
/**
   * No-op; it is provided for consistency with the method available on
   * other libSBML object classes but has no effect on {@link SBMLDocument}.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::connectToChild "
/**
   * Sets this SBML object to child SBML objects (if any).
   * (Creates a child-parent relationship by the parent)
   * <p>
   * Subclasses must override this function if they define
   * one ore more child elements.
   * Basically, this function needs to be called in
   * constructor, copy constructor and assignment operator.
   * <p>
   * @see setSBMLDocument
   * @see enablePackageInternal
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::convert(const ConversionProperties& props) "
/**
   * Converts this document using the converter that best matches
   * the given conversion properties. 
   * <p>
   * @param props the conversion properties to use
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED  }
   * <li> {@link  libsbmlConstants#LIBSBML_CONV_CONVERSION_NOT_AVAILABLE LIBSBML_CONV_CONVERSION_NOT_AVAILABLE  }
   * </ul>
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::enablePackageInternal(const std::string& pkgURI,const std::string& pkgPrefix, bool flag) "
/**
   * Enables/Disables the given package with this element and child
   * elements (if any).
   * (This is an internal implementation for enablePackage function)
   * <p>
   * @note Subclasses of the SBML Core package in which one or more child
   * elements are defined must override this function.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see SBMLDocument#getElementName()
   */ public
";


%javamethodmodifiers SBMLDocument::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link SBMLDocument},
   * is always <code>\'sbml\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'sbml\'.</code>
   */ public
";


%javamethodmodifiers SBMLDocument::getErrorLog "
/**
   * Returns the list of errors or warnings logged during parsing, 
   * consistency checking, or attempted translation of this model.
   * <p>
   * @return the {@link SBMLErrorLog} used for this {@link SBMLDocument}
   * <p>
   * @see SBMLDocument#getNumErrors()
   */ public
";


%javamethodmodifiers SBMLDocument::getErrorLog() const "
/**
   * Returns a constant pointer to the list of errors or warnings 
   * logged during parsing, consistency checking, or attempted translation 
   * of this model.
   * <p>
   * @return the {@link SBMLErrorLog} used for this {@link SBMLDocument}
   * <p>
   * @see SBMLDocument#getNumErrors()
   */ public
";


%javamethodmodifiers SBMLDocument::getNamespaces() const "
/**
   * Returns a list of XML Namespaces associated with the XML content
   * of this SBML document.
   * <p>
   * @return the XML Namespaces associated with this SBML object
   */ public
";


%javamethodmodifiers SBMLDocument::enableDefaultNS(const std::string& package, bool flag) "
/**
   * Set/unset default namespace to each top-level element defined in the
   * given package extension.
   * <p>
   * This works by adding a <code>xmlns=&quot;...&quot;</code> attribute.  No
   * prefix will be written when writing elements defined in the given
   * package extension if <code>true</code> is given as second argument.
   * <p>
   * @param package the name or URI of the package extension.
   * @param flag boolean value to indicate whether to write a namespace
   * prefix.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION }
   * </ul>
   */ public
";


%javamethodmodifiers SBMLDocument::isEnabledDefaultNS(const std::string& package) "
/**
   * Returns <code>true</code> if a default namespace is added to each top-level
   * element defined in the given package extension, otherwise returns
   * <code>false.</code>
   * <p>
   * This basically checks if the attribute
   * <code>xmlns=&quot;...&quot;</code> is present.
   * <p>
   * @param package the name or URI of the package extension.
   * <p>
   * @return a boolean 
   */ public
";


%javamethodmodifiers SBMLDocument::setPackageRequired(const std::string& package, bool flag) "
/**
   * Sets the <code>required</code> attribute value of the given package
   * extension.
   * <p>
   * @note The name of package must not be given if the package is not
   * enabled.
   * <p>
   * @param package the name or URI of the package extension.
   * @param flag Boolean value indicating whether the package is required.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION }
   * </ul>
   */ public
";


%javamethodmodifiers SBMLDocument::getPackageRequired(const std::string& package) "
/**
   * Returns the <code>required</code> attribute of the given package
   * extension.
   * <p>
   * @note The name of package must not be given if the package is not
   * enabled.
   * <p>
   * @param package the name or URI of the package extension.
   * <p>
   * @return Boolean flag indicating whether the package is flagged as
   * being required.
   */ public
";


%javamethodmodifiers SBMLDocument::isSetPackageRequired(const std::string& package) "
/**
   * Returns <code>true</code> if the required attribute of the given package extension
   * is defined, otherwise returns <code>false.</code>
   * <p>
   * @note The name of package must not be given if the package is not
   * enabled.
   * <p>
   * @param package the name or URI of the package extension.
   * <p>
   * @return a Boolean
   */ public
";


%javamethodmodifiers SBMLDocument::isIgnoredPackage(const std::string& pkgURI) "
/**
   * Returns <code>true</code> if the given package extension is one of an ignored
   * packages, otherwise returns <code>false.</code>
   * <p>
   * An ignored package is one that is defined to be used in this SBML
   * document, but the package is not enabled in this copy of libSBML.
   * <p>
   * @param pkgURI the URI of the package extension.
   * <p>
   * @return a Boolean, <code>true</code> if the package is being ignored and
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers SBMLDocument::setPkgRequired(const std::string& package, bool flag) "
/**
   * Sets the value of the <code>required</code> attribute for the given
   * package.
   * <p>
   * @note The name of package must not be given if the package is not
   * enabled.
   * <p>
   * @param package the name or URI of the package extension.
   * @param flag a Boolean value.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION }
   * </ul>
   * <p>
   * @deprecated <div class=\'deprecated\'>Replaced in libSBML 5.2.0 by
   * setPackageRequired(String package, boolean flag)
   </div>
   * */ public
";


%javamethodmodifiers SBMLDocument::getPkgRequired(const std::string& package) "
/**
   * Returns the <code>required</code> attribute of the given package
   * extension.
   * <p>
   * @note The name of package must not be given if the package is not
   * enabled.
   * <p>
   * @param package the name or URI of the package extension.
   * <p>
   * @return a Boolean value indicating whether the package is flagged as
   * being required in this SBML document.
   * <p>
   * @deprecated <div class=\'deprecated\'>Replaced in libSBML 5.2.0 by
   * getPackageRequired(String package flag)
   </div>
   * */ public
";


%javamethodmodifiers SBMLDocument::isSetPkgRequired(const std::string& package) "
/**
   * Returns <code>true</code> if the required attribute of the given package extension
   * is defined, otherwise returns <code>false.</code>
   * <p>
   * @note The name of package must not be given if the package is not
   * enabled.
   * <p>
   * @param package the name or URI of the package extension.
   * <p>
   * @return a Boolean value.
   * <p>
   * @deprecated <div class=\'deprecated\'>Replaced in libSBML 5.2.0 by
   * isSetPackageRequired(String package flag)
   </div>
   * */ public
";


%javamethodmodifiers SBMLDocument::isIgnoredPkg(const std::string& pkgURI) "
/**
   * Returns <code>true</code> if the given package extension is one of ignored
   * packages, otherwise returns <code>false.</code>
   * <p>
   * An ignored package is one that is defined to be used in this SBML
   * document, but the package is not enabled in this copy of libSBML.
   * <p>
   * @param pkgURI the URI of the package extension.
   * <p>
   * @return a boolean
   * <p>
   * @deprecated <div class=\'deprecated\'>Replaced in libSBML 5.2.0 by
   * isIgnoredPackage(String pkgURI flag)
   </div>
   * */ public
";


%javamethodmodifiers SBMLDocument::getElementPosition() const "
/**
   * @return the ordinal position of the element with respect to its
   * siblings or -1 (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::getApplicableValidators() const "
/**
   * Validation system.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::getConversionValidators() const "
/**
   * Validation system.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::setApplicableValidators(unsigned char appl) "
/**
   * Validation system.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::setConversionValidators(unsigned char appl) "
/**
   * Validation system.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::getNumValidators() const "
/**
   * Validation system.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::clearValidators "
/**
   * Validation system.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::addValidator(const SBMLValidator* validator) "
/**
   * Validation system.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::getValidator(unsigned int index) "
/**
   * Validation system.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers SBMLDocument::writeXMLNS(XMLOutputStream& stream) const "
/**
   * <p>
   * Subclasses should override this method to write their xmlns attriubutes
   * (if any) to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @internal
   */ public
";


%typemap(javaimports) FunctionDefinition "
/** 
 * LibSBML implementation of SBML\'s FunctionDefinition construct.
 * <p>
 * The {@link FunctionDefinition} structure associates an identifier with a
 * function definition.  This identifier can then be used as the function
 * called in subsequent MathML content elsewhere in an SBML model.
 * <p>
 * {@link FunctionDefinition} has one required attribute, \'id\', to give the
 * function a unique identifier by which other parts of an SBML model
 * definition can refer to it.  A {@link FunctionDefinition} instance can also have
 * an optional \'name\' attribute of type <code>string.</code>  Identifiers and names
 * must be used according to the guidelines described in the SBML
 * specification (e.g., Section 3.3 in the Level 2 Version 4
 * specification).
 * <p>
 * {@link FunctionDefinition} has a required \'math\' subelement containing a MathML
 * expression defining the function body.  The content of this element can
 * only be a MathML \'lambda\' element.  The \'lambda\' element must begin with
 * zero or more \'bvar\' elements, followed by any other of the elements in
 * the MathML subset allowed in SBML Level 2 <em>except</em> \'lambda\' (i.e., a
 * \'lambda\' element cannot contain another \'lambda\' element).  This is the
 * only place in SBML where a \'lambda\' element can be used.  The function
 * defined by a {@link FunctionDefinition} is only available for use in other
 * MathML elements that <em>follow</em> the {@link FunctionDefinition} definition in the
 * model.  (These restrictions prevent recursive and mutually-recursive
 * functions from being expressed.)
 * <p>
 * A further restriction on the content of \'math\' is that it cannot contain
 * references to variables other than the variables declared to the
 * \'lambda\' itself.  That is, the contents of MathML \'ci\' elements inside
 * the body of the \'lambda\' can only be the variables declared by its
 * \'bvar\' elements, or the identifiers of other {@link FunctionDefinition}
 * instances in the model.  This means must be written so that all
 * variables or parameters used in the MathML content are passed to them
 * via their function parameters.  In SBML Level&nbsp;2, this restriction
 * applies also to the MathML <code>csymbol</code> elements for <em>time</em> and 
 * <em>delay</em>; in SBML Level&nbsp;3, it additionally applies to the <code>csymbol</code>
 * element for <em>avogadro</em>.
 * <p>
 * @note Function definitions (also informally known as user-defined
 * functions) were introduced in SBML Level 2.  They have purposefully
 * limited capabilities.  A function cannot reference parameters or other
 * model quantities outside of itself; values must be passed as parameters
 * to the function.  Moreover, recursive and mutually-recursive functions
 * are not permitted.  The purpose of these limitations is to balance power
 * against complexity of implementation.  With the restrictions as they
 * are, function definitions could be implemented as textual
 * substitutions&mdash;they are simply macros.  Software implementations
 * therefore do not need the full function-definition machinery typically
 * associated with programming languages.
 * <br><br>
 * Another important point to note is {@link FunctionDefinition} does not
 * have a separate attribute for defining the units of the value returned
 * by the function.  The units associated with the function\'s return value,
 * when the function is called from within MathML expressions elsewhere in
 * SBML, are simply the overall units of the expression in
 * {@link FunctionDefinition}\'s \'math\' subelement when applied to the arguments
 * supplied in the call to the function.  Ascertaining these units requires
 * performing dimensional analysis on the expression.  (Readers may wonder
 * why there is no attribute.  The reason is that having a separate
 * attribute for declaring the units would not only be redundant, but also
 * lead to the potential for having conflicting information.  In the case
 * of a conflict between the declared units and those of the value actually
 * returned by the function, the only logical resolution rule would be to
 * assume that the correct units are those of the expression anyway.)
 * <p>
 * <!---------------------------------------------------------------------- -->
 * <p>
 */
"


%typemap(javaimports) ListOfFunctionDefinitions "
/** 
 * LibSBML implementation of SBML\'s ListOfFunctionDefinitions construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers FunctionDefinition::FunctionDefinition(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link FunctionDefinition} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link FunctionDefinition}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link FunctionDefinition}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link FunctionDefinition} object to an
   * {@link SBMLDocument} (e.g., using {@link Model#addFunctionDefinition(FunctionDefinition f)}), the SBML
   * Level, SBML Version and XML namespace of the document 
   * <em>override</em> the values used when creating the {@link FunctionDefinition} object
   * via this constructor.  This is necessary to ensure that an SBML
   * document is a consistent structure.  Nevertheless, the ability to
   * supply the values at the time of creation of a {@link FunctionDefinition} is
   * an important aid to producing valid SBML.  Knowledge of the intented
   * SBML Level and Version determine whether it is valid to assign a
   * particular value to an attribute, or whether it is valid to add an
   * object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers FunctionDefinition::FunctionDefinition(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link FunctionDefinition} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link FunctionDefinition} object to an
   * {@link SBMLDocument} (e.g., using {@link Model#addFunctionDefinition(FunctionDefinition f)}), the SBML
   * XML namespace of the document <em>overrides</em> the value used when
   * creating the {@link FunctionDefinition} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link FunctionDefinition} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers FunctionDefinition::FunctionDefinition(const FunctionDefinition& orig) "
/**
   * Copy constructor; creates a copy of this {@link FunctionDefinition}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers FunctionDefinition::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link FunctionDefinition}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link FunctionDefinition} in
   * the list of function definitions.
   */ public
";


%javamethodmodifiers FunctionDefinition::clone() const "
/**
   * Creates and returns a deep copy of this {@link FunctionDefinition}.
   * <p>
   * @return a (deep) copy of this {@link FunctionDefinition}.
   */ public
";


%javamethodmodifiers FunctionDefinition::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link FunctionDefinition}.
   * <p>
   * @return the id of this {@link FunctionDefinition}.
   */ public
";


%javamethodmodifiers FunctionDefinition::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link FunctionDefinition}.
   * <p>
   * @return the name of this {@link FunctionDefinition}.
   */ public
";


%javamethodmodifiers FunctionDefinition::getMath() const "
/**
   * Get the mathematical formula of this {@link FunctionDefinition}.
   * <p>
   * @return an {@link ASTNode}, the value of the \'math\' subelement of this
   * {@link FunctionDefinition}
   */ public
";


%javamethodmodifiers FunctionDefinition::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link FunctionDefinition}\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link FunctionDefinition} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers FunctionDefinition::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link FunctionDefinition}\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link FunctionDefinition} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers FunctionDefinition::isSetMath() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link FunctionDefinition}\'s \'math\' subelement contains a value.
   * <p>
   * @return <code>true</code> if the \'math\' for this {@link FunctionDefinition} is set,
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers FunctionDefinition::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link FunctionDefinition}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link FunctionDefinition}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers FunctionDefinition::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link FunctionDefinition}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link FunctionDefinition}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers FunctionDefinition::setMath(const ASTNode* math) "
/**
   * Sets the \'math\' subelement of this {@link FunctionDefinition} to the Abstract
   * Syntax Tree given in <code>math</code>.
   * <p>
   * @param math an AST containing the mathematical expression to
   * be used as the formula for this {@link FunctionDefinition}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   */ public
";


%javamethodmodifiers FunctionDefinition::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link FunctionDefinition}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers FunctionDefinition::getArgument "
/**
   * Get the <code>n</code>th argument to this function.
   * <p>
   * Callers should first find out the number of arguments to the function
   * by calling getNumArguments().
   * <p>
   * @param n an integer index for the argument sought.
   * <p>
   * @return the nth argument (bound variable) passed to this
   * {@link FunctionDefinition}.
   * <p>
   * @see #getNumArguments()
   */ public
";


%javamethodmodifiers FunctionDefinition::getArgument(const std::string& name) const "
/**
   * Get the argument named <code>name</code> to this {@link FunctionDefinition}.
   * <p>
   * @param name the exact name (case-sensitive) of the sought-after
   * argument
   * <p>
   * @return the argument (bound variable) having the given name, or <code>null</code> if
   * no such argument exists.
   */ public
";


%javamethodmodifiers FunctionDefinition::getBody() const "
/**
   * Get the mathematical expression that is the body of this
   * {@link FunctionDefinition} object.
   * <p>
   * @return the body of this {@link FunctionDefinition} as an Abstract Syntax
   * Tree, or <code>null</code> if no body is defined.
   */ public
";


%javamethodmodifiers FunctionDefinition::getBody "
/**
   * Get the mathematical expression that is the body of this
   * {@link FunctionDefinition} object.
   * <p>
   * @return the body of this {@link FunctionDefinition} as an Abstract Syntax
   * Tree, or <code>null</code> if no body is defined.
   */ public
";


%javamethodmodifiers FunctionDefinition::isSetBody() const "
/**
   * Predicate returning <code>true</code> if the body of this
   * {@link FunctionDefinition} has set.
   * <p>
   * @return <code>true</code> if the body of this {@link FunctionDefinition} is 
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers FunctionDefinition::getNumArguments() const "
/**
   * Get the number of arguments (bound variables) taken by this
   * {@link FunctionDefinition}.
   * <p>
   * @return the number of arguments (bound variables) that must be passed
   * to this {@link FunctionDefinition}.
   */ public
";


%javamethodmodifiers FunctionDefinition::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers FunctionDefinition::getElementName() const "
/**
   * Returns the XML element name of this object, which for
   * {@link FunctionDefinition}, is always <code>\'functionDefinition\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'functionDefinition\'.</code>
   */ public
";


%javamethodmodifiers FunctionDefinition::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers FunctionDefinition::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link FunctionDefinition} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link FunctionDefinition} object are:
   * <ul>
   * <li> \'id\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers FunctionDefinition::hasRequiredElements() const "
/**
   * Predicate returning <code>true</code> if
   * all the required elements for this {@link FunctionDefinition} object
   * have been set.
   * <p>
   * @note The required elements for a {@link FunctionDefinition} object are:
   * <ul>
   * <li> \'math\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers FunctionDefinition::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers FunctionDefinition::readOtherXML(XMLInputStream& stream) "
/**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   * @internal
   */ public
";


%javamethodmodifiers FunctionDefinition::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers FunctionDefinition::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers FunctionDefinition::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers FunctionDefinition::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::ListOfFunctionDefinitions(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfFunctionDefinitions} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::ListOfFunctionDefinitions(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfFunctionDefinitions} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfFunctionDefinitions} object to be created.
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfFunctionDefinitions} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfFunctionDefinitions}.
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link FunctionDefinition} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfFunctionDefinitions}, the XML element name is 
   * <code>\'listOfFunctionDefinitions\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfFunctionDefinitions\'.</code>
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::get(unsigned int n) "
/**
   * Get a {@link FunctionDefinition} from the {@link ListOfFunctionDefinitions}.
   * <p>
   * @param n the index number of the {@link FunctionDefinition} to get.
   * <p>
   * @return the nth {@link FunctionDefinition} in this {@link ListOfFunctionDefinitions}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::get "
/**
   * Get a {@link FunctionDefinition} from the {@link ListOfFunctionDefinitions}.
   * <p>
   * @param n the index number of the {@link FunctionDefinition} to get.
   * <p>
   * @return the nth {@link FunctionDefinition} in this {@link ListOfFunctionDefinitions}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::get(const std::string& sid) "
/**
   * Get a {@link FunctionDefinition} from the {@link ListOfFunctionDefinitions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link FunctionDefinition} to get.
   * <p>
   * @return {@link FunctionDefinition} in this {@link ListOfFunctionDefinitions}
   * with the given id or <code>null</code> if no such
   * {@link FunctionDefinition} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::get(const std::string& sid) const "
/**
   * Get a {@link FunctionDefinition} from the {@link ListOfFunctionDefinitions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link FunctionDefinition} to get.
   * <p>
   * @return {@link FunctionDefinition} in this {@link ListOfFunctionDefinitions}
   * with the given id or <code>null</code> if no such
   * {@link FunctionDefinition} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfFunctionDefinitions} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::remove(const std::string& sid) "
/**
   * Removes item in this {@link ListOfFunctionDefinitions} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the
   * {@link ListOfFunctionDefinitions} in a model is (in SBML Level 2 Version 4)
   * the first ListOf___.  (However, it differs for different Levels and
   * Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfFunctionDefinitions::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%javamethodmodifiers UnitKind_equals(UnitKind_t uk1, UnitKind_t uk2) "
/**
 * Tests for logical equality between two given <code>UNIT_KIND_</code>
 * code values.
 * <p>
 * This function behaves exactly like C\'s <code>==</code> operator, except
 * for the following two cases:
 * <ul>
 * <li>{@link  libsbmlConstants#UNIT_KIND_LITER UNIT_KIND_LITER} <code>==</code> {@link  libsbmlConstants#UNIT_KIND_LITRE UNIT_KIND_LITRE}
 * <li>{@link  libsbmlConstants#UNIT_KIND_METER UNIT_KIND_METER} <code>==</code> {@link  libsbmlConstants#UNIT_KIND_METRE UNIT_KIND_METRE}
 * </ul>
 * <p>
 * In the two cases above, C equality comparison would yield <code>false</code>
 * (because each of the above is a distinct enumeration value), but
 * this function returns <code>true.</code>
 * <p>
 * @param uk1 a <code>UNIT_KIND_</code> value 
 * @param uk2 a second <code>UNIT_KIND_</code> value to compare to <code>uk1</code>
 * <p>
 * @return nonzero (for <code>true</code>) if <code>uk1</code> is logically equivalent to 
 * <code>uk2</code>, zero (for <code>false</code>) otherwise.
 * <p>
 * @note For more information about the libSBML unit codes, please refer to
 * the class documentation for {@link Unit}.
 */ public
";


%javamethodmodifiers UnitKind_forName(const char *name) "
/**
 * Converts a text string naming a kind of unit to its corresponding
 * libSBML <code>UNIT_KIND_</code> constant/enumeration value.
 * <p>
 * @param name a string, the name of a predefined base unit in SBML
 * <p>
 * @return a value the set of <code>UNIT_KIND_</code> codes
 * defined in class {@link libsbmlConstants}, corresponding to the string
 * <code>name</code> (determined in a case-insensitive manner).
 * <p>
 * @note For more information about the libSBML unit codes, please refer to
 * the class documentation for {@link Unit}.
 */ public
";


%javamethodmodifiers UnitKind_toString(UnitKind_t uk) "
/**
 * Converts a unit code to a text string equivalent.
 * <p>
 * @param uk a value from the set of <code>UNIT_KIND_</code> codes
 * defined in the class {@link libsbmlConstants}
 * <p>
 * <p>
 * @return the name corresponding to the given unit code.
 * <p>
 * @note For more information about the libSBML unit codes, please refer to
 * the class documentation for {@link Unit}.
 * <p>
 * @warning The string returned is a static data value.  The caller does not
 * own the returned string and is therefore not allowed to modify it.
 */ public
";


%javamethodmodifiers UnitKind_isValidUnitKindString(const char *str, unsigned int level, unsigned int version) "
/**
 * Predicate for testing whether a given string corresponds to a
 * predefined libSBML unit code.
 * <p>
 * @param str a text string naming a base unit defined by SBML
 * @param level the Level of SBML
 * @param version the Version within the Level of SBML
 * <p>
 * @return nonzero (for <code>true</code>) if string is the name of a valid
 * <code>UNIT_KIND_</code> value, zero (for <code>false</code>) otherwise.
 * <p>
 * @note For more information about the libSBML unit codes, please refer to
 * the class documentation for {@link Unit}.
 */ public
";


%typemap(javaimports) Unit "
/** 
 * LibSBML implementation of SBML\'s Unit construct.
 * <p>
 * The SBML unit definition facility uses two classes of objects,
 * {@link UnitDefinition} and {@link Unit}.  The approach to defining units in SBML is
 * compositional; for example, <em>meter second<sup> &ndash;2</sup></em> is
 * constructed by combining a {@link Unit} object representing <em>meter</em> with
 * another {@link Unit} object representing <em>second<sup> &ndash;2</sup></em>.
 * The combination is wrapped inside a {@link UnitDefinition}, which provides for
 * assigning an identifier and optional name to the combination.  The
 * identifier can then be referenced from elsewhere in a model.  Thus, the
 * {@link UnitDefinition} class is the container, and {@link Unit} instances are placed
 * inside {@link UnitDefinition} instances.
 * <p>
 * A {@link Unit} structure has four attributes named \'kind\', \'exponent\', \'scale\'
 * and \'multiplier\'.  It represents a (possibly transformed) reference to a
 * base unit.  The attribute \'kind\' on {@link Unit} indicates the chosen base unit.
 * Its value must be one of the text strings listed below; this list
 * corresponds to SBML Level&nbsp;3 Version&nbsp;1 Core:
 * <p>
 * <center>
<table border=\'0\' class=\'text-table width80 normal-font\'
       style=\'border-top: none !important; border-bottom: none !important; color: black; font-weight: bold; font-family: Courier New, Courier, monospace, fixed;\'>
<tr>
<td>ampere</td><td>farad</td><td>joule</td><td>lux</td><td>radian</td><td>volt</td>
</tr>
<tr>
<td>avogadro</td><td>gram</td><td>katal</td><td>metre</td><td>second</td><td>watt</td>
</tr>
<tr>
<td>becquerel</td><td>gray</td><td>kelvin</td><td>mole</td><td>siemens</td><td>weber</td>
</tr>
<tr>
<td>candela</td><td>henry</td><td>kilogram</td><td>newton</td><td>sievert</td>
</tr>
<tr>
<td>coulomb</td><td>hertz</td><td>litre</td><td>ohm</td><td>steradian</td>
</tr>
<tr>
<td>dimensionless</td><td>item</td><td>lumen</td><td>pascal</td><td>tesla</td>
</tr>
</table>
</center>

 * <p>
 * A few small differences exist between the Level&nbsp;3 list of base
 * units and the list defined in other Level/Version combinations of SBML.
 * Specifically, Levels of SBML before Level&nbsp;3 do not define 
 * <code>avogadro</code>; conversely, Level&nbsp;2 Version&nbsp;1 defines <code>Celsius</code>,
 * and Level&nbsp;1 defines <code>celsius</code>, <code>meter</code>, and <code>liter</code>, none of
 * which are available in Level&nbsp;3.  In libSBML, each of the predefined
 * base unit names is represented by an enumeration value whose name begins with the characters
 * <code>UNIT_KIND_</code>, discussed in a separate section below.
 * <p>
 * The attribute named \'exponent\' on {@link Unit} represents an exponent on the
 * unit.  In SBML Level&nbsp;2, the attribute is optional and has a default
 * value of <code>1</code> (one); in SBML Level&nbsp;3, the attribute is mandatory
 * and there is no default value.  A {@link Unit} structure also has an attribute
 * called \'scale\'; its value must be an integer exponent for a power-of-ten
 * multiplier used to set the scale of the unit.  For example, a unit
 * having a \'kind\' value of <code>gram</code> and a \'scale\' value of <code>-3</code> signifies
 * 10<sup>&nbsp;&ndash;3</sup> &#215; gram, or milligrams.  In SBML
 * Level&nbsp;2, the attribute is optional and has a default value of <code>0</code>
 * (zero), because 10<sup> 0</sup> = 1; in SBML Level&nbsp;3, the attribute
 * is mandatory and has no default value.  Lastly, the attribute named
 * \'multiplier\' can be used to multiply the unit by a real-numbered factor;
 * this enables the definition of units that are not power-of-ten multiples
 * of SI units.  For instance, a multiplier of 0.3048 could be used to
 * define <code>foot</code> as a measure of length in terms of a <code>metre.</code>  The
 * \'multiplier\' attribute is optional in SBML Level&nbsp;2, where it has a
 * default value of <code>1</code> (one); in SBML Level&nbsp;3, the attribute is
 * mandatory and has not default value.
 * <p>
 * * <h3><a class=\'anchor\' name=\'UnitKind_t\'>%Unit identification codes</a></h3>
 * <p>
 * <p>
 * As discussed above, SBML defines a set of base units which serves as the
 * starting point for new unit definitions.  This set of base units
 * consists of the SI units and a small number of additional convenience
 * units.
 * <p>
 * In SBML Level&nbsp;2 Versions before
 * Version&nbsp;3, there existed an enumeration of units called 
 * <code>UnitKind.</code>  In Version&nbsp;3, this enumeration was removed and the
 * identifier class <code>UnitSId</code> redefined to include the previous 
 * <code>UnitKind</code> values as reserved symbols.  This change has no net effect on
 * permissible models, their representation or their syntax.  The purpose
 * of the change in the SBML specification was simply to clean up an
 * inconsistency about the contexts in which these values were usable.
 * However, libSBML maintains UnitKind in the form of of a set of static
 * integer constants whose names begin with the characters
 * <code>UNIT_KIND_</code>.  These constants are defined in the class
 * <code><a href=\'libsbmlConstants.html\'>libsbmlConstants</a></code>.
 * <p>
 * <p>
 * As a consequence of the fact that libSBML supports models in all Levels
 * and Versions of SBML, libSBML\'s set of <code>UNIT_KIND_</code> values is a union
 * of all the possible base unit names defined in the different SBML
 * specifications.  However, not every base unit is allowed in every
 * Level+Version combination of SBML.  Note in particular the following
 * exceptions:
 * <ul>
 * <li> The alternate spelling <code>\'meter\'</code> is included in
 * addition to the official SI spelling <code>\'metre\'.</code>  This spelling is only
 * permitted in SBML Level&nbsp;1 models.
 * <p>
 * <li> The alternate spelling <code>\'liter\'</code> is included in addition to the
 * official SI spelling <code>\'litre\'.</code>  This spelling is only permitted in
 * SBML Level&nbsp;1 models.
 * <p>
 * <li> The unit <code>\'Celsius\'</code> is included because of its presence in
 * specifications of SBML prior to SBML Level&nbsp;2 Version&nbsp;3.
 * <p>
 * <li> The unit <code>avogadro</code> was introduced in SBML Level&nbsp;3, and
 * is only permitted for use in SBML Level&nbsp;3 models.
 * </ul>
 * <p>
 * The table below lists the unit
 * constants defined in libSBML, and their meanings. 
 * <p>
 * <center>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
 <tr>
     <th align=\'left\' width=\'200\'>Enumerator</th>
     <th align=\'left\'>Meaning</th>
 </tr>
<tr><td><code>UNIT_KIND_AMPERE</code></td><td>The ampere unit.</td></tr>
<tr><td><code>UNIT_KIND_AVOGADRO</code></td><td>The unit
<code>dimensionless</code> multiplied by the numerical value of Avogadro\'s
constant. (<span class=\'warning\'>Only usable in SBML Level&nbsp;3 models.</span>)</td></tr>
<tr><td><code>UNIT_KIND_BECQUEREL</code></td><td>The becquerel unit.</td></tr>
<tr><td><code>UNIT_KIND_CANDELA</code></td><td>The candela unit.</td></tr>
<tr><td><code>UNIT_KIND_CELSIUS</code></td><td>The Celsius unit. (<span
class=\'warning\'>Only usable in SBML Level&nbsp;1 and SBML Level&nbsp;2
Version&nbsp;1 models.</span>)</td></tr>
<tr><td><code>UNIT_KIND_COULOMB</code></td><td>The coulomb unit.</td></tr>
<tr><td><code>UNIT_KIND_DIMENSIONLESS</code></td><td>A pseudo-unit
indicating a dimensionless quantity.</td></tr>
<tr><td><code>UNIT_KIND_FARAD</code></td><td>The farad unit.</td></tr>
<tr><td><code>UNIT_KIND_GRAM</code></td><td>The gram unit.</td></tr>
<tr><td><code>UNIT_KIND_GRAY</code></td><td>The gray unit.</td></tr>
<tr><td><code>UNIT_KIND_HENRY</code></td><td>The henry unit.</td></tr>
<tr><td><code>UNIT_KIND_HERTZ</code></td><td>The hertz unit.</td></tr>
<tr><td><code>UNIT_KIND_ITEM</code></td><td>A pseudo-unit representing a
single \'thing\'.</td></tr>
<tr><td><code>UNIT_KIND_JOULE</code></td><td>The joule unit.</td></tr>
<tr><td><code>UNIT_KIND_KATAL</code></td><td>The katal unit.</td></tr>
<tr><td><code>UNIT_KIND_KELVIN</code></td><td>The kelvin unit.</td></tr>
<tr><td><code>UNIT_KIND_KILOGRAM</code></td><td>The kilogram unit.</td></tr>
<tr><td><code>UNIT_KIND_LITER</code></td><td>Alternate spelling of litre.</td></tr>
<tr><td><code>UNIT_KIND_LITRE</code></td><td>The litre unit.</td></tr>
<tr><td><code>UNIT_KIND_LUMEN</code></td><td>The lumen unit.</td></tr>
<tr><td><code>UNIT_KIND_LUX</code></td><td>The lux unit.</td></tr>
<tr><td><code>UNIT_KIND_METER</code></td><td>Alternate spelling of metre.</td></tr>
<tr><td><code>UNIT_KIND_METRE</code></td><td>The metre unit.</td></tr>
<tr><td><code>UNIT_KIND_MOLE</code></td><td>The mole unit.</td></tr>
<tr><td><code>UNIT_KIND_NEWTON</code></td><td>The newton unit.</td></tr>
<tr><td><code>UNIT_KIND_OHM</code></td><td>The ohm unit.</td></tr>
<tr><td><code>UNIT_KIND_PASCAL</code></td><td>The pascal unit.</td></tr>
<tr><td><code>UNIT_KIND_RADIAN</code></td><td>The radian unit.</td></tr>
<tr><td><code>UNIT_KIND_SECOND</code></td><td>The second unit.</td></tr>
<tr><td><code>UNIT_KIND_SIEMENS</code></td><td>The siemens unit.</td></tr>
<tr><td><code>UNIT_KIND_SIEVERT</code></td><td>The sievert unit.</td></tr>
<tr><td><code>UNIT_KIND_STERADIAN</code></td><td>The steradian unit.</td></tr>
<tr><td><code>UNIT_KIND_TESLA</code></td><td>The tesla unit.</td></tr>
<tr><td><code>UNIT_KIND_VOLT</code></td><td>The volt unit.</td></tr>
<tr><td><code>UNIT_KIND_WATT</code></td><td>The watt unit.</td></tr>
<tr><td><code>UNIT_KIND_WEBER</code></td><td>The weber unit.</td></tr>
<tr><td><code>UNIT_KIND_INVALID</code></td><td>Marker used by libSBML
to indicate an invalid or unset unit.</td></tr>
</table>
</center>

 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%typemap(javaimports) ListOfUnits "
/** 
 * LibSBML implementation of SBML\'s ListOfUnits construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * {@link ListOfUnits} is entirely contained within {@link UnitDefinition}.
 */
"


%javamethodmodifiers Unit::Unit(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link Unit} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Unit}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Unit}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Unit} object to an {@link SBMLDocument}, the SBML
   * Level, SBML Version and XML namespace of the document 
   * <em>override</em> the values used when creating the {@link Unit} object via this
   * constructor.  This is necessary to ensure that an SBML document is a
   * consistent structure.  Nevertheless, the ability to supply the values
   * at the time of creation of a {@link Unit} is an important aid to producing
   * valid SBML.  Knowledge of the intented SBML Level and Version
   * determine whether it is valid to assign a particular value to an
   * attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Unit::Unit(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link Unit} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Unit} object to an {@link SBMLDocument}, the SBML
   * XML namespace of the document <em>overrides</em> the value used when
   * creating the {@link Unit} object via this constructor.  This is necessary to
   * ensure that an SBML document is a consistent structure.  Nevertheless,
   * the ability to supply the values at the time of creation of a {@link Unit} is
   * an important aid to producing valid SBML.  Knowledge of the intented
   * SBML Level and Version determine whether it is valid to assign a
   * particular value to an attribute, or whether it is valid to add an
   * object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Unit::Unit(const Unit& orig) "
/**
   * Copy constructor; creates a copy of this {@link Unit}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers Unit::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Unit}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link Unit} in the list
   * of units within which this {@link Unit} is embedded (i.e., in the {@link ListOfUnits}
   * located in the enclosing {@link UnitDefinition} instance).
   */ public
";


%javamethodmodifiers Unit::clone() const "
/**
   * Creates and returns a deep copy of this {@link Unit}.
   * <p>
   * @return a (deep) copy of this {@link Unit}.
   */ public
";


%javamethodmodifiers Unit::initDefaults "
/**
   * Initializes the fields of this {@link Unit} object to \'typical\' default
   * values.
   * <p>
   * The SBML {@link Unit} component has slightly different aspects and default
   * attribute values in different SBML Levels and Versions.  This method
   * sets the values to certain common defaults, based mostly on what they
   * are in SBML Level&nbsp;2.  Specifically:
   * <ul>
   * <li> Sets attribute \'exponent\' to <code>1</code>
   * <li> Sets attribute \'scale\' to <code>0</code>
   * <li> Sets attribute \'multiplier\' to <code>1.0</code>
   * </ul>
   * <p>
   * The \'kind\' attribute is left unchanged.
   */ public
";


%javamethodmodifiers Unit::getKind() const "
/**
   * Returns the \'kind\' of {@link Unit} this is.
   * <p>
   * @return the value of the \'kind\' attribute of this {@link Unit} as a
   * value from the set of constants whose names begin
   * with <code>UNIT_KIND_</code> defined in the class
   * <code><a href=\'libsbmlConstants.html\'>libsbmlConstants</a></code>.
   * <p>
   */ public
";


%javamethodmodifiers Unit::getExponent() const "
/**
   * Returns the value of the \'exponent\' attribute of this unit.
   * <p>
   * @return the \'exponent\' value of this {@link Unit}, as an integer.
   */ public
";


%javamethodmodifiers Unit::getExponentAsDouble() const "
/**
   * Returns the value of the \'exponent\' attribute of this unit.
   * <p>
   * @return the \'exponent\' value of this {@link Unit}, as a double.
   */ public
";


%javamethodmodifiers Unit::getScale() const "
/**
   * Returns the value of the \'scale\' attribute of this unit.
   * <p>
   * @return the \'scale\' value of this {@link Unit}, as an integer.
   */ public
";


%javamethodmodifiers Unit::getMultiplier() const "
/**
   * Returns the value of the \'multiplier\' attribute of this {@link Unit}.
   * <p>
   * @return the \'multiplier\' value of this {@link Unit}, as a double.
   */ public
";


%javamethodmodifiers Unit::getOffset() const "
/**
   * Returns the value of the \'offset\' attribute of this {@link Unit}.
   * <p>
   * @warning The \'offset\' attribute is only available in SBML Level&nbsp;2
   * Version&nbsp;1.  This attribute is not present in SBML Level&nbsp;2
   * Version&nbsp;2 or above.  When producing SBML models using these later
   * specifications, modelers and software tools need to account for units
   * with offsets explicitly.  The SBML specification document offers a
   * number of suggestions for how to achieve this.  LibSBML methods such
   * as this one related to \'offset\' are retained for compatibility with
   * earlier versions of SBML Level&nbsp;2, but their use is strongly
   * discouraged.
   * <p>
   * @return the \'offset\' value of this {@link Unit}, as a double.
   */ public
";


%javamethodmodifiers Unit::isAmpere() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>ampere.</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>ampere</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isAvogadro() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>avogadro.</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>avogadro</code>, <code>false</code>
   * otherwise.
   * <p>
   * @note The unit <code>avogadro</code> was introduced in SBML Level&nbsp;3, and
   * is only permitted for use in SBML Level&nbsp;3 models.
   */ public
";


%javamethodmodifiers Unit::isBecquerel() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>becquerel</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>becquerel</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isCandela() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>candela</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>candela</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isCelsius() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>Celsius</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>Celsius</code>, <code>false</code>
   * otherwise. 
   * <p>
   * @warning The predefined unit <code>Celsius</code> was removed from the list of
   * predefined units in SBML Level&nbsp;2 Version&nbsp;2 at the same time
   * that the \'offset\' attribute was removed from {@link Unit} definitions.
   * LibSBML methods such as this one related to <code>Celsius</code> are retained in
   * order to support SBML Level&nbsp;2 Version&nbsp;1, but their use is
   * strongly discouraged.
   */ public
";


%javamethodmodifiers Unit::isCoulomb() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>coulomb</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>coulomb</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isDimensionless() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind 
   * <code>dimensionless.</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>dimensionless</code>, <code>false</code>
   * <p>
   * otherwise.
   */ public
";


%javamethodmodifiers Unit::isFarad() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>farad</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>farad</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isGram() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>gram</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>gram</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isGray() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>gray</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>gray</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isHenry() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>henry</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>henry</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isHertz() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>hertz</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>hertz</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isItem() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>item</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>item</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isJoule() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>joule</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>joule</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isKatal() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>katal</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>katal</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isKelvin() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>kelvin</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>kelvin</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isKilogram() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>kilogram</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>kilogram</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isLitre() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>litre</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>litre</code> or \'liter\', 
   * <code>false</code> 
   * otherwise.
   */ public
";


%javamethodmodifiers Unit::isLumen() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>lumen</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>lumen</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isLux() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>lux</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>lux</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isMetre() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>metre</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>metre</code> or \'meter\', 
   * <code>false</code> 
   * otherwise.
   */ public
";


%javamethodmodifiers Unit::isMole() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>mole</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>mole</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isNewton() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>newton</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>newton</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isOhm() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>ohm</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>ohm</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isPascal() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>pascal</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>pascal</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isRadian() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>radian</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>radian</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isSecond() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>second</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>second</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isSiemens() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>siemens</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>siemens</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isSievert() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>sievert</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>sievert</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isSteradian() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>steradian</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>steradian</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isTesla() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>tesla</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>tesla</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isVolt() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>volt</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>volt</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isWatt() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>watt</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>watt</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isWeber() const "
/**
   * Predicate for testing whether this {@link Unit} is of the kind <code>weber</code>
   * <p>
   * @return <code>true</code> if the kind of this {@link Unit} is <code>weber</code>, <code>false</code>
   * otherwise. 
   */ public
";


%javamethodmodifiers Unit::isSetKind() const "
/**
   * Predicate to test whether the \'kind\' attribute of this {@link Unit} is set.
   * <p>
   * @return <code>true</code> if the \'kind\' attribute of this {@link Unit} is set, 
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Unit::isSetExponent() const "
/**
   * Predicate to test whether the \'exponent\' attribute of this {@link Unit} 
   * is set.
   * <p>
   * @return <code>true</code> if the \'exponent\' attribute of this {@link Unit} is set, 
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Unit::isSetScale() const "
/**
   * Predicate to test whether the \'scale\' attribute of this {@link Unit} 
   * is set.
   * <p>
   * @return <code>true</code> if the \'scale\' attribute of this {@link Unit} is set, 
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Unit::isSetMultiplier() const "
/**
   * Predicate to test whether the \'multiplier\' attribute of this {@link Unit} 
   * is set.
   * <p>
   * @return <code>true</code> if the \'multiplier\' attribute of this {@link Unit} is set, 
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Unit::setKind(UnitKind_t kind) "
/**
   * Sets the \'kind\' attribute value of this {@link Unit}.
   * <p>
   * @param kind a unit identifier chosen from the set of constants whose
   * names begin with <code>UNIT_KIND_</code> in <code><a
   * href=\'libsbmlConstants.html\'>libsbmlConstants</a></code>.
   * <p>
   * <p>
   * @return integer value indicating success/failure of the
   * function. The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Unit::setExponent(int value) "
/**
   * Sets the \'exponent\' attribute value of this {@link Unit}.
   * <p>
   * @param value the integer to which the attribute \'exponent\' should be set
   * <p>
   * @return integer value indicating success/failure of the
   * function. The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Unit::setExponent(double value) "
/**
   * Sets the \'exponent\' attribute value of this {@link Unit}.
   * <p>
   * @param value the double to which the attribute \'exponent\' should be set
   * <p>
   * @return integer value indicating success/failure of the
   * function. The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   */ public
";


%javamethodmodifiers Unit::setScale(int value) "
/**
   * Sets the \'scale\' attribute value of this {@link Unit}.
   * <p>
   * @param value the integer to which the attribute \'scale\' should be set
   * <p>
   * @return integer value indicating success/failure of the
   * function. The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   */ public
";


%javamethodmodifiers Unit::setMultiplier(double value) "
/**
   * Sets the \'multipler\' attribute value of this {@link Unit}.
   * <p>
   * @param value the floating-point value to which the attribute
   * \'multiplier\' should be set
   * <p>
   * @return integer value indicating success/failure of the
   * function. The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   */ public
";


%javamethodmodifiers Unit::setOffset(double value) "
/**
   * Sets the \'offset\' attribute value of this {@link Unit}.
   * <p>
   * @param value the float-point value to which the attribute \'offset\'
   * should set
   * <p>
   * @return integer value indicating success/failure of the
   * function. The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @warning The \'offset\' attribute is only available in SBML Level&nbsp;2
   * Version&nbsp;1.  This attribute is not present in SBML Level&nbsp;2
   * Version&nbsp;2 or above.  When producing SBML models using these later
   * specifications, modelers and software tools need to account for units
   * with offsets explicitly.  The SBML specification document offers a
   * number of suggestions for how to achieve this.  LibSBML methods such
   * as this one related to \'offset\' are retained for compatibility with
   * earlier versions of SBML Level&nbsp;2, but their use is strongly
   * discouraged.
   */ public
";


%javamethodmodifiers Unit::getTypeCode() const "
/**
   * Returns the libSBML type code of this object instance.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers Unit::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link Unit}, is
   * always <code>\'unit\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'unit\'.</code> 
   */ public
";


%javamethodmodifiers Unit::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Unit::isBuiltIn(const std::string& name, unsigned int level) "
/**
   * Predicate to test whether a given string is the name of a
   * predefined SBML unit.
   * <p>
   * @param name a string to be tested against the predefined unit names
   * <p>
   * @param level the Level of SBML for which the determination should be
   * made.  This is necessary because there are a few small differences
   * in allowed units between SBML Level&nbsp;1 and Level&nbsp;2.
   * <p>
   * @return <code>true</code> if <code>name</code> is one of the five SBML predefined unit
   * identifiers (<code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>, <code>\'length\'</code> or 
   * <code>\'time\'</code>), <code>false</code> otherwise.
   * <p>
   * @note The predefined unit identifiers <code>\'length\'</code> and <code>\'area\'</code> were
   * added in Level&nbsp;2 Version&nbsp;1.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link Unit}), and the
   * other will be a standalone top-level function with the name
   * Unit_isBuiltIn(). They are functionally
   * identical. 
   */ public
";


%javamethodmodifiers Unit::isUnitKind(const std::string& name, unsigned int level, unsigned int version) "
/**
   * Predicate to test whether a given string is the name of a valid
   * base unit in SBML (such as <code>\'gram\'</code> or <code>\'mole\'</code>).
   * <p>
   * This method exists because prior to SBML Level&nbsp;2 Version&nbsp;3,
   * an enumeration called <code>UnitKind</code> was defined by SBML.  This enumeration
   * was removed in SBML Level&nbsp;2 Version&nbsp;3 and its values were
   * folded into the space of values of a type called <code>UnitSId.</code>  This method
   * therefore has less significance in SBML Level&nbsp;2 Version&nbsp;3
   * and Level&nbsp;2 Version&nbsp;4, but remains for backward
   * compatibility and support for reading models in older Versions of
   * Level&nbsp;2.
   * <p>
   * @param name a string to be tested
   * <p>
   * @param level a long integer representing the SBML specification
   * Level 
   * <p>
   * @param version a long integer representing the SBML specification
   * Version
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * <p>
   * @note The allowed unit names differ between SBML Levels&nbsp;1
   * and&nbsp;2 and again slightly between Level&nbsp;2 Versions&nbsp;1
   * and&nbsp;2.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link Unit}), and the
   * other will be a standalone top-level function with the name
   * Unit_isUnitKind(). They are functionally
   * identical. 
   */ public
";


%javamethodmodifiers Unit::areIdentical(Unit * unit1, Unit * unit2) "
/** 
   * Predicate returning <code>true</code> if two
   * {@link Unit} objects are identical.
   * <p>
   * Two {@link Unit} objects are considered to be <em>identical</em> if they match in
   * all attributes.  (Contrast this to the method areEquivalent(* {@link Unit} u1, Unit u2), which compares {@link Unit} objects only with respect
   * to certain attributes.)
   * <p>
   * @param unit1 the first {@link Unit} object to compare
   * @param unit2 the second {@link Unit} object to compare
   * <p>
   * @return <code>true</code> if all the attributes of unit1 are identical
   * to the attributes of unit2, <code>false</code> otherwise.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link Unit}), and the
   * other will be a standalone top-level function with the name
   * Unit_areIdentical(). They are functionally
   * identical. 
   * <p>
   * @see Unit#areEquivalent(Unit u1, Unit u2) 
   */ public
";


%javamethodmodifiers Unit::areEquivalent(Unit * unit1, Unit * unit2) "
/** 
   * Predicate returning <code>true</code> if 
   * {@link Unit} objects are equivalent.
   * <p>
   * Two {@link Unit} objects are considered to be <em>equivalent</em> either if (1) both
   * have a \'kind\' attribute value of <code>dimensionless</code>, or (2) their \'kind\',
   * \'exponent\' and (for SBML Level&nbsp;2 Version&nbsp;1) \'offset\'
   * attribute values are equal. (Contrast this to the method
   * areIdentical(Unit u1, Unit u2), which compares {@link Unit} objects with respect to all
   * attributes, not just the \'kind\' and \'exponent\'.)
   * <p>
   * @param unit1 the first {@link Unit} object to compare
   * @param unit2 the second {@link Unit} object to compare
   * <p>
   * @return <code>true</code> if the \'kind\' and \'exponent\' attributes of unit1 are
   * identical to the kind and exponent attributes of unit2, <code>false</code>
   * otherwise.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link Unit}), and the
   * other will be a standalone top-level function with the name
   * Unit_areEquivalent(). They are functionally
   * identical. 
   * <p>
   * @see Unit#areIdentical(Unit u1, Unit u2) 
   */ public
";


%javamethodmodifiers Unit::removeScale(Unit * unit) "
/** 
   * Manipulates the attributes of the {@link Unit} to express the unit with the 
   * value of the scale attribute reduced to zero.
   * <p>
   * For example, 1 millimetre can be expressed as a {@link Unit} with kind=
   * <code>\'metre\'</code> multiplier=<code>\'1\'</code> scale=<code>\'-3\'</code> exponent=<code>\'1\'.</code> It can also be
   * expressed as a {@link Unit} with kind=<code>\'metre\'</code>
   * multiplier=<code>\'0.001\'</code> scale=<code>\'0\'</code> exponent=<code>\'1\'.</code>
   * <p>
   * @param unit the {@link Unit} object to manipulate.
   * <p>
   * @return integer value indicating success/failure of the function.  The
   * possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link Unit}), and the
   * other will be a standalone top-level function with the name
   * Unit_removeScale(). They are functionally
   * identical. 
   * <p>
   * @see Unit#convertToSI(Unit u) 
   * @see Unit#merge(Unit u1, Unit u2) 
   */ public
";


%javamethodmodifiers Unit::merge(Unit * unit1, Unit * unit2) "
/** 
   * Merges two {@link Unit} objects with the same \'kind\' attribute value into a
   * single {@link Unit}.
   * <p>
   * For example, the following,
   * <div class=\'fragment\'><pre>
 &lt;unit kind=\'metre\' exponent=\'2\'/&gt;
 &lt;unit kind=\'metre\' exponent=\'1\'/&gt;
 </pre></div>
   * would be merged to become
   * <div class=\'fragment\'><pre>
 &lt;unit kind=\'metre\' exponent=\'3\'/&gt;
 </pre></div>
   * <p>
   * @param unit1 the first {@link Unit} object; the result of the operation is
   * left as a new version of this unit, modified in-place.
   * <p>
   * @param unit2 the second {@link Unit} object to merge with the first
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link Unit}), and the
   * other will be a standalone top-level function with the name
   * Unit_merge(). They are functionally
   * identical. 
   * <p>
   * @see Unit#convertToSI(Unit u) 
   * @see Unit#removeScale(Unit u) 
   */ public
";


%javamethodmodifiers Unit::convertToSI(const Unit * unit) "
/**
   * Returns a {@link UnitDefinition} object containing the given <code>unit</code> converted
   * to the appropriate SI unit.
   * <p>
   * This method exists because some units can be expressed in terms of
   * others when the same physical dimension is involved.  For example, one
   * hertz is identical to 1&nbsp;sec<sup>-1</sup>, one litre is equivalent
   * to 1 cubic decametre, and so on.
   * <p>
   * @param unit the {@link Unit} object to convert to SI
   * <p>
   * @return a {@link UnitDefinition} object containing the SI unit.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link Unit}), and the
   * other will be a standalone top-level function with the name
   * Unit_convertToSI(). They are functionally
   * identical. 
   * <p>
   * @see Unit#merge(Unit u1, Unit u2) 
   */ public
";


%javamethodmodifiers Unit::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Unit} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Unit} object are:
   * <ul>
   * <li> \'kind\'
   * <li> \'exponent\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'multiplier\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'scale\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers Unit::setExponentUnitChecking(double value) "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Unit} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Unit} object are:
   * <ul>
   * <li> \'kind\'
   * <li> \'exponent\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'multiplier\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'scale\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   * @internal
   */ public
";


%javamethodmodifiers Unit::getExponentUnitChecking "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Unit} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Unit} object are:
   * <ul>
   * <li> \'kind\'
   * <li> \'exponent\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'multiplier\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'scale\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   * @internal
   */ public
";


%javamethodmodifiers Unit::getExponentUnitChecking() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Unit} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Unit} object are:
   * <ul>
   * <li> \'kind\'
   * <li> \'exponent\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'multiplier\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'scale\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   * @internal
   */ public
";


%javamethodmodifiers Unit::isUnitChecking "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Unit} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Unit} object are:
   * <ul>
   * <li> \'kind\'
   * <li> \'exponent\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'multiplier\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'scale\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   * @internal
   */ public
";


%javamethodmodifiers Unit::isUnitChecking() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Unit} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Unit} object are:
   * <ul>
   * <li> \'kind\'
   * <li> \'exponent\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'multiplier\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * <li> \'scale\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   * @internal
   */ public
";


%javamethodmodifiers Unit::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers Unit::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Unit::readL1Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Unit::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Unit::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Unit::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Unit::isL1UnitKind(const std::string& name) "
/**
   * Predicate to test whether a given string is the name of a valid
   * base unit in SBML Level 1 (such as <code>\'gram\'</code> or <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * @internal
   */ public
";


%javamethodmodifiers Unit::isL2V1UnitKind(const std::string& name) "
/**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;1 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * @internal
   */ public
";


%javamethodmodifiers Unit::isL2UnitKind(const std::string& name) "
/**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;2 Version&nbsp;2, 3 or 4 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * @internal
   */ public
";


%javamethodmodifiers Unit::isL3UnitKind(const std::string& name) "
/**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;3 Version&nbsp;1 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * @internal
   */ public
";


%javamethodmodifiers Unit::isExplicitlySetExponent() const "
/**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;3 Version&nbsp;1 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * @internal
   */ public
";


%javamethodmodifiers Unit::isExplicitlySetMultiplier() const "
/**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;3 Version&nbsp;1 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * @internal
   */ public
";


%javamethodmodifiers Unit::isExplicitlySetScale() const "
/**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;3 Version&nbsp;1 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * @internal
   */ public
";


%javamethodmodifiers Unit::isExplicitlySetOffset() const "
/**
   * Predicate to test whether a given string is the name of a valid base
   * unit in SBML Level&nbsp;3 Version&nbsp;1 (such as <code>\'gram\'</code> or 
   * <code>\'mole\'</code>)
   * <p>
   * @param name a string to be tested
   * <p>
   * @return <code>true</code> if name is a valid SBML UnitKind, <code>false</code> otherwise
   * @internal
   */ public
";


%javamethodmodifiers ListOfUnits::ListOfUnits(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfUnits} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfUnits::ListOfUnits(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfUnits} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfUnits} object to be created.
   */ public
";


%javamethodmodifiers ListOfUnits::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfUnits}.
   * <p>
   * @return a (deep) copy of this {@link ListOfUnits}.
   */ public
";


%javamethodmodifiers ListOfUnits::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfUnits::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Unit} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfUnits::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfUnits}, the XML element name is <code>\'listOfUnits\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfUnits\'.</code>
   */ public
";


%javamethodmodifiers ListOfUnits::get(unsigned int n) "
/**
   * Get a {@link Unit} from the {@link ListOfUnits}.
   * <p>
   * @param n the index number of the {@link Unit} to get.
   * <p>
   * @return the nth {@link Unit} in this {@link ListOfUnits}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfUnits::get "
/**
   * Get a {@link Unit} from the {@link ListOfUnits}.
   * <p>
   * @param n the index number of the {@link Unit} to get.
   * <p>
   * @return the nth {@link Unit} in this {@link ListOfUnits}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfUnits::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfUnits} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfUnits::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfUnits::createObject(XMLInputStream& stream) "
/**
   * Create a {@link ListOfUnits} object corresponding to the next token
   * in the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) UnitDefinition "
/** 
 * LibSBML implementation of SBML\'s UnitDefinition construct.
 * <p>
 * Units of measurement may be supplied in a number of contexts in an SBML
 * model.  The SBML unit definition facility uses two classes of objects,
 * {@link UnitDefinition} and {@link Unit}.  The approach to defining units in SBML is
 * compositional; for example, <em>meter second<sup> &ndash;2</sup></em> is
 * constructed by combining a {@link Unit} object representing <em>meter</em> with
 * another {@link Unit} object representing <em>second<sup> &ndash;2</sup></em>.
 * The combination is wrapped inside a {@link UnitDefinition}, which provides for
 * assigning an identifier and optional name to the combination.  The
 * identifier can then be referenced from elsewhere in a model.  Thus, the
 * {@link UnitDefinition} class is the container, and {@link Unit} instances are placed
 * inside {@link UnitDefinition} instances.
 * <p>
 * Two points are worth discussing in the context of SBML units.  First,
 * unit declarations in SBML models are <em>optional</em>.  The consequence of
 * this is that a model must be numerically self-consistent independently
 * of unit declarations, for the benefit of software tools that cannot
 * interpret or manipulate units.  {@link Unit} declarations in SBML are thus more
 * akin to a type of annotation; they can indicate intentions, and can be
 * used by model readers for checking the consistency of the model,
 * labeling simulation output, etc., but any transformations of values
 * implied by different units must be incorporated <em>explicitly</em> into a
 * model.
 * <p>
 * Second, the vast majority of situations that require new SBML unit
 * definitions involve simple multiplicative combinations of base units and
 * factors.  An example is <em>moles per litre per second</em>.  What
 * distinguishes these sorts of unit definitions from more complex ones is
 * that they may be expressed without the use of an additive offset from a
 * zero point.  The use of offsets complicates all unit definition systems,
 * yet in the domain of SBML, the real-life cases requiring offsets are few
 * (and in fact, to the best of our knowledge, only involve temperature).
 * Consequently, the SBML unit system has been consciously designed to
 * simplify implementation of unit support for the most common cases in
 * systems biology.  The cost of this simplification is to require units
 * with offsets to be handled explicitly by the modeler.
 * <p>
 * <h2>Summary of the {@link UnitDefinition} construct</h2>
 * <p>
 * {@link UnitDefinition} has two attributes and one subelement.  The two
 * attributes are \'id\' and \'name\', and the subelement is {@link ListOfUnits}.
 * <p>
 * The required attribute \'id\' and optional attribute \'name\' are both
 * strings.  The \'id\' attribute is used to give the defined unit a unique
 * identifier by which other parts of an SBML model definition can refer to
 * it.  The \'name\' attribute is intended to be used for giving the unit
 * definition an optional human-readable name.  Please see the <a
 * href=\'#unitdef-id\'>next section</a> for information about the values
 * permitted for \'id\'.
 * <p>
 * A {@link UnitDefinition} must contain exactly one {@link ListOfUnits}, and this list
 * must contain one or more {@link Unit} definitions; see the definitions of these
 * other object classes for more information about them.  The following
 * example illustrates a complete unit definition (when written in XML)
 * when they all the pieces are combined together.  This defines \'mmls\'
 * to be millimoles per litre per second.
 * <div class=\'fragment\'><pre>
 &lt;listOfUnitDefinitions&gt;
     &lt;unitDefinition id=\'mmls\'&gt;
         &lt;listOfUnits&gt;
             &lt;unit kind=\'mole\'   scale=\'-3\'/&gt;
             &lt;unit kind=\'litre\'  exponent=\'-1\'/&gt;
             &lt;unit kind=\'second\' exponent=\'-1\'/&gt;
         &lt;/listOfUnits&gt;
     &lt;/unitDefinition&gt;
 &lt;/listOfUnitDefinitions&gt;
 </pre></div>
 * <p>
 * <h2>Special considerations for {@link Unit} object identifiers</h2>
 * <p>
 * The attribute \'id\' in {@link UnitDefinition} cannot be given simply any value,
 * and the precise details of the values permitted differ slightly between
 * Levels of SBML:
 * <ul>
 * <p>
 * <li> The \'id\' of a {@link UnitDefinition} must <em>not</em> contain a value from the
 * list of SBML\'s predefined base unit names (i.e., the strings <code>gram</code>, 
 * <code>litre</code>, etc.).  In SBML Level&nbsp;3, this list consists of the
 * following:
 * <p>
 * <center>
<table border=\'0\' class=\'text-table width80 normal-font\'
       style=\'border-top: none !important; border-bottom: none !important; color: black; font-weight: bold; font-family: Courier New, Courier, monospace, fixed;\'>
<tr>
<td>ampere</td><td>farad</td><td>joule</td><td>lux</td><td>radian</td><td>volt</td>
</tr>
<tr>
<td>avogadro</td><td>gram</td><td>katal</td><td>metre</td><td>second</td><td>watt</td>
</tr>
<tr>
<td>becquerel</td><td>gray</td><td>kelvin</td><td>mole</td><td>siemens</td><td>weber</td>
</tr>
<tr>
<td>candela</td><td>henry</td><td>kilogram</td><td>newton</td><td>sievert</td>
</tr>
<tr>
<td>coulomb</td><td>hertz</td><td>litre</td><td>ohm</td><td>steradian</td>
</tr>
<tr>
<td>dimensionless</td><td>item</td><td>lumen</td><td>pascal</td><td>tesla</td>
</tr>
</table>
</center>

 * <p>
 * This list of predefined base units is nearly identical in SBML
 * Level&nbsp;2 Version&nbsp;4, the exception being that Level&nbsp;2 does
 * not define <code>avogadro.</code>  SBML Level&nbsp;2 Version&nbsp;1 (and <em>only</em>
 * this Level+Version combination) provides an additional predefined unit
 * name, <code>Celsius</code>, not available in Level&nbsp;3.  Finally, SBML
 * Level&nbsp;1 Versions&nbsp;2&ndash;3 provide two more additional
 * predefined unit names, <code>meter</code> and <code>liter.</code>  This is explained in
 * somewhat greater detail in the description of the {@link Unit} class.
 * <p>
 * <li> In SBML Level&nbsp;2 (all Versions), there is an additional set of
 * reserved identifiers: <code>substance</code>, <code>volume</code>, <code>area</code>, <code>length</code>, and
 * <code>time.</code>  Using one of these values for the attribute \'id\' of a
 * {@link UnitDefinition} has the effect of redefining the model-wide default units
 * for the corresponding quantities.  The list of special unit names in
 * SBML Level&nbsp;2 is given in the table below:
 * <p>
 *   <center>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
 <tr>
     <th align=\'left\'>Identifier</th>
     <th align=\'left\'>Possible scalable units</th>
     <th align=\'left\'>Default units</th>
 </tr>
<tr><td><code>substance</code></td><td>mole, item, gram, kilogram, dimensionless</td><td>mole</td></tr>
<tr><td><code>volume</code></td><td>litre, cubic metre, dimensionless</td><td>litre</td></tr>
<tr><td><code>area</code></td><td>square metre, dimensionless</td><td>square metre</td></tr>
<tr><td><code>length</code></td><td>metre, dimensionless</td><td>metre</td></tr>
<tr><td><code>time</code></td><td>second, dimensionless</td><td>second</td></tr>
</table>
</center>

 * <p>
 * Also, SBML Level&nbsp;2 imposes two limitations on redefining the
 * predefined unit <code>substance</code>, <code>volume</code>, <code>area</code>, <code>length</code>, and 
 * <code>time:</code> (1) The {@link UnitDefinition} of a predefined SBML unit can only contain
 * a single {@link Unit} object within it.  (2) The value of the \'kind\' attribute
 * in a {@link Unit} instance must be drawn from one of the values in the second
 * column of the table above.
 * <p>
 * The special unit names <code>substance</code>, <code>volume</code>, <code>area</code>, <code>length</code>, and
 * <code>time</code> are not defined by SBML Level&nbsp;3, which uses a different
 * approach to setting model-wide inherited units.
 * <p>
 * </ul>
 * <p>
 * <p>
 * <h2>Further comments about SBML\'s unit definition system</h2>
 * <p>
 * The vast majority of modeling situations requiring new SBML unit
 * definitions involve simple multiplicative combinations of base units and
 * factors.  An example of this might be <em>moles per litre per
 * second</em>.  What distinguishes these sorts of simpler unit definitions
 * from more complex ones is that they may be expressed without the use of
 * an additive offset from a zero point.  The use of offsets complicates
 * all unit definition systems, yet in the domain of SBML the real-life
 * cases requiring offsets are few (and in fact, to the best of our
 * knowledge, only involve temperature).  Consequently, the SBML unit
 * system has been consciously designed in a way that attempts to simplify
 * implementation of unit support for the most common cases in systems
 * biology.
 * <p>
 * As of SBML Level&nbsp;2 Version&nbsp;2, {@link Unit} no longer has the
 * attribute called \'offset\' introduced in SBML Level&nbsp;2
 * Version&nbsp;1.  It turned out that the general case involving units
 * with offsets was incorrectly defined, and few (if any) developers even
 * attempted to support offset-based units in their software.  In the
 * development of Level&nbsp;2 Version&nbsp;2, a consensus among SBML
 * developers emerged that a fully generalized unit scheme is <em>so</em>
 * confusing and complicated that it actually <em>impedes</em> interoperability.
 * SBML Level&nbsp;2 Version&nbsp;2, Version&nbsp;3 and Version&nbsp;4 acknowledge this
 * reality by reducing and simplifying the unit system, specifically by
 * removing the \'offset\' attribute on {@link Unit} and <code>Celsius</code> as a pre-defined
 * unit.
 * <p>
 * The following guidelines suggest methods for handling units that do
 * require the use of zero offsets for their definitions:
 * <ul>
 * <li> <em>Handling Celsius</em>.  A model in which certain quantities are
 *   temperatures measured in degrees Celsius can be converted
 *   straightforwardly to a model in which those temperatures are in
 *   kelvin.  A software tool could do this by performing a straightforward
 *   substitution using the following relationship: <em>T<sub> kelvin</sub> =
 *   T<sub>Celsius</sub> + 273.15</em>.  In every mathematical formula of the
 *   model where a quantity (call it <em>x</em>) in degrees Celsius appears,
 *   replace <em>x</em> with <em>x<sub>k</sub>+ 273.15</em>, where
 *   <em>x<sub>k</sub></em> is now in kelvin.  An alternative approach would 
 *   be to use a {@link FunctionDefinition} object to define a function encapsulating this
 *   relationship above and then using that in the rest of the model as
 *   needed.  Since Celsius is a commonly-used unit, software tools could
 *   help users by providing users with the ability to express temperatures
 *   in Celsius in the tools\' interfaces, and making substitutions
 *   automatically when writing out the SBML.
 * <p>
 * <li> <em>Other units requiring offsets</em>.  One approach to handling
 *   other kinds of units is to use a {@link FunctionDefinition} to define a function
 *   encapsulating the necessary mathematical relationship, then
 *   substituting a call to this function wherever the original quantity
 *   appeared in the model.  For example, here is a possible definition for
 *   converting Fahrenheit to Celsius degrees:
 *   <div class=\'fragment\'><pre>
 &lt;functionDefinition id=\'Fahrenheit_to_kelvin\'&gt;
     &lt;math xmlns=\'http://www.w3.org/1998/Math/MathML\'&gt;
         &lt;lambda&gt;
             &lt;bvar&gt;&lt;ci&gt; temp_in_fahrenheit &lt;/ci&gt;&lt;/bvar&gt;
             &lt;apply&gt;
                 &lt;divide/&gt;
                 &lt;apply&gt;
                     &lt;plus/&gt;
                     &lt;ci&gt; temp_in_fahrenheit &lt;/ci&gt;
                     &lt;cn&gt; 459.67 &lt;/cn&gt;
                 &lt;/apply&gt;
                 &lt;cn&gt; 1.8 &lt;/cn&gt;
             &lt;/apply&gt;
         &lt;/lambda&gt;
     &lt;/math&gt;
 &lt;/functionDefinition&gt;
 </pre></div>
 * <p>
 * <li> An alternative approach not requiring the use of function definitions
 *   is to use an {@link AssignmentRule} for each variable in Fahrenheit units.
 *   The {@link AssignmentRule} could compute the conversion from Fahrenheit to
 *   (say) kelvin, assign its value to a variable (in Kelvin units), and
 *   then that variable could be used elsewhere in the model.
 * <p>
 * <li> Still another approach is to rewrite the mathematical formulas of a
 *   model to directly incorporate the conversion formula wherever the
 *   original quantity appeared.
 * </ul>
 * <p>
 * Please consult the SBML specifications for more information about this
 * and other issues involving units.
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%typemap(javaimports) ListOfUnitDefinitions "
/** 
 * LibSBML implementation of SBML\'s ListOfUnitDefinitions construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers UnitDefinition::UnitDefinition(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link UnitDefinition} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link UnitDefinition}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link UnitDefinition}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link UnitDefinition} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addUnitDefinition(UnitDefinition ud)}), the SBML Level, SBML Version
   * and XML namespace of the document <em>override</em> the values used
   * when creating the {@link UnitDefinition} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link UnitDefinition} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers UnitDefinition::UnitDefinition(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link UnitDefinition} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link UnitDefinition} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addUnitDefinition(UnitDefinition ud)}), the SBML XML namespace of
   * the document <em>overrides</em> the value used when creating the
   * {@link UnitDefinition} object via this constructor.  This is necessary to
   * ensure that an SBML document is a consistent structure.  Nevertheless,
   * the ability to supply the values at the time of creation of a
   * {@link UnitDefinition} is an important aid to producing valid SBML.  Knowledge
   * of the intented SBML Level and Version determine whether it is valid
   * to assign a particular value to an attribute, or whether it is valid
   * to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers UnitDefinition::UnitDefinition(const UnitDefinition& orig) "
/**
   * Copy constructor; creates a copy of this {@link UnitDefinition}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers UnitDefinition::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link UnitDefinition}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link UnitDefinition} in the
   * list of units within which this {@link UnitDefinition} is embedded (i.e., in
   * the {@link ListOfUnitDefinitions} located in the enclosing {@link Model} instance).
   */ public
";


%javamethodmodifiers UnitDefinition::clone() const "
/**
   * Creates and returns a deep copy of this {@link UnitDefinition}.
   * <p>
   * @return a (deep) copy of this {@link UnitDefinition}.
   */ public
";


%javamethodmodifiers UnitDefinition::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.
   * <p>
   * @param id string representing the id of objects to find.
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers UnitDefinition::getElementByMetaId(std::string metaid) "
/**
   * Returns the first child element it can find with the given metaid, or null if no such object is found.
   * <p>
   * @param metaid string representing the metaid of objects to find
   * <p>
   * @return pointer to the first element found with the given metaid.
   */ public
";


%javamethodmodifiers UnitDefinition::getAllElements "
/**
   * Returns a {@link List} of all child {@link SBase} objects, including those nested to an arbitrary depth
   * <p>
   * @return a {@link List} of pointers to all children objects.
   */ public
";


%javamethodmodifiers UnitDefinition::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link UnitDefinition}.
   * <p>
   * @return the id of this {@link UnitDefinition}.
   */ public
";


%javamethodmodifiers UnitDefinition::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link UnitDefinition}.
   * <p>
   * @return the name of this {@link UnitDefinition}.
   */ public
";


%javamethodmodifiers UnitDefinition::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link UnitDefinition}\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link UnitDefinition} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link UnitDefinition}\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link UnitDefinition} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link UnitDefinition}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link UnitDefinition}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers UnitDefinition::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link UnitDefinition}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link UnitDefinition}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers UnitDefinition::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link UnitDefinition}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers UnitDefinition::isVariantOfArea() const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'area\'.</code>
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>area</code>, meaning square metres with only abritrary variations
   * in scale or multiplier values; <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::isVariantOfLength() const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'length\'.</code>
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>length</code>, meaning metres with only abritrary variations in scale
   * or multiplier values; <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::isVariantOfSubstance() const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'substance\'.</code>
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>substance</code>, meaning moles or items (and grams or kilograms from
   * SBML Level&nbsp;2 Version&nbsp;2 onwards) with only abritrary variations
   * in scale or multiplier values; <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::isVariantOfTime() const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'time\'.</code>
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>time</code>, meaning seconds with only abritrary variations in scale or
   * multiplier values; <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::isVariantOfVolume() const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'volume\'.</code>
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>volume</code>, meaning litre or cubic metre with only abritrary
   * variations in scale or multiplier values; <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::isVariantOfDimensionless() const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the unit <code>\'dimensionless\'.</code>
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of 
   * <code>dimensionless</code>, meaning dimensionless with only abritrary variations in
   * scale or multiplier values; <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::isVariantOfMass() const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier <code>\'mass\'.</code>
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of mass units,
   * meaning gram or kilogram with only abritrary variations in scale or
   * multiplier values; <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::isVariantOfSubstancePerTime() const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit <code>\'substance\'</code> divided by the predefined
   * unit <code>\'time\'.</code>
   * <p>
   * @return <code>true</code> if this {@link UnitDefinition} is a variant of the predefined
   * unit <code>substance</code> per predefined unit <code>time</code>, meaning it contains two
   * units one of which is a variant of substance and the other is a
   * variant of time which an exponent of -1; <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers UnitDefinition::addUnit(const Unit* u) "
/**
   * Adds a copy of the given {@link Unit} to this {@link UnitDefinition}.
   * <p>
   * @param u the {@link Unit} instance to add to this {@link UnitDefinition}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link UnitDefinition}.  Changes made to the original
   * object instance (such as resetting attribute values) will <em>not
   * affect the instance in the {@link UnitDefinition}</em>.  In addition, the
   * caller should make sure to free the original object if it is no longer
   * being used, or else a memory leak will result.  Please see
   * {@link UnitDefinition#createUnit()} for a method that does not lead to these
   * issues.
   * <p>
   * @see #createUnit()
   */ public
";


%javamethodmodifiers UnitDefinition::createUnit "
/**
   * Creates a new and empty {@link Unit}, adds it to this {@link UnitDefinition}\'s list of
   * units, and returns it.
   * <p>
   * @return a newly constructed (and empty) {@link Unit} instance.
   * <p>
   * @note It is worth emphasizing that the attribute \'kind\' value of a
   * {@link Unit} is a required attribute for a valid {@link Unit} definition.  The
   * createUnit() method does not assign a valid kind to the constructed
   * unit (instead, it sets the \'kind\' to {@link  libsbmlConstants#UNIT_KIND_INVALID UNIT_KIND_INVALID}).
   * Callers are cautioned to set the newly-constructed {@link Unit}\'s kind using
   * {@link Unit#setKind(int kind)} soon after calling this method.
   * <p>
   * @see #addUnit(Unit u)
   */ public
";


%javamethodmodifiers UnitDefinition::getListOfUnits() const "
/**
   * Returns the list of Units for this {@link UnitDefinition} instance.
   * @return the {@link ListOfUnits} value for this {@link UnitDefinition}.
   */ public
";


%javamethodmodifiers UnitDefinition::getListOfUnits "
/**
   * Returns the list of Units for this {@link UnitDefinition} instance.
   * @return the {@link ListOfUnits} value for this {@link UnitDefinition}.
   */ public
";


%javamethodmodifiers UnitDefinition::getUnit(unsigned int n) "
/**
   * Returns a specific {@link Unit} instance belonging to this {@link UnitDefinition}.
   * <p>
   * @param n an integer, the index of the {@link Unit} to be returned.
   * <p>
   * @return the nth {@link Unit} of this {@link UnitDefinition}.
   * <p>
   * @see #getNumUnits()
   */ public
";


%javamethodmodifiers UnitDefinition::getUnit "
/**
   * Returns a specific {@link Unit} instance belonging to this {@link UnitDefinition}.
   * <p>
   * @param n an integer, the index of the {@link Unit} to be returned.
   * <p>
   * @return the nth {@link Unit} of this {@link UnitDefinition}.
   */ public
";


%javamethodmodifiers UnitDefinition::getNumUnits() const "
/**
   * Returns the number of {@link Unit} objects contained within this
   * {@link UnitDefinition}.
   * <p>
   * @return an integer representing the number of Units in this
   * {@link UnitDefinition}.
   */ public
";


%javamethodmodifiers UnitDefinition::removeUnit(unsigned int n) "
/**
   * Removes the nth {@link Unit} object from this {@link UnitDefinition} object and
   * returns a pointer to it.
   * <p>
   * The caller owns the returned object and is responsible for deleting it.
   * <p>
   * @param n the index of the {@link Unit} object to remove
   * <p>
   * @return the {@link Unit} object removed, or <code>null</code> if the given index 
   * is out of range.
   * <p>
   */ public
";


%javamethodmodifiers UnitDefinition::setSBMLDocument(SBMLDocument* d) "
/**
   * Sets the parent {@link SBMLDocument} of this SBML object.
   * <p>
   * @param d the {@link SBMLDocument} to use
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::connectToChild "
/**
   * Sets this SBML object to child SBML objects (if any).
   * (Creates a child-parent relationship by the parent)
   * <p>
   * Subclasses must override this function if they define
   * one ore more child elements.
   * Basically, this function needs to be called in
   * constructor, copy constructor and assignment operator.
   * <p>
   * @see setSBMLDocument
   * @see enablePackageInternal
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::enablePackageInternal(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/**
   * Enables/Disables the given package with this element and child
   * elements (if any).
   * (This is an internal implementation for enablePackage function)
   * <p>
   * @note Subclasses of the SBML Core package in which one or more child
   * elements are defined must override this function.
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::getTypeCode() const "
/**
   * Returns the libSBML type code for this object instance.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers UnitDefinition::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link UnitDefinition},
   * is always <code>\'unitDefinition\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'unitDefinition\'.</code>
   */ public
";


%javamethodmodifiers UnitDefinition::simplify(UnitDefinition * ud) "
/** 
   * Simplifies the {@link UnitDefinition} such that any given kind of {@link Unit} object
   * occurs only once in the {@link ListOfUnits}.
   * <p>
   * For example, the following definition,
   * <div class=\'fragment\'><pre>
 &lt;unitDefinition&gt;
  &lt;listOfUnits&gt;
    &lt;unit kind=\'metre\' exponent=\'1\'/&gt;
    &lt;unit kind=\'metre\' exponent=\'2\'/&gt;
  &lt;/listOfUnits&gt;
 &lt;unitDefinition&gt;
 </pre></div>
   * will be simplified to 
   * <div class=\'fragment\'><pre>
 &lt;unitDefinition&gt;
   &lt;listOfUnits&gt;
     &lt;unit kind=\'metre\' exponent=\'3\'/&gt;
   &lt;/listOfUnits&gt;
 &lt;unitDefinition&gt;
 </pre></div>
   * <p>
   * @param ud the {@link UnitDefinition} object to be simplified.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link UnitDefinition}), and the
   * other will be a standalone top-level function with the name
   * UnitDefinition_simplify(). They are functionally identical. 
   */ public
";


%javamethodmodifiers UnitDefinition::reorder(UnitDefinition * ud) "
/** 
   * Alphabetically orders the {@link Unit} objects within the {@link ListOfUnits} of a
   * {@link UnitDefinition}.
   * <p>
   * @param ud the {@link UnitDefinition} object whose units are to be reordered.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link UnitDefinition}), and the
   * other will be a standalone top-level function with the name
   * UnitDefinition_reorder(). They are functionally identical. 
   */ public
";


%javamethodmodifiers UnitDefinition::convertToSI(const UnitDefinition *ud) "
/**
   * Convert a given {@link UnitDefinition} into a new {@link UnitDefinition} object
   * that uses SI units.
   * <p>
   * @param ud the {@link UnitDefinition} object to convert to SI
   * <p>
   * @return a new {@link UnitDefinition} object representing the results of the
   * conversion.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link UnitDefinition}), and the
   * other will be a standalone top-level function with the name
   * UnitDefinition_convertToSI(). They are functionally identical. 
   */ public
";


%javamethodmodifiers UnitDefinition::areIdentical(const UnitDefinition * ud1, const UnitDefinition * ud2) "
/** 
   * Predicate returning <code>true</code> if two
   * {@link UnitDefinition} objects are identical.
   * <p>
   * For the purposes of performing this comparison, two {@link UnitDefinition}
   * objects are considered identical when they contain identical lists of
   * {@link Unit} objects.  Pairs of {@link Unit} objects in the lists are in turn
   * considered identical if they satisfy the predicate
   * {@link Unit#areIdentical(Unit u1, Unit u2)}.
   * The predicate compares every attribute of the
   * {@link Unit} objects.
   * <p>
   * @param ud1 the first {@link UnitDefinition} object to compare
   * @param ud2 the second {@link UnitDefinition} object to compare
   * <p>
   * @return <code>true</code> if all the {@link Unit} objects in ud1 are identical to the
   * {@link Unit} objects of ud2, <code>false</code> otherwise.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link UnitDefinition}), and the
   * other will be a standalone top-level function with the name
   * UnitDefinition_areIdentical(). They are functionally identical. 
   * <p>
   * @see UnitDefinition#areEquivalent(UnitDefinition  ud1, UnitDefinition  ud2)
   * @see Unit#areIdentical(Unit  unit1, Unit  unit2)
   */ public
";


%javamethodmodifiers UnitDefinition::areEquivalent(const UnitDefinition *ud1 , const UnitDefinition * ud2) "
/** 
   * Predicate returning <code>true</code> if two
   * {@link UnitDefinition} objects are equivalent.
   * <p>
   * For the purposes of performing this comparison, two {@link UnitDefinition}
   * objects are considered equivalent when they contain <em>equivalent</em>
   * list of {@link Unit} objects.  {@link Unit} objects are in turn considered equivalent
   * if they satisfy the predicate
   * {@link Unit#areEquivalent(Unit u1, Unit u2)}.
   * The predicate tests a subset of the objects\'s attributes.
   * <p>
   * @param ud1 the first {@link UnitDefinition} object to compare
   * <p>
   * @param ud2 the second {@link UnitDefinition} object to compare
   * <p>
   * @return <code>true</code> if all the {@link Unit} objects in ud1 are equivalent
   * to the {@link Unit} objects in ud2, <code>false</code> otherwise.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link UnitDefinition}), and the
   * other will be a standalone top-level function with the name
   * UnitDefinition_areEquivalent(). They are functionally identical. 
   * <p>
   * @see UnitDefinition#areIdentical(UnitDefinition  ud1, UnitDefinition  ud2)
   * @see Unit#areEquivalent(Unit  unit1, Unit  unit2)
   */ public
";


%javamethodmodifiers UnitDefinition::areIdenticalSIUnits(const UnitDefinition * ud1, const UnitDefinition * ud2) "
/** 
   * Predicate returning <code>true</code> if two
   * {@link UnitDefinition} objects are equivalent.
   * <p>
   * For the purposes of performing this comparison, two {@link UnitDefinition}
   * objects are considered equivalent when they contain <em>equivalent</em>
   * list of {@link Unit} objects.  {@link Unit} objects are in turn considered equivalent
   * if they satisfy the predicate
   * {@link Unit#areEquivalent(Unit u1, Unit u2)}.
   * The predicate tests a subset of the objects\'s attributes.
   * <p>
   * @param ud1 the first {@link UnitDefinition} object to compare
   * <p>
   * @param ud2 the second {@link UnitDefinition} object to compare
   * <p>
   * @return <code>true</code> if all the {@link Unit} objects in ud1 are equivalent
   * to the {@link Unit} objects in ud2, <code>false</code> otherwise.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link UnitDefinition}), and the
   * other will be a standalone top-level function with the name
   * UnitDefinition_areEquivalent(). They are functionally identical. 
   * <p>
   * @see UnitDefinition#areIdentical(UnitDefinition  ud1, UnitDefinition  ud2)
   * @see Unit#areEquivalent(Unit  unit1, Unit  unit2)
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::combine(UnitDefinition * ud1, UnitDefinition * ud2) "
/** 
   * Combines two {@link UnitDefinition} objects into a single {@link UnitDefinition}.
   * <p>
   * This takes {@link UnitDefinition} objects <code>ud1</code> and <code>ud2</code>, and creates a
   * {@link UnitDefinition} object that expresses the product of the units of 
   * <code>ud1</code> and <code>ud2</code>.
   * <p>
   * @param ud1 the first {@link UnitDefinition} object 
   * @param ud2 the second {@link UnitDefinition} object
   * <p>
   * @return a {@link UnitDefinition} which represents the product of the 
   * units of the two argument UnitDefinitions.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link UnitDefinition}), and the
   * other will be a standalone top-level function with the name
   * UnitDefinition_combine(). They are functionally identical. 
   */ public
";


%javamethodmodifiers UnitDefinition::printUnits(const UnitDefinition * ud, bool compact = false) "
/** 
   * Expresses the given definition in a plain-text form.
   * <p>
   * For example,
   * {@link UnitDefinition#printUnits(UnitDefinition u)}
   * applied to
   * <div class=\'fragment\'><pre>
 &lt;unitDefinition&gt;
  &lt;listOfUnits&gt;
    &lt;unit kind=\'metre\' exponent=\'1\'/&gt;
    &lt;unit kind=\'second\' exponent=\'-2\'/&gt;
  &lt;/listOfUnits&gt;
 &lt;unitDefinition&gt;
 </pre></div>
   * will return the string <code>\'metre (exponent = 1, multiplier = 1,
   * scale = 0) second (exponent = -2, multiplier = 1, scale = 0)\'</code>
   * or, if the optional parameter <code>compact</code> is given the value <code>true</code>,
   * the string <code>\'(1 metre)^1 (1 second)^-2\'</code>.  This method may
   * be useful for printing unit information to human users, or in
   * debugging software, or other situations.
   * <p>
   * @param ud the {@link UnitDefinition} object
   * @param compact boolean indicating whether the compact form
   * should be used (defaults to false)
   * <p>
   * @return a string expressing the unit definition defined by the given
   * {@link UnitDefinition} object <code>ud</code>.
   * <p>
   * @note Because this is a <em>static</em> method, the
   * non-C++ language interfaces for libSBML will contain two variants.  One
   * will be a static method on the class (i.e., {@link UnitDefinition}), and the
   * other will be a standalone top-level function with the name
   * UnitDefinition_printUnits(). They are functionally identical. 
   */ public
";


%javamethodmodifiers UnitDefinition::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link UnitDefinition} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link UnitDefinition} object are:
   * <ul>
   * <li> \'id\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers UnitDefinition::hasRequiredElements() const "
/**
   * Predicate returning <code>true</code> if
   * all the required elements for this {@link UnitDefinition} object
   * have been set.
   * <p>
   * @note The required elements for a {@link Constraint} object are:
   * <ul>
   * <li> \'listOfUnits\' (required in SBML Level&nbsp;2 only, optional in Level&nbsp;3)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers UnitDefinition::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::readL1Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::ListOfUnitDefinitions(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfUnitDefinitions} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::ListOfUnitDefinitions(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfUnitDefinitions} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfUnitDefinitions} object to be created.
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfUnitDefinitions} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfUnitDefinitions}.
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link UnitDefinition} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfUnitDefinitions}, the XML element name is 
   * <code>\'listOfUnitDefinitions\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfUnitDefinitions\'.</code>
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::get(unsigned int n) "
/**
   * Get a {@link UnitDefinition} from the {@link ListOfUnitDefinitions}.
   * <p>
   * @param n the index number of the {@link UnitDefinition} to get.
   * <p>
   * @return the nth {@link UnitDefinition} in this {@link ListOfUnitDefinitions}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::get "
/**
   * Get a {@link UnitDefinition} from the {@link ListOfUnitDefinitions}.
   * <p>
   * @param n the index number of the {@link UnitDefinition} to get.
   * <p>
   * @return the nth {@link UnitDefinition} in this {@link ListOfUnitDefinitions}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::get(const std::string& sid) "
/**
   * Get a {@link UnitDefinition} from the {@link ListOfUnitDefinitions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link UnitDefinition} to get.
   * <p>
   * @return {@link UnitDefinition} in this {@link ListOfUnitDefinitions}
   * with the given id or <code>null</code> if no such
   * {@link UnitDefinition} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::get(const std::string& sid) const "
/**
   * Get a {@link UnitDefinition} from the {@link ListOfUnitDefinitions}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link UnitDefinition} to get.
   * <p>
   * @return {@link UnitDefinition} in this {@link ListOfUnitDefinitions}
   * with the given id or <code>null</code> if no such
   * {@link UnitDefinition} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.  Note that UnitDefinitions themselves are in the UnitId namespace, not the SId namespace, so no {@link UnitDefinition} object will be returned from this function (and is the reason we override the base ListOf.getElementBySId function here).
   * <p>
   * @param id string representing the id of objects to find
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfUnitDefinitions} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::remove(const std::string& sid) "
/**
   * Removes item in this {@link ListOfUnitDefinitions} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the
   * {@link ListOfUnitDefinitions} in a model is (in SBML Level&nbsp;2
   * Version&nbsp;4) the second ListOf___.  (However, it differs for
   * different Levels and Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfUnitDefinitions::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or null if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) CompartmentType "
/** 
 * LibSBML implementation of SBML\'s Level&nbsp;2\'s CompartmentType construct.
 * <p>
 * SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 provide the <em>compartment
 * type</em> as a grouping construct that can be used to establish a
 * relationship between multiple {@link Compartment} objects.  A {@link CompartmentType}
 * object only has an identity, and this identity can only be used to
 * indicate that particular {@link Compartment} objects in the model belong to this
 * type.  This may be useful for conveying a modeling intention, such as
 * when a model contains many similar compartments, either by their
 * biological function or the reactions they carry.  Without a compartment
 * type construct, it would be impossible within SBML itself to indicate
 * that all of the compartments share an underlying conceptual relationship
 * because each SBML compartment must be given a unique and separate
 * identity.  {@link Compartment} types have no mathematical meaning in
 * SBML&mdash;they have no effect on a model\'s mathematical interpretation.
 * Simulators and other numerical analysis software may ignore
 * {@link CompartmentType} definitions and references to them in a model.
 * <p>
 * There is no mechanism in SBML Level 2 for representing hierarchies of
 * compartment types.  One {@link CompartmentType} instance cannot be the subtype
 * of another {@link CompartmentType} instance; SBML provides no means of defining
 * such relationships.
 * <p>
 * As with other major structures in SBML, {@link CompartmentType} has a mandatory
 * attribute, \'id\', used to give the compartment type an identifier.  The
 * identifier must be a text string conforming to the identifer syntax
 * permitted in SBML.  {@link CompartmentType} also has an optional \'name\'
 * attribute, of type <code>string.</code>  The \'id\' and \'name\' must be used
 * according to the guidelines described in the SBML specification (e.g.,
 * Section 3.3 in the Level 2 Version 4 specification).
 * <p>
 * {@link CompartmentType} was introduced in SBML Level 2 Version 2.  It is not
 * available in SBML Level&nbsp;1 nor in Level&nbsp;3.
 * <p>
 * @see Compartment
 * @see ListOfCompartmentTypes
 * @see SpeciesType
 * @see ListOfSpeciesTypes
 * <p>
 * <p>
 */
"


%typemap(javaimports) ListOfCompartmentTypes "
/** 
 * LibSBML implementation of SBML\'s ListOfCompartmentTypes construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers CompartmentType::CompartmentType(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link CompartmentType} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link CompartmentType}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link CompartmentType}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link CompartmentType} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addCompartmentType(CompartmentType ct)}), the SBML Level, SBML
   * Version and XML namespace of the document <em>override</em> the
   * values used when creating the {@link CompartmentType} object via this
   * constructor.  This is necessary to ensure that an SBML document is a
   * consistent structure.  Nevertheless, the ability to supply the values
   * at the time of creation of a {@link CompartmentType} is an important aid to
   * producing valid SBML.  Knowledge of the intented SBML Level and
   * Version determine whether it is valid to assign a particular value to
   * an attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers CompartmentType::CompartmentType(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link CompartmentType} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link CompartmentType} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method setId(String id).
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link CompartmentType} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addCompartmentType(CompartmentType ct)}), the SBML XML namespace of
   * the document <em>overrides</em> the value used when creating the
   * {@link CompartmentType} object via this constructor.  This is necessary to
   * ensure that an SBML document is a consistent structure.  Nevertheless,
   * the ability to supply the values at the time of creation of a
   * {@link CompartmentType} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers CompartmentType::CompartmentType(const CompartmentType& orig) "
/**
   * Copy constructor; creates a copy of this {@link CompartmentType}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers CompartmentType::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link CompartmentType}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link CompartmentType} in
   * the list of compartment types.
   */ public
";


%javamethodmodifiers CompartmentType::clone() const "
/**
   * Creates and returns a deep copy of this {@link CompartmentType}.
   * <p>
   * @return a (deep) copy of this {@link CompartmentType}.
   */ public
";


%javamethodmodifiers CompartmentType::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link CompartmentType}.
   * <p>
   * @return the id of this {@link CompartmentType}.
   */ public
";


%javamethodmodifiers CompartmentType::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link CompartmentType}.
   * <p>
   * @return the name of this {@link CompartmentType}.
   */ public
";


%javamethodmodifiers CompartmentType::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link CompartmentType}\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link CompartmentType} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers CompartmentType::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link CompartmentType}\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this CompartmentTypeType is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers CompartmentType::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link CompartmentType}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link CompartmentType}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers CompartmentType::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link CompartmentType}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link CompartmentType}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers CompartmentType::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link CompartmentType}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers CompartmentType::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers CompartmentType::getElementName() const "
/**
   * Returns the XML element name of this object, which for
   * {@link CompartmentType}, is always <code>\'compartmentType\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'compartmentType\'.</code>
   */ public
";


%javamethodmodifiers CompartmentType::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers CompartmentType::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link CompartmentType} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link CompartmentType} object are:
   * <ul>
   * <li> \'id\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers CompartmentType::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers CompartmentType::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * <p>
   * @param attributes the {@link XMLAttributes} to use.
   * @internal
   */ public
";


%javamethodmodifiers CompartmentType::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * <p>
   * @param attributes the {@link XMLAttributes} to use.
   * @internal
   */ public
";


%javamethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   * @internal
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::ListOfCompartmentTypes(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfCompartmentTypes} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::ListOfCompartmentTypes(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfCompartmentTypes} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfCompartmentTypes} object to be created.
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfCompartmentTypes} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfCompartmentTypes}.
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link CompartmentType} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfCompartmentTypes}, the XML element name is 
   * <code>\'listOfCompartmentTypes\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfCompartmentTypes\'.</code>
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::get(unsigned int n) "
/**
   * Get a {@link CompartmentType} from the {@link ListOfCompartmentTypes}.
   * <p>
   * @param n the index number of the {@link CompartmentType} to get.
   * <p>
   * @return the nth {@link CompartmentType} in this {@link ListOfCompartmentTypes}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::get "
/**
   * Get a {@link CompartmentType} from the {@link ListOfCompartmentTypes}.
   * <p>
   * @param n the index number of the {@link CompartmentType} to get.
   * <p>
   * @return the nth {@link CompartmentType} in this {@link ListOfCompartmentTypes}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::get(const std::string& sid) "
/**
   * Get a {@link CompartmentType} from the {@link ListOfCompartmentTypes}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link CompartmentType} to get.
   * <p>
   * @return {@link CompartmentType} in this {@link ListOfCompartmentTypes}
   * with the given id or <code>null</code> if no such
   * {@link CompartmentType} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::get(const std::string& sid) const "
/**
   * Get a {@link CompartmentType} from the {@link ListOfCompartmentTypes}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link CompartmentType} to get.
   * <p>
   * @return {@link CompartmentType} in this {@link ListOfCompartmentTypes}
   * with the given id or <code>null</code> if no such
   * {@link CompartmentType} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfCompartmentTypes} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::remove(const std::string& sid) "
/**
   * Removes item in this {@link ListOfCompartmentTypes} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  For example, the
   * {@link ListOfCompartmentTypes} in a model (in SBML Level 2 Version 4) is the
   * third ListOf___.  (However, it differs for different Levels and
   * Versions of SBML, so calling code should not hardwire this number.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfCompartmentTypes::createObject(XMLInputStream& stream) "
/**
   * Create a {@link ListOfCompartmentTypes} object corresponding to the next token
   * in the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) SpeciesType "
/** 
 * LibSBML implementation of SBML Level&nbsp;2\'s SpeciesType construct.
 * <p>
 * The term <em>species</em> <em>type</em> refers to reacting entities independent of
 * location.  These include simple ions (e.g., protons, calcium), simple
 * molecules (e.g., glucose, ATP), large molecules (e.g., RNA,
 * polysaccharides, and proteins), and others.
 * <p>
 * SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 provide an explicit
 * {@link SpeciesType} class of object to enable {@link Species} objects of the same type
 * to be related together.  {@link SpeciesType} is a conceptual construct; the
 * existence of {@link SpeciesType} objects in a model has no effect on the model\'s
 * numerical interpretation.  Except for the requirement for uniqueness of
 * species/species type combinations located in compartments, simulators
 * and other numerical analysis software may ignore {@link SpeciesType} definitions
 * and references to them in a model.
 * <p>
 * There is no mechanism in SBML Level 2 for representing hierarchies of
 * species types.  One {@link SpeciesType} object cannot be the subtype of another
 * {@link SpeciesType} object; SBML provides no means of defining such
 * relationships.
 * <p>
 * As with other major structures in SBML, {@link SpeciesType} has a mandatory
 * attribute, \'id\', used to give the species type an identifier.  The
 * identifier must be a text string conforming to the identifer syntax
 * permitted in SBML.  {@link SpeciesType} also has an optional \'name\' attribute,
 * of type <code>string.</code>  The \'id\' and \'name\' must be used according to the
 * guidelines described in the SBML specification (e.g., Section 3.3 in
 * the Level&nbsp;2 Version&nbsp;4 specification).
 * <p>
 * {@link SpeciesType} was introduced in SBML Level 2 Version 2.  It is not
 * available in SBML Level&nbsp;1 nor in Level&nbsp;3.
 * <p>
 * @see Species
 * @see ListOfSpeciesTypes
 * @see CompartmentType
 * @see ListOfCompartmentTypes
 * <p>
 * <!---------------------------------------------------------------------- -->
 * <p>
 */
"


%typemap(javaimports) ListOfSpeciesTypes "
/** 
 * LibSBML implementation of SBML\'s ListOfSpeciesTypes construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers SpeciesType::SpeciesType(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link SpeciesType} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link SpeciesType}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link SpeciesType}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link SpeciesType} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addSpeciesType(SpeciesType st)}),
   * the SBML Level, SBML Version and XML namespace of the document 
   * <em>override</em> the values used when creating the {@link SpeciesType} object via this
   * constructor.  This is necessary to ensure that an SBML document is a
   * consistent structure.  Nevertheless, the ability to supply the values
   * at the time of creation of a {@link SpeciesType} is an important aid to
   * producing valid SBML.  Knowledge of the intented SBML Level and
   * Version determine whether it is valid to assign a particular value to
   * an attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers SpeciesType::SpeciesType(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link SpeciesType} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link SpeciesType} object is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method {@link SBase#setId(String id)}.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link SpeciesType} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addSpeciesType(SpeciesType st)}),
   * the SBML XML namespace of the document <em>overrides</em> the value used
   * when creating the {@link SpeciesType} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link SpeciesType} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers SpeciesType::SpeciesType(const SpeciesType& orig) "
/**
   * Copy constructor; creates a copy of this {@link SpeciesType}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers SpeciesType::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link SpeciesType}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link SpeciesType} in
   * the list of compartment types.
   */ public
";


%javamethodmodifiers SpeciesType::clone() const "
/**
   * Creates and returns a deep copy of this {@link SpeciesType}.
   * <p>
   * @return a (deep) copy of this {@link SpeciesType}.
   */ public
";


%javamethodmodifiers SpeciesType::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link SpeciesType}.
   * <p>
   * @return the id of this {@link SpeciesType}.
   */ public
";


%javamethodmodifiers SpeciesType::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link SpeciesType}.
   * <p>
   * @return the name of this {@link SpeciesType}.
   */ public
";


%javamethodmodifiers SpeciesType::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link SpeciesType}\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link SpeciesType} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers SpeciesType::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link SpeciesType}\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link SpeciesType} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers SpeciesType::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link SpeciesType}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link SpeciesType}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers SpeciesType::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link SpeciesType}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link SpeciesType}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers SpeciesType::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link SpeciesType}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers SpeciesType::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers SpeciesType::getElementName() const "
/**
   * Returns the XML element name of this object, which for
   * {@link SpeciesType}, is always <code>\'compartmentType\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'compartmentType\'.</code>
   */ public
";


%javamethodmodifiers SpeciesType::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers SpeciesType::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link SpeciesType} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link SpeciesType} object are:
   * <ul>
   * <li> \'id\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers SpeciesType::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers SpeciesType::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * <p>
   * @param attributes the {@link XMLAttributes} to use.
   * @internal
   */ public
";


%javamethodmodifiers SpeciesType::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * <p>
   * @param attributes the {@link XMLAttributes} to use.
   * @internal
   */ public
";


%javamethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * <p>
   * @param stream the XMLOutputStream to use.
   * @internal
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::ListOfSpeciesTypes(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfSpeciesTypes} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::ListOfSpeciesTypes(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfSpeciesTypes} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfSpeciesTypes} object to be created.
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfSpeciesTypes} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfSpeciesTypes}.
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link SpeciesType} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfSpeciesTypes}, the XML element name is 
   * <code>\'listOfSpeciesTypes\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfSpeciesTypes\'.</code>
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::get(unsigned int n) "
/**
   * Get a {@link SpeciesType} from the {@link ListOfSpeciesTypes}.
   * <p>
   * @param n the index number of the {@link SpeciesType} to get.
   * <p>
   * @return the nth {@link SpeciesType} in this {@link ListOfSpeciesTypes}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::get "
/**
   * Get a {@link SpeciesType} from the {@link ListOfSpeciesTypes}.
   * <p>
   * @param n the index number of the {@link SpeciesType} to get.
   * <p>
   * @return the nth {@link SpeciesType} in this {@link ListOfSpeciesTypes}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::get(const std::string& sid) "
/**
   * Get a {@link SpeciesType} from the {@link ListOfSpeciesTypes}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link SpeciesType} to get.
   * <p>
   * @return {@link SpeciesType} in this {@link ListOfSpeciesTypes}
   * with the given id or <code>null</code> if no such
   * {@link SpeciesType} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::get(const std::string& sid) const "
/**
   * Get a {@link SpeciesType} from the {@link ListOfSpeciesTypes}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link SpeciesType} to get.
   * <p>
   * @return {@link SpeciesType} in this {@link ListOfSpeciesTypes}
   * with the given id or <code>null</code> if no such
   * {@link SpeciesType} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfSpeciesTypes} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::remove(const std::string& sid) "
/**
   * Removes item in this {@link ListOfSpeciesTypes} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  For example, the
   * {@link ListOfSpeciesTypes} in a model (in SBML Level 2 Version 4) is the
   * third ListOf___.  (However, it differs for different Levels and
   * Versions of SBML, so calling code should not hardwire this number.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfSpeciesTypes::createObject(XMLInputStream& stream) "
/**
   * Create a {@link ListOfSpeciesTypes} object corresponding to the next token
   * in the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) Compartment "
/** 
 * LibSBML implementation of SBML\'s Compartment construct.
 * <p>
 * A compartment in SBML represents a bounded space in which species are
 * located.  Compartments do not necessarily have to correspond to actual
 * structures inside or outside of a biological cell.
 * <p>
 * It is important to note that although compartments are optional in the
 * overall definition of {@link Model}, every species in an SBML model must be
 * located in a compartment.  This in turn means that if a model defines
 * any species, the model must also define at least one compartment.  The
 * reason is simply that species represent physical things, and therefore
 * must exist <em>somewhere</em>.  Compartments represent the <em>somewhere</em>.
 * <p>
 * {@link Compartment} has one required attribute, \'id\', to give the compartment a
 * unique identifier by which other parts of an SBML model definition can
 * refer to it.  A compartment can also have an optional \'name\' attribute
 * of type <code>string.</code>  Identifiers and names must be used according to the
 * guidelines described in the SBML specifications.
 * <p>
 * {@link Compartment} also has an optional attribute \'spatialDimensions\' that is
 * used to indicate the number of spatial dimensions possessed by the
 * compartment.  Most modeling scenarios involve compartments with integer
 * values of \'spatialDimensions\' of <code>3</code> (i.e., a three-dimensional
 * compartment, which is to say, a volume), or 2 (a two-dimensional
 * compartment, a surface), or <code>1</code> (a one-dimensional compartment, a
 * line).  In SBML Level&nbsp;3, the type of this attribute is <code>double</code>,
 * there are no restrictions on the permitted values of the
 * \'spatialDimensions\' attribute, and there are no default values.  In SBML
 * Level&nbsp;2, the value must be a positive <code>integer</code>, and the default
 * value is <code>3</code>; the permissible values in SBML Level&nbsp;2 are <code>3</code>, 
 * <code>2</code>, <code>1</code>, and <code>0</code> (for a point).
 * <p>
 * Another optional attribute on {@link Compartment} is \'size\', representing the
 * <em>initial</em> total size of that compartment in the model.  The \'size\'
 * attribute must be a floating-point value and may represent a volume (if
 * the compartment is a three-dimensional one), or an area (if the
 * compartment is two-dimensional), or a length (if the compartment is
 * one-dimensional).  There is no default value of compartment size in SBML
 * Level&nbsp;2 or Level&nbsp;3.  In particular, a missing \'size\' value
 * <em>does not imply that the compartment size is 1</em>.  (This is unlike
 * the definition of compartment \'volume\' in SBML Level&nbsp;1.)  When the
 * compartment\'s \'spatialDimensions\' attribute does not have a value of 
 * <code>0</code>, a missing value of \'size\' for a given compartment signifies that the
 * value either is unknown, or to be obtained from an external source, or
 * determined by an {@link InitialAssignment}, {@link AssignmentRule}, {@link AlgebraicRule} or
 * {@link RateRule} elsewhere in the model.  In SBML Level&nbsp;2, there are
 * additional special requirements on the values of \'size\'; we discuss them
 * in a <a href=\'#comp-l2\'>separate section below</a>.
 * <p>
 * The units associated with a compartment\'s \'size\' attribute value may be
 * set using the optional attribute \'units\'.  The rules for setting and
 * using compartment size units differ between SBML Level&nbsp;2 and
 * Level&nbsp;3, and are discussed separately below.
 * <p>
 * Finally, the optional {@link Compartment} attribute named \'constant\' is used to
 * indicate whether the compartment\'s size stays constant after simulation
 * begins.  A value of <code>true</code> indicates the compartment\'s \'size\' cannot be
 * changed by any other construct except {@link InitialAssignment}; a value of 
 * <code>false</code> indicates the compartment\'s \'size\' can be changed by other
 * constructs in SBML.  In SBML Level&nbsp;2, there is an additional
 * explicit restriction that if \'spatialDimensions\'=<code>\'0\'</code>, the value
 * cannot be changed by {@link InitialAssignment} either.  Further, in
 * Level&nbsp;2, \'constant\' has a default value of <code>true.</code>  In SBML
 * Level&nbsp;3, there is no default value for the \'constant\' attribute.
 * <p>
 * <p>
 * <h2>Additional considerations in SBML Level&nbsp;2</h2>
 * <p>
 * In SBML Level&nbsp;2, the default units of compartment size, and the
 * kinds of units allowed as values of the attribute \'units\', interact with
 * the number of spatial dimensions of the compartment.  The value of the
 * \'units\' attribute of a {@link Compartment} object must be one of the base units
 * (see {@link Unit}), or the predefined unit identifiers <code>volume</code>, <code>area</code>, 
 * <code>length</code> or <code>dimensionless</code>, or a new unit defined by a {@link UnitDefinition}
 * object in the enclosing {@link Model}, subject to the restrictions detailed in
 * the following table:
 * <p>
 * <center style=\'padding-bottom: 0.5em\'>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
<caption class=\'top-caption\'>Restrictions on values permitted for
compartment <code>size</code> and <code>units</code> attributes.</caption>
 <tr>
     <th align=\'left\' valign=\'bottom\'>
       Value of<br><code>spatialDimensions</code>
     </th>
     <th align=\'left\' valign=\'bottom\'>
       <code>size</code><br>allowed?
     </th>
     <th align=\'left\' valign=\'bottom\'>
       <code>units</code><br>allowed?
     </th>
     <th align=\'left\' valign=\'bottom\'>
	  Allowable kinds of units
     </th>
     <th align=\'left\' valign=\'bottom\'>
	  Default value of attribute <code>units</code>
     </th>
 </tr>
 <tr>
     <td>\'<code>3</code>\'</td>
     <td>yes</td>
     <td>yes</td>
     <td>units of volume, or <code>dimensionless</code></td>
     <td>\'<code>volume</code>\'</td>
 </tr>
 <tr>
     <td>\'<code>2</code>\'</td>
     <td>yes</td>
     <td>yes</td>
     <td>units of area, or <code>dimensionless</code></td>
     <td>\'<code>area</code>\'</td>
 </tr>
 <tr>
     <td>\'<code>1</code>\'</td>
     <td>yes</td>
     <td>yes</td>
     <td>units of length, or <code>dimensionless</code></td>
     <td>\'<code>length</code>\'</td>
 </tr>
 <tr>
     <td>\'<code>0</code>\'</td>
     <td>no</td>
     <td>no</td>
     <td>(no units allowed)</td>
     <td></td>
 </tr>
 </tr>
</table>
</center>
 
 * <p>
 * In SBML Level&nbsp;2, the units of the compartment size, as defined by the
 * \'units\' attribute or (if \'units\' is not set) the default value listed in
 * the table above, are used in the following ways when the compartment has
 * a \'spatialDimensions\' value greater than <code>0:</code>
 * <ul>
 * <li> The value of the \'units\' attribute is used as the units of the
 * compartment identifier when the identifier appears as a numerical
 * quantity in a mathematical formula expressed in MathML.
 * <p>
 * <li> The <code>math</code> element of an {@link AssignmentRule} or {@link InitialAssignment}
 * referring to this compartment must have identical units.
 * <p>
 * <li> In {@link RateRule} objects that set the rate of change of the compartment\'s
 * size, the units of the rule\'s <code>math</code> element must be identical to the
 * compartment\'s \'units\' attribute divided by the default <em>time</em> units.
 * (In other words, the units for the rate of change of compartment size
 * are <em>compartment size</em>/<em>time</em> units.
 * <p>
 * <li> When a {@link Species} is to be treated in terms of concentrations or
 * density, the units of the spatial size portion of the concentration
 * value (i.e., the denominator in the units formula <em>substance</em>/
 * <em>size</em>) are those indicated by the value of the \'units\' attribute on the
 * compartment in which the species is located.
 * </ul>
 * <p>
 * Compartments with \'spatialDimensions\'=<code>0</code> require special treatment in
 * this framework.  As implied above, the \'size\' attribute must not have a
 * value on an SBML Level&nbsp;2 {@link Compartment} object if the
 * \'spatialDimensions\' attribute has a value of <code>0.</code>  An additional
 * related restriction is that the \'constant\' attribute must default to or
 * be set to <code>true</code> if the value of the \'spatialDimensions\' attribute is
 * <code>0</code>, because a zero-dimensional compartment cannot ever have a size.
 * <p>
 * If a compartment has no size or dimensional units, how should such a
 * compartment\'s identifier be interpreted when it appears in mathematical
 * formulas?  The answer is that such a compartment\'s identifier should not
 * appear in mathematical formulas in the first place&mdash;it has no
 * value, and its value cannot change.  Note also that a zero-dimensional
 * compartment is a point, and species located at points can only be
 * described in terms of amounts, not spatially-dependent measures such as
 * concentration.  Since SBML {@link KineticLaw} formulas are already in terms of
 * <em>substance</em>/<em>time</em> and not (say) <em>concentration</em>/<em>time</em>, volume
 * or other factors in principle are not needed for species located in
 * zero-dimensional compartments.
 * <p>
 * Finally, in SBML Level&nbsp;2 Versions 2&ndash;4, each compartment in a
 * model may optionally be designated as belonging to a particular
 * compartment <em>type</em>.  The optional attribute \'compartmentType\' is used
 * identify the compartment type represented by the {@link Compartment} structure.
 * The \'compartmentType\' attribute\'s value must be the identifier of a
 * {@link CompartmentType} instance defined in the model.  If the \'compartmentType\'
 * attribute is not present on a particular compartment definition, a
 * unique virtual compartment type is assumed for that compartment, and no
 * other compartment can belong to that compartment type.  The values of
 * \'compartmentType\' attributes on compartments have no effect on the
 * numerical interpretation of a model.  Simulators and other numerical
 * analysis software may ignore \'compartmentType\' attributes.  The
 * \'compartmentType\' attribute and the {@link CompartmentType} class of objects are
 * not present in SBML Level&nbsp;3 Core nor in SBML Level&nbsp;1.
 * <p>
 * <p>
 * <h2>Additional considerations in SBML Level&nbsp;3</h2>
 * <p>
 * One difference between SBML Level&nbsp;3 and lower Levels of SBML is
 * that there are no restrictions on the permissible values of the
 * \'spatialDimensions\' attribute, and there is no default value defined for
 * the attribute.  The value of \'spatialDimensions\' does not have to be an
 * integer, either; this is to allow for the possibility of representing
 * structures with fractal dimensions.
 * <p>
 * The number of spatial dimensions possessed by a compartment cannot enter
 * into mathematical formulas, and therefore cannot directly alter the
 * numerical interpretation of a model.  However, the value of
 * \'spatialDimensions\' <em>does</em> affect the interpretation of the units
 * associated with a compartment\'s size.  Specifically, the value of
 * \'spatialDimensions\' is used to select among the {@link Model} attributes
 * \'volumeUnits\', \'areaUnits\' and \'lengthUnits\' when a {@link Compartment} object
 * does not define a value for its \'units\' attribute.
 * <p>
 * The \'units\' attribute may be left unspecified for a given compartment in
 * a model; in that case, the compartment inherits the unit of measurement
 * specified by one of the attributes on the enclosing {@link Model} object
 * instance.  The applicable attribute on {@link Model} depends on the value of the
 * compartment\'s \'spatialDimensions\' attribute; the relationship is shown
 * in the table below.  If the {@link Model} object does not define the relevant
 * attribute (\'volumeUnits\', \'areaUnits\' or \'lengthUnits\') for a given
 * \'spatialDimensions\' value, the unit associated with that {@link Compartment}
 * object\'s size is undefined.  If <em>both</em> \'spatialDimensions\' and \'units\'
 * are left unset on a given {@link Compartment} object instance, then no unit can
 * be chosen from among the {@link Model}\'s \'volumeUnits\', \'areaUnits\' or
 * \'lengthUnits\' attributes (even if the {@link Model} instance provides values for
 * those attributes), because there is no basis to select between them and
 * there is no default value of \'spatialDimensions\'.  Leaving the units of
 * compartments\' sizes undefined in an SBML model does not render the model
 * invalid; however, as a matter of best practice, we strongly recommend
 * that all models specify the units of measurement for all compartment
 * sizes.
 * <p>
 * <center style=\'padding-bottom: 0.5em\'>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
<caption class=\'top-caption\'>Interpretation of the {@link Compartment} \'units\' attribute.</caption>
 <tr>
     <th align=\'left\' valign=\'bottom\'>
       Value of attribute<br>\'spatialDimensions\'
     </th>
     <th align=\'left\' valign=\'bottom\'>
       Attribute of {@link Model} used<br>for inheriting the unit
     </th>
     <th align=\'left\' valign=\'bottom\'>
       Recommended candidate units
     </th>
 </tr>
 <tr>
     <td>\'<code>3</code>\'</td>
     <td>\'volumeUnits\'</td>
     <td>units of volume, or <code>dimensionless</code></td>
 </tr>
 <tr>
     <td>\'<code>2</code>\'</td>
     <td>\'areaUnits\'</td>
     <td>units of area, or <code>dimensionless</code></td>
 </tr>
 <tr>
     <td>\'<code>1</code>\'</td>
     <td>\'lengthUnits\'</td>
     <td>units of length, or <code>dimensionless</code></td>
 </tr>
 <tr>
     <td><em>other</em></td>
     <td><em>no units inherited</em></td>
     <td><em>no specific recommendations</em></td>
 </tr>
 </tr>
</table>
</center>

 * <p>
 * The unit of measurement associated with a compartment\'s size, as defined
 * by the \'units\' attribute or (if \'units\' is not set) the inherited value
 * from {@link Model} according to the table above, is used in the following ways:
 * <p>
 * <ul>
 * <p>
 * <li> When the identifier of the compartment appears as a numerical
 * quantity in a mathematical formula expressed in MathML, it represents
 * the size of the compartment, and the unit associated with the size is
 * the value of the \'units\' attribute.
 * <p>
 * <li> When a {@link Species} is to be treated in terms of concentrations or
 * density, the unit associated with the spatial size portion of the
 * concentration value (i.e., the denominator in the formula
 * <em>amount</em>/<em>size</em>) is specified by the value of the \'units\'
 * attribute on the compartment in which the species is located.
 * <p>
 * <li> The \'math\' elements of {@link AssignmentRule}, {@link InitialAssignment} and
 * {@link EventAssignment} objects setting the value of the compartment size
 * should all have the same units as the unit associated with the
 * compartment\'s size.
 * <p>
 * <li> In a {@link RateRule} object that defines a rate of change for a
 * compartment\'s size, the unit of the rule\'s \'math\' element should be
 * identical to the compartment\'s \'units\' attribute divided by the
 * model-wide unit of <em>time</em>.  (In other words, {<em>unit of
 * compartment size</em>}/{<em>unit of time</em>}.)
 * <p>
 * </ul>
 * <p>
 * <p>
 * <h2>Other aspects of {@link Compartment}</h2>
 * <p>
 * In SBML Level&nbsp;1 and Level&nbsp;2, {@link Compartment} has an optional
 * attribute named \'outside\', whose value can be the identifier of another
 * {@link Compartment} object defined in the enclosing {@link Model} object.  Doing so
 * means that the other compartment contains it or is outside of it.  This
 * enables the representation of simple topological relationships between
 * compartments, for those simulation systems that can make use of the
 * information (e.g., for drawing simple diagrams of compartments).  It is
 * worth noting that in SBML, there is no relationship between compartment
 * sizes when compartment positioning is expressed using the \'outside\'
 * attribute.  The size of a given compartment does not in any sense
 * include the sizes of other compartments having it as the value of their
 * \'outside\' attributes.  In other words, if a compartment <em>B</em> has the
 * identifier of compartment <em>A</em> as its \'outside\' attribute value, the
 * size of <em>A</em> does not include the size of <em>B</em>.  The compartment sizes
 * are separate.
 * <p>
 * In Level&nbsp;2, there are two restrictions on the \'outside\' attribute.
 * First, because a compartment with \'spatialDimensions\' of <code>0</code> has no
 * size, such a compartment cannot act as the container of any other
 * compartment <em>except</em> compartments that <em>also</em> have
 * \'spatialDimensions\' values of <code>0.</code>  Second, the directed graph formed
 * by representing {@link Compartment} structures as vertexes and the \'outside\'
 * attribute values as edges must be acyclic.  The latter condition is
 * imposed to prevent a compartment from being contained inside itself.  In
 * the absence of a value for \'outside\', compartment definitions in SBML
 * Level&nbsp;2 do not have any implied spatial relationships between each
 * other.
 * <p>
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%typemap(javaimports) ListOfCompartments "
/** 
 * LibSBML implementation of SBML Level&nbsp;2\'s ListOfCompartments construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers Compartment::Compartment(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link Compartment} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Compartment}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Compartment}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Compartment} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addCompartment(Compartment c)}), the SBML Level, SBML Version
   * and XML namespace of the document <em>override</em> the values used
   * when creating the {@link Compartment} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link Compartment} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Compartment::Compartment(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link Compartment} using the given {@link SBMLNamespaces} object 
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link Compartment} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method {@link Compartment#setId(String id)}.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Compartment} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addCompartment(Compartment c)}), the SBML XML namespace of the
   * document <em>overrides</em> the value used when creating the {@link Compartment}
   * object via this constructor.  This is necessary to ensure that an SBML
   * document is a consistent structure.  Nevertheless, the ability to
   * supply the values at the time of creation of a {@link Compartment} is an
   * important aid to producing valid SBML.  Knowledge of the intented SBML
   * Level and Version determine whether it is valid to assign a particular
   * value to an attribute, or whether it is valid to add an object to an
   * existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Compartment::Compartment(const Compartment& orig) "
/**
   * Copy constructor; creates a copy of a {@link Compartment}.
   * <p>
   * @param orig the {@link Compartment} instance to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers Compartment::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Compartment}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link Compartment} in the
   * list of compartments within which this {@link Compartment} is embedded (i.e.,
   * the {@link ListOfCompartments} in the parent {@link Model}).
   */ public
";


%javamethodmodifiers Compartment::clone() const "
/**
   * Creates and returns a deep copy of this {@link Compartment} object.
   * <p>
   * @return a (deep) copy of this {@link Compartment}.
   */ public
";


%javamethodmodifiers Compartment::initDefaults "
/**
   * Initializes the fields of this {@link Compartment} object to \'typical\' default
   * values.
   * <p>
   * The SBML {@link Compartment} component has slightly different aspects and
   * default attribute values in different SBML Levels and Versions.
   * This method sets the values to certain common defaults, based
   * mostly on what they are in SBML Level&nbsp;2.  Specifically:
   * <ul>
   * <li> Sets attribute \'spatialDimensions\' to <code>3</code>
   * <li> Sets attribute \'constant\' to <code>true</code>
   * <li> (Applies to Level&nbsp;1 models only) Sets attribute \'volume\' to <code>1.0</code>
   * <li> (Applies to Level&nbsp;3 models only) Sets attribute \'units\' to <code>litre</code>
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link Compartment} object.
   * <p>
   * @return the id of this {@link Compartment}.
   */ public
";


%javamethodmodifiers Compartment::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link Compartment} object.
   * <p>
   * @return the name of this {@link Compartment}.
   */ public
";


%javamethodmodifiers Compartment::getCompartmentType() const "
/**
   * Get the value of the \'compartmentType\' attribute of this {@link Compartment}
   * object.
   * <p>
   * @return the value of the \'compartmentType\' attribute of this
   * {@link Compartment} as a string.
   * <p>
   * @note The \'compartmentType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%javamethodmodifiers Compartment::getSpatialDimensions() const "
/**
   * Get the number of spatial dimensions of this {@link Compartment} object.
   * <p>
   * @note In SBML Level&nbsp;3, the data type of the \'spatialDimensions\'
   * attribute is <code>double</code>, whereas in Level&nbsp;2, it is <code>integer.</code>
   * LibSBML provides a separate method for obtaining the value as a double,
   * for models where it is relevant.
   * <p>
   * @return the value of the \'spatialDimensions\' attribute of this
   * {@link Compartment} as a long integereger
   * <p>
   * @see #getSpatialDimensionsAsDouble()
   */ public
";


%javamethodmodifiers Compartment::getSpatialDimensionsAsDouble() const "
/**
   * Get the number of spatial dimensions of this {@link Compartment} object
   * as a double.
   * <p>
   * @note In SBML Level&nbsp;3, the data type of the \'spatialDimensions\'
   * attribute is <code>double</code>, whereas in Level&nbsp;2, it is <code>integer.</code>  To
   * avoid backward compatibility issues, libSBML provides a separate
   * method for obtaining the value as a double, for models where it is
   * relevant.
   * <p>
   * @return the value of the \'spatialDimensions\' attribute of this
   * {@link Compartment} as a double, or <code>NaN</code> if this model is not in SBML
   * Level&nbsp;3 format.
   * <p>
   * @see #getSpatialDimensions()
   */ public
";


%javamethodmodifiers Compartment::getSize() const "
/**
   * Get the size of this {@link Compartment}.
   * <p>
   * This method is identical to
   * {@link Compartment#getVolume()}.
   * In SBML Level&nbsp;1, compartments are always three-dimensional
   * constructs and only have volumes, whereas in SBML Level&nbsp;2,
   * compartments may be other than three-dimensional and therefore the
   * \'volume\' attribute is named \'size\' in Level&nbsp;2.  LibSBML provides
   * both
   * {@link Compartment#getSize()} and
   * {@link Compartment#getVolume()} for
   * easier compatibility between SBML Levels.
   * <p>
   * @return the value of the \'size\' attribute (\'volume\' in Level&nbsp;1) of
   * this {@link Compartment} as a float-point number.
   * <p>
   * @see #isSetSize()
   * @see #getVolume()
   */ public
";


%javamethodmodifiers Compartment::getVolume() const "
/**
   * Get the volume of this {@link Compartment}.
   * <p>
   * This method is identical to
   * {@link Compartment#getSize()}.  In
   * SBML Level&nbsp;1, compartments are always three-dimensional
   * constructs and only have volumes, whereas in SBML Level&nbsp;2,
   * compartments may be other than three-dimensional and therefore the
   * \'volume\' attribute is named \'size\' in Level&nbsp;2.  LibSBML provides
   * both
   * {@link Compartment#getSize()} and
   * {@link Compartment#getVolume()}
   * for easier compatibility between SBML Levels.
   * <p>
   * @return the value of the \'volume\' attribute (\'size\' in Level&nbsp;2) of
   * this {@link Compartment}, as a floating-point number.
   * <p>
   * @note The attribute \'volume\' only exists by that name in SBML
   * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
   * named \'size\'.
   * <p>
   * @see #isSetVolume()
   * @see #getSize()
   */ public
";


%javamethodmodifiers Compartment::getUnits() const "
/**
   * Get the units of this compartment\'s size.
   * <p>
   * The value of an SBML compartment\'s \'units\' attribute establishes the
   * unit of measurement associated with the compartment\'s size.
   * <p>
   * @return the value of the \'units\' attribute of this {@link Compartment}, as a
   * string.  An empty string indicates that no units have been assigned to
   * the value of the size.
   * <p>
   * @note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

   * <p>
   * @see #isSetUnits()
   * @see Compartment#setUnits(String sid)
   * @see #getSize()
   */ public
";


%javamethodmodifiers Compartment::getOutside() const "
/**
   * Get the identifier, if any, of the compartment that is designated
   * as being outside of this one.
   * <p>
   * @return the value of the \'outside\' attribute of this {@link Compartment}.
   * <p>
   * @note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
   * Level&nbsp;2, but does not exist in SBML Level&nbsp;3 Version&nbsp;1
   * Core.
   */ public
";


%javamethodmodifiers Compartment::getConstant() const "
/**
   * Get the value of the \'constant\' attribute of this {@link Compartment}.
   * <p>
   * @return <code>true</code> if this {@link Compartment}\'s size is flagged as being
   * constant, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Compartment::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Compartment}\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Compartment} is 
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Compartment::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Compartment}\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Compartment} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Compartment::isSetCompartmentType() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Compartment}\'s \'compartmentType\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'compartmentType\' attribute of this {@link Compartment}
   * is set, <code>false</code> otherwise.
   * <p>
   * @note The \'compartmentType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%javamethodmodifiers Compartment::isSetSize() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Compartment}\'s \'size\' attribute is set.
   * <p>
   * This method is similar but not identical to
   * {@link Compartment#isSetVolume()}.  The latter
   * should be used in the context of SBML Level&nbsp;1 models instead of
   * {@link Compartment#isSetSize()}
   * because {@link Compartment#isSetVolume()}
   * performs extra processing to take into account the difference in
   * default values between SBML Levels 1 and 2.
   * <p>
   * @return <code>true</code> if the \'size\' attribute (\'volume\' in Level&nbsp;2) of
   * this {@link Compartment} is set, <code>false</code> otherwise.
   * <p>
   * @see #isSetVolume()
   * @see #setSize(double value)
   */ public
";


%javamethodmodifiers Compartment::isSetVolume() const "
/**
   * Predicate returning <code>true</code> if this {@link Compartment}\'s
   * \'volume\' attribute is set.
   * <p>
   * This method is similar but not identical to
   * {@link Compartment#isSetSize()}.  The latter
   * should not be used in the context of SBML Level&nbsp;1 models because this
   * method performs extra processing to take into account
   * the difference in default values between SBML Levels 1 and 2.
   * <p>
   * @return <code>true</code> if the \'volume\' attribute (\'size\' in Level&nbsp;2 and
   * above) of this {@link Compartment} is set, <code>false</code> otherwise.
   * <p>
   * @note The attribute \'volume\' only exists by that name in SBML
   * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
   * named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a
   * default value (<code>1.0</code>) and therefore this method will always return 
   * <code>true.</code>  In Level 2, a compartment\'s size (the equivalent of SBML
   * Level&nbsp;1\'s \'volume\') is optional and has no default value, and
   * therefore may or may not be set.
   * <p>
   * @see #isSetSize()
   * @see Compartment#setVolume(double value)
   */ public
";


%javamethodmodifiers Compartment::isSetUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Compartment}\'s \'units\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'units\' attribute of this {@link Compartment} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

   */ public
";


%javamethodmodifiers Compartment::isSetOutside() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Compartment}\'s \'outside\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'outside\' attribute of this {@link Compartment} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
   * Level&nbsp;2, but does not exist in SBML Level&nbsp;3 Version&nbsp;1
   * Core.
   */ public
";


%javamethodmodifiers Compartment::isSetSpatialDimensions() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Compartment}\'s \'spatialDimensions\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'spatialDimensions\' attribute of this
   * {@link Compartment} is set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Compartment::isSetConstant() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Compartment}\'s \'constant\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'constant\' attribute of this {@link Compartment} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Compartment::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link Compartment}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link Compartment}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link Compartment}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link Compartment}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::setCompartmentType(const std::string& sid) "
/**
   * Sets the \'compartmentType\' attribute of this {@link Compartment}.
   * <p>
   * @param sid the identifier of a {@link CompartmentType} object defined
   * elsewhere in this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note The \'compartmentType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%javamethodmodifiers Compartment::setSpatialDimensions(unsigned int value) "
/**
   * Sets the \'spatialDimensions\' attribute of this {@link Compartment}.
   * <p>
   * If <code>value</code> is not one of <code>0</code>, <code>1</code>, <code>2</code>, or <code>3</code>, this method will
   * have no effect (i.e., the \'spatialDimensions\' attribute will not be
   * set).
   * <p>
   * @param value a long integereger indicating the number of dimensions
   * of this compartment.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::setSpatialDimensions(double value) "
/**
   * Sets the \'spatialDimensions\' attribute of this {@link Compartment} as a double.
   * <p>
   * @param value a double indicating the number of dimensions
   * of this compartment.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::setSize(double value) "
/**
   * Sets the \'size\' attribute (or \'volume\' in SBML Level&nbsp;1) of this
   * {@link Compartment}.
   * <p>
   * This method is identical to
   * {@link Compartment#setVolume(double value)}
   * and is provided for compatibility between
   * SBML Level&nbsp;1 and Level&nbsp;2.
   * <p>
   * @param value a <code>double</code> representing the size of this compartment
   * instance in whatever units are in effect for the compartment.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @note The attribute \'volume\' only exists by that name in SBML
   * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
   * named \'size\'.
   */ public
";


%javamethodmodifiers Compartment::setVolume(double value) "
/**
   * Sets the \'volume\' attribute (or \'size\' in SBML Level&nbsp;2) of this
   * {@link Compartment}.
   * <p>
   * This method is identical to
   * {@link Compartment#setVolume(double value)}
   * and is provided for compatibility between SBML Level&nbsp;1 and
   * Level&nbsp;2.
   * <p>
   * @param value a <code>double</code> representing the volume of this compartment
   * instance in whatever units are in effect for the compartment.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @note The attribute \'volume\' only exists by that name in SBML
   * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
   * named \'size\'.
   */ public
";


%javamethodmodifiers Compartment::setUnits(const std::string& sid) "
/**
   * Sets the \'units\' attribute of this {@link Compartment}.
   * <p>
   * @param sid the identifier of the defined units to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::setOutside(const std::string& sid) "
/**
   * Sets the \'outside\' attribute of this {@link Compartment}.
   * <p>
   * @param sid the identifier of a compartment that encloses this one.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
   * Level&nbsp;2, but does not exist in SBML Level&nbsp;3 Version&nbsp;1
   * Core.
   */ public
";


%javamethodmodifiers Compartment::setConstant(bool value) "
/**
   * Sets the value of the \'constant\' attribute of this {@link Compartment}.
   * <p>
   * @param value a boolean indicating whether the size/volume of this
   * compartment should be considered constant (<code>true</code>) or variable
   * (<code>false</code>)
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%javamethodmodifiers Compartment::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers Compartment::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link Compartment}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::unsetCompartmentType "
/**
   * Unsets the value of the \'compartmentType\'
   * attribute of this {@link Compartment}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note The \'compartmentType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   * <p>
   * @see #setCompartmentType(String sid)
   * @see #isSetCompartmentType()
   */ public
";


%javamethodmodifiers Compartment::unsetSize "
/**
   * Unsets the value of the \'size\' attribute of this {@link Compartment}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The attribute \'volume\' only exists by that name in SBML
   * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
   * named \'size\'.
   */ public
";


%javamethodmodifiers Compartment::unsetVolume "
/**
   * Unsets the value of the \'volume\' attribute of this
   * {@link Compartment}.
   * <p>
   * In SBML Level&nbsp;1, a {@link Compartment} volume has a default value (<code>1.0</code>) and
   * therefore <em>should always be set</em>.  In Level&nbsp;2, \'size\' is
   * optional with no default value and as such may or may not be set.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The attribute \'volume\' only exists by that name in SBML
   * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
   * named \'size\'.
   */ public
";


%javamethodmodifiers Compartment::unsetUnits "
/**
   * Unsets the value of the \'units\' attribute of this {@link Compartment}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Compartment::unsetOutside "
/**
   * Unsets the value of the \'outside\' attribute of this {@link Compartment}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
   * Level&nbsp;2, but does not exist in SBML Level&nbsp;3 Version&nbsp;1
   * Core.
   */ public
";


%javamethodmodifiers Compartment::unsetSpatialDimensions "
/**
   * Unsets the value of the \'spatialDimensions\' attribute of this {@link Compartment}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note This function is only valid for SBML Level&nbsp;3.
   */ public
";


%javamethodmodifiers Compartment::getDerivedUnitDefinition "
/**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Compartment}\'s designated size.
   * <p>
   * Compartments in SBML have an attribute (\'units\') for declaring the
   * units of measurement intended for the value of the compartment\'s size.
   * In the absence of a value given for this attribute, the units are
   * inherited from values either defined on the enclosing {@link Model} (in SBML
   * Level&nbsp;3) or in defaults (in SBML Level&nbsp;2).  This method
   * returns a {@link UnitDefinition} object based on how this compartment\'s units
   * are interpreted according to the relevant SBML guidelines, or it
   * returns <code>null</code> if no units have been declared and no defaults are
   * defined by the relevant SBML specification.
   * <p>
   * Note that unit declarations for {@link Compartment} objects are specified in
   * terms of the <em>identifier</em> of a unit (e.g., using
   * {@link Compartment#setUnits(String sid)}), but
   * <em>this</em> method returns a {@link UnitDefinition} object, not a unit
   * identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  For SBML Level&nbsp;2 models, it will do this even
   * when the value of the \'units\' attribute is one of the special SBML
   * Level&nbsp;2 unit identifiers <code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>,
   * <code>\'length\'</code> or <code>\'time\'.</code>  Callers may find this useful in conjunction
   * with the helper methods provided by the {@link UnitDefinition} class for
   * comparing different {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Compartment}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @note The libSBML system for unit analysis depends on the model as a
   * whole.  In cases where the {@link Compartment} object has not yet been added
   * to a model, or the model itself is incomplete, unit analysis is not
   * possible, and consequently this method will return <code>null.</code>
   * <p>
   * @see #isSetUnits()
   * @see #getUnits()
   */ public
";


%javamethodmodifiers Compartment::getDerivedUnitDefinition() const "
/**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Compartment}\'s designated size.
   * <p>
   * Compartments in SBML have an attribute (\'units\') for declaring the
   * units of measurement intended for the value of the compartment\'s size.
   * In the absence of a value given for this attribute, the units are
   * inherited from values either defined on the enclosing {@link Model} (in SBML
   * Level&nbsp;3) or in defaults (in SBML Level&nbsp;2).  This method
   * returns a {@link UnitDefinition} object based on how this compartment\'s units
   * are interpreted according to the relevant SBML guidelines, or it
   * returns <code>null</code> if no units have been declared and no defaults are
   * defined by the relevant SBML specification.
   * <p>
   * Note that unit declarations for {@link Compartment} objects are specified in
   * terms of the <em>identifier</em> of a unit (e.g., using setUnits(String sid)), but
   * <em>this</em> method returns a {@link UnitDefinition} object, not a unit
   * identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  For SBML Level&nbsp;2 models, it will do this even
   * when the value of the \'units\' attribute is one of the special SBML
   * Level&nbsp;2 unit identifiers <code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>,
   * <code>\'length\'</code> or <code>\'time\'.</code>  Callers may find this useful in conjunction
   * with the helper methods provided by the {@link UnitDefinition} class for
   * comparing different {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Compartment}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @note The libSBML system for unit analysis depends on the model as a
   * whole.  In cases where the {@link Compartment} object has not yet been added
   * to a model, or the model itself is incomplete, unit analysis is not
   * possible, and consequently this method will return <code>null.</code>
   * <p>
   * @see #isSetUnits()
   * @see #getUnits()
   */ public
";


%javamethodmodifiers Compartment::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or
   * {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers Compartment::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link Compartment}, is
   * always <code>\'compartment\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'compartment\'.</code>
   */ public
";


%javamethodmodifiers Compartment::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Compartment::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Compartment} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Compartment} object are:
   * <ul>
   * <li> \'id\' (or \'name\' in SBML Level&nbsp;1)
   * <li> \'constant\' (in SBML Level&nbsp;3 only)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers Compartment::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers Compartment::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Compartment::readL1Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Compartment::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Compartment::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Compartment::isExplicitlySetSpatialDimensions() const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Compartment::isExplicitlySetConstant() const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfCompartments::ListOfCompartments(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfCompartments} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfCompartments::ListOfCompartments(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfCompartments} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfCompartments} object to be created.
   */ public
";


%javamethodmodifiers ListOfCompartments::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfCompartments} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfCompartments}.
   */ public
";


%javamethodmodifiers ListOfCompartments::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfCompartments::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Compartment} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfCompartments::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfCompartments}, the XML element name is <code>\'listOfCompartments\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfCompartments\'.</code>
   */ public
";


%javamethodmodifiers ListOfCompartments::get(unsigned int n) "
/**
   * Get a {@link Compartment} from the {@link ListOfCompartments}.
   * <p>
   * @param n the index number of the {@link Compartment} to get.
   * <p>
   * @return the nth {@link Compartment} in this {@link ListOfCompartments}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartments::get "
/**
   * Get a {@link Compartment} from the {@link ListOfCompartments}.
   * <p>
   * @param n the index number of the {@link Compartment} to get.
   * <p>
   * @return the nth {@link Compartment} in this {@link ListOfCompartments}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartments::get(const std::string& sid) "
/**
   * Get a {@link Compartment} from the {@link ListOfCompartments}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Compartment} to get.
   * <p>
   * @return {@link Compartment} in this {@link ListOfCompartments}
   * with the given id or <code>null</code> if no such
   * {@link Compartment} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartments::get(const std::string& sid) const "
/**
   * Get a {@link Compartment} from the {@link ListOfCompartments}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Compartment} to get.
   * <p>
   * @return {@link Compartment} in this {@link ListOfCompartments}
   * with the given id or <code>null</code> if no such
   * {@link Compartment} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartments::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfCompartments} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfCompartments::remove(const std::string& sid) "
/**
   * Removes item in this {@link ListOfCompartments} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOfCompartments::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the {@link ListOfCompartments}
   * in a model is (in SBML Level&nbsp;2 Version&nbsp;4) the fifth
   * ListOf___.  (However, it differs for different Levels and Versions of
   * SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfCompartments::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) Species "
/** 
 * LibSBML implementation of SBML\'s Species construct.
 * <p>
 * A <em>species</em> in SBML refers to a pool of entities that (a) are
 * considered indistinguishable from each other for the purposes of the
 * model, (b) participate in reactions, and (c) are located in a specific
 * <em>compartment</em>.  The SBML {@link Species} object class is intended to represent
 * these pools.
 * <p>
 * As with other major constructs in SBML, {@link Species} has a mandatory
 * attribute, \'id\', used to give the species type an identifier in the
 * model.  The identifier must be a text string conforming to the identifer
 * syntax permitted in SBML.  {@link Species} also has an optional \'name\'
 * attribute, of type <code>string.</code>  The \'id\' and \'name\' must be used
 * according to the guidelines described in the SBML specifications.
 * <p>
 * The required attribute \'compartment\' is used to identify the compartment
 * in which the species is located.  The attribute\'s value must be the
 * identifier of an existing {@link Compartment} object.  It is important to note
 * that there is no default value for the \'compartment\' attribute on
 * {@link Species}; every species in an SBML model must be assigned a compartment
 * <em>explicitly</em>.  (This also implies that every model with one or more
 * {@link Species} objects must define at least one {@link Compartment} object.)
 * <p>
 * <p>
 * <h2>The initial amount and concentration of a species</h2>
 * <p>
 * The optional attributes \'initialAmount\' and \'initialConcentration\', both
 * having a data type of <code>double</code>, can be used to set the <em>initial</em>
 * quantity of the species in the compartment where the species is located.
 * These attributes are mutually exclusive; i.e., <em>only one</em> can
 * have a value on any given instance of a {@link Species} object.  Missing
 * \'initialAmount\' and \'initialConcentration\' values implies that their
 * values either are unknown, or to be obtained from an external source, or
 * determined by an {@link InitialAssignment} or other SBML construct elsewhere in
 * the model.
 * <p>
 * A species\' initial quantity in SBML is set by the \'initialAmount\' or
 * \'initialConcentration\' attribute exactly once.  If the \'constant\'
 * attribute is <code>true</code>, then the value of the species\' quantity is fixed
 * and cannot be changed except by an {@link InitialAssignment}.  These methods
 * differ in that the \'initialAmount\' and \'initialConcentration\' attributes
 * can only be used to set the species quantity to a literal floating-point
 * number, whereas the use of an {@link InitialAssignment} object allows the value
 * to be set using an arbitrary mathematical expression (which, thanks to
 * MathML\'s expressiveness, may evaluate to a rational number).  If the
 * species\' \'constant\' attribute is <code>false</code>, the species\' quantity value
 * may be overridden by an {@link InitialAssignment} or changed by {@link AssignmentRule}
 * or {@link AlgebraicRule}, and in addition, for <em>t &gt; 0</em>, it may also be
 * changed by a {@link RateRule}, {@link Event} objects, and as a result of being a
 * reactant or product in one or more {@link Reaction} objects.  (However, some
 * constructs are mutually exclusive; see the SBML specifications for the
 * precise details.)  It is not an error to define \'initialAmount\' or
 * \'initialConcentration\' on a species and also redefine the value using an
 * {@link InitialAssignment}, but the \'initialAmount\' or \'initialConcentration\'
 * setting in that case is ignored.  The SBML specifications provide
 * additional information about the semantics of assignments, rules and
 * values for simulation time <em>t</em> &#8804; <em>0</em>.
 * <p>
 * SBML Level&nbsp;2 additionally stipulates that in cases where a species\'
 * compartment has a \'spatialDimensions\' value of <code>0</code> (zero), the species
 * cannot have a value for \'initialConcentration\' because the concepts of
 * concentration and density break down when a container has zero
 * dimensions.
 * <p>
 * <h2>The units of a species\' amount or concentration</h2>
 * <p>
 * When the attribute \'initialAmount\' is set, the unit of measurement
 * associated with the value of \'initialAmount\' is specified by the {@link Species}
 * attribute \'substanceUnits\'.  When the \'initialConcentration\' attribute
 * is set, the unit of measurement associated with this concentration value
 * is {<em>unit of amount</em>} divided by {<em>unit of size</em>}, where
 * the {<em>unit of amount</em>} is specified by the {@link Species}
 * \'substanceUnits\' attribute, and the {<em>unit of size</em>} is specified
 * by the \'units\' attribute of the {@link Compartment} object in which the species
 * is located.  Note that in either case, a unit of <em>amount</em> is
 * involved and determined by the \'substanceUnits\' attribute.  Note
 * <strong>these two attributes alone do not determine the units of the
 * species when the species identifier appears in a mathematical
 * expression</strong>; <em>that</em> aspect is determined by the attribute
 * \'hasOnlySubstanceUnits\' discussed below.
 * <p>
 * In SBML Level&nbsp;3, if the \'substanceUnits\' attribute is not set on a
 * given {@link Species} object instance, then the unit of <em>amount</em> for that
 * species is inherited from the \'substanceUnits\' attribute on the
 * enclosing {@link Model} object instance.  If that attribute on {@link Model} is not set
 * either, then the unit associated with the species\' quantity is
 * undefined.
 * <p>
 * In SBML Level&nbsp;2, if the \'substanceUnits\' attribute is not set on a
 * given {@link Species} object instance, then the unit of <em>amount</em> for that
 * species is taken from the predefined SBML unit identifier 
 * <code>\'substance\'.</code>  The value assigned to \'substanceUnits\' must be chosen from
 * one of the following possibilities: one of the base unit identifiers
 * defined in SBML, the built-in unit identifier <code>\'substance\'</code>, or the
 * identifier of a new unit defined in the list of unit definitions in the
 * enclosing {@link Model} object.  The chosen units for \'substanceUnits\' must be
 * be <code>\'dimensionless\'</code>, <code>\'mole\'</code>, <code>\'item\'</code>, <code>\'kilogram\'</code>, <code>\'gram\'</code>,
 * or units derived from these.
 * <p>
 * As noted at the beginning of this section, simply setting
 * \'initialAmount\' or \'initialConcentration\' alone does <em>not</em> determine
 * whether a species identifier represents an amount or a concentration
 * when it appears elsewhere in an SBML model.  The role of the attribute
 * \'hasOnlySubstanceUnits\' is to indicate whether the units of the species,
 * when the species identifier appears in mathematical formulas, are
 * intended to be concentration or amount.  The attribute takes on a
 * boolean value.  In SBML Level&nbsp;3, the attribute has no default value
 * and must always be set in a model; in SBML Level&nbsp;2, it has a
 * default value of <code>false.</code>
 * <p>
 * The <em>units of the species</em> are used in the following ways:
 * <ul>
<p>
 * <li> When the species\' identifier appears in a MathML formula, it
 * represents the species\' quantity, and the unit of measurement associated
 * with the quantity is as described above.
 * <p>
 * <li> The \'math\' elements of {@link AssignmentRule}, {@link InitialAssignment} and
 * {@link EventAssignment} objects referring to this species should all have the
 * same units as the unit of measurement associated with the species
 * quantity.
 * <p>
 * <li> In a {@link RateRule} object that defines the rate of change of the
 * species\' quantity, the unit associated with the rule\'s \'math\' element
 * should be equal to the unit of the species\' quantity divided by the
 * model-wide unit of <em>time</em>; in other words, {<em>unit of species
 * quantity</em>}/{<em>unit of time</em>}.
 * <p>
 * </ul>
 * <p>
 * <p>
 * <h2>The \'constant\' and \'boundaryCondition\' attributes</h2>
 * <p>
 * The {@link Species} object class has two boolean attributes named \'constant\' and
 * \'boundaryCondition\', used to indicate whether and how the quantity of
 * that species can vary during a simulation.  In SBML Level&nbsp;2 they
 * are optional; in SBML Level&nbsp;3 they are mandatory.  The following
 * table shows how to interpret the combined values of these attributes.
 * <p>
 * <center>
<table border=\'0\' class=\'text-table width80 normal-font alt-row-colors\'>
<caption class=\'top-caption\'>Interpretation
of species\' <code>constant</code> and <code>boundaryCondition</code>
attributes.</caption>
 <tr>
     <th align=\'left\' valign=\'bottom\'>
       <code>constant</code><br>value
     </th>
     <th align=\'left\' valign=\'bottom\'>
       <code>boundaryCondition</code><br>value
     </th>
     <th align=\'left\' valign=\'bottom\'>
       Can have<br>assignment<br>or rate rule?
     </th>
     <th align=\'left\' valign=\'bottom\'>
       Can be<br>reactant<br>or product?
     </th>
     <th align=\'left\' valign=\'bottom\'>
       {@link Species}\' quantity<br>can be changed by
     </th>
 </tr>
 <tr>
     <td><code>true</code></td>
     <td><code>true</code></td>
     <td>no</td>
     <td>yes</td>
     <td>(never changes)</td>
 </tr>
 <tr>
     <td><code>false</code></td>
     <td><code>true</code></td>
     <td>yes</td>
     <td>yes</td>
     <td>rules and events</td>
 </tr>
 <tr>
     <td><code>true</code></td>
     <td><code>false</code></td>
     <td>no</td>
     <td>no</td>
     <td>(never changes)</td>
 </tr>
 <tr>
     <td><code>false</code></td>
     <td><code>false</code></td>
     <td>yes</td>
     <td>yes</td>
     <td>reactions <em>or</em> rules (but not both at the same time), and events</td>
 </tr>
</table>
</center>

 * <p>
 * By default, when a species is a product or reactant of one or more
 * reactions, its quantity is determined by those reactions.  In SBML, it
 * is possible to indicate that a given species\' quantity is <em>not</em>
 * determined by the set of reactions even when that species occurs as a
 * product or reactant; i.e., the species is on the <em>boundary</em> of
 * the reaction system, and its quantity is not determined by the
 * reactions.  The boolean attribute \'boundaryCondition\' can be used to
 * indicate this.  A value of <code>false</code> indicates that the species <em>is</em>
 * part of the reaction system.  In SBML Level&nbsp;2, the attribute has a
 * default value of <code>false</code>, while in SBML Level&nbsp;3, it has no
 * default.
 * <p>
 * The \'constant\' attribute indicates whether the species\' quantity can be
 * changed at all, regardless of whether by reactions, rules, or constructs
 * other than {@link InitialAssignment}.  A value of <code>false</code> indicates that the
 * species\' quantity can be changed.  (This is also a common value because
 * the purpose of most simulations is precisely to calculate changes in
 * species quantities.)  In SBML Level&nbsp;2, the attribute has a default
 * value of <code>false</code>, while in SBML Level&nbsp;3, it has no default.  Note
 * that the initial quantity of a species can be set by an
 * {@link InitialAssignment} irrespective of the value of the \'constant\' attribute.
 * <p>
 * In practice, a \'boundaryCondition\' value of <code>true</code> means a differential
 * equation derived from the reaction definitions should not be generated
 * for the species.  However, the species\' quantity may still be changed by
 * {@link AssignmentRule}, {@link RateRule}, {@link AlgebraicRule}, {@link Event}, and {@link InitialAssignment}
 * constructs if its \'constant\' attribute is <code>false.</code>  Conversely, if the
 * species\' \'constant\' attribute is <code>true</code>, then its value cannot be
 * changed by anything except {@link InitialAssignment}.
 * <p>
 * A species having \'boundaryCondition\'=<code>false</code> and \'constant\'=<code>false</code>
 * can appear as a product and/or reactant of one or more reactions in the
 * model.  If the species is a reactant or product of a reaction, it must
 * <em>not</em> also appear as the target of any {@link AssignmentRule} or {@link RateRule}
 * object in the model.  If instead the species has \'boundaryCondition\'=
 * <code>false</code> and \'constant\'=<code>true</code>, then it cannot appear as a reactant or
 * product, or as the target of any {@link AssignmentRule}, {@link RateRule} or
 * {@link EventAssignment} object in the model.
 * <p>
 * <p>
 * <h2>The conversionFactor attribute in SBML Level&nbsp;3</h2>
 * <p>
 * In SBML Level&nbsp;3, {@link Species} has an additional optional attribute,
 * \'conversionFactor\', that defines a conversion factor that applies to a
 * particular species.  The value must be the identifier of a {@link Parameter}
 * object instance defined in the model.  That {@link Parameter} object must be a
 * constant, meaning its \'constant\' attribute must be set to <code>true.</code>
 * If a given {@link Species} object definition defines a value for its
 * \'conversionFactor\' attribute, it takes precedence over any factor
 * defined by the {@link Model} object\'s \'conversionFactor\' attribute.
 * <p>
 * The unit of measurement associated with a species\' quantity can be
 * different from the unit of extent of reactions in the model.  SBML
 * Level&nbsp;3 avoids implicit unit conversions by providing an explicit
 * way to indicate any unit conversion that might be required.  The use of
 * a conversion factor in computing the effects of reactions on a species\'
 * quantity is explained in detail in the SBML Level&nbsp;3 specification
 * document.  Because the value of the \'conversionFactor\' attribute is the
 * identifier of a {@link Parameter} object, and because parameters can have units
 * attached to them, the transformation from reaction extent units to
 * species units can be completely specified using this approach.
 * <p>
 * Note that the unit conversion factor is <strong>only applied when
 * calculating the effect of a reaction on a species</strong>.  It is not
 * used in any rules or other SBML constructs that affect the species, and
 * it is also not used when the value of the species is referenced in a
 * mathematical expression.
 * <p>
 * <p>
 * <h2>The speciesType attribute in SBML Level&nbsp;2 Versions&nbsp;2&ndash;4</h2>
 * <p>
 * In SBML Level&nbsp;2 Versions&nbsp;2&ndash;4, each species in a model
 * may optionally be designated as belonging to a particular species type.
 * The optional attribute \'speciesType\' is used to identify the species
 * type of the chemical entities that make up the pool represented by the
 * {@link Species} objects.  The attribute\'s value must be the identifier of an
 * existing {@link SpeciesType} object in the model.  If the \'speciesType\'
 * attribute is not present on a particular species definition, it means
 * the pool contains chemical entities of a type unique to that pool; in
 * effect, a virtual species type is assumed for that species, and no other
 * species can belong to that species type.  The value of \'speciesType\'
 * attributes on species have no effect on the numerical interpretation of
 * a model; simulators and other numerical analysis software may ignore
 * \'speciesType\' attributes.
 * <p>
 * There can be only one species of a given species type in any given
 * compartment of a model.  More specifically, for all {@link Species} objects
 * having a value for the \'speciesType\' attribute, the pair
 * <center>
 * (\'speciesType\' attribute value, \'compartment\' attribute value)
 * </center>
 * <p>
 * must be unique across the set of all {@link Species} object in a model.
 * <p>
 * <p>
 * <h2>The spatialSizeUnits attribute in SBML Level&nbsp;2 Versions&nbsp;1&ndash;2</h2>
 * <p>
 * In versions of SBML Level&nbsp;2 before Version&nbsp;3, the class
 * {@link Species} included an attribute called \'spatialSizeUnits\', which allowed
 * explicitly setting the units of size for initial concentration.  LibSBML
 * retains this attribute for compatibility with older definitions of
 * Level&nbsp;2, but its use is strongly discouraged because many software
 * tools do no properly interpret this unit declaration and it is
 * incompatible with all SBML specifications after Level&nbsp;2
 * Version&nbsp;3.
 * <p>
 * <p>
 * <h2>Additional considerations for interpreting the numerical value of a species</h2>
 * <p>
 * {@link Species} are unique in SBML in that they have a kind of duality: a
 * species identifier may stand for either substance amount (meaning, a
 * count of the number of individual entities) or a concentration or
 * density (meaning, amount divided by a compartment size).  The previous
 * sections explain the meaning of a species identifier when it is
 * referenced in a mathematical formula or in rules or other SBML
 * constructs; however, it remains to specify what happens to a species
 * when the compartment in which it is located changes in size.
 * <p>
 * When a species definition has a \'hasOnlySubstanceUnits\' attribute value
 * of <code>false</code> and the size of the compartment in which the species is
 * located changes, the default in SBML is to assume that it is the
 * concentration that must be updated to account for the size change.  This
 * follows from the principle that, all other things held constant, if a
 * compartment simply changes in size, the size change does not in itself
 * cause an increase or decrease in the number of entities of any species
 * in that compartment.  In a sense, the default is that the <em>amount</em> of
 * a species is preserved across compartment size changes.  Upon such size
 * changes, the value of the concentration or density must be recalculated
 * from the simple relationship <em>concentration = amount / size</em> if
 * the value of the concentration is needed (for example, if the species
 * identifier appears in a mathematical formula or is otherwise referenced
 * in an SBML construct).  There is one exception: if the species\' quantity
 * is determined by an {@link AssignmentRule}, {@link RateRule}, {@link AlgebraicRule}, or an
 * {@link EventAssignment} and the species has a \'hasOnlySubstanceUnits\' attribute
 * value of <code>false</code>, it means that the <em>concentration</em> is assigned
 * by the rule or event; in that case, the <em>amount</em> must be
 * calculated when the compartment size changes.  (Events also require
 * additional care in this situation, because an event with multiple
 * assignments could conceivably reassign both a species quantity and a
 * compartment size simultaneously.  Please refer to the SBML
 * specifications for the details.)
 * <p>
 * Note that the above only matters if a species has a
 * \'hasOnlySubstanceUnits\' attribute value of <code>false</code>, meaning that the
 * species identifier refers to a concentration wherever the identifier
 * appears in a mathematical formula.  If instead the attribute\'s value is
 * <code>true</code>, then the identifier of the species <em>always</em> stands for
 * an amount wherever it appears in a mathematical formula or is referenced
 * by an SBML construct.  In that case, there is never a question about
 * whether an assignment or event is meant to affect the amount or
 * concentration: it is always the amount.
 * <p>
 * A particularly confusing situation can occur when the species has
 * \'constant\' attribute value of <code>true</code> in combination with a
 * \'hasOnlySubstanceUnits\' attribute value of <code>false.</code>  Suppose this
 * species is given a value for \'initialConcentration\'.  Does a \'constant\'
 * value of <code>true</code> mean that the concentration is held constant if the
 * compartment size changes?  No; it is still the amount that is kept
 * constant across a compartment size change.  The fact that the species
 * was initialized using a concentration value is irrelevant.
 * <p>
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%typemap(javaimports) ListOfSpecies "
/** 
 * LibSBML implementation of SBML Level&nbsp;2\'s ListOfSpecies construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers Species::Species(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link Species} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Species}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Species}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Species} object to an {@link SBMLDocument} (e.g.,
   * using {@link Model#addSpecies(Species s)}), the SBML Level, SBML Version and XML
   * namespace of the document <em>override</em> the values used when creating
   * the {@link Species} object via this constructor.  This is necessary to ensure
   * that an SBML document is a consistent structure.  Nevertheless, the
   * ability to supply the values at the time of creation of a {@link Species} is
   * an important aid to producing valid SBML.  Knowledge of the intented
   * SBML Level and Version determine whether it is valid to assign a
   * particular value to an attribute, or whether it is valid to add an
   * object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Species::Species(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link Species} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link Species} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method {@link Species#setId(String id)}.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Species} object to an {@link SBMLDocument} (e.g.,
   * using {@link Model#addSpecies(Species s)}), the SBML XML namespace of the document 
   * <em>overrides</em> the value used when creating the {@link Species} object via this
   * constructor.  This is necessary to ensure that an SBML document is a
   * consistent structure.  Nevertheless, the ability to supply the values
   * at the time of creation of a {@link Species} is an important aid to producing
   * valid SBML.  Knowledge of the intented SBML Level and Version
   * determine whether it is valid to assign a particular value to an
   * attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Species::Species(const Species& orig) "
/**
   * Copy constructor; creates a copy of this {@link Species} object.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers Species::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Species}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */ public
";


%javamethodmodifiers Species::clone() const "
/**
   * Creates and returns a deep copy of this {@link Species} object.
   * <p>
   * @return a (deep) copy of this {@link Species} object.
   */ public
";


%javamethodmodifiers Species::initDefaults "
/**
   * Initializes the fields of this {@link Species} object to \'typical\' defaults
   * values.
   * <p>
   * The SBML {@link Species} component has slightly different aspects and
   * default attribute values in different SBML Levels and Versions.
   * This method sets the values to certain common defaults, based
   * mostly on what they are in SBML Level&nbsp;2.  Specifically:
   * <ul>
   * <li> Sets \'boundaryCondition\' to <code>false</code>
   * <li> Sets \'constant\' to <code>false</code>
   * <li> sets \'hasOnlySubstanceUnits\' to <code>false</code>
   * <li> (Applies to Level&nbsp;3 models only) Sets attribute \'substanceUnits\' to <code>mole</code>
   * </ul>
   */ public
";


%javamethodmodifiers Species::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link Species} object.
   * <p>
   * @return the id of this {@link Species} object.
   */ public
";


%javamethodmodifiers Species::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link Species} object.
   * <p>
   * @return the name of this {@link Species} object.
   */ public
";


%javamethodmodifiers Species::getSpeciesType() const "
/**
   * Get the type of this {@link Species} object object.
   * <p>
   * @return the value of the \'speciesType\' attribute of this
   * {@link Species} as a string.
   * <p>
   * @note The \'speciesType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%javamethodmodifiers Species::getCompartment() const "
/**
   * Get the compartment in which this species is located.
   * <p>
   * The compartment is designated by its identifier.
   * <p>
   * @return the value of the \'compartment\' attribute of this {@link Species}
   * object, as a string.
   */ public
";


%javamethodmodifiers Species::getInitialAmount() const "
/**
   * Get the value of the \'initialAmount\' attribute.
   * <p>
   * @return the initialAmount of this {@link Species}, as a float-point number.
   */ public
";


%javamethodmodifiers Species::getInitialConcentration() const "
/**
   * Get the value of the \'initialConcentration\' attribute.
   * <p>
   * @return the initialConcentration of this {@link Species},, as a float-point
   * number.
   * <p>
   * @note The attribute \'initialConcentration\' is only available in SBML
   * Level&nbsp;2 and&nbsp;3.  It does not exist on {@link Species} in Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::getSubstanceUnits() const "
/**
   * Get the value of the \'substanceUnits\' attribute.
   * <p>
   * @return the value of the \'substanceUnits\' attribute of this {@link Species},
   * as a string.  An empty string indicates that no units have been
   * assigned.
   * <p>
   * @note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

   * <p>
   * @see #isSetSubstanceUnits()
   * @see #setSubstanceUnits(String sid)
   */ public
";


%javamethodmodifiers Species::getSpatialSizeUnits() const "
/**
   * Get the value of the \'spatialSizeUnits\' attribute.
   * <p>
   * @return the value of the \'spatialSizeUnits\' attribute of this {@link Species}
   * object, as a string.
   * <p>
   * @warning In versions of SBML Level&nbsp;2 before Version&nbsp;3, the
   * class {@link Species} included an attribute called \'spatialSizeUnits\', which
   * allowed explicitly setting the units of size for initial
   * concentration.  This attribute was removed in SBML Level&nbsp;2
   * Version&nbsp;3.  LibSBML retains this attribute for compatibility with
   * older definitions of Level&nbsp;2, but its use is strongly discouraged
   * because it is incompatible with Level&nbsp;2 Version&nbsp;3 and
   * Level&nbsp;2 Version&nbsp;4.
   */ public
";


%javamethodmodifiers Species::getUnits() const "
/**
   * Get the value of the \'units\' attribute.
   * <p>
   * @return the units of this {@link Species} (L1 only).
   * <p>
   * @note The \'units\' attribute is defined only in SBML Level&nbsp;1.  In
   * SBML Level&nbsp;2 and Level&nbsp;3, it has been replaced by a
   * combination of \'substanceUnits\' and the units of the {@link Compartment}
   * object in which a species is located.  In SBML Level&nbsp;2
   * Versions&nbsp;1&ndash;2, an additional attribute \'spatialSizeUnits\'
   * helps determine the units of the species quantity, but this attribute
   * was removed in later versions of SBML Level&nbsp;2.
   */ public
";


%javamethodmodifiers Species::getHasOnlySubstanceUnits() const "
/**
   * Get the value of the \'hasOnlySubstanceUnits\' attribute.
   * <p>
   * @return <code>true</code> if this {@link Species}\' \'hasOnlySubstanceUnits\' attribute
   * value is nonzero, <code>false</code> otherwise.
   * <p>
   * @note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
   * Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::getBoundaryCondition() const "
/**
   * Get the value of the \'boundaryCondition\' attribute.
   * <p>
   * @return <code>true</code> if this {@link Species}\' \'boundaryCondition\' attribute value
   * is nonzero, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Species::getCharge() const "
/**
   * Get the value of the \'charge\' attribute.
   * <p>
   * @return the charge of this {@link Species} object.
   * <p>
   * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
   * attribute on {@link Species} is deprecated and in SBML Level&nbsp;3 it does
   * not exist at all.  Its use strongly discouraged.  Its presence is
   * considered a misfeature in earlier definitions of SBML because its
   * implications for the mathematics of a model were never defined, and in
   * any case, no known modeling system ever used it.  Instead, models take
   * account of charge values directly in their definitions of species by
   * (for example) having separate species identities for the charged and
   * uncharged versions of the same species.  This allows the condition to
   * affect model mathematics directly.  LibSBML retains this method for
   * easier compatibility with SBML Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::getConstant() const "
/**
   * Get the value of the \'constant\' attribute.
   * <p>
   * @return <code>true</code> if this {@link Species}\'s \'constant\' attribute value is
   * nonzero, <code>false</code> otherwise.
   * <p>
   * @note The attribute \'constant\' is only available in SBML Levels&nbsp;2
   * and&nbsp;3.  It does not exist on {@link Species} in Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::getConversionFactor() const "
/**
   * Get the value of the \'conversionFactor\' attribute.
   * <p>
   * @return the conversionFactor of this {@link Species}, as a string.
   * <p>
   * @note The \'conversionFactor\' attribute was introduced in SBML
   * Level&nbsp;3.  It does not exist on {@link Species} in SBML Levels&nbsp;1
   * and&nbsp;2.
   */ public
";


%javamethodmodifiers Species::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Species::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Species::isSetSpeciesType() const "
/**
   * Predicate returning <code>true</code> if this {@link Species} object\'s
   * \'speciesType\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'speciesType\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'speciesType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%javamethodmodifiers Species::isSetCompartment() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'compartment\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'compartment\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Species::isSetInitialAmount() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'initialAmount\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'initialAmount\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note In SBML Level&nbsp;1, {@link Species}\' \'initialAmount\' is required and
   * therefore <em>should always be set</em>.  (However, in Level&nbsp;1, the
   * attribute has no default value either, so this method will not return
   * <code>true</code> until a value has been assigned.)  In SBML Level&nbsp;2,
   * \'initialAmount\' is optional and as such may or may not be set.
   */ public
";


%javamethodmodifiers Species::isSetInitialConcentration() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'initialConcentration\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'initialConcentration\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The attribute \'initialConcentration\' is only available in SBML
   * Level&nbsp;2 and&nbsp;3.  It does not exist on {@link Species} in Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::isSetSubstanceUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'substanceUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'substanceUnits\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Species::isSetSpatialSizeUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'spatialSizeUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'spatialSizeUnits\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   * <p>
   * @warning In versions of SBML Level~2 before Version&nbsp;3, the class
   * {@link Species} included an attribute called \'spatialSizeUnits\', which allowed
   * explicitly setting the units of size for initial concentration.  This
   * attribute was removed in SBML Level&nbsp;2 Version&nbsp;3.  LibSBML
   * retains this attribute for compatibility with older definitions of
   * Level&nbsp;2, but its use is strongly discouraged because it is
   * incompatible with Level&nbsp;2 Version&nbsp;3 and Level&nbsp;2 Version&nbsp;4.
   */ public
";


%javamethodmodifiers Species::isSetUnits() const "
/**
   * Predicate returning <code>true</code> if
   * this {@link Species} object\'s \'units\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'units\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Species::isSetCharge() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'charge\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'charge\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
   * attribute on {@link Species} in SBML is deprecated and in SBML Level&nbsp;3 it
   * does not exist at all.  Its use strongly discouraged.  Its presence is
   * considered a misfeature in earlier definitions of SBML because its
   * implications for the mathematics of a model were never defined, and in
   * any case, no known modeling system ever used it.  Instead, models take
   * account of charge values directly in their definitions of species by
   * (for example) having separate species identities for the charged and
   * uncharged versions of the same species.  This allows the condition to
   * affect model mathematics directly.  LibSBML retains this method for
   * easier compatibility with SBML Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::isSetConversionFactor() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'conversionFactor\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'conversionFactor\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'conversionFactor\' attribute was introduced in SBML
   * Level&nbsp;3.  It does not exist on {@link Species} in SBML Levels&nbsp;1
   * and&nbsp;2.
   */ public
";


%javamethodmodifiers Species::isSetBoundaryCondition() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'boundaryCondition\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'boundaryCondition\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Species::isSetHasOnlySubstanceUnits() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'hasOnlySubstanceUnits\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'hasOnlySubstanceUnits\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
   * Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::isSetConstant() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Species} object\'s \'constant\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'constant\' attribute of this {@link Species} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The attribute \'constant\' is only available in SBML Levels&nbsp;2
   * and&nbsp;3.  It does not exist on {@link Species} in Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link Species} object.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link Species}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Species::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link Species} object.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link Species}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Species::setSpeciesType(const std::string& sid) "
/**
   * Sets the \'speciesType\' attribute of this {@link Species} object.
   * <p>
   * @param sid the identifier of a {@link SpeciesType} object defined elsewhere
   * in this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note The \'speciesType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%javamethodmodifiers Species::setCompartment(const std::string& sid) "
/**
   * Sets the \'compartment\' attribute of this {@link Species} object.
   * <p>
   * @param sid the identifier of a {@link Compartment} object defined elsewhere
   * in this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Species::setInitialAmount(double value) "
/**
   * Sets the \'initialAmount\' attribute of this {@link Species} and marks the field
   * as set.
   * <p>
   * This method also unsets the \'initialConcentration\' attribute.
   * <p>
   * @param value the value to which the \'initialAmount\' attribute should
   * be set.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   */ public
";


%javamethodmodifiers Species::setInitialConcentration(double value) "
/**
   * Sets the \'initialConcentration\' attribute of this {@link Species} and marks
   * the field as set.
   * <p>
   * This method also unsets the \'initialAmount\' attribute.
   * <p>
   * @param value the value to which the \'initialConcentration\' attribute
   * should be set.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note The attribute \'initialConcentration\' is only available in SBML
   * Level&nbsp;2 and&nbsp;3.  It does not exist on {@link Species} in Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::setSubstanceUnits(const std::string& sid) "
/**
   * Sets the \'substanceUnits\' attribute of this {@link Species} object.
   * <p>
   * @param sid the identifier of the unit to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Species::setSpatialSizeUnits(const std::string& sid) "
/**
   * (SBML Level&nbsp;2 Versions&nbsp;1&ndash;2) Sets the \'spatialSizeUnits\' attribute of this {@link Species} object.
   * <p>
   * @param sid the identifier of the unit to use.
   * <p>
   * @warning In versions of SBML Level~2 before Version&nbsp;3, the class
   * {@link Species} included an attribute called \'spatialSizeUnits\', which allowed
   * explicitly setting the units of size for initial concentration.  This
   * attribute was removed in SBML Level&nbsp;2 Version&nbsp;3.  LibSBML
   * retains this attribute for compatibility with older definitions of
   * Level&nbsp;2, but its use is strongly discouraged because it is
   * incompatible with Level&nbsp;2 Version&nbsp;3 and Level&nbsp;2 Version&nbsp;4.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   */ public
";


%javamethodmodifiers Species::setUnits(const std::string& sname) "
/**
   * (SBML Level&nbsp;1 only) Sets the units of this {@link Species} object.
   * <p>
   * @param sname the identifier of the unit to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
  */ public
";


%javamethodmodifiers Species::setHasOnlySubstanceUnits(bool value) "
/**
   * Sets the \'hasOnlySubstanceUnits\' attribute of this {@link Species} object.
   * <p>
   * @param value boolean value for the \'hasOnlySubstanceUnits\' attribute.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
   * Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::setBoundaryCondition(bool value) "
/**
   * Sets the \'boundaryCondition\' attribute of this {@link Species} object.
   * <p>
   * @param value boolean value for the \'boundaryCondition\' attribute.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   */ public
";


%javamethodmodifiers Species::setCharge(int value) "
/**
   * Sets the \'charge\' attribute of this {@link Species} object.
   * <p>
   * @param value an integer to which to set the \'charge\' to.
   * <p>
   * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
   * attribute on {@link Species} in SBML is deprecated and its use strongly
   * discouraged, and it does not exist in SBML Level&nbsp;3 at all.  Its
   * presence is considered a misfeature in earlier definitions of SBML
   * because its implications for the mathematics of a model were never
   * defined, and in any case, no known modeling system ever used it.
   * Instead, models take account of charge values directly in their
   * definitions of species by (for example) having separate species
   * identities for the charged and uncharged versions of the same species.
   * This allows the condition to affect model mathematics directly.
   * LibSBML retains this method for easier compatibility with SBML
   * Level&nbsp;1.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   */ public
";


%javamethodmodifiers Species::setConstant(bool value) "
/**
   * Sets the \'constant\' attribute of this {@link Species} object.
   * <p>
   * @param value a boolean value for the \'constant\' attribute
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note The attribute \'constant\' is only available in SBML Levels&nbsp;2
   * and&nbsp;3.  It does not exist on {@link Species} in Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::setConversionFactor(const std::string& sid) "
/**
   * Sets the value of the \'conversionFactor\' attribute of this {@link Species} object.
   * <p>
   * The string in <code>sid</code> is copied.
   * <p>
   * @param sid the new conversionFactor for the {@link Species}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'conversionFactor\' attribute was introduced in SBML
   * Level&nbsp;3.  It does not exist on {@link Species} in SBML Levels&nbsp;1
   * and&nbsp;2.
   */ public
";


%javamethodmodifiers Species::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Species::unsetSpeciesType "
/**
   * Unsets the \'speciesType\' attribute value of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The attribute \'speciesType\' is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%javamethodmodifiers Species::unsetInitialAmount "
/**
   * Unsets the \'initialAmount\' attribute value of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Species::unsetInitialConcentration "
/**
   * Unsets the \'initialConcentration\' attribute value of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The attribute \'initialConcentration\' is only available in SBML
   * Level&nbsp;2 and&nbsp;3.  It does not exist on {@link Species} in Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::unsetSubstanceUnits "
/**
   * Unsets the \'substanceUnits\' attribute value of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Species::unsetSpatialSizeUnits "
/**
   * Unsets the \'spatialSizeUnits\' attribute value of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @warning In versions of SBML Level~2 before Version&nbsp;3, the class
   * {@link Species} included an attribute called \'spatialSizeUnits\', which allowed
   * explicitly setting the units of size for initial concentration.  This
   * attribute was removed in SBML Level&nbsp;2 Version&nbsp;3.  LibSBML
   * retains this attribute for compatibility with older definitions of
   * Level&nbsp;2, but its use is strongly discouraged because it is
   * incompatible with Level&nbsp;2 Version&nbsp;3 and Level&nbsp;2 Version&nbsp;4.
   */ public
";


%javamethodmodifiers Species::unsetUnits "
/**
   * Unsets the \'units\' attribute value of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Species::unsetCharge "
/**
   * Unsets the \'charge\' attribute
   * value of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
   * attribute on {@link Species} in SBML is deprecated and its use strongly
   * discouraged, and it does not exist in SBML Level&nbsp;3 at all.  Its
   * presence is considered a misfeature in earlier definitions of SBML
   * because its implications for the mathematics of a model were never
   * defined, and in any case, no known modeling system ever used it.
   * Instead, models take account of charge values directly in their
   * definitions of species by (for example) having separate species
   * identities for the charged and uncharged versions of the same species.
   * This allows the condition to affect model mathematics directly.
   * LibSBML retains this method for easier compatibility with SBML
   * Level&nbsp;1.
   */ public
";


%javamethodmodifiers Species::unsetConversionFactor "
/**
   * Unsets the \'conversionFactor\' attribute value of this {@link Species} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function. The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'conversionFactor\' attribute was introduced in SBML
   * Level&nbsp;3.  It does not exist on {@link Species} in SBML Levels&nbsp;1
   * and&nbsp;2.
   */ public
";


%javamethodmodifiers Species::getDerivedUnitDefinition "
/**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Species}\' amount or concentration.
   * <p>
   * {@link Species} in SBML have an attribute (\'substanceUnits\') for declaring the
   * units of measurement intended for the species\' amount or concentration
   * (depending on which one applies).  In the absence of a value given for
   * \'substanceUnits\', the units are taken from the enclosing {@link Model}\'s
   * definition of <code>\'substance\'</code> or <code>\'substance\'</code>/<em>(size of the
   * compartment)</em> in which the species is located, or finally, if
   * these are not redefined by the {@link Model}, the relevant SBML default units
   * for those quantities.  Following that procedure, the method
   * {@link Species#getDerivedUnitDefinition()}
   * returns a {@link UnitDefinition} based on the
   * interpreted units of this species\'s amount or concentration.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return <code>null.</code>
   * <p>
   * Note also that unit declarations for {@link Species} are in terms of the 
   * <em>identifier</em> of a unit, but this method returns a {@link UnitDefinition} object,
   * not a unit identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  Callers may find this particularly useful when used
   * in conjunction with the helper methods on {@link UnitDefinition} for comparing
   * different {@link UnitDefinition} objects.
   * <p>
   * In SBML Level&nbsp;2 specifications prior to Version&nbsp;3, {@link Species}
   * includes an additional attribute named \'spatialSizeUnits\', which
   * allows explicitly setting the units of size for initial concentration.
   * The {@link Species#getDerivedUnitDefinition()}
   * takes this into account for models
   * expressed in SBML Level&nbsp;2 Versions&nbsp;1 and&nbsp;2.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Species}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @see #getSubstanceUnits()
   */ public
";


%javamethodmodifiers Species::getDerivedUnitDefinition() const "
/**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Species}\' amount or concentration.
   * <p>
   * {@link Species} in SBML have an attribute (\'substanceUnits\') for declaring the
   * units of measurement intended for the species\' amount or concentration
   * (depending on which one applies).  In the absence of a value given for
   * \'substanceUnits\', the units are taken from the enclosing {@link Model}\'s
   * definition of <code>\'substance\'</code> or <code>\'substance\'</code>/<em>(size of the
   * compartment)</em> in which the species is located, or finally, if
   * these are not redefined by the {@link Model}, the relevant SBML default units
   * for those quantities.  Following that procedure, the method
   * {@link Species#getDerivedUnitDefinition()}
   * returns a {@link UnitDefinition} based on the
   * interpreted units of this species\'s amount or concentration.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return <code>null.</code>
   * <p>
   * Note also that unit declarations for {@link Species} are in terms of the 
   * <em>identifier</em> of a unit, but this method returns a {@link UnitDefinition} object,
   * not a unit identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  Callers may find this particularly useful when used
   * in conjunction with the helper methods on {@link UnitDefinition} for comparing
   * different {@link UnitDefinition} objects.
   * <p>
   * In SBML Level&nbsp;2 specifications prior to Version&nbsp;3, {@link Species}
   * includes an additional attribute named \'spatialSizeUnits\', which
   * allows explicitly setting the units of size for initial concentration.
   * The {@link Species#getDerivedUnitDefinition()}
   * takes this into account for models
   * expressed in SBML Level&nbsp;2 Versions&nbsp;1 and&nbsp;2.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Species}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @see #getSubstanceUnits()
   */ public
";


%javamethodmodifiers Species::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers Species::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link Species}, is
   * always <code>\'species\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'species\'.</code>
   */ public
";


%javamethodmodifiers Species::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Species::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Species} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Species} object are:
   * <ul>
   * <li> \'id\' (or \'name\' in SBML Level&nbsp;1)
   * <li> \'compartment\'
   * <li> \'initialAmount\' (required in SBML Level&nbsp;1 only; optional otherwise)
   * <li> \'hasOnlySubstanceUnits\' (required in SBML Level&nbsp;3; optional in SBML Level&nbsp;2)
   * <li> \'boundaryCondition\' (required in SBML Level&nbsp;3; optional in Levels&nbsp;1 and&nbsp;2)
   * <li> \'constant\' (required in SBML Level&nbsp;3; optional in SBML Level&nbsp;2)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers Species::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%javamethodmodifiers Species::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers Species::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers Species::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Species::readL1Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Species::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Species::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Species::isExplicitlySetBoundaryCondition() const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Species::isExplicitlySetConstant() const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Species::isExplicitlySetHasOnlySubsUnits() const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfSpecies::ListOfSpecies(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfSpecies} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfSpecies::ListOfSpecies(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfSpecies} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfSpecies} object to be created.
   */ public
";


%javamethodmodifiers ListOfSpecies::clone() const "
/**
   * Creates and returns a deep copy of this ListOfSpeciess instance.
   * <p>
   * @return a (deep) copy of this ListOfSpeciess.
   */ public
";


%javamethodmodifiers ListOfSpecies::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfSpecies::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Species} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfSpecies::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For ListOfSpeciess, the XML element name is <code>\'listOfSpeciess\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfSpeciess\'.</code>
   */ public
";


%javamethodmodifiers ListOfSpecies::get(unsigned int n) "
/**
   * Get a {@link Species} from the {@link ListOfSpecies}.
   * <p>
   * @param n the index number of the {@link Species} to get.
   * <p>
   * @return the nth {@link Species} in this {@link ListOfSpecies}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpecies::get "
/**
   * Get a {@link Species} from the {@link ListOfSpecies}.
   * <p>
   * @param n the index number of the {@link Species} to get.
   * <p>
   * @return the nth {@link Species} in this {@link ListOfSpecies}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpecies::get(const std::string& sid) "
/**
   * Get a {@link Species} from the {@link ListOfSpecies}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Species} to get.
   * <p>
   * @return {@link Species} in this {@link ListOfSpecies}
   * with the given id or <code>null</code> if no such
   * {@link Species} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpecies::get(const std::string& sid) const "
/**
   * Get a {@link Species} from the {@link ListOfSpecies}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Species} to get.
   * <p>
   * @return {@link Species} in this {@link ListOfSpecies}
   * with the given id or <code>null</code> if no such
   * {@link Species} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpecies::remove(unsigned int n) "
/**
   * Removes the nth item from this ListOfSpeciess items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfSpecies::remove(const std::string& sid) "
/**
   * Removes item in this ListOfSpeciess items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOfSpecies::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the ListOfSpeciess in
   * a model is (in SBML Level&nbsp;2 Version&nbsp;4) the sixth
   * ListOf___.  (However, it differs for different Levels and Versions of
   * SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfSpecies::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) Parameter "
/** 
 * LibSBML implementation of SBML\'s Parameter construct.
 * <p>
 * A {@link Parameter} is used in SBML to define a symbol associated with a value;
 * this symbol can then be used in mathematical formulas in a model.  By
 * default, parameters have constant value for the duration of a
 * simulation, and for this reason are called <em>parameters</em> instead of 
 * <em>variables</em> in SBML, although it is crucial to understand that <em>SBML
 * parameters represent both concepts</em>.  Whether a given SBML
 * parameter is intended to be constant or variable is indicated by the
 * value of its \'constant\' attribute.
 * <p>
 * SBML\'s {@link Parameter} has a required attribute, \'id\', that gives the
 * parameter a unique identifier by which other parts of an SBML model
 * definition can refer to it.  A parameter can also have an optional
 * \'name\' attribute of type <code>string.</code>  Identifiers and names must be used
 * according to the guidelines described in the SBML specifications.
 * <p>
 * The optional attribute \'value\' determines the value (of type <code>double</code>)
 * assigned to the parameter.  A missing value for \'value\' implies that
 * the value either is unknown, or to be obtained from an external source,
 * or determined by an initial assignment.  The unit of measurement
 * associated with the value of the parameter can be specified using the
 * optional attribute \'units\'.  Here we only mention briefly some notable
 * points about the possible unit choices, but readers are urged to consult
 * the SBML specification documents for more information:
 * <ul>
 * <p>
 * <li> In SBML Level&nbsp;3, there are no constraints on the units that
 * can be assigned to parameters in a model; there are also no units to
 * inherit from the enclosing {@link Model} object (unlike the case for, e.g.,
 * {@link Species} and {@link Compartment}).
 * <p>
 * <li> In SBML Level&nbsp;2, the value assigned to the parameter\'s \'units\'
 * attribute must be chosen from one of the following possibilities: one of
 * the base unit identifiers defined in SBML; one of the built-in unit
 * identifiers <code>\'substance\'</code>, <code>\'time\'</code>, <code>\'volume\'</code>, <code>\'area\'</code> or 
 * <code>\'length\'</code>; or the identifier of a new unit defined in the list of unit
 * definitions in the enclosing {@link Model} structure.  There are no constraints
 * on the units that can be chosen from these sets.  There are no default
 * units for parameters.
 * </ul>
 * <p>
 * The {@link Parameter} structure has another boolean attribute named \'constant\'
 * that is used to indicate whether the parameter\'s value can vary during a
 * simulation.  (In SBML Level&nbsp;3, the attribute is mandatory and must
 * be given a value; in SBML Levels below Level&nbsp;3, the attribute is
 * optional.)  A value of <code>true</code> indicates the parameter\'s value cannot be
 * changed by any construct except {@link InitialAssignment}.  Conversely, if the
 * value of \'constant\' is <code>false</code>, other constructs in SBML, such as rules
 * and events, can change the value of the parameter.
 * <p>
 * SBML Level&nbsp;3 uses a separate object class, {@link LocalParameter}, for
 * parameters that are local to a {@link Reaction}\'s {@link KineticLaw}.  In Levels prior
 * to SBML Level&nbsp;3, the {@link Parameter} class is used both for definitions
 * of global parameters, as well as reaction-local parameters stored in a
 * list within {@link KineticLaw} objects.  {@link Parameter} objects that are local to a
 * reaction (that is, those defined within the {@link KineticLaw} structure of a
 * {@link Reaction}) cannot be changed by rules and therefore are <em>implicitly
 * always constant</em>; consequently, in SBML Level&nbsp;2, parameter
 * definitions within {@link Reaction} structures should <em>not</em> have their
 * \'constant\' attribute set to <code>false.</code>
 * <p>
 * What if a global parameter has its \'constant\' attribute set to <code>false</code>,
 * but the model does not contain any rules, events or other constructs
 * that ever change its value over time?  Although the model may be
 * suspect, this situation is not strictly an error.  A value of <code>false</code>
 * for \'constant\' only indicates that a parameter <em>can</em> change value, not
 * that it <em>must</em>.
 * <p>
 * As with all other major SBML components, {@link Parameter} is derived from
 * {@link SBase}, and the methods defined on {@link SBase} are available on {@link Parameter}.
 * <p>
 * @note The use of the term <em>parameter</em> in SBML sometimes leads to
 * confusion among readers who have a particular notion of what something
 * called \'parameter\' should be.  It has been the source of heated debate,
 * but despite this, no one has yet found an adequate replacement term that
 * does not have different connotations to different people and hence leads
 * to confusion among <em>some</em> subset of users.  Perhaps it would have been
 * better to have two constructs, one called <em>constants</em> and the other
 * called <em>variables</em>.  The current approach in SBML is simply more
 * parsimonious, using a single {@link Parameter} construct with the boolean flag
 * \'constant\' indicating which flavor it is.  In any case, readers are
 * implored to look past their particular definition of a <em>parameter</em> and
 * simply view SBML\'s {@link Parameter} as a single mechanism for defining both
 * constants and (additional) variables in a model.  (We write 
 * <em>additional</em> because the species in a model are usually considered to be
 * the central variables.)  After all, software tools are not required to
 * expose to users the actual names of particular SBML constructs, and
 * thus tools can present to their users whatever terms their designers
 * feel best matches their target audience.
 * <p>
 * @see ListOfParameters
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%typemap(javaimports) ListOfParameters "
/** 
 * LibSBML implementation of SBML\'s ListOfParameters construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers Parameter::Parameter(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link Parameter} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Parameter}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Parameter}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Parameter} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addParameter(Parameter p)}), the SBML Level, SBML Version
   * and XML namespace of the document <em>override</em> the values used
   * when creating the {@link Parameter} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link Parameter} is an important aid to producing valid SBML.  Knowledge
   * of the intented SBML Level and Version determine whether it is valid
   * to assign a particular value to an attribute, or whether it is valid
   * to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Parameter::Parameter(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link Parameter} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link Parameter} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor if no identifier is provided as an argument.  Setting the
   * identifier can be accomplished using the method
   * setId(String id).
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Parameter} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addParameter(Parameter p)}), the SBML XML namespace of the
   * document <em>overrides</em> the value used when creating the {@link Parameter}
   * object via this constructor.  This is necessary to ensure that an SBML
   * document is a consistent structure.  Nevertheless, the ability to
   * supply the values at the time of creation of a {@link Parameter} is an
   * important aid to producing valid SBML.  Knowledge of the intented SBML
   * Level and Version determine whether it is valid to assign a particular
   * value to an attribute, or whether it is valid to add an object to an
   * existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Parameter::Parameter(const Parameter& orig) "
/**
   * Copy constructor; creates a copy of a {@link Parameter}.
   * <p>
   * @param orig the {@link Parameter} instance to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers Parameter::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Parameter}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, indicating
   * whether the Visitor would like to visit the next {@link Parameter} object in
   * the list of parameters within which <em>the</em> present object is
   * embedded.
   */ public
";


%javamethodmodifiers Parameter::clone() const "
/**
   * Creates and returns a deep copy of this {@link Parameter}.
   * <p>
   * @return a (deep) copy of this {@link Parameter}.
   */ public
";


%javamethodmodifiers Parameter::initDefaults "
/**
   * Initializes the fields of this {@link Parameter} object to \'typical\' defaults
   * values.
   * <p>
   * The SBML {@link Parameter} component has slightly different aspects and
   * default attribute values in different SBML Levels and Versions.  Many
   * SBML object classes defined by libSBML have an initDefaults() method
   * to set the values to certain common defaults, based mostly on what
   * they are in SBML Level&nbsp;2.  In the case of {@link Parameter}, this method
   * only sets the value of the \'constant\' attribute to <code>true.</code>
   * <p>
   * @see #getConstant()
   * @see #isSetConstant()
   * @see #setConstant(boolean flag)
   */ public
";


%javamethodmodifiers Parameter::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link Parameter}.
   * <p>
   * @return the id of this {@link Parameter}.
   */ public
";


%javamethodmodifiers Parameter::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link Parameter}.
   * <p>
   * @return the name of this {@link Parameter}.
   */ public
";


%javamethodmodifiers Parameter::getValue() const "
/**
   * Gets the numerical value of this {@link Parameter}.
   * <p>
   * @return the value of the \'value\' attribute of this {@link Parameter}, as a
   * number of type <code>double.</code>
   * <p>
   * @note <b>It is crucial</b> that callers not blindly call
   * {@link Parameter#getValue()} without first using {@link Parameter#isSetValue()} to
   * determine whether a value has ever been set.  Otherwise, the value
   * return by {@link Parameter#getValue()} may not actually represent a value
   * assigned to the parameter.  The reason is simply that the data type
   * <code>double</code> in a program always has <em>some</em> value.  A separate test is
   * needed to determine whether the value is a true model value, or
   * uninitialized data in a computer\'s memory location.
   * <p>
   * @see #isSetValue()
   * @see #setValue(double value)
   * @see #getUnits()
   */ public
";


%javamethodmodifiers Parameter::getUnits() const "
/**
   * Gets the units defined for this {@link Parameter}.
   * <p>
   * The value of an SBML parameter\'s \'units\' attribute establishes the
   * unit of measurement associated with the parameter\'s value.
   * <p>
   * @return the value of the \'units\' attribute of this {@link Parameter}, as a
   * string.  An empty string indicates that no units have been assigned.
   * <p>
   * @note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

   * <p>
   * @see #isSetUnits()
   * @see #setUnits(String units)
   * @see #getValue()
   */ public
";


%javamethodmodifiers Parameter::getConstant() const "
/**
   * Gets the value of the \'constant\' attribute of this {@link Parameter} instance.
   * <p>
   * @return <code>true</code> if this {@link Parameter} is declared as being constant,
   * <code>false</code> otherwise.
   * <p>
   * @note Readers who view the documentation for {@link LocalParameter} may be
   * confused about the presence of this method.  LibSBML derives
   * {@link LocalParameter} from {@link Parameter}; however, this does not precisely match
   * the object hierarchy defined by SBML Level&nbsp;3, where
   * {@link LocalParameter} is derived directly from {@link SBase} and not {@link Parameter}.  We
   * believe this arrangement makes it easier for libSBML users to program
   * applications that work with both SBML Level&nbsp;2 and SBML
   * Level&nbsp;3, but programmers should also keep in mind this difference
   * exists.  A side-effect of libSBML\'s scheme is that certain methods on
   * {@link LocalParameter} that are inherited from {@link Parameter} do not actually have
   * relevance to {@link LocalParameter} objects.  An example of this is the
   * methods pertaining to {@link Parameter}\'s attribute \'constant\' (i.e.,
   * isSetConstant(), setConstant(), and getConstant()).
   * <p>
   * @see #isSetConstant()
   * @see #setConstant(boolean flag)
   */ public
";


%javamethodmodifiers Parameter::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Parameter}\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Parameter} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Parameter::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Parameter}\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Parameter} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Parameter::isSetValue() const "
/**
   * Predicate returning <code>true</code> if the
   * \'value\' attribute of this {@link Parameter} is set.
   * <p>
   * In SBML definitions after SBML Level&nbsp;1 Version&nbsp;1,
   * parameter values are optional and have no defaults.  If a model read
   * from a file does not contain a setting for the \'value\' attribute of a
   * parameter, its value is considered unset; it does not default to any
   * particular value.  Similarly, when a {@link Parameter} object is created in
   * libSBML, it has no value until given a value.  The
   * {@link Parameter#isSetValue()} method allows calling applications to
   * determine whether a given parameter\'s value has ever been set.
   * <p>
   * In SBML Level&nbsp;1 Version&nbsp;1, parameters are required to have
   * values and therefore, the value of a {@link Parameter} <b>should always be
   * set</b>.  In Level&nbsp;1 Version&nbsp;2 and beyond, the value is
   * optional and as such, the \'value\' attribute may or may not be set.
   * <p>
   * @return <code>true</code> if the value of this {@link Parameter} is set,
   * <code>false</code> otherwise.
   * <p>
   * @see #getValue()
   * @see #setValue(double value)
   */ public
";


%javamethodmodifiers Parameter::isSetUnits() const "
/**
   * Predicate returning <code>true</code> if the
   * \'units\' attribute of this {@link Parameter} is set.
   * <p>
   * @return <code>true</code> if the \'units\' attribute of this {@link Parameter} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note There is an important distinction to be made between <em>no units
assigned</em>, and assuming a value without units has any specific unit
such as <code>dimensionless</code>.  In SBML, default units are never
attributed to numbers, and numbers without units are not automatically
assumed to have the unit <code>dimensionless</code>.  Please consult the
relevant SBML specification document for a more in-depth explanation of
this topic and the SBML unit system.

   */ public
";


%javamethodmodifiers Parameter::isSetConstant() const "
/**
   * Predicate returning <code>true</code> if the
   * \'constant\' attribute of this {@link Parameter} is set.
   * <p>
   * @return <code>true</code> if the \'constant\' attribute of this {@link Parameter} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note Readers who view the documentation for {@link LocalParameter} may be
   * confused about the presence of this method.  LibSBML derives
   * {@link LocalParameter} from {@link Parameter}; however, this does not precisely match
   * the object hierarchy defined by SBML Level&nbsp;3, where
   * {@link LocalParameter} is derived directly from {@link SBase} and not {@link Parameter}.  We
   * believe this arrangement makes it easier for libSBML users to program
   * applications that work with both SBML Level&nbsp;2 and SBML
   * Level&nbsp;3, but programmers should also keep in mind this difference
   * exists.  A side-effect of libSBML\'s scheme is that certain methods on
   * {@link LocalParameter} that are inherited from {@link Parameter} do not actually have
   * relevance to {@link LocalParameter} objects.  An example of this is the
   * methods pertaining to {@link Parameter}\'s attribute \'constant\' (i.e.,
   * isSetConstant(), setConstant(), and getConstant()).
   * <p>
   * @see #getConstant()
   * @see #setConstant(boolean flag)
   */ public
";


%javamethodmodifiers Parameter::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link Parameter}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link Parameter}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Parameter::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link Parameter}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link Parameter}
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Parameter::setValue(double value) "
/**
   * Sets the \'value\' attribute of this {@link Parameter} to the given <code>double</code>
   * value and marks the attribute as set.
   * <p>
   * @param value a <code>double</code>, the value to assign
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   */ public
";


%javamethodmodifiers Parameter::setUnits(const std::string& units) "
/**
   * Sets the \'units\' attribute of this {@link Parameter} to a copy of the given
   * units identifier <code>units</code>.
   * <p>
   * @param units a string, the identifier of the units to assign to this
   * {@link Parameter} instance
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Parameter::setConstant(bool flag) "
/**
   * Sets the \'constant\' attribute of this {@link Parameter} to the given boolean
   * <code>flag</code>.
   * <p>
   * @param flag a boolean, the value for the \'constant\' attribute of this
   * {@link Parameter} instance
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note Readers who view the documentation for {@link LocalParameter} may be
   * confused about the presence of this method.  LibSBML derives
   * {@link LocalParameter} from {@link Parameter}; however, this does not precisely match
   * the object hierarchy defined by SBML Level&nbsp;3, where
   * {@link LocalParameter} is derived directly from {@link SBase} and not {@link Parameter}.  We
   * believe this arrangement makes it easier for libSBML users to program
   * applications that work with both SBML Level&nbsp;2 and SBML
   * Level&nbsp;3, but programmers should also keep in mind this difference
   * exists.  A side-effect of libSBML\'s scheme is that certain methods on
   * {@link LocalParameter} that are inherited from {@link Parameter} do not actually have
   * relevance to {@link LocalParameter} objects.  An example of this is the
   * methods pertaining to {@link Parameter}\'s attribute \'constant\' (i.e.,
   * isSetConstant(), setConstant(), and getConstant()).
   * <p>
   * @see #getConstant()
   * @see #isSetConstant()
   */ public
";


%javamethodmodifiers Parameter::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link Parameter}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Parameter::unsetValue "
/**
   * Unsets the \'value\' attribute of this {@link Parameter} instance.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * In SBML Level&nbsp;1 Version&nbsp;1, parameters are required to have
   * values and therefore, the value of a {@link Parameter} <b>should always be
   * set</b>.  In SBML Level&nbsp;1 Version&nbsp;2 and beyond, the value
   * is optional and as such, the \'value\' attribute may or may not be set.
   */ public
";


%javamethodmodifiers Parameter::unsetUnits "
/**
   * Unsets the \'units\' attribute of this {@link Parameter} instance.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Parameter::getDerivedUnitDefinition "
/**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Parameter}\'s value.
   * <p>
   * Parameters in SBML have an attribute (\'units\') for declaring the units
   * of measurement intended for the parameter\'s value.  <b>No defaults are
   * defined</b> by SBML in the absence of a definition for \'units\'.  This
   * method returns a {@link UnitDefinition} object based on the units declared for
   * this {@link Parameter} using its \'units\' attribute, or it returns <code>null</code> if
   * no units have been declared.
   * <p>
   * Note that unit declarations for {@link Parameter} objects are specified in
   * terms of the <em>identifier</em> of a unit (e.g., using setUnits()), but
   * <em>this</em> method returns a {@link UnitDefinition} object, not a unit
   * identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.For SBML Level&nbsp;2 models, it will do this even when
   * the value of the \'units\' attribute is one of the special SBML
   * Level&nbsp;2 unit identifiers <code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>,
   * <code>\'length\'</code> or <code>\'time\'.</code>  Callers may find this useful in conjunction
   * with the helper methods provided by the {@link UnitDefinition} class for
   * comparing different {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Parameter}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @note The libSBML system for unit analysis depends on the model as a
   * whole.  In cases where the {@link Parameter} object has not yet been added to
   * a model, or the model itself is incomplete, unit analysis is not
   * possible, and consequently this method will return <code>null.</code>
   * <p>
   * @see #isSetUnits()
   */ public
";


%javamethodmodifiers Parameter::getDerivedUnitDefinition() const "
/**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link Parameter}\'s value.
   * <p>
   * Parameters in SBML have an attribute (\'units\') for declaring the units
   * of measurement intended for the parameter\'s value.  <b>No defaults are
   * defined</b> by SBML in the absence of a definition for \'units\'.  This
   * method returns a {@link UnitDefinition} object based on the units declared for
   * this {@link Parameter} using its \'units\' attribute, or it returns <code>null</code> if
   * no units have been declared.
   * <p>
   * Note that unit declarations for {@link Parameter} objects are specified in
   * terms of the <em>identifier</em> of a unit (e.g., using setUnits()), but
   * <em>this</em> method returns a {@link UnitDefinition} object, not a unit
   * identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  For SBML Level&nbsp;2 models, it will do this even
   * when the value of the \'units\' attribute is one of the predefined SBML
   * units <code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>, <code>\'length\'</code> or 
   * <code>\'time\'.</code>  Callers may find this useful in conjunction with the helper
   * methods provided by the {@link UnitDefinition} class for comparing different
   * {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link Parameter}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @note The libSBML system for unit analysis depends on the model as a
   * whole.  In cases where the {@link Parameter} object has not yet been added to
   * a model, or the model itself is incomplete, unit analysis is not
   * possible, and consequently this method will return <code>null.</code>
   * <p>
   * @see #isSetUnits()
   */ public
";


%javamethodmodifiers Parameter::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or
   * {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers Parameter::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link Parameter}, is
   * always <code>\'parameter\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'parameter\'.</code>
   */ public
";


%javamethodmodifiers Parameter::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Parameter::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link Parameter} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link Parameter} object are:
   * <ul>
   * <li> \'id\' (or \'name\' in SBML Level&nbsp;1)
   * <li> \'value\' (required in Level&nbsp;1, optional otherwise)
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers Parameter::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers Parameter::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers Parameter::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Parameter::readL1Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Parameter::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Parameter::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Parameter::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Parameter::isExplicitlySetConstant() const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfParameters::ListOfParameters(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfParameters} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfParameters::ListOfParameters(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfParameters} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfParameters} object to be created.
   */ public
";


%javamethodmodifiers ListOfParameters::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfParameters} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfParameters}.
   */ public
";


%javamethodmodifiers ListOfParameters::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link 
   * libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfParameters::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Parameter} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN}
   * (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfParameters::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfParameters}, the XML element name is <code>\'listOfParameters\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfParameters\'.</code>
   */ public
";


%javamethodmodifiers ListOfParameters::get(unsigned int n) "
/**
   * Returns the {@link Parameter} object located at position <code>n</code> within this
   * {@link ListOfParameters} instance.
   * <p>
   * @param n the index number of the {@link Parameter} to get.
   * <p>
   * @return the nth {@link Parameter} in this {@link ListOfParameters}.  If the index <code>n</code>
   * is out of bounds for the length of the list, then <code>null</code> is returned.
   * <p>
   * @see #size()
   * @see #get(String sid)
   */ public
";


%javamethodmodifiers ListOfParameters::get "
/**
   * Returns the {@link Parameter} object located at position <code>n</code> within this
   * {@link ListOfParameters} instance.
   * <p>
   * @param n the index number of the {@link Parameter} to get.
   * <p>
   * @return the nth {@link Parameter} in this {@link ListOfParameters}.  If the index <code>n</code>
   * is out of bounds for the length of the list, then <code>null</code> is returned.
   * <p>
   * @see #size()
   * @see #get(String sid)
   */ public
";


%javamethodmodifiers ListOfParameters::get(const std::string& sid) "
/**
   * Returns the first {@link Parameter} object matching the given identifier.
   * <p>
   * @param sid a string, the identifier of the {@link Parameter} to get.
   * <p>
   * @return the {@link Parameter} object found.  The caller owns the returned
   * object and is responsible for deleting it.  If none of the items have
   * an identifier matching <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfParameters::get(const std::string& sid) const "
/**
   * Returns the first {@link Parameter} object matching the given identifier.
   * <p>
   * @param sid a string representing the identifier of the {@link Parameter} to
   * get.
   * <p>
   * @return the {@link Parameter} object found.  The caller owns the returned
   * object and is responsible for deleting it.  If none of the items have
   * an identifier matching <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfParameters::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfParameters}, and returns a pointer
   * to it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @return the item removed.  The caller owns the returned object and is
   * responsible for deleting it.  If the index number <code>n</code> is out of
   * bounds for the length of the list, then <code>null</code> is returned.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfParameters::remove(const std::string& sid) "
/**
   * Removes the first {@link Parameter} object in this {@link ListOfParameters}
   * matching the given identifier, and returns a pointer to it.
   * <p>
   * @param sid the identifier of the item to remove.
   * <p>
   * @return the item removed.  The caller owns the returned object and is
   * responsible for deleting it.  If none of the items have an identifier
   * matching <code>sid</code>, then <code>null</code> is returned.
   */ public
";


%javamethodmodifiers ListOfParameters::getElementPosition() const "
/**
   * Gets the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the {@link ListOfParameters}
   * in a model is (in SBML Level&nbsp;2 Version&nbsp;4) the seventh
   * ListOf___.  (However, it differs for different Levels and Versions of
   * SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfParameters::createObject(XMLInputStream& stream) "
/**
   * Create a {@link ListOfParameters} object corresponding to the next token in
   * the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) LocalParameter "
/** 
 * LibSBML implementation of SBML Level&nbsp;3\'s LocalParameter construct.
 * <p>
 * {@link LocalParameter} has been introduced in SBML Level&nbsp;3 to serve as the
 * object class for parameter definitions that are intended to be local to
 * a {@link Reaction}.  Objects of class {@link LocalParameter} never appear at the {@link Model}
 * level; they are always contained within {@link ListOfLocalParameters} lists
 * which are in turn contained within {@link KineticLaw} objects.
 * <p>
 * Like its global {@link Parameter} counterpart, the {@link LocalParameter} object class
 * is used to define a symbol associated with a value; this symbol can then
 * be used in a model\'s mathematical formulas (and specifically, for
 * {@link LocalParameter}, reaction rate formulas).  Unlike {@link Parameter}, the
 * {@link LocalParameter} class does not have a \'constant\' attribute: local
 * parameters within reactions are <em>always</em> constant.
 * <p>
 * {@link LocalParameter} has one required attribute, \'id\', to give the
 * parameter a unique identifier by which other parts of an SBML model
 * definition can refer to it.  A parameter can also have an optional
 * \'name\' attribute of type <code>string.</code>  Identifiers and names must be used
 * according to the guidelines described in the SBML specifications.
 * <p>
 * The optional attribute \'value\' determines the value (of type <code>double</code>)
 * assigned to the parameter.  A missing value for \'value\' implies that
 * the value either is unknown, or to be obtained from an external source,
 * or determined by an initial assignment.  The unit of measurement
 * associated with the value of the parameter can be specified using the
 * optional attribute \'units\'.  Here we only mention briefly some notable
 * points about the possible unit choices, but readers are urged to consult
 * the SBML specification documents for more information:
 * <ul>
 * <p>
 * <li> In SBML Level&nbsp;3, there are no constraints on the units that
 * can be assigned to parameters in a model; there are also no units to
 * inherit from the enclosing {@link Model} object.
 * <p>
 * <li> In SBML Level&nbsp;2, the value assigned to the parameter\'s \'units\'
 * attribute must be chosen from one of the following possibilities: one of
 * the base unit identifiers defined in SBML; one of the built-in unit
 * identifiers <code>\'substance\'</code>, <code>\'time\'</code>, <code>\'volume\'</code>, <code>\'area\'</code> or 
 * <code>\'length\'</code>; or the identifier of a new unit defined in the list of unit
 * definitions in the enclosing {@link Model} structure.  There are no constraints
 * on the units that can be chosen from these sets.  There are no default
 * units for local parameters.
 * </ul>
 * <p>
 * As with all other major SBML components, {@link LocalParameter} is derived from
 * {@link SBase}, and the methods defined on {@link SBase} are available on {@link LocalParameter}.
 * <p>
 * @warning LibSBML derives {@link LocalParameter} from {@link Parameter}; however, this
 * does not precisely match the object hierarchy defined by SBML
 * Level&nbsp;3, where {@link LocalParameter} is derived directly from {@link SBase} and not
 * {@link Parameter}.  We believe this arrangement makes it easier for libSBML
 * users to program applications that work with both SBML Level&nbsp;2 and
 * SBML Level&nbsp;3, but programmers should also keep in mind this
 * difference exists.  A side-effect of libSBML\'s scheme is that certain
 * methods on {@link LocalParameter} that are inherited from {@link Parameter} do not
 * actually have relevance to {@link LocalParameter} objects.  An example of this
 * is the methods pertaining to {@link Parameter}\'s attribute \'constant\'
 * (i.e., isSetConstant(), setConstant(), and getConstant()).
 * <p>
 * @see ListOfLocalParameters
 * @see KineticLaw
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%typemap(javaimports) ListOfLocalParameters "
/** 
 * LibSBML implementation of SBML Level&nbsp;3\'s ListOfLocalParameters construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * {@link ListOfLocalParameters} is a subsidiary object class used only within
 * {@link KineticLaw} in SBML Level&nbsp;3.  It is not defined in SBML Levels
 * 1&ndash;2.  In Level&nbsp;3, a {@link KineticLaw} object can have a single
 * object of class {@link ListOfLocalParameters} containing a set of local
 * parameters used in that kinetic law definition.
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers LocalParameter::LocalParameter(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link LocalParameter} object with the given SBML <code>level</code> and
   * <code>version</code> values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this
   * {@link LocalParameter}.
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link LocalParameter}.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link LocalParameter} object to an {@link SBMLDocument}
   * (e.g., using {@link KineticLaw#addLocalParameter(LocalParameter p)}), the SBML Level, SBML
   * Version and XML namespace of the document <em>override</em> the
   * values used when creating the {@link LocalParameter} object via this
   * constructor.  This is necessary to ensure that an SBML document is a
   * consistent structure.  Nevertheless, the ability to supply the values
   * at the time of creation of a {@link LocalParameter} is an important aid to
   * producing valid SBML.  Knowledge of the intented SBML Level and
   * Version determine whether it is valid to assign a particular value to
   * an attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers LocalParameter::LocalParameter(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link LocalParameter} object with the given {@link SBMLNamespaces}
   * object <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a {@link LocalParameter} is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor if no identifier is provided as an argument.  Setting the
   * identifier can be accomplished using the method
   * setId(String id).
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link LocalParameter} object to an {@link SBMLDocument}
   * (e.g., using {@link KineticLaw#addLocalParameter(LocalParameter p)}), the SBML XML namespace of
   * the document <em>overrides</em> the value used when creating the
   * {@link LocalParameter} object via this constructor.  This is necessary to
   * ensure that an SBML document is a consistent structure.  Nevertheless,
   * the ability to supply the values at the time of creation of a
   * {@link LocalParameter} is an important aid to producing valid SBML.  Knowledge
   * of the intented SBML Level and Version determine whether it is valid
   * to assign a particular value to an attribute, or whether it is valid
   * to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers LocalParameter::LocalParameter(const LocalParameter& orig) "
/**
   * Copy constructor; creates a copy of a given {@link LocalParameter} object.
   * <p>
   * @param orig the {@link LocalParameter} instance to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers LocalParameter::LocalParameter(const Parameter& orig) "
/**
   * Copy constructor; creates a {@link LocalParameter} object by copying
   * the attributes of a given {@link Parameter} object.
   * <p>
   * @param orig the {@link Parameter} instance to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers LocalParameter::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link LocalParameter}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link LocalParameter} in the list
   * of parameters within which this {@link LocalParameter} is embedded (i.e., either
   * the list of parameters in the parent {@link Model} or the list of parameters
   * in the enclosing {@link KineticLaw}).
   */ public
";


%javamethodmodifiers LocalParameter::clone() const "
/**
   * Creates and returns a deep copy of this {@link LocalParameter}.
   * <p>
   * @return a (deep) copy of this {@link LocalParameter}.
   */ public
";


%javamethodmodifiers LocalParameter::getDerivedUnitDefinition "
/**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link LocalParameter}\'s value.
   * <p>
   * LocalParameters in SBML have an attribute (\'units\') for declaring the
   * units of measurement intended for the parameter\'s value.  <b>No
   * defaults are defined</b> by SBML in the absence of a definition for
   * \'units\'.  This method returns a {@link UnitDefinition} object based on the
   * units declared for this {@link LocalParameter} using its \'units\' attribute, or
   * it returns <code>null</code> if no units have been declared.
   * <p>
   * Note that unit declarations for {@link LocalParameter} objects are specified
   * in terms of the <em>identifier</em> of a unit (e.g., using setUnits()), but
   * <em>this</em> method returns a {@link UnitDefinition} object, not a unit
   * identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  For SBML Level&nbsp;2 models, it will do this even
   * when the value of the \'units\' attribute is one of the predefined SBML
   * units <code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>, <code>\'length\'</code> or 
   * <code>\'time\'.</code>  Callers may find this useful in conjunction with the helper
   * methods provided by the {@link UnitDefinition} class for comparing different
   * {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link LocalParameter}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @note The libSBML system for unit analysis depends on the model as a
   * whole.  In cases where the {@link LocalParameter} object has not yet been
   * added to a model, or the model itself is incomplete, unit analysis is
   * not possible, and consequently this method will return <code>null.</code>
   * <p>
   * @see #isSetUnits()
   */ public
";


%javamethodmodifiers LocalParameter::getDerivedUnitDefinition() const "
/**
   * Constructs and returns a {@link UnitDefinition} that corresponds to the units
   * of this {@link LocalParameter}\'s value.
   * <p>
   * LocalParameters in SBML have an attribute (\'units\') for declaring the
   * units of measurement intended for the parameter\'s value.  <b>No
   * defaults are defined</b> by SBML in the absence of a definition for
   * \'units\'.  This method returns a {@link UnitDefinition} object based on the
   * units declared for this {@link LocalParameter} using its \'units\' attribute, or
   * it returns <code>null</code> if no units have been declared.
   * <p>
   * Note that unit declarations for {@link LocalParameter} objects are specified
   * in terms of the <em>identifier</em> of a unit (e.g., using setUnits()), but
   * <em>this</em> method returns a {@link UnitDefinition} object, not a unit
   * identifier.  It does this by constructing an appropriate
   * {@link UnitDefinition}.  For SBML Level&nbsp;2 models, it will do this even
   * when the value of the \'units\' attribute is one of the predefined SBML
   * units <code>\'substance\'</code>, <code>\'volume\'</code>, <code>\'area\'</code>, <code>\'length\'</code> or 
   * <code>\'time\'.</code>  Callers may find this useful in conjunction with the helper
   * methods provided by the {@link UnitDefinition} class for comparing different
   * {@link UnitDefinition} objects.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of this 
   * {@link LocalParameter}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @note The libSBML system for unit analysis depends on the model as a
   * whole.  In cases where the {@link LocalParameter} object has not yet been
   * added to a model, or the model itself is incomplete, unit analysis is
   * not possible, and consequently this method will return <code>null.</code>
   * <p>
   * @see #isSetUnits()
   */ public
";


%javamethodmodifiers LocalParameter::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link 
   * libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers LocalParameter::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link LocalParameter},
   * is always <code>\'localParameter\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'localParameter\'.</code>
   */ public
";


%javamethodmodifiers LocalParameter::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link LocalParameter} object
   * have been set.
   * <p>
   * @note The required attributes for a {@link LocalParameter} object are:
   * <ul>
   * <li> \'id\'
   * <li> \'value\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers LocalParameter::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers LocalParameter::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers LocalParameter::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers LocalParameter::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfLocalParameters::ListOfLocalParameters(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfLocalParameters} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfLocalParameters::ListOfLocalParameters(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfLocalParameters} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfLocalParameters} object to be created.
   */ public
";


%javamethodmodifiers ListOfLocalParameters::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfLocalParameters} object.
   * <p>
   * @return a (deep) copy of this {@link ListOfLocalParameters}.
   */ public
";


%javamethodmodifiers ListOfLocalParameters::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link 
   * libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfLocalParameters::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link LocalParameter} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN}
   * (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfLocalParameters::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfLocalParameters}, the XML element name is <code>\'listOfLocalParameters\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfLocalParameters\'.</code>
   */ public
";


%javamethodmodifiers ListOfLocalParameters::get(unsigned int n) "
/**
   * Returns the {@link LocalParameter} object located at position <code>n</code> within this
   * {@link ListOfLocalParameters} instance.
   * <p>
   * @param n the index number of the {@link LocalParameter} to get.
   * <p>
   * @return the nth {@link LocalParameter} in this {@link ListOfLocalParameters}.  If the
   * index <code>n</code> is out of bounds for the length of the list, then <code>null</code>
   * is returned.
   * <p>
   * @see #size()
   * @see #get(String sid)
   */ public
";


%javamethodmodifiers ListOfLocalParameters::get "
/**
   * Returns the {@link LocalParameter} object located at position <code>n</code> within this
   * {@link ListOfLocalParameters} instance.
   * <p>
   * @param n the index number of the {@link LocalParameter} to get.
   * <p>
   * @return the item at position <code>n</code>.  The caller owns the returned
   * object and is responsible for deleting it.  If the index number <code>n</code>
   * is out of bounds for the length of the list, then <code>null</code> is returned.
   * <p>
   * @see #size()
   * @see #get(String sid)
   */ public
";


%javamethodmodifiers ListOfLocalParameters::get(const std::string& sid) "
/**
   * Returns the first {@link LocalParameter} object matching the given identifier.
   * <p>
   * @param sid a string, the identifier of the {@link LocalParameter} to get.
   * <p>
   * @return the {@link LocalParameter} object found.  The caller owns the returned
   * object and is responsible for deleting it.  If none of the items have
   * an identifier matching <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfLocalParameters::get(const std::string& sid) const "
/**
   * Returns the first {@link LocalParameter} object matching the given identifier.
   * <p>
   * @param sid a string representing the identifier of the {@link LocalParameter}
   * to get.
   * <p>
   * @return the {@link LocalParameter} object found.  The caller owns the returned
   * object and is responsible for deleting it.  If none of the items have
   * an identifier matching <code>sid</code>, then <code>null</code> is returned.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfLocalParameters::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.  Note that LocalParameters, while they use the SId namespace, are not in the model-wide SId namespace, so no {@link LocalParameter} object will be returned from this function (and is the reason we override the base ListOf.getElementBySId function here).
   * <p>
   * @param id string representing the id of objects to find
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers ListOfLocalParameters::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfLocalParameters}, and returns a
   * pointer to it.
   * <p>
   * @param n the index of the item to remove.  
   * <p>
   * @return the item removed.  The caller owns the returned object and is
   * responsible for deleting it.  If the index number <code>n</code> is out of
   * bounds for the length of the list, then <code>null</code> is returned.
   * <p>
   * @see #size()
   * @see #remove(String sid)
   */ public
";


%javamethodmodifiers ListOfLocalParameters::remove(const std::string& sid) "
/**
   * Removes the first {@link LocalParameter} object in this {@link ListOfLocalParameters}
   * matching the given identifier, and returns a pointer to it.
   * <p>
   * @param sid the identifier of the item to remove.
   * <p>
   * @return the item removed.  The caller owns the returned object and is
   * responsible for deleting it.  If none of the items have an identifier
   * matching <code>sid</code>, then <code>null</code> is returned.
   */ public
";


%javamethodmodifiers ListOfLocalParameters::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the {@link ListOfLocalParameters}
   * in a model is (in SBML Level&nbsp;2 Version&nbsp;4) the seventh
   * ListOf___.  (However, it differs for different Levels and Versions of
   * SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfLocalParameters::createObject(XMLInputStream& stream) "
/**
   * Create a {@link ListOfLocalParameters} object corresponding to the next token in
   * the XML input stream.
   * <p>
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream, or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) InitialAssignment "
/** 
 * LibSBML implementation of SBML\'s InitialAssignment construct.
 * <p>
 * SBML Level 2 Versions 2&ndash;4 and SBML Level&nbsp;3 provide two ways of assigning initial
 * values to entities in a model.  The simplest and most basic is to set
 * the values of the appropriate attributes in the relevant components; for
 * example, the initial value of a model parameter (whether it is a
 * constant or a variable) can be assigned by setting its \'value\' attribute
 * directly in the model definition.  However, this approach is not
 * suitable when the value must be calculated, because the initial value
 * attributes on different components such as species, compartments, and
 * parameters are single values and not mathematical expressions.  In those
 * situations, the {@link InitialAssignment} construct can be used; it permits the
 * calculation of the value of a constant or the initial value of a
 * variable from the values of <em>other</em> quantities in a model.
 * <p>
 * As explained below, the provision of {@link InitialAssignment} does not mean
 * that models necessarily must use this construct when defining initial
 * values of quantities in a model.  If a value can be set directly using
 * the relevant attribute of a component in a model, then that
 * approach may be more efficient and more portable to other software
 * tools.  {@link InitialAssignment} should be used when the other mechanism is
 * insufficient for the needs of a particular model.
 * <p>
 * The {@link InitialAssignment} construct has some similarities to {@link AssignmentRule}.
 * The main differences are: (a) an {@link InitialAssignment} can set the value of
 * a constant whereas an {@link AssignmentRule} cannot, and (b) unlike
 * {@link AssignmentRule}, an {@link InitialAssignment} definition only applies up to and
 * including the beginning of simulation time, i.e., <em>t &#8804; 0</em>,
 * while an {@link AssignmentRule} applies at all times.
 * <p>
 * {@link InitialAssignment} has a required attribute, \'symbol\', whose value must
 * follow the guidelines for identifiers described in the SBML
 * specification (e.g., Section 3.3 in the Level 2 Version 4
 * specification).  The value of this attribute in an {@link InitialAssignment}
 * object can be the identifier of a {@link Compartment}, {@link Species} or global
 * {@link Parameter} elsewhere in the model.  The {@link InitialAssignment} defines the
 * initial value of the constant or variable referred to by the \'symbol\'
 * attribute.  (The attribute\'s name is \'symbol\' rather than \'variable\'
 * because it may assign values to constants as well as variables in a
 * model.)  Note that an initial assignment cannot be made to reaction
 * identifiers, that is, the \'symbol\' attribute value of an
 * {@link InitialAssignment} cannot be an identifier that is the \'id\' attribute
 * value of a {@link Reaction} object in the model.  This is identical to a
 * restriction placed on rules.
 * <p>
 * {@link InitialAssignment} also has a required \'math\' subelement that contains a
 * MathML expression used to calculate the value of the constant or the
 * initial value of the variable.  The units of the value computed by the
 * formula in the \'math\' subelement should (in SBML Level&nbsp;2
 * Version&nbsp;4 and in SBML Level&nbsp;3) or must (in previous Versions) be identical to be the
 * units associated with the identifier given in the \'symbol\' attribute.
 * (That is, the units are the units of the species, compartment, or
 * parameter, as appropriate for the kind of object identified by the value
 * of \'symbol\'.)
 * <p>
 * {@link InitialAssignment} was introduced in SBML Level 2 Version 2.  It is not
 * available in SBML Level&nbsp;2 Version&nbsp;1 nor in any version of Level 1.
 * <p>
 * <h2>Semantics of Initial Assignments</h2>
 * <p>
 * The value calculated by an {@link InitialAssignment} object overrides the value
 * assigned to the given symbol by the object defining that symbol.  For
 * example, if a compartment\'s \'size\' attribute is set in its definition,
 * and the model also contains an {@link InitialAssignment} having that
 * compartment\'s identifier as its \'symbol\' attribute value, then the
 * interpretation is that the \'size\' assigned in the {@link Compartment} object
 * should be ignored and the value assigned based on the computation
 * defined in the {@link InitialAssignment}.  Initial assignments can take place
 * for {@link Compartment}, {@link Species} and global {@link Parameter} objects regardless of the
 * value of their \'constant\' attribute.
 * <p>
 * The actions of all {@link InitialAssignment} objects are in general terms
 * the same, but differ in the precise details depending on the type
 * of variable being set:
 * <ul>
 * <li> <em>In the case of a species</em>, an {@link InitialAssignment} sets the
 * referenced species\' initial quantity (concentration or amount of
 * substance) to the value determined by the formula in the \'math\'
 * subelement.    The overall units of the formula should (in SBML
 * Level&nbsp;2 Version&nbsp;4 and in SBML Level&nbsp;3) or must (in previous Versions) be the same
 * as the units specified for the species.
 * <p>
 * <li> <em>In the case of a compartment</em>, an {@link InitialAssignment} sets
 * the referenced compartment\'s initial size to the size determined by the
 * formula in \'math\'.  The overall units of the formula should (in SBML
 * Level&nbsp;2 Version&nbsp;4 and in SBML Level&nbsp;3) or must (in previous Versions) be the same
 * as the units specified for the size of the compartment.
 * <p>
 * <li> <em>In the case of a parameter</em>, an {@link InitialAssignment} sets the
 * referenced parameter\'s initial value to that determined by the formula
 * in \'math\'.  The overall units of the formula should (in SBML
 * Level&nbsp;2 Version&nbsp;4 and SBML Level&nbsp;3) or must (in previous Versions) be the same
 * as the units defined for the parameter.  </ul>
 * <p>
 * In the context of a simulation, initial assignments establish values
 * that are in effect prior to and including the start of simulation time,
 * i.e., <em>t &#8804; 0</em>.  Section 3.4.8 in the SBML Level 2
 * Version 4  and SBML Level&nbsp;3 Version&nbsp;1 Core specifications provides information about the interpretation of
 * assignments, rules, and entity values for simulation time up to and
 * including the start time <em>t = 0</em>; this is important for
 * establishing the initial conditions of a simulation if the model
 * involves expressions containing the <em>delay</em> \'csymbol\'.
 * <p>
 * There cannot be two initial assignments for the same symbol in a model;
 * that is, a model must not contain two or more {@link InitialAssignment} objects
 * that both have the same identifier as their \'symbol\' attribute value.  A
 * model must also not define initial assignments <em>and</em> assignment
 * rules for the same entity.  That is, there cannot be <em>both</em> an
 * {@link InitialAssignment} and an {@link AssignmentRule} for the same symbol in a model,
 * because both kinds of constructs apply prior to and at the start of
 * simulated time&mdash;allowing both to exist for a given symbol would
 * result in indeterminism).
 * <p>
 * The ordering of {@link InitialAssignment} objects is not significant.  The
 * combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
 * objects form a set of assignment statements that must be considered as a
 * whole.  The combined set of assignment statements should not contain
 * algebraic loops: a chain of dependency between these statements should
 * terminate.  (More formally, consider the directed graph of assignment
 * statements where nodes are a model\'s assignment statements and directed
 * arcs exist for each occurrence of a symbol in an assignment statement
 * \'math\' attribute.  The directed arcs in this graph start from the
 * statement assigning the symbol and end at the statement that contains
 * the symbol in their math elements.  Such a graph must be acyclic.)
 * <p>
 * Finally, it is worth being explicit about the expected behavior in the
 * following situation.  Suppose (1) a given symbol has a value <em>x</em>
 * assigned to it in its definition, and (2) there is an initial assignment
 * having the identifier as its \'symbol\' value and reassigning the value to
 * <em>y</em>, <em>and</em> (3) the identifier is also used in the
 * mathematical formula of a second initial assignment.  What value should
 * the second initial assignment use?  It is <em>y</em>, the value assigned
 * to the symbol by the first initial assignment, not whatever value was
 * given in the symbol\'s definition.  This follows directly from the
 * behavior described above: if an {@link InitialAssignment} object exists for a
 * given symbol, then the symbol\'s value is overridden by that initial
 * assignment.
 * <p>
 * <!---------------------------------------------------------------------- -->
 * <p>
 */
"


%typemap(javaimports) ListOfInitialAssignments "
/** 
 * LibSBML implementation of SBML\'s ListOfInitialAssignments construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers InitialAssignment::InitialAssignment(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link InitialAssignment} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link InitialAssignment}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link InitialAssignment}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link InitialAssignment} object to an
   * {@link SBMLDocument} (e.g., using {@link Model#addInitialAssignment(InitialAssignment ia)}), the SBML
   * Level, SBML Version and XML namespace of the document 
   * <em>override</em> the values used when creating the {@link InitialAssignment} object
   * via this constructor.  This is necessary to ensure that an SBML
   * document is a consistent structure.  Nevertheless, the ability to
   * supply the values at the time of creation of a {@link InitialAssignment} is an
   * important aid to producing valid SBML.  Knowledge of the intented SBML
   * Level and Version determine whether it is valid to assign a particular
   * value to an attribute, or whether it is valid to add an object to an
   * existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers InitialAssignment::InitialAssignment(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link InitialAssignment} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link InitialAssignment} object to an
   * {@link SBMLDocument} (e.g., using {@link Model#addInitialAssignment(InitialAssignment ia)}), the SBML XML
   * namespace of the document <em>overrides</em> the value used when creating
   * the {@link InitialAssignment} object via this constructor.  This is necessary
   * to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link InitialAssignment} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers InitialAssignment::InitialAssignment(const InitialAssignment& orig) "
/**
   * Copy constructor; creates a copy of this {@link InitialAssignment}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers InitialAssignment::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link InitialAssignment}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link InitialAssignment} in
   * the list of compartment types.
   */ public
";


%javamethodmodifiers InitialAssignment::clone() const "
/**
   * Creates and returns a deep copy of this {@link InitialAssignment}.
   * <p>
   * @return a (deep) copy of this {@link InitialAssignment}.
   */ public
";


%javamethodmodifiers InitialAssignment::getSymbol() const "
/**
   * Get the value of the \'symbol\' attribute of this {@link InitialAssignment}.
   * <p>
   * @return the identifier string stored as the \'symbol\' attribute value
   * in this {@link InitialAssignment}.
   */ public
";


%javamethodmodifiers InitialAssignment::getMath() const "
/**
   * Get the mathematical formula of this {@link InitialAssignment}.
   * <p>
   * @return an {@link ASTNode}, the value of the \'math\' subelement of this
   * {@link InitialAssignment}
   */ public
";


%javamethodmodifiers InitialAssignment::isSetSymbol() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link InitialAssignment}\'s \'symbol\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'symbol\' attribute of this {@link InitialAssignment}
   * is set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers InitialAssignment::isSetMath() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link InitialAssignment}\'s \'math\' subelement contains a value.
   * <p>
   * @return <code>true</code> if the \'math\' for this {@link InitialAssignment} is set,
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers InitialAssignment::setSymbol(const std::string& sid) "
/**
   * Sets the \'symbol\' attribute value of this {@link InitialAssignment}.
   * <p>
   * @param sid the identifier of a {@link Species}, {@link Compartment} or {@link Parameter}
   * object defined elsewhere in this {@link Model}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers InitialAssignment::setMath(const ASTNode* math) "
/**
   * Sets the \'math\' subelement of this {@link InitialAssignment}.
   * <p>
   * The AST passed in <code>math</code> is copied.
   * <p>
   * @param math an AST containing the mathematical expression to
   * be used as the formula for this {@link InitialAssignment}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   */ public
";


%javamethodmodifiers InitialAssignment::getDerivedUnitDefinition "
/**
   * Calculates and returns a {@link UnitDefinition} that expresses the units
   * of measurement assumed for the \'math\' expression of this
   * {@link InitialAssignment}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link InitialAssignment} and the model quantities referenced by
   * <code>&lt;ci&gt;</code> elements used within that expression.  The
   * {@link InitialAssignment#getDerivedUnitDefinition()}
   * method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return <code>null.</code>
   * <p>
   * @warning Note that it is possible the \'math\' expression in the
   * {@link InitialAssignment} contains pure numbers or parameters with undeclared
   * units.  In those cases, it is not possible to calculate the units of
   * the overall expression without making assumptions.  LibSBML does not
   * make assumptions about the units, and
   * {@link InitialAssignment#getDerivedUnitDefinition()}
   * only returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong>
   * {@link InitialAssignment#containsUndeclaredUnits()}
   * <strong>to determine whether this situation holds</strong>.  Callers
   * may wish to take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link InitialAssignment}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @see #containsUndeclaredUnits()
   */ public
";


%javamethodmodifiers InitialAssignment::getDerivedUnitDefinition() const "
/**
   * Calculates and returns a {@link UnitDefinition} that expresses the units
   * of measurement assumed for the \'math\' expression of this
   * {@link InitialAssignment}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link InitialAssignment} and the model quantities referenced by
   * <code>&lt;ci&gt;</code> elements used within that expression.  The
   * {@link InitialAssignment#getDerivedUnitDefinition()}
   * method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return <code>null.</code>
   * <p>
   * @warning Note that it is possible the \'math\' expression in the
   * {@link InitialAssignment} contains pure numbers or parameters with undeclared
   * units.  In those cases, it is not possible to calculate the units of
   * the overall expression without making assumptions.  LibSBML does not
   * make assumptions about the units, and
   * {@link InitialAssignment#getDerivedUnitDefinition()}
   * only returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong>
   * {@link InitialAssignment#containsUndeclaredUnits()}
   * <strong>to determine whether this situation holds</strong>.  Callers
   * may wish to take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link InitialAssignment}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @see #containsUndeclaredUnits()
   */ public
";


%javamethodmodifiers InitialAssignment::containsUndeclaredUnits "
/**
   * Predicate returning <code>true</code> if 
   * the math expression of this {@link InitialAssignment} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link InitialAssignment}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by
   * {@link InitialAssignment#getDerivedUnitDefinition()}
   * may not accurately represent the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */ public
";


%javamethodmodifiers InitialAssignment::containsUndeclaredUnits() const "
/**
   * Predicate returning <code>true</code> if 
   * the math expression of this {@link InitialAssignment} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link InitialAssignment}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by
   * {@link InitialAssignment#getDerivedUnitDefinition()}
   * may not accurately represent the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */ public
";


%javamethodmodifiers InitialAssignment::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers InitialAssignment::getElementName() const "
/**
   * Returns the XML element name of this object, which for
   * {@link InitialAssignment}, is always <code>\'initialAssignment\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'initialAssignment\'.</code>
   */ public
";


%javamethodmodifiers InitialAssignment::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers InitialAssignment::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link InitialAssignment} object
   * have been set.
   * <p>
   * @note The required attributes for an {@link InitialAssignment} object are:
   * <ul>
   * <li> \'symbol\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * attributes for this object have been defined.
   */ public
";


%javamethodmodifiers InitialAssignment::hasRequiredElements() const "
/**
   * Predicate returning <code>true</code> if
   * all the required elements for this {@link InitialAssignment} object
   * have been set.
   * <p>
   * @note The required elements for a {@link InitialAssignment} object are:
   * <ul>
   * <li> \'math\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers InitialAssignment::getId() const "
/**
   * Predicate returning <code>true</code> if
   * all the required elements for this {@link InitialAssignment} object
   * have been set.
   * <p>
   * @note The required elements for a {@link InitialAssignment} object are:
   * <ul>
   * <li> \'math\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers InitialAssignment::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%javamethodmodifiers InitialAssignment::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers InitialAssignment::replaceSIDWithFunction(const std::string& id, const ASTNode* function) "
/**
   * Replace all nodes with the name \'id\' from the child \'math\' object with the provided function. 
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers InitialAssignment::divideAssignmentsToSIdByFunction(const std::string& id, const ASTNode* function) "
/**
   * If this assignment assigns a value to the \'id\' element, replace the \'math\' object with the function (existing/function). 
   * @internal 
   */ public
";


%javamethodmodifiers InitialAssignment::multiplyAssignmentsToSIdByFunction(const std::string& id, const ASTNode* function) "
/**
   * If this assignment assigns a value to the \'id\' element, replace the \'math\' object with the function (existing*function). 
   * @internal 
   */ public
";


%javamethodmodifiers InitialAssignment::readOtherXML(XMLInputStream& stream) "
/**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   * @internal
   */ public
";


%javamethodmodifiers InitialAssignment::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers InitialAssignment::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers InitialAssignment::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers InitialAssignment::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers InitialAssignment::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::ListOfInitialAssignments(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfInitialAssignments} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::ListOfInitialAssignments(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfInitialAssignments} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfInitialAssignments} object to be created.
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfInitialAssignments} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfInitialAssignments}.
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link InitialAssignment} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfInitialAssignments}, the XML element name is 
   * <code>\'listOfInitialAssignments\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfInitialAssignments\'.</code>
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::get(unsigned int n) "
/**
   * Get a {@link InitialAssignment} from the {@link ListOfInitialAssignments}.
   * <p>
   * @param n the index number of the {@link InitialAssignment} to get.
   * <p>
   * @return the nth {@link InitialAssignment} in this {@link ListOfInitialAssignments}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::get "
/**
   * Get a {@link InitialAssignment} from the {@link ListOfInitialAssignments}.
   * <p>
   * @param n the index number of the {@link InitialAssignment} to get.
   * <p>
   * @return the nth {@link InitialAssignment} in this {@link ListOfInitialAssignments}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::get(const std::string& sid) "
/**
   * Get a {@link InitialAssignment} from the {@link ListOfInitialAssignments}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link InitialAssignment} to get.
   * <p>
   * @return {@link InitialAssignment} in this {@link ListOfInitialAssignments}
   * with the given id or <code>null</code> if no such
   * {@link InitialAssignment} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::get(const std::string& sid) const "
/**
   * Get a {@link InitialAssignment} from the {@link ListOfInitialAssignments}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link InitialAssignment} to get.
   * <p>
   * @return {@link InitialAssignment} in this {@link ListOfInitialAssignments}
   * with the given id or <code>null</code> if no such
   * {@link InitialAssignment} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfInitialAssignments} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::remove(const std::string& sid) "
/**
   * Removes item in this {@link ListOfInitialAssignments} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.  Note that InitialAssignments do not actually have IDs, though the {@link libsbml} interface pretends that they do:  no initial assignment is returned by this function.
   * <p>
   * @param id string representing the id of objects to find
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the
   * {@link ListOfInitialAssignments} in a model is (in SBML Level 2 Version 4)
   * the eighth ListOf___.  (However, it differs for different Levels and
   * Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfInitialAssignments::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) Rule "
/** 
 * LibSBML implementation of SBML\'s Rule construct.
 * <p>
 * In SBML, <em>rules</em> provide additional ways to define the values of
 * variables in a model, their relationships, and the dynamical behaviors
 * of those variables.  They enable encoding relationships that cannot be
 * expressed using {@link Reaction} nor {@link InitialAssignment} objects alone.
 * <p>
 * The libSBML implementation of rules mirrors the SBML Level&nbsp;3
 * Version&nbsp;1 Core definition (which is in turn is very similar to the
 * Level&nbsp;2 Version&nbsp;4 definition), with {@link Rule} being the parent
 * class of three subclasses as explained below.  The {@link Rule} class itself
 * cannot be instantiated by user programs and has no constructor; only the
 * subclasses {@link AssignmentRule}, {@link AlgebraicRule} and {@link RateRule} can be
 * instantiated directly.
 * <p>
 * <h2>General summary of SBML rules</h2>
 * <p>
 * In SBML Level&nbsp;3 as well as Level&nbsp;2, rules are separated into three
subclasses for the benefit of model analysis software.  The three
subclasses are based on the following three different possible functional
forms (where <em>x</em> is a variable, <em>f</em> is some arbitrary
function returning a numerical result, <b><em>V</em></b> is a vector of
variables that does not include <em>x</em>, and <b><em>W</em></b> is a
vector of variables that may include <em>x</em>):

<center>
<table border=\'0\' cellpadding=\'0\' style=\'font-size: small\'>
<tr><td width=\'120px\'><em>Algebraic:</em></td><td width=\'250px\'>left-hand side is zero</td><td><em>0 = f(<b>W</b>)</em></td></tr>
<tr><td><em>Assignment:</em></td><td>left-hand side is a scalar:</td><td><em>x = f(<b>V</b>)</em></td></tr>
<tr><td><em>Rate:</em></td><td>left-hand side is a rate-of-change:</td><td><em>dx/dt = f(<b>W</b>)</em></td></tr>
</table>
</center>

In their general form given above, there is little to distinguish
between <em>assignment</em> and <em>algebraic</em> rules.  They are treated as
separate cases for the following reasons:

<ul>
<li> <em>Assignment</em> rules can simply be evaluated to calculate
intermediate values for use in numerical methods.  They are statements
of equality that hold at all times.  (For assignments that are only
performed once, see {@link InitialAssignment}.)<p>

<li> SBML needs to place restrictions on assignment rules, for example
the restriction that assignment rules cannot contain algebraic loops.<p>

<li> Some simulators do not contain numerical solvers capable of solving
unconstrained algebraic equations, and providing more direct forms such
as assignment rules may enable those simulators to process models they
could not process if the same assignments were put in the form of
general algebraic equations;<p>

<li> Those simulators that <em>can</em> solve these algebraic equations make a
distinction between the different categories listed above; and<p>

<li> Some specialized numerical analyses of models may only be applicable
to models that do not contain <em>algebraic</em> rules.
</ul>

The approach taken to covering these cases in SBML is to define an
abstract {@link Rule} structure containing a subelement, \'math\', to hold the
right-hand side expression, then to derive subtypes of {@link Rule} that add
attributes to distinguish the cases of algebraic, assignment and rate
rules.  The \'math\' subelement must contain a MathML expression defining the
mathematical formula of the rule.  This MathML formula must return a
numerical value.  The formula can be an arbitrary expression referencing
the variables and other entities in an SBML model.

Each of the three subclasses of {@link Rule} (AssignmentRule, {@link AlgebraicRule},
{@link RateRule}) inherit the the \'math\' subelement and other fields from {@link SBase}.
The {@link AssignmentRule} and {@link RateRule} classes add an additional attribute,
\'variable\'.  See the definitions of {@link AssignmentRule}, {@link AlgebraicRule} and
{@link RateRule} for details about the structure and interpretation of each one.

 * <p>
 * <h2>Additional restrictions on SBML rules</h2>
 * <p>
 * An important design goal of SBML rule semantics is to ensure that a
model\'s simulation and analysis results will not be dependent on when or
how often rules are evaluated.  To achieve this, SBML needs to place two
restrictions on rule use.  The first concerns algebraic loops in the system
of assignments in a model, and the second concerns overdetermined systems.

<h3><a class=\'anchor\' id=\'no-algebraic-loops\'>A model must not contain
algebraic loops</a></h3>

The combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
objects in a model constitute a set of assignment statements that should be
considered as a whole.  (A {@link KineticLaw} object is counted as an assignment
because it assigns a value to the symbol contained in the \'id\' attribute of
the {@link Reaction} object in which it is defined.)  This combined set of
assignment statements must not contain algebraic loops&mdash;dependency
chains between these statements must terminate.  To put this more formally,
consider a directed graph in which nodes are assignment statements and
directed arcs exist for each occurrence of an SBML species, compartment or
parameter symbol in an assignment statement\'s \'math\' subelement.  Let the
directed arcs point from the statement assigning the symbol to the
statements that contain the symbol in their \'math\' subelement expressions.
This graph must be acyclic.

SBML does not specify when or how often rules should be evaluated.
Eliminating algebraic loops ensures that assignment statements can be
evaluated any number of times without the result of those evaluations
changing.  As an example, consider the set of equations <em>x = x + 1</em>,
<em>y = z + 200</em> and <em>z = y + 100</em>.  If this set of equations
were interpreted as a set of assignment statements, it would be invalid
because the rule for <em>x</em> refers to <em>x</em> (exhibiting one type
of loop), and the rule for <em>y</em> refers to <em>z</em> while the rule
for <em>z</em> refers back to <em>y</em> (exhibiting another type of loop).
Conversely, the following set of equations would constitute a valid set of
assignment statements: <em>x = 10</em>, <em>y = z + 200</em>, and <em>z = x
+ 100</em>.

<h3><a class=\'anchor\' id=\'no-overdetermined\'>A model must not be
overdetermined</a></h3>

An SBML model must not be overdetermined; that is, a model must not
define more equations than there are unknowns in a model.  An SBML model
that does not contain {@link AlgebraicRule} structures cannot be overdetermined.

LibSBML implements the static analysis procedure described in
Appendix&nbsp;B of the SBML Level&nbsp;3 Version&nbsp;1 Core
specification for assessing whether a model is overdetermined.

(In summary, assessing whether a given continuous, deterministic,
mathematical model is overdetermined does not require dynamic analysis; it
can be done by analyzing the system of equations created from the model.
One approach is to construct a bipartite graph in which one set of vertices
represents the variables and the other the set of vertices represents the
equations.  Place edges between vertices such that variables in the system
are linked to the equations that determine them.  For algebraic equations,
there will be edges between the equation and each variable occurring in the
equation.  For ordinary differential equations (such as those defined by
rate rules or implied by the reaction rate definitions), there will be a
single edge between the equation and the variable determined by that
differential equation.  A mathematical model is overdetermined if the
maximal matchings of the bipartite graph contain disconnected vertexes
representing equations.  If one maximal matching has this property, then
all the maximal matchings will have this property; i.e., it is only
necessary to find one maximal matching.)

 * <p>
 * <h2>Rule types for SBML Level 1</h2>
 * <p>
 * SBML Level 1 uses a different scheme than SBML Level 2 and Level 3 for
 * distinguishing rules; specifically, it uses an attribute whose value is
 * drawn from an enumeration of 3 values.  LibSBML supports this using methods
 * that work with the enumeration values  listed below.
 * <p>
 * <ul>
 * <li> {@link  libsbmlConstants#RULE_TYPE_RATE RULE_TYPE_RATE}: Indicates
 * the rule is a \'rate\' rule.
 * <li> {@link  libsbmlConstants#RULE_TYPE_SCALAR RULE_TYPE_SCALAR}:
 * Indicates the rule is a \'scalar\' rule.
 * <li> {@link  libsbmlConstants#RULE_TYPE_INVALID RULE_TYPE_INVALID}:
 * Indicates the rule type is unknown or not yet set.
 * </ul>
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%typemap(javaimports) ListOfRules "
/** 
 * LibSBML implementation of SBML\'s ListOfRules construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;3 and later versions
 * of SBML Level&nbsp;2 as well):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers Rule::Rule(const Rule& orig) "
/**
   * Copy constructor; creates a copy of this {@link Rule}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers Rule::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Rule}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link Rule} object in the
   * list of rules within which <em>the</em> present object is embedded.
   */ public
";


%javamethodmodifiers Rule::clone() const "
/**
   * Creates and returns a deep copy of this {@link Rule}.
   * <p>
   * @return a (deep) copy of this {@link Rule}.
   */ public
";


%javamethodmodifiers Rule::getFormula() const "
/**
   * Returns the mathematical expression of this {@link Rule} in text-string form.
   * <p>
   * The text string is produced by
   * <code><a href=\'libsbml.html#formulaToString(org.sbml.libsbml.ASTNode)\'>libsbml.formulaToString()</a></code>; please consult
   * the documentation for that function to find out more about the format
   * of the text-string formula.
   * <p>
   * @return the formula text string for this {@link Rule}.
   * <p>
   * @note The attribute \'formula\' is specific to SBML Level&nbsp;1; in
   * higher Levels of SBML, it has been replaced with a subelement named
   * \'math\'.  However, libSBML provides a unified interface to the
   * underlying math expression and this method can be used for models
   * of all Levels of SBML.
   * <p>
   * @see #getMath()
   */ public
";


%javamethodmodifiers Rule::getMath() const "
/**
   * Get the mathematical formula of this {@link Rule} as an {@link ASTNode} tree.
   * <p>
   * @return an {@link ASTNode}, the value of the \'math\' subelement of this {@link Rule}.
   * <p>
   * @note The subelement \'math\' is present in SBML Levels&nbsp;2
   * and&nbsp;3.  In SBML Level&nbsp;1, the equivalent construct is the
   * attribute named \'formula\'.  LibSBML provides a unified interface to
   * the underlying math expression and this method can be used for models
   * of all Levels of SBML.
   * <p>
   * @see #getFormula()
   */ public
";


%javamethodmodifiers Rule::getVariable() const "
/**
   * Get the value of the \'variable\' attribute of this {@link Rule} object.
   * <p>
   * In SBML Level&nbsp;1, the different rule types each have a different
   * name for the attribute holding the reference to the object
   * constituting the left-hand side of the rule.  (E.g., for
   * SBML Level&nbsp;1\'s SpeciesConcentrationRule the attribute is \'species\', for
   * CompartmentVolumeRule it is \'compartment\', etc.)  In SBML
   * Levels&nbsp;2 and&nbsp;3, the only two types of {@link Rule} objects with a
   * left-hand side object reference are {@link AssignmentRule} and {@link RateRule}, and
   * both of them use the same name for attribute: \'variable\'.  In order to
   * make it easier for application developers to work with all Levels of
   * SBML, libSBML uses a uniform name for all of such attributes, and it
   * is \'variable\', regardless of whether Level&nbsp;1 rules or
   * Level&nbsp;2&ndash;3 rules are being used.
   * <p>
   * @return the identifier string stored as the \'variable\' attribute value
   * in this {@link Rule}, or <code>null</code> if this object is an {@link AlgebraicRule} object.
   */ public
";


%javamethodmodifiers Rule::getUnits() const "
/**
   * Returns the units for the
   * mathematical formula of this {@link Rule}.
   * <p>
   * @return the identifier of the units for the expression of this {@link Rule}.
   * <p>
   * @note The attribute \'units\' exists on SBML Level&nbsp;1 ParameterRule
   * objects only.  It is not present in SBML Levels&nbsp;2 and&nbsp;3.
   */ public
";


%javamethodmodifiers Rule::isSetFormula() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Rule}\'s mathematical expression is set.
   * <p>
   * This method is equivalent to isSetMath().  This version is present for
   * easier compatibility with SBML Level&nbsp;1, in which mathematical
   * formulas were written in text-string form.
   * <p>
   * @return <code>true</code> if the mathematical formula for this {@link Rule} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The attribute \'formula\' is specific to SBML Level&nbsp;1; in
   * higher Levels of SBML, it has been replaced with a subelement named
   * \'math\'.  However, libSBML provides a unified interface to the
   * underlying math expression and this method can be used for models
   * of all Levels of SBML.
   * <p>
   * @see #isSetMath()
   */ public
";


%javamethodmodifiers Rule::isSetMath() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Rule}\'s mathematical expression is set.
   * <p>
   * This method is equivalent to isSetFormula().
   * <p>
   * @return <code>true</code> if the formula (or equivalently the math) for this
   * {@link Rule} is set, <code>false</code> otherwise.
   * <p>
   * @note The subelement \'math\' is present in SBML Levels&nbsp;2
   * and&nbsp;3.  In SBML Level&nbsp;1, the equivalent construct is the
   * attribute named \'formula\'.  LibSBML provides a unified interface to
   * the underlying math expression and this method can be used for models
   * of all Levels of SBML.
   * <p>
   * @see #isSetFormula()
   */ public
";


%javamethodmodifiers Rule::isSetVariable() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Rule}\'s \'variable\' attribute is set.
   * <p>
   * In SBML Level&nbsp;1, the different rule types each have a different
   * name for the attribute holding the reference to the object
   * constituting the left-hand side of the rule.  (E.g., for
   * SBML Level&nbsp;1\'s SpeciesConcentrationRule the attribute is \'species\', for
   * CompartmentVolumeRule it is \'compartment\', etc.)  In SBML
   * Levels&nbsp;2 and&nbsp;3, the only two types of {@link Rule} objects with a
   * left-hand side object reference are {@link AssignmentRule} and {@link RateRule}, and
   * both of them use the same name for attribute: \'variable\'.  In order to
   * make it easier for application developers to work with all Levels of
   * SBML, libSBML uses a uniform name for all such attributes, and it is
   * \'variable\', regardless of whether Level&nbsp;1 rules or
   * Level&nbsp;2&ndash;3 rules are being used.
   * <p>
   * @return <code>true</code> if the \'variable\' attribute value of this {@link Rule} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Rule::isSetUnits() const "
/**
   * Predicate returning <code>true</code>
   * if this {@link Rule}\'s \'units\' attribute is set.
   * <p>
   * @return <code>true</code> if the units for this {@link Rule} is set, <code>false</code>
   * otherwise
   * <p>
   * @note The attribute \'units\' exists on SBML Level&nbsp;1 ParameterRule
   * objects only.  It is not present in SBML Levels&nbsp;2 and&nbsp;3.
   */ public
";


%javamethodmodifiers Rule::setFormula(const std::string& formula) "
/**
   * Sets the \'math\' subelement of this {@link Rule} to an expression in
   * text-string form.
   * <p>
   * This is equivalent to setMath(ASTNode math).  The provision of
   * using text-string formulas is retained for easier SBML Level&nbsp;1
   * compatibility.  The formula is converted to an {@link ASTNode} internally.
   * <p>
   * @param formula a mathematical formula in text-string form.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   * <p>
   * @note The attribute \'formula\' is specific to SBML Level&nbsp;1; in
   * higher Levels of SBML, it has been replaced with a subelement named
   * \'math\'.  However, libSBML provides a unified interface to the
   * underlying math expression and this method can be used for models
   * of all Levels of SBML.
   * <p>
   * @see #setMath(ASTNode math)
   */ public
";


%javamethodmodifiers Rule::setMath(const ASTNode* math) "
/**
   * Sets the \'math\' subelement of this {@link Rule} to a copy of the given
   * {@link ASTNode}.
   * <p>
   * @param math the {@link ASTNode} structure of the mathematical formula.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   * <p>
   * @note The subelement \'math\' is present in SBML Levels&nbsp;2
   * and&nbsp;3.  In SBML Level&nbsp;1, the equivalent construct is the
   * attribute named \'formula\'.  LibSBML provides a unified interface to
   * the underlying math expression and this method can be used for models
   * of all Levels of SBML.
   * <p>
   * @see #setFormula(String formula)
   */ public
";


%javamethodmodifiers Rule::setVariable(const std::string& sid) "
/**
   * Sets the \'variable\' attribute value of this {@link Rule} object.
   * <p>
   * In SBML Level&nbsp;1, the different rule types each have a different
   * name for the attribute holding the reference to the object
   * constituting the left-hand side of the rule.  (E.g., for
   * SBML Level&nbsp;1\'s SpeciesConcentrationRule the attribute is \'species\', for
   * CompartmentVolumeRule it is \'compartment\', etc.)  In SBML
   * Levels&nbsp;2 and&nbsp;3, the only two types of {@link Rule} objects with a
   * left-hand side object reference are {@link AssignmentRule} and {@link RateRule}, and
   * both of them use the same name for attribute: \'variable\'.  In order to
   * make it easier for application developers to work with all Levels of
   * SBML, libSBML uses a uniform name for all such attributes, and it is
   * \'variable\', regardless of whether Level&nbsp;1 rules or
   * Level&nbsp;2&ndash;3 rules are being used.
   * <p>
   * @param sid the identifier of a {@link Compartment}, {@link Species} or {@link Parameter}
   * elsewhere in the enclosing {@link Model} object.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   */ public
";


%javamethodmodifiers Rule::setUnits(const std::string& sname) "
/**
   * Sets the units for this {@link Rule}.
   * <p>
   * @param sname the identifier of the units
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note The attribute \'units\' exists on SBML Level&nbsp;1 ParameterRule
   * objects only.  It is not present in SBML Levels&nbsp;2 and&nbsp;3.
   */ public
";


%javamethodmodifiers Rule::unsetUnits "
/**
   * Unsets the \'units\' for this {@link Rule}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * </ul>
   * <p>
   * @note The attribute \'units\' exists on SBML Level&nbsp;1 ParameterRule
   * objects only.  It is not present in SBML Levels&nbsp;2 and&nbsp;3.
   */ public
";


%javamethodmodifiers Rule::getDerivedUnitDefinition "
/**
   * Calculates and returns a {@link UnitDefinition} that expresses the units of
   * measurement assumed for the \'math\' expression of this {@link Rule}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link Rule} and the model quantities referenced by <code>&lt;ci&gt;</code>
   * elements used within that expression.  The getDerivedUnitDefinition()
   * method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return <code>null.</code>
   * <p>
   * @warning Note that it is possible the \'math\' expression in the {@link Rule}
   * contains pure numbers or parameters with undeclared units.  In those
   * cases, it is not possible to calculate the units of the overall
   * expression without making assumptions.  LibSBML does not make
   * assumptions about the units, and getDerivedUnitDefinition() only
   * returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong>
   * {@link Rule#containsUndeclaredUnits()}
   * <strong>to determine whether this situation holds</strong>.  Callers may
   * wish to take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link Rule}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @see #containsUndeclaredUnits()
   */ public
";


%javamethodmodifiers Rule::getDerivedUnitDefinition() const "
/**
   * Calculates and returns a {@link UnitDefinition} that expresses the units of
   * measurement assumed for the \'math\' expression of this {@link Rule}.
   * <p>
   * The units are calculated based on the mathematical expression in the
   * {@link Rule} and the model quantities referenced by <code>&lt;ci&gt;</code>
   * elements used within that expression.  The getDerivedUnitDefinition()
   * method returns the calculated units.
   * <p>
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return <code>null.</code>
   * <p>
   * @warning Note that it is possible the \'math\' expression in the {@link Rule}
   * contains pure numbers or parameters with undeclared units.  In those
   * cases, it is not possible to calculate the units of the overall
   * expression without making assumptions.  LibSBML does not make
   * assumptions about the units, and getDerivedUnitDefinition() only
   * returns the units as far as it is able to determine them.  For
   * example, in an expression <em>X + Y</em>, if <em>X</em> has
   * unambiguously-defined units and <em>Y</em> does not, it will return
   * the units of <em>X</em>.  <strong>It is important that callers also
   * invoke the method</strong>
   * {@link Rule#containsUndeclaredUnits()}
   * <strong>to determine whether this situation holds</strong>.  Callers
   * may wish to take suitable actions in those scenarios.
   * <p>
   * @return a {@link UnitDefinition} that expresses the units of the math 
   * expression of this {@link Rule}, or <code>null</code> if one cannot be constructed.
   * <p>
   * @see #containsUndeclaredUnits()
   */ public
";


%javamethodmodifiers Rule::containsUndeclaredUnits "
/**
   * Predicate returning <code>true</code> if 
   * the math expression of this {@link Rule} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link Rule}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by getDerivedUnitDefinition() may not accurately represent
   * the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */ public
";


%javamethodmodifiers Rule::containsUndeclaredUnits() const "
/**
   * Predicate returning <code>true</code> if 
   * the math expression of this {@link Rule} contains
   * parameters/numbers with undeclared units.
   * <p>
   * @return <code>true</code> if the math expression of this {@link Rule}
   * includes parameters/numbers 
   * with undeclared units, <code>false</code> otherwise.
   * <p>
   * @note A return value of <code>true</code> indicates that the {@link UnitDefinition}
   * returned by getDerivedUnitDefinition() may not accurately represent
   * the units of the expression.
   * <p>
   * @see #getDerivedUnitDefinition()
   */ public
";


%javamethodmodifiers Rule::getType() const "
/**
   * Get the type of rule this is.
   * <p>
   * @return the rule type (a value drawn from the enumeration <a
   * class=\'el\' href=\'#RuleType_t\'>RuleType_t</a>) of this {@link Rule}.  The value
   * will be either {@link  libsbmlConstants#RULE_TYPE_RATE RULE_TYPE_RATE}
   * or {@link  libsbmlConstants#RULE_TYPE_SCALAR RULE_TYPE_SCALAR}.
   * <p>
   * @note The attribute \'type\' on {@link Rule} objects is present only in SBML
   * Level&nbsp;1.  In SBML Level&nbsp;2 and later, the type has been
   * replaced by subclassing the {@link Rule} object.
   */ public
";


%javamethodmodifiers Rule::isAlgebraic() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Rule} is an {@link AlgebraicRule}.
   * <p>
   * @return <code>true</code> if this {@link Rule} is an {@link AlgebraicRule}, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Rule::isAssignment() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Rule} is an {@link AssignmentRule}.
   * <p>
   * @return <code>true</code> if this {@link Rule} is an {@link AssignmentRule}, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Rule::isCompartmentVolume() const "
/**
   * Predicate returning <code>true</code> if this {@link Rule} is an CompartmentVolumeRule
   * or equivalent.
   * <p>
   * This libSBML method works for SBML Level&nbsp;1 models (where there is
   * such a thing as an explicit CompartmentVolumeRule), as well as other Levels of
   * SBML.  For Levels above Level&nbsp;1, this method checks the symbol
   * being affected by the rule, and returns <code>true</code> if the symbol is the
   * identifier of a {@link Compartment} object defined in the model.
   * <p>
   * @return <code>true</code> if this {@link Rule} is a CompartmentVolumeRule, <code>false</code>
   * otherwise.
   */ public
";


%javamethodmodifiers Rule::isParameter() const "
/**
   * Predicate returning <code>true</code> if this {@link Rule} is an ParameterRule or
   * equivalent.
   * <p>
   * This libSBML method works for SBML Level&nbsp;1 models (where there is
   * such a thing as an explicit ParameterRule), as well as other Levels of
   * SBML.  For Levels above Level&nbsp;1, this method checks the symbol
   * being affected by the rule, and returns <code>true</code> if the symbol is the
   * identifier of a {@link Parameter} object defined in the model.
   * <p>
   * @return <code>true</code> if this {@link Rule} is a ParameterRule, <code>false</code>
   * otherwise.
   */ public
";


%javamethodmodifiers Rule::isRate() const "
/**
   * Predicate returning <code>true</code> if this {@link Rule}
   * is a {@link RateRule} (SBML Levels&nbsp;2&ndash;3) or has a \'type\' attribute
   * value of <code>\'rate\'</code> (SBML Level&nbsp;1).
   * <p>
   * @return <code>true</code> if this {@link Rule} is a {@link RateRule} (Level&nbsp;2) or has
   * type \'rate\' (Level&nbsp;1), <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Rule::isScalar() const "
/**
   * Predicate returning <code>true</code> if this {@link Rule}
   * is an {@link AssignmentRule} (SBML Levels&nbsp;2&ndash;3) or has a \'type\'
   * attribute value of <code>\'scalar\'</code> (SBML Level&nbsp;1).
   * <p>
   * @return <code>true</code> if this {@link Rule} is an {@link AssignmentRule} (Level&nbsp;2) or has
   * type \'scalar\' (Level&nbsp;1), <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Rule::isSpeciesConcentration() const "
/**
   * Predicate returning <code>true</code> if this {@link Rule} is a
   * SpeciesConcentrationRule or equivalent.
   * <p>
   * This libSBML method works for SBML Level&nbsp;1 models (where there is
   * such a thing as an explicit SpeciesConcentrationRule), as well as
   * other Levels of SBML.  For Levels above Level&nbsp;1, this method
   * checks the symbol being affected by the rule, and returns <code>true</code> if
   * the symbol is the identifier of a {@link Species} object defined in the model.
   * <p>
   * @return <code>true</code> if this {@link Rule} is a SpeciesConcentrationRule, <code>false</code>
   * otherwise.
   */ public
";


%javamethodmodifiers Rule::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link 
   * libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers Rule::getL1TypeCode() const "
/**
   * Returns the SBML Level&nbsp;1 type code for this {@link Rule} object.
   * <p>
   * This method only applies to SBML Level&nbsp;1 model objects.  If this
   * is not an SBML Level&nbsp;1 rule object, this method will return {@link 
   * libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN}.
   * <p>
   * @return the SBML Level&nbsp;1 type code for this {@link Rule} (namely, {@link 
   * libsbmlConstants#SBML_COMPARTMENT_VOLUME_RULE
   * SBML_COMPARTMENT_VOLUME_RULE}, {@link 
   * libsbmlConstants#SBML_PARAMETER_RULE SBML_PARAMETER_RULE}, {@link 
   * libsbmlConstants#SBML_SPECIES_CONCENTRATION_RULE
   * SBML_SPECIES_CONCENTRATION_RULE}, or {@link 
   * libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN}).
   */ public
";


%javamethodmodifiers Rule::getElementName() const "
/**
   * Returns the XML element name of this object
   * <p>
   * The returned value can be any of a number of different strings,
   * depending on the SBML Level in use and the kind of {@link Rule} object this
   * is.  The rules as of libSBML version 5.6.0
   * are the following:
   * <ul>
   * <li> (Level&nbsp;2 and&nbsp;3) RateRule: returns <code>\'rateRule\'</code>
   * <li> (Level&nbsp;2 and&nbsp;3) AssignmentRule: returns <code>\'assignmentRule\'</code> 
   * <li> (Level&nbsp;2 and&nbsp;3) AlgebraicRule: returns <code>\'algebraicRule\'</code>
   * <li> (Level&nbsp;1 Version&nbsp;1) SpecieConcentrationRule: returns <code>\'specieConcentrationRule\'</code>
   * <li> (Level&nbsp;1 Version&nbsp;2) SpeciesConcentrationRule: returns <code>\'speciesConcentrationRule\'</code>
   * <li> (Level&nbsp;1) CompartmentVolumeRule: returns <code>\'compartmentVolumeRule\'</code>
   * <li> (Level&nbsp;1) ParameterRule: returns <code>\'parameterRule\'</code>
   * <li> Unknown rule type: returns <code>\'unknownRule\'</code>
   * </ul>
   * <p>
   * Beware that the last (<code>\'unknownRule\'</code>) is not a valid SBML element
   * name.
   * <p>
   * @return the name of this element
   */ public
";


%javamethodmodifiers Rule::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Rule::setL1TypeCode(int type) "
/**
   * Sets the SBML Level&nbsp;1 type code for this {@link Rule}.
   * <p>
   * @param type the SBML Level&nbsp;1 type code for this {@link Rule}. The
   * allowable values are {@link  libsbmlConstants#SBML_COMPARTMENT_VOLUME_RULE
   * SBML_COMPARTMENT_VOLUME_RULE}, {@link 
   * libsbmlConstants#SBML_PARAMETER_RULE SBML_PARAMETER_RULE}, and
   * {@link  libsbmlConstants#SBML_SPECIES_CONCENTRATION_RULE
   * SBML_SPECIES_CONCENTRATION_RULE}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.  The possible values returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS}
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE}
   * if given <code>type</code> value is not one of the above.
   * </ul>
   */ public
";


%javamethodmodifiers Rule::hasRequiredElements() const "
/**
   * Predicate returning <code>true</code> if all the
   * required elements for this {@link Rule} object have been set.
   * <p>
   * The only required element for a {@link Rule} object is the \'math\' subelement.
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers Rule::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if all the
   * required attributes for this {@link Rule} object have been set.
   * <p>
   * The required attributes for a {@link Rule} object depend on the type of {@link Rule}
   * it is.  For {@link AssignmentRule} and {@link RateRule} objects (and SBML
   * Level&nbsp1\'s SpeciesConcentrationRule, CompartmentVolumeRule, and
   * ParameterRule objects), the required attribute is \'variable\'; for
   * {@link AlgebraicRule} objects, there is no required attribute.
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers Rule::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%javamethodmodifiers Rule::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers Rule::getInternalId() const "
/**
   * Renames all the UnitSIdRef attributes on this element
   * @internal
   */ public
";


%javamethodmodifiers Rule::setInternalId(std::string id) "
/**
   * Renames all the UnitSIdRef attributes on this element
   * @internal
   */ public
";


%javamethodmodifiers Rule::getId() const "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers Rule::replaceSIDWithFunction(const std::string& id, const ASTNode* function) "
/**
   * Replace all nodes with the name \'id\' from the child \'math\' object with the provided function. 
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers Rule::divideAssignmentsToSIdByFunction(const std::string& id, const ASTNode* function) "
/**
   * If this rule assigns a value or a change to the \'id\' element, replace the \'math\' object with the function (existing/function). 
   * @internal 
   */ public
";


%javamethodmodifiers Rule::multiplyAssignmentsToSIdByFunction(const std::string& id, const ASTNode* function) "
/**
   * If this assignment assigns a value to the \'id\' element, replace the \'math\' object with the function (existing*function). 
   * @internal 
   */ public
";


%javamethodmodifiers Rule::Rule(  int      type , unsigned int        level , unsigned int        version ) "
/**
   * Only subclasses may create Rules.
   * @internal
   */ public
";


%javamethodmodifiers Rule::Rule(  int      type , SBMLNamespaces *    sbmlns ) "
/**
   * Only subclasses may create Rules.
   * @internal
   */ public
";


%javamethodmodifiers Rule::readOtherXML(XMLInputStream& stream) "
/**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   * @internal
   */ public
";


%javamethodmodifiers Rule::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers Rule::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Rule::readL1Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Rule::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Rule::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Rule::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfRules::ListOfRules(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfRules} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfRules::ListOfRules(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfRules} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfRules} object to be created.
   */ public
";


%javamethodmodifiers ListOfRules::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfRules} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfRules}.
   */ public
";


%javamethodmodifiers ListOfRules::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfRules::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Rule} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfRules::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfRules}, the XML element name is <code>\'listOfRules\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfRules\'.</code>
   */ public
";


%javamethodmodifiers ListOfRules::get(unsigned int n) "
/**
   * Get a {@link Rule} from the {@link ListOfRules}.
   * <p>
   * @param n the index number of the {@link Rule} to get.
   * <p>
   * @return the nth {@link Rule} in this {@link ListOfRules}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfRules::get "
/**
   * Get a {@link Rule} from the {@link ListOfRules}.
   * <p>
   * @param n the index number of the {@link Rule} to get.
   * <p>
   * @return the nth {@link Rule} in this {@link ListOfRules}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfRules::get(const std::string& sid) "
/**
   * Get a {@link Rule} from the {@link ListOfRules}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Rule} to get.
   * <p>
   * @return {@link Rule} in this {@link ListOfRules}
   * with the given id or <code>null</code> if no such
   * {@link Rule} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfRules::get(const std::string& sid) const "
/**
   * Get a {@link Rule} from the {@link ListOfRules}
   * based on its identifier.
   * <p>
   * @param sid a string representing the identifier 
   * of the {@link Rule} to get.
   * <p>
   * @return {@link Rule} in this {@link ListOfRules}
   * with the given id or <code>null</code> if no such
   * {@link Rule} exists.
   * <p>
   * @see #get(long n)
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfRules::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfRules} items and returns a pointer to
   * it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfRules::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.  Note that AssignmentRules and RateRules do not actually have IDs, but the {@link libsbml} interface pretends that they do:  no assignment rule or rate rule is returned by this function.
   * <p>
   * @param id string representing the id of objects to find
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers ListOfRules::remove(const std::string& sid) "
/**
   * Removes item in this {@link ListOfRules} items with the given identifier.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier <code>sid</code>, then 
   * <code>null</code> is returned.
   * <p>
   * @param sid the identifier of the item to remove
   * <p>
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public
";


%javamethodmodifiers ListOfRules::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfRules::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%javamethodmodifiers ListOfRules::isValidTypeForList(SBase * item) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) AlgebraicRule "
/** 
 * LibSBML implementation of SBML\'s AlgebraicRule construct.
 * <p>
 * The rule type {@link AlgebraicRule} is derived from the parent class {@link Rule}.  It
 * is used to express equations that are neither assignments of model
 * variables nor rates of change.  {@link AlgebraicRule} does not add any
 * attributes to the basic {@link Rule}; its role is simply to distinguish this
 * case from the other cases.
 * <p>
 * In the context of a simulation, algebraic rules are in effect at all
 * times, <em>t</em> &#8805; <em>0</em>.  For purposes of evaluating
 * expressions that involve the delay \'csymbol\' (see the SBML
 * specification), algebraic rules are considered to apply also at
 * <em>t</em> &#8804; <em>0</em>.  Please consult the relevant SBML
 * specification for additional information about the semantics of
 * assignments, rules, and entity values for simulation time <em>t</em>
 * &#8804; <em>0</em>.
 * <p>
 * An SBML model must not be overdetermined.  The ability to define
 * arbitrary algebraic expressions in an SBML model introduces the
 * possibility that a model is mathematically overdetermined by the overall
 * system of equations constructed from its rules, reactions and events.
 * Therefore, if an algebraic rule is introduced in a model, for at least
 * one of the entities referenced in the rule\'s \'math\' element the value of
 * that entity must not be completely determined by other constructs in the
 * model.  This means that at least this entity must not have the attribute
 * \'constant\'=<code>true</code> and there must also not be a rate rule or assignment
 * rule for it.  Furthermore, if the entity is a {@link Species} object, its value
 * must not be determined by reactions, which means that it must either
 * have the attribute \'boundaryCondition\'=<code>true</code> or else not be involved
 * in any reaction at all.  These restrictions are explained in more detail
 * in the SBML specification documents.
 * <p>
 * In SBML Levels 2 and&nbsp;3, {@link Reaction} object identifiers can be
 * referenced in the \'math\' expression of an algebraic rule, but reaction
 * rates can never be <em>determined</em> by algebraic rules.  This is true
 * even when a reaction does not contain a {@link KineticLaw} element.  (In such
 * cases of missing {@link KineticLaw} elements, the model is valid but incomplete;
 * the rates of reactions lacking kinetic laws are simply undefined, and
 * not determined by the algebraic rule.)
 * <p>
 * <h2>General summary of SBML rules
</h2>
 * <p>
 * In SBML Level&nbsp;3 as well as Level&nbsp;2, rules are separated into three
subclasses for the benefit of model analysis software.  The three
subclasses are based on the following three different possible functional
forms (where <em>x</em> is a variable, <em>f</em> is some arbitrary
function returning a numerical result, <b><em>V</em></b> is a vector of
variables that does not include <em>x</em>, and <b><em>W</em></b> is a
vector of variables that may include <em>x</em>):

<center>
<table border=\'0\' cellpadding=\'0\' style=\'font-size: small\'>
<tr><td width=\'120px\'><em>Algebraic:</em></td><td width=\'250px\'>left-hand side is zero</td><td><em>0 = f(<b>W</b>)</em></td></tr>
<tr><td><em>Assignment:</em></td><td>left-hand side is a scalar:</td><td><em>x = f(<b>V</b>)</em></td></tr>
<tr><td><em>Rate:</em></td><td>left-hand side is a rate-of-change:</td><td><em>dx/dt = f(<b>W</b>)</em></td></tr>
</table>
</center>

In their general form given above, there is little to distinguish
between <em>assignment</em> and <em>algebraic</em> rules.  They are treated as
separate cases for the following reasons:

<ul>
<li> <em>Assignment</em> rules can simply be evaluated to calculate
intermediate values for use in numerical methods.  They are statements
of equality that hold at all times.  (For assignments that are only
performed once, see {@link InitialAssignment}.)<p>

<li> SBML needs to place restrictions on assignment rules, for example
the restriction that assignment rules cannot contain algebraic loops.<p>

<li> Some simulators do not contain numerical solvers capable of solving
unconstrained algebraic equations, and providing more direct forms such
as assignment rules may enable those simulators to process models they
could not process if the same assignments were put in the form of
general algebraic equations;<p>

<li> Those simulators that <em>can</em> solve these algebraic equations make a
distinction between the different categories listed above; and<p>

<li> Some specialized numerical analyses of models may only be applicable
to models that do not contain <em>algebraic</em> rules.
</ul>

The approach taken to covering these cases in SBML is to define an
abstract {@link Rule} structure containing a subelement, \'math\', to hold the
right-hand side expression, then to derive subtypes of {@link Rule} that add
attributes to distinguish the cases of algebraic, assignment and rate
rules.  The \'math\' subelement must contain a MathML expression defining the
mathematical formula of the rule.  This MathML formula must return a
numerical value.  The formula can be an arbitrary expression referencing
the variables and other entities in an SBML model.

Each of the three subclasses of {@link Rule} (AssignmentRule, {@link AlgebraicRule},
{@link RateRule}) inherit the the \'math\' subelement and other fields from {@link SBase}.
The {@link AssignmentRule} and {@link RateRule} classes add an additional attribute,
\'variable\'.  See the definitions of {@link AssignmentRule}, {@link AlgebraicRule} and
{@link RateRule} for details about the structure and interpretation of each one.

 * <p>
 * <h2>Additional restrictions on SBML rules
</h2>
 * <p>
 * An important design goal of SBML rule semantics is to ensure that a
model\'s simulation and analysis results will not be dependent on when or
how often rules are evaluated.  To achieve this, SBML needs to place two
restrictions on rule use.  The first concerns algebraic loops in the system
of assignments in a model, and the second concerns overdetermined systems.

<h3><a class=\'anchor\' id=\'no-algebraic-loops\'>A model must not contain
algebraic loops</a></h3>

The combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
objects in a model constitute a set of assignment statements that should be
considered as a whole.  (A {@link KineticLaw} object is counted as an assignment
because it assigns a value to the symbol contained in the \'id\' attribute of
the {@link Reaction} object in which it is defined.)  This combined set of
assignment statements must not contain algebraic loops&mdash;dependency
chains between these statements must terminate.  To put this more formally,
consider a directed graph in which nodes are assignment statements and
directed arcs exist for each occurrence of an SBML species, compartment or
parameter symbol in an assignment statement\'s \'math\' subelement.  Let the
directed arcs point from the statement assigning the symbol to the
statements that contain the symbol in their \'math\' subelement expressions.
This graph must be acyclic.

SBML does not specify when or how often rules should be evaluated.
Eliminating algebraic loops ensures that assignment statements can be
evaluated any number of times without the result of those evaluations
changing.  As an example, consider the set of equations <em>x = x + 1</em>,
<em>y = z + 200</em> and <em>z = y + 100</em>.  If this set of equations
were interpreted as a set of assignment statements, it would be invalid
because the rule for <em>x</em> refers to <em>x</em> (exhibiting one type
of loop), and the rule for <em>y</em> refers to <em>z</em> while the rule
for <em>z</em> refers back to <em>y</em> (exhibiting another type of loop).
Conversely, the following set of equations would constitute a valid set of
assignment statements: <em>x = 10</em>, <em>y = z + 200</em>, and <em>z = x
+ 100</em>.

<h3><a class=\'anchor\' id=\'no-overdetermined\'>A model must not be
overdetermined</a></h3>

An SBML model must not be overdetermined; that is, a model must not
define more equations than there are unknowns in a model.  An SBML model
that does not contain {@link AlgebraicRule} structures cannot be overdetermined.

LibSBML implements the static analysis procedure described in
Appendix&nbsp;B of the SBML Level&nbsp;3 Version&nbsp;1 Core
specification for assessing whether a model is overdetermined.

(In summary, assessing whether a given continuous, deterministic,
mathematical model is overdetermined does not require dynamic analysis; it
can be done by analyzing the system of equations created from the model.
One approach is to construct a bipartite graph in which one set of vertices
represents the variables and the other the set of vertices represents the
equations.  Place edges between vertices such that variables in the system
are linked to the equations that determine them.  For algebraic equations,
there will be edges between the equation and each variable occurring in the
equation.  For ordinary differential equations (such as those defined by
rate rules or implied by the reaction rate definitions), there will be a
single edge between the equation and the variable determined by that
differential equation.  A mathematical model is overdetermined if the
maximal matchings of the bipartite graph contain disconnected vertexes
representing equations.  If one maximal matching has this property, then
all the maximal matchings will have this property; i.e., it is only
necessary to find one maximal matching.)

 * <p>
 * <h2>Rule types for SBML Level 1
</h2>
 * <p>
 * SBML Level 1 uses a different scheme than SBML Level 2 and Level 3 for
 * distinguishing rules; specifically, it uses an attribute whose value is
 * drawn from an enumeration of 3 values.  LibSBML supports this using methods
 * that work with the enumeration values  listed below.
 * <p>
 * <ul>
 * <li> {@link  libsbmlConstants#RULE_TYPE_RATE RULE_TYPE_RATE}: Indicates
 * the rule is a \'rate\' rule.
 * <li> {@link  libsbmlConstants#RULE_TYPE_SCALAR RULE_TYPE_SCALAR}:
 * Indicates the rule is a \'scalar\' rule.
 * <li> {@link  libsbmlConstants#RULE_TYPE_INVALID RULE_TYPE_INVALID}:
 * Indicates the rule type is unknown or not yet set.
 * </ul>
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%javamethodmodifiers AlgebraicRule::AlgebraicRule(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link AlgebraicRule} using the given SBML <code>level</code> and <code>version
</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link AlgebraicRule}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link AlgebraicRule}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of an {@link AlgebraicRule} object to an {@link SBMLDocument}
   * (e.g., using&nbsp; {@link Model#addRule(Rule r)}), the SBML Level, SBML Version
   * and XML namespace of the document <em>override</em> the values used
   * when creating the {@link AlgebraicRule} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link AlgebraicRule} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers AlgebraicRule::AlgebraicRule(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link AlgebraicRule} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link AlgebraicRule} object to an {@link SBMLDocument}
   * (e.g., using&nbsp; {@link Model#addRule(Rule r)}, the SBML XML namespace of the
   * document <em>overrides</em> the value used when creating the {@link AlgebraicRule}
   * object via this constructor.  This is necessary to ensure that an SBML
   * document is a consistent structure.  Nevertheless, the ability to
   * supply the values at the time of creation of a {@link AlgebraicRule} is an
   * important aid to producing valid SBML.  Knowledge of the intented SBML
   * Level and Version determine whether it is valid to assign a particular
   * value to an attribute, or whether it is valid to add an object to an
   * existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers AlgebraicRule::clone() const "
/**
   * Creates and returns a deep copy of this {@link Rule}.
   * <p>
   * @return a (deep) copy of this {@link Rule}.
   */ public
";


%javamethodmodifiers AlgebraicRule::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link AlgebraicRule}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link AlgebraicRule} object
   * in the list of rules within which <em>the</em> present object is embedded.
   */ public
";


%javamethodmodifiers AlgebraicRule::setInternalIdOnly "
/**
   * sets the mInternalIdOnly flag
   * @internal
   */ public
";


%javamethodmodifiers AlgebraicRule::getInternalIdOnly() const "
/**
   * sets the mInternalIdOnly flag
   * @internal
   */ public
";


%javamethodmodifiers AlgebraicRule::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link AlgebraicRule} object
   * have been set.
   * <p>
   * @note In SBML Levels&nbsp;2&ndash;3, there is no required attribute
   * for an {@link AlgebraicRule} object.  For Level&nbsp;1, the only required
   * attribute is \'formula\'.
   * <p>
   * @return <code>true</code> if the required attributes have been set, <code>false
</code>
   * otherwise.
   */ public
";


%typemap(javaimports) AssignmentRule "
/** 
 * LibSBML implementation of SBML\'s AssignmentRule construct.
 * <p>
 * The rule type {@link AssignmentRule} is derived from the parent class {@link Rule}.  It
 * is used to express equations that set the values of variables.  The
 * left-hand side (the attribute named \'variable\') of an assignment rule
 * can refer to the identifier of a {@link Species}, {@link SpeciesReference} (in SBML
 * Level&nbsp;3), {@link Compartment}, or {@link Parameter} object in the model (but not a
 * {@link Reaction}).  The entity identified must have its \'constant\' attribute set
 * to <code>false.</code>  The effects of an {@link AssignmentRule} are in general terms the
 * same, but differ in the precise details depending on the type of
 * variable being set: <ul>

<p>
 * <li> <em>In the case of a species</em>, an {@link AssignmentRule} sets the
 * referenced species\' quantity (whether a \'concentration\' or \'amount\') to
 * the value determined by the formula in the MathML subelement \'math\'.
 * The unit associated with the value produced by the \'math\' formula <em>
</em>
 * should (in SBML Level&nbsp;2 Version&nbsp;4 and in SBML Level&nbsp;3) or <em>must</em> (in
 * SBML releases prior to Level&nbsp;2 version&nbsp;4) be equal to the unit
 * associated with the species\' quantity.  <em>Restrictions</em>: There
 * must not be both an {@link AssignmentRule} \'variable\' attribute and a
 * {@link SpeciesReference} \'species\' attribute having the same value, unless the
 * referenced {@link Species} object has its \'boundaryCondition\' attribute set to
 * <code>true.</code>  In other words, an assignment rule cannot be defined for a
 * species that is created or destroyed in a reaction unless that species
 * is defined as a boundary condition in the model.
 * <p>
 * <li> (For SBML Level&nbsp;3 only) <em>In the case of a species
 * reference</em>, an {@link AssignmentRule} sets the stoichiometry of the
 * referenced reactant or product to the value determined by the formula in
 * \'math\'.  The unit associated with the value produced by the \'math\'
 * formula should be consistent with the unit \'dimensionless\', because
 * reactant and product stoichiometries in reactions are dimensionless
 * quantities.
  * <p>
 * <li> <em>In the case of a compartment</em>, an {@link AssignmentRule} sets the
 * referenced compartment\'s size to the value determined by the formula in
 * the \'math\' subelement of the {@link AssignmentRule} object.  The overall units
 * of the formula in \'math\' <em>should</em> (in SBML Level&nbsp;2 Version&nbsp;4
 * and in SBML Level&nbsp;3) or <em>must</em> (in SBML releases prior to Level&nbsp;2
 * version&nbsp;4) be the same as the units of the size of the compartment.
 * <p>
 * <li> <em>In the case of a parameter</em>, an {@link AssignmentRule} sets the
 * referenced parameter\'s value to that determined by the formula in the
 * \'math\' subelement of the {@link AssignmentRule} object.  The overall units of
 * the formula in the \'math\' subelement <em>should</em> (in SBML Level&nbsp;2
 * Version&nbsp;4 and in SBML Level&nbsp;3) or <em>must</em> (in SBML releases prior to
 * Level&nbsp;2 version&nbsp;4) be the same as the units defined for the
 * parameter.  </ul>
 * <p>
 * In the context of a simulation, assignment rules are in effect at all
 * times, <em>t</em> &#8805; <em>0</em>.  For purposes of evaluating
 * expressions that involve the <em>delay</em> \'csymbol\' (see the SBML
 * Level&nbsp;2 specification), assignment rules are considered to apply
 * also at <em>t</em> &#8804; <em>0</em>.  Please consult the relevant
 * SBML specification for additional information about the semantics of
 * assignments, rules, and entity values for simulation time <em>t</em>
 * &#8804; <em>0</em>.
 * <p>
 * A model must not contain more than one {@link AssignmentRule} or {@link RateRule}
 * object having the same value of \'variable\'; in other words, in the set
 * of all assignment rules and rate rules in an SBML model, each variable
 * appearing in the left-hand sides can only appear once.  This simply
 * follows from the fact that an indeterminate system would result if a
 * model contained more than one assignment rule for the same variable or
 * both an assignment rule and a rate rule for the same variable.
 * <p>
 * Similarly, a model must also not contain <em>both</em> an {@link AssignmentRule}
 * and an {@link InitialAssignment} for the same variable, because both kinds of
 * constructs apply prior to and at the start of simulation time, i.e.,
 * <em>t</em> &#8804; <em>0</em>.  If a model contained both an initial
 * assignment and an assignment rule for the same variable, an
 * indeterminate system would result.
 * <p>
 * The value calculated by an {@link AssignmentRule} object overrides the value
 * assigned to the given symbol by the object defining that symbol.  For
 * example, if a {@link Compartment} object\'s \'size\' attribute value is set in its
 * definition, and the model also contains an {@link AssignmentRule} object having
 * that compartment\'s \'id\' as its \'variable\' value, then the \'size\'
 * assigned in the {@link Compartment} object definition is ignored and the value
 * assigned based on the computation defined in the {@link AssignmentRule}.  This
 * does <em>not</em> mean that a definition for a given symbol can be
 * omitted if there is an {@link AssignmentRule} object for it.  For example, there
 * must be a {@link Parameter} definition for a given parameter if there is an
 * {@link AssignmentRule} for that parameter.  It is only a question of which value
 * definition takes precedence.
 * <p>
 * <h2>General summary of SBML rules
</h2>
 * <p>
 * In SBML Level&nbsp;3 as well as Level&nbsp;2, rules are separated into three
subclasses for the benefit of model analysis software.  The three
subclasses are based on the following three different possible functional
forms (where <em>x</em> is a variable, <em>f</em> is some arbitrary
function returning a numerical result, <b><em>V</em></b> is a vector of
variables that does not include <em>x</em>, and <b><em>W</em></b> is a
vector of variables that may include <em>x</em>):

<center>
<table border=\'0\' cellpadding=\'0\' style=\'font-size: small\'>
<tr><td width=\'120px\'><em>Algebraic:</em></td><td width=\'250px\'>left-hand side is zero</td><td><em>0 = f(<b>W</b>)</em></td></tr>
<tr><td><em>Assignment:</em></td><td>left-hand side is a scalar:</td><td><em>x = f(<b>V</b>)</em></td></tr>
<tr><td><em>Rate:</em></td><td>left-hand side is a rate-of-change:</td><td><em>dx/dt = f(<b>W</b>)</em></td></tr>
</table>
</center>

In their general form given above, there is little to distinguish
between <em>assignment</em> and <em>algebraic</em> rules.  They are treated as
separate cases for the following reasons:

<ul>
<li> <em>Assignment</em> rules can simply be evaluated to calculate
intermediate values for use in numerical methods.  They are statements
of equality that hold at all times.  (For assignments that are only
performed once, see {@link InitialAssignment}.)<p>

<li> SBML needs to place restrictions on assignment rules, for example
the restriction that assignment rules cannot contain algebraic loops.<p>

<li> Some simulators do not contain numerical solvers capable of solving
unconstrained algebraic equations, and providing more direct forms such
as assignment rules may enable those simulators to process models they
could not process if the same assignments were put in the form of
general algebraic equations;<p>

<li> Those simulators that <em>can</em> solve these algebraic equations make a
distinction between the different categories listed above; and<p>

<li> Some specialized numerical analyses of models may only be applicable
to models that do not contain <em>algebraic</em> rules.
</ul>

The approach taken to covering these cases in SBML is to define an
abstract {@link Rule} structure containing a subelement, \'math\', to hold the
right-hand side expression, then to derive subtypes of {@link Rule} that add
attributes to distinguish the cases of algebraic, assignment and rate
rules.  The \'math\' subelement must contain a MathML expression defining the
mathematical formula of the rule.  This MathML formula must return a
numerical value.  The formula can be an arbitrary expression referencing
the variables and other entities in an SBML model.

Each of the three subclasses of {@link Rule} (AssignmentRule, {@link AlgebraicRule},
{@link RateRule}) inherit the the \'math\' subelement and other fields from {@link SBase}.
The {@link AssignmentRule} and {@link RateRule} classes add an additional attribute,
\'variable\'.  See the definitions of {@link AssignmentRule}, {@link AlgebraicRule} and
{@link RateRule} for details about the structure and interpretation of each one.

 * <p>
 * <h2>Additional restrictions on SBML rules
</h2>
 * <p>
 * An important design goal of SBML rule semantics is to ensure that a
model\'s simulation and analysis results will not be dependent on when or
how often rules are evaluated.  To achieve this, SBML needs to place two
restrictions on rule use.  The first concerns algebraic loops in the system
of assignments in a model, and the second concerns overdetermined systems.

<h3><a class=\'anchor\' id=\'no-algebraic-loops\'>A model must not contain
algebraic loops</a></h3>

The combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
objects in a model constitute a set of assignment statements that should be
considered as a whole.  (A {@link KineticLaw} object is counted as an assignment
because it assigns a value to the symbol contained in the \'id\' attribute of
the {@link Reaction} object in which it is defined.)  This combined set of
assignment statements must not contain algebraic loops&mdash;dependency
chains between these statements must terminate.  To put this more formally,
consider a directed graph in which nodes are assignment statements and
directed arcs exist for each occurrence of an SBML species, compartment or
parameter symbol in an assignment statement\'s \'math\' subelement.  Let the
directed arcs point from the statement assigning the symbol to the
statements that contain the symbol in their \'math\' subelement expressions.
This graph must be acyclic.

SBML does not specify when or how often rules should be evaluated.
Eliminating algebraic loops ensures that assignment statements can be
evaluated any number of times without the result of those evaluations
changing.  As an example, consider the set of equations <em>x = x + 1</em>,
<em>y = z + 200</em> and <em>z = y + 100</em>.  If this set of equations
were interpreted as a set of assignment statements, it would be invalid
because the rule for <em>x</em> refers to <em>x</em> (exhibiting one type
of loop), and the rule for <em>y</em> refers to <em>z</em> while the rule
for <em>z</em> refers back to <em>y</em> (exhibiting another type of loop).
Conversely, the following set of equations would constitute a valid set of
assignment statements: <em>x = 10</em>, <em>y = z + 200</em>, and <em>z = x
+ 100</em>.

<h3><a class=\'anchor\' id=\'no-overdetermined\'>A model must not be
overdetermined</a></h3>

An SBML model must not be overdetermined; that is, a model must not
define more equations than there are unknowns in a model.  An SBML model
that does not contain {@link AlgebraicRule} structures cannot be overdetermined.

LibSBML implements the static analysis procedure described in
Appendix&nbsp;B of the SBML Level&nbsp;3 Version&nbsp;1 Core
specification for assessing whether a model is overdetermined.

(In summary, assessing whether a given continuous, deterministic,
mathematical model is overdetermined does not require dynamic analysis; it
can be done by analyzing the system of equations created from the model.
One approach is to construct a bipartite graph in which one set of vertices
represents the variables and the other the set of vertices represents the
equations.  Place edges between vertices such that variables in the system
are linked to the equations that determine them.  For algebraic equations,
there will be edges between the equation and each variable occurring in the
equation.  For ordinary differential equations (such as those defined by
rate rules or implied by the reaction rate definitions), there will be a
single edge between the equation and the variable determined by that
differential equation.  A mathematical model is overdetermined if the
maximal matchings of the bipartite graph contain disconnected vertexes
representing equations.  If one maximal matching has this property, then
all the maximal matchings will have this property; i.e., it is only
necessary to find one maximal matching.)

 * <p>
 * <h2>Rule types for SBML Level 1
</h2>
 * <p>
 * SBML Level 1 uses a different scheme than SBML Level 2 and Level 3 for
 * distinguishing rules; specifically, it uses an attribute whose value is
 * drawn from an enumeration of 3 values.  LibSBML supports this using methods
 * that work with the enumeration values  listed below.
 * <p>
 * <ul>
 * <li> {@link  libsbmlConstants#RULE_TYPE_RATE RULE_TYPE_RATE}: Indicates
 * the rule is a \'rate\' rule.
 * <li> {@link  libsbmlConstants#RULE_TYPE_SCALAR RULE_TYPE_SCALAR}:
 * Indicates the rule is a \'scalar\' rule.
 * <li> {@link  libsbmlConstants#RULE_TYPE_INVALID RULE_TYPE_INVALID}:
 * Indicates the rule type is unknown or not yet set.
 * </ul>
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%javamethodmodifiers AssignmentRule::AssignmentRule(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link AssignmentRule} using the given SBML <code>level</code> and <code>version
</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link AssignmentRule}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link AssignmentRule}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of an {@link AssignmentRule} object to an {@link SBMLDocument}
   * (e.g., using&nbsp; {@link Model#addRule(Rule r)}, the SBML Level, SBML Version
   * and XML namespace of the document <em>override</em> the values used
   * when creating the {@link AssignmentRule} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link AssignmentRule} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers AssignmentRule::AssignmentRule(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link AssignmentRule} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link AssignmentRule} object to an {@link SBMLDocument}
   * (e.g., using&nbsp; {@link Model#addRule(Rule r)}, the SBML XML namespace of
   * the document <em>overrides</em> the value used when creating the
   * {@link AssignmentRule} object via this constructor.  This is necessary to
   * ensure that an SBML document is a consistent structure.  Nevertheless,
   * the ability to supply the values at the time of creation of a
   * {@link AssignmentRule} is an important aid to producing valid SBML.  Knowledge
   * of the intented SBML Level and Version determine whether it is valid
   * to assign a particular value to an attribute, or whether it is valid
   * to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers AssignmentRule::clone() const "
/**
   * Creates and returns a deep copy of this {@link Rule}.
   * <p>
   * @return a (deep) copy of this {@link Rule}.
   */ public
";


%javamethodmodifiers AssignmentRule::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link AssignmentRule}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link AssignmentRule} object
   * in the list of rules within which <em>the</em> present object is embedded.
   */ public
";


%javamethodmodifiers AssignmentRule::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link AssignmentRule} object
   * have been set.
   * <p>
   * @note In SBML Levels&nbsp;2&ndash;3, the only required attribute for
   * an {@link AssignmentRule} object is \'variable\'.  For Level&nbsp;1, where the
   * equivalent attribute is known by different names (\'compartment\',
   * \'species\', or \'name\', depending on the type of object), there is an
   * additional required attribute called \'formula\'.
   * <p>
   * @return <code>true</code> if the required attributes have been set, <code>false
</code>
   * otherwise.
   */ public
";


%javamethodmodifiers AssignmentRule::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%typemap(javaimports) RateRule "
/** 
 * LibSBML implementation of SBML\'s RateRule construct.
 * <p>
 * The rule type {@link RateRule} is derived from the parent class {@link Rule}.  It is
 * used to express equations that determine the rates of change of
 * variables.  The left-hand side (the \'variable\' attribute) can refer to
 * the identifier of a species, compartment, or parameter (but not a
 * reaction).  The entity identified must have its \'constant\' attribute set
 * to <code>false.</code>  The effects of a {@link RateRule} are in general terms the same,
 * but differ in the precise details depending on which variable is being
 * set:
 * <p>
 * <ul> <li> <em>In the case of a species</em>, a {@link RateRule} sets the rate of
 * change of the species\' quantity (<em>concentration</em> or <em>amount of
 * substance</em>) to the value determined by the formula in the \'math\'
 * subelement of the {@link RateRule} object.  The overall units of the formula in
 * \'math\' <em>should</em> (in SBML Level&nbsp;2 Version&nbsp;4 and in SBML Level&nbsp;3) or <em>
</em>
 * must (in SBML releases prior to Level&nbsp;2 version&nbsp;4) be equal to
 * the unit of <em>species quantity</em> divided by the model-wide unit of
 * <em>time</em>.  <em>Restrictions</em>: There must not be both a {@link RateRule}
 * \'variable\' attribute and a {@link SpeciesReference} \'species\' attribute having
 * the same value, unless that species has its \'boundaryCondition\'
 * attribute is set to <code>true.</code>  This means a rate rule cannot be defined
 * for a species that is created or destroyed in a reaction, unless that
 * species is defined as a boundary condition in the model.
 * <p>
 * <li> (For SBML Level&nbsp;3 only) <em>In the case of a species
 * reference</em>, a {@link RateRule} sets the rate of change of the stoichiometry
 * of the referenced reactant or product to the value determined by the
 * formula in \'math\'.  The unit associated with the value produced by the
 * \'math\' formula should be consistent with the unit \'dimensionless\'
 * divided by the model-wide unit of <em>time</em>.
 * <p>
 * <li> <em>In the case of a compartment</em>, a {@link RateRule} sets the rate of
 * change of the compartment\'s size to the value determined by the formula
 * in the \'math\' subelement of the {@link RateRule} object.  The overall units of
 * the formula <em>should</em> (in SBML Level&nbsp;2 Version&nbsp;4 and in SBML
 * Level&nbsp;3) or <em>must</em> (in SBML releases prior to Level&nbsp;2
 * version&nbsp;4) be the units of the compartment\'s <em>size</em> divided
 * by the model-wide unit of <em>time</em>.
 * <p>
 * <li> <em>In the case of a parameter</em>, a {@link RateRule} sets the rate of
 * change of the parameter\'s value to that determined by the formula in the
 * \'math\' subelement of the {@link RateRule} object.  The overall units of the
 * formula <em>should</em> (in SBML Level&nbsp;2 Version&nbsp;4 and in SBML
 * Level&nbsp;3) or <em>must</em> (in SBML releases prior to Level&nbsp;2
 * version&nbsp;4) be the {@link Parameter} object\'s \'unit\' attribute value divided
 * by the model-wide unit of <em>time</em>.  </ul>
 * <p>
 * In the context of a simulation, rate rules are in effect for simulation
 * time <em>t</em> &lt; <em>0</em>.  Please consult the relevant SBML
 * specification for additional information about the semantics of
 * assignments, rules, and entity values for simulation time <em>t</em>
 * &#8804; <em>0</em>.
 * <p>
 * As mentioned in the description of {@link AssignmentRule}, a model must not
 * contain more than one {@link RateRule} or {@link AssignmentRule} object having the same
 * value of \'variable\'; in other words, in the set of all assignment rules
 * and rate rules in an SBML model, each variable appearing in the
 * left-hand sides can only appear once.  This simply follows from the fact
 * that an indeterminate system would result if a model contained more than
 * one assignment rule for the same variable or both an assignment rule and
 * a rate rule for the same variable.
 * <p>
 * <h2>General summary of SBML rules
</h2>
 * <p>
 * In SBML Level&nbsp;3 as well as Level&nbsp;2, rules are separated into three
subclasses for the benefit of model analysis software.  The three
subclasses are based on the following three different possible functional
forms (where <em>x</em> is a variable, <em>f</em> is some arbitrary
function returning a numerical result, <b><em>V</em></b> is a vector of
variables that does not include <em>x</em>, and <b><em>W</em></b> is a
vector of variables that may include <em>x</em>):

<center>
<table border=\'0\' cellpadding=\'0\' style=\'font-size: small\'>
<tr><td width=\'120px\'><em>Algebraic:</em></td><td width=\'250px\'>left-hand side is zero</td><td><em>0 = f(<b>W</b>)</em></td></tr>
<tr><td><em>Assignment:</em></td><td>left-hand side is a scalar:</td><td><em>x = f(<b>V</b>)</em></td></tr>
<tr><td><em>Rate:</em></td><td>left-hand side is a rate-of-change:</td><td><em>dx/dt = f(<b>W</b>)</em></td></tr>
</table>
</center>

In their general form given above, there is little to distinguish
between <em>assignment</em> and <em>algebraic</em> rules.  They are treated as
separate cases for the following reasons:

<ul>
<li> <em>Assignment</em> rules can simply be evaluated to calculate
intermediate values for use in numerical methods.  They are statements
of equality that hold at all times.  (For assignments that are only
performed once, see {@link InitialAssignment}.)<p>

<li> SBML needs to place restrictions on assignment rules, for example
the restriction that assignment rules cannot contain algebraic loops.<p>

<li> Some simulators do not contain numerical solvers capable of solving
unconstrained algebraic equations, and providing more direct forms such
as assignment rules may enable those simulators to process models they
could not process if the same assignments were put in the form of
general algebraic equations;<p>

<li> Those simulators that <em>can</em> solve these algebraic equations make a
distinction between the different categories listed above; and<p>

<li> Some specialized numerical analyses of models may only be applicable
to models that do not contain <em>algebraic</em> rules.
</ul>

The approach taken to covering these cases in SBML is to define an
abstract {@link Rule} structure containing a subelement, \'math\', to hold the
right-hand side expression, then to derive subtypes of {@link Rule} that add
attributes to distinguish the cases of algebraic, assignment and rate
rules.  The \'math\' subelement must contain a MathML expression defining the
mathematical formula of the rule.  This MathML formula must return a
numerical value.  The formula can be an arbitrary expression referencing
the variables and other entities in an SBML model.

Each of the three subclasses of {@link Rule} (AssignmentRule, {@link AlgebraicRule},
{@link RateRule}) inherit the the \'math\' subelement and other fields from {@link SBase}.
The {@link AssignmentRule} and {@link RateRule} classes add an additional attribute,
\'variable\'.  See the definitions of {@link AssignmentRule}, {@link AlgebraicRule} and
{@link RateRule} for details about the structure and interpretation of each one.

 * <p>
 * <h2>Additional restrictions on SBML rules
</h2>
 * <p>
 * An important design goal of SBML rule semantics is to ensure that a
model\'s simulation and analysis results will not be dependent on when or
how often rules are evaluated.  To achieve this, SBML needs to place two
restrictions on rule use.  The first concerns algebraic loops in the system
of assignments in a model, and the second concerns overdetermined systems.

<h3><a class=\'anchor\' id=\'no-algebraic-loops\'>A model must not contain
algebraic loops</a></h3>

The combined set of {@link InitialAssignment}, {@link AssignmentRule} and {@link KineticLaw}
objects in a model constitute a set of assignment statements that should be
considered as a whole.  (A {@link KineticLaw} object is counted as an assignment
because it assigns a value to the symbol contained in the \'id\' attribute of
the {@link Reaction} object in which it is defined.)  This combined set of
assignment statements must not contain algebraic loops&mdash;dependency
chains between these statements must terminate.  To put this more formally,
consider a directed graph in which nodes are assignment statements and
directed arcs exist for each occurrence of an SBML species, compartment or
parameter symbol in an assignment statement\'s \'math\' subelement.  Let the
directed arcs point from the statement assigning the symbol to the
statements that contain the symbol in their \'math\' subelement expressions.
This graph must be acyclic.

SBML does not specify when or how often rules should be evaluated.
Eliminating algebraic loops ensures that assignment statements can be
evaluated any number of times without the result of those evaluations
changing.  As an example, consider the set of equations <em>x = x + 1</em>,
<em>y = z + 200</em> and <em>z = y + 100</em>.  If this set of equations
were interpreted as a set of assignment statements, it would be invalid
because the rule for <em>x</em> refers to <em>x</em> (exhibiting one type
of loop), and the rule for <em>y</em> refers to <em>z</em> while the rule
for <em>z</em> refers back to <em>y</em> (exhibiting another type of loop).
Conversely, the following set of equations would constitute a valid set of
assignment statements: <em>x = 10</em>, <em>y = z + 200</em>, and <em>z = x
+ 100</em>.

<h3><a class=\'anchor\' id=\'no-overdetermined\'>A model must not be
overdetermined</a></h3>

An SBML model must not be overdetermined; that is, a model must not
define more equations than there are unknowns in a model.  An SBML model
that does not contain {@link AlgebraicRule} structures cannot be overdetermined.

LibSBML implements the static analysis procedure described in
Appendix&nbsp;B of the SBML Level&nbsp;3 Version&nbsp;1 Core
specification for assessing whether a model is overdetermined.

(In summary, assessing whether a given continuous, deterministic,
mathematical model is overdetermined does not require dynamic analysis; it
can be done by analyzing the system of equations created from the model.
One approach is to construct a bipartite graph in which one set of vertices
represents the variables and the other the set of vertices represents the
equations.  Place edges between vertices such that variables in the system
are linked to the equations that determine them.  For algebraic equations,
there will be edges between the equation and each variable occurring in the
equation.  For ordinary differential equations (such as those defined by
rate rules or implied by the reaction rate definitions), there will be a
single edge between the equation and the variable determined by that
differential equation.  A mathematical model is overdetermined if the
maximal matchings of the bipartite graph contain disconnected vertexes
representing equations.  If one maximal matching has this property, then
all the maximal matchings will have this property; i.e., it is only
necessary to find one maximal matching.)

 * <p>
 * <h2>Rule types for SBML Level 1
</h2>
 * <p>
 * SBML Level 1 uses a different scheme than SBML Level 2 and Level 3 for
 * distinguishing rules; specifically, it uses an attribute whose value is
 * drawn from an enumeration of 3 values.  LibSBML supports this using methods
 * that work with the enumeration values  listed below.
 * <p>
 * <ul>
 * <li> {@link  libsbmlConstants#RULE_TYPE_RATE RULE_TYPE_RATE}: Indicates
 * the rule is a \'rate\' rule.
 * <li> {@link  libsbmlConstants#RULE_TYPE_SCALAR RULE_TYPE_SCALAR}:
 * Indicates the rule is a \'scalar\' rule.
 * <li> {@link  libsbmlConstants#RULE_TYPE_INVALID RULE_TYPE_INVALID}:
 * Indicates the rule type is unknown or not yet set.
 * </ul>
 * <p>
 * <!-- leave this next break as-is to work around some doxygen bug -->
 */
"


%javamethodmodifiers RateRule::RateRule(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link RateRule} using the given SBML <code>level</code> and <code>version
</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link RateRule}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link RateRule}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link RateRule} object to an {@link SBMLDocument}
   * (e.g., using&nbsp; {@link Model#addRule(Rule r)}, the SBML Level, SBML Version
   * and XML namespace of the document <em>override</em> the values used
   * when creating the {@link RateRule} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link RateRule} is an important aid to producing valid SBML.  Knowledge
   * of the intented SBML Level and Version determine whether it is valid
   * to assign a particular value to an attribute, or whether it is valid
   * to add an object to an existing {@link SBMLDocument}.
  */ public
";


%javamethodmodifiers RateRule::RateRule(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link RateRule} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link RateRule} object to an {@link SBMLDocument} (e.g.,
   * using {@link Model#addRule(Rule r)}, the SBML XML namespace of the document
   * <em>overrides</em> the value used when creating the {@link RateRule} object via
   * this constructor.  This is necessary to ensure that an SBML document
   * is a consistent structure.  Nevertheless, the ability to supply the
   * values at the time of creation of a {@link RateRule} is an important aid to
   * producing valid SBML.  Knowledge of the intented SBML Level and
   * Version determine whether it is valid to assign a particular value to
   * an attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers RateRule::clone() const "
/**
   * Creates and returns a deep copy of this {@link Rule}.
   * <p>
   * @return a (deep) copy of this {@link Rule}.
   */ public
";


%javamethodmodifiers RateRule::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link RateRule} object
   * in the list of rules within which <em>the</em> present object is embedded.
   */ public
";


%javamethodmodifiers RateRule::hasRequiredAttributes() const "
/**
   * Predicate returning <code>true</code> if
   * all the required attributes for this {@link RateRule} object
   * have been set.
   * <p>
   * @note In SBML Levels&nbsp;2&ndash;3, the only required attribute for a
   * {@link RateRule} object is \'variable\'.  For Level&nbsp;1, where the equivalent
   * attribute is known by different names (\'compartment\', \'species\', or
   * \'name\', depending on the type of object), there is an additional
   * required attribute called \'formula\'.
   * <p>
   * @return <code>true</code> if the required attributes have been set, <code>false
</code>
   * otherwise.
   */ public
";


%javamethodmodifiers RateRule::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%typemap(javaimports) Constraint "
/** 
 * LibSBML implementation of SBML\'s Constraint construct.
 * <p>
 * The {@link Constraint} object class was introduced in SBML Level&nbsp;2
 * Version&nbsp;2 as a mechanism for stating the assumptions under which a
 * model is designed to operate.  The <em>constraints</em> are statements
 * about permissible values of different quantities in a model.
 * Constraints are not used to compute dynamical values for simulation or
 * analysis, but rather, they serve an advisory role for
 * simulation/analysis tools.
 * <p>
 * SBML\'s {@link Constraint} object class has one required attribute, \'id\', to
 * give the parameter a unique identifier by which other parts of an SBML
 * model definition can refer to it.  A {@link Constraint} object can also have an
 * optional \'name\' attribute of type <code>string.</code>  Identifiers and names must
 * be used according to the guidelines described in the SBML specification
 * (e.g., Section 3.3 in the Level&nbsp;2 Version 4 specification).  
 * <p>
 * {@link Constraint} has one required subelement, \'math\', containing a MathML
 * formula defining the condition of the constraint.  This formula must
 * return a boolean value of <code>true</code> when the model is a <em>valid</em>
 * state.  The formula can be an arbitrary expression referencing the
 * variables and other entities in an SBML model.  The evaluation of \'math\'
 * and behavior of constraints are described in more detail below.
 * <p>
 * A {@link Constraint} structure also has an optional subelement called \'message\'.
 * This can contain a message in XHTML format that may be displayed to the
 * user when the condition of the formula in the \'math\' subelement
 * evaluates to a value of <code>false.</code>  Software tools are not required to
 * display the message, but it is recommended that they do so as a matter
 * of best practice.  The XHTML content within a \'message\' subelement must
 * follow the same restrictions as for the \'notes\' element on {@link SBase}
 * described in in the SBML Level&nbsp;2 specification; please consult the
 * <a target=\'_blank\' href=\'http://sbml.org/Documents/Specifications\'>SBML
 * specification document</a> corresponding to the SBML Level and Version
 * of your model for more information about the requirements for \'notes\'
 * content.
 * <p>
 * {@link Constraint} was introduced in SBML Level&nbsp;2 Version&nbsp;2.  It is
 * not available in earlier versions of Level&nbsp;2 nor in any version of
 * Level&nbsp;1.
 * <p>
 * <h2>Semantics of Constraints</h2>
 * <p>
 * In the context of a simulation, a {@link Constraint} has effect at all times
 * <em>t &#8805; 0</em>.  Each {@link Constraint}\'s \'math\' subelement is first
 * evaluated after any {@link InitialAssignment} definitions in a model at <em>t =
 * 0</em> and can conceivably trigger at that point.  (In other words, a
 * simulation could fail a constraint immediately.)
 * <p>
 * {@link Constraint} structures <em>cannot and should not</em> be used to compute
 * the dynamical behavior of a model as part of, for example, simulation.
 * Constraints may be used as input to non-dynamical analysis, for instance
 * by expressing flux constraints for flux balance analysis.
 * <p>
 * The results of a simulation of a model containing a constraint are
 * invalid from any simulation time at and after a point when the function
 * given by the \'math\' subelement returns a value of <code>false.</code>  Invalid
 * simulation results do not make a prediction of the behavior of the
 * biochemical reaction network represented by the model.  The precise
 * behavior of simulation tools is left undefined with respect to
 * constraints.  If invalid results are detected with respect to a given
 * constraint, the \'message\' subelement may optionally be displayed to the
 * user.  The simulation tool may also halt the simulation or clearly
 * delimit in output data the simulation time point at which the simulation
 * results become invalid.
 * <p>
 * SBML does not impose restrictions on duplicate {@link Constraint} definitions or
 * the order of evaluation of {@link Constraint} objects in a model.  It is
 * possible for a model to define multiple constraints all with the same
 * mathematical expression.  Since the failure of any constraint indicates
 * that the model simulation has entered an invalid state, a system is not
 * required to attempt to detect whether other constraints in the model
 * have failed once any one constraint has failed.
 * <p>
 * <!---------------------------------------------------------------------- -->
 * <p>
 */
"


%typemap(javaimports) ListOfConstraints "
/** 
 * LibSBML implementation of SBML\'s ListOfConstraints construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers Constraint::Constraint(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link Constraint} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Constraint}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Constraint}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Constraint} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addConstraint(Constraint c)}), the
   * SBML Level, SBML Version and XML namespace of the document 
   * <em>override</em> the values used when creating the {@link Constraint} object via this
   * constructor.  This is necessary to ensure that an SBML document is a
   * consistent structure.  Nevertheless, the ability to supply the values
   * at the time of creation of a {@link Constraint} is an important aid to
   * producing valid SBML.  Knowledge of the intented SBML Level and
   * Version determine whether it is valid to assign a particular value to
   * an attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Constraint::Constraint(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link Constraint} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Constraint} object to an {@link SBMLDocument}
   * (e.g., using {@link Model#addConstraint(Constraint c)}), the
   * SBML XML namespace of the document <em>overrides</em> the value used when
   * creating the {@link Constraint} object via this constructor.  This is
   * necessary to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link Constraint} is an important aid to producing valid SBML.
   * Knowledge of the intented SBML Level and Version determine whether it
   * is valid to assign a particular value to an attribute, or whether it
   * is valid to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Constraint::Constraint(const Constraint& orig) "
/**
   * Copy constructor; creates a copy of this {@link Constraint}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers Constraint::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Constraint}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>, which indicates
   * whether the Visitor would like to visit the next {@link Constraint} in the
   * list of constraints within which this {@link Constraint} is embedded (i.e., in
   * the {@link ListOfConstraints} located in the enclosing {@link Model} instance).
   */ public
";


%javamethodmodifiers Constraint::clone() const "
/**
   * Creates and returns a deep copy of this {@link Constraint}.
   * <p>
   * @return a (deep) copy of this {@link Constraint}.
   */ public
";


%javamethodmodifiers Constraint::getMessage() const "
/**
   * Get the message, if any, associated with this {@link Constraint}
   * <p>
   * @return the message for this {@link Constraint}, as an {@link XMLNode}.
   */ public
";


%javamethodmodifiers Constraint::getMessageString() const "
/**
   * Get the message string, if any, associated with this {@link Constraint}
   * <p>
   * @return the message for this {@link Constraint}, as a string.
   */ public
";


%javamethodmodifiers Constraint::getMath() const "
/**
   * Get the mathematical expression of this {@link Constraint}
   * <p>
   * @return the math for this {@link Constraint}, as an {@link ASTNode}.
   */ public
";


%javamethodmodifiers Constraint::isSetMessage() const "
/**
   * Predicate returning <code>true</code> if a
   * message is defined for this {@link Constraint}.
   * <p>
   * @return <code>true</code> if the message of this {@link Constraint} is set,
   * <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Constraint::isSetMath() const "
/**
   * Predicate returning <code>true</code> if a
   * mathematical formula is defined for this {@link Constraint}.
   * <p>
   * @return <code>true</code> if the \'math\' subelement for this {@link Constraint} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Constraint::setMessage(const XMLNode* xhtml) "
/**
   * Sets the message of this {@link Constraint}.
   * <p>
   * The {@link XMLNode} tree passed in <code>xhtml</code> is copied.
   * <p>
   * @param xhtml an XML tree containing XHTML content.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   */ public
";


%javamethodmodifiers Constraint::setMath(const ASTNode* math) "
/**
   * Sets the mathematical expression of this {@link Constraint} to a copy of the
   * AST given as <code>math</code>.
   * <p>
   * @param math an {@link ASTNode} expression to be assigned as the \'math\'
   * subelement of this {@link Constraint}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * </ul>
   */ public
";


%javamethodmodifiers Constraint::unsetMessage "
/**
   * Unsets the \'message\' subelement of this {@link Constraint}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Constraint::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%javamethodmodifiers Constraint::renameUnitSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the UnitSIdRef attributes on this element
   */ public
";


%javamethodmodifiers Constraint::replaceSIDWithFunction(const std::string& id, const ASTNode* function) "
/**
   * Replace all nodes with the name \'id\' from the child \'math\' object with the provided function. 
   * <p>
   * @internal
   */ public
";


%javamethodmodifiers Constraint::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers Constraint::getElementName() const "
/**
   * Returns the XML element name of this object, which for {@link Constraint}, is
   * always <code>\'constraint\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'constraint\'.</code>
   */ public
";


%javamethodmodifiers Constraint::writeElements(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write out their contained
   * SBML objects as XML elements.  Be sure to call your parents
   * implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Constraint::hasRequiredElements() const "
/**
   * Predicate returning <code>true</code> if
   * all the required elements for this {@link Constraint} object
   * have been set.
   * <p>
   * @note The required elements for a {@link Constraint} object are:
   * <ul>
   * <li> \'math\'
   * </ul>
   * <p>
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public
";


%javamethodmodifiers Constraint::readOtherXML(XMLInputStream& stream) "
/**
   * Subclasses should override this method to read (and store) XHTML,
   * MathML, etc. directly from the XMLInputStream.
   * <p>
   * @return true if the subclass read from the stream, false otherwise.
   * @internal
   */ public
";


%javamethodmodifiers Constraint::addExpectedAttributes(ExpectedAttributes& attributes) "
/**
   * Subclasses should override this method to get the list of
   * expected attributes.
   * This function is invoked from corresponding readAttributes()
   * function.
   * @internal
   */ public
";


%javamethodmodifiers Constraint::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Constraint::readL2Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Constraint::readL3Attributes(const XMLAttributes& attributes) "
/**
   * Subclasses should override this method to read values from the given
   * {@link XMLAttributes} set into their specific fields.  Be sure to call your
   * parents implementation of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers Constraint::writeAttributes(XMLOutputStream& stream) const "
/**
   * Subclasses should override this method to write their XML attributes
   * to the XMLOutputStream.  Be sure to call your parents implementation
   * of this method as well.
   * @internal
   */ public
";


%javamethodmodifiers ListOfConstraints::ListOfConstraints(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link ListOfConstraints} object.
   * <p>
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   * <p>
   * @param level the SBML Level
   * <p>
   * @param version the Version within the SBML Level
   */ public
";


%javamethodmodifiers ListOfConstraints::ListOfConstraints(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link ListOfConstraints} object.
   * <p>
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the {@link SBMLNamespaces} object in 
   * <code>sbmlns</code>.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object that is used to determine the
   * characteristics of the {@link ListOfConstraints} object to be created.
   */ public
";


%javamethodmodifiers ListOfConstraints::clone() const "
/**
   * Creates and returns a deep copy of this {@link ListOfConstraints} instance.
   * <p>
   * @return a (deep) copy of this {@link ListOfConstraints}.
   */ public
";


%javamethodmodifiers ListOfConstraints::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for this object, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfConstraints::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this {@link ListOf}
   * (i.e., {@link Constraint} objects, if the list is non-empty).
   * <p>
   * LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters <code>SBML_.</code> 
   * <p>
   * @return the SBML type code for the objects contained in this {@link ListOf}
   * instance, or {@link  libsbmlConstants#SBML_UNKNOWN SBML_UNKNOWN} (default).
   * <p>
   * @see #getElementName()
   */ public
";


%javamethodmodifiers ListOfConstraints::getElementName() const "
/**
   * Returns the XML element name of this object.
   * <p>
   * For {@link ListOfConstraints}, the XML element name is <code>\'listOfConstraints\'.</code>
   * <p>
   * @return the name of this element, i.e., <code>\'listOfConstraints\'.</code>
   */ public
";


%javamethodmodifiers ListOfConstraints::get(unsigned int n) "
/**
   * Get a {@link Constraint} from the {@link ListOfConstraints}.
   * <p>
   * @param n the index number of the {@link Constraint} to get.
   * <p>
   * @return the nth {@link Constraint} in this {@link ListOfConstraints}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfConstraints::get "
/**
   * Get a {@link Constraint} from the {@link ListOfConstraints}.
   * <p>
   * @param n the index number of the {@link Constraint} to get.
   * <p>
   * @return the nth {@link Constraint} in this {@link ListOfConstraints}.
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfConstraints::remove(unsigned int n) "
/**
   * Removes the nth item from this {@link ListOfConstraints} items and returns a
   * pointer to it.
   * <p>
   * The caller owns the returned item and is responsible for deleting it.
   * <p>
   * @param n the index of the item to remove
   * <p>
   * @see #size()
   */ public
";


%javamethodmodifiers ListOfConstraints::getElementPosition() const "
/**
   * Get the ordinal position of this element in the containing object
   * (which in this case is the {@link Model} object).
   * <p>
   * The ordering of elements in the XML form of SBML is generally fixed
   * for most components in SBML.  So, for example, the {@link ListOfConstraints}
   * in a model is (in SBML Level&nbsp;2 Version 4) the tenth ListOf___.
   * (However, it differs for different Levels and Versions of SBML.)
   * <p>
   * @return the ordinal position of the element with respect to its
   * siblings, or <code>-1</code> (default) to indicate the position is not significant.
   * @internal
   */ public
";


%javamethodmodifiers ListOfConstraints::createObject(XMLInputStream& stream) "
/**
   * @return the SBML object corresponding to next {@link XMLToken} in the
   * XMLInputStream or <code>null</code> if the token was not recognized.
   * @internal
   */ public
";


%typemap(javaimports) Reaction "
/** 
 * LibSBML implementation of SBML\'s Reaction construct.
 * <p>
 * A <em>reaction</em> represents any transformation, transport or binding
 * process, typically a chemical reaction, that can change the quantity of
 * one or more species.  In SBML, a reaction is defined primarily in terms
 * of the participating reactants and products (and their corresponding
 * stoichiometries), along with optional modifier species, an optional rate
 * at which the reaction takes place, and optional parameters.  
 * <p>
 * As with other major object in SBML, {@link Reaction} has a mandatory attribute,
 * \'id\', used to give the compartment type an identifier.  The identifier
 * must be a text string conforming to the identifer syntax permitted in
 * SBML.  In SBML Level&nbsp;2 and Level&nbsp;3, the reaction \'id\'
 * identifier can be used in mathematical formulas elsewhere in an SBML
 * model to represent the rate of that reaction; this usage is explained
 * below.  {@link Reaction} also has an optional \'name\' attribute, of type 
 * <code>string.</code>  The \'id\' and \'name\' must be used according to the guidelines
 * described in the SBML specification.
 * <p>
 * The species participating as reactants, products, and/or modifiers in a
 * reaction are declared using lists of {@link SpeciesReference} and/or
 * {@link ModifierSpeciesReference} instances stored in subelements
 * \'listOfReactants\', \'listOfProducts\' and \'listOfModifiers\'.  Certain
 * restrictions are placed on the appearance of species in reaction
 * definitions:
 * <ul>
 * <li> The ability of a species to appear as a reactant or product of any
 * reaction in a model is governed by certain flags in that species\'
 * definition; see the definition of {@link Species} for more information.
 * <p>
 * <li> Any species appearing in the mathematical formula of the subelement
 * \'kineticLaw\' (described below) of a {@link Reaction} must be declared in at
 * least one of that {@link Reaction}\'s lists of reactants, products, and/or
 * modifiers.  Put another way, it is an error for a reaction\'s kinetic law
 * formula to refer to species that have not been declared for that
 * reaction.
 * <p>
 * <li> A reaction definition can contain an empty list of reactants
 * <em>or</em> an empty list of products, but it must have at least one
 * reactant or product; in other words, a reaction without any reactant or
 * product species is not permitted.  (This restriction does not apply to
 * modifier species, which remain optional in all cases.)
 * </ul>
 * <p>
 * A reaction can contain up to one {@link KineticLaw} object in a subelement named
 * \'kineticLaw\'.  It defines the speed at which the process defined by the
 * reaction takes place.  The description of {@link KineticLaw} provides more
 * details about its use.  Note that although the inclusion of a {@link KineticLaw}
 * object in an instance of a {@link Reaction} component is optional, there is no
 * useful default that can be substituted in place of a missing rate
 * expression in a reaction.  Moreover, a reaction\'s rate cannot be defined
 * in any other way in SBML&mdash;{@link InitialAssignment}, {@link AssignmentRule},
 * {@link RateRule}, {@link AlgebraicRule}, {@link Event}, and other constructs in SBML cannot be
 * used to set the reaction rate separately.  Nevertheless, for some
 * modeling applications, reactions without any defined rate can be
 * perfectly acceptable.
 * <p>
 * {@link Reaction} also has a boolean attribute named \'reversible\' for indicating
 * whether the reaction is reversible.  This attribute is optional in SBML
 * Level&nbsp;2, with a default of <code>true</code>; it is mandatory in SBML
 * Level&nbsp;3 (with no default value).  To say that a reaction is 
 * <em>reversible</em> is to say it can proceed in either the forward or the reverse
 * direction.  Although the reversibility of a reaction can sometimes be
 * deduced by inspecting its rate expression, this is not always the case,
 * especially for complicated expressions.  Moreover, the need in SBML to
 * allow rate expressions (i.e., {@link KineticLaw}) to be optional leads to the
 * need for a separate flag indicating reversibility.  Note that labeling a
 * reaction as irreversible is an assertion that the reaction always
 * proceeds in the given forward direction.  (Why else would it be flagged
 * as irreversible?)  This implies the rate expression in the {@link KineticLaw}
 * always has a non-negative value during simulations.  Software tools
 * could provide a means of optionally testing that this condition holds.
 * The presence of reversibility information in two places (i.e., the rate
 * expression and the \'reversible\' attribute on {@link Reaction}) leaves open the
 * possibility that a model could contain contradictory information, but
 * the creation of such a model would be an error on the part of the
 * software generating it.
 * <p>
 * The {@link Reaction} object class has another boolean attribute called \'fast\'.
 * This attribute is optional in SBML Level&nbsp;2, with a default of 
 * <code>false</code>; it is mandatory in SBML Level&nbsp;3 (with no default value).  It
 * is used to indicate that a reaction occurs on a vastly faster time scale
 * than others in a system.  Readers are directed to the SBML Level&nbsp;2
 * Version&nbsp;4 specification, which provides more detail about the
 * conditions under which a reaction can be considered to be fast in this
 * sense.  The attribute\'s default value is <code>false.</code>  SBML Level&nbsp;1
 * and Level&nbsp;2 Version&nbsp;1 incorrectly claimed that software tools
 * could ignore this attribute if they did not implement support for the
 * corresponding concept; however, further research in SBML has revealed
 * that this is not true, and \'fast\' <em>cannot be ignored</em> if it is
 * set to <code>true.</code>  SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 therefore
 * stipulate that if a model has any reactions with \'fast\' set to <code>true</code>,
 * a software tool must be able to respect the attribute or else indicate
 * to the user that it does not have the capacity to do so.  Analysis
 * software cannot ignore the value of the \'fast\' attribute because doing
 * so may lead to different results as compared to a software system that
 * <em>does</em> make use of \'fast\'.
 * <p>
 * In SBML Level&nbsp;3 Version&nbsp;1, the {@link Reaction} object has an
 * additional optional attribute named \'compartment\', whose value must be
 * the identifier of a compartment defined in the enclosing {@link Model} object.
 * The \'compartment\' attribute can be used to indicate the compartment in
 * which the reaction is assumed to take place.  If the attribute is
 * present, its value must be the identifier of a {@link Compartment} object
 * defined in the enclosing {@link Model} object.  Similar to the \'reversible\'
 * attribute, the value of the \'compartment\' attribute has no direct impact
 * on the construction of mathematical equations for the SBML model.  When
 * a kinetic law is given for a reaction, the compartment location may
 * already be implicit in the kinetic law (although this cannot always be
 * guaranteed).  Nevertheless, software tools may find the \'compartment\'
 * attribute value useful for such purposes as analyzing the structure of
 * the model, guiding the modeler in constructing correct rate formulas,
 * and visualization purposes.
 * <p>
 * Readers are urged to read the SBML specification for more details about
 * the proper use of {@link Reaction}.
 * <p>
 * <p>
 */
"


%typemap(javaimports) ListOfReactions "
/** 
 * LibSBML implementation of SBML\'s ListOfReactions construct.
 * <p>
 * The various ListOf___ classes in SBML are merely containers used for
 * organizing the main components of an SBML model.  All are derived from
 * the abstract class {@link SBase}, and inherit the various attributes and
 * subelements of {@link SBase}, such as \'metaid\' as and \'annotation\'.  The
 * ListOf___ classes do not add any attributes of their own.
 * <p>
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 * <p>
 * <center><img src=\'listof-illustration.jpg\'></center><br>
 * 
 * <p>
 * Readers may wonder about the motivations for using the ListOf___
 * containers.  A simpler approach in XML might be to place the components
 * all directly at the top level of the model definition.  The choice made
 * in SBML is to group them within XML elements named after
 * {@link ListOf}<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from {@link SBase} means that software tools can add information <em>about</em>
 * the lists themselves into each list container\'s \'annotation\'.
 * <p>
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 */
"


%javamethodmodifiers Reaction::Reaction(unsigned int level, unsigned int version) "
/**
   * Creates a new {@link Reaction} using the given SBML <code>level</code> and <code>version</code>
   * values.
   * <p>
   * @param level a long integer, the SBML Level to assign to this {@link Reaction}
   * <p>
   * @param version a long integer, the SBML Version to assign to this
   * {@link Reaction}
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Reaction} object to an {@link SBMLDocument} (e.g.,
   * using {@link Model#addReaction(Reaction r)}), the SBML Level, SBML Version and
   * XML namespace of the document <em>override</em> the values used when
   * creating the {@link Reaction} object via this constructor.  This is necessary
   * to ensure that an SBML document is a consistent structure.
   * Nevertheless, the ability to supply the values at the time of creation
   * of a {@link Reaction} is an important aid to producing valid SBML.  Knowledge
   * of the intented SBML Level and Version determine whether it is valid
   * to assign a particular value to an attribute, or whether it is valid
   * to add an object to an existing {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Reaction::Reaction(SBMLNamespaces* sbmlns) "
/**
   * Creates a new {@link Reaction} using the given {@link SBMLNamespaces} object
   * <code>sbmlns</code>.
   * <p>
   * The {@link SBMLNamespaces} object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * A common approach to using this class constructor is to create an
   * {@link SBMLNamespaces} object somewhere in a program, once, then pass it to
   * object constructors such as this one when needed.
   * <p>
   * @param sbmlns an {@link SBMLNamespaces} object.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the given <code>level</code> and <code>version</code> combination, or this kind
   * of SBML object, are either invalid or mismatched with respect to the
   * parent {@link SBMLDocument} object.
   * <p>
   * @note Upon the addition of a {@link Reaction} object to an {@link SBMLDocument} (e.g.,
   * using {@link Model#addReaction(Reaction r)}), the SBML XML namespace of the document
   * <em>overrides</em> the value used when creating the {@link Reaction} object via
   * this constructor.  This is necessary to ensure that an SBML document
   * is a consistent structure.  Nevertheless, the ability to supply the
   * values at the time of creation of a {@link Reaction} is an important aid to
   * producing valid SBML.  Knowledge of the intented SBML Level and
   * Version determine whether it is valid to assign a particular value to
   * an attribute, or whether it is valid to add an object to an existing
   * {@link SBMLDocument}.
   */ public
";


%javamethodmodifiers Reaction::Reaction(const Reaction& orig) "
/**
   * Copy constructor; creates a copy of this {@link Reaction}.
   * <p>
   * @param orig the object to copy.
   * <p>
   * @throws SBMLConstructorException 
   * Thrown if the argument <code>orig</code> is <code>null.</code>
   */ public
";


%javamethodmodifiers Reaction::accept(SBMLVisitor& v) const "
/**
   * Accepts the given {@link SBMLVisitor} for this instance of {@link Reaction}.
   * <p>
   * @param v the {@link SBMLVisitor} instance to be used.
   * <p>
   * @return the result of calling <code>v.visit()</code>.
   */ public
";


%javamethodmodifiers Reaction::clone() const "
/**
   * Creates and returns a deep copy of this {@link Reaction}.
   * <p>
   * @return a (deep) copy of this {@link Reaction}.
   */ public
";


%javamethodmodifiers Reaction::getElementBySId(std::string id) "
/**
   * Returns the first child element found that has the given id in the model-wide SId namespace, or null if no such object is found.
   * <p>
   * @param id string representing the id of objects to find.
   * <p>
   * @return pointer to the first element found with the given id.
   */ public
";


%javamethodmodifiers Reaction::getElementByMetaId(std::string metaid) "
/**
   * Returns the first child element it can find with the given metaid, or null if no such object is found.
   * <p>
   * @param metaid string representing the metaid of objects to find
   * <p>
   * @return pointer to the first element found with the given metaid.
   */ public
";


%javamethodmodifiers Reaction::getAllElements "
/**
   * Returns a {@link List} of all child {@link SBase} objects, including those nested to an arbitrary depth
   * <p>
   * @return a {@link List} of pointers to all children objects.
   */ public
";


%javamethodmodifiers Reaction::renameSIdRefs(std::string oldid, std::string newid) "
/**
   * Renames all the SIdRef attributes on this element, including any found in MathML
   */ public
";


%javamethodmodifiers Reaction::initDefaults "
/**
   * Initializes the fields of this {@link Reaction} object to \'typical\' default
   * values.
   * <p>
   * The SBML {@link Reaction} component has slightly different aspects and
   * default attribute values in different SBML Levels and Versions.
   * This method sets the values to certain common defaults, based
   * mostly on what they are in SBML Level&nbsp;2.  Specifically:
   * <ul>
   * <li> Sets the \'reversible\' attribute to <code>true</code>
   * <li> Sets the \'fast\' attribute to <code>false</code>
   * <li> Marks the \'fast\' attribute as <em>not</em> having been set
   * </ul>
   * <p>
   * @warning The \'fast\' attribute must be used with care.  SBML
   * definitions before SBML Level&nbsp;2 Version&nbsp;2 incorrectly
   * indicated that software tools could ignore this attribute if they did
   * not implement support for the corresponding concept; however, further
   * research in SBML has revealed that this is not true, and \'fast\'
   * <em>cannot be ignored</em> if it is set to <code>true.</code>  Beginning with
   * SBML Level&nbsp;2 Versions&nbsp;2, the SBML specifications therefore
   * stipulate that if a model has any reactions with \'fast\' set to 
   * <code>true</code>, a software tool must be able to respect the attribute or else
   * indicate to the user that it does not have the capacity to do so.
   * Readers are directed to the SBML specifications, which provides more
   * detail about the conditions under which a reaction can be considered
   * to be fast in this sense.
   */ public
";


%javamethodmodifiers Reaction::getId() const "
/**
   * Returns the value of the \'id\' attribute of this {@link Reaction}.
   * <p>
   * @return the id of this {@link Reaction}.
   */ public
";


%javamethodmodifiers Reaction::getName() const "
/**
   * Returns the value of the \'name\' attribute of this {@link Reaction}.
   * <p>
   * @return the name of this {@link Reaction}.
   */ public
";


%javamethodmodifiers Reaction::getKineticLaw() const "
/**
   * Returns the {@link KineticLaw} object contained in this {@link Reaction}.
   * <p>
   * @return the {@link KineticLaw} instance.
   */ public
";


%javamethodmodifiers Reaction::getKineticLaw "
/**
   * Returns the {@link KineticLaw} object contained in this {@link Reaction}.
   * <p>
   * @return the {@link KineticLaw} instance.
   */ public
";


%javamethodmodifiers Reaction::getReversible() const "
/**
   * Returns the value of the \'reversible\' attribute on the {@link Reaction} as a
   * boolean value.
   * <p>
   * @return the reversibility status of this {@link Reaction}.
   */ public
";


%javamethodmodifiers Reaction::getFast() const "
/**
   * Returns the value of the \'fast\' attribute of this {@link Reaction}.
   * <p>
   * @return the \'fast\' status of this {@link Reaction}.
   * <p>
   * @warning SBML definitions before SBML Level&nbsp;2 Version&nbsp;2 incorrectly
   * indicated that software tools could ignore this attribute if they did
   * not implement support for the corresponding concept; however, further
   * research in SBML has revealed that this is not true, and \'fast\'
   * <em>cannot be ignored</em> if it is set to <code>true.</code>  SBML Level&nbsp;2
   * Versions 2, 3 and 4 therefore stipulate that if a model has any reactions
   * with \'fast\' set to <code>true</code>, a software tool must be able to respect
   * the attribute or else indicate to the user that it does not have the
   * capacity to do so.  Readers are directed to the SBML Level&nbsp;2 Version
   * 4 specification, which provides more detail about the conditions under
   * which a reaction can be considered to be fast in this sense.
   */ public
";


%javamethodmodifiers Reaction::getCompartment() const "
/**
   * (SBML Level&nbsp;3 only) Returns the value of the \'compartment\'
   * attribute on the {@link Reaction}.
   * <p>
   * @return the compartment of this {@link Reaction}.
   * <p>
   * @note The \'compartment\' attribute is available in SBML Level&nbsp;3
   * Version&nbsp;1 Core, but is not present on {@link Reaction} in lower Levels of
   * SBML.
   */ public
";


%javamethodmodifiers Reaction::isSetId() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Reaction}\'s \'id\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'id\' attribute of this {@link Reaction} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Reaction::isSetName() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Reaction}\'s \'name\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'name\' attribute of this {@link Reaction} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Reaction::isSetKineticLaw() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Reaction} contains a kinetic law object.
   * <p>
   * @return <code>true</code> if a {@link KineticLaw} is present in this {@link Reaction},, <code>false</code>
   * otherwise.
   */ public
";


%javamethodmodifiers Reaction::isSetFast() const "
/**
   * Predicate returning <code>true</code> if the value of
   * the \'fast\' attribute on this {@link Reaction}.
   * <p>
   * @return <code>true</code> if the \'fast\' attribute is true, <code>false</code> otherwise.
   * <p>
   * @warning SBML definitions before SBML Level&nbsp;2 Version&nbsp;2 incorrectly
   * indicated that software tools could ignore this attribute if they did
   * not implement support for the corresponding concept; however, further
   * research in SBML has revealed that this is not true, and \'fast\'
   * <em>cannot be ignored</em> if it is set to <code>true.</code>  SBML Level&nbsp;2
   * Versions 2, 3 and 4 therefore stipulate that if a model has any reactions
   * with \'fast\' set to <code>true</code>, a software tool must be able to respect
   * the attribute or else indicate to the user that it does not have the
   * capacity to do so.  Readers are directed to the SBML Level&nbsp;2 Version
   * 4 specification, which provides more detail about the conditions under
   * which a reaction can be considered to be fast in this sense.  Note
   * also that in SBML Level&nbsp;1, \'fast\' is defined as optional with a
   * default of <code>false</code>, which means it is effectively always set.
   */ public
";


%javamethodmodifiers Reaction::isSetCompartment() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Reaction}\'s \'compartment\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'compartment\' attribute of this {@link Reaction} is
   * set, <code>false</code> otherwise.
   * <p>
   * @note The \'compartment\' attribute is available in SBML
   * Level&nbsp;3 Version&nbsp;1 Core, but is not present on {@link Reaction} in
   * lower Levels of SBML.
   */ public
";


%javamethodmodifiers Reaction::isSetReversible() const "
/**
   * Predicate returning <code>true</code> if this
   * {@link Reaction}\'s \'reversible\' attribute is set.
   * <p>
   * @return <code>true</code> if the \'reversible\' attribute of this {@link Reaction} is
   * set, <code>false</code> otherwise.
   */ public
";


%javamethodmodifiers Reaction::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this {@link Reaction}.
   * <p>
   * The string <code>sid</code> is copied.  Note that SBML has strict requirements
   * for the syntax of identifiers.  The following is a summary of the definition of the SBML identifier type 
<code>SId</code>, which defines the permitted syntax of identifiers.  We
express the syntax using an extended form of BNF notation: 
<pre style=\'margin-left: 2em; border: none; font-weight: bold; font-size: 13px; color: black\'>
letter .= \'a\'..\'z\',\'A\'..\'Z\'
digit  .= \'0\'..\'9\'
idChar .= letter | digit | \'_\'
SId    .= ( letter | \'_\' ) idChar*
</pre>
The characters <code>(</code> and <code>)</code> are used for grouping, the
character <code>*</code> \'zero or more times\', and the character
<code>|</code> indicates logical \'or\'.  The equality of SBML identifiers is
determined by an exact character sequence match; i.e., comparisons must be
performed in a case-sensitive manner.  In addition, there are a few
conditions for the uniqueness of identifiers in an SBML model.  Please
consult the SBML specifications for the exact formulations.
<p>

   * <p>
   * @param sid the string to use as the identifier of this {@link Reaction}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Reaction::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this {@link Reaction}.
   * <p>
   * The string in <code>name</code> is copied.
   * <p>
   * @param name the new name for the {@link Reaction}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   */ public
";


%javamethodmodifiers Reaction::setKineticLaw(const KineticLaw* kl) "
/**
   * Sets the \'kineticLaw\' subelement of this {@link Reaction} to a copy of the
   * given {@link KineticLaw} object.
   * <p>
   * @param kl the {@link KineticLaw} object to use.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * </ul>
  */ public
";


%javamethodmodifiers Reaction::setReversible(bool value) "
/**
   * Sets the value of the \'reversible\' attribute of this {@link Reaction}.
   * <p>
   * @param value the value of the \'reversible\' attribute.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   */ public
";


%javamethodmodifiers Reaction::setFast(bool value) "
/**
   * Sets the value of the \'fast\' attribute of this {@link Reaction}.
   * <p>
   * @param value the value of the \'fast\' attribute.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * </ul>
   * <p>
   * @warning SBML definitions before SBML Level&nbsp;2 Version&nbsp;2 incorrectly
   * indicated that software tools could ignore this attribute if they did
   * not implement support for the corresponding concept; however, further
   * research in SBML has revealed that this is not true, and \'fast\'
   * <em>cannot be ignored</em> if it is set to <code>true.</code>  SBML Level&nbsp;2
   * Versions 2, 3 and 4 therefore stipulate that if a model has any reactions
   * with \'fast\' set to <code>true</code>, a software tool must be able to respect
   * the attribute or else indicate to the user that it does not have the
   * capacity to do so.  Readers are directed to the SBML Level&nbsp;2 Version
   * 4 specification, which provides more detail about the conditions under
   * which a reaction can be considered to be fast in this sense.
   */ public
";


%javamethodmodifiers Reaction::setCompartment(const std::string& sid) "
/**
   * Sets the value of the \'compartment\' attribute of this {@link Reaction}.
   * <p>
   * The string <code>sid</code> is copied.  
   * <p>
   * @param sid the string to use as the compartment of this {@link Reaction}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE }
   * </ul>
   * <p>
   * @note The \'compartment\' attribute is available in SBML
   * Level&nbsp;3 Version&nbsp;1 Core, but is not present on {@link Reaction} in
   * lower Levels of SBML.
   */ public
";


%javamethodmodifiers Reaction::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this {@link Reaction}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Reaction::unsetKineticLaw "
/**
   * Unsets the \'kineticLaw\' subelement of this {@link Reaction}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   */ public
";


%javamethodmodifiers Reaction::unsetFast "
/**
   * Unsets the value of the \'fast\' attribute of this {@link Reaction}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @warning In SBML Level&nbsp;1, \'fast\' is optional with a default of 
   * <code>false</code>, which means it is effectively always set (and reset to <code>false</code>
   * if this method is called).  Further, SBML definitions before SBML
   * Level&nbsp;2 Version&nbsp;2 incorrectly indicated that software tools could
   * ignore this attribute if they did not implement support for the
   * corresponding concept; however, further research in SBML has revealed
   * that this is not true, and \'fast\' <em>cannot be ignored</em> if it is
   * set to <code>true.</code>  SBML Level&nbsp;2 Versions 2, 3 and 4 therefore stipulate
   * that if a model has any reactions with \'fast\' set to <code>true</code>, a
   * software tool must be able to respect the attribute or else indicate
   * to the user that it does not have the capacity to do so.  Readers are
   * directed to the SBML Level&nbsp;2 Version&nbsp;4 specification, which provides
   * more detail about the conditions under which a reaction can be
   * considered to be fast in this sense.
   */ public
";


%javamethodmodifiers Reaction::unsetCompartment "
/**
   * Unsets the value of the \'compartment\' attribute of this {@link Reaction}.
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note The \'compartment\' attribute is available in SBML
   * Level&nbsp;3 Version&nbsp;1 Core, but is not present on {@link Reaction} in
   * lower Levels of SBML.
   */ public
";


%javamethodmodifiers Reaction::addReactant(const SpeciesReference* sr) "
/**
   * Adds a given {@link SpeciesReference} object as a reactant in this {@link Reaction}.
   * <p>
   * The {@link SpeciesReference} instance in <code>sr</code> is copied.
   * <p>
   * @param sr a {@link SpeciesReference} object referring to a {@link Species} in the
   * enclosing {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Reaction}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Reaction}</em>.  In addition, the caller should make
   * sure to free the original object if it is no longer being used, or
   * else a memory leak will result.  Please see {@link Reaction#createReactant()}
   * for a method that does not lead to these issues.
   * <p>
   * @see #createReactant()
   */ public
";


%javamethodmodifiers Reaction::addProduct(const SpeciesReference* sr) "
/**
   * Adds a given {@link SpeciesReference} object as a product in this {@link Reaction}.
   * <p>
   * The {@link SpeciesReference} instance in <code>sr</code> is copied.
   * <p>
   * @param sr a {@link SpeciesReference} object referring to a {@link Species} in the
   * enclosing {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Reaction}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Reaction}</em>.  In addition, the caller should make
   * sure to free the original object if it is no longer being used, or
   * else a memory leak will result.  Please see {@link Reaction#createProduct()}
   * for a method that does not lead to these issues.
   * <p>
   * @see #createProduct()
   */ public
";


%javamethodmodifiers Reaction::addModifier(const ModifierSpeciesReference* msr) "
/**
   * Adds a given {@link ModifierSpeciesReference} object as a product in this
   * {@link Reaction}.
   * <p>
   * The {@link ModifierSpeciesReference} instance in <code>msr</code> is copied.
   * <p>
   * @param msr a {@link ModifierSpeciesReference} object referring to a {@link Species} in
   * the enclosing {@link Model}
   * <p>
   * @return integer value indicating success/failure of the
   * function.   The possible values
   * returned by this function are:
   * <ul>
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS }
   * <li> {@link  libsbmlConstants#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE }
   * <li> {@link  libsbmlConstants#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID }
   * <li> {@link  libsbmlConstants#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH }
   * <li> {@link  libsbmlConstants#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED }
   * </ul>
   * <p>
   * @note This method should be used with some caution.  The fact that
   * this method <em>copies</em> the object passed to it means that the caller
   * will be left holding a physically different object instance than the
   * one contained in this {@link Reaction}.  Changes made to the original object
   * instance (such as resetting attribute values) will <em>not affect the
   * instance in the {@link Reaction}</em>.  In addition, the caller should make
   * sure to free the original object if it is no longer being used, or
   * else a memory leak will result.  Please see {@link Reaction#createModifier()}
   * for a method that does not lead to these issues.
   * <p>
   * @see #createModifier()
   */ public
";


%javamethodmodifiers Reaction::createReactant "
/**
   * Creates a new {@link SpeciesReference}, adds it to this {@link Reaction}\'s list of
   * reactants, and returns it.
   * <p>
   * @return a new {@link SpeciesReference} object.
   */ public
";


%javamethodmodifiers Reaction::createProduct "
/**
   * Creates a new {@link SpeciesReference}, adds it to this {@link Reaction}\'s list of
   * products, and returns it.
   * <p>
   * @return a new {@link SpeciesReference} object.
   */ public
";


%javamethodmodifiers Reaction::createModifier "
/**
   * Creates a new {@link ModifierSpeciesReference}, adds it to this {@link Reaction}\'s
   * list of modifiers and returns it.
   * <p>
   * @return a new {@link ModifierSpeciesReference} object.
   */ public
";


%javamethodmodifiers Reaction::createKineticLaw "
/**
   * Creates a new {@link KineticLaw} object, installs it as this {@link Reaction}\'s
   * \'kineticLaw\' subelement, and returns it.
   * <p>
   * If this {@link Reaction} had a previous {@link KineticLaw}, it will be destroyed.
   * <p>
   * @return the new {@link KineticLaw} object
   */ public
";


%javamethodmodifiers Reaction::getListOfReactants() const "
/**
   * Returns the list of reactants in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as reactants in this reaction
   */ public
";


%javamethodmodifiers Reaction::getListOfReactants "
/**
   * Returns the list of reactants in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as reactants in this reaction
   */ public
";


%javamethodmodifiers Reaction::getListOfProducts() const "
/**
   * Returns the list of products in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as products in this reaction
   */ public
";


%javamethodmodifiers Reaction::getListOfProducts "
/**
   * Returns the list of products in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as products in this reaction
   */ public
";


%javamethodmodifiers Reaction::getListOfModifiers() const "
/**
   * Returns the list of modifiers in this {@link Reaction} object.
   * <p>
   * @return the {@link ListOfSpeciesReferences} containing the references to the
   * species acting as modifiers in this reaction
   */ public
";


%javamethodmodifiers Reaction::getL