/*-*- c++ -*-******************************************************************
 * Qwt Widget Library 
 * Copyright (C) 1997   Josef Wilgen
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#include <qpainter.h>
#include "qwt_arrbtn.h"
#include "qwt_math.h"
#include "qwt_symbol.h"


static const int MaxNum = 3;


/*!
  \brief Constructor
  \param num Number of arrows
  \param dir Direction. One of \c QwtArrowButton::Up, \c QwtArrowButton::Down,
  \c QwtArrowButton::Left, or \c QwtArrowButton::Right
  \param parent
  \param name Forwarded to QPushButton's ctor
*/
QwtArrowButton::QwtArrowButton (int num, Direction dir, QWidget *parent,
                const char *name)
: QPushButton (parent, name)
{
    d_num = qwtLim(num,1,MaxNum);
    d_dir = dir;
    setAutoDefault(FALSE);
}

/*!
  \brief The direction of the arrows
*/
QwtArrowButton::Direction QwtArrowButton::direction() const 
{ 
    return d_dir; 
}

/*!
  \brief The number of arrows
*/
int QwtArrowButton::num() const 
{ 
    return d_num; 
}

/*!
  \brief Draw the button label
  \sa The Qt Manual on QPushButton
*/
void QwtArrowButton::drawButtonLabel(QPainter *p)
{
    QwtSymbol sym;
    
    int i,ws, hs, offs;
    
    QRect r = this->rect();
    QRect symRect[MaxNum];

    r.setHeight(r.height() - 4);
    r.setWidth(r.width() - 4);
    r.moveBy(2,2);
    
    switch(d_dir)
    {
        case Up:
            sym.setStyle(QwtSymbol::UTriangle);
            break;
        case Down:
            sym.setStyle(QwtSymbol::DTriangle);
            break;
        case Right:
            sym.setStyle(QwtSymbol::RTriangle);
            break;
        case Left:
            sym.setStyle(QwtSymbol::LTriangle);
            break;
    }

    switch(d_dir)
    {
        case Up:
        case Down:
            hs = qwtMin(r.height() / 3 - 1, r.width() / 2 - 2);
            ws = hs * 2 - 1;
            offs = (r.height() - d_num * hs) / 2;
            for (i = 0; i < d_num; i++)
            {
                symRect[i] = QRect(r.left() + (r.width() - ws) / 2,
                       r.top() + offs + i * hs,
                       ws, hs);
            }
            break;

        case Left:
        case Right:
            ws = qwtMin(r.width() / 3 - 1, r.height() / 2 - 2);
            hs = ws * 2 - 1;
            offs = (r.width() - d_num * ws) / 2;
            for (i = 0; i < d_num; i++)
            {
                symRect[i] = QRect(r.left() + offs + i * ws,
                       r.top() + (r.height() - hs) / 2,
                       ws, hs);
            }
            break;
    }

    sym.setPen(colorGroup().text());
    sym.setBrush(colorGroup().text());

    p->save();

    for (i=0; i<d_num;i++)
       sym.draw(p, symRect[i]);

    p->restore();
}

//! \return Fixed/Expanding
QSizePolicy QwtArrowButton::sizePolicy() const
{
    return QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);
}

/*!
  \return a size hint
*/
QSize QwtArrowButton::sizeHint() const
{
    return minimumSizeHint();
}

/*!
  \brief Return a minimum size hint
*/
QSize QwtArrowButton::minimumSizeHint() const
{
    return QSize( 16, 16 );
}

