#include <stdlib.h>
#include <qapplication.h>
#include <qwt_plot.h>
#include <qwt_math.h>


const int PLOT_SIZE = 201;      // 0 to 200

//-----------------------------------------------------------------
//      data_plot.cpp
//
//  This example shows how to display time-varying data
//  using QwtPlot.
//
//-----------------------------------------------------------------

class DataPlot : public QwtPlot
{
public:
    DataPlot();

protected:
    virtual void timerEvent(QTimerEvent *e);

private:
    double x[PLOT_SIZE], y[PLOT_SIZE], z[PLOT_SIZE];
};


//
//  Initialize main window
//
DataPlot::DataPlot() 
{
    //  Initialize data
    for (int i = 0; i< PLOT_SIZE; i++)
    {
        x[i] = 0.5 * double(i);     // time axis
        y[i] = 0;
        z[i] = 0;
    }

    // Assign a title
    setTitle("Another Simple QwtPlot Demonstration");
    setAutoLegend(TRUE);

    // Insert new curves
    long cRight = insertCurve("Data Moving Right");
    long cLeft = insertCurve("Data Moving Left");

    // Set curve styles
    setCurvePen(cRight, QPen(red));
    setCurvePen(cLeft, QPen(blue));

    // Attach (don't copy) data. Both curves use the same x array.
    setCurveRawData(cRight, x, y, PLOT_SIZE);
    setCurveRawData(cLeft, x, z, PLOT_SIZE);

    //  Insert zero line at y = 0
    long mY = insertLineMarker("", QwtPlot::yLeft);     
    setMarkerYPos(mY, 0.0);

    // Set axis titles
    setAxisTitle(QwtPlot::xBottom, "Time/seconds");
    setAxisTitle(QwtPlot::yLeft, "Values");
    
    // Generate timer event every 50ms
    (void)startTimer(50);
}


//
//  Generate new values every 500ms. 
//
void DataPlot::timerEvent(QTimerEvent *)
{
    static double phase = 0.0;

    if (phase > (M_PI - 0.0001)) phase = 0;

    // y moves from left to right:
    // Shift y array right and assign new value to y[0].
    qwtShiftArray(y, PLOT_SIZE, 1);
    y[0] = sin(phase) * (-1.0 + 2.0 * double(rand()) / double(RAND_MAX));

    // z moves from right to left:
    // Shift z array left and assign new value to z[n-1].
    qwtShiftArray(z, PLOT_SIZE, -1);
    z[PLOT_SIZE - 1] = 0.8 - (2.0 * phase/M_PI) + 0.4 * 
        double(rand()) / double(RAND_MAX);

    // update the display
    replot();

    phase += M_PI * 0.02;
}

int main(int argc, char **argv)
{
    QApplication a(argc, argv);

    DataPlot plot;
    a.setMainWidget(&plot);

    plot.resize(500,300);
    plot.show();

    return a.exec(); 
}
