/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifndef PXMCHEM_POLCHEMDEF_H
#define PXMCHEM_POLCHEMDEF_H

#include "libpolyxmass-globals.h"
#include "pxmchem-ionizerule.h"
#include "pxmchem-calcopt.h"

#define POLCHEMDEF_XML_VERSION  "0.1"

typedef struct _PxmPolchemdef PxmPolchemdef;

struct _PxmPolchemdef
{
  /* version identifies the version of the polymer definition file
     format. This string, which can be transformed to a float for
     comparisons, is used to ensure that the file format of a polymer
     chemistry definition is readable by the library. When a polymer
     chemistry definition is saved, it will be saved with version
     string defined in #define POLCHEMDEF_XML_VERSION.
   */
  gchar *version;
  
  /* type identifies the polymer definition, like "protein" or "dna",
   * for example.
   */
  gchar *type;

  /* file fully qualifies the file which contains the polymer definition.
   */
  gchar *file;

  /* dir is the directory where all the housekeeping files and data
   * pertaining to a given polymer definition are stored. This is
   * the place where the monomer icon files are stored, for example.
   */
  gchar *dir;

  /* leftcap and rightcap are two actform strings that describe
   * how the polymer should be capped on its left and right ends.
   */
  gchar *leftcap;
  gchar *rightcap;

  gint codelen;
  
  /* A string made with each code of each monomer separated using '@',
   * like "@Ala@Tyr@Phe@".
   */
  gchar *delim_codes;

  /* A boolean value that tells if the polymer definition has been
   * modified or not. This is a convenience for the user of the 
   * PxmPolchemdef object.
   */
  gboolean modified;
  
  /* ionizerule is a PxmIonizerule structure that describes how 
   * the polymer is to be ionized.
   */
  PxmIonizerule *ionizerule;
  
  /* polymerGPA is an array of PxmPolymer instances which will hold
   * all the polymer sequences currently in memory that have the same
   * polymer type as type (which means that are of _this_ polymer
   * definition type.
   */
  GPtrArray *polymerGPA;

  /* atomsGPA is an array of PxmAtom instances holding the definitions
     of all the atoms available in the polymer definition.
  */
  GPtrArray *atomGPA;

  /* The GPtrArray arrays that will hold all the polymer definition
   * data pertaining to the chemistry: monomers, modifs, cleavespecifs...
   */
  GPtrArray *monomerGPA;
  GPtrArray *modifGPA;
  GPtrArray *cleavespecGPA;
  GPtrArray *fragspecGPA;
  
  /* The conventional system of prop instances' array.
   */
  GPtrArray *propGPA;
};


/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS, INITING FUNCTIONS ...
 */
PxmPolchemdef *
pxmchem_polchemdef_new (void);

gboolean
pxmchem_polchemdef_set_version (PxmPolchemdef * polchemdef, gchar *version);

gboolean
pxmchem_polchemdef_set_type (PxmPolchemdef * polchemdef, gchar *type);

gboolean
pxmchem_polchemdef_set_file (PxmPolchemdef * polchemdef, gchar *file);

gboolean
pxmchem_polchemdef_set_dir (PxmPolchemdef * polchemdef, gchar *dir);

gboolean
pxmchem_polchemdef_set_leftcap (PxmPolchemdef * polchemdef, gchar *leftcap);

gboolean
pxmchem_polchemdef_set_rightcap (PxmPolchemdef * polchemdef, gchar *rightcap);

gboolean
pxmchem_polchemdef_set_codelen (PxmPolchemdef *polchemdef, gint codelen);

gboolean
pxmchem_polchemdef_set_delim_codes (PxmPolchemdef * polchemdef, gchar *delim_codes);

void
pxmchem_polchemdef_set_modified (PxmPolchemdef *polchemdef, gboolean modified);


gboolean
pxmchem_polchemdef_get_modified (PxmPolchemdef *polchemdef);


gboolean
pxmchem_polchemdef_invert_modified (PxmPolchemdef *polchemdef);

gboolean
pxmchem_polchemdef_update_delim_codes (PxmPolchemdef * polchemdef, gchar delim);

/* INTEGRITY CHECKING FUNCTIONS
 */
gboolean
pxmchem_polchemdef_validate_all (PxmPolchemdef *polchemdef, gchar **valid);



/* LOCATING FUNCTIONS
 */
PxmPolchemdef *
pxmchem_polchemdef_get_ptr_by_type (GPtrArray *GPA, gchar *type);


gint
pxmchem_polchemdef_get_index_by_type (GPtrArray *GPA, gchar *type);


gint
pxmchem_polchemdef_get_index_by_ptr (GPtrArray *GPA, PxmPolchemdef *polchemdef);



/* UTILITY FUNCTIONS
 */
gchar*
pxmchem_polchemdef_make_delim_codes (PxmPolchemdef *polchemdef, gchar delim);

gchar *
pxmchem_polchemdef_get_current_xml_version (void);



/* XML-format TRANSACTIONS
 */
gchar *
pxmchem_polchemdef_format_xml_string_DTD (void);


gchar *
pxmchem_polchemdef_format_xml_string_polchemdefdata (PxmPolchemdef *polchemdef, 
					     gchar *indent, gint offset);

gchar *
pxmchem_polchemdef_get_xml_file_version (gchar *file);


PxmPolchemdef *
pxmchem_polchemdef_render_xml_node_polchemdefdata (xmlDocPtr xml_doc,
					   xmlNodePtr xml_node,
					   gpointer user_data);




PxmPolchemdef *
pxmchem_polchemdef_render_xml_file (gchar *file);





/* FREE'ING FUNCTIONS
 */
gboolean
pxmchem_polchemdef_free (PxmPolchemdef *polchemdef);


/* GPtrArray-RELATED FUNCTIONS
 */
gint
pxmchem_polchemdef_GPA_free (GPtrArray *GPA);


#endif /* PXMCHEM_POLCHEMDEF_H */
