/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include "pxmchem-polchemdef-plugins.h"
#include "pxmchem-monomer.h"
#include "pxmchem-modif.h"
#include "pxmchem-cleavespec.h"
#include "pxmchem-fragspec.h"



/* This one was declared extern in pxmchem-polchemdef-plugins.h
 */
PxmPolchemdef *
(*pxmchem_polchemdefdata_xml_node_render_plugin) (xmlDocPtr xml_doc,
						  xmlNodePtr xml_node,
						  gpointer data);



gpointer
pxmchem_polchemdefdata_xml_node_choose_renderer (gchar *version)
{

  if (version == NULL)
    return pxmchem_polchemdefdata_xml_node_renderer_v_none;
  
  if (0 == strcmp (version, "0.1"))
    return pxmchem_polchemdefdata_xml_node_renderer_v_0_1;
  

  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	_("%s@%d: plugin for version: '%s' has no registered xml renderer\n"),
	 __FILE__, __LINE__, version);
  
  return NULL;
}



PxmPolchemdef *
pxmchem_polchemdefdata_xml_node_renderer_v_none (xmlDocPtr xml_doc,
						 xmlNodePtr xml_node,
						 gpointer data)
{
  /* The xml node we are in is structured in this way (note no version
   * here):
   * 
   * <polchemdefdata>
   *   <type>protein</type>
   *   <leftcap>+H1</leftcap>
   *   <rightcap>+O1H1</rightcap>
   *   <codelen>1</codelen>
   *   <ionizerule>
   *     <actform>+H1</actform>
   *     <charge>1</charge>
   *   </ionizerule>
   *   <monomers>
   *     <mnm>
   *       <name>Glycine</name>
   *       <code>G</code>
   *       <formula>C2H3N1O1</formula>
   *     </mnm>
   *
   * And the xml_node parameter points to the 
   *
   * <polchemdefdata> element tag:
   *  ^
   *  |
   *  +----- here we are right now.
   * 
   * Which means that xml_node->name == "polchemdefdata" and that
   * we'll have to go one step down to the first child of the 
   * current node in order to get to the <type> element.
   *
   */
  PxmPolchemdef *polchemdef = NULL;

  PxmMonomer *monomer = NULL;
  PxmModif *modif = NULL;
  PxmCleaveSpec *cleavespec = NULL;
  PxmFragSpec *fragspec = NULL;
  
  gchar *help = NULL;

  gint test = 0;
  
  xmlNodePtr xml_child_node = NULL;



  /* Make sure we have parameters pointing bona fide to the right
   * xml element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == strcmp ((gchar *) xml_node->name, "polchemdefdata"));

  polchemdef = pxmchem_polchemdef_new ();

  /* Now go to the first child of current node: <type>.
   */
  xml_node = xml_node->children;
  
  /* From a rigorous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <type> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "type"));

  /* Since we have allocated the polchemdef instance at the line above, we
     know that its member data are NULL, so we can make direct
     assignements, without recoursing to the _set_xxx ().
  */
  polchemdef->type = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (polchemdef->type != NULL && strlen (polchemdef->type) > 0);
    
  /* Go to the next sibling, which according to DTD has to be 
   * element <leftcap>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <leftcap> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "leftcap"));
  
  polchemdef->leftcap = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (polchemdef->leftcap != NULL && strlen (polchemdef->leftcap) > 0);

  /* Go to the next sibling, which according to DTD has to be 
   * element <rightcap>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <rightcap> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "rightcap"));
  
  polchemdef->rightcap = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (polchemdef->rightcap != NULL
	    && strlen (polchemdef->rightcap) > 0);

  /* Go to the next sibling, which according to DTD has to be 
   * element <codelen>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <codelen> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "codelen"));
  
  help = (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);

  if (FALSE == libpolyxmass_globals_strtoi (help, &test, 10))
    {
      g_free (help);
      pxmchem_polchemdef_free (polchemdef);
      
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: failed to  convert from '%s' to gint\n"),
	     __FILE__, __LINE__, xml_node->xmlChildrenNode->content);
      
      return NULL;
    }
  
  g_free (help);
  pxmchem_polchemdef_set_codelen (polchemdef, test);
  
  /* Go to the next sibling, which according to DTD has to be 
   * element <ionizerule>. We do this and then copy the new 
   * pointer value to another variable, because <ionizerule> will
   * modify the received pointer to go to child nodes. So we want
   * to keep a copy of were we were in the xml tree before rendering
   * the ionizerule node.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <ionizerule> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "ionizerule"));

  /* Now deal with the <ionizerule> element, and ask the specific 
   * function to read the xml data and make the proper allocations.
   */
  xml_child_node = xml_node;
  
  /* Since we have allocated the polchemdef above in this same function,
   * we have by the same allocation allocated polchemdef->ionizerule in
   * pxmchem_polchemdef_new (), so we first have to free it because the
   * function below also will allocate one instance of ionizerule.
   */
  pxmchem_ionizerule_free (polchemdef->ionizerule);
  polchemdef->ionizerule = NULL;

  polchemdef->ionizerule = 
    pxmchem_ionizerule_render_xml_node_ionizerule (xml_doc, 
						   xml_child_node, NULL);
  
  if (polchemdef->ionizerule == NULL)
    {
      pxmchem_polchemdef_free (polchemdef);
      
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: failed to render xml ionizerule node\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  /* Now go to the next child of the <polchemdefdata> element, which
   * must be, according to the DTD, the <monomers> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <monomers> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "monomers"));
  
  /* The <monomers> element may contain no, one, more <mnm> elements.
   * So we should go to the child node, and if the gotten node is non
   * NULL it should be a <mnm> node.
   */
  xml_child_node = xml_node->children;
  while (TRUE == xmlIsBlankNode (xml_child_node))
    xml_child_node = xml_child_node->next;
  
  while (xml_child_node != NULL)
    {
      /* A node is found, which should be <mnm>.
       */
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "mnm"));
      
      monomer = pxmchem_monomer_render_xml_node_mnm (xml_doc,
						     xml_child_node,
						     NULL);
      
      if (monomer == NULL)
	{
	  pxmchem_polchemdef_free (polchemdef);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to create monomer from xml data\n"),
		 __FILE__, __LINE__);
      
	  return NULL;
	}

      g_ptr_array_add (polchemdef->monomerGPA, monomer);
      
      xml_child_node = xml_child_node->next;

      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
    }
  
  /* Now go to the next child of the <polchemdefdata> element, which
   * must be, according to the DTD, the <modifs> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <modifs> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "modifs"));
  
  /* The <modifs> element may contain no, one, more <mdf> elements.
   * So we should go to the child node, and if the gotten node is non
   * NULL it should be a <mdf> node.
   */
  xml_child_node = xml_node->children;
  while (TRUE == xmlIsBlankNode (xml_child_node))
    xml_child_node = xml_child_node->next;
  
  while (xml_child_node != NULL)
    {
      /* A node is found, which should be <mdf>.
       */
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "mdf"));
      
      modif = pxmchem_modif_render_xml_node_mdf (xml_doc,
					     xml_child_node,
					     NULL);
      
      if (modif == NULL)
	{
	  pxmchem_polchemdef_free (polchemdef);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to create modif from xml data\n"),
		 __FILE__, __LINE__);
      
	  return NULL;
	}
      
      g_ptr_array_add (polchemdef->modifGPA, modif);
            
      xml_child_node = xml_child_node->next;

      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
    }
  
  /* Now go to the next child of the <polchemdefdata> element, which
   * must be, according to the DTD, the <cleavespecs> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <cleavespecs> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "cleavespecs"));
  
  /* The <cleavespecs> element may contain no, one, more <cls> elements.
   * So we should go to the child node, and if the gotten node is non
   * NULL it should be a <cls> node.
   */
  xml_child_node = xml_node->children;
  while (TRUE == xmlIsBlankNode (xml_child_node))
    xml_child_node = xml_child_node->next;
  
  while (xml_child_node != NULL)
    {
      /* A node is found, which should be <cls>.
       */
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "cls"));
      
      cleavespec = pxmchem_cleavespec_render_xml_node_cls (xml_doc,
						       xml_child_node,
						       NULL);
      
      if (cleavespec == NULL)
	{
	  pxmchem_polchemdef_free (polchemdef);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to create cleavespec from xml data\n"),
		 __FILE__, __LINE__);
      
	  return NULL;
	}
      
      g_ptr_array_add (polchemdef->cleavespecGPA, cleavespec);
            
      xml_child_node = xml_child_node->next;

      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
    }
  
  /* Now go to the next child of the <polchemdefdata> element, which
   * must be, according to the DTD, the <fragspecs> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <fragspecs> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "fragspecs"));
  
  /* The <fragspecs> element may contain no, one, more <fgs> elements.
   * So we should go to the child node, and if the gotten node is non
   * NULL it should be a <fgs> node.
   */
  xml_child_node = xml_node->children;
  while (TRUE == xmlIsBlankNode (xml_child_node))
    xml_child_node = xml_child_node->next;
  
  while (xml_child_node != NULL)
    {
      /* A node is found, which should be <fgs>.
       */
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "fgs"));
      
      fragspec = pxmchem_fragspec_render_xml_node_fgs (xml_doc,
						       xml_child_node,
						       NULL);
      
      if (fragspec == NULL)
	{
	  pxmchem_polchemdef_free (polchemdef);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to create fragspec from xml data\n"),
		 __FILE__, __LINE__);
      
	  return NULL;
	}
      
      g_ptr_array_add (polchemdef->fragspecGPA, fragspec);
            
      xml_child_node = xml_child_node->next;

      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
    }
  
  /* At this point all the <polchemdefdata> node has been rendered.
   */

  return polchemdef;
}


PxmPolchemdef *
pxmchem_polchemdefdata_xml_node_renderer_v_0_1 (xmlDocPtr xml_doc,
						xmlNodePtr xml_node,
						gpointer data)
{
  /* The xml node we are in is structured in this way:
   * 
   * <polchemdefdata version="0.1">
   *   <type>protein</type>
   *   <leftcap>+H1</leftcap>
   *   <rightcap>+O1H1</rightcap>
   *   <codelen>1</codelen>
   *   <ionizerule>
   *     <actform>+H1</actform>
   *     <charge>1</charge>
   *   </ionizerule>
   *   <monomers>
   *     <mnm>
   *       <name>Glycine</name>
   *       <code>G</code>
   *       <formula>C2H3N1O1</formula>
   *     </mnm>
   *
   * And the xml_node parameter points to the 
   *
   * <polchemdefdata> element tag:
   *  ^
   *  |
   *  +----- here we are right now.
   * 
   * Which means that xml_node->name == "polchemdefdata" and that
   * we'll have to go one step down to the first child of the 
   * current node in order to get to the <type> element.
   *
   */
  PxmPolchemdef *polchemdef = NULL;

  PxmMonomer *monomer = NULL;
  PxmModif *modif = NULL;
  PxmCleaveSpec *cleavespec = NULL;
  PxmFragSpec *fragspec = NULL;
  
  gchar *help = NULL;

  gint test = 0;
  
  xmlNodePtr xml_child_node = NULL;



  /* Make sure we have parameters pointing bona fide to the right
   * xml element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == strcmp ((gchar *) xml_node->name, "polchemdefdata"));

  polchemdef = pxmchem_polchemdef_new ();

  polchemdef->version = 
    (gchar *) xmlGetProp (xml_node, (guchar *) "version");
  g_assert (polchemdef->version != NULL);
  
  
  /* Now go to the first child of current node: <type>.
   */
  xml_node = xml_node->children;
  
  /* From a rigorous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <type> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "type"));

  /* Since we have allocated the polchemdef instance at the line above, we
     know that its member data are NULL, so we can make direct
     assignements, without recoursing to the _set_xxx ().
  */
  polchemdef->type = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (polchemdef->type != NULL && strlen (polchemdef->type) > 0);
    
  /* Go to the next sibling, which according to DTD has to be 
   * element <leftcap>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <leftcap> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "leftcap"));
  
  polchemdef->leftcap = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (polchemdef->leftcap != NULL && strlen (polchemdef->leftcap) > 0);

  /* Go to the next sibling, which according to DTD has to be 
   * element <rightcap>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <rightcap> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "rightcap"));
  
  polchemdef->rightcap = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (polchemdef->rightcap != NULL
	    && strlen (polchemdef->rightcap) > 0);

  /* Go to the next sibling, which according to DTD has to be 
   * element <codelen>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <codelen> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "codelen"));
  
  help = (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);

  if (FALSE == libpolyxmass_globals_strtoi (help, &test, 10))
    {
      g_free (help);
      pxmchem_polchemdef_free (polchemdef);
      
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: failed to convert from '%s' to gint\n"),
	     __FILE__, __LINE__, xml_node->xmlChildrenNode->content);
      
      return NULL;
    }
  
  g_free (help);
  pxmchem_polchemdef_set_codelen (polchemdef, test);
  
  /* Go to the next sibling, which according to DTD has to be 
   * element <ionizerule>. We do this and then copy the new 
   * pointer value to another variable, because <ionizerule> will
   * modify the received pointer to go to child nodes. So we want
   * to keep a copy of were we were in the xml tree before rendering
   * the ionizerule node.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <ionizerule> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "ionizerule"));

  /* Now deal with the <ionizerule> element, and ask the specific 
   * function to read the xml data and make the proper allocations.
   */
  xml_child_node = xml_node;
  
  /* Since we have allocated the polchemdef above in this same function,
   * we have by the same allocation allocated polchemdef->ionizerule in
   * pxmchem_polchemdef_new (), so we first have to free it because the
   * function below also will allocate one instance of ionizerule.
   */
  pxmchem_ionizerule_free (polchemdef->ionizerule);
  polchemdef->ionizerule = NULL;

  polchemdef->ionizerule = 
    pxmchem_ionizerule_render_xml_node_ionizerule (xml_doc, 
						   xml_child_node, NULL);
  
  if (polchemdef->ionizerule == NULL)
    {
      pxmchem_polchemdef_free (polchemdef);
      
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: failed to render xml ionizerule node\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  /* Now go to the next child of the <polchemdefdata> element, which
   * must be, according to the DTD, the <monomers> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <monomers> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "monomers"));
  
  /* The <monomers> element may contain no, one, more <mnm> elements.
   * So we should go to the child node, and if the gotten node is non
   * NULL it should be a <mnm> node.
   */
  xml_child_node = xml_node->children;
  while (TRUE == xmlIsBlankNode (xml_child_node))
    xml_child_node = xml_child_node->next;
  
  while (xml_child_node != NULL)
    {
      /* A node is found, which should be <mnm>.
       */
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "mnm"));
      
      monomer = pxmchem_monomer_render_xml_node_mnm (xml_doc,
						     xml_child_node,
						     NULL);
      
      if (monomer == NULL)
	{
	  pxmchem_polchemdef_free (polchemdef);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to create monomer from xml data\n"),
		 __FILE__, __LINE__);
      
	  return NULL;
	}

      g_ptr_array_add (polchemdef->monomerGPA, monomer);
      
      xml_child_node = xml_child_node->next;

      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
    }
  
  /* Now go to the next child of the <polchemdefdata> element, which
   * must be, according to the DTD, the <modifs> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <modifs> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "modifs"));
  
  /* The <modifs> element may contain no, one, more <mdf> elements.
   * So we should go to the child node, and if the gotten node is non
   * NULL it should be a <mdf> node.
   */
  xml_child_node = xml_node->children;
  while (TRUE == xmlIsBlankNode (xml_child_node))
    xml_child_node = xml_child_node->next;
  
  while (xml_child_node != NULL)
    {
      /* A node is found, which should be <mdf>.
       */
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "mdf"));
      
      modif = pxmchem_modif_render_xml_node_mdf (xml_doc,
					     xml_child_node,
					     NULL);
      
      if (modif == NULL)
	{
	  pxmchem_polchemdef_free (polchemdef);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to create modif from xml data\n"),
		 __FILE__, __LINE__);
      
	  return NULL;
	}
      
      g_ptr_array_add (polchemdef->modifGPA, modif);
            
      xml_child_node = xml_child_node->next;

      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
    }
  
  /* Now go to the next child of the <polchemdefdata> element, which
   * must be, according to the DTD, the <cleavespecs> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <cleavespecs> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "cleavespecs"));
  
  /* The <cleavespecs> element may contain no, one, more <cls> elements.
   * So we should go to the child node, and if the gotten node is non
   * NULL it should be a <cls> node.
   */
  xml_child_node = xml_node->children;
  while (TRUE == xmlIsBlankNode (xml_child_node))
    xml_child_node = xml_child_node->next;
  
  while (xml_child_node != NULL)
    {
      /* A node is found, which should be <cls>.
       */
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "cls"));
      
      cleavespec = pxmchem_cleavespec_render_xml_node_cls (xml_doc,
						       xml_child_node,
						       NULL);
      
      if (cleavespec == NULL)
	{
	  pxmchem_polchemdef_free (polchemdef);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to create cleavespec from xml data\n"),
		 __FILE__, __LINE__);
      
	  return NULL;
	}
      
      g_ptr_array_add (polchemdef->cleavespecGPA, cleavespec);
            
      xml_child_node = xml_child_node->next;

      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
    }
  
  /* Now go to the next child of the <polchemdefdata> element, which
   * must be, according to the DTD, the <fragspecs> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <fragspecs> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "fragspecs"));
  
  /* The <fragspecs> element may contain no, one, more <fgs> elements.
   * So we should go to the child node, and if the gotten node is non
   * NULL it should be a <fgs> node.
   */
  xml_child_node = xml_node->children;
  while (TRUE == xmlIsBlankNode (xml_child_node))
    xml_child_node = xml_child_node->next;
  
  while (xml_child_node != NULL)
    {
      /* A node is found, which should be <fgs>.
       */
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "fgs"));
      
      fragspec = pxmchem_fragspec_render_xml_node_fgs (xml_doc,
						       xml_child_node,
						       NULL);
      
      if (fragspec == NULL)
	{
	  pxmchem_polchemdef_free (polchemdef);
	  
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: failed to create fragspec from xml data\n"),
		 __FILE__, __LINE__);
      
	  return NULL;
	}
      
      g_ptr_array_add (polchemdef->fragspecGPA, fragspec);
            
      xml_child_node = xml_child_node->next;

      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
    }
  
  /* At this point all the <polchemdefdata version="0.1"> node has
     been rendered.
   */

  return polchemdef;
}


