#include "osl/record/kisen.h"
#include "osl/stl/vector.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

#include <iostream>
#include <iterator>

class KisenTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE( KisenTest );
  CPPUNIT_TEST( testShow );
  CPPUNIT_TEST( testConsistent );
  CPPUNIT_TEST( testGetPlayer );
  CPPUNIT_TEST( testGetRating );
  CPPUNIT_TEST( testResult );
  CPPUNIT_TEST( testGetIpxFileName );
  CPPUNIT_TEST_SUITE_END();
 private:
  osl::Record rec;
  osl::SimpleState state;
  osl::vector<osl::Move> moves;
 public:
  void setUp();
  void testShow();
  void testConsistent();
  void testGetPlayer();
  void testGetRating();
  void testResult();
  void testGetIpxFileName();
};

CPPUNIT_TEST_SUITE_REGISTRATION( KisenTest );

using namespace osl;
extern bool isShortTest;

void KisenTest::setUp(){
}

const char *filename = OslConfig::testFile("kisen/01.kif");
const char *ipxFilename = OslConfig::testFile("kisen/01.ipx");

void KisenTest::testShow(){
  KisenFile kisenFile(filename);
  moves = kisenFile.getMoves(0);  
  CPPUNIT_ASSERT( !moves.empty() );
  if (! isShortTest)
      copy(moves.begin(), moves.end(), 
	   std::ostream_iterator<Move>( std::cout , "\n" ));
}

void KisenTest::testConsistent(){
  KisenFile kisenFile(filename);
  extern bool isShortTest;
  const size_t count=isShortTest ? (size_t)100u : std::min((size_t)10000u, kisenFile.size());
  for (size_t i=0;i<count;i++) {
    if (i==172518) 
      continue;
    SimpleState state=kisenFile.getInitialState();
    vector<Move> moves=kisenFile.getMoves(i);
    for (size_t j=0;j<moves.size();j++) {
      ApplyMoveOfTurn::doMove(state, moves[j]);
      CPPUNIT_ASSERT(state.isConsistent(true));
    }
  }
}

void KisenTest::testGetPlayer(){
  KisenIpxFile kisenIpxFile(ipxFilename);
  CPPUNIT_ASSERT("raymond"==kisenIpxFile.getPlayer(2,BLACK) ||
		 (std::cerr <<kisenIpxFile.getPlayer(2,BLACK) << std::endl,0)
		 );
  CPPUNIT_ASSERT("Yang_WengLi"==kisenIpxFile.getPlayer(2,WHITE) ||
		 (std::cerr <<kisenIpxFile.getPlayer(2,WHITE) << std::endl,0)
		 );
}

void KisenTest::testGetRating(){
  KisenIpxFile kisenIpxFile(ipxFilename);
  CPPUNIT_ASSERT_EQUAL(2149u,kisenIpxFile.getRating(6,BLACK));
  CPPUNIT_ASSERT_EQUAL(2165u,kisenIpxFile.getRating(6,WHITE));
}

void KisenTest::testResult(){
  KisenIpxFile kisenIpxFile(ipxFilename);
  KisenFile kisenFile(filename);
  extern bool isShortTest;
  const size_t count=isShortTest ? (size_t)100u : std::min((size_t)5000u, kisenFile.size());
  for(size_t i=0;i<count;i++){
    if(i==553 || i==1683 || i==2187 || i==2545 || i==2803 || i==4432 || i==4641) continue; // 王手放置
    moves = kisenFile.getMoves(i);  
    unsigned int result=kisenIpxFile.getResult(i);
    if(result==KisenIpxFile::BLACK_WIN || result==KisenIpxFile::BLACK_WIN_256){
      CPPUNIT_ASSERT((moves.size()%2)==1);
    }
    else if(result==KisenIpxFile::WHITE_WIN || result==KisenIpxFile::WHITE_WIN_256){
      CPPUNIT_ASSERT((moves.size()%2)==0);
    }
  }
}

void KisenTest::testGetIpxFileName() {
  KisenFile kisenFile(filename);
  CPPUNIT_ASSERT_EQUAL(std::string(ipxFilename), kisenFile.ipxFileName());
}


/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
