/* piecePairIndexTest.cc
 */
#include "osl/eval/ppair/piecePairIndex.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <bitset>
#include <iostream>
#include <fstream>

class PiecePairIndexTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PiecePairIndexTest);
  CPPUNIT_TEST(testForEachRelation);
  CPPUNIT_TEST(testUniqIndex);
  CPPUNIT_TEST(testIndex);
  CPPUNIT_TEST_SUITE_END();
public:
  void testUniqIndex();
  void testForEachRelation();
  void testIndex();
};

CPPUNIT_TEST_SUITE_REGISTRATION(PiecePairIndexTest);

using namespace osl;
using namespace osl::eval;
using namespace osl::eval::ppair;

typedef std::bitset<PiecePairIndex::maxPairIndex> set_t;
set_t visited;
void visitRelation(size_t index)
{
  size_t i1, i2;
  PiecePairIndex::meltIndex(index, i1, i2);
  const size_t i = PiecePairIndex::canonicalIndexOf(i1, i2);

  if (visited[i])
  {
    std::cerr << "oops " << i << "\n";
    size_t i1, i2;
    PiecePairIndex::meltIndex(i, i1, i2);
    std::cerr << i1 << " " << i2 << "\n";
    Position pos;
    PtypeO ptypeo;
    PiecePairIndex::meltIndex(i1, pos, ptypeo);
    std::cerr << pos << " " << ptypeo << "\n";
    PiecePairIndex::meltIndex(i2, pos, ptypeo);
    std::cerr << pos << " " << ptypeo << "\n";
  }
  CPPUNIT_ASSERT(! visited[i]);
  visited.flip(i);
}

void PiecePairIndexTest::testForEachRelation()
{
  PiecePairIndex::forEachRelation(visitRelation);
  size_t numCanons = 0;
  for (size_t i=0; i<PiecePairIndex::maxPairIndex; ++i)
  {
    size_t i1, i2;
    Position pos1, pos2;
    PtypeO ptypeo1=PTYPEO_MIN, ptypeo2=PTYPEO_MIN;

    PiecePairIndex::meltIndex(i, i1, i2);
    if (PiecePairIndex::canonicalIndexOf(i1, i2) != i)
      goto confirm_not_visited;

    PiecePairIndex::meltIndex(i1, pos1, ptypeo1);
    if (! pos1.isValid())
      goto confirm_not_visited;
    if (! isPiece(ptypeo1))
      goto confirm_not_visited;
    PiecePairIndex::meltIndex(i2, pos2, ptypeo2);
    if (! pos2.isValid())
      goto confirm_not_visited;
    if (! isPiece(ptypeo2))
      goto confirm_not_visited;
    if ((pos1 == pos2)
	&& pos1.isOnBoard()
	&& (ptypeo1 != ptypeo2))
      goto confirm_not_visited;
    if (pos1.isPieceStand() && isPromoted(ptypeo1))
      goto confirm_not_visited;
    if (pos2.isPieceStand() && isPromoted(ptypeo2))
      goto confirm_not_visited;

    // must be visited
    if (! visited[i])
    {
      std::cerr << "must visit " << i1 << " " << i2 << "\n"
		<< pos1 << " " << ptypeo1 << " " << pos2 << " " << ptypeo2
		<< "\n";
    }
    ++numCanons;
    continue;
  confirm_not_visited:
    if (visited[i])
    {
      std::cerr << "must not visit " << i1 << " " << i2 << "\n"
		<< pos1 << " " << ptypeo1 << " " << pos2 << " " << ptypeo2
		<< "\n";
    }
  }
  CPPUNIT_ASSERT_EQUAL((size_t)2578852, numCanons);
  CPPUNIT_ASSERT_EQUAL(numCanons, visited.count());
}


void PiecePairIndexTest::testUniqIndex()
{
  set_t s;
  s.flip(PiecePairIndex::positionIndexOf(Position::STAND()));
  for (int x=1; x<=9; ++x)
  {
    for (int y=1; y<=9; ++y)
    {
      const Position pos1 = Position(x,y);
      const unsigned int index = PiecePairIndex::positionIndexOf(pos1);
      CPPUNIT_ASSERT(index < PiecePairIndex::maxPositionIndex);
      CPPUNIT_ASSERT(! s.test(index));
      s.flip(index);
    }
  }
  s.reset();
  for (int p=PPAWN; p<=ROOK; ++p)
  {
    const Ptype ptype = static_cast<Ptype>(p);
    const unsigned int indexBlack = PiecePairIndex::ptypeOIndexOf(newPtypeO(BLACK,ptype));
    const unsigned int indexWhite = PiecePairIndex::ptypeOIndexOf(newPtypeO(WHITE,ptype));
    
    CPPUNIT_ASSERT(indexBlack < PiecePairIndex::maxPtypeOIndex);
    CPPUNIT_ASSERT(! s.test(indexBlack));
    s.flip(indexBlack);
    CPPUNIT_ASSERT(indexWhite < PiecePairIndex::maxPtypeOIndex);
    CPPUNIT_ASSERT(! s.test(indexWhite));
    s.flip(indexWhite);
  }
}

/** 本当はmember 変数にするべきだが手抜き*/
static set_t dejavu;
void testPieces(Piece p1, Piece p2)
{
  const unsigned int index12 = PiecePairIndex::indexOf(p1, p2);
  // std::cerr << index12 << " " << p1 << " " << p2 << "\n";
  CPPUNIT_ASSERT(index12 < PiecePairIndex::maxPairIndex);
  CPPUNIT_ASSERT(! dejavu.test(index12)
		 || (std::cerr << p1 << p2, 0));
  dejavu.flip(index12);
  if ((p1.position() == p2.position())
      && (p1.ptype() == p2.ptype())
      && (p1.owner() == p2.owner()))
    return;
  
  const unsigned int index21 = PiecePairIndex::indexOf(p2, p1);
  // std::cerr << index21 << " " << p2 << "\n";
  CPPUNIT_ASSERT(index21 < PiecePairIndex::maxPairIndex);
  CPPUNIT_ASSERT(! dejavu.test(index21)
		 || (std::cerr << p1 << p2, 0));
  dejavu.flip(index21);
}

void PiecePairIndexTest::testIndex()
{
  for (int x=1; x<=9; ++x)
  {
    std::cerr << x << " ";
    for (int y=1; y<=9; ++y)
    {
      const Position pos1 = Position(x,y);
      for (int ptype=PPAWN; ptype<=PTYPE_MAX; ++ptype)
      {
	const Piece p1 = Piece(BLACK, static_cast<Ptype>(ptype), 0/*?*/, pos1);
	const unsigned int index = PiecePairIndex::indexOf(p1);
	CPPUNIT_ASSERT(index < PiecePairIndex::maxPieceIndex
		       || (std::cerr << p1 << "\n" << index << "\n" << PiecePairIndex::maxPieceIndex << "\n", 0));
	{
	  PtypeO ptypeo;
	  Position pos;
	  PiecePairIndex::meltIndex(index, pos, ptypeo);
	  CPPUNIT_ASSERT_EQUAL(newPtypeO(BLACK, static_cast<Ptype>(ptype)),
			       ptypeo);
	  CPPUNIT_ASSERT_EQUAL(pos1, pos);
	}
	const Piece p1w = Piece(WHITE, static_cast<Ptype>(ptype), 0/*?*/, pos1);
	CPPUNIT_ASSERT(PiecePairIndex::indexOf(p1w) < PiecePairIndex::maxPieceIndex);
	{
	  PtypeO ptypeo;
	  Position pos;
	  PiecePairIndex::meltIndex(PiecePairIndex::indexOf(p1w),
					pos, ptypeo);
	  CPPUNIT_ASSERT_EQUAL(newPtypeO(WHITE, static_cast<Ptype>(ptype)),
			       ptypeo);
	  CPPUNIT_ASSERT_EQUAL(pos1, pos);
	}

	testPieces(p1, p1);
	testPieces(p1, p1w);
	testPieces(p1w, p1w);

	for (int x2=x; x2<=9; ++x2)
	{
	  for (int y2=y; y2<=9; ++y2)
	  {
	    Position pos2 = Position(x2,y2);
	    if (pos1 == pos2)
	      continue;
	    
	    for (int ptype2=PPAWN; ptype2<=PTYPE_MAX; ++ptype2)
	    {
	      const Piece p2 = Piece(BLACK, static_cast<Ptype>(ptype2), 0/*?*/, pos2);
	      const unsigned int index2 = PiecePairIndex::indexOf(p2);
	      CPPUNIT_ASSERT(index2 < PiecePairIndex::maxPieceIndex
			     || (std::cerr << p1 << "\n" << index << "\n" << PiecePairIndex::maxPieceIndex << "\n", 0));
	      const Piece p2w = Piece(WHITE, static_cast<Ptype>(ptype2), 0/*?*/, pos2);
	      CPPUNIT_ASSERT(PiecePairIndex::indexOf(p2w) < PiecePairIndex::maxPieceIndex);

	      testPieces(p1, p2);
	      testPieces(p1, p2w);
	      testPieces(p1w, p2);
	      testPieces(p1w, p2w);
	    }
	  }
	}

	for (int ptype2=KING; ptype2<=PTYPE_MAX; ++ptype2)
	{
	  const Piece p2 = Piece(BLACK, static_cast<Ptype>(ptype2), 0/*?*/, Position::STAND());
	  const unsigned int index2 = PiecePairIndex::indexOf(p2);
	  CPPUNIT_ASSERT(index2 < PiecePairIndex::maxPieceIndex
			 || (std::cerr << p1 << "\n" << index << "\n" << PiecePairIndex::maxPieceIndex << "\n", 0));
	  const Piece p2w = Piece(WHITE, static_cast<Ptype>(ptype2), 0/*?*/, Position::STAND());
	  CPPUNIT_ASSERT(PiecePairIndex::indexOf(p2w) < PiecePairIndex::maxPieceIndex);

	  testPieces(p1, p2);
	  testPieces(p1, p2w);
	  testPieces(p1w, p2);
	  testPieces(p1w, p2w);
	}
      }	// ptype
    } // x
  } // y

  // 持駒同士は最後に
  for (int ptype2=KING; ptype2<=PTYPE_MAX; ++ptype2)
  {
    const Piece p2 = Piece(BLACK, static_cast<Ptype>(ptype2), 0/*?*/, Position::STAND());
    const unsigned int index2 = PiecePairIndex::indexOf(p2);
    CPPUNIT_ASSERT(index2 < PiecePairIndex::maxPieceIndex);
    const Piece p2w = Piece(WHITE, static_cast<Ptype>(ptype2), 0/*?*/, Position::STAND());
    CPPUNIT_ASSERT(PiecePairIndex::indexOf(p2w) < PiecePairIndex::maxPieceIndex);
    testPieces(p2, p2);
    testPieces(p2, p2w);
    testPieces(p2w, p2w);
  }
#if 0
  std::cerr << PiecePairIndex::maxPieceIndex << "\n";
  std::cerr << PiecePairIndex::maxPairIndex << "\n";
#endif
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
