/* simpleState.cc
 */

#include "osl/state/simpleState.h"
#include "osl/state/simpleState.tcc"
#include "osl/apply_move/applyMove.h"
#include "osl/record/csa.h"
#include "osl/record/csaIOError.h"
#include "osl/pieceTable.h"
#include "osl/pieceStand.h"
#include <boost/foreach.hpp>
#include <iostream>
#include <stdexcept>

osl::SimpleState::SimpleState() {
  init();
}

osl::SimpleState::SimpleState(Handicap h) {
  init(h);
}

void osl::SimpleState::initPawnMask(){
  BOOST_FOREACH(Ptype ptype, PieceStand::order) {
    stand_count[BLACK][ptype - PTYPE_BASIC_MIN] = countPiecesOnStandBit(BLACK, ptype);
    stand_count[WHITE][ptype - PTYPE_BASIC_MIN] = countPiecesOnStandBit(WHITE, ptype);
  }

  pawnMask[0].clearAll();
  pawnMask[1].clearAll();
  for(int num=PtypeTraits<PAWN>::indexMin;
      num< PtypeTraits<PAWN>::indexLimit; num++){
    Piece p=getPieceOf(num);
    Player player=p.owner();
    Position pos=p.position();
    if(!pos.isPieceStand() && !p.isPromotedNotKingGold()){
      if (isPawnMaskSet(player,pos.x()))
      {
	throw CsaIOError("2FU!");
      }
      pawnMask[player].set(pos);
    }
  }
  assert(isConsistent(true));
}

void osl::SimpleState::init() {
  turn=BLACK;
  for (int ipos=0;ipos<Position::SIZE;ipos++) {
    setBoard(Position::nth(ipos),Piece::EDGE());
  }
  for (int y=1;y<=9;y++)
    for (int x=9;x>0;x--) {
      setBoard(Position(x,y),Piece::EMPTY());
    }
  //  promoteMask.clearAll();
  stand_mask[BLACK].resetAll();
  stand_mask[WHITE].resetAll();
  stand_count[BLACK].fill(0);
  stand_count[WHITE].fill(0);
  used_mask.resetAll();
  pawnMask[0].clearAll();
  pawnMask[1].clearAll();
}
  

void osl::SimpleState::init(Handicap h) {
  init();
  if (h != HIRATE) {
    std::cerr << "unsupported handicap\n";
    throw std::runtime_error("unsupported handicap");
  }
  // 歩
  for (int x=9;x>0;x--) {
    setPiece(BLACK,Position(x,7),PAWN);
    setPiece(WHITE,Position(x,3),PAWN);
  }
  // 
  setPiece(BLACK,Position(1,9),LANCE);
  setPiece(BLACK,Position(9,9),LANCE);
  setPiece(WHITE,Position(1,1),LANCE);
  setPiece(WHITE,Position(9,1),LANCE);
  //
  setPiece(BLACK,Position(2,9),KNIGHT);
  setPiece(BLACK,Position(8,9),KNIGHT);
  setPiece(WHITE,Position(2,1),KNIGHT);
  setPiece(WHITE,Position(8,1),KNIGHT);
  //
  setPiece(BLACK,Position(3,9),SILVER);
  setPiece(BLACK,Position(7,9),SILVER);
  setPiece(WHITE,Position(3,1),SILVER);
  setPiece(WHITE,Position(7,1),SILVER);
  //
  setPiece(BLACK,Position(4,9),GOLD);
  setPiece(BLACK,Position(6,9),GOLD);
  setPiece(WHITE,Position(4,1),GOLD);
  setPiece(WHITE,Position(6,1),GOLD);
  //
  setPiece(BLACK,Position(5,9),KING);
  setPiece(WHITE,Position(5,1),KING);
  //
  setPiece(BLACK,Position(8,8),BISHOP);
  setPiece(WHITE,Position(2,2),BISHOP);
  //
  setPiece(BLACK,Position(2,8),ROOK);
  setPiece(WHITE,Position(8,2),ROOK);

  initPawnMask();
}
  

osl::SimpleState::~SimpleState() {}

void osl::SimpleState::setPiece(Player player,Position pos,Ptype ptype) {
  int num;
  for (num=0;num<40;num++) {
    if (!used_mask.test(num) && Piece_Table.getPtypeOf(num)==unpromote(ptype)
	&& (ptype!=KING || 
	    num==PtypeTraits<KING>::indexMin+playerToIndex(player))) {
      used_mask.set(num);
	
      Piece p(player,ptype,num,pos);
      setPieceOf(num,p);
      if (pos.isPieceStand())
	stand_mask[player].set(num);
      else{
	setBoard(pos,p);
	if (ptype==PAWN)
	  pawnMask[player].set(pos);
      }
      return;
    }
  }
  std::cerr << "osl::SimpleState::setPiece! maybe too many pieces " 
	    << ptype << " " << pos << " " << player << "\n";
  abort();
}

void osl::SimpleState::setPieceAll(Player player) {
  for (int num=0;num<40;num++) {
    if (!used_mask.test(num)) {
      used_mask.set(num);
      stand_mask[player].set(num);
      Player pplayer = player;
      /* 片玉しかない問題のため */
      if (num==PtypeTraits<KING>::indexMin+playerToIndex(alt(player)))
      {
	pplayer=alt(player);
      }
      Piece p(pplayer,Piece_Table.getPtypeOf(num),num,Position::STAND());
      setPieceOf(num,p);
    }
  }
}
  
// check
bool osl::SimpleState::isConsistent(bool show_error) const
{
  // board上の要素のconsistency
  for (int y=1;y<=9;y++)
  {
    for (int x=9;x>=1;x--)
    {
      const Position pos(x,y);
      const Piece p0=getPieceAt(pos);
      if (p0.isPiece())
      {
	if (p0.position()!=pos)
	{
	  if (show_error) {
	    std::cerr << p0 << " must be put at " << pos << std::endl;
	  }
	  return false;
	}
	int num=p0.number();
	if (! PieceTable::validNumber(num) || !used_mask.test(num)) {
	  if (show_error) std::cerr << "NotUsed, num=" << num << std::endl;
	  return false;
	}
	Piece p1=getPieceOf(num);
	if (p0!=p1) {
	  if (show_error) std::cerr << "board[" << pos << "]!=" 
				    << "piece[" << num << "]" << std::endl;
	  return false;
	}
      }
    }
  }
  // piecesのconsistency
  for (int num0=0; num0<Piece::SIZE; num0++)
  {
    if (isOnBoard(num0))
    {
      Piece p0=getPieceOf(num0);
      Ptype ptype=p0.ptype();
      if (unpromote(ptype)!=Piece_Table.getPtypeOf(num0)) {
	if (show_error) std::cerr << "ptype of piece[" << num0 << "]=" 
				  << ptype << std::endl;
	return false;
      }
      if (!p0.isOnBoard()) {
	if (show_error) std::cerr << "mochigoma[" << num0 << "]=true" << std::endl;
	return false;
      }
      Position pos=p0.position();
      if (!pos.isOnBoard()) {
	if (show_error) std::cerr << "position " << pos << " is not onboard" << std::endl;
	return false;
      }
      Piece p1=getPieceAt(pos);
      int num1=p1.number();
      if (num0 !=num1) {
	if (show_error) std::cerr << "pieces[" << num0 << "]=" << p0 << ",board[" << pos << "] is " << p1 << std::endl;
	return false;
      }
    }
    else
    {
      Piece p0=getPieceOf(num0);
      Ptype ptype=p0.ptype();
#ifdef ALLOW_KING_ABSENCE
      if (p0.isEmpty() && Piece_Table.getPtypeOf(num0) == KING)
	continue;
#endif
      if (p0.number()!=num0) {
	if (show_error) 
	  std::cerr << "pieces[" << num0 << "] (" 
		    << Piece_Table.getPtypeOf(num0)  <<  ") ="
		    << p0 << std::endl;
	return false;
	  
      }
      if (ptype!=Piece_Table.getPtypeOf(num0)) {
	if (show_error) std::cerr << "ptype of piece[" << num0 << "]=" 
				  << ptype << std::endl;
	return false;
      }
      if (! p0.position().isPieceStand()) {
	if (show_error) std::cerr << p0 << " must be offboard" << std::endl;
	return false;
      }
    }
  }
  // mask
  BOOST_FOREACH(Ptype ptype, PieceStand::order) {
    if (countPiecesOnStand(BLACK, ptype) 
	!= countPiecesOnStandBit(BLACK, ptype)) {
      if (show_error) std::cerr << "count stand BLACK " << ptype << " inconsistent\n"
				<< *this << countPiecesOnStand(BLACK, ptype)
				<< " " << countPiecesOnStandBit(BLACK, ptype) << std::endl;
      return false;
    }
    if (countPiecesOnStand(WHITE, ptype)
	!= countPiecesOnStandBit(WHITE, ptype)) {
      if (show_error) std::cerr << "count stand WHITE " << ptype << " inconsistent\n" 
				<< *this << countPiecesOnStand(WHITE, ptype)
				<< " " << countPiecesOnStandBit(WHITE, ptype) << std::endl;
      return false;
    }
  }
  // pawnMask;
  {
    CArray<BitXmask,2> pawnMask1;
    pawnMask1[0].clearAll();
    pawnMask1[1].clearAll();
    for (int num=PtypeTraits<PAWN>::indexMin;
	 num<PtypeTraits<PAWN>::indexLimit;num++){
      if (isOnBoard(num)){
	Piece p=getPieceOf(num);
	if (!p.isPromotedNotKingGold()){
	  pawnMask1[playerToIndex(p.owner())].set(p.position());
	}
      }
    }
    if ((pawnMask[0]!=pawnMask1[0])
	|| (pawnMask[1]!=pawnMask1[1]))
    {
      if (show_error) 
	std::cerr << "pawnMask "
		  << pawnMask[0] << "!=" << pawnMask1[0] 
		  << " || " <<  pawnMask[1] << "!=" << pawnMask1[1]
		  << std::endl;
      return false;
    }
  }
  return true;
}

bool osl::SimpleState::isAlmostValidMove(Move move,bool show_error) const
{
  if (show_error)
  {
    const bool valid = isAlmostValidMove<true>(move);
    if (! valid)
      std::cerr << *this << " " << move << std::endl;
    return valid;
  }
  else
    return isAlmostValidMove<false>(move);
}

template <bool show_error>
bool osl::SimpleState::isAlmostValidMove(Move move) const
{
  assert(move.isValid());
  assert(getTurn() == move.player());
  assert(isValidMoveByRule(move, true));

  const Position from=move.from();
  const Position to=move.to();
  const Piece toPiece=getPieceAt(to);
  const Ptype ptype=move.ptype();
  const Player turn = move.player();
    
  if (from.isPieceStand()) // 打つ手
  {      
    // ターゲットが空白か
    if (! toPiece.isEmpty()) {
      if (show_error) std::cerr << "drop on to piece : " << move << std::endl;
      return false;
    }
    // そもそもその駒を持っているか?
    if (! hasPieceOnStand(turn,ptype)) {
      if (show_error) std::cerr << turn << " don't have : " << ptype << std::endl;
      return false;
    }
    // 二歩のチェック
    if (ptype==PAWN && isPawnMaskSet(turn, to.x())) {
      if (show_error) std::cerr << " Double Pawn : " << move << std::endl;
      return false;
    }
  }
  else
  {
    const Piece from_piece = getPieceAt(from);
    // fromにあるのがその駒か
    if (from_piece.isEmpty() 
	|| (from_piece.owner() != turn))
    {
      if (show_error) 
	std::cerr << " No such piece0 : " << move << std::endl;
      return false;
    }
    // promoteしている時にpromote可能か
    if (move.isPromote())
    {
      // fromにあるのがその駒か
      if (from_piece.ptype() != unpromote(move.ptype()))
      {
	if (show_error) 
	  std::cerr << " No such piece1  : " << move << std::endl;
	return false;
      }
      if (from_piece.isPromotedNotKingGold())
      {
	if (show_error) 
	  std::cerr << " can't promote promoted piece : " << move << std::endl;
	return false;
      }
    }
    else
    {
      // fromにあるのがその駒か
      if (from_piece.ptype() != move.ptype())
      {
	if (show_error) 
	  std::cerr << " No such piece2  : " << move << std::endl;
	return false;
      }
    }
    // toにあるのが，相手の駒か空白か?
    if (!toPiece.isEmpty() && toPiece.owner()==turn) {
      if (show_error) std::cerr << " No move on  : " << move << std::endl;
      return false;
    }
    // その offsetの動きがptypeに関してvalidか?
    EffectContent effect=Ptype_Table.getEffect(from_piece.ptypeO(),from,to);
    if (!effect.hasUnblockableEffect())
    {
      const Offset o=effect.offset();
      if (o.zero()) {
	if (show_error) {
	  std::cerr << " No such move2 : " << move << std::endl;
	}
	return false;
      }
      // 離れた動きの時に間が全部空いているか?
      for (Position p=from+o;p!=to;p+=o) {
	if (! getPieceAt(p).isEmpty()) {
	  if (show_error) 
	    std::cerr << " Not space to move : " << move << std::endl;
	  return false;
	}
      }
    }
    // capturePtypeが一致しているか?
    if (toPiece.ptype()!=move.capturePtype()) {
      if (show_error) std::cerr << " Not such capture : " << move 
				<< std::endl << *this;
      return false;
    }
  }

  assert(isValidMoveByRule(move, true));
  return true;
}

bool osl::SimpleState::isValidMoveByRule(Move move,bool show_error) const
{
  assert(move.isNormal());
  const Position from=move.from();
  const Position to=move.to();
  const Ptype ptype=move.ptype();
  const Player turn = move.player();
    
  if (from.isPieceStand()) // 打つ手
  { 
    // 動けない場所ではないか?
    if (! Ptype_Table.canDropTo(turn,ptype,to))
    {
      if (show_error) std::cerr << " can't drop to : " << move << std::endl;
      return false;
    }
  }
  else
  {
    const PtypeO old_ptypeo = move.oldPtypeO();
    const EffectContent effect
      = Ptype_Table.getEffect(old_ptypeo, Offset32(to,from));
    // その offsetの動きがptypeに関してvalidか?
    if (!effect.hasUnblockableEffect())
    {
      const Offset o = effect.offset();
      if (o.zero()) {
	if (show_error) {
	  std::cerr << " No such move1 : " << move << std::endl;
	}
	return false;
      }
    }
    // promoteしている時にpromote可能か
    if (move.isPromote())
    {
      if (! (canPromote(unpromote(move.ptype()))
	     && (to.canPromote(move.player()) 
		 || from.canPromote(move.player()))))
      {
	if (show_error) 
	  std::cerr << " illegal promote type or position : " << move << std::endl;
	return false;
      }
    }
    // promoteしていない時に強制promoteでないか?
    if ((! isPromoted(ptype)
	 && ! Ptype_Table.canDropTo(turn,getPtype(old_ptypeo),to)) 
	&& !move.isPromote()) 
    {
      if (show_error) 
	std::cerr << " must promote to this position : " << move << std::endl;
      return false;
    }
  }
  return true;
}

bool osl::SimpleState::isValidMove(Move move,bool show_error) const
{
  if (getTurn() != move.player()) {
    if (show_error) {
      std::cerr << "invalid player move : " << move << std::endl;
      std::cerr << *this;
    }
    return false;
  }
  if (! isValidMoveByRule(move, show_error))
    return false;
  return isAlmostValidMove(move, show_error);
}
  
#ifndef MINIMAL
bool osl::SimpleState::dump() const
{
  return std::cerr << *this << "\n";
}
#endif
  
void osl::SimpleState::doSimpleMove(Position from, Position to, int promoteMask)
{
  Piece oldPiece;
  int num;
  if(getTurn()==BLACK) {
    apply_move::ApplyDoUndoSimpleMove<BLACK,SimpleState>::
      prologue(*this, from, to, promoteMask, oldPiece, num);
  }
  else{
    apply_move::ApplyDoUndoSimpleMove<WHITE,SimpleState>::
      prologue(*this, from, to, promoteMask, oldPiece, num);
  }
  if (promoteMask!=0 && num<PtypeTraits<PAWN>::indexLimit)
    clearPawn(getTurn(), from);
}

void osl::SimpleState::doDropMove(Position to,Ptype ptype)
{
  Piece oldPiece;
  int num, numIndex, numLow;
  if(getTurn()==BLACK) {
    apply_move::ApplyDoUndoDropMove<BLACK,SimpleState>::
      prologue(*this, ptype, to, oldPiece, num, numIndex, numLow);
  }
  else{
    apply_move::ApplyDoUndoDropMove<WHITE,SimpleState>::
      prologue(*this, ptype, to, oldPiece, num, numIndex, numLow);
  }
  if (ptype==PAWN)
    setPawn(getTurn(),to);
}

void osl::SimpleState::doCaptureMove(Position from, Position to, Piece target,int promoteMask)

{
  Piece oldPiece;
  int num0, num1;
  int num1Index;
  mask_t num1Mask;
  Ptype capturePtype;
  if(getTurn()==BLACK) {
    apply_move::ApplyDoUndoCaptureMove<BLACK,SimpleState>::
      prologue(*this, from, to, target, promoteMask,
	       capturePtype, oldPiece, num0, num1, num1Index, num1Mask);
  }
  else{
    apply_move::ApplyDoUndoCaptureMove<WHITE,SimpleState>::
      prologue(*this, from, to, target, promoteMask,
	       capturePtype, oldPiece, num0, num1, num1Index, num1Mask);
  }
  if (capturePtype==PAWN)
    clearPawn(alt(getTurn()),to);
  if (promoteMask!=0 && num0<PtypeTraits<PAWN>::indexLimit)
    clearPawn(getTurn(),from);
}

/**
 * from で表現されたPieceをnew_ownerの持駒にした局面を作る.
 */
const osl::SimpleState 
osl::SimpleState::emulateCapture(Piece from, Player new_owner) const {
  osl::SimpleState newState;
  for(int i=0;i<40;i++){
    Piece p=getPieceOf(i);
    if(p==from){
      newState.setPiece(new_owner,Position::STAND(),unpromote(p.ptype()));
    }
    else{
      newState.setPiece(p.owner(),p.position(),p.ptype());
    }
  }
  newState.setTurn(getTurn());
  newState.initPawnMask();
  return newState;
}

/**
 * from からto に ptypeの持駒を一枚渡した局面を作る.
 */
const osl::SimpleState 
osl::SimpleState::emulateHandPiece(Player from, Player to, Ptype ptype) const {
  assert(hasPieceOnStand(from, ptype));
  assert(from==alt(to));
  osl::SimpleState newState;
  bool done=false;
  for(int i=0;i<40;i++){
    Piece p=getPieceOf(i);
    if(!done &&
       p.owner()==from &&
       !p.isOnBoard() &&
       p.ptype()==ptype){
      newState.setPiece(to,Position::STAND(),ptype);
      done=true;
    }
    else{
      newState.setPiece(p.owner(),p.position(),p.ptype());
    }
  }
  assert(done);
  newState.setTurn(getTurn());
  newState.initPawnMask();
  return newState;
}

const osl::state::SimpleState osl::state::SimpleState::rotate180() const
{
  SimpleState ret;
  for (int i=0; i<40; ++i) {
    const Piece p = getPieceOf(i);
    ret.setPiece(alt(p.owner()), p.position().rotate180Safe(), p.ptype());
  }
  ret.setTurn(alt(getTurn()));
  ret.initPawnMask();
  return ret;
}

const osl::state::SimpleState osl::state::SimpleState::flipHorizontal() const
{
  SimpleState ret;
  for (int i=0; i<40; ++i) {
    const Piece p = getPieceOf(i);
    ret.setPiece(p.owner(), p.position().flipHorizontal(), p.ptype());
  }
  ret.setTurn(getTurn());
  ret.initPawnMask();
  return ret;
}

bool osl::state::operator==(const SimpleState& st1,const SimpleState& st2)
{
  assert(st1.isConsistent(false));
  assert(st2.isConsistent(false));
  if (st1.turn!=st2.turn) 
    return false;
  if (st1.pawnMask[0]!=st2.pawnMask[0]) return false;
  if (st1.pawnMask[1]!=st2.pawnMask[1]) return false;
  for (int y=1;y<=9;y++)
    for (int x=9;x>0;x--) {
      Piece p1=st1.getPieceAt(Position(x,y));
      Piece p2=st2.getPieceAt(Position(x,y));
      if (p1.ptypeO()!=p2.ptypeO()) return false;
    }
  return true;
      
}

namespace osl
{
  namespace 
  {
    void showStand(std::ostream& os, Player player, PieceStand stand)
    {
      using namespace osl::record;
      if (! stand.any())
	return;
      
      os << "P" << csa::show(player);
      BOOST_FOREACH(Ptype ptype, PieceStand::order) {
	for (unsigned int j=0; j<stand.get(ptype); ++j)
	{
	  os << "00" << csa::show(ptype);
	}
      }
      os << "\n";
    }
  } // anonymous namespace
} // namespace osl

std::ostream& osl::state::operator<<(std::ostream& os,const SimpleState& state)
{
  for (int y=1;y<=9;y++) {
    os << 'P' << y;  
    for (int x=9;x>0;x--) {
      csaShow(os,state.getPieceOnBoard(Position(x,y)));
    }
    os << std::endl;
  }
  // 持ち駒の表示
  const PieceStand black_stand(BLACK, state);
  const PieceStand white_stand(WHITE, state);
  showStand(os, BLACK, black_stand);
  showStand(os, WHITE, white_stand);
  
  os << state.getTurn() << std::endl;
  return os;
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
