/* numEffectState.cc
 */
#include "osl/state/numEffectState.h"
#include "osl/state/numEffectState.tcc"
#include "osl/effect/numSimpleEffect.tcc"

#include <iostream>

bool osl::state::operator==(const NumEffectState& st1,
			    const NumEffectState& st2)
{
  assert(st1.isConsistent(true));
  assert(st2.isConsistent(true));
  if (!(st1.effects == st2.effects)) 
    return false;
  if (!(st1.onBoardMask == st2.onBoardMask)) 
    return false;
  if (!(st1.promoted == st2.promoted)) 
    return false;
  if (!(st1.pin_or_open == st2.pin_or_open)) 
    return false;
  if (!(st1.king_mobility == st2.king_mobility)) 
    return false;
  return (static_cast<const SimpleState&>(st1)
	  == static_cast<const SimpleState&>(st2));
}

const osl::checkmate::King8Info osl::state::
NumEffectState::king8Info(Player king) const
{
  return King8Info(Iking8Info(king));
}

template<osl::Player P>
void osl::state::NumEffectState::makeKing8Info()
{
  const Player altP=PlayerTraits<P>::opponent;
#ifdef ALLOW_KING_ABSENCE
  if (getKingPosition<P>().isPieceStand())
    return;
#endif
  king8infos[P]=King8Info::make<altP>(*this,getKingPosition<P>()).value;
}

osl::state::
NumEffectState::NumEffectState(const SimpleState& st) 
  : SimpleState(st),effects(st)
{
  onBoardMask[0].resetAll();
  onBoardMask[1].resetAll();
  promoted.resetAll();
  effects.effected_mask[0].resetAll();
  effects.effected_mask[1].resetAll();
  effects.effected_changed_mask[0].resetAll();
  effects.effected_changed_mask[1].resetAll();
  for(int num=0;num<40;num++){
    Piece p=getPieceOf(num);
    if (p.isOnBoard()){
      mutableOnBoardMask(p.owner()).set(num);
      if (p.isPromoted())
	promoted.set(num);
      for(int i=0;i<2;i++){
	Player pl=indexToPlayer(i);
	if(hasEffectBy(pl,p.position()))
	{
	  effects.effected_mask[i].set(num);
	  effects.effected_changed_mask[i].set(num);
	}
      }
    }
  }
  makePinOpen(BLACK);
  makePinOpen(WHITE);
  if(getKingPosition<BLACK>().isOnBoard())
    makeKing8Info<BLACK>();
  if(getKingPosition<WHITE>().isOnBoard())
    makeKing8Info<WHITE>();
}
osl::state::
NumEffectState::~NumEffectState() 
{
}

const osl::Piece osl::state::
NumEffectState::selectCheapPiece(PieceMask effect) const
{
  if (! effect.any())
    return Piece::EMPTY();
  mask_t pieces = effect.selectBit<PAWN>(), ppieces;
  if (pieces.any()) 
  {
    ppieces = pieces & promoted.getMask<PAWN>();
    pieces &= ~ppieces;
    if (pieces.any())
      return getPieceOf(pieces.bsf()+PtypeFuns<PAWN>::indexNum*32);
    return getPieceOf(ppieces.bsf()+PtypeFuns<PAWN>::indexNum*32);
  }
  pieces = effect.selectBit<LANCE>();
  if (pieces.any()) 
  {
    ppieces = pieces & promoted.getMask<LANCE>();
    pieces &= ~ppieces;
    if (pieces.any())
      return getPieceOf(pieces.bsf()+PtypeFuns<LANCE>::indexNum*32);
    return getPieceOf(ppieces.bsf()+PtypeFuns<LANCE>::indexNum*32);
  }
  mask_t king = effect.selectBit<KING>();
  effect.clearBit<KING>();
  if (effect.none())
    return getPieceOf(king.bsf()+PtypeFuns<KING>::indexNum*32);
  // depends on current piece numbers: <FU 0>, KE 18, GI 22, KI 26, <OU 30>, <KY 32>, KA 36, HI 38, 
#if OSL_WORDSIZE == 64
  const int index = 0;
#else
  const int index = effect.getMask(0).any() ? 0 : 1;
#endif
  ppieces = effect.getMask(index) & promoted.getMask(index);
  pieces = effect.getMask(index) & ~ppieces;
  const int num = pieces.any() 
    ? (ppieces.any() ? std::min(pieces.bsf(),ppieces.bsf()) : pieces.bsf()) 
    : ppieces.bsf();
  return getPieceOf(num + index*32);
}

const osl::Piece osl::state::
NumEffectState::findThreatenedPiece(Player P) const
{
  assert(! inCheck(P));
  PieceMask pieces = getOnBoardMask(P) & effectedMask(alt(P));
  PieceMask nolance = pieces; nolance.clearBit<LANCE>();
  int pp=-1, npp=-1, ret=-1;
  const int lance_index = PtypeFuns<LANCE>::indexNum; // 64bit: 0, 32bit: 1
  for (int i=lance_index; i>=0; --i) {
    mask_t all = nolance.getMask(i);
    mask_t promoted = all & promotedPieces().getMask(i);
    mask_t notpromoted = all & ~promoted;
    if (promoted.any()) {
      pp = promoted.bsr() + i*32;
      notpromoted &= ~Ptype_Table.getMaskLow(Piece_Table.getPtypeOf(pp));
    }
    if (notpromoted.any())
      npp = notpromoted.bsr() + i*32;
    ret = std::max(pp, npp);
    if (ret >= PtypeTraits<KNIGHT>::indexMin)
      return getPieceOf(ret);  
  }
  mask_t lance = pieces.selectBit<LANCE>();
  if (lance.any()) {
    mask_t plance = lance & promotedPieces().getMask(lance_index);
    if (plance.any())
      return getPieceOf(plance.bsr()+lance_index*32);
    return getPieceOf(lance.bsr()+lance_index*32);
  }
  if (ret >= 0) {
    assert(Piece_Table.getPtypeOf(ret) == PAWN);
    return getPieceOf(ret);
  }
  return Piece::EMPTY();
}

void osl::state::NumEffectState::
doSimpleMove(Position from, Position to, int promoteMask)
{
  Piece oldPiece;
  int num;
  PtypeO oldPtypeO, newPtypeO;
  CArray<PieceMask,2> pin_or_open_backup;
  KingMobility king_mobility_backup;
  PieceMask promoted_backup;
  CArray<PieceMask,2> effected_mask_backup;
  CArray<PieceMask,2> effected_changed_mask_backup;
  CArray<uint64_t,2> king8infos_backup;
  mobility::MobilityTable mobilityTable;
  if (getTurn()==BLACK){
    apply_move::ApplyDoUndoSimpleMove<BLACK,NumEffectState >
      ::prologue(*this, from, to, promoteMask, 
		 oldPiece, num, oldPtypeO, newPtypeO, 
		 pin_or_open_backup, king_mobility_backup,
		 promoted_backup, effected_mask_backup, effected_changed_mask_backup,king8infos_backup,mobilityTable);
  }
  else{
    apply_move::ApplyDoUndoSimpleMove<WHITE,NumEffectState >
      ::prologue(*this, from, to, promoteMask, 
		 oldPiece, num, oldPtypeO, newPtypeO, 
		 pin_or_open_backup, king_mobility_backup,
		 promoted_backup, effected_mask_backup, effected_changed_mask_backup,king8infos_backup,mobilityTable);
  }
  if (promoteMask!=0 && num < PtypeTraits<PAWN>::indexLimit)
    clearPawn(getTurn(),from);
}
void osl::state::NumEffectState::
doCaptureMove(Position from, Position to, Piece target, int promoteMask)
{
  Piece oldPiece;
  PtypeO oldPtypeO, capturePtypeO, newPtypeO;
  int num0, num1, num1Index;
  mask_t num1Mask;
  CArray<PieceMask,2> pin_or_open_backup;
  KingMobility king_mobility_backup;
  PieceMask promoted_backup;
  CArray<PieceMask,2> effected_mask_backup;
  CArray<PieceMask,2> effected_changed_mask_backup;
  CArray<uint64_t,2> king8infos_backup;
  mobility::MobilityTable mobilityTable;
  if(getTurn()==BLACK){
    apply_move::ApplyDoUndoCaptureMove<BLACK,NumEffectState >
      ::prologue(*this, from, to, target, promoteMask, oldPiece, oldPtypeO, 
		 capturePtypeO, newPtypeO, num0, num1, num1Index,num1Mask, 
		 pin_or_open_backup, king_mobility_backup,
		 promoted_backup, effected_mask_backup, effected_changed_mask_backup,king8infos_backup,mobilityTable);
  }
  else{
    apply_move::ApplyDoUndoCaptureMove<WHITE,NumEffectState >
      ::prologue(*this, from, to, target, promoteMask, oldPiece, oldPtypeO, 
		 capturePtypeO, newPtypeO, num0, num1, num1Index,num1Mask, 
		 pin_or_open_backup, king_mobility_backup,
		 promoted_backup, effected_mask_backup, effected_changed_mask_backup,king8infos_backup,mobilityTable);
  }
  const Ptype capturePtype=target.ptype();
  if (capturePtype==PAWN)
    clearPawn(alt(getTurn()),to);
  if (promoteMask!=0 && num0<PtypeTraits<PAWN>::indexLimit)
    clearPawn(getTurn(),from);
}

void osl::state::NumEffectState::
doDropMove(Position to,Ptype ptype)
{
  Piece oldPiece;
  PtypeO ptypeO;
  int num, numIndex;
  mask_t numMask;
  CArray<PieceMask,2> pin_or_open_backup;
  KingMobility king_mobility_backup;
  CArray<PieceMask,2> effected_mask_backup;
  CArray<PieceMask,2> effected_changed_mask_backup;
  CArray<uint64_t,2> king8infos_backup;
  mobility::MobilityTable mobilityTable;
  if(getTurn()==BLACK){
    apply_move::ApplyDoUndoDropMove<BLACK,NumEffectState >
      ::prologue(*this, to, ptype, oldPiece, num, ptypeO, numIndex, numMask, 
		 pin_or_open_backup, king_mobility_backup,
		 effected_mask_backup, effected_changed_mask_backup,king8infos_backup,mobilityTable);
  }
  else{
    apply_move::ApplyDoUndoDropMove<WHITE,NumEffectState >
      ::prologue(*this, to, ptype, oldPiece, num, ptypeO, numIndex, numMask, 
		 pin_or_open_backup, king_mobility_backup,
		 effected_mask_backup, effected_changed_mask_backup,king8infos_backup,mobilityTable);
  }
  if (ptype==PAWN)
    setPawn(getTurn(),to);
}

#ifndef MINIMAL
bool osl::state::NumEffectState::isConsistent(bool showError) const
{
  if (!SimpleState::isConsistent(showError)) 
  {
    if (showError)
      std::cerr << "error before effect\n";
    return false;
  }
  effect::NumSimpleEffectTable effects1(*this);
  if (!(effects1==effects))
  {
    if (showError)
    {
      std::cerr << "Effect error 1" << std::endl;
      std::cerr << *this;
      for(int y=1;y<=9;y++)
	for(int x=9;x>0;x--)
	{
	  Position pos(x,y);
	  if (!(effects1.getEffect(pos)==effects.getEffect(pos)))
	  {
	    std::cerr << pos << ",real=" << effects.getEffect(pos) << ",ideal=" << effects1.getEffect(pos) << std::endl;
	  }
	}
      for(int num=0;num<=39;num++){
	for(int i=0;i<8;i++){
	  Direction d=static_cast<Direction>(i);
	  if(effects.effectedNumTable[num][d]!=effects1.effectedNumTable[num][d]){
	    std::cerr << "piece=" << getPieceOf(num) << ",num=" << num << ",d=" << d << ",v1=" << effects.effectedNumTable[num][d] << ",v2=" << effects1.effectedNumTable[num][d] << std::endl;
	  }
	}
      }
      std::cerr << effects.effectedNumTable << std::endl;
    }
    return false;
  }
  for (int z=0; z<2; ++z) {
    const Player p = indexToPlayer(z);
#ifdef ALLOW_KING_ABSENCE
    if (getKingPosition(p).isPieceStand())
      continue;
#endif
    const PieceMask pin2 = effect_util::Pin::make(*this, p);
    if (pin(p) != pin2) {
      if (showError)
	std::cerr << "pin for " << p << " differs " << pin(p) << " " << pin2 << "\n";
      return false;
    }
    King8Info king8info2 = King8Info::make(alt(p), *this);
    if (King8Info(Iking8Info(p)).value != king8info2.value) {
      if (showError)
	std::cerr << "king8info for " << p << " differs \n" << King8Info(Iking8Info(p)) << "\n" << king8info2 << "\n";
      return false;
    }      
  }
  for (int i=0; i<Piece::SIZE; ++i) {
    const Piece p = getPieceOf(i);
    if (p.isOnBoard()) {
      if (promoted.test(i) != p.isPromoted()) {
	if (showError)
	  std::cerr << "promoted differs " << p << " " << promoted << " " << promoted.test(i) << "\n";
	return false;
      }
    }
  }
  return true;
}
#endif

bool osl::state::NumEffectState::isConsistent(const NumEffectState& prev, Move moved, bool show_error) const
{
#ifdef MINIMAL
  show_error=false;
#endif
  // test changedEffects
  const CArray<BoardMask,2> changed_squares
    = {{ changedEffects(BLACK), changedEffects(WHITE) }};
  const BoardMask changed_all = changed_squares[BLACK] | changed_squares[WHITE];
  CArray<BoardMask, Piece::SIZE> each_effect, prev_effect;
  for (int i=0; i<Piece::SIZE; ++i) {
    each_effect[i].clear();
    prev_effect[i].clear();
  }
  for (int x=1; x<=9; ++x) {
    for (int y=1; y<=9; ++y) {
      const Position sq(x, y);
      for (int i=0; i<Piece::SIZE; ++i) {
	if (getEffect(sq).test(i))
	  each_effect[i].set(sq);
	if (prev.getEffect(sq).test(i))
	  prev_effect[i].set(sq);
      }
      if (! changed_all.test(sq))
      {
	if (getEffect(sq) != prev.getEffect(sq)) {
#ifndef MINIMAL
	  if (show_error)
	    std::cerr << "changedEffects unset\n" << *this << moved << sq << "\n";	  
#endif
	  return false;
	}
      }
      for (int i=0; i<2; ++i) 
      {
	const Player pl = indexToPlayer(i);
	if (! changed_squares[pl].test(sq))
	{
	  if ((getEffect(sq) & getOnBoardMask(pl))
	      != (prev.getEffect(sq) & prev.getOnBoardMask(pl))) {
#ifndef MINIMAL
	    if (show_error)
	      std::cerr << "changedEffects unset for " << pl << "\n" << *this << moved << sq << "\n";
#endif
	    return false;
	  }
	}
      }
    }
  }
  // test changedPieces()
  const NumBitmapEffect changed_effect_pieces = changedPieces(); 
  for (int i=0; i<Piece::SIZE; ++i) {
    if (each_effect[i] == prev_effect[i])
      continue;
    if (! changed_effect_pieces.test(i)) {
#ifndef MINIMAL
      if (show_error)
	std::cerr << "changedPieces() unset\n" << *this << moved << i 
		  << " " << each_effect[i] << " != " <<  prev_effect[i] << "\n";
#endif
      return false;
    }
  }
  // test effectedChanged(Player pl)
  for (int i=0; i<Piece::SIZE; ++i) 
  {
    for (int j=0; j<2; ++j) 
    {
      const Player pl = indexToPlayer(j);
      if (prev.getPieceOf(i).position() == moved.to())
	continue;		// captured
      if (prev.effectedMask(pl).test(i) != effectedMask(pl).test(i)) {
	if (! effectedChanged(pl).test(i)) {
#ifndef MINIMAL
	  if (show_error)
	    std::cerr << "effectedChanged(" << pl << ") unset\n" << *this << moved << i 
		      << " " << prev.effectedChanged(pl) << " != " << prev.effectedChanged(WHITE) << "\n";
#endif
	  return false;
	}
      }
    }
  }
  return true;
}

template <bool show_error>
bool
#if (defined __GNUC__) && (! defined GPSONE) && (! defined GPSUSIONE)
__attribute__ ((used))
#endif
osl::state::NumEffectState::isAlmostValidMove(Move move) const{
  assert(move.isValid());
  assert(move.isNormal());
  assert(this->getTurn() == move.player());
  assert(isValidMoveByRule(move, true));

  const Position from=move.from();
  const Position to=move.to();
  const Piece toPiece=this->getPieceAt(to);
  const Ptype ptype=move.ptype();
  const Player turn = move.player();
    
  if (from.isPieceStand()){ // 打つ手
    // ターゲットが空白か
    if (! toPiece.isEmpty()) {
      if (show_error) std::cerr << "drop on to piece : " << move << std::endl;
      return false;
    }
    // そもそもその駒を持っているか?
    if (! this->hasPieceOnStand(turn,ptype)) {
      if (show_error) std::cerr << turn << " don't have : " << ptype << std::endl;
      return false;
    }
    // 二歩のチェック
    if (ptype==PAWN && isPawnMaskSet(turn, to.x())) {
      if (show_error) std::cerr << " Double Pawn : " << move << std::endl;
      return false;
    }
  }
  else{
    const Piece from_piece = this->getPieceAt(from);
    // fromにあるのがその駒か
    if (from_piece.isEmpty() || (from_piece.owner() != turn)){
      if (show_error) 
	std::cerr << " No such piece0 : " << move << std::endl;
      return false;
    }
    // promoteしている時にpromote可能か
    if (move.isPromote()){
      // fromにあるのがその駒か
      if (from_piece.ptype() != unpromote(move.ptype()))
	{
	  if (show_error) 
	    std::cerr << " No such piece1  : " << move << std::endl;
	  return false;
	}
      if (from_piece.isPromotedNotKingGold()){
	if (show_error) 
	  std::cerr << " can't promote promoted piece : " << move << std::endl;
	return false;
      }
    }
    else{
      // fromにあるのがその駒か
      if (from_piece.ptype() != move.ptype()){
	if (show_error) 
	  std::cerr << " No such piece2  : " << move << std::endl;
	return false;
      }
    }
    // toにあるのが，相手の駒か空白か?
    if (!toPiece.isEmpty() && toPiece.owner()==turn) {
      if (show_error) std::cerr << " No move on  : " << move << std::endl;
      return false;
    }
    if(!hasEffectByPiece(from_piece,to)){
      if (show_error) {
	std::cerr << " No such move2 : " << move << std::endl;
      }
      return false;
    }
    // capturePtypeが一致しているか?
    if (toPiece.ptype()!=move.capturePtype()) {
      if (show_error) std::cerr << " Not such capture : " << move 
				<< std::endl << *this;
      return false;
    }
  }

  assert(isValidMoveByRule(move, true));
  return true;
}

bool osl::state::NumEffectState::
isAlmostValidMove(Move move,bool show_error) const{
#ifdef MINIMAL
  show_error=false;
#endif
  if(show_error)
    return isAlmostValidMove<true>(move);
  else
    return isAlmostValidMove<false>(move);
}

#ifndef MINIMAL
void osl::state::NumEffectState::showEffect(std::ostream& os) const
{
  os<< static_cast<SimpleState const&>(*this);
  for(int y=1;y<=9;y++){
    os << 'P' << y;  
    for(int x=9;x>0;x--){
      Position pos(x,y);
      os << record::csa::show(getPieceAt(pos)) << getEffect(pos);
    }
    os << std::endl;
  }
  // 持ち駒の表示
  for(int num=0;num<Piece::SIZE;num++){
    if (standMask(BLACK).test(num)){
      os << "P+00" << record::csa::show(Piece_Table.getPtypeOf(num))
	 << std::endl;
    }
    else if (standMask(WHITE).test(num)){
      os << "P-00" << record::csa::show(Piece_Table.getPtypeOf(num))
	 << std::endl;
    }
  }
}
#endif

osl::container::PieceMask osl::state::NumEffectState::
makePinOpen(osl::Position target,osl::Player defense)
{
  PieceMask pins;
  if(target.isPieceStand()) return pins;
  PieceMask mask=getOnBoardMask(alt(defense));
  makePinOpenDir<UL>(target,pins,mask,defense);
  makePinOpenDir<U>(target,pins,mask,defense);
  makePinOpenDir<UR>(target,pins,mask,defense);
  makePinOpenDir<L>(target,pins,mask,defense);
  makePinOpenDir<R>(target,pins,mask,defense);
  makePinOpenDir<DL>(target,pins,mask,defense);
  makePinOpenDir<D>(target,pins,mask,defense);
  makePinOpenDir<DR>(target,pins,mask,defense);
  return pins;
}

void osl::state::NumEffectState::
makePinOpen(osl::Player defense)
{
  pin_or_open[defense]=makePinOpen(getKingPosition(defense),defense);
}

const osl::mask_t osl::state::NumEffectState::
effectBit(Player attack, Ptype ptype, Position target) const
{
  switch (ptype) {
  case PAWN: case PPAWN:
    return effectBit<PAWN>(attack, target);
  case LANCE: case PLANCE:
    return effectBit<LANCE>(attack, target);
  case KNIGHT: case PKNIGHT:
    return effectBit<KNIGHT>(attack, target);
  case SILVER: case PSILVER:
    return effectBit<SILVER>(attack, target);
  case GOLD:
    return effectBit<GOLD>(attack, target);
  case BISHOP: case PBISHOP:
    return effectBit<BISHOP>(attack, target);
  case ROOK: case PROOK:
    return effectBit<ROOK>(attack, target);
  case KING:
    return effectBit<KING>(attack, target);
  default:
    assert(0);
  }
  return mask_t();
}

namespace osl
{
  // explicit template instantiation

  template bool NumEffectState:: 
  hasEffectByWithRemove<BLACK>(Position, Position) const;
  template bool NumEffectState:: 
  hasEffectByWithRemove<WHITE>(Position, Position) const;
  template void NumEffectState::makeKing8Info<BLACK>();
  template void NumEffectState::makeKing8Info<WHITE>();
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
