#include "osl/record/record.h"
#include "osl/record/compactBoard.h"
#include "osl/misc/base64.h"
#include <boost/dynamic_bitset.hpp>
#include <boost/foreach.hpp>
#include <iostream>
#include <algorithm>
#include <sstream>

int osl::record::
OPiece::position2Bits(const Position& pos)
{
  return pos.isPieceStand() ? 0 : pos.x() << 4 | pos.y(); // 8 bits
}

osl::Position osl::record::
OPiece::bits2Position(const int bit_position)
{
  if ((bit_position & 0xff) == 0)
    return Position::STAND();
  else
    return Position((bit_position >> 4) & 0xf, bit_position & 0xf);
}

namespace osl
{
  namespace record
  {

    struct opiece_sort
    { 
      bool operator()(const OPiece& l, const OPiece& r)
      {
	// need to special case pieces on stand
	if (l.getPosition() == Position::STAND() || r.getPosition() == Position::STAND())
	  {
	    if (l.getPosition() == Position::STAND() && r.getPosition() != Position::STAND())
	      return true;
	    else if (l.getPosition() != Position::STAND() && r.getPosition() == Position::STAND())
	      return false;
	    else
	      {
		if (l.getOwner() != r.getOwner())
		  return l.getOwner() == WHITE;
		return l.getPtype() < r.getPtype();
	      }
	  }
	else
	  {
	    if (l.getPosition().x() < r.getPosition().x())
	      return true;
	    else if (l.getPosition().x() > r.getPosition().x())
	      return false;
	    else
	      {
		if (l.getPosition().y() <= r.getPosition().y())
		  return true;
		else
		  return false;
	      }
	  }
      }
    };
  }
}

osl::record::
CompactBoard::CompactBoard(const SimpleState& state)
{
  pieces.reserve(40);
  for (int i = 0; i < 40; i++)
  {
    pieces.push_back(OPiece(state.getPieceOf(i)));
  }
  std::sort(pieces.begin(), pieces.end(), opiece_sort());
  turn = state.getTurn();
}

osl::SimpleState osl::record::
CompactBoard::getState() const
{
  assert(pieces.size() == 40);

  SimpleState state;
  state.init();

  BOOST_FOREACH(const OPiece& p, pieces) {
    state.setPiece(p.getOwner(), p.getPosition(), p.getPtype());
  }
  state.setTurn(turn);
  state.initPawnMask();
  return state;
}

bool osl::record::
operator==(const CompactBoard& lhs, const CompactBoard& rhs)
{
  return (lhs.turn == rhs.turn) && (lhs.pieces == rhs.pieces);
}

std::ostream& osl::record::
operator<<(std::ostream& os, const CompactBoard& c)
{
  assert(c.pieces.size() == 40);

  for (unsigned int i = 0; i < c.pieces.size(); i++)
  {
    writeInt(os, static_cast<int>(c.pieces[i]));
  }
  writeInt(os, static_cast<int>(c.turn));
  return os;
}

std::istream& osl::record::
operator>>(std::istream& is, CompactBoard& c)
{
  assert(c.pieces.size() == 0);

  for (unsigned int i = 0; i < 40; i++)
  {
    c.pieces.push_back(OPiece(readInt(is)));
  }
  c.turn = static_cast<Player>(readInt(is));
  return is;
}
#ifndef MINIMAL
std::string osl::record::
CompactBoard::toBase64() const
{
  const static size_t ninteger = 41;
  const static size_t integer_size = 32;
  const static size_t size = ninteger*integer_size;

  std::stringstream ss;
  ss << *this;

  ss.clear();
  ss.seekg(0, std::ios::beg);

  boost::dynamic_bitset<> bits(size);

  for (size_t i = 0; i < ninteger; ++i)
  {
    const unsigned int tmp = static_cast<unsigned int>(readInt(ss));
    const boost::dynamic_bitset<> mask(size, static_cast<unsigned long>(tmp));
    bits = (bits << integer_size) | mask;
  }

  return misc::base64Encode(bits);
}

const osl::record::CompactBoard osl::record::
CompactBoard::fromBase64(const std::string& str)
{
  const boost::dynamic_bitset<> bits = misc::base64Decode(str);
  std::stringstream ss;
  assert(bits.size()%32 == 0);
  const boost::dynamic_bitset<> mask(bits.size(), 4294967295ul);
  for (size_t i=0; i<bits.size()/32; ++i)
  {
    const unsigned long tmp = ((bits >> ((bits.size()/32-1-i)*32)) & mask).to_ulong();
    writeInt(ss, static_cast<int>(tmp));
  }
      
  ss.clear();
  ss.seekg(0, std::ios::beg);

  CompactBoard cb;
  ss >> cb;
  return cb;
}
#endif


/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
