/* fixedCapacityVector.h
 */
#ifndef OSL_FIXED_CAPACITY_VECTOR_H
#define OSL_FIXED_CAPACITY_VECTOR_H

#include "osl/misc/carray.h"
#include "osl/misc/cstdint.h"
#include "osl/misc/construct.h"
#include <algorithm>
#include <cstddef>
#include <cassert>

#if (__GNUC__ >= 4 && __GNUC_MINOR__ >=4)
#  pragma GCC diagnostic ignored "-Wstrict-aliasing"
#endif

namespace osl
{
  namespace misc
  {
  template <typename T>
  class FixedCapacityVectorPushBack
  {
    T *ptr;
    T **vPtr;
#if ! (defined NDEBUG && defined MINIMAL)
    T *limit;
#endif
  public:
    FixedCapacityVectorPushBack(T** vPtr_, T* limit_)
      : ptr(*vPtr_), vPtr(vPtr_)
#if ! (defined NDEBUG && defined MINIMAL)
      ,limit(limit_)
#endif
    {
    }
    ~FixedCapacityVectorPushBack()
    {
      assert( *vPtr == ptr );
      *vPtr = ptr;
    }
    void push_back(const T& e) 
    {
      assert(ptr < limit);
      assert( *vPtr == ptr );
      if(misc::detail::BitCopyTraits<T>::value)
	*ptr++ = e;
      else
	construct(ptr++,e);
#ifndef NDEBUG
      (*vPtr)++;
#endif
    }
  };
  template <typename T, size_t Capacity>
  class FixedCapacityVector
  {
  protected:
    struct Array : public CArray<T, Capacity> {}
#ifdef __GNUC__
    __attribute__((__may_alias__))
#endif
      ;
    typedef Array array_t;
    T* ptr;
    CArray<int64_t, (sizeof(T[Capacity])+sizeof(int64_t)-1)/sizeof(int64_t)> relements;
  private:
    const array_t &elements() const{
      return *reinterpret_cast<const array_t*>(&relements);
    }
    array_t &elements(){
      return *reinterpret_cast<array_t*>(&relements);
    }
  public:
    typedef typename array_t::value_type value_type;
    typedef typename array_t::iterator iterator;
    typedef typename array_t::const_iterator const_iterator;
    
    FixedCapacityVector() : ptr(&(elements()[0])) {}
    ~FixedCapacityVector() 
    {
      destroy(begin(),end());
    }
    FixedCapacityVector(FixedCapacityVector const& rhs){
      ptr= &*begin()+rhs.size();
      std::uninitialized_copy(rhs.begin(),rhs.end(),begin());
    }
    FixedCapacityVector& operator=(FixedCapacityVector const& rhs){
      if (this == &rhs)
	return *this;
      
      if(size()>rhs.size()){
	iterator it=std::copy(rhs.begin(),rhs.end(),begin());
	destroy(it,end());
      }
      else{
	iterator it=std::copy(&(rhs.elements()[0]),
			      &(rhs.elements()[0])+size(),begin());
	std::uninitialized_copy(&(rhs.elements()[0])+size(),
				&(rhs.elements()[0])+rhs.size(),it);
      }
      ptr= &*begin()+rhs.size();
      return *this;
    }

    T& operator[] (size_t i)
    {
      assert(i <= size());
      return elements()[i];
    }

    iterator begin() {  return &elements()[0]; }
    iterator end() { return static_cast<iterator>(ptr); }

    T& front() { return *begin(); }
    T& back() { return *(end() - 1); }

    void push_back(const T& e)
    {
      assert(size() < Capacity);
      construct(ptr,e);
      ++ptr;
    }
    template <class RangeIterator>
    void push_back(const RangeIterator& first, const RangeIterator& last);
    void pop_back() { 
      --ptr;
      destroy(ptr+1);
    }
    void clear() { 
      size_t s=size();
      ptr= &(elements()[0]);
      // 該当する部分のdestructorを呼ぶ
      destroy(begin(),begin()+(int)s);
    }
    void resize(size_t new_length)
    {
      while (size() < new_length)
	push_back(T());
      if (new_length < size()) {
	destroy(begin()+(int)new_length,end());
	ptr= &(elements()[new_length]);
      }
    }

    void erase(const T& e)
    {
      const iterator new_end = std::remove(begin(), end(), e);
      ptr= &*new_end;
      destroy(new_end,end());
    }

    /** 重複する要素を取り除く */
    void unique()
    {
      std::sort(begin(),end());
      iterator last = std::unique(begin(), end());
      ptr = &*last;
      destroy(last,end());
    }

    size_t size() const { return ptr-&*begin(); }
    bool empty() const { return ptr==&*begin(); }
    size_t capacity() const { return Capacity; }

    T const& operator[] (size_t i) const
    {
      assert(i < size());
      return elements()[i];
    }
    const_iterator begin() const { return &elements()[0]; }
    const_iterator end()   const { return ptr; }

    const T& front() const { return *begin(); }
    const T& back() const { return *(end() - 1); }
    
    bool isMember(const T& e, const_iterator first, const_iterator last) const
    {
      return std::find(first, last, e) != last;
    }
    bool isMember(const T& e) const
    {
      return isMember(e, begin(), end());
    }
    FixedCapacityVectorPushBack<T> pushBackHelper() 
    {
      return FixedCapacityVectorPushBack<T>
	(&ptr, &*begin()+Capacity);
    }
  };
    template <typename T, size_t C> inline
    bool operator==(const FixedCapacityVector<T,C>& l, const FixedCapacityVector<T,C>& r) 
    {
      return l.size() == r.size() && std::equal(l.begin(), l.end(), r.begin());
    }
    template <typename T, size_t C> inline
    bool operator<(const FixedCapacityVector<T,C>& l, const FixedCapacityVector<T,C>& r) 
    {
      return std::lexicographical_compare(l.begin(), l.end(), r.begin(), r.end());
    }
  } // namespace misc
  using misc::FixedCapacityVector;
  using misc::FixedCapacityVectorPushBack;
} // namespace osl

template <typename T, size_t Capacity>
template <class RangeIterator>
void osl::misc::FixedCapacityVector<T,Capacity>::push_back(const RangeIterator& first, const RangeIterator& last)
{
  iterator insert_point = end();
  std::uninitialized_copy(first, last, insert_point);
  ptr += last-first;
  assert(size() <= Capacity);
}


#endif /* OSL_FIXED_CAPACITY_VECTOR_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
