/* boardKey.h
 */
#ifndef _BOARD_KEY_H
#define _BOARD_KEY_H
#include "osl/misc/carray.h"
#include "osl/pieceStand.h"
#include <cstddef>

namespace osl
{
namespace hash
{
  /**
   * 手番を含んだ盤面の状態のハッシュ値を保持するためのクラス.
   *
   * - 64bit プロセッサでは unsigned long long, 2 (衝突がなければ1も)
   * - 32bit プロセッサでは unsigned int, 4 
   * を想定.
   *
   * elements[0] の最下位bit をplayer bit とする．
   */
  template<typename Integer,size_t SIZE>
  class GeneralBoardKey
  {
    CArray<Integer,SIZE> elements;
  public:
    GeneralBoardKey();
    GeneralBoardKey(const GeneralBoardKey& src)
    {
      for (size_t i=0; i<SIZE; ++i)
	elements[i] = src[i];	// better than memmove
    }
    typedef Integer int_t;
    size_t size() const{ return SIZE; }
    int_t operator[](size_t i) const{
      return elements[i];
    }
    int_t& operator[](size_t i){
      return elements[i];
    }
    unsigned int signature() const { return elements[1]; }
    /**
     * メモリ破壊を検出するためにrのplayer bitが0であることを確認
     */
    GeneralBoardKey& operator+=(const GeneralBoardKey& r)
    {
      assert(r.playerBit() == 0);
      for(size_t i=0;i<SIZE;i++)
	elements[i]+=r.elements[i];
      return *this;
    }
    /**
     * メモリ破壊を検出するためにrのplayer bitが0であることを確認
     */
    GeneralBoardKey& operator-=(const GeneralBoardKey& r)
    {
      assert(r.playerBit() == 0);
      for(size_t i=0;i<SIZE;i++)
	elements[i]-=r.elements[i];
      return *this;
    }
    void changeTurn()
    {
      elements[0]^=static_cast<int_t>(1);
    }
    /**
     * 元々 whiteの時もblackにsetできるようにする
     */
    void setPlayer(Player p) 
    { 
      elements[0]=(elements[0]& ~static_cast<int_t>(1))|playerToIndex(p);
    }
    bool playerBit() const
    {
      return (static_cast<int>(elements[0]) & 1);
    }
    bool isPlayerOfTurn(Player p) const
    {
      return playerBit() == playerToIndex(p);
    }
    Player turn() const { return isPlayerOfTurn(BLACK) ? BLACK : WHITE; }
    /**
     * elements[0]の最下位ビットは0にする
     */
    void setRandom();
  };

  template<typename Integer,size_t SIZE>
  inline bool operator==(const GeneralBoardKey<Integer,SIZE>& l,
			 const GeneralBoardKey<Integer,SIZE>& r)
  {
    for(size_t i=0;i<SIZE;i++)
      if (l[i]!=r[i]) return false;
    return true;
  }
  template<typename Integer,size_t SIZE>
  inline bool operator!=(const GeneralBoardKey<Integer,SIZE>& l,
			 const GeneralBoardKey<Integer,SIZE>& r)
  {
    return !(l==r);
  }
  /**
   * set等で使うためのみの不等号.  
   * full orderであること以外に深い意味はない
   */
  template<typename Integer,size_t SIZE>
  inline bool operator<(const GeneralBoardKey<Integer,SIZE>& l,
			 const GeneralBoardKey<Integer,SIZE>& r)
  {
    for(size_t i=0;i<SIZE-1;i++)
      if (l[i]!=r[i]) return l[i]<r[i];
    return l[SIZE-1]<r[SIZE-1];
  }

  /**
   * BoardKey + 駒台(piece stand)
   */
  template<typename BoardKeyBase>
  struct GeneralHashKey
  {
    typedef BoardKeyBase base_t;
    BoardKeyBase board_key;
    PieceStand piece_stand;

    typedef typename BoardKeyBase::int_t int_t;

    GeneralHashKey()
    {
    }
    const base_t& boardKey() const{
      return board_key;
    }
    unsigned int signature() const
    {
      return board_key.signature();
    }
    const PieceStand& pieceStand() const{
      return piece_stand;
    }
    void setPieceStand(const PieceStand& p){
      piece_stand=p;
    }
    size_t size() const{ 
      return board_key.size(); 
    }
    int_t operator[](size_t i) const{
      return board_key[i];
    }
    int_t& operator[](size_t i){
      return board_key[i];
    }
    /**
     * 駒台の情報を除いて同じかどうか.
     * 手番が異なるものは異なると定義する
     */
    bool isSameBoard(const GeneralHashKey& key) const
    {
      return board_key == key.boardKey();
    }
    GeneralHashKey& operator+=(const GeneralHashKey& r)
    {
      board_key+=r.board_key;
      piece_stand.addAtmostOnePiece(r.piece_stand);
      return *this;
    }
    GeneralHashKey& operator-=(const GeneralHashKey& r)
    {
      board_key-=r.board_key;
      piece_stand.subAtmostOnePiece(r.piece_stand);
      return *this;
    }
    const PieceStand blackStand() const 
    { 
      return piece_stand;
    }
    void changeTurn()
    {
      board_key.changeTurn();
    }
    void setPlayer(Player p) 
    { 
      board_key.setPlayer(p);
    }
    bool isPlayerOfTurn(Player p) const
    {
      return board_key.isPlayerOfTurn(p);
    }
    Player turn() const { return isPlayerOfTurn(BLACK) ? BLACK : WHITE; }
    /**
     * pieceStandには触らない
     */
    void setRandom();
    
  };
  template<typename T>
  inline bool operator==(const GeneralHashKey<T>& l,
			 const GeneralHashKey<T>& r)
  {
    /**
     * board_keyが違う場合の方が多いだろう
     */
    return l.board_key == r.board_key &&
      l.piece_stand == r.piece_stand;
  }
  template<typename T>
  inline bool operator!=(const GeneralHashKey<T>& l,
			 const GeneralHashKey<T>& r)
  {
    return !(l==r);
  }
  /**
   * set等で使うためのみの不等号
   * full orderであること以外に深い意味はない
   */
  template<typename T>
  inline bool operator<(const GeneralHashKey<T>& l,
			const GeneralHashKey<T>& r)
  {
    if (l.pieceStand < r.pieceStand) return true;
    else if (r.pieceStand < l.pieceStand) return false;
    return l.board_key < r.board_key;
  }

  typedef GeneralBoardKey<unsigned int,4> BoardKey32;
  typedef GeneralBoardKey<unsigned long long,2> BoardKey64;
  typedef GeneralHashKey<BoardKey32> HashKey32;
  typedef GeneralHashKey<BoardKey64> HashKey64;
} // namespace hash
} // namespace osl

#endif /* _BOARD_KEY_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:

