MODULE VO:OS:Display;

(*
    Implementation Classes for visualisation.
    Copyright (C) 2000  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT s    := SYSTEM,

       D    := VO:Base:Display,
       DD   := VO:Base:DragDrop,
       E    := VO:Base:Event,
       O    := VO:Base:Object,
       U    := VO:Base:Util,

       EO   := VO:OS:Event,
       LI   := VO:OS:Interface,
       LM   := VO:OS:Procedure,

       P    := VO:Prefs:Base,

       a    := Xatom,
               C,
       c    := Calendar,
       ch   := Channel,
       co   := IntStr,
               Err,
               Out,
       p    := PosixFileDescr,
       sc   := SysClock,
       str  := Strings,
       str2 := Strings2,
       t    := Time,

<* IF LIB_HAVE_LIBIMLIB=TRUE THEN *>

               Imlib,

<* END *>

               X11,
       xu   := Xutil;


(* ============== Implementation classes ============== *)

CONST
  contextTimeOutSec = 3; (* seconds until context help will displayed *)
  contextTimeOutMil = 0; (* microseconds until context help will displayed *)

  (* Size of the various stack arrays *)
  initialStackSize   = 30;
  stackSizeIncrement = 30;

  (* constants for Xdnd *)

  XDND_VERSION = 3;

  sleepCursor = 150; (* from /usr/include/X11/cursorFont.h *)
  popCursor   =  94;
  dndCursor   =  50;

TYPE

  Sleep           = POINTER TO SleepDesc;
  (**
    This is the class, sleep information are stored in. VODisplay
    bilds a sorted list of Sleeps to store all current sleep callbacks in it.
  *)

  SleepDesc       = RECORD (D.SleepDesc)
                      next   : Sleep;
                      object : O.MsgObject;
                    END;

  TimeOut         = POINTER TO TimeOutDesc;

  (**
    This is the class, timeout information are stored in. VODisplay
    bilds a sorted list of TimeOuts to store all current timeouts in it.
  *)

  TimeOutDesc = RECORD (D.TimeOutDesc)
                  next   : TimeOut;
                  time   : t.TimeStamp;
                  object : O.MsgObject;
                END;


  Channel         = POINTER TO ChannelDesc;

  (**
    This structure holds information about channels the main event loop
    should wait for. The display object notifies the given object if the
    handed channel becomes available (currently this means, there is some data
    in the channel that can be read, the event loop engine may support
    event sneding on channel getting available for writing in the future, too).
  *)

  ChannelDesc     = RECORD (D.ChannelDesc)
                      next    : Channel;
                      channel : ch.Channel;
                      object  : O.MsgObject;
                    END;

  Font*     = POINTER TO FontDesc;
  FontDesc* = RECORD (D.FontDesc)
                fullName     : U.Text;
                loaded       : BOOLEAN;
                fonts        : ARRAY D.maxStyleNum OF X11.XFontStructPtr;
                useCount     : LONGINT;
              END;

  PointArray = POINTER TO ARRAY OF X11.XPoint;

  Display*     = POINTER TO DisplayDesc;
  DisplayDesc* = RECORD (D.DisplayDesc)
                    scrNum-      : LONGINT;

                    display*     : X11.DisplayPtr;
                    visual       : X11.VisualPtr;
                    xim-         : X11.XIM;
                    xic-         : X11.XIC;

<* IF LIB_HAVE_LIBIMLIB=TRUE THEN *>
                    imlib*       : Imlib.DataPtr;
<* END *>

                    (* various cursors *)
                    sleepCursor,
                    popCursor,
                    dndCursor,
                    copyCursor,
                    moveCursor,
                    linkCursor   : X11.Cursor;

                    colorMap     : X11.Colormap;

                    (* fonts *)
                    font         : ARRAY D.fontCount OF Font;
                    fonts        : Font;

                    deleteProt   : ARRAY 1 OF X11.Atom;

                    (* windowing information *)
                    winList      : D.Window;

                    (* selection stuff *)
                    selectObject,
                    querySelectObject : D.Object;

                    (* timeouts *)
                    timeOutList  : TimeOut;
                    timeOut      : TimeOut; (* The current timeout for context help *)
                    contextHelp  : BOOLEAN;

                    (* sleep stuff *)
                    sleepList    : Sleep;

                    (* fd stuff *)

                    channelList  : Channel;

                    (* X property types *)
                    atom         : X11.Atom;

                    (* Xdnd stuff *)
                    XdndAware,
                    XdndEnter,
                    XdndLeave,
                    XdndPosition,
                    XdndStatus,
                    XdndFinished,
                    XdndDrop,
                    XdndActionCopy,
                    XdndActionMove,
                    XdndActionLink,
                    XdndActionAsk,
                    XdndActionPrivate,
                    XdndActionList,
                    XdndSelection,
                    XdndTypeList : X11.Atom;

                    xSelection,
                    xDrop,
                    dSelection,
                    wmState      : X11.Atom;
                    dropWindow   : X11.Window;

                    (* general DnD *)
                    dragObject   : D.Object;
                    dragInfo     : DD.DnDDataInfo;
                    dndAction    : LONGINT;
                    dragX,dragY  : LONGINT; (* x,y coord of the drag start point *)
                    dragging,               (* we are currently dragging (not handled yet) *)
                    dragStart    : BOOLEAN; (* Button down, check for dragging on mouse move *)

                    exit         : BOOLEAN;

                    lastEventIsButtonPress : BOOLEAN;
                    lastButton             : SHORTINT;
                    lastBut1Button         : SHORTINT;
                    lastPressTime          : X11.Time;
                    lastBut1PressTime      : X11.Time;
                    multiClickTime         : X11.Time;
                    pointArray             : PointArray;
                 END;

  Bitmap*     = POINTER TO BitmapDesc;
  BitmapDesc* = RECORD (D.BitmapDesc)
                    pixmap- : X11.Pixmap;
                  END;

  DrawInfo* = POINTER TO DrawInfoDesc;

  ClipEntry     = POINTER TO ClipEntryDesc;
  ClipEntryDesc = RECORD
                    region  : xu.Region;
                    draw    : DrawInfo;
                  END;


  PenColor      = POINTER TO PenColorDesc;
  PenColorDesc  = RECORD
                    color : D.Color;
                    count : LONGINT;
                  END;

  DrawMode      = POINTER TO DrawModeDesc;
  DrawModeDesc  = RECORD
                    next : DrawMode;
                    mode : LONGINT;
                  END;

  PenStyle      = POINTER TO PenStyleDesc;
  PenStyleDesc  = RECORD
                    next  : PenStyle;
                    size,
                    mode,
                    cap,
                    join  : LONGINT;
                  END;

  PenDash       = POINTER TO PenDashDesc;
  PenDashDesc   = RECORD
                    next : PenDash;
                    list : POINTER TO ARRAY OF CHAR;
                    mode : LONGINT;
                  END;

  Pattern       = POINTER TO PatternDesc;
  PatternDesc   = RECORD
                    next   : Pattern;
                    pixMap : X11.Pixmap;
                    mode   : LONGINT;
                  END;

  FontEntry     = POINTER TO FontEntryDesc;
  FontEntryDesc = RECORD
                    font  : Font;
                    style : SET;
                    count : LONGINT;
                  END;

  ClipStack = POINTER TO ARRAY OF ClipEntryDesc;
  PenStack  = POINTER TO ARRAY OF PenColorDesc;
  FontStack = POINTER TO ARRAY OF FontEntryDesc;

  DrawInfoDesc* = RECORD (D.DrawInfoDesc)
                        clipStack    : ClipStack;
                        clipPos      : LONGINT;
                        
                        fPenStack    : PenStack;
                        fPenPos      : LONGINT;
                        
                        bPenStack    : PenStack;
                        bPenPos      : LONGINT;

                        fontStack    : FontStack;
                        fontPos      : LONGINT;
                        
                        styleStack   : PenStyle;
                        dashStack    : PenDash;
                        patternStack : Pattern;
                        modeStack    : DrawMode;
                        window-      : X11.Window;
                        gc-          : X11.GC;
                      END;

  Window*     = POINTER TO WindowDesc;
  WindowDesc* = RECORD (D.WindowDesc)
                  last,next      : D.Window;
                  window         : X11.Window;

                  draw           : DrawInfo;

                  modalCount     : LONGINT;

                  exposed,
                  grab,
                  exit           : BOOLEAN;
                END;

  FactoryImpl*      = POINTER TO FactoryImplDesc;
  FactoryImplDesc*  = RECORD (D.FactoryDesc)
                      END;

  DisplayPrefs     = POINTER TO DisplayPrefsDesc;
  DisplayPrefsDesc = RECORD (D.DisplayPrefsDesc)
                     END;

VAR
  factory : FactoryImpl;

  (* copy cursor *)

  copyCursorData,
  copyMaskData   : ARRAY 30 OF CHAR;

  moveCursorData,
  moveMaskData   : ARRAY 26 OF CHAR;

  linkCursorData,
  linkMaskData   : ARRAY 34 OF CHAR;

  tOutMsg        : D.TimeOutMsg;

  PROCEDURE (p : DisplayPrefs) EvaluateColor;

  BEGIN
    IF D.display.colorMode=D.monochromeMode THEN
      p.colors[D.backgroundColorIndex]:="white";
      p.colors[D.tableBackgroundColorIndex]:=p.colors[D.backgroundColorIndex];
      p.colors[D.textBackgroundColorIndex]:=p.colors[D.backgroundColorIndex];
      p.colors[D.buttonBackgroundColorIndex]:=p.colors[D.backgroundColorIndex];
      p.colors[D.tableTextColorIndex]:="black";
      p.colors[D.halfShineColorIndex]:="black";
      p.colors[D.halfShadowColorIndex]:="black";
      p.colors[D.textColorIndex]:="black";
      p.colors[D.shineColorIndex]:="black";
      p.colors[D.shadowColorIndex]:="black";
      p.colors[D.fillColorIndex]:="black";
      p.colors[D.fillTextColorIndex]:="white";
      p.colors[D.warnColorIndex]:="black";
      p.colors[D.disabledColorIndex]:="black";
      p.colors[D.focusColorIndex]:="black";
      p.colors[D.blackColorIndex]:="black";
      p.colors[D.whiteColorIndex]:="white";
      p.colors[D.helpBackgroundColorIndex]:=p.colors[D.backgroundColorIndex];
    ELSIF D.display.colorMode=D.greyScaleMode THEN
      p.colors[D.backgroundColorIndex]:="grey70";
      p.colors[D.tableBackgroundColorIndex]:="white";
      p.colors[D.textBackgroundColorIndex]:=p.colors[D.backgroundColorIndex];
      p.colors[D.buttonBackgroundColorIndex]:=p.colors[D.backgroundColorIndex];
      p.colors[D.tableTextColorIndex]:="black";
      p.colors[D.textColorIndex]:="black";
      p.colors[D.shineColorIndex]:="grey95";
      p.colors[D.halfShineColorIndex]:="grey82";
      p.colors[D.halfShadowColorIndex]:="grey45";
      p.colors[D.shadowColorIndex]:="grey20";
      p.colors[D.fillColorIndex]:="grey60";
      p.colors[D.fillTextColorIndex]:="black";
      p.colors[D.warnColorIndex]:="grey92";
      p.colors[D.disabledColorIndex]:="grey20";
      p.colors[D.focusColorIndex]:="grey30";
      p.colors[D.blackColorIndex]:="black";
      p.colors[D.whiteColorIndex]:="white";
      p.colors[D.helpBackgroundColorIndex]:=p.colors[D.backgroundColorIndex];
    ELSE
      p.colors[D.backgroundColorIndex]:="grey70";
      p.colors[D.tableBackgroundColorIndex]:="white";
      p.colors[D.tableTextColorIndex]:="black";
      p.colors[D.textBackgroundColorIndex]:=p.colors[D.backgroundColorIndex];
      p.colors[D.buttonBackgroundColorIndex]:=p.colors[D.textBackgroundColorIndex];
      p.colors[D.textColorIndex]:="black";
      p.colors[D.shineColorIndex]:="grey95";
      p.colors[D.halfShineColorIndex]:="grey82";
      p.colors[D.halfShadowColorIndex]:="grey45";
      p.colors[D.shadowColorIndex]:="grey20";
      p.colors[D.fillColorIndex]:="royal blue";
      p.colors[D.fillTextColorIndex]:="white";
      p.colors[D.warnColorIndex]:="red";
      p.colors[D.disabledColorIndex]:="grey20";
      p.colors[D.focusColorIndex]:="grey30";
      p.colors[D.blackColorIndex]:="black";
      p.colors[D.whiteColorIndex]:="white";
      p.colors[D.helpBackgroundColorIndex]:="light yellow";
    END;
  END EvaluateColor;

  PROCEDURE (p : DisplayPrefs) Init*;
  (**
    Initializes an instance.
  *)

  BEGIN
    p.Init^;

    p.EvaluateColor;

    p.fonts[D.tinyFontIndex]      :="-adobe-helvetica-medium-r-*--8-*-*-*-*-*-iso8859-1";
    p.fonts[D.scriptFontIndex]    :="-adobe-helvetica-medium-r-*--8-*-*-*-*-*-iso8859-1";
    p.fonts[D.footnoteFontIndex]  :="-adobe-helvetica-medium-r-*--10-*-*-*-*-*-iso8859-1";
    p.fonts[D.smallFontIndex]     :="-adobe-helvetica-medium-r-*--10-*-*-*-*-*-iso8859-1";
    p.fonts[D.normalFontIndex]    :="-adobe-helvetica-medium-r-*--12-*-*-*-*-*-iso8859-1";
    p.fonts[D.largeFontIndex]     :="-adobe-helvetica-medium-r-*--14-*-*-*-*-*-iso8859-1";
    p.fonts[D.LargeFontIndex]     :="-adobe-helvetica-medium-r-*--18-*-*-*-*-*-iso8859-1";
    p.fonts[D.LARGEFontIndex]     :="-adobe-helvetica-medium-r-*--18-*-*-*-*-*-iso8859-1";
    p.fonts[D.hugeFontIndex]      :="-adobe-helvetica-medium-r-*--24-*-*-*-*-*-iso8859-1";
    p.fonts[D.HUGEFontIndex]      :="-adobe-helvetica-medium-r-*--25-*-*-*-*-*-iso8859-1";

    p.fonts[D.smallFixedFontIndex]:="-adobe-courier-medium-r-*--9-*-*-*-*-*-iso8859-1";
    p.fonts[D.fixedFontIndex]     :="-adobe-courier-medium-r-*--10-*-*-*-*-*-iso8859-1";
    p.fonts[D.hugeFixedFontIndex] :="-adobe-courier-medium-r-*--12-*-*-*-*-*-iso8859-1";
  END Init;

(* ============== Implementation classes ============== *)

  (**
     Frees memory allocated using X11 if the pointer is not @code{NIL}.
  *)

  PROCEDURE XFree(pointer : s.PTR);

  BEGIN
    IF pointer#NIL THEN
      X11.XFree(s.VAL(LONGINT,pointer));
    END;
  END XFree;

  (* -------- font stuff ------------- *)

  (**
    Calculates the position in the fontstyle array for the given style.
  *)

  PROCEDURE (font : Font) StyleToPos(style : SET):LONGINT;

  VAR
    x,y,val,res  : LONGINT;

  BEGIN
    res:=0;
    style:=style*D.styleMask;
    FOR x:=0 TO SIZE(SET)-1 DO
      IF x IN style THEN
        val:=1;
        FOR y:=1 TO x DO
          val:=val*2;
        END;
        INC(res,val);
      END;
    END;

    RETURN res;
  END StyleToPos;

  (**
    Returns the offset in the font array, that corresponds to the given style.
    uses Font.StyleToPos.

    NOTE
    GetFontPos tries to find a "close" font if a font with the given style bits
    does not exist. Currently it sees italic and slanted as equal, so if no italic
    font does exist, it tries a slanted one and vice versa.
  *)

  PROCEDURE (font : Font) GetFontPos(style : SET):LONGINT;

  VAR
    x : LONGINT;

  BEGIN
    x:=font.StyleToPos(style);
    IF font.fonts[x]#NIL THEN
      RETURN x;
    END;

    IF D.slanted IN style THEN
      EXCL(style,D.slanted);
      INCL(style,D.italic);
    ELSIF D.italic IN style THEN
      EXCL(style,D.italic);
      INCL(style,D.slanted);
    END;

    x:=font.StyleToPos(style);
    IF font.fonts[x]#NIL THEN
      RETURN x;
    END;

    EXCL(style,D.slanted);
    EXCL(style,D.italic);

    RETURN 0;
  END GetFontPos;

  (**
    Returns a pointer to the OS-specific font handler.
  *)

  PROCEDURE (font : Font) GetFontHandle(style : SET):X11.XFontStructPtr;

  BEGIN
    RETURN font.fonts[font.GetFontPos(style)];
  END GetFontHandle;

  PROCEDURE (font : Font) TextWidth*(text : ARRAY OF CHAR; length : LONGINT; style : SET):LONGINT;

  BEGIN
    RETURN X11.XTextWidth(font.fonts[font.GetFontPos(style)],text,length);
  END TextWidth;

  PROCEDURE (font : Font) LongTextWidth*(text : ARRAY OF LONGCHAR; length : LONGINT; style : SET):LONGINT;

  VAR x : LONGINT;

  BEGIN
    FOR x:=0 TO LEN(text)-1 DO
      IF ORD(text[x])#0 THEN
        text[x]:=CHR(((ORD(text[x]) MOD 256)*256) + (ORD(text[x]) DIV 256));
      END;
    END;

    RETURN X11.XTextWidth16(font.fonts[font.GetFontPos(style)],text,length);
  END LongTextWidth;

  PROCEDURE (font : Font) TextExtent*(text : ARRAY OF CHAR;
                                      length : LONGINT; style : SET;
                                      VAR extent : D.FontExtentDesc);

  VAR
    a,b,c : LONGINT;
    char  : X11.XCharStruct;

  BEGIN
    X11.XTextExtents(font.fonts[font.GetFontPos(style)],text,length,a,b,c,char);
    extent.lbearing:=char.lbearing;
    extent.rbearing:=char.rbearing;
    extent.width:=char.width;
    extent.ascent:=char.ascent;
    extent.descent:=char.descent;
    extent.height:=char.ascent+char.descent;
  END TextExtent;

  PROCEDURE (font : Font) LongTextExtent*(text : ARRAY OF LONGCHAR;
                                          length : LONGINT; style : SET;
                                          VAR extent : D.FontExtentDesc);

  VAR
    a,b,c : LONGINT;
    char  : X11.XCharStruct;
    x     : LONGINT;

  BEGIN
    FOR x:=0 TO LEN(text)-1 DO
      IF ORD(text[x])#0 THEN
        text[x]:=CHR(((ORD(text[x]) MOD 256)*256) + (ORD(text[x]) DIV 256));
      END;
    END;

    X11.XTextExtents16(font.fonts[font.GetFontPos(style)],text,length,a,b,c,char);
    extent.lbearing:=char.lbearing;
    extent.rbearing:=char.rbearing;
    extent.width:=char.width;
    extent.ascent:=char.ascent;
    extent.descent:=char.descent;
    extent.height:=char.ascent+char.descent;
  END LongTextExtent;

  PROCEDURE (font : Font) Init;

  (**
    Initialize a font object.
  *)

  VAR
    x : LONGINT;

  BEGIN
    font.last:=NIL;
    font.next:=NIL;

    font.loaded:=FALSE;
    font.features:={};
    font.useCount:=0;

    FOR x:=0 TO LEN(font.fonts)-1 DO
      font.fonts[x]:=NIL;
    END;
  END Init;

  PROCEDURE (font : Font) InitFromFontInfo(name : ARRAY OF CHAR; info : X11.XFontStructPtr);

  VAR
    buffer,
    buffer2 : ARRAY 256 OF CHAR;
    res     : co.ConvResults;

    PROCEDURE GetField(VAR source : ARRAY OF CHAR; pos : INTEGER; VAR string : ARRAY OF CHAR);

    VAR
      x1,x2 : INTEGER;

    BEGIN
      x1:=0;
      WHILE (source[x1]#0X) & (pos>0) DO
        IF source[x1]="-" THEN
          DEC(pos);
        END;
        INC(x1);
      END;

      x2:=x1;
      WHILE (source[x2+1]#0X) & (source[x2+1]#"-") DO
        INC(x2);
      END;

      str.Extract(source,x1,x2-x1+1,string);
    END GetField;

  BEGIN
    GetField(name,1,buffer);
    GetField(name,2,buffer2);
    NEW(font.name,str.Length(buffer)+1+str.Length(buffer2)+1);
    COPY(buffer,font.name^);
    str.Append("-",font.name^);
    str.Append(buffer2,font.name^);
    INCL(font.features,D.fontName);

    GetField(name,5,buffer);
    IF buffer="normal" THEN
      font.setWidth:=D.fontNormal;
      INCL(font.features,D.fontSetWidth);
    ELSIF buffer="semicondensed" THEN
      font.setWidth:=D.fontCondensed;
      INCL(font.features,D.fontSetWidth);
    ELSIF buffer="narrow" THEN
      font.setWidth:=D.fontNarrow;
      INCL(font.features,D.fontSetWidth);
    ELSIF buffer="double" THEN
      font.setWidth:=D.fontDouble;
      INCL(font.features,D.fontSetWidth);
    END;

    GetField(name,7,buffer);
    co.StrToInt(buffer,font.height,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,D.fontHeight);

    GetField(name,8,buffer);
    co.StrToInt(buffer,font.pointHeight,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,D.fontPointHeight);

    GetField(name,9,buffer);
    co.StrToInt(buffer,font.horizRes,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,D.fontHorizRes);

    GetField(name,10,buffer);
    co.StrToInt(buffer,font.vertRes,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,D.fontVertRes);

    GetField(name,11,buffer);
    IF buffer="p" THEN
      font.spacing:=D.fontProportional;
      INCL(font.features,D.fontSpacing);
    ELSIF buffer="m" THEN
      font.spacing:=D.fontFixed;
      INCL(font.features,D.fontSpacing);
    END;

    GetField(name,12,buffer);
    co.StrToInt(buffer,font.avWidth,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,D.fontAvWidth);

    GetField(name,13,buffer);
    GetField(name,14,buffer2);
    NEW(font.charSet,str.Length(buffer)+1+str.Length(buffer2)+1);
    COPY(buffer,font.charSet^);
    str.Append("-",font.charSet^);
    str.Append(buffer2,font.charSet^);
    INCL(font.features,D.fontCharSet);

(*    IF info#NIL THEN
      font.ascent:=info.ascent;
      INCL(font.features,fontAscent);

      font.descent:=info.descent;
      INCL(font.features,fontDescent);
    END;*)

  END InitFromFontInfo;

  PROCEDURE (font : Font) LoadFont(display : Display;
                                   name: ARRAY OF CHAR;
                                   full : BOOLEAN):BOOLEAN;
  (**
    Loads the given font. First tries to find an ressource entry
    with the given name and load the font stored there before falling
    back to the given default.

    This method is X11 specific.
  *)

  VAR
    help : ARRAY 256 OF CHAR;

    PROCEDURE ConvertField(VAR source : ARRAY OF CHAR; pos : INTEGER; string : ARRAY OF CHAR);

    VAR
      x1,x2 : INTEGER;

    BEGIN
      x1:=0;
      WHILE (source[x1]#0X) & (pos>0) DO
        IF source[x1]="-" THEN
          DEC(pos);
        END;
        INC(x1);
      END;

      x2:=x1;
      WHILE (source[x2+1]#0X) & (source[x2+1]#"-") DO
        INC(x2);
      END;

      str.Delete(source,x1,x2-x1+1);
      str.Insert(string,x1,source);
    END ConvertField;


  BEGIN
    NEW(font.fullName,str.Length(name)+1);
    COPY(name,font.fullName^);

    IF ~full THEN
      font.fonts[0]:=X11.XLoadQueryFont(display.display,name);
      IF font.fonts[0]=NIL THEN
        RETURN FALSE;
      END;
(*      font.height:=font.fonts[0].max_bounds.ascent+font.fonts[0].max_bounds.descent;
      font.ascent:=font.fonts[0].max_bounds.ascent;
      font.descent:=font.fonts[0].max_bounds.descent;*)

      font.height:=font.fonts[0].ascent+font.fonts[0].descent;
      font.ascent:=font.fonts[0].ascent;
      font.descent:=font.fonts[0].descent;
    ELSE
      font.loaded:=TRUE;
      font.useCount:=1;

      font.features:={D.fontHeight,D.fontAscent,D.fontDescent};

      IF str2.Match("-*-*-medium-r-*-*-*-*-*-*-*-*-*-*",name) THEN

        (* bold *)
        COPY(name,help);
        ConvertField(help,3,"bold");
        font.fonts[font.StyleToPos({D.bold})]:=X11.XLoadQueryFont(display.display,help);

        (* italic *)
        COPY(name,help);
        ConvertField(help,4,"i");
        font.fonts[font.StyleToPos({D.italic})]:=X11.XLoadQueryFont(display.display,help);

        (* slanted *)
        COPY(name,help);
        ConvertField(help,4,"o");
        font.fonts[font.StyleToPos({D.slanted})]:=X11.XLoadQueryFont(display.display,help);

        (* bold+italic *)
        COPY(name,help);
        ConvertField(help,3,"bold");
        ConvertField(help,4,"i");
        font.fonts[font.StyleToPos({D.bold,D.italic})]:=X11.XLoadQueryFont(display.display,help);

        (* bold+slanted *)
        COPY(name,help);
        ConvertField(help,3,"bold");
        ConvertField(help,4,"o");
        font.fonts[font.StyleToPos({D.bold,D.slanted})]:=X11.XLoadQueryFont(display.display,help);
      END;
    END;
    RETURN TRUE;
  END LoadFont;

  PROCEDURE (d : Display) AddFont(font : Font);

  BEGIN
    font.last:=NIL;
    font.next:=d.fonts;
    IF d.fonts#NIL THEN
      d.fonts.last:=font;
    END;
    d.fonts:=font;
  END AddFont;

  PROCEDURE (d : Display) EnlargePointArray(len : LONGINT);

  BEGIN
    IF (d.pointArray=NIL) OR (LEN(d.pointArray^)<len) THEN
      NEW(d.pointArray,len);
    END;
  END EnlargePointArray;

  (**
    Loads the font. The OS specific font will be matched by evaluating the
    handed features. Every font loaded must be free using Free.

    RETURNS
    If the font can be loaded the method returns a new instance of font that must be used
    for future uses exspecially when calling Free.
  *)

  PROCEDURE (font : Font) Load(display : Display):BOOLEAN;

  VAR
    name   : ARRAY 256 OF CHAR;
    buffer : ARRAY 20 OF CHAR;
    f      : D.Font;

  BEGIN
    name:="-";

    (* name *)
    IF D.fontName IN font.features THEN
      str.Append(font.name^,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-medium-r-",name);

    (* setWidth *)
    IF D.fontSetWidth IN font.features THEN
      CASE font.setWidth OF
        D.fontNormal:
          str.Append("normal",name);
      | D.fontCondensed:
          str.Append("semicondensed",name);
      | D.fontNarrow:
          str.Append("narrow",name);
      | D.fontDouble:
          str.Append("double",name);
      END;
    ELSE
      str.Append("*",name);
    END;

    str.Append("--",name);

    (* height *)
    IF D.fontHeight IN font.features THEN
      co.IntToStr(font.height,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* pointHeight *)
    IF D.fontPointHeight IN font.features THEN
      co.IntToStr(font.pointHeight,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* horizRes *)
    IF D.fontHorizRes IN font.features THEN
      co.IntToStr(font.horizRes,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* vertRes *)
    IF D.fontVertRes IN font.features THEN
      co.IntToStr(font.vertRes,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* spacing *)
    IF D.fontSpacing IN font.features THEN
      CASE font.spacing OF
        D.fontFixed:
          str.Append("m",name);
      | D.fontProportional:
          str.Append("p",name);
      END;
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* avWidth *)
    IF D.fontAvWidth IN font.features THEN
      co.IntToStr(font.avWidth,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);
    IF D.fontCharSet IN font.features THEN
      str.Append(font.charSet^,name);
    ELSE
      str.Append("*",name);
    END;

    f:=display.fonts;
    WHILE (f#NIL) & ~str2.Match(name,f(Font).fullName^) DO
      f:=f(Font).next(Font);
    END;

    IF f#NIL THEN
      INC(f(Font).useCount);
      RETURN TRUE;
    END;

    f:=factory.CreateFont();
    IF f(Font).LoadFont(display,name,FALSE) THEN
      display.AddFont(f(Font));
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END Load;

  PROCEDURE (font : Font) Free(display : Display);

  VAR
    x : LONGINT;

  BEGIN
    DEC(font.useCount);

    IF font.useCount<=0 THEN
      FOR x:=0 TO LEN(font.fonts)-1 DO
        IF font.fonts[x]#NIL THEN
          X11.XFreeFont(display.display,font.fonts[x]);
        END;
      END;

      IF font=display.fonts THEN
        IF font.next#NIL THEN
          display.fonts:=font.next(Font);
          display.fonts.last:=NIL;
        END;
      ELSE
        IF font.last#NIL THEN
          font.last.next:=font.next;
        END;
        IF font.next#NIL THEN
          font.next.last:=font.last;
        END;
      END;
    END;
  END Free;

  (* -------- drawinfo stuff ------------- *)

  (* ----------- Clipping ---------------- *)

  (**
    Recalces the current clipping regions by analysing the
    current clipping stack.
  *)

  PROCEDURE (d : DrawInfo) RecalcClipRegion;

  VAR
    pos    : LONGINT;
    region : xu.Region;

  BEGIN
    region:=xu.XCreateRegion();
    xu.XUnionRegion(d.clipStack[d.clipPos].region,region,region);
    pos:=d.clipPos-1;
    WHILE pos>=0 DO
      xu.XIntersectRegion(d.clipStack[pos].region,region,region);
      DEC(pos);
    END;
    xu.XSetRegion(D.display(Display).display,d.gc,region);

    xu.XDestroyRegion(region);
  END RecalcClipRegion;

  (**
    Initializes a clip entry object.
  *)

  PROCEDURE (VAR c : ClipEntryDesc) Init(draw : DrawInfo);

  BEGIN
    c.region:=xu.XCreateRegion();
    c.draw:=draw;
  END Init;

  (**
    Recalcs the clipping rectangle.
  *)

  PROCEDURE (VAR c : ClipEntryDesc) Install;

  BEGIN
    c.draw.RecalcClipRegion;
  END Install;

  (**
    adds the given rectangle to the current clip entry.
  *)

  PROCEDURE (VAR c : ClipEntryDesc) Add(x,y,width,height : LONGINT);

  VAR
    rect : X11.XRectanglePtr;

  BEGIN
    s.NEW(rect,SIZE(X11.XRectangle));
    rect.x:=SHORT(x);
    rect.y:=SHORT(y);
    rect.width:=SHORT(width);
    rect.height:=SHORT(height);
    xu.XUnionRectWithRegion(rect,c.region,c.region);
    c.draw.RecalcClipRegion;
  END Add;

  (**
    adds the given rectangle to the current clip entry.
  *)

  PROCEDURE (VAR c : ClipEntryDesc) Sub(x,y,width,height : LONGINT);

  VAR
    rect   : X11.XRectanglePtr;
    region : xu.Region;

  BEGIN
    s.NEW(rect,SIZE(X11.XRectangle));
    rect.x:=SHORT(x);
    rect.y:=SHORT(y);
    rect.width:=SHORT(width);
    rect.height:=SHORT(height);

    region:=xu.XCreateRegion();
    xu.XUnionRectWithRegion(rect,region,region);
    xu.XSubtractRegion(c.region,region,c.region);
    xu.XDestroyRegion(region);

    c.draw.RecalcClipRegion;
  END Sub;

  (**
    Frees the given clip entry.
  *)

  PROCEDURE (VAR c : ClipEntryDesc) Free;

  BEGIN
    xu.XDestroyRegion(c.region);
  END Free;

  PROCEDURE (d : DrawInfo) InstallClip*(x,y,w,h : LONGINT);

  VAR
    help : ClipStack;
    pos  : LONGINT;

  BEGIN
    IF d.clipPos>=LEN(d.clipStack^)-1 THEN
      NEW(help,LEN(d.clipStack^)+stackSizeIncrement);
      FOR pos:=0 TO LEN(d.clipStack^)-1 DO
        help[pos]:=d.clipStack[pos];
      END;
      d.clipStack:=help;
    END;
    INC(d.clipPos);

    d.clipStack[d.clipPos].Init(d);
    d.clipStack[d.clipPos].Install;

    d.clipStack[d.clipPos].Add(x,y,w,h);
  END InstallClip;

  PROCEDURE (d : DrawInfo) ReinstallClip*;

  BEGIN
    d.clipStack[d.clipPos].Install;
  END ReinstallClip;

  PROCEDURE (d : DrawInfo) AddRegion*(x,y,width,height : LONGINT);

  BEGIN
    d.clipStack[d.clipPos].Add(x,y,width,height);
  END AddRegion;

  PROCEDURE (d : DrawInfo) SubRegion*(x,y,width,height : LONGINT);

  BEGIN
    d.clipStack[d.clipPos].Sub(x,y,width,height);
  END SubRegion;

  PROCEDURE (d : DrawInfo) GetClipRegion*(VAR x,y,w,h : LONGINT);

  VAR
    rect : X11.XRectangle;

  BEGIN
    xu.XClipBox(d.clipStack[d.clipPos].region,rect);
    x:=rect.x;
    y:=rect.y;
    w:=rect.width;
    h:=rect.height;
  END GetClipRegion;

  PROCEDURE (d : DrawInfo) FreeLastClip*;

  BEGIN
    d.clipStack[d.clipPos].Free;
    DEC(d.clipPos);
    IF d.clipPos>=0 THEN
      d.clipStack[d.clipPos].Install;
    END;
  END FreeLastClip;

  (**
    Initialize an instance of the  DrawInfo class.
  *)

  PROCEDURE (d : DrawInfo) Init(window : X11.Drawable);

  VAR
    mask   : X11.ulongmask;
    values : X11.XGCValues;

  BEGIN
    d.window:=window;
    mask:={};
    d.gc:=X11.XCreateGC(D.display(Display).display,window,mask,values);
    X11.XSetArcMode(D.display(Display).display,d.gc,X11.ArcPieSlice);
    d.mode:={};

    (* The array stacks *)
    d.fPenPos:=-1;
    NEW(d.fPenStack,initialStackSize);
    d.bPenPos:=-1;
    NEW(d.bPenStack,initialStackSize);
    d.fontPos:=-1;
    NEW(d.fontStack,initialStackSize);
    d.clipPos:=-1;
    NEW(d.clipStack,initialStackSize);

    (* The list stacks *)
    d.styleStack:=NIL;
    d.dashStack:=NIL;
    d.patternStack:=NIL;
    d.modeStack:=NIL;

    (* convinience stuff *)
    d.PushFont(D.normalFont,{});
  END Init;

  (**
    Deinitializes the drawInfo
  *)

  PROCEDURE (d : DrawInfo) Deinit;

  BEGIN
    d.PopFont;

    ASSERT(d.fPenPos=-1);
    ASSERT(d.bPenPos=-1);
    ASSERT(d.fontPos=-1);
    ASSERT(d.styleStack=NIL);
    ASSERT(d.dashStack=NIL);
    ASSERT(d.clipPos=-1);
    ASSERT(d.patternStack=NIL);
    ASSERT(d.modeStack=NIL);

    X11.XFreeGC(D.display(Display).display,d.gc);
  END Deinit;

  PROCEDURE (d : DrawInfo) PushFont*(font : D.Font; style : SET);

  VAR
    help      : FontStack;
    pos       : LONGINT;
    sysHandle : X11.XFontStructPtr;

  BEGIN
    IF d.fontPos>=LEN(d.fontStack^)-1 THEN
      NEW(help,LEN(d.fontStack^)+stackSizeIncrement);
      FOR pos:=0 TO LEN(d.fontStack^)-1 DO
        help[pos]:=d.fontStack[pos];
      END;
      d.fontStack:=help;
    END;
    IF (d.fontPos>=0) & (d.fontStack[d.fontPos].font=font)
    &  (d.fontStack[d.fontPos].style=style) THEN
      INC(d.fontStack[d.fontPos].count);
    ELSE  
      INC(d.fontPos);        
    
      IF ~font(Font).loaded THEN
        IF font(Font).LoadFont(D.display(Display),font(Font).fullName^,TRUE) THEN END;
      END;

      d.fontStack[d.fontPos].font:=font(Font);
      d.fontStack[d.fontPos].style:=style;
      d.fontStack[d.fontPos].count:=1;
      sysHandle:=d.fontStack[d.fontPos].font.GetFontHandle(style);
      X11.XSetFont(D.display(Display).display,d.gc,sysHandle.fid);
    END;  
  END PushFont;

  PROCEDURE (d : DrawInfo) PopFont*;

  VAR
    handle : X11.XFontStructPtr;

  BEGIN
    IF d.fontStack[d.fontPos].count>1 THEN
      DEC(d.fontStack[d.fontPos].count);
    ELSE  
      DEC(d.fontPos);
      IF d.fontPos>=0 THEN
        handle:=d.fontStack[d.fontPos].font.GetFontHandle(d.fontStack[d.fontPos].style);
        X11.XSetFont(D.display(Display).display,d.gc,handle.fid);
      END;  
    END;
  END PopFont;

  PROCEDURE (d : DrawInfo) DrawString*(x,y : LONGINT; text : ARRAY OF CHAR; length : LONGINT);

  BEGIN
    X11.XDrawString(D.display(Display).display,d.window,d.gc,x,y,text,length);
  END DrawString;

  PROCEDURE (d : DrawInfo) DrawLongString*(x,y : LONGINT; text : ARRAY OF LONGCHAR; length : LONGINT);

  VAR i : LONGINT;

  BEGIN
    FOR i:=0 TO LEN(text)-1 DO
      IF ORD(text[i])#0 THEN
        text[i]:=LONGCHR(((ORD(text[i]) MOD 256)*256) + (ORD(text[i]) DIV 256));
      END;
    END;

    X11.XDrawString16(D.display(Display).display,d.window,d.gc,x,y,text,length);
  END DrawLongString;

  PROCEDURE (d : DrawInfo) DrawFillString*(x,y : LONGINT; text : ARRAY OF CHAR; length : LONGINT);

  BEGIN
    X11.XDrawImageString(D.display(Display).display,d.window,d.gc,x,y,text,length);
  END DrawFillString;

  PROCEDURE (d : DrawInfo) DrawFillLongString*(x,y : LONGINT; text : ARRAY OF LONGCHAR; length : LONGINT);

  VAR i : LONGINT;

  BEGIN
    FOR i:=0 TO LEN(text)-1 DO
      IF ORD(text[i])#0 THEN
        text[i]:=LONGCHR(((ORD(text[i]) MOD 256)*256) + (ORD(text[i]) DIV 256));
      END;
    END;

    X11.XDrawImageString16(D.display(Display).display,d.window,d.gc,x,y,text,length);
  END DrawFillLongString;

  PROCEDURE (d : DrawInfo) PushForeground*(color : D.Color);

  VAR
    help : PenStack;
    pos  : LONGINT;

  BEGIN
    IF d.fPenPos>=LEN(d.fPenStack^)-1 THEN
      NEW(help,LEN(d.fPenStack^)+stackSizeIncrement);
      FOR pos:=0 TO LEN(d.fPenStack^)-1 DO
        help[pos]:=d.fPenStack[pos];
      END;
      d.fPenStack:=help;
    END;
    
    IF (d.fPenPos>=0) & (d.fPenStack[d.fPenPos].color=color) THEN
      INC(d.fPenStack[d.fPenPos].count);
    ELSE  
      INC(d.fPenPos);

      d.fPenStack[d.fPenPos].color:=color;
      d.fPenStack[d.fPenPos].count:=1;

      X11.XSetForeground(D.display(Display).display,d.gc,color);
    END;  
  END PushForeground;

  PROCEDURE (d : DrawInfo) PopForeground*;

  BEGIN
    IF d.fPenStack[d.fPenPos].count>1 THEN
      DEC(d.fPenStack[d.fPenPos].count);
    ELSE  
      DEC(d.fPenPos);
      IF d.fPenPos>=0 THEN
        X11.XSetForeground(D.display(Display).display,d.gc,
                           d.fPenStack[d.fPenPos].color);
      END;
    END;  
  END PopForeground;

  PROCEDURE (d : DrawInfo) PushDrawMode*(mode : LONGINT);

  VAR
    m     : DrawMode;
    xMode : INTEGER;

  BEGIN
    NEW(m);
    m.mode:=mode;
    CASE mode OF
      D.copy:
        xMode:=X11.GXcopy;
    | D.invert:
        xMode:=X11.GXinvert;
    END;
    X11.XSetFunction(D.display(Display).display,d.gc,xMode);
    m.next:=d.modeStack;
    d.modeStack:=m;
  END PushDrawMode;

  PROCEDURE (d : DrawInfo) PopDrawMode*;

  VAR
   xMode : INTEGER;

  BEGIN
    d.modeStack:=d.modeStack.next;
    IF d.modeStack#NIL THEN
      CASE d.modeStack.mode OF
        D.copy:
          xMode:=X11.GXcopy;
      | D.invert:
          xMode:=X11.GXinvert;
      END;
      X11.XSetFunction(D.display(Display).display,d.gc,xMode);
    ELSE
      X11.XSetFunction(D.display(Display).display,d.gc,X11.GXcopy);
    END;
  END PopDrawMode;

  PROCEDURE (d : DrawInfo) PushBackground*(color : D.Color);

  VAR
    help : PenStack;
    pos  : LONGINT;

  BEGIN
    IF d.bPenPos>=LEN(d.bPenStack^)-1 THEN
      NEW(help,LEN(d.bPenStack^)+stackSizeIncrement);
      FOR pos:=0 TO LEN(d.bPenStack^)-1 DO
        help[pos]:=d.bPenStack[pos];
      END;
      d.bPenStack:=help;
    END;
    IF (d.bPenPos>=0) & (d.bPenStack[d.bPenPos].color=color) THEN
      INC(d.bPenStack[d.bPenPos].count);
    ELSE  
      INC(d.bPenPos);

      d.bPenStack[d.bPenPos].color:=color;
      d.bPenStack[d.bPenPos].count:=1;

      X11.XSetBackground(D.display(Display).display,d.gc,color);
    END;  
  END PushBackground;

  PROCEDURE (d : DrawInfo) PopBackground*;

  BEGIN
    IF d.bPenStack[d.bPenPos].count>1 THEN
      DEC(d.bPenStack[d.bPenPos].count);
    ELSE  
      DEC(d.bPenPos);
      IF d.bPenPos>=0 THEN
        X11.XSetBackground(D.display(Display).display,d.gc,
                           d.bPenStack[d.bPenPos].color);
      END;
    END;  
  END PopBackground;

  PROCEDURE (d : DrawInfo) PushStyle*(size, mode : LONGINT);

  VAR
    pen  : PenStyle;
    lMode : LONGINT;

  BEGIN
    NEW(pen);
    pen.size:=size;
    pen.mode:=mode;
    IF mode=D.roundPen THEN
      pen.cap:=X11.CapRound;
      pen.join:=X11.JoinRound;
    ELSE
      pen.cap:=X11.CapButt;
      pen.join:=X11.JoinBevel;
    END;

    IF d.dashStack#NIL THEN
      lMode:=d.dashStack.mode;
    ELSE
      lMode:=X11.LineSolid;
    END;

    X11.XSetLineAttributes(D.display(Display).display,d.gc,pen.size,lMode,pen.cap,pen.join);
    pen.next:=d.styleStack;
    d.styleStack:=pen;

  END PushStyle;

  PROCEDURE (d : DrawInfo) PopStyle*;

  VAR
    mode : LONGINT;

  BEGIN
    d.styleStack:=d.styleStack.next;

    IF d.dashStack#NIL THEN
      mode:=d.dashStack.mode;
    ELSE
      mode:=X11.LineSolid;
    END;

    IF d.styleStack#NIL THEN
      X11.XSetLineAttributes(D.display(Display).display,d.gc,d.styleStack.size,mode,d.styleStack.cap,d.styleStack.join);
    ELSE
      X11.XSetLineAttributes(D.display(Display).display,d.gc,0,mode,X11.CapButt,X11.JoinBevel);
    END;
  END PopStyle;

  PROCEDURE (d : DrawInfo) PushDash*(dashList : ARRAY OF CHAR; mode : LONGINT);

  VAR
    dash   : PenDash;
    size,
    cap,
    join,x : LONGINT;

  BEGIN
    NEW(dash);
    dash.next:=NIL;
    NEW(dash.list,LEN(dashList));
    FOR x:=0 TO LEN(dashList)-1 DO
      dash.list[x]:=dashList[x];
    END;

    IF mode=D.fMode THEN
      dash.mode:=X11.LineOnOffDash;
    ELSE
      dash.mode:=X11.LineDoubleDash;
    END;

    IF d.styleStack#NIL THEN
      size:=d.styleStack.size;
      cap:=d.styleStack.cap;
      join:=d.styleStack.join;
    ELSE
      size:=0;
      cap:=X11.CapButt;
      join:=X11.JoinBevel;
    END;

    X11.XSetLineAttributes(D.display(Display).display,d.gc,size,dash.mode,cap,join);
    X11.XSetDashes(D.display(Display).display,d.gc,0,dash.list^,LEN(dashList));

    dash.next:=d.dashStack;
    d.dashStack:=dash;
  END PushDash;

  PROCEDURE (d : DrawInfo) PopDash*;

  VAR
    size,
    cap,
    join : LONGINT;

  BEGIN
    d.dashStack:=d.dashStack.next;

    IF d.styleStack#NIL THEN
      size:=d.styleStack.size;
      cap:=d.styleStack.cap;
      join:=d.styleStack.join;
    ELSE
      size:=0;
      cap:=X11.CapButt;
      join:=X11.JoinBevel;
    END;


    IF d.dashStack#NIL THEN
      X11.XSetLineAttributes(D.display(Display).display,d.gc,size,d.dashStack.mode,cap,join);
    ELSE
      X11.XSetLineAttributes(D.display(Display).display,d.gc,0,X11.LineSolid,X11.CapButt,X11.JoinBevel);
    END;
  END PopDash;

  PROCEDURE (d : DrawInfo) PushPattern*(pattern : ARRAY OF CHAR; width, height : LONGINT; mode : LONGINT);

  VAR
    pat : Pattern;

  BEGIN
    NEW(pat);
    pat.pixMap:=X11.XCreateBitmapFromData(D.display(Display).display,
                    X11.XRootWindow(D.display(Display).display,D.display(Display).scrNum),
                    pattern,width,height);
    IF pat.pixMap=0 THEN
      Err.String("Cannot create pimap!");Err.Ln;
    END;
    pat.mode:=mode;

    X11.XSetStipple(D.display(Display).display,d.gc,pat.pixMap);
    CASE mode OF
      D.fgPattern:
        X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillStippled);
    | D.fbPattern:
        X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillOpaqueStippled);
    ELSE
      Err.String("Unsuported patternMode!"); Err.Ln;
    END;
    pat.next:=d.patternStack;
    d.patternStack:=pat;
  END PushPattern;

  PROCEDURE (d : DrawInfo) PopPattern*;

  BEGIN
    X11.XFreePixmap(D.display(Display).display,d.patternStack.pixMap);
    d.patternStack:=d.patternStack.next;
    IF d.patternStack#NIL THEN

      CASE d.patternStack.mode OF
        D.fgPattern:
          X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillStippled);
      | D.fbPattern:
          X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillOpaqueStippled);
      ELSE
       Err.String("Unsuported patternMode!"); Err.Ln;
      END;
      X11.XSetStipple(D.display(Display).display,d.gc,d.patternStack.pixMap);

    ELSE
      X11.XSetFillStyle(D.display(Display).display,d.gc,X11.FillSolid);
    END;
  END PopPattern;

  PROCEDURE (d : DrawInfo) PushUniqueFillPattern*(pos, maximum : LONGINT);

  VAR
   pattern : LONGINT;

  BEGIN
    pattern:=pos MOD 3;
    IF (pos=maximum) & (pattern=1) THEN
      pattern:=2;
    END;

    CASE D.display.colorMode OF
      D.colorMode:
        CASE pattern OF
          0: d.PushForeground(D.warnColor);
        | 1: d.PushForeground(D.fillColor);
        | 2: d.PushForeground(D.halfShadowColor);
        END;
    | D.greyScaleMode:
        CASE pattern OF
          0: d.PushForeground(D.shineColor);
        | 1: d.PushForeground(D.halfShineColor);
        | 2: d.PushForeground(D.halfShadowColor);
        END;
    | D.monochromeMode: (* only use black and white and some shadings *)
        CASE pattern OF
        ELSE
          ASSERT(FALSE);
(*          0: d.PushForeground(whiteColor);
        | 1: d.PushForeground(whiteColor);
             d.PushBackground(whiteColor);
             d.PushPattern(disablePattern,disableWidth,disableHeight,fbPattern);
        | 2: d.PushForeground(blackColor);*)
        END;
    END;
  END PushUniqueFillPattern;

  PROCEDURE (d : DrawInfo) PopUniqueFillPattern*(pos, maximum : LONGINT);

  VAR
   pattern : LONGINT;

  BEGIN
    pattern:=pos MOD 3;
    IF (pos=maximum) & (pattern=1) THEN
      pattern:=2;
    END;

    IF (D.display.colorMode=D.monochromeMode) & (pattern=1) THEN
      d.PopForeground;
      d.PopBackground;
      d.PopPattern;
    ELSE
      d.PopForeground;
    END;
  END PopUniqueFillPattern;

  (* Drawing functions *)

  PROCEDURE (d : DrawInfo) DrawPoint*(x,y : LONGINT);

  BEGIN
    X11.XDrawPoint(D.display(Display).display,d.window,d.gc,x,y);
  END DrawPoint;

  PROCEDURE (d : DrawInfo) DrawLine*(x1,y1,x2,y2 : LONGINT);

  BEGIN
    X11.XDrawLine(D.display(Display).display,d.window,d.gc,x1,y1,x2,y2);
  END DrawLine;

  PROCEDURE (d : DrawInfo) FillRectangle*(x,y,width,height : LONGINT);

  BEGIN
    X11.XFillRectangle(D.display(Display).display,d.window,d.gc,x,y,width,height);
  END FillRectangle;

  PROCEDURE (d : DrawInfo) DrawArc*(x,y,width,height,angle1,angle2 : LONGINT);

  BEGIN
    X11.XDrawArc(D.display(Display).display,d.window,d.gc,x,y,width-1,height-1,angle1,angle2);
  END DrawArc;

  PROCEDURE (d : DrawInfo) FillArc*(x,y,width,height,angle1,angle2 : LONGINT);

  BEGIN
    X11.XFillArc(D.display(Display).display,d.window,d.gc,x,y,width,height,angle1,angle2);
  END FillArc;

  PROCEDURE (d : DrawInfo) FillPolygon*(points : ARRAY OF D.PointDesc;
                                        count : LONGINT);

  VAR
    x : LONGINT;

  BEGIN
    D.display(Display).EnlargePointArray(count);
    FOR x:=0 TO count-1 DO
      D.display(Display).pointArray[x].x:=SHORT(points[x].x);
      D.display(Display).pointArray[x].y:=SHORT(points[x].y);
    END;
    X11.XFillPolygon(D.display(Display).display,d.window,d.gc,D.display(Display).pointArray^,count,X11.Complex,X11.CoordModeOrigin);
  END FillPolygon;

  PROCEDURE (d : DrawInfo) FillBackground*(x,y,width,height : LONGINT);

(*  VAR
    Pat : ARRAY 2 OF CHAR;*)

  BEGIN
(*    Pat[0]:=CHR(170);  (* 10101010 *)
    Pat[1]:=CHR(85);   (* 01010101 *)*)
    d.PushForeground(D.backgroundColor);
(*    d.PushBackground(halfShineColor);
    d.PushPattern(Pat,8,2,fbPattern);*)
    X11.XFillRectangle(D.display(Display).display,d.window,d.gc,x,y,width,height);
(*    d.PopPattern;*)
    d.PopForeground;
(*    d.PopBackground;*)
  END FillBackground;

  PROCEDURE (d : DrawInfo) CopyArea*(sX,sY,width,height,dX,dY : LONGINT);

  BEGIN
    X11.XSetGraphicsExposures(D.display(Display).display,d.gc,X11.True);
    X11.XCopyArea(D.display(Display).display,d.window,d.window,d.gc,
                  sX,sY,width,height,dX,dY);
    X11.XSetGraphicsExposures(D.display(Display).display,d.gc,X11.False);
  END CopyArea;

  PROCEDURE (d : DrawInfo) CopyFromBitmap*(bitmap : D.Bitmap;
                                               sX,sY,width,height,dX,dY : LONGINT);

  BEGIN
    X11.XCopyArea(D.display(Display).display,
                  bitmap(Bitmap).pixmap,
                  d.window,
                  d.gc,
                  sX,sY,width,height,dX,dY);
  END CopyFromBitmap;

  PROCEDURE (d : DrawInfo) CopyToBitmap*(sX,sY,width,height,dX,dY : LONGINT;
                                             bitmap : D.Bitmap);

  BEGIN
    X11.XSetGraphicsExposures(D.display(Display).display,d.gc,X11.True);
    X11.XCopyArea(D.display(Display).display,
                  bitmap(Bitmap).pixmap,d.window,d.gc,
                  sX,sY,width,height,dX,dY);
    X11.XSetGraphicsExposures(D.display(Display).display,d.gc,X11.False);
  END CopyToBitmap;

  (* ------------ Data Exchange stuff --------------- *)

  (**
    Initializes an Data object.
  *)

(*  PROCEDURE (data : Data) Init*;

  BEGIN
    data.type:=none;
    data.length:=0;

    data.string:=NIL;
    data.ints:=NIL;
    data.longs:=NIL;

    data.xData:=NIL;
  END Init;

  PROCEDURE (data : Data) Free*;

  BEGIN
    IF data.xData#NIL THEN
      XFree(data.xData);
    END;
  END Free;

  PROCEDURE (data : Data) InitFromX(window : Window; event : X11.XSelectionEvent):BOOLEAN;

  VAR
    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    delete      : X11.Bool;

  BEGIN
    data.Init;

    IF event.property=X11.None THEN
      Err.String("Selection denied"); Err.Ln;
      RETURN FALSE;
    END;

    IF window.display.dragging THEN
      delete:=X11.False;
    ELSE
      delete:=X11.True;
    END;

    IF X11.XGetWindowProperty(window.display.display,
        event.requestor,event.property,0,MAX(LONGINT),delete,
        X11.AnyPropertyType,retType,retFormat,itemsReturn,bytesLeft,
        data.xData)#X11.Success THEN
        Err.String("Cannot get property"); Err.Ln;
        RETURN FALSE;
    END;

    IF retType=X11.None THEN
      RETURN FALSE;
    END;

    CASE retType OF
      a.XA_STRING: data.type:=text;
    ELSE
      RETURN FALSE;
    END;

    CASE retFormat OF
       8: NEW(data.string,itemsReturn+1);
          s.MOVE(s.VAL(LONGINT,data.xData),
                 s.VAL(LONGINT,data.string),itemsReturn);
          data.string[itemsReturn]:=0X;

          Err.String(">>"); Err.String(data.string^); Err.String("<<"); Err.Ln;

    | 16: NEW(data.ints,itemsReturn+1);
          s.MOVE(s.VAL(LONGINT,data.xData),
                 s.VAL(LONGINT,data.ints),itemsReturn*2);
    | 32: NEW(data.longs,itemsReturn+1);
          s.MOVE(s.VAL(LONGINT,data.xData),
                 s.VAL(LONGINT,data.longs),itemsReturn*4);
    END;

    data.length:=itemsReturn;

    IF window.display.dragging THEN
      Err.String("Finishing drop selection"); Err.Ln;
      X11.XConvertSelection(window.display.display,
                            window.display.dSelection,
                            window.display.dndSuccess,
                            window.display.xSelection,
                            window.window,X11.CurrentTime);
      window.display.dragging:=FALSE;
    END;

    RETURN TRUE;
  END InitFromX;

  PROCEDURE (data : Data) InitToX(VAR event : X11.XSelectionEvent):BOOLEAN;

  VAR
    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;

  BEGIN
    IF (event.target=a.XA_STRING) & (data.string#NIL) THEN

      IF X11.XGetWindowProperty(event.display,
        event.requestor,event.property,0,MAX(LONGINT),X11.False,
        event.target,retType,retFormat,itemsReturn,bytesLeft,
        data.xData)#X11.Success THEN
          Err.String("Cannot get property"); Err.Ln;
          RETURN FALSE;
      END;

      X11.XChangeProperty(event.display,event.requestor,event.property,
                          a.XA_STRING,8,
                          X11.PropModeReplace,data.string^,
                          str.Length(data.string^));
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END InitToX;*)

  (**
    Returns the window currently under the mouse pointer. It also returns the
    screen relative and window relative mouse coordinates.
  *)

  PROCEDURE (d : Display) GetWindowOnScreen(VAR rX,rY,cX,cY : LONGINT):X11.Window;

  VAR
    help,rW,cW  : LONGINT;
    dragWin     : X11.Window;
    mask        : SET;

    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    data        : C.charPtr1d;

  BEGIN
    dragWin:=0;
    data:=NIL;
    itemsReturn:=0;

    IF X11.XQueryPointer(d.display,X11.XRootWindow(d.display,0),
                         rW,cW,rX,rY,cX,cY,mask)=X11.True THEN

      WHILE (rW#0) & (dragWin=0) & (cW#0) & (rW#cW) DO
        IF (X11.XGetWindowProperty(d.display,
                                   cW,d.wmState,0,MAX(LONGINT),X11.False,
                                   X11.AnyPropertyType,retType,retFormat,itemsReturn,
                                   bytesLeft,data)=X11.Success) & (retType#X11.None) THEN
          dragWin:=cW;
        END;

        IF rW#0 THEN
          help:=cW;
          IF X11.XTranslateCoordinates(d.display,
                                       rW,cW,cX,cY,cX,cY,cW)=X11.True THEN
            rW:=help;
          ELSE
            Err.String("Cannot convert coordinates!"); Err.Ln;
            rW:=0;
          END;
        END;
      END;
    END;
    RETURN dragWin;
  END GetWindowOnScreen;

  (**
    Get the VO window matching the given X11 window.
  *)

  PROCEDURE (d : Display) GetWindow(window : X11.Window):D.Window;

  VAR
    help : D.Window;

  BEGIN
    help:=d.winList;
    WHILE help#NIL DO
      IF help(Window).window=window THEN
        RETURN help;
      END;
      help:=help(Window).next;
    END;
    RETURN NIL;
  END GetWindow;

  (**
    Call this method if you want the Display to stop generating
    QuickHelp calls to windows. This is necessesarry, if you are
    opening a QuickHelp and don't want to have a second after the
    second timeout.
  *)

  PROCEDURE (d : Display) StopContextHelp;

  BEGIN
    IF d.contextHelp THEN
(*      Err.String("Stop contexthelp"); Err.Ln;*)
      d.contextHelp:=FALSE;
      IF d.timeOut#NIL THEN
        d.RemoveTimeOut(d.timeOut);
        d.timeOut:=NIL;
      END;
    END;
  END StopContextHelp;

  (**
    Restart the generation of QuickHelp calls to windows stoped
    with Display.StopContextHelp.
  *)

  PROCEDURE (d : Display) RestartContextHelp;

  VAR
    timeOut : D.TimeOut;

  BEGIN
    IF ~d.contextHelp THEN
(*      Err.String("Restart contexthelp"); Err.Ln;*)
      timeOut:=d.AddTimeOut(contextTimeOutSec,contextTimeOutMil,d);
      d.timeOut:=timeOut(TimeOut);
      d.contextHelp:=TRUE;
    END;
  END RestartContextHelp;

  (**
    Restart the generation of QuickHelp calls to windows stoped
    with Display.StopContextHelp.
  *)

  PROCEDURE (d : Display) StartContextHelp;


  VAR
    timeOut : D.TimeOut;

  BEGIN
    IF ~d.contextHelp THEN
(*      Err.String("Start contexthelp"); Err.Ln;*)
      timeOut:=d.AddTimeOut(contextTimeOutSec,contextTimeOutMil,d);
      d.timeOut:=timeOut(TimeOut);
      d.contextHelp:=TRUE;
    END;
  END StartContextHelp;

  (**
    Adds window to the internal list of windows.
  *)

  PROCEDURE (d : Display) AddWindow(w : D.Window);

  BEGIN
    IF d.winList#NIL THEN
      w(Window).last:=NIL;
      w(Window).next:=d.winList;
      d.winList(Window).last:=w;
    ELSE
      w(Window).last:=NIL;
      w(Window).next:=NIL;
    END;
    d.winList:=w;
  END AddWindow;

  (**
    Removes window from the internal list of windows.
  *)

  PROCEDURE (d : Display) RemoveWindow(w : D.Window);

  BEGIN
    IF d.winList=w THEN
      d.winList:=d.winList(Window).next;
    END;

    IF w(Window).last#NIL THEN
      w(Window).last(Window).next:=w(Window).next;
    END;

    IF w(Window).next#NIL THEN
      w(Window).next(Window).last:=w(Window).last;
    END;
  END RemoveWindow;

  PROCEDURE (d : Display) LoadFont*(font : D.Font):BOOLEAN;

  BEGIN
    RETURN font(Font).Load(d);
  END LoadFont;

  PROCEDURE (d : Display) FreeFont*(font : D.Font);

  BEGIN
    ASSERT(font#NIL);

    font(Font).Free(d);
  END FreeFont;

  PROCEDURE (d : Display) GetFontList*():D.Font;

  VAR
    count,
    x,y    : LONGINT;
    names  : C.charPtr2d;
    fonts  : X11.XFontStructPtr1d;
    first,
    last,
    font   : D.Font;
    name   : D.FontName;

  BEGIN
    names:=X11.XListFontsWithInfo(d.display,"-*-*-medium-r-*--*-*-*-*-*-*-*-*",
                                  MAX(LONGINT),count,fonts);

    first:=NIL;

    IF (names#NIL) & (count>0) THEN
      FOR x:=0 TO count-1 DO
         y:=0;
         WHILE names[x][y]#0X DO
           name[y]:=names[x][y];
           INC(y);
         END;
         name[y]:=0X;

         font:=factory.CreateFont();
         font(Font).Init;
         font(Font).InitFromFontInfo(name,fonts[x]);

         IF x=0 THEN
           first:=font;
           font.last:=NIL;
         ELSE
           last.next:=font;
           font.last:=last;
         END;
         last:=font;
         font.next:=NIL;
      END;
    END;

    X11.XFreeFontInfo(names,fonts[0],count);

    RETURN first;
  END GetFontList;

  (* -------- color stuff of D.display ------------- *)

  PROCEDURE (d : Display) AllocateColor*(r,g,b,default : LONGINT):D.Color;

  VAR
    color : X11.XColor;

  BEGIN
    (*
      Scaling down color values from 31 bit to 16 bit,
      because X11 only supports 16 color values.
    *)
    color.red:=SHORT(r DIV 32768);
    color.green:=SHORT(g DIV 32768);
    color.blue:=SHORT(b DIV 32768);

    IF X11.XAllocColor(d.display,d.colorMap,color)=0 THEN
      color.pixel:=default;
      X11.XQueryColor(d.display,d.colorMap,color);
      IF X11.XAllocColor(d.display,d.colorMap,color)=0 THEN
        ASSERT(FALSE);
      END;
    END;
    RETURN color.pixel;
  END AllocateColor;

  PROCEDURE (d : Display) AllocateNamedColor*(name : ARRAY OF CHAR;
                                                  default : D.Color):D.Color;
  VAR
    exact,color : X11.XColor;

  BEGIN
    IF X11.XLookupColor(d.display,d.colorMap,name,exact,color)#0 THEN
      RETURN d.AllocateColor(color.red*32768,color.green*32768,color.blue*32768,default);
    ELSE
      color.pixel:=default;
      X11.XQueryColor(d.display,d.colorMap,color);
      IF X11.XAllocColor(d.display,d.colorMap,color)=0 THEN
        ASSERT(FALSE);
      END;
      RETURN color.pixel;
    END;
  END AllocateNamedColor;

  PROCEDURE (d : Display) IsAllocatedColor*(color : D.Color):BOOLEAN;

  BEGIN
    RETURN TRUE; (*color>=0;*)
  END IsAllocatedColor;

  PROCEDURE (d : Display) FreeColor*(color : D.Color);

  VAR
    pixel : ARRAY 1 OF LONGINT;

  BEGIN
    pixel[0]:=color;
    X11.XFreeColors(d.display,d.colorMap,pixel,1,0);
  END FreeColor;

  (* ------------ Display ----------------- *)

  PROCEDURE (d : Display) AddTimeOut*(sec, msec : LONGINT; object : O.MsgObject):TimeOut;

  VAR
    timeOut,
    help     : TimeOut;
    dt       : sc.DateTime;
    interval : t.Interval;
    milli    : LONGINT;

  BEGIN
    NEW(timeOut);
    timeOut.object:=object;
    sc.GetClock(dt);
    c.SetTimeStamp(dt,timeOut.time);

    milli:=sec*1000+msec;

    t.InitInterval(interval,0,milli);

    timeOut.time.Add(interval);

    IF (d.timeOutList=NIL) OR (timeOut.time.Cmp(d.timeOutList.time)<=0) THEN
      timeOut.next:=d.timeOutList;
      d.timeOutList:=timeOut;
    ELSE
      help:=d.timeOutList;
      WHILE (help.next#NIL) & (help.next.time.Cmp(timeOut.time)<0) DO
        help:=help.next;
      END;
      timeOut.next:=help.next;
      help.next:=timeOut;
    END;

    RETURN timeOut;
  END AddTimeOut;

  PROCEDURE (d : Display) RemoveTimeOut*(timeOut : D.TimeOut);

  VAR
    help,
    last : TimeOut;

  BEGIN
    IF d.timeOutList=NIL THEN
      RETURN;
    END;

    IF d.timeOutList=timeOut THEN
      d.timeOutList:=d.timeOutList.next;
      RETURN;
    END;

    help:=d.timeOutList.next;
    last:=d.timeOutList;
    WHILE (help#NIL) & (help#timeOut) DO
      last:=help;
      help:=help.next;
    END;
    IF help#NIL THEN
      last.next:=help.next;
    END;
  END RemoveTimeOut;

  PROCEDURE (d : Display) AddSleep*(object : O.MsgObject):Sleep;

  VAR
    sleep : Sleep;

  BEGIN
    NEW(sleep);
    sleep.object:=object;

    sleep.next:=d.sleepList;
    d.sleepList:=sleep;

    RETURN sleep;
  END AddSleep;

  PROCEDURE (d : Display) RemoveSleep*(sleep : D.Sleep);

  VAR
    help,
    last : Sleep;

  BEGIN
    IF d.sleepList=NIL THEN
      RETURN;
    END;

    IF d.sleepList=sleep THEN
      d.sleepList:=d.sleepList.next;
      RETURN;
    END;

    help:=d.sleepList.next;
    last:=d.sleepList;
    WHILE (help#NIL) & (help#sleep) DO
      last:=help;
      help:=help.next;
    END;
    IF help#NIL THEN
      last.next:=help.next;
    END;
  END RemoveSleep;

  PROCEDURE (d : Display) AddChannel*(channel : ch.Channel; object : O.MsgObject):Channel;

  VAR
    entry : Channel;

  BEGIN
    NEW(entry);
    entry.channel:=channel;
    entry.object:=object;

    entry.next:=d.channelList;
    d.channelList:=entry;

    RETURN entry;
  END AddChannel;

  PROCEDURE (d : Display) RemoveChannel*(channel : D.Channel);

  VAR
    help,
    last : Channel;

  BEGIN
    IF d.channelList=NIL THEN
      RETURN;
    END;

    IF d.channelList=channel THEN
      d.channelList:=d.channelList.next;
      RETURN;
    END;

    help:=d.channelList.next;
    last:=d.channelList;
    WHILE (help#NIL) & (help#channel) DO
      last:=help;
      help:=help.next;
    END;
    IF help#NIL THEN
      last.next:=help.next;
    END;
  END RemoveChannel;

  PROCEDURE (d : Display) CreateBitmap*(width, height : LONGINT):D.Bitmap;

  VAR
    bitmap : D.Bitmap;

  BEGIN
    bitmap:=factory.CreateBitmap();
    bitmap(Bitmap).pixmap:=X11.XCreatePixmap(d.display,
                                             X11.XDefaultRootWindow(d.display),
                                             width,
                                             height,d.colorDepth);

    IF bitmap(Bitmap).pixmap=0 THEN
      RETURN NIL;
    END;

    bitmap.draw:=factory.CreateDrawInfo();
    bitmap.draw(DrawInfo).Init(bitmap(Bitmap).pixmap);

    bitmap.width:=width;
    bitmap.height:=height;

    RETURN bitmap;
  END CreateBitmap;

  PROCEDURE (d : Display) FreeBitmap*(bitmap : D.Bitmap);

  BEGIN
    X11.XFreePixmap(d.display,bitmap(Bitmap).pixmap);
  END FreeBitmap;

  PROCEDURE (d : Display) Open*():BOOLEAN;

  VAR
    result     : C.charPtr1d;
    i,j        : LONGINT;
    exactColor : X11.XColor;
    error      : BOOLEAN;
    pixel      : ARRAY 1 OF LONGINT;
    font       : D.Font;
    colors     : ARRAY D.colorCount OF X11.XColor;
    prefs      : DisplayPrefs;

    PROCEDURE CreatePixmapCursor(bit1,bit2 : C.address; w,h : LONGINT):X11.Cursor;

    VAR
      pix1,pix2   : X11.Pixmap;
      cursor      : X11.Cursor;
      white,black : X11.XColor;

    BEGIN
      pix1:=X11.XCreatePixmapFromBitmapData(d.display,X11.XRootWindow(d.display,d.scrNum),
                                            bit1,w,h,
                                            X11.XBlackPixel(d.display,d.scrNum),
                                            X11.XWhitePixel(d.display,d.scrNum),1);

      pix2:=X11.XCreatePixmapFromBitmapData(d.display,X11.XRootWindow(d.display,d.scrNum),
                                            bit2,w,h,
                                            X11.XWhitePixel(d.display,d.scrNum),
                                            X11.XBlackPixel(d.display,d.scrNum),1);

      IF (pix1#0) & (pix2#0) THEN
        white.pixel:=D.whiteColor;
        X11.XQueryColor(d.display,d.colorMap,white);
        black.pixel:=D.blackColor;
        X11.XQueryColor(d.display,d.colorMap,black);

        cursor:=X11.XCreatePixmapCursor(d.display,pix1,pix2,
                                        white,
                                        black,
                                        w DIV 2,h DIV 2);
      ELSE
        cursor:=0;
      END;

      IF pix1#0 THEN
        X11.XFreePixmap(d.display,pix1);
      END;
      IF pix2#0 THEN
        X11.XFreePixmap(d.display,pix2);
      END;

      RETURN cursor;
    END CreatePixmapCursor;


  BEGIN
    d.name:="";
    result:=X11.XDisplayName(NIL);
    IF result#NIL THEN
      COPY(result^,d.name);
    END;

    d.display:=X11.XOpenDisplay(d.name);
    IF D.display=NIL THEN
      RETURN FALSE;
    END;

<* IF LIB_HAVE_LIBIMLIB=TRUE THEN *>

    d.imlib:=Imlib.Imlib_init(d.display);

    d.display:=d.imlib.x.display;
    d.scrNum:=d.imlib.x.screen;
    d.visual:=d.imlib.x.visual;
    d.colorDepth:=d.imlib.x.depth;
    d.colorMap:=d.imlib.x.root_cmap;

<* ELSE *>

    d.scrNum:=X11.XDefaultScreen(d.display);
    d.visual:=X11.XDefaultVisual(d.display,d.scrNum);
    IF d.visual=NIL THEN
      Err.String("Cannot get visual!"); Err.Ln;
      RETURN FALSE;
    END;
    d.colorDepth:=X11.XDefaultDepth(d.display,d.scrNum);
    d.colorMap:=X11.XDefaultColormap(d.display,d.scrNum);

<* END *>

    d.scrWidth:=X11.XDisplayWidth(d.display,d.scrNum);
    d.scrHeight:=X11.XDisplayHeight(d.display,d.scrNum);
    
    d.xim:=X11.XOpenIM(d.display,0,"",""); (* TODO: Add d.appName *)
    IF d.xim=NIL THEN
      Err.String("Cannot open im"); Err.Ln;
      RETURN FALSE;
    END;

    d.xic:=X11.XCreateIC(d.xim,
                         X11.XNInputStyle, X11.XIMPreeditNothing+X11.XIMStatusNothing,
                         X11.XNClientWindow, 0,
                         X11.XNFocusWindow, 0,
                         NIL);
    IF d.xic=NIL THEN
      Err.String("Cannot open ic"); Err.Ln;
      RETURN FALSE;
    END;

    IF (d.visual.class=X11.GrayScale)
    OR (d.visual.class=X11.StaticGray) THEN
      IF d.colorDepth=1 THEN
        d.colorMode:=D.monochromeMode;
      ELSE
        d.colorMode:=D.greyScaleMode;
      END;
    ELSIF (d.visual.class=X11.PseudoColor)
    OR (d.visual.class=X11.StaticColor)
    OR (d.visual.class=X11.DirectColor)
    OR (d.visual.class=X11.TrueColor) THEN
      d.colorMode:=D.colorMode;
    ELSE
      Err.String("Unsupported visual class!"); Err.Ln;
      RETURN FALSE;
    END;

    NEW(prefs);
    prefs.Init;
    D.prefs:=prefs;

    IF D.prefsCallback#NIL THEN
      IF d.appName#NIL THEN
        D.prefsCallback.LoadPrefs(d.appName^);
      ELSE
        D.prefsCallback.LoadPrefs("");
      END;
      D.prefsCallback.ReadDisplayPrefs;
    END;

    REPEAT
      error:=FALSE;
      i:=0;
      WHILE (i<D.colorCount) & ~error DO
        IF X11.XLookupColor(d.display,
                            d.colorMap,
                            D.prefs.colors[i],
                            exactColor,
                            colors[i])=0 THEN
          X11.XCloseDisplay(d.display);
          Err.String("Cannot parse color '");
          Err.String(D.prefs.colors[i]); Err.String("'"); Err.Ln;
          error:=TRUE;
        END;
        IF X11.XAllocColor(d.display,d.colorMap,colors[i])=0 THEN
          Err.String("Cannot allocate color '");
          Err.String(D.prefs.colors[i]); Err.String("'"); Err.Ln;
          error:=TRUE;
        END;
        INC(i);
      END;

      IF error THEN
        FOR j:=0 TO i-2 DO
          pixel[0]:=colors[j].pixel;
          X11.XFreeColors(d.display,d.colorMap,pixel,1,0);
        END;
        DEC(d.colorMode);
        IF d.colorMode=D.greyScaleMode THEN
          prefs.EvaluateColor;
        ELSIF d.colorMode=D.monochromeMode THEN
          prefs.EvaluateColor;
        END;
      END;

    UNTIL ~error OR (d.colorMode<0);

    IF d.colorMode<0 THEN
      Err.String("cannot allocate enough colors, giving up"); Err.Ln;
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

    D.backgroundColor       := colors[D.backgroundColorIndex].pixel;
    D.tableBackgroundColor  := colors[D.tableBackgroundColorIndex].pixel;
    D.tableTextColor        := colors[D.tableTextColorIndex].pixel;
    D.textBackgroundColor   := colors[D.textBackgroundColorIndex].pixel;
    D.buttonBackgroundColor := colors[D.buttonBackgroundColorIndex].pixel;
    D.textColor             := colors[D.textColorIndex].pixel;
    D.shineColor            := colors[D.shineColorIndex].pixel;
    D.halfShineColor        := colors[D.halfShineColorIndex].pixel;
    D.halfShadowColor       := colors[D.halfShadowColorIndex].pixel;
    D.shadowColor           := colors[D.shadowColorIndex].pixel;
    D.fillColor             := colors[D.fillColorIndex].pixel;
    D.fillTextColor         := colors[D.fillTextColorIndex].pixel;
    D.warnColor             := colors[D.warnColorIndex].pixel;
    D.disabledColor         := colors[D.disabledColorIndex].pixel;
    D.focusColor            := colors[D.focusColorIndex].pixel;
    D.blackColor            := colors[D.blackColorIndex].pixel;
    D.whiteColor            := colors[D.whiteColorIndex].pixel;
    D.helpBackgroundColor   := colors[D.helpBackgroundColorIndex].pixel;
                
    d.fonts:=NIL;
    FOR i:=0 TO D.fontCount-1 DO
      IF D.prefs.fonts[i]#"" THEN
        font:=factory.CreateFont();
        d.font[i]:=font(Font);
        d.font[i].Init;
        IF ~d.font[i].LoadFont(d,D.prefs.fonts[i],FALSE) THEN
          Err.String("Cannot load font '");
          Err.String(D.prefs.fonts[i]); Err.String("'"); Err.Ln;
          (* TODO: Free colors *)
          X11.XCloseDisplay(d.display);
          RETURN FALSE;
        ELSE
          d.AddFont(d.font[i]);
        END;
      END;

    END;

    D.tinyFont:=d.font[D.tinyFontIndex];
    D.scriptFont:=d.font[D.scriptFontIndex];
    D.footnoteFont:=d.font[D.footnoteFontIndex];
    D.smallFont:=d.font[D.smallFontIndex];
    D.normalFont:=d.font[D.normalFontIndex];
    D.largeFont:=d.font[D.largeFontIndex];
    D.LargeFont:=d.font[D.LargeFontIndex];
    D.LARGEFont:=d.font[D.LARGEFontIndex];
    D.hugeFont:=d.font[D.hugeFontIndex];
    D.HUGEFont:=d.font[D.HUGEFontIndex];
  
    D.smallFixedFont:=d.font[D.smallFixedFontIndex];
    D.fixedFont:=d.font[D.fixedFontIndex];
    D.hugeFixedFont:=d.font[D.hugeFixedFontIndex];

    d.spaceHeight:=(D.fixedFont.height+D.normalFont.height) DIV 4;
    d.spaceWidth:=d.spaceHeight;

    d.deleteProt[0]:=X11.XInternAtom(d.display,"WM_DELETE_WINDOW",X11.True); (* Check result *)
    IF d.deleteProt[0]=X11.None THEN
      Err.String("Window manager does not support WM_DELETE_WINDOW"); Err.Ln;
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

(*    d.dndMessage:=X11.XInternAtom(d.display,"_MOTIF_DRAG_AND_DROP_MESSAGE",X11.False);
    d.dndRecProp:=X11.XInternAtom(d.display,"_MOTIF_DRAG_RECEIVER_INFO",X11.False);
    d.dndIniProp:=X11.XInternAtom(d.display,"_MOTIF_DRAG_INITIATOR_INFO",X11.False);
    d.dndSuccess:=X11.XInternAtom(d.display,"XmTRANSFER_SUCCESS",X11.False);
    d.dndFailure:=X11.XInternAtom(d.display,"XmTRANSFER_FAILURE",X11.False);*)

    (* X11 *)
    d.atom:=X11.XInternAtom(d.display,"ATOM",X11.False);

    (* Xdnd *)
    d.XdndAware:=X11.XInternAtom(d.display,"XdndAware",X11.False);
    d.XdndEnter:=X11.XInternAtom(d.display,"XdndEnter",X11.False);
    d.XdndLeave:=X11.XInternAtom(d.display,"XdndLeave",X11.False);
    d.XdndPosition:=X11.XInternAtom(d.display,"XdndPosition",X11.False);
    d.XdndStatus:=X11.XInternAtom(d.display,"XdndStatus",X11.False);
    d.XdndFinished:=X11.XInternAtom(d.display,"XdndFinished",X11.False);
    d.XdndDrop:=X11.XInternAtom(d.display,"XdndDrop",X11.False);
    d.XdndActionCopy:=X11.XInternAtom(d.display,"XdndActionCopy",X11.False);
    d.XdndActionMove:=X11.XInternAtom(d.display,"XdndActionMove",X11.False);
    d.XdndActionLink:=X11.XInternAtom(d.display,"XdndActionLink",X11.False);
    d.XdndActionAsk:=X11.XInternAtom(d.display,"XdndActionAsk",X11.False);
    d.XdndActionPrivate:=X11.XInternAtom(d.display,"XdndActionPrivate",X11.False);
    d.XdndActionList:=X11.XInternAtom(d.display,"XdndActionList",X11.False);
    d.XdndSelection:=X11.XInternAtom(d.display,"XdndSelection",X11.False);
    d.XdndTypeList:=X11.XInternAtom(d.display,"XdndTypeList",X11.False);

    (* X selection mechzanism *)
    d.xSelection:=X11.XInternAtom(d.display,"_VISUALOBERON_SELECTION_DATA",X11.False);
    d.xDrop:=X11.XInternAtom(d.display,"_VISUALOBERON_DROP_DATA",X11.False);
    d.wmState:=X11.XInternAtom(d.display,"WM_STATE",X11.False);

    d.sleepCursor:=X11.XCreateFontCursor(d.display,sleepCursor);
    IF d.sleepCursor=0 THEN
      Err.String("Cannot get sleep cursor!"); Err.Ln;
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

    d.popCursor:=X11.XCreateFontCursor(d.display,popCursor);
    IF d.popCursor=0 THEN
      Err.String("Cannot get popup cursor!"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

    d.dndCursor:=X11.XCreateFontCursor(d.display,dndCursor);
    IF d.dndCursor=0 THEN
      Err.String("Cannot get drag & drop cursor!"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XFreeCursor(d.display,d.popCursor);
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

    d.copyCursor:=CreatePixmapCursor(s.ADR(copyCursorData),s.ADR(copyMaskData),14,15);
    IF d.copyCursor=0 THEN
      Err.String("Cannot create copy cursor"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XFreeCursor(d.display,d.popCursor);
      X11.XFreeCursor(d.display,d.dndCursor);
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

    d.moveCursor:=CreatePixmapCursor(s.ADR(moveCursorData),s.ADR(moveMaskData),11,13);
    IF d.moveCursor=0 THEN
      Err.String("Cannot create move cursor"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XFreeCursor(d.display,d.popCursor);
      X11.XFreeCursor(d.display,d.dndCursor);
      X11.XFreeCursor(d.display,d.copyCursor);
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

    d.linkCursor:=CreatePixmapCursor(s.ADR(linkCursorData),s.ADR(linkMaskData),15,17);
    IF d.linkCursor=0 THEN
      Err.String("Cannot create move cursor"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XFreeCursor(d.display,d.popCursor);
      X11.XFreeCursor(d.display,d.dndCursor);
      X11.XFreeCursor(d.display,d.copyCursor);
      X11.XFreeCursor(d.display,d.moveCursor);
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;


    d.winList:=NIL;
    d.exit:=TRUE;
    d.currentWin:=NIL;

    d.timeOutList:=NIL;
    d.timeOut:=NIL;
    d.contextHelp:=TRUE;
    
    d.sleepList:=NIL;
    d.channelList:=NIL;

    d.selectObject:=NIL;
    d.querySelectObject:=NIL;

    d.dragging:=FALSE;
    d.dragObject:=NIL;

    d.lastEventIsButtonPress:=FALSE;
    d.lastButton:=-2;
    d.lastBut1Button:=-1;
    d.multiClickTime:=200;

    d.pointArray:=NIL;

    IF D.prefsCallback#NIL THEN
      D.prefsCallback.ReadOtherPrefs;
      D.prefsCallback.FreePrefs;
    END;

    RETURN TRUE;
  END Open;

  PROCEDURE (d : Display) IsDoubleClicked*():BOOLEAN;
  BEGIN
    IF ~d.lastEventIsButtonPress  THEN
      RETURN FALSE;
    END;
    RETURN (d.lastButton = d.lastBut1Button)
          & (d.lastPressTime - d.lastBut1PressTime <= d.multiClickTime);
  END IsDoubleClicked;

  PROCEDURE (d : Display) SetMultiClickTime*(t : LONGINT);
  BEGIN
    IF t>0 THEN d.multiClickTime:=t END;
  END SetMultiClickTime;

  (**
    Flushes all pending events. makes only sense for asnsycronous
    windowing systems like X11.
  *)

  PROCEDURE (d : Display) Flush;

  BEGIN
    X11.XSync(d.display,X11.False);
  END Flush;

  PROCEDURE (d : Display) Beep*;

  BEGIN
    X11.XBell(d.display,100);
  END Beep;

  PROCEDURE (d : Display) RegisterSelection*(object : D.Object;
                                             window : D.Window):BOOLEAN;

  BEGIN
    IF d.selectObject#object THEN

      IF (d.selectObject#NIL) THEN
        d.selectObject.Deselect;
        d.selectObject:=NIL;
      END;

      X11.XSetSelectionOwner(d.display,a.XA_PRIMARY,window(Window).window,X11.CurrentTime);
      d.selectObject:=object;
      RETURN TRUE;
    ELSE
      RETURN TRUE;
    END;
  END RegisterSelection;

  PROCEDURE (d : Display) CancelSelection*;

  BEGIN
    X11.XSetSelectionOwner(d.display,a.XA_PRIMARY,X11.None,X11.CurrentTime);
    ASSERT(d.selectObject#NIL);
    d.selectObject.Deselect;
    d.selectObject:=NIL;
  END CancelSelection;

  PROCEDURE (d : Display) QuerySelection*(window: D.Window;
                                          object: D.Object;
                                          type: LONGINT):BOOLEAN;

  VAR
    propType : X11.Atom;

  BEGIN
    CASE type OF
      D.text: propType:=a.XA_STRING;
    ELSE
      RETURN FALSE;
    END;

    X11.XConvertSelection(d.display,
                          a.XA_PRIMARY,propType,
                          d.xSelection,
                          window(Window).window,
                          X11.CurrentTime);

    d.querySelectObject:=object;
    RETURN TRUE;
  END QuerySelection;

  PROCEDURE (d : Display) PutBackEvent*(event : E.Event; destWin : D.Window);

  VAR
    x11Event : X11.XEvent;

  BEGIN
    EO.GetX11Event(event,x11Event);
    x11Event.xany.window:=destWin(Window).window;
    IF X11.XSendEvent(d.display,destWin(Window).window,X11.True,X11.NoEventMask,
                     s.VAL(X11.XEventPtr,s.ADR(x11Event)))=0 THEN
      Err.String("Cannot resend event!"); Err.Ln;
    END;
  END PutBackEvent;

  PROCEDURE (d : Display) Exit*;

  BEGIN
    ASSERT(~d.exit);
    d.exit:=TRUE;
  END Exit;

  PROCEDURE (d : Display) ReinitWindows*;

  VAR
    win : D.Window;

  BEGIN
    win:=d.winList;
    WHILE win#NIL DO
      win.ReinitWindow;
      win:=win(Window).next;
    END;
  END ReinitWindows;

  PROCEDURE (d : Display) Close*;

  VAR
    w    : D.Window;
    font : Font;

  BEGIN
    w:=d.winList;
    WHILE w#NIL DO
(*
      Err.String("Warning: window ");
      IF w.title#NIL THEN
        Err.String(w.title^);
      END;
      Err.String(" not explicitely closed"); Err.Ln;
*)
      w.Close;
      w:=d.winList;
    END;

    (* Freeing all preferences *)
    P.Free;

    (* Freeing colors *)

(*    IF d.userColor#NIL THEN
      x:=0;
      FOR x:=0 TO LEN(d.userColor^)-1 DO
        (* TODO: Free colors *)
(*        IF d.userColor[x].useCount>0 THEN
          Err.String("Color ");
          Err.LongInt(d.userColor[x].color.red,0);
          Err.String(" ");
          Err.LongInt(d.userColor[x].color.green,0);
          Err.String(" ");
          Err.LongInt(d.userColor[x].color.blue,0);
          Err.String(" ");
          Err.String("not explicitely freed"); Err.Ln;
        END;*)
      END;
    END;
*)
    (* Freeing fonts *)
    font:=d.fonts;
    WHILE font#NIL DO
      font.useCount:=0;
      font.Free(d);
      IF font.next#NIL THEN
        font:=font.next(Font);
      ELSE
        font:=NIL;
      END;  
    END;

    IF d.sleepCursor#0 THEN
      X11.XFreeCursor(d.display,d.sleepCursor);
    END;
                          
    IF d.xic#NIL THEN                    
      X11.XDestroyIC(d.xic);
    END;  
        
    IF d.xim#NIL THEN
      IF X11.XCloseIM(d.xim)#0 THEN END;
    END;  

    IF d.display#NIL THEN
      X11.XCloseDisplay(d.display);
    END;
  END Close;

  PROCEDURE (d : Display) Receive*(message : O.Message);

  BEGIN
    WITH
      message : D.ExitMsg DO
        d.Exit;
    | message : D.TimeOutMsg DO
        IF (d.currentWin#NIL) & (message.timeOut=d.timeOut) THEN
          d.timeOut:=NIL;
          d.currentWin.ContextHelp;
        END;
    ELSE
    END;
  END Receive;

  (**
    Check the list of registered sleeps and send a message for every elapsed
    timeout.
  *)

  PROCEDURE (d : Display) CheckSleeps;

  VAR
    sleep : Sleep;
    msg   : D.SleepMsg;

  BEGIN
    sleep:=d.sleepList;
    WHILE sleep#NIL DO
      NEW(msg);
      msg.sleep:=sleep;
      sleep.object.Receive(msg);
      sleep:=sleep.next;
    END;
  END CheckSleeps;

  (**
    Check the list of registered timeouts and send a message for every elapsed
    timeout.
  *)

  PROCEDURE (d : Display) CheckTimeOuts;

  VAR
    timeOut,
    help     : TimeOut;
    dt       : sc.DateTime;
    time     : t.TimeStamp;

  BEGIN
    sc.GetClock(dt);
    c.SetTimeStamp(dt,time);

    timeOut:=d.timeOutList;
    WHILE timeOut#NIL DO
      IF (timeOut.time.Cmp(time)<0) THEN
        tOutMsg.timeOut:=timeOut;
        help:=timeOut.next;
        d.RemoveTimeOut(timeOut);
        timeOut.object.Receive(tOutMsg);
        timeOut:=help;
      ELSE
        timeOut:=timeOut.next;
      END;
    END;
  END CheckTimeOuts;

  PROCEDURE (d : Display) GetEvent;

  VAR
    e     : X11.XEvent;
    event : E.Event;

  BEGIN
    X11.XNextEvent(d.display,e);

    IF d.timeOut#NIL THEN
      d.RemoveTimeOut(d.timeOut);
    END;
    IF d.contextHelp THEN
      d.timeOut:=d.AddTimeOut(contextTimeOutSec,contextTimeOutMil,d);
    END;

    d.currentWin:=d.GetWindow(e.xany.window);

    IF d.currentWin#NIL THEN

      IF ~(
        (e.type=X11.ButtonPress) & (e.xbutton.state={})
      & (e.xbutton.button=X11.Button3) & (d.currentWin(Window).modalCount=0)
      & d.currentWin.ContextMenu()) THEN
        event:=EO.GetEvent(e,d.xic);
        IF event#NIL THEN
          REPEAT
            event.reUse:=FALSE;
            IF d.currentWin.HandleEvent(event) THEN END;
          UNTIL event.reUse=FALSE;
        END;  
      END;
    END;
  END GetEvent;

  (**
    Check the list of registered file notifier and send a message for every
    notifier matching the file descriptor.
  *)

  PROCEDURE (entry : Channel) SendNotify;

  VAR
    msg : D.ChannelMsg;

  BEGIN
    NEW(msg);
    msg.channel:=entry;
    entry.object.Receive(msg);
    entry:=entry.next;
  END SendNotify;

  (**
    Waits for events on the given filedescriptor with the given timeout.

    Returns TRUE, if the wait exceeds the given timeout, else FALSE, if Wait
    returned because the filedescriptor got available.

    You can overload this method to wait for other events (f.e. other
    filedescriptors like sockets, etc...), too.

    NOTE
    This method must be seen as lowlevel stuff. Not all windowing systems
    or OSs may use filedescriptors. It maybe that ported versions of
    VisualOberon hand other parameters to this method.
  *)

  PROCEDURE (d : Display) Wait(x11FD : C.int; interval : t.Interval):BOOLEAN;

  VAR
    channel : Channel;
    fds     : LI.fd_set;
    timeout : LI.timeval;
    maxFD   : C.int;
    res     : C.int;

  BEGIN
    LOOP
      timeout.tv_sec:=interval.dayInt*(t.msecPerDay DIV t.msecPerSec)+interval.msecInt DIV 1000;
      timeout.tv_usec:=(interval.msecInt MOD 1000)*1000;

      IF (timeout.tv_sec<0) OR (timeout.tv_usec<0) THEN
        timeout.tv_sec:=0;
        timeout.tv_usec:=0;
      END;

      LM.FD_ZERO(fds);
      LM.FD_SET(x11FD,fds);
      maxFD:=x11FD;

      channel:=d.channelList;
      WHILE channel#NIL DO
        IF channel.channel IS p.Channel THEN
          LM.FD_SET(channel.channel(p.Channel).fd,fds);
          IF channel.channel(p.Channel).fd>maxFD THEN
            maxFD:=channel.channel(p.Channel).fd;
          END;
        END;
        channel:=channel.next;
      END;
      res:=LI.select(maxFD+1,s.VAL(LI.fd_setPtr,s.ADR(fds)),NIL,NIL,timeout);

      IF res=0 THEN
        RETURN TRUE;
      ELSIF res=-1 THEN
        Err.String("select: error!"); Err.Ln;
      ELSE
        channel:=d.channelList;
        WHILE channel#NIL DO
          IF channel.channel IS p.Channel THEN
            IF LM.FD_ISSET(channel.channel(p.Channel).fd,fds) THEN
              channel.SendNotify;
            END;
          END;
          channel:=channel.next;
        END;

        IF LM.FD_ISSET(x11FD,fds) THEN
          RETURN FALSE;
        END;
      END;
    END;
  END Wait;

  (**
    Return the next pending future time event. If there is no event pending it
    will return a event 10 seconds in the future.
  *)

  PROCEDURE (d : Display) GetNextTimeOut(VAR interval : t.Interval);

  VAR
    dt   : sc.DateTime;
    time : t.TimeStamp;

  BEGIN
    IF d.timeOutList#NIL THEN

      sc.GetClock(dt);
      c.SetTimeStamp(dt,time);
      d.timeOutList.time.Delta(time,interval);
    ELSE
      t.InitInterval(interval,0,30*1000);
    END;
  END GetNextTimeOut;

  PROCEDURE (d : Display) EventLoop*;

  VAR
    interval : t.Interval;

  BEGIN
    ASSERT(d.exit);

    d.exit:=FALSE;

    LOOP
      IF d.sleepList#NIL THEN
        IF X11.XEventsQueued(d.display,X11.QueuedAlready)=X11.False THEN
          d.CheckSleeps;
          d.CheckTimeOuts;
        ELSE
          d.GetEvent;
        END;

      ELSE
        IF X11.XEventsQueued(d.display,X11.QueuedAlready)=X11.False THEN

          IF X11.XEventsQueued(d.display,X11.QueuedAfterFlush)=X11.False THEN
            d.CheckTimeOuts;
            d.GetNextTimeOut(interval);
            IF d.Wait(d.display.fd,interval) THEN
              d.CheckTimeOuts;
            END;
          ELSE
            d.CheckTimeOuts;
          END;
        ELSE
          d.CheckTimeOuts;
        END;
      END;

      IF X11.XEventsQueued(d.display,X11.QueuedAfterReading)#X11.False THEN

        d.GetEvent;

        d.CheckTimeOuts;

      END;

      IF d.exit THEN
        EXIT;
      END;

    END;
  END EventLoop;

  PROCEDURE (w : Window) Init*;

  BEGIN
    w.Init^;

    w.modalCount:=0;

    w.last:=NIL;
    w.next:=NIL;

    w.window:=0;

    w.exposed:=FALSE;
    w.grab:=FALSE;
    w.exit:=TRUE;
  END Init;

  PROCEDURE (w : Window) SetTitle*(name : ARRAY OF CHAR);

  BEGIN
    w.SetTitle^(name);

    IF w.IsOpen() THEN
      X11.XStoreName(D.display(Display).display,w.window,name);
    END;
  END SetTitle;

  PROCEDURE (w : Window) GetDrawInfo*():D.DrawInfo;

  BEGIN
    RETURN w.draw;
  END GetDrawInfo;

  PROCEDURE (w : Window) Resize*(width,height : LONGINT);

  BEGIN
    w.Resize^(width,height);

    X11.XResizeWindow(D.display(Display).display,w.window,width,height);
  END Resize;

  (**
    Grabs the mouse cursor and keyboard.
  *)

  PROCEDURE (w : Window) GrabOn;

  BEGIN
    D.display(Display).StopContextHelp;
    IF X11.XGrabPointer(D.display(Display).display,w.window,X11.False,
                         X11.ButtonPressMask
                        +X11.ButtonReleaseMask
                        +X11.PointerMotionMask
                        +X11.PointerMotionHintMask
                        +X11.LeaveWindowMask
                        +X11.EnterWindowMask
                        +X11.StructureNotifyMask
                        +X11.VisibilityChangeMask
                        +X11.FocusChangeMask
                        +X11.ButtonMotionMask,
                        X11.GrabModeAsync,
                        X11.GrabModeAsync,
                        X11.None,
                        D.display(Display).popCursor,
                        X11.CurrentTime)#X11.GrabSuccess THEN
      Err.String("Can't grab cursor"); Err.Ln;
    END;
    IF X11.XGrabKeyboard(D.display(Display).display,w.window,X11.False,
                         X11.GrabModeAsync,
                         X11.GrabModeAsync,
                         X11.CurrentTime)#X11.GrabSuccess THEN
      Err.String("Can't grab cursor"); Err.Ln;
    END;
    X11.XSetInputFocus(D.display(Display).display,w.window,X11.RevertToParent,X11.CurrentTime);
  END GrabOn;

  (**
    Releases the grab of the mouse cursor and the keyboard.
  *)

  PROCEDURE (w : Window) GrabOff;

  BEGIN
    X11.XUngrabPointer(D.display(Display).display,X11.CurrentTime);
    X11.XUngrabKeyboard(D.display(Display).display,X11.CurrentTime);
    D.display(Display).RestartContextHelp;
  END GrabOff;

  PROCEDURE (w : Window) Grab*(grab : BOOLEAN);

  BEGIN
    IF w.grab=grab THEN
      RETURN;
    END;

    IF ~w.IsOpen() THEN
      w.grab:=grab;
    ELSE
      IF grab THEN
        w.GrabOn;
      ELSE
        w.GrabOff;
      END;
      w.grab:=grab;
    END;
  END Grab;

  PROCEDURE (w : Window) InternalOpen;

  VAR
    sHints  : xu.XSizeHintsPtr;
    wHints  : xu.XWMHintsPtr;
    cHints  : xu.XClassHintPtr;
    wName,
    iName   : xu.XTextProperty;
    tName   : POINTER TO ARRAY OF POINTER TO  ARRAY OF CHAR;
    attr    : X11.XSetWindowAttributes;
    window  : D.Window;
    parent  : X11.Window;
    info    : LONGINT;
    rInfo   :  ARRAY SIZE(LONGINT) OF CHAR;
    draw    : D.DrawInfo;

  BEGIN
    sHints:=xu.XAllocSizeHints();
    wHints:=xu.XAllocWMHints();
    cHints:=xu.XAllocClassHint();

    IF (sHints=NIL) OR (wHints=NIL) OR (cHints=NIL) THEN
      XFree(sHints);
      XFree(wHints);
      XFree(cHints);
    END;

(*    attr.background_pixmap:=X11.None;
    attr.background_pixel:=D.display(Display).GetX11Color(w.background);*)
    attr.border_pixel:=w.background;
  (*  attr.backing_store:=X11.NotUseful; (* Give a little boost *)*)
    attr.bit_gravity:=X11.NorthWestGravity;
    attr.event_mask:= X11.KeyPressMask
                     +X11.KeyReleaseMask
                     +X11.ExposureMask
                     +X11.StructureNotifyMask
                     +X11.ButtonPressMask
                     +X11.ButtonReleaseMask
                     +X11.ButtonMotionMask
                     +X11.PointerMotionMask
                     +X11.FocusChangeMask;

    IF w.borderless THEN
      attr.override_redirect:=X11.True;
      attr.save_under:=X11.True;
    ELSE
      attr.override_redirect:=X11.False;
      attr.save_under:=X11.False;
    END;

    CASE w.horizontalPos OF
      D.centerOnParent:
        IF w.parent#NIL THEN
          w.x:=w.parent.x+(w.parent.width-w.width) DIV 2
        END;
    | D.osPos,
      D.centerOnScreen:
        w.x:=(D.display.scrWidth-w.width) DIV 2;
    ELSE
    END;

    CASE w.verticalPos OF
      D.centerOnParent:
        IF w.parent#NIL THEN
          w.y:=w.parent.y+(w.parent.height-w.height) DIV 2
        END;
    | D.osPos,
      D.centerOnScreen:
        w.y:=(D.display.scrHeight-w.height) DIV 2;
    ELSE
    END;

(*    IF w.parent=NIL THEN*)
      parent:=X11.XRootWindow(D.display(Display).display,
                              D.display(Display).scrNum);
(*    ELSE
      parent:=w.parent(Window).window;
    END;*)

    w.window:=X11.XCreateWindow(D.display(Display).display,
                                parent,
                                w.x,w.y,
                                w.width,w.height,
                                0,
                                X11.CopyFromParent,
                                X11.InputOutput,
                                D.display(Display).visual^,
(*                                 X11.CWBackPixmap
                                +X11.CWBackPixel*)
                                +X11.CWBorderPixel
                                +X11.CWBitGravity
(*                                +X11.CWBackingStore*)
                                +X11.CWOverrideRedirect
                                +X11.CWSaveUnder
                                +X11.CWEventMask,
                                attr);

    IF w.window=0 THEN
      XFree(sHints);
      XFree(wHints);
      XFree(cHints);
    END;

    sHints.flags:=xu.PMinSize+xu.PMaxSize;

    sHints.min_width:=w.minWidth;
    sHints.max_width:=w.maxWidth;

    sHints.min_height:=w.minHeight;
    sHints.max_height:=w.maxHeight;

    NEW(tName,2);
    NEW(tName[0],str.Length(w.title^)+1);
    COPY(w.title^,tName[0]^);
    tName[1]:=NIL;
    IF xu.XStringListToTextProperty(s.VAL(C.charPtr2d,tName),1,wName)=0 THEN END;
    IF xu.XStringListToTextProperty(s.VAL(C.charPtr2d,tName),1,iName)=0 THEN END;

    wHints.initial_state:=xu.NormalState;
    wHints.input:=1;
    wHints.flags:=xu.StateHint+xu.InputHint+xu.WindowGroupHint;

    window:=w.GetTopWindow();
    IF window#NIL THEN
      wHints.window_group:=window(Window).window;
    ELSE
      wHints.window_group:=w.window;
    END;

    IF window#NIL THEN
      X11.XSetTransientForHint(D.display(Display).display,
                               w.window,
                               window(Window).window);
    END;

    cHints.res_name:=s.VAL(C.charPtr1d,w.title);
    cHints.res_class:=s.VAL(C.charPtr1d,w.title);

    xu.XSetWMProperties(D.display(Display).display,
                        w.window,
                        s.VAL(xu.XTextPropertyPtr,s.ADR(wName)),
                        s.VAL(xu.XTextPropertyPtr,s.ADR(iName)),
                        NIL,0,
                        sHints,
                        wHints,
                        cHints);

    XFree(sHints);
    XFree(wHints);
    XFree(cHints);

    IF X11.XSetWMProtocols(D.display(Display).display,
                           w.window,
                           D.display(Display).deleteProt,1)=0 THEN
      X11.XDestroyWindow(D.display(Display).display,w.window);
      XFree(sHints);
      XFree(wHints);
      XFree(cHints);
    END;

    (* We mark the windows as Xdnd aware *)

    info:=XDND_VERSION;
    s.MOVE(s.ADR(info),s.ADR(rInfo),SIZE(LONGINT));

    X11.XChangeProperty(D.display(Display).display,
                        w.window,
                        D.display(Display).XdndAware,
                        D.display(Display).atom,32,
                        X11.PropModeReplace,
                        rInfo,
                        1);

    draw:=factory.CreateDrawInfo();
    w.draw:=draw(DrawInfo);
    w.draw.Init(w.window);

    w.draw.InstallClip(0,0,D.display.scrWidth,D.display.scrHeight);

    D.display(Display).AddWindow(w);
  END InternalOpen;

  PROCEDURE (w : Window) EnableParents;

  VAR
   window : D.Window;

  BEGIN
    window:=w.parent;
    WHILE window#NIL DO
      DEC(window(Window).modalCount);
      IF window(Window).modalCount=0 THEN
        X11.XUndefineCursor(D.display(Display).display,window(Window).window);
      END;
      window:=window.parent;
    END;
  END EnableParents;

  PROCEDURE (w : Window) DisableParents;

  VAR
   window : D.Window;

  BEGIN
    window:=w.parent;
    WHILE window#NIL DO
      INC(window(Window).modalCount);
      IF window(Window).modalCount=1 THEN
        X11.XDefineCursor(D.display(Display).display,
                          window(Window).window,
                          D.display(Display).sleepCursor);
      END;
      window:=window.parent;
    END;
  END DisableParents;

  PROCEDURE (w : Window) Open*;

  BEGIN
    w.Open^;

    w.exposed:=FALSE;
    w.modalCount:=0;

    w.InternalOpen;

    X11.XMapWindow(D.display(Display).display,w.window);

    IF w.grab THEN
      w.GrabOn;
    END;
  END Open;

  PROCEDURE (w : Window) Close*;

  BEGIN
    IF w.grab THEN
      w.GrabOff;
    END;

    IF w.IsOpen() & (w.window#0) THEN
      X11.XUnmapWindow(D.display(Display).display,w.window);
    END;

    w.modalCount:=-1;

    IF w.window#0 THEN
      w.draw.FreeLastClip;
      w.draw.Deinit;
      X11.XDestroyWindow(D.display(Display).display,w.window);
      w.window:=0;
    END;

    D.display(Display).RemoveWindow(w);

    w.Close^;
  END Close;

  PROCEDURE (w : Window) GetXY*(VAR x,y : LONGINT);

  VAR
    retAttr : X11.XWindowAttributes;
    root,
    parent,
    current : X11.Window;
    childs  : X11.WindowPtr1d;
    count   : LONGINT;

  BEGIN
    x:=0;
    y:=0;

    current:=w.window;

    IF X11.XQueryTree(D.display(Display).display,
                      current,root,parent,childs,count)#0 THEN
      XFree(childs);
      WHILE current#root DO
        IF X11.XGetWindowAttributes(D.display(Display).display,current,retAttr)#0 THEN
          INC(x,retAttr.x+retAttr.border_width);
          INC(y,retAttr.y+retAttr.border_width);
        END;
        current:=parent;
        IF X11.XQueryTree(D.display(Display).display,current,root,parent,childs,count)#0 THEN
          XFree(childs);
        END;
      END;
    END;
  END GetXY;

  PROCEDURE (w : Window) Maped*;

  BEGIN
    w.Maped^;

    w.GetXY(w.x,w.y);
  END Maped;

  PROCEDURE (w : Window) GetMousePos*(VAR rx, ry, wx, wy : LONGINT);

  VAR
    root,
    child  : X11.Window;
    bmask  : X11.ulongmask;

  BEGIN
    IF X11.XQueryPointer(D.display(Display).display,w.window,root,child,rx,ry,wx,wy,bmask)=X11.False THEN
      Err.String("Can't get cursorpos"); Err.Ln;
    END;
  END GetMousePos;

  PROCEDURE (w : Window) FocusIn*;

  BEGIN
    w.FocusIn^;

    IF ~w.grab THEN
      D.display(Display).StartContextHelp;
    END;
  END FocusIn;

  PROCEDURE (w : Window) FocusOut*;

  BEGIN
    w.FocusOut^;

    IF ~w.grab THEN
      D.display(Display).StopContextHelp;
    END;
  END FocusOut;

  (**
    Called when some drag action should be started.
  *)

  PROCEDURE (w : Window) HandleDrag(event : E.MotionEvent):BOOLEAN;

  VAR
    x11Event : X11.XEvent;

  BEGIN
    D.display(Display).dragStart:=FALSE;

    IF w.modalCount=0 THEN
      D.display(Display).dragObject:=w.GetDnDObject(D.display(Display).dragX,
                                                      D.display(Display).dragY,TRUE);
      IF D.display(Display).dragObject#NIL THEN
        NEW(D.display(Display).dragInfo);
        D.display(Display).dragInfo.Init;
        D.display(Display).dragObject.GetDragInfo(D.display(Display).dragInfo);

        D.display(Display).StopContextHelp;

        (* Calculation action *)
        IF event.qualifier*E.keyMask=E.shiftMask THEN
          D.display(Display).dndAction:=DD.move;
          X11.XDefineCursor(D.display(Display).display,
                            w.window,
                            D.display(Display).moveCursor);
        ELSIF event.qualifier*E.keyMask=E.controlMask THEN
          D.display(Display).dndAction:=DD.copy;
          X11.XDefineCursor(D.display(Display).display,
                            w.window,
                            D.display(Display).copyCursor);
        ELSIF event.qualifier*E.keyMask=E.shiftMask+E.controlMask THEN
          D.display(Display).dndAction:=DD.link;
          X11.XDefineCursor(D.display(Display).display,
                            w.window,
                            D.display(Display).linkCursor);
        ELSE
          D.display(Display).dndAction:=DD.default;
          X11.XDefineCursor(D.display(Display).display,
                            w.window,
                            D.display(Display).dndCursor);
        END;

        EO.GetX11Event(event,x11Event);
        D.display(Display).dropWindow:=x11Event.xmotion.window;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleDrag;

  (**
    Called when a drop action should be handled.
  *)

  PROCEDURE (w : Window) HandleDragDrop(event : E.Event):BOOLEAN;

  VAR
    dragWin    : X11.Window;
    rX,rY,
    cX,cY      : LONGINT;
    return     : BOOLEAN;
    window     : D.Window;
    dropObject : D.Object;
    dragData   : DD.DnDData;
    group,type : LONGINT;

  BEGIN
    return:=FALSE;
    D.display(Display).dragStart:=FALSE;

    IF (D.display(Display).dropWindow#0) THEN

      D.display(Display).RestartContextHelp;

      dragWin:=D.display(Display).GetWindowOnScreen(rX,rY,cX,cY);
      IF dragWin#0 THEN
        window:=D.display(Display).GetWindow(dragWin);
        IF window#NIL THEN
          dropObject:=window.GetDnDObject(cX,cY,FALSE);
          IF (window(Window).modalCount=0)
          & (dropObject#NIL) & (dropObject#D.display(Display).dragObject) THEN
            IF dropObject.GetDropDataType(D.display(Display).dragInfo,
                                          group,type,D.display(Display).dndAction) THEN
              dragData:=D.display(Display).dragObject.GetDragData(group,type,D.display(Display).dndAction);
              IF dragData#NIL THEN
                IF dropObject.HandleDrop(dragData,D.display(Display).dndAction) THEN
                  (* nothing to do yet*)
                ELSE
                  X11.XBell(D.display(Display).display,100);
                END;
              END;
            END;
            return:=TRUE;
          END;
        ELSE
          Err.String("End external drag -> drop"); Err.Ln;
          (*w.HandleMotifDragDrop(dragWin,rX,rY,event.event.xbutton);*)
        END;
      END;
      X11.XUndefineCursor(D.display(Display).display,w.window);
      D.display(Display).dropWindow:=0;
      D.display(Display).dragObject:=NIL;
    END;

    RETURN return;
  END HandleDragDrop;

  (**
    Handle the XSelecitionNotify of X11. A SelectionNotify gets send
    when someone made a request for the selection value. The event states
    where one can get the selection value from.
  *)

  PROCEDURE (w : Window) HandleXSelectionNotify(event : X11.XSelectionEvent);

  VAR
    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    data        : C.charPtr1d;
    dndData     : DD.DnDStringData;
    queryObject : D.Object;
    text        : U.Text;

  BEGIN
    queryObject:=D.display(Display).querySelectObject;
    D.display(Display).querySelectObject:=NIL;

    IF event.property=X11.None THEN
      Out.String("No property"); Out.Ln;
      RETURN;
    END;

    IF X11.XGetWindowProperty(D.display(Display).display,
                              event.requestor,event.property,0,MAX(LONGINT),X11.True,
                              X11.AnyPropertyType,
                              retType,retFormat,itemsReturn,bytesLeft,
                              data)#X11.Success THEN
      Out.String("Cannot get property data"); Out.Ln;
      RETURN;
    END;

    IF retType=X11.None THEN
      Out.String("Illegal property data type"); Out.Ln;
      RETURN;
    END;

    IF event.property=D.display(Display).xSelection THEN
      IF (retType=a.XA_STRING) & (retFormat=8) THEN
        NEW(dndData);
        NEW(dndData.string,itemsReturn+1);
        s.MOVE(s.VAL(LONGINT,data),
               s.VAL(LONGINT,dndData.string),itemsReturn);
        dndData.string[itemsReturn]:=0X;

        IF queryObject#NIL THEN
          IF queryObject.HandleDrop(dndData,DD.insert) THEN
            (* nothing to do *)
          ELSE
            X11.XBell(D.display(Display).display,100);
          END;
        END;
      END;
    ELSE
      Out.String("Drop result: ");
      text:=U.CStringToText(data);
      Out.String(text^);
      Out.Ln;
    END;
  END HandleXSelectionNotify;

  (**
    handle XSelectionrequest of X11. A SelectionRequest gets send when some application
    wants the selection value and your window has registered the selection. We
    ask the object which has registered the selection for the selection value
    and return a notify message to the requestor.
  *)

  PROCEDURE (w : Window) HandleXSelectionRequest(event : X11.XSelectionRequestEvent);

  VAR
    data        : DD.DnDData;
    selNotify   : X11.XEvent;
    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    string      : C.charPtr1d;

  BEGIN
    selNotify.xselection.type:=X11.SelectionNotify;
    selNotify.xselection.display:=event.display;
    selNotify.xselection.requestor:=event.requestor;
    selNotify.xselection.selection:=event.selection;
    selNotify.xselection.target:=event.target;
    selNotify.xselection.property:=event.property;
    selNotify.xselection.time:=event.time;


    IF (event.target=a.XA_STRING) & (D.display(Display).selectObject#NIL) THEN
      data:=D.display(Display).selectObject.GetDragData(D.text,D.none,D.copy);
      IF (data#NIL) & (data IS DD.DnDStringData) THEN
        string:=s.VAL(C.charPtr1d,data(DD.DnDStringData).string);
        IF X11.XGetWindowProperty(event.display,
                                  event.requestor,event.property,0,MAX(LONGINT),X11.False,
                                  event.target,retType,retFormat,itemsReturn,bytesLeft,
                                  string)#X11.Success THEN
            Err.String("Cannot get property"); Err.Ln;
            selNotify.xselection.property:=X11.None;
        ELSE
          X11.XChangeProperty(event.display,event.requestor,event.property,
                              a.XA_STRING,8,
                              X11.PropModeReplace,data(DD.DnDStringData).string^,
                              str.Length(data(DD.DnDStringData).string^));
        END;
      ELSE
        selNotify.xselection.property:=X11.None;
      END;
    ELSE
      selNotify.xselection.property:=X11.None;
    END;

    IF X11.XSendEvent(selNotify.xselection.display,
                      selNotify.xselection.requestor,X11.True,X11.NoEventMask,
                      s.VAL(X11.XEventPtr,s.ADR(selNotify)))=0 THEN
      Err.String("Error sending selection notify"); Err.Ln;
    END;
  END HandleXSelectionRequest;

  PROCEDURE (w : Window) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    x,y,
    width,
    height      : LONGINT;
    name        : C.charPtr1d;
    text        : U.Text;
    msg         : X11.XClientMessageEvent;

    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    xData       : C.charPtr1d;
    data        : POINTER TO ARRAY OF LONGINT;
    x11Event    : X11.XEvent;

  BEGIN
    D.display(Display).lastEventIsButtonPress:=FALSE;

    EO.GetX11Event(event,x11Event);

    CASE x11Event.type OF
      X11.ClientMessage:
        IF (x11Event.xclient.data.l[0]=D.display(Display).deleteProt[0]) THEN
        END;
        IF (x11Event.xclient.data.l[0]=D.display(Display).deleteProt[0])
        & (w.modalCount=0) THEN
          w.ClosePressed;
        ELSIF (x11Event.xclient.message_type=D.display(Display).XdndEnter) THEN
          Out.String("XDND Enter"); Out.Ln;
          Out.String("XID: "); Out.Hex(x11Event.xclient.data.l[0],0); Out.Ln;
          Out.String("Protokoll: "); Out.LongInt(s.VAL(SHORTINT,x11Event.xclient.data.b[7]),0); Out.Ln;

          FOR x:=2 TO 4 DO
            name:=X11.XGetAtomName(D.display(Display).display,x11Event.xclient.data.l[x]);
            IF name#NIL THEN
              Out.String("Datatype: ");
              text:=U.CStringToText(name);
              Out.String(text^); Out.Ln;
            END;
          END;

          IF x11Event.xclient.data.b[4]#0X THEN (* TODO: fix *)
            IF X11.XGetWindowProperty(D.display(Display).display,
                                      x11Event.xclient.data.l[0],
                                      D.display(Display).XdndTypeList,
                                      0,
                                      MAX(LONGINT),
                                      X11.False,
                                      D.display(Display).atom,
                                      retType,
                                      retFormat,
                                      itemsReturn,
                                      bytesLeft,
                                      xData)#X11.Success THEN
              Err.String("Cannot get property"); Err.Ln;
            ELSE
              NEW(data,itemsReturn);
              s.MOVE(s.VAL(LONGINT,xData),s.VAL(LONGINT,data),SIZE(LONGINT)*itemsReturn);
              XFree(xData);
              FOR x:=0 TO itemsReturn-1 DO
                name:=X11.XGetAtomName(D.display(Display).display,data[x]);
                IF name#NIL THEN
                  Out.String("Datatype: ");
                  text:=U.CStringToText(name);
                  Out.String(text^); Out.Ln;
                END;
              END;
            END;
          END;

          Out.Ln;

        ELSIF (x11Event.xclient.message_type=D.display(Display).XdndLeave) THEN
          Out.String("XDND Leave"); Out.Ln;
          Out.String("XID: "); Out.Hex(x11Event.xclient.data.l[0],0); Out.Ln;
          Out.Ln;

        ELSIF (x11Event.xclient.message_type=D.display(Display).XdndPosition) THEN
          Out.String("XDND Position"); Out.Ln;
          Out.String("XID: "); Out.Hex(x11Event.xclient.data.l[0],0); Out.Ln;
          Out.String("X: "); Out.LongInt(x11Event.xclient.data.s[5],0); Out.Ln;
          Out.String("Y: "); Out.LongInt(x11Event.xclient.data.s[4],0); Out.Ln;

          IF x11Event.xclient.data.l[4]=D.display(Display).XdndActionMove THEN
            Out.String("action move"); Out.Ln;
          ELSIF x11Event.xclient.data.l[4]=D.display(Display).XdndActionCopy THEN
            Out.String("action copy"); Out.Ln;
          ELSIF x11Event.xclient.data.l[4]=D.display(Display).XdndActionLink THEN
            Out.String("action link"); Out.Ln;
          END;
          Out.Ln;

          msg.type:=X11.ClientMessage;
          msg.format:=32;
          msg.display:=D.display(Display).display;
          msg.window:=x11Event.xclient.data.l[0];
          msg.message_type:=D.display(Display).XdndStatus;
          msg.data.l[0]:=w.window;
          msg.data.l[1]:=3;
          msg.data.s[4]:=0; (* y *)
          msg.data.s[5]:=0; (* x *)
          msg.data.s[6]:=0; (* h *)
          msg.data.s[7]:=0; (* w *)
          msg.data.l[4]:=D.display(Display).XdndActionCopy;

          IF X11.XSendEvent(D.display(Display).display,
                            x11Event.xclient.data.l[0],
                            X11.False,X11.NoEventMask,
                            s.VAL(X11.XEventPtr,s.ADR(msg)))=0 THEN
            Out.String("Could not send XDNDStatus message"); Out.Ln;
          END;


        ELSIF (x11Event.xclient.message_type=D.display(Display).XdndStatus) THEN
          Out.String("XDND Status"); Out.Ln;

        ELSIF (x11Event.xclient.message_type=D.display(Display).XdndDrop) THEN
          Out.String("XDND Drop"); Out.Ln;

          msg.type:=X11.ClientMessage;
          msg.format:=32;
          msg.display:=D.display(Display).display;
          msg.window:=w.window;
          msg.message_type:=D.display(Display).XdndFinished;
          msg.data.l[0]:=w.window;
          msg.data.l[1]:=0;

          X11.XConvertSelection(D.display(Display).display,
                                D.display(Display).XdndSelection,
                                297, (* text/plain *)
                                D.display(Display).xDrop,
                                w.window,
                                x11Event.xclient.data.l[2]);

(*          dropObject:=window.GetDnDObject(cX,cY,FALSE);
                IF dropObject.HandleDrop(dragData,D.display.dndAction) THEN*)

(*          IF X11.XSendEvent(D.display.display,
                            x11Event.xclient.data.l[0],
                            X11.False,X11.NoEventMask,
                            s.VAL(X11.XEventPtr,s.ADR(msg)))=0 THEN
            Out.String("Could not send XDNDStatus message"); Out.Ln;
          END;*)

        ELSIF (x11Event.xclient.message_type=D.display(Display).XdndFinished) THEN
          Out.String("XDND Finished"); Out.Ln;
          Out.String("XID: "); Out.Hex(x11Event.xclient.data.l[0],0); Out.Ln;
          Out.Ln;
        ELSE
          Out.String("Unknown client message: ");
          Out.LongInt(x11Event.xclient.message_type,0); Out.Ln;
        END;
    | X11.ConfigureNotify:
        w.GetXY(w.x,w.y);
        IF (x11Event.xconfigure.width#w.width)
        OR (x11Event.xconfigure.height#w.height) THEN
          w.width:=x11Event.xconfigure.width;  (* Where get they set the first time? *)
          w.height:=x11Event.xconfigure.height;
          w.Resized(x11Event.xconfigure.width -x11Event.xconfigure.border_width,
                    x11Event.xconfigure.height-x11Event.xconfigure.border_width);
        END;
    | X11.Expose,
      X11.GraphicsExpose:
        IF ((x11Event.type=X11.Expose) & (x11Event.xexpose.count=0))
        OR ((x11Event.type=X11.GraphicsExpose) & (x11Event.xgraphicsexpose.count=0)) THEN
          IF ~w.exposed THEN
            w.draw.InstallClip(x11Event.xexpose.x,x11Event.xexpose.y,
                               x11Event.xexpose.width,x11Event.xexpose.height);
            w.exposed:=TRUE;
          ELSE
            w.draw.AddRegion(x11Event.xexpose.x,x11Event.xexpose.y,
                             x11Event.xexpose.width,x11Event.xexpose.height);
          END;
          w.draw.GetClipRegion(x,y,width,height);
          w.Redraw(x,y,width,height);
          w.draw.FreeLastClip;
          w.exposed:=FALSE;
        ELSE
          IF ~w.exposed THEN
            w.exposed:=TRUE;
            w.draw.InstallClip(x11Event.xexpose.x,x11Event.xexpose.y,
                               x11Event.xexpose.width,x11Event.xexpose.height);
          ELSE
            w.draw.AddRegion(x11Event.xexpose.x,x11Event.xexpose.y,
                             x11Event.xexpose.width,x11Event.xexpose.height);
          END;
        END;
    | X11.CreateNotify:
    | X11.ReparentNotify:
    | X11.MappingNotify:
        X11.XRefreshKeyboardMapping(s.VAL(X11.XMappingEventPtr,s.ADR(x11Event)));
    | X11.UnmapNotify:
        w.Unmaped;
    | X11.MapNotify:
        w.Maped;
    | X11.FocusIn:
        X11.XSetICFocus(D.display(Display).xic);
        w.FocusIn;
    | X11.FocusOut:
        w.FocusOut;
        X11.XUnsetICFocus(D.display(Display).xic);
    | X11.EnterNotify:
        (* TODO w.Entered *)
    | X11.LeaveNotify:
        w.Left;
    | X11.VisibilityNotify:
        IF x11Event.xvisibility.state>0 THEN
          w.Hidden;
        END;
    | X11.SelectionNotify:
        Out.String("SelectionNotify"); Out.Ln;
        w.HandleXSelectionNotify(x11Event.xselection);
    | X11.SelectionClear:
        IF D.display(Display).selectObject#NIL THEN
          D.display(Display).selectObject.Deselect;
          D.display(Display).selectObject:=NIL;
        END;
    | X11.SelectionRequest:
        w.HandleXSelectionRequest(x11Event.xselectionrequest);
    | X11.ButtonPress:
        IF (x11Event.xbutton.button=E.dragDropButton) THEN
          D.display(Display).dragStart:=TRUE;
          D.display(Display).dragX:=x11Event.xbutton.x;
          D.display(Display).dragY:=x11Event.xbutton.y;
        END;
        D.display(Display).lastEventIsButtonPress:=TRUE;
        D.display(Display).lastBut1Button:=D.display(Display).lastButton;
        D.display(Display).lastBut1PressTime:=D.display(Display).lastPressTime;
        D.display(Display).lastButton:=SHORT(SHORT(x11Event.xbutton.button));
        D.display(Display).lastPressTime:=x11Event.xbutton.time;
        IF w.modalCount=0 THEN
          RETURN FALSE;
        ELSE
          RETURN TRUE;
        END;
    | X11.ButtonRelease:
        IF (x11Event.xbutton.type=X11.ButtonRelease)
        & (x11Event.xbutton.button=E.dragDropButton) THEN
          IF w.HandleDragDrop(event) THEN
            RETURN TRUE;
          END;
        END;
        IF w.modalCount=0 THEN
          RETURN FALSE;
        ELSE
          RETURN TRUE;
        END;
    | X11.MotionNotify:
        IF (event(E.MotionEvent).qualifier*E.buttonMask={E.dragDropButton})
         & (D.display(Display).dragStart) THEN
          IF   (x11Event.xmotion.x>D.display(Display).dragX+D.display.spaceWidth DIV 2)
            OR (x11Event.xmotion.x<D.display(Display).dragX-D.display.spaceWidth DIV 2)
            OR (x11Event.xmotion.y>D.display(Display).dragY+D.display.spaceHeight DIV 2)
            OR (x11Event.xmotion.y<D.display(Display).dragY-D.display.spaceHeight DIV 2) THEN
            IF w.HandleDrag(event(E.MotionEvent)) THEN
              RETURN TRUE;
            END;
          END;
        END;
        IF w.modalCount=0 THEN
          RETURN FALSE;
        ELSE
          RETURN TRUE;
        END;
    ELSE
      IF w.modalCount=0 THEN
        RETURN FALSE;
      ELSE
        RETURN TRUE;
      END;
    END;
    RETURN TRUE;
  END HandleEvent;

  PROCEDURE (w : Window) EventLoop*;

  VAR
    interval : t.Interval;

  BEGIN
    ASSERT(w.exit);

    w.exit:=FALSE;

    w.DisableParents;

    LOOP
      IF D.display(Display).sleepList#NIL THEN
        IF X11.XEventsQueued(D.display(Display).display,X11.QueuedAlready)=X11.False THEN
          D.display(Display).CheckSleeps;
          D.display(Display).CheckTimeOuts;
        ELSE
          D.display(Display).GetEvent;
        END;

      ELSE
        IF X11.XEventsQueued(D.display(Display).display,X11.QueuedAlready)=X11.False THEN

          IF X11.XEventsQueued(D.display(Display).display,X11.QueuedAfterFlush)=X11.False THEN
            D.display(Display).CheckTimeOuts;
            D.display(Display).GetNextTimeOut(interval);
            IF D.display(Display).Wait(D.display(Display).display.fd,interval) THEN
              D.display(Display).CheckTimeOuts;
            END;
          ELSE
            D.display(Display).CheckTimeOuts;
          END;
        ELSE
          D.display(Display).CheckTimeOuts;
        END;
      END;

      IF X11.XEventsQueued(D.display(Display).display,X11.QueuedAfterReading)#X11.False THEN

        D.display(Display).GetEvent;

        D.display(Display).CheckTimeOuts;

      END;
      IF w.exit THEN
        EXIT;
      END;
    END;

    w.EnableParents;
  END EventLoop;

  PROCEDURE (w : Window) IsInEventLoop*():BOOLEAN;

  BEGIN
    RETURN ~w.exit;
  END IsInEventLoop;

  PROCEDURE (w : Window) Exit*;

  BEGIN
    ASSERT(~w.exit);

    w.exit:=TRUE;
  END Exit;

  (* ------------ object factory --------------- *)

  PROCEDURE (f : FactoryImpl) CreateFont*():D.Font;

  VAR
    font : Font;

  BEGIN
    NEW(font);

    RETURN font;
  END CreateFont;

  PROCEDURE (f : FactoryImpl) CreateDrawInfo*():D.DrawInfo;

  VAR
    draw : DrawInfo;

  BEGIN
    NEW(draw);

    RETURN draw;
  END CreateDrawInfo;

  PROCEDURE (f : FactoryImpl) CreateDisplay*():D.Display;

  VAR
    display : Display;

  BEGIN
    NEW(display);

    RETURN display;
  END CreateDisplay;

  PROCEDURE (f : FactoryImpl) CreateBitmap*():D.Bitmap;

  VAR
    bitmap : Bitmap;

  BEGIN
    NEW(bitmap);

    RETURN bitmap;
  END CreateBitmap;

BEGIN
  NEW(tOutMsg);

  NEW(factory);
  D.SetFactory(factory);

  copyCursorData[ 0]:=000X;
  copyCursorData[ 1]:=000X;
  copyCursorData[ 2]:=0FEX;
  copyCursorData[ 3]:=003X;
  copyCursorData[ 4]:=002X;
  copyCursorData[ 5]:=002X;
  copyCursorData[ 6]:=002X;
  copyCursorData[ 7]:=01EX;
  copyCursorData[ 8]:=072X;
  copyCursorData[ 9]:=012X;
  copyCursorData[10]:=002X;
  copyCursorData[11]:=012X;
  copyCursorData[12]:=072X;
  copyCursorData[13]:=012X;
  copyCursorData[14]:=002X;
  copyCursorData[15]:=012X;
  copyCursorData[16]:=072X;
  copyCursorData[17]:=012X;
  copyCursorData[18]:=002X;
  copyCursorData[19]:=012X;
  copyCursorData[20]:=002X;
  copyCursorData[21]:=012X;
  copyCursorData[22]:=0FEX;
  copyCursorData[23]:=013X;
  copyCursorData[24]:=010X;
  copyCursorData[25]:=010X;
  copyCursorData[26]:=0F0X;
  copyCursorData[27]:=01FX;
  copyCursorData[28]:=000X;
  copyCursorData[29]:=000X;

  copyMaskData[ 0]:=0FFX;
  copyMaskData[ 1]:=007X;
  copyMaskData[ 2]:=0FFX;
  copyMaskData[ 3]:=007X;
  copyMaskData[ 4]:=0FFX;
  copyMaskData[ 5]:=03FX;
  copyMaskData[ 6]:=0FFX;
  copyMaskData[ 7]:=03FX;
  copyMaskData[ 8]:=0FFX;
  copyMaskData[ 9]:=03FX;
  copyMaskData[10]:=0FFX;
  copyMaskData[11]:=03FX;
  copyMaskData[12]:=0FFX;
  copyMaskData[13]:=03FX;
  copyMaskData[14]:=0FFX;
  copyMaskData[15]:=03FX;
  copyMaskData[16]:=0FFX;
  copyMaskData[17]:=03FX;
  copyMaskData[18]:=0FFX;
  copyMaskData[19]:=03FX;
  copyMaskData[20]:=0FFX;
  copyMaskData[21]:=03FX;
  copyMaskData[22]:=0FFX;
  copyMaskData[23]:=03FX;
  copyMaskData[24]:=0FFX;
  copyMaskData[25]:=03FX;
  copyMaskData[26]:=0F8X;
  copyMaskData[27]:=03FX;
  copyMaskData[28]:=0F8X;
  copyMaskData[29]:=03FX;

  moveCursorData[ 0]:=000X;
  moveCursorData[ 1]:=000X;
  moveCursorData[ 2]:=0FEX;
  moveCursorData[ 3]:=003X;
  moveCursorData[ 4]:=002X;
  moveCursorData[ 5]:=002X;
  moveCursorData[ 6]:=002X;
  moveCursorData[ 7]:=002X;
  moveCursorData[ 8]:=072X;
  moveCursorData[ 9]:=002X;
  moveCursorData[10]:=002X;
  moveCursorData[11]:=002X;
  moveCursorData[12]:=072X;
  moveCursorData[13]:=002X;
  moveCursorData[14]:=002X;
  moveCursorData[15]:=002X;
  moveCursorData[16]:=072X;
  moveCursorData[17]:=002X;
  moveCursorData[18]:=002X;
  moveCursorData[19]:=002X;
  moveCursorData[20]:=002X;
  moveCursorData[21]:=002X;
  moveCursorData[22]:=0FEX;
  moveCursorData[23]:=003X;
  moveCursorData[24]:=000X;
  moveCursorData[25]:=000X;

  moveMaskData[ 0]:=0FFX;
  moveMaskData[ 1]:=007X;
  moveMaskData[ 2]:=0FFX;
  moveMaskData[ 3]:=007X;
  moveMaskData[ 4]:=0FFX;
  moveMaskData[ 5]:=007X;
  moveMaskData[ 6]:=0FFX;
  moveMaskData[ 7]:=007X;
  moveMaskData[ 8]:=0FFX;
  moveMaskData[ 9]:=007X;
  moveMaskData[10]:=0FFX;
  moveMaskData[11]:=007X;
  moveMaskData[12]:=0FFX;
  moveMaskData[13]:=007X;
  moveMaskData[14]:=0FFX;
  moveMaskData[15]:=007X;
  moveMaskData[16]:=0FFX;
  moveMaskData[17]:=007X;
  moveMaskData[18]:=0FFX;
  moveMaskData[19]:=007X;
  moveMaskData[20]:=0FFX;
  moveMaskData[21]:=007X;
  moveMaskData[22]:=0FFX;
  moveMaskData[23]:=007X;
  moveMaskData[24]:=0FFX;
  moveMaskData[25]:=007X;

  linkCursorData[ 0]:=000X;
  linkCursorData[ 1]:=000X;
  linkCursorData[ 2]:=0FEX;
  linkCursorData[ 3]:=003X;
  linkCursorData[ 4]:=002X;
  linkCursorData[ 5]:=02AX;
  linkCursorData[ 6]:=002X;
  linkCursorData[ 7]:=002X;
  linkCursorData[ 8]:=072X;
  linkCursorData[ 9]:=022X;
  linkCursorData[10]:=002X;
  linkCursorData[11]:=002X;
  linkCursorData[12]:=072X;
  linkCursorData[13]:=022X;
  linkCursorData[14]:=002X;
  linkCursorData[15]:=002X;
  linkCursorData[16]:=072X;
  linkCursorData[17]:=03EX;
  linkCursorData[18]:=002X;
  linkCursorData[19]:=022X;
  linkCursorData[20]:=002X;
  linkCursorData[21]:=02EX;
  linkCursorData[22]:=0FEX;
  linkCursorData[23]:=023X;
  linkCursorData[24]:=000X;
  linkCursorData[25]:=021X;
  linkCursorData[26]:=000X;
  linkCursorData[27]:=02DX;
  linkCursorData[28]:=000X;
  linkCursorData[29]:=021X;
  linkCursorData[30]:=000X;
  linkCursorData[31]:=03FX;
  linkCursorData[32]:=000X;
  linkCursorData[33]:=000X;

  linkMaskData[ 0]:=0FFX;
  linkMaskData[ 1]:=007X;
  linkMaskData[ 2]:=0FFX;
  linkMaskData[ 3]:=07FX;
  linkMaskData[ 4]:=0FFX;
  linkMaskData[ 5]:=07FX;
  linkMaskData[ 6]:=0FFX;
  linkMaskData[ 7]:=07FX;
  linkMaskData[ 8]:=0FFX;
  linkMaskData[ 9]:=077X;
  linkMaskData[10]:=0FFX;
  linkMaskData[11]:=077X;
  linkMaskData[12]:=0FFX;
  linkMaskData[13]:=077X;
  linkMaskData[14]:=0FFX;
  linkMaskData[15]:=07FX;
  linkMaskData[16]:=0FFX;
  linkMaskData[17]:=07FX;
  linkMaskData[18]:=0FFX;
  linkMaskData[19]:=07FX;
  linkMaskData[20]:=0FFX;
  linkMaskData[21]:=07FX;
  linkMaskData[22]:=0FFX;
  linkMaskData[23]:=07FX;
  linkMaskData[24]:=0FFX;
  linkMaskData[25]:=07FX;
  linkMaskData[26]:=080X;
  linkMaskData[27]:=07FX;
  linkMaskData[28]:=080X;
  linkMaskData[29]:=07FX;
  linkMaskData[30]:=080X;
  linkMaskData[31]:=07FX;
  linkMaskData[32]:=080X;
  linkMaskData[33]:=07FX;

END VO:OS:Display.