(**
  A image class that implements a number of frames. Use the frames
  implemented here everywhere where possible. Do @emph{not} implement
  your own frames, add them here. Add your frames also here, when you
  want to implement another look.

  TODO
  * nicer frame for string gadget

  * Make some frames support label (groupFrame)

  * Isn't a line a special frame?
**)

MODULE VO:Base:Frame;

(*
    Frames.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D  := VO:Base:Display,
       PP := VO:Prefs:Parser;

CONST

  (* Internal Frames *)

  none              * =  0;
  noneSingle        * =  1;
  noneDouble        * =  2;
  single3DIn        * =  3;
  single3DOut       * =  4;
  single            * =  5;
  double            * =  6;
  double3DIn        * =  7;
  double3DOut       * =  8;
  double3DTool      * =  9;
  group3D           * = 10;
  ridge             * = 11;
  doubleFocus       * = 12; (* A thicker frame for motif-like focus *)
  dottedFocus       * = 13; (* A dotted-line focus frame            *)
  w95BO             * = 14; (* Win95 button, released               *)
  w95BI             * = 15; (* Win95 button, pressed                *)
  w95IO             * = 16; (* Win95 image button, released         *)
  w95II             * = 17; (* Win95 image button, pressed          *)
  w95CI             * = 18; (* Container in w95 look                *)
  w95CO             * = 19; (* Container in w95 look                *)
  BBBI              * = 20; (* BeBox Button in                      *)
  BBBO              * = 21; (* BeBox Button out                     *)

  internalFrameCount* = 22;

TYPE
  FrameName       = ARRAY 30 OF CHAR;

  Frame *     = POINTER TO FrameDesc;
  FrameDesc * = RECORD
                  type-          : LONGINT;
                  topBorder-,
                  bottomBorder-,
                  leftBorder-,
                  rightBorder-,
                  x-,y-,w-,h-,
                  gx-,gy-,gw-,gh-,
                  minWidth-,
                  minHeight-     : LONGINT;
                  draw           : D.DrawInfo;
                END;


VAR
  internalFrames- : ARRAY internalFrameCount OF FrameName;

  PROCEDURE (f : Frame) Init*;

  BEGIN
    f.type:=none;
    f.gw:=0;
    f.gh:=0;
    f.draw:=NIL;
  END Init;

  PROCEDURE (f : Frame) HasGap*():BOOLEAN;

  BEGIN
    RETURN (f.gw>0) & (f.gh>0);
  END HasGap;

  (**
    Set the frame to be used.

    NOTE
    The frame can be changed at runtime.
  **)

  PROCEDURE (f : Frame) SetFrame*(type : LONGINT);

  BEGIN
    f.type:=type;

    CASE f.type OF
      none:
        f.topBorder:=0;
        f.bottomBorder:=0;
        f.leftBorder:=0;
        f.rightBorder:=0;
    | single3DIn,
      single3DOut,
      noneSingle,
      single:
        f.topBorder:=1;
        f.bottomBorder:=1;
        f.leftBorder:=1;
        f.rightBorder:=1;
    | noneDouble,
      double,
      double3DIn,
      double3DOut,
      double3DTool,
      ridge:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | group3D:
        IF f.HasGap() THEN
          f.topBorder:=f.gh; (*G.MaxLong(f.gh,2);*)
(*          f.width:=f.gw+4*display.spaceWidth;*)
        ELSE
          f.topBorder:=2;
        END;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | doubleFocus:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | dottedFocus:
        f.topBorder:=1;
        f.bottomBorder:=1;
        f.leftBorder:=1;
        f.rightBorder:=1;
    | w95BO,
      w95BI:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | w95CI,
      w95CO:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | w95II,
      w95IO:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | BBBI,
      BBBO:
        f.topBorder:=4;
        f.bottomBorder:=4;
        f.leftBorder:=4;
        f.rightBorder:=4;
    ELSE
        f.topBorder:=0;
        f.bottomBorder:=0;
        f.leftBorder:=0;
        f.rightBorder:=0;
    END;

    f.minWidth:=f.leftBorder+f.rightBorder;
    f.minHeight:=f.topBorder+f.bottomBorder;

    IF f.HasGap() THEN
      INC(f.minWidth,f.gw+4*8);
    END;
  END SetFrame;

  PROCEDURE (f : Frame) SetGap*(width,height : LONGINT);

  BEGIN
    f.gw:=width;
    f.gh:=height;
    (* recalculate sizes *)
    f.SetFrame(f.type);
  END SetGap;

  PROCEDURE (f : Frame) Draw*(draw : D.DrawInfo; x,y,w,h : LONGINT);

  VAR
    shine,
    shadow,
    halfShadow,
    halfShine,
    top,type    : LONGINT;

  BEGIN
    f.x:=x;
    f.y:=y;
    f.w:=w;
    f.h:=h;

    f.draw:=draw;

    shine:=D.shineColor;
    shadow:=D.shadowColor;
    halfShadow:=D.halfShadowColor;
    halfShine:=D.halfShineColor;

    type:=f.type;

    IF D.selected IN draw.mode THEN
      IF type=double3DOut THEN
        type:=double3DIn;
      ELSIF type=single3DOut THEN
        type:=single3DIn;
      ELSIF type=w95BO THEN
        type:=w95BI;
      ELSIF type=BBBO THEN
        type:=BBBI;
      ELSIF type=w95IO THEN
        type:=w95II;
      END;
    ELSIF D.hidden IN draw.mode THEN
      shine:=D.backgroundColor;
      shadow:=D.backgroundColor;
      halfShadow:=D.backgroundColor;
      halfShine:=D.backgroundColor;
    END;

    CASE type OF
      none,
      noneSingle,
      noneDouble:

    | double:
        draw.PushForeground(shadow);
        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-1,y);
        draw.DrawLine(x+1,y+h-2,x+1,y+2);
        draw.DrawLine(x+1,y+1,x+w-2,y+1);

        draw.DrawLine(x+w-1,y+1,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
        draw.DrawLine(x+w-2,y+2,x+w-2,y+h-2);
        draw.DrawLine(x+w-2,y+h-2,x+1,y+h-2);
        draw.PopForeground;

    | double3DIn,
      double3DOut:
        IF type=double3DIn THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-1,y);
        draw.DrawLine(x+1,y+h-2,x+1,y+2);
        draw.DrawLine(x+1,y+1,x+w-2,y+1);
        draw.PopForeground;

        IF type=double3DIn THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(x+w-1,y+1,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
        draw.DrawLine(x+w-2,y+2,x+w-2,y+h-2);
        draw.DrawLine(x+w-2,y+h-2,x+1,y+h-2);
        draw.PopForeground;

    | double3DTool:
        IF (draw.mode={}) OR (draw.mode={D.hidden}) THEN
          draw.PushForeground(D.backgroundColor);
        ELSIF draw.mode={D.selected} THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;

        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-1,y);
(*        draw.DrawLine(x+1,y+h-2,x+1,y+2);
        draw.DrawLine(x+1,y+1,x+w-2,y+1);*)
        draw.PopForeground;


        IF (draw.mode={}) OR (draw.mode={D.hidden}) THEN
          draw.PushForeground(D.backgroundColor);
        ELSIF draw.mode={D.selected} THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(x+w-1,y+1,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
(*        draw.DrawLine(x+w-2,y+2,x+w-2,y+h-2);
        draw.DrawLine(x+w-2,y+h-2,x+1,y+h-2);*)
        draw.PopForeground;

    | single3DIn,
      single3DOut :
        IF type=single3DIn THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-1,y);
        draw.PopForeground;

        IF type=single3DIn THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(x+w-1,y+1,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
        draw.PopForeground;

    | single:
        draw.PushForeground(shadow);
        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-1,y);
        draw.DrawLine(x+w-1,y+1,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
        draw.PopForeground;

    | group3D:
        IF f.HasGap() THEN
          top:=y+f.gh DIV 2;
        ELSE
          top:=y;
        END;
        draw.PushForeground(shadow);
        draw.DrawLine(x,y+h-1,x,top);
        draw.DrawLine(x+w-2,top+1,x+w-2,y+h-2);
        draw.DrawLine(x+1,y+h-2,x+w-2,y+h-2);
        IF f.HasGap() THEN
          draw.DrawLine(x,top,x+D.display.spaceWidth-1,top);
          draw.DrawLine(x+3*D.display.spaceWidth+f.gw+1,top,x+w-1,top);
          draw.DrawLine(x+3*D.display.spaceWidth+f.gw,top,
                        x+3*D.display.spaceWidth+f.gw,top+1);
        ELSE
          draw.DrawLine(x,top,x+w-1,top);
        END;
        draw.PopForeground;

        draw.PushForeground(shine);
        draw.DrawLine(x+1,y+h-2,x+1,top+1);
        draw.DrawLine(x+w-1,top+1,x+w-1,y+h-1);
        draw.DrawLine(x+1,y+h-1,x+w-2,y+h-1);
        IF f.HasGap() THEN
          draw.DrawLine(x+2,top+1,x+D.display.spaceWidth-1,top+1);
          draw.DrawLine(x+3*D.display.spaceWidth+f.gw+1,top+1,x+w-2,top+1);
          draw.DrawLine(x+D.display.spaceWidth,top,x+D.display.spaceWidth,top+1);
        ELSE
          draw.DrawLine(x+2,top+1,x+w-2,top+1);
        END;
        draw.PopForeground;

        f.gx:=x+f.leftBorder+2*D.display.spaceWidth;
        f.gy:=y;

    | ridge:
        draw.PushForeground(shine);
        draw.DrawLine(x,y,x+w-2,y);
        draw.DrawLine(x+2,y+h-2,x+w-2,y+h-2);
        draw.DrawLine(x,y+h-1,x,y);
        draw.DrawLine(x+w-2,y+2,x+w-2,y+h-2);
        draw.PopForeground;

        draw.PushForeground(shadow);
        draw.DrawLine(x+w-1,y,x+w-1,y+h-1);
        draw.DrawLine(x+1,y+h-2,x+1,y+1);
        draw.DrawLine(x+1,y+h-1,x+w-1,y+h-1);
        draw.DrawLine(x+1,y+1,x+w-2,y+1);
        draw.PopForeground;

    | doubleFocus:
        IF D.hidden IN draw.mode THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(D.focusColor);
        END;
        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-1,y);
        draw.DrawLine(x+1,y+h-2,x+1,y+2);
        draw.DrawLine(x+1,y+1,x+w-2,y+1);
        draw.DrawLine(x+w-1,y+1,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
        draw.DrawLine(x+w-2,y+2,x+w-2,y+h-2);
        draw.DrawLine(x+w-2,y+h-2,x+1,y+h-2);
        draw.PopForeground;
    | dottedFocus:
        IF D.hidden IN draw.mode THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(D.textColor);
        END;
        draw.PushDash(D.sPointLine,D.fMode);
        draw.DrawLine(x          ,y,           x+w-1,y);
        draw.DrawLine(x+w-1,y,           x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,          y+h-1);
        draw.DrawLine(x          ,y+h-1,x,          y);
        draw.PopDash;
        draw.PopForeground;
    | w95BO,
      w95BI:
        IF type=w95BI THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-2,y);
        draw.PopForeground;

        IF type=w95BI THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(D.backgroundColor);
        END;
        draw.DrawLine(x+1,y+h-2,x+1,y+2);
        draw.DrawLine(x+1,y+1,x+w-3,y+1);
        draw.PopForeground;

        IF type=w95BI THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(x+w-1,y,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
        draw.PopForeground;

        IF type=w95BI THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(halfShadow);
        END;
        draw.DrawLine(x+w-2,y+1,x+w-2,y+h-2);
        draw.DrawLine(x+w-2,y+h-2,x+1,y+h-2);
        draw.PopForeground;
    | w95CI,
      w95CO:
        IF type=w95CI THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(halfShine);
        END;
        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-2,y);
        draw.PopForeground;

        IF type=w95CI THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(x+1,y+h-2,x+1,y+2);
        draw.DrawLine(x+1,y+1,x+w-3,y+1);
        draw.PopForeground;

        IF type=w95CI THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(x+w-1,y,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
        draw.PopForeground;

        IF type=w95CI THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(halfShadow);
        END;
        draw.DrawLine(x+w-2,y+1,x+w-2,y+h-2);
        draw.DrawLine(x+w-2,y+h-2,x+1,y+h-2);
        draw.PopForeground;
    | w95II,
      w95IO:
        IF type=w95II THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(D.backgroundColor);
        END;
        draw.DrawLine(x,y+h-1,x,y+1);
        draw.DrawLine(x,y,x+w-2,y);
        draw.PopForeground;

        IF type=w95II THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(x+1,y+h-2,x+1,y+2);
        draw.DrawLine(x+1,y+1,x+w-3,y+1);
        draw.PopForeground;

        IF type=w95II THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(x+w-1,y,x+w-1,y+h-1);
        draw.DrawLine(x+w-1,y+h-1,x,y+h-1);
        draw.PopForeground;

        IF type=w95II THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(halfShadow);
        END;
        draw.DrawLine(x+w-2,y+1,x+w-2,y+h-2);
        draw.DrawLine(x+w-2,y+h-2,x+1,y+h-2);
        draw.PopForeground;
    | BBBI,
      BBBO:
        (* outest frame *)
        IF D.hidden IN draw.mode THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(D.blackColor);
        END;
        draw.DrawLine(x+1,y,x+w-2,y);
        draw.DrawLine(x+1,y+h-1,x+w-2,y+h-1);
        draw.DrawLine(x,y+1,x,y+h-2);
        draw.DrawLine(x+w-1,y+1,x+w-1,y+h-2);
        draw.PopForeground;

        (* 2nd frame *)

        draw.PushForeground(D.backgroundColor);
        draw.DrawLine(x+1,y+1,x+w-2,y+1);
        draw.DrawLine(x+1,y+2,x+1,y+h-2);
        draw.PopForeground;

        IF type=BBBI THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(x+w-2,y+2,x+w-2,y+h-2);
        draw.DrawLine(x+2,y+h-2,x+w-2,y+h-2);
        draw.PopForeground;

        (* 3rd frame *)

        IF type=BBBI THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(halfShine);
        END;
        draw.DrawLine(x+2,y+2,x+w-3,y+2);
        draw.DrawLine(x+2,y+3,x+2,y+h-3);
        draw.PopForeground;

        IF type=BBBI THEN
          draw.PushForeground(halfShine);
        ELSE
          draw.PushForeground(halfShadow);
        END;
        draw.DrawLine(x+w-3,y+3,x+w-3,y+h-3);
        draw.DrawLine(x+4,y+h-3,x+w-4,y+h-3);
        draw.PopForeground;

        (* 4th frame *)

        IF type=BBBI THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(x+3,y+3,x+w-4,y+3);
        draw.DrawLine(x+3,y+4,x+3,y+h-4);
        draw.PopForeground;
    END;
  END Draw;

  PROCEDURE (f : Frame) Redraw*;

  BEGIN
    f.Draw(f.draw,f.x,f.y,f.w,f.h);
  END Redraw;

  PROCEDURE (f : Frame) Hide*;

  BEGIN
    f.draw.mode:={D.hidden};
    f.Draw(f.draw,f.x,f.y,f.w,f.h);
    f.draw.mode:={};
  END Hide;

  PROCEDURE CreateFrame*():Frame;

  VAR
    frame : Frame;

  BEGIN
    NEW(frame);
    frame.Init;

    RETURN frame;
  END CreateFrame;

  PROCEDURE GetFrameEntry*(name : ARRAY OF CHAR):LONGINT;

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=0 TO internalFrameCount-1 DO
      IF name=internalFrames[x] THEN
        RETURN x;
      END;
    END;
    RETURN -1;
  END GetFrameEntry;

  PROCEDURE LoadFrame*(name : ARRAY OF CHAR; top : PP.Item; VAR value : LONGINT);

  VAR
    buffer : ARRAY 256 OF CHAR;
    pos    : LONGINT;

  BEGIN
    IF top.GetStringEntry(name,buffer) THEN
      pos:=GetFrameEntry(buffer);
      IF pos>=0 THEN
        value:=pos;
      END;
    END;
  END LoadFrame;

BEGIN
  internalFrames[none]        :="none";
  internalFrames[noneSingle]  :="noneSingle";
  internalFrames[noneDouble]  :="noneDouble";
  internalFrames[single3DIn]  :="single3DIn";
  internalFrames[single3DOut] :="single3DOut";
  internalFrames[single]      :="single";
  internalFrames[double]      :="double";
  internalFrames[double3DIn]  :="double3DIn";
  internalFrames[double3DOut] :="double3DOut";
  internalFrames[double3DTool]:="double3DTool";
  internalFrames[group3D]     :="group3D";
  internalFrames[ridge]       :="ridge";
  internalFrames[doubleFocus] :="doubleFocus";
  internalFrames[dottedFocus] :="dottedFocus";
  internalFrames[w95BO]       :="w95BO";
  internalFrames[w95BI]       :="w95BI";
  internalFrames[w95IO]       :="w95IO";
  internalFrames[w95II]       :="w95II";
  internalFrames[w95CI]       :="w95CI";
  internalFrames[w95CO]       :="w95CO";
  internalFrames[BBBI]        :="BBBI";
  internalFrames[BBBO]        :="BBBO";
END VO:Base:Frame.