/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package test.mx4j.tools.adaptor.rmi;

import java.io.File;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.rmi.RemoteException;
import java.util.Set;
import javax.management.MBeanInfo;
import javax.management.MBeanServer;
import javax.management.MBeanServerInvocationHandler;
import javax.management.MBeanServerNotification;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.naming.Context;
import javax.naming.NamingException;

import mx4j.tools.adaptor.interceptor.TimingAdaptorInterceptor;
import mx4j.tools.adaptor.rmi.SSLAdaptorRMIServerSocketFactory;
import mx4j.tools.adaptor.rmi.SSLRMIClientSocketFactory;
import mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor;
import mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptorMBean;
import mx4j.tools.adaptor.ssl.SSLAdaptorServerSocketFactory;
import mx4j.tools.adaptor.ssl.SSLAdaptorServerSocketFactoryMBean;
import mx4j.tools.connector.RemoteMBeanServer;
import mx4j.tools.connector.rmi.jrmp.JRMPConnector;
import mx4j.tools.connector.rmi.jrmp.JRMPRemoteNotificationListenerSupport;
import test.MX4JTestCase;
import test.MutableBoolean;
import test.MutableInteger;

/**
 * @version $Revision: 1.4 $
 */
public class JRMPAdaptorTest extends MX4JTestCase
{
   private MBeanServer m_server;
   private ObjectName m_naming;

   public JRMPAdaptorTest(String s)
   {
      super(s);
   }

   protected void setUp() throws Exception
   {
      m_server = newMBeanServer();
      m_naming = new ObjectName("Naming:type=rmiregistry");
      m_server.createMBean("mx4j.tools.naming.NamingService", m_naming, null);
      m_server.invoke(m_naming, "start", null, null);
      System.setProperty(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.rmi.registry.RegistryContextFactory");
      System.setProperty(Context.PROVIDER_URL, "rmi://localhost:1099");
   }

   protected void tearDown() throws Exception
   {
      System.getProperties().remove(Context.INITIAL_CONTEXT_FACTORY);
      System.getProperties().remove(Context.PROVIDER_URL);
      m_server.invoke(m_naming, "stop", null, null);
      m_server.unregisterMBean(m_naming);
   }

   public void testAdaptorNoName() throws Exception
   {
      JRMPAdaptor adaptor = new JRMPAdaptor();
      adaptor.setMBeanServer(m_server);

      // Do not set the JNDI name, but call start()
      try
      {
         adaptor.start();
         fail("Cannot start adaptor with no name");
      }
      catch (NamingException ignored)
      {
      }
   }

   public void testAdaptorNotRegistered() throws Exception
   {
      JRMPAdaptor adaptor = new JRMPAdaptor();
      adaptor.setMBeanServer(m_server);
      String jndiName = "jrmp";
      adaptor.setJNDIName(jndiName);

      try
      {
         adaptor.start();

         JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         System.out.println("NAME: " + connector.getRemoteHostName());
         System.out.println("IP: " + connector.getRemoteHostAddress());
         System.out.println("COUNT: " + connector.getRemoteMBeanServer().getMBeanCount());
      }
      finally
      {
         adaptor.stop();
      }
   }

   public void testAddInterceptorNotRegistered() throws Exception
   {
      JRMPAdaptor adaptor = new JRMPAdaptor();
      adaptor.setMBeanServer(m_server);
      String jndiName = "jrmp";
      adaptor.setJNDIName(jndiName);

      TimingAdaptorInterceptor intr = new TimingAdaptorInterceptor();
      intr.setEnabled(true);
      adaptor.addInterceptor(intr);

      try
      {
         adaptor.start();

         JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         System.out.println("NAME: " + connector.getRemoteHostName());
         System.out.println("IP: " + connector.getRemoteHostAddress());
         System.out.println("COUNT: " + connector.getRemoteMBeanServer().getMBeanCount());
      }
      finally
      {
         adaptor.stop();
      }
   }

   public void testAddInterceptorRegistered() throws Exception
   {
      int initialCount = m_server.getMBeanCount().intValue();
      System.out.println("MBEANS: " + m_server.queryMBeans(null, null));

      JRMPAdaptor adaptor = new JRMPAdaptor();
      String jndiName = "jrmp";
      adaptor.setJNDIName(jndiName);
      ObjectName adaptorName = new ObjectName("Adaptor:protocol=JRMP");

      TimingAdaptorInterceptor intr = new TimingAdaptorInterceptor();
      ObjectName interceptorName = new ObjectName("AdaptorInterceptor:type=timing");
      intr.setObjectName(interceptorName);
      intr.setEnabled(true);
      adaptor.addInterceptor(intr);

      m_server.registerMBean(adaptor, adaptorName);

      try
      {
         adaptor.start();

         JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         RemoteMBeanServer server = connector.getRemoteMBeanServer();
         System.out.println("NAME: " + connector.getRemoteHostName());
         System.out.println("IP: " + connector.getRemoteHostAddress());
         int count = server.getMBeanCount().intValue();
         System.out.println("COUNT: " + count);
         System.out.println("MBEANS: " + server.queryMBeans(null, null));
// Five more MBean: the adaptor, the logger interceptor, the context classloader interceptor,
// the timing interceptor and the invoker interceptor
         if (count - initialCount != 5) fail("Interceptors not registered");

// Check that the interceptor has been registered with the given name
         Set mbeans = m_server.queryMBeans(interceptorName, null);
         if (mbeans.size() != 1) fail("Interceptor registered with wrong name");

         ObjectInstance instance = (ObjectInstance)mbeans.iterator().next();
         if (!instance.getClassName().equals(intr.getClass().getName())) fail("Interceptor registered with wrong name");
      }
      finally
      {
         adaptor.stop();
      }
   }

   public void testStartStopConnect() throws Exception
   {
      ObjectName adaptor = new ObjectName("Adaptor:protocol=JRMP");
      m_server.createMBean("mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor", adaptor, null);
      JRMPAdaptorMBean mbean = (JRMPAdaptorMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, adaptor, JRMPAdaptorMBean.class, false);
      String jndiName = "jrmp";
      mbean.setJNDIName(jndiName);

      mbean.start();

      // Now simulate a client
      JRMPConnector connector = new JRMPConnector();
      connector.connect(jndiName, null);
      RemoteMBeanServer server = connector.getRemoteMBeanServer();
      System.out.println("NAME: " + connector.getRemoteHostName());
      System.out.println("IP: " + connector.getRemoteHostAddress());

      server.invoke(adaptor, "stop", null, null);

      try
      {
         System.out.println("IP: " + connector.getRemoteHostAddress());
         fail("Connector should be disabled");
      }
      catch (RemoteException ignored)
      {
      }

      try
      {
// Restart mbean
         mbean.start();

// Reconnect
         connector.connect(jndiName, null);
         System.out.println("IP: " + connector.getRemoteHostAddress());
         System.out.println("COUNT: " + server.getMBeanCount());
      }
      finally
      {
         mbean.stop();
      }
   }

   public void testAddRemoveListeners() throws Exception
   {
      ObjectName adaptor = new ObjectName("Adaptor:protocol=JRMP");
      m_server.createMBean("mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor", adaptor, null);
      JRMPAdaptorMBean mbean = (JRMPAdaptorMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, adaptor, JRMPAdaptorMBean.class, false);
      String jndiName = "jrmp";
      mbean.setJNDIName(jndiName);

      try
      {
         mbean.start();

// Now simulate a client
         JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         RemoteMBeanServer server = connector.getRemoteMBeanServer();

         MutableInteger integer = new MutableInteger(0);
         Listener listener = new Listener(integer);

// Add a listener for registrations and unregistrations
         server.addNotificationListener(new ObjectName("JMImplementation:type=MBeanServerDelegate"), listener, null, null);

// OK, now register an MBean; the server should notify listeners of this new mbean
         server.createMBean("javax.management.loading.MLet", new ObjectName(":type=MLet1"), null);

// Wait for a while to have the time to receive the remote notification
         Thread.sleep(1000);

         if (integer.get() < 1) fail("Remote notifications are not working");

// Remove the listener
         server.removeNotificationListener(new ObjectName("JMImplementation:type=MBeanServerDelegate"), listener, null, null);

// Unregister the MBean, the server should notify again
         server.unregisterMBean(new ObjectName(":type=MLet1"));

         if (integer.get() != 1) fail("Remote NotificationListener not removed");
      }
      finally
      {
         mbean.stop();
      }
   }

   public void testRemoteNotificationListeners() throws Exception
   {
      ObjectName adaptor = new ObjectName("Adaptor:protocol=JRMP");
      m_server.createMBean("mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor", adaptor, null);
      JRMPAdaptorMBean mbean = (JRMPAdaptorMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, adaptor, JRMPAdaptorMBean.class, false);
      String jndiName = "jrmp";
      mbean.setJNDIName(jndiName);

      try
      {
         mbean.start();

// Now simulate a client
         JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         RemoteMBeanServer server = connector.getRemoteMBeanServer();

         Listener listener = new Listener(new MutableInteger(0));

// Add a listener for registrations and unregistrations
         server.addNotificationListener(new ObjectName("JMImplementation:type=MBeanServerDelegate"), listener, null, null);

// OK, now register an MBean; the server should notify listeners of this new mbean
         server.createMBean("javax.management.loading.MLet", new ObjectName(":type=MLet1"), null);

         MutableInteger integer = listener.getInteger();
         if (integer == null)
         {
// Means that this listener has been serialized to the server, bad
            fail("Listener must not be serialized");
         }

// Wait for a while to have the time to receive the remote notification
         Thread.sleep(1000);

         if (integer.get() < 1) fail("Remote notifications are not working");
      }
      finally
      {
         mbean.stop();
      }
   }

   public void testRemoteNotificationHandbacks() throws Exception
   {
      ObjectName adaptor = new ObjectName("Adaptor:protocol=JRMP");
      m_server.createMBean("mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor", adaptor, null);
      JRMPAdaptorMBean mbean = (JRMPAdaptorMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, adaptor, JRMPAdaptorMBean.class, false);
      String jndiName = "jrmp";
      mbean.setJNDIName(jndiName);

      try
      {
         mbean.start();

// Now simulate a client
         JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         RemoteMBeanServer server = connector.getRemoteMBeanServer();

         Listener handback = new Listener(new MutableInteger(0));
         Listener listener = new Listener(new MutableInteger(0));

// Add a listener for registrations and unregistrations, with handback
         server.addNotificationListener(new ObjectName("JMImplementation:type=MBeanServerDelegate"), listener, null, handback);

// OK, now register an MBean; the server should notify listeners of this new mbean
         server.createMBean("javax.management.loading.MLet", new ObjectName(":type=MLet2"), null);

// Wait for a while to have the time to receive the remote notification
         Thread.sleep(1000);

         MutableInteger integer = listener.getInteger();
         if (integer == null) fail("Listener must not be serialized");
         if (integer.get() < 1) fail("Remote notifications are not working");

// Test the handback, must have been serialized to the server
         Listener hb = (Listener)listener.getHandBack();
// Must have been serialized
         if (hb.getInteger() != null) fail("Handback must be serialized");
      }
      finally
      {
         mbean.stop();
      }
   }

   public void testRemoteListenerCallback() throws Exception
   {
      ObjectName adaptor = new ObjectName("Adaptor:protocol=JRMP");

      m_server.createMBean("mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor", adaptor, null);
      JRMPAdaptorMBean mbean = (JRMPAdaptorMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, adaptor, JRMPAdaptorMBean.class, false);
      String jndiName = "jrmp";
      mbean.setJNDIName(jndiName);

      try
      {
         mbean.start();

         // Now simulate a client
         final JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         RemoteMBeanServer server = connector.getRemoteMBeanServer();

         final MutableInteger integer = new MutableInteger(0);

         // Add a listener for registrations and unregistrations
         NotificationListener listener = new NotificationListener()
         {
            public void handleNotification(Notification notification, Object handback)
            {
               // Call back the server, should not block
               if (notification instanceof MBeanServerNotification)
               {
                  try
                  {
                     MBeanServerNotification n = (MBeanServerNotification)notification;
                     System.out.println("Before callback...");
                     MBeanInfo info = connector.getRemoteMBeanServer().getMBeanInfo(n.getMBeanName());
                     System.out.println("After callback");
                     if (info == null)
                     {
                        fail("Null MBeanInfo");
                     }
                     integer.set(integer.get() + 1);
                  }
                  catch (Exception x)
                  {
                     x.printStackTrace();
                     integer.set(0);
                  }
               }
               else
               {
                  integer.set(0);
               }
            }
         };

         ObjectName delegate = new ObjectName("JMImplementation:type=MBeanServerDelegate");
         server.addNotificationListener(delegate, listener, null, null);

         // OK, now register an MBean; the server should notify listeners of this new mbean
         server.createMBean("javax.management.loading.MLet", new ObjectName(":type=MLet2"), null);

         // Wait for a while to have the time to receive the remote notification
         Thread.sleep(1000);

         // Remove the listener, so that it will not be called upon deregistration of MBeans
         server.removeNotificationListener(delegate, listener, null, null);

         // If, after this while, we're not received the notification, then we're in trouble
         if (integer.get() != 1) fail("Problems with notification callbacks");
      }
      finally
      {
         mbean.stop();
      }
   }

   public void testStandardArgumentOperationMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testStandardArgumentOperation", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testCustomReturnValueOperationMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testCustomReturnValueOperation", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testCustomArgumentOperationMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testCustomArgumentOperation", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testGetCustomAttributeMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testGetCustomAttribute", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testSetCustomAttributeMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testSetCustomAttribute", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testGetCustomAttributesMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testGetCustomAttributes", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testSetCustomAttributesMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testSetCustomAttributes", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testCreateMBeanWithRepositoryMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testCreateMBeanWithRepository", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testCreateMBeanWithNullLoaderMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testCreateMBeanWithNullLoader", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testCreateMBeanWithLoaderMarshalling() throws Exception
   {
      // Must do bad tricks with classloaders. Hierarchy is this one:
      //
      //                   Ext-ClassLoader
      //                          |
      //                   JMX-ClassLoader
      //                     |         |
      //       Tools-ClassLoader     MBean-ClassLoader

      ClassLoader jmxCl = createMX4JClassLoader();

      File toolsJar = new File("dist/test/mx4j-tools.jar");
      URLClassLoader toolsCl = new URLClassLoader(new URL[]{toolsJar.toURL()}, jmxCl);

      File mbeanJar = new File("dist/test/mx4j-tests.jar");
      URLClassLoader mbeanCl = new URLClassLoader(new URL[]{mbeanJar.toURL()}, jmxCl);

      Object test = mbeanCl.loadClass("test.mx4j.tools.adaptor.rmi.support.TestMarshalling").newInstance();
      Method method = test.getClass().getMethod("testCreateMBeanWithLoader", new Class[]{ClassLoader.class});
      method.invoke(test, new Object[]{toolsCl});
   }

   public void testJRMPOverSSL() throws Exception
   {
//		System.setProperty("javax.net.debug", "all");

      ObjectName adaptor = new ObjectName("Adaptor:protocol=JRMP");

      // Clients need to have this system property set
      String baseDir = System.getProperty("user.dir");
      String trust = baseDir + "/dist/test/trust.store";
      System.setProperty("javax.net.ssl.trustStore", trust);

      ObjectName ssl = new ObjectName(":type=SSL");
      m_server.createMBean("mx4j.tools.adaptor.ssl.SSLAdaptorServerSocketFactory", ssl, null);
      SSLAdaptorServerSocketFactoryMBean factory = (SSLAdaptorServerSocketFactoryMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, ssl, SSLAdaptorServerSocketFactoryMBean.class, false);
      factory.setKeyStoreName("key.store");
      factory.setKeyStorePassword("storepwd");

      m_server.createMBean("mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor", adaptor, null);
      JRMPAdaptorMBean mbean = (JRMPAdaptorMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, adaptor, JRMPAdaptorMBean.class, false);
      String jndiName = "jrmp";
      mbean.setJNDIName(jndiName);
      mbean.setSSLFactory(ssl.toString());

      try
      {
         mbean.start();

// Now simulate a client
         JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         System.out.println("IP: " + connector.getRemoteHostAddress());
         System.out.println("COUNT: " + connector.getRemoteMBeanServer().getMBeanCount());
      }
      finally
      {
         mbean.stop();
      }
   }

   public void testRemoteNotificationListenersOverSSL() throws Exception
   {
//		System.setProperty("javax.net.debug", "all");

      // Clients need to have this system property set
      String baseDir = System.getProperty("user.dir");
      String trust = baseDir + "/dist/test/trust.store";
//		System.out.println("trust = " + trust);
      System.setProperty("javax.net.ssl.trustStore", trust);

      ObjectName sslFactory = new ObjectName("Factory:type=SSL");
      m_server.createMBean("mx4j.tools.adaptor.ssl.SSLAdaptorServerSocketFactory", sslFactory, null);
      SSLAdaptorServerSocketFactoryMBean factory = (SSLAdaptorServerSocketFactoryMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, sslFactory, SSLAdaptorServerSocketFactoryMBean.class, false);
      factory.setKeyStoreName("key.store");
      factory.setKeyStorePassword("storepwd");

      ObjectName adaptor = new ObjectName("Adaptor:protocol=JRMP");
      m_server.createMBean("mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor", adaptor, null);
      JRMPAdaptorMBean mbean = (JRMPAdaptorMBean)MBeanServerInvocationHandler.newProxyInstance(m_server, adaptor, JRMPAdaptorMBean.class, false);
      String jndiName = "jrmp";
      mbean.setJNDIName(jndiName);
      mbean.setSSLFactory(sslFactory.toString());

      try
      {
         mbean.start();

// Now simulate a client

         JRMPConnector connector = new JRMPConnector();
         connector.connect(jndiName, null);
         RemoteMBeanServer server = connector.getRemoteMBeanServer();
         System.out.println("IP: " + connector.getRemoteHostAddress());

// Register a remote listener
         final MutableBoolean bool = new MutableBoolean(false);
         NotificationListener listener = new NotificationListener()
         {
            public void handleNotification(Notification notification, Object handback)
            {
               bool.set(true);
            }
         };

         SSLAdaptorServerSocketFactory clientSSLFactory = new SSLAdaptorServerSocketFactory();
         clientSSLFactory.setKeyStoreName("key.store");
         clientSSLFactory.setKeyStorePassword("storepwd");
         JRMPRemoteNotificationListenerSupport sslListener = new JRMPRemoteNotificationListenerSupport(listener);
         sslListener.export(0, new SSLRMIClientSocketFactory(), new SSLAdaptorRMIServerSocketFactory(clientSSLFactory));
         server.addNotificationListener(new ObjectName("JMImplementation:type=MBeanServerDelegate"), sslListener, null, null);

// OK, now register an MBean; the server should notify listeners of this new mbean
         server.createMBean("javax.management.loading.MLet", new ObjectName(":type=MLet2"), null);

// Wait for a while to have the time to receive the remote notification
         Thread.sleep(1000);

         if (!bool.get()) fail("SSL Remote listeners not working");
      }
      finally
      {
         mbean.stop();
      }
   }

   private static class Listener implements NotificationListener, Serializable
   {
      private transient MutableInteger m_notified;
      private Object m_handback;

      public Listener(MutableInteger notified)
      {
         m_notified = notified;
      }

      public void handleNotification(Notification notification, Object handback)
      {
         m_notified.set(m_notified.get() + 1);
         m_handback = handback;
      }

      private MutableInteger getInteger()
      {
         return m_notified;
      }

      private Object getHandBack()
      {
         return m_handback;
      }
   }
}
