#include <ruby.h>
#include <st.h>

#include <locale.h>
#define __USE_GNU
#include <langinfo.h>

static VALUE mLocale;
static VALUE mLocale_LangInfo;

static VALUE rb_setlocale(VALUE category_val, VALUE locale_val)
{
	int category;
	const char* locale;
	char* ret_locale;

	Check_Type(category_val, T_FIXNUM);
	category = NUM2INT(category_val);

#ifdef SafeStringValue
	SafeStringValue(locale_val);
#else
	Check_SafeStr(locale_val);
#endif
	locale = RSTRING(locale_val)->ptr;

	ret_locale = setlocale(category, locale);

	if (ret_locale == NULL)
		rb_raise(rb_eSystemCallError, "call to setlocale failed for args (%d, %s)", category, locale);

	return rb_tainted_str_new2(ret_locale);
}

static VALUE rb_setlocale_i(VALUE key, VALUE val, VALUE hash)
{
	VALUE locale;
	if (key == Qundef) return ST_CONTINUE;
	locale = rb_setlocale(key, val);
	rb_hash_aset(hash, key, locale);
	return ST_CONTINUE;
}

static VALUE mLocale_setlocale(int argc, VALUE *argv, VALUE obj)
{
	VALUE retval;

	if (0 > argc || argc > 2)
	{
		rb_raise(rb_eArgError, "argument must be (category, value) or a hash thereof");
	}
	else if (argc == 2)
	{
		/*
		 * Assume Locale.setlocal(<category>, <locale>)
		 * eg; Locale.setlocale(Locale::LC_NUMERIC, 'en_GB')
		 */
		retval = rb_setlocale(argv[0], argv[1]);
	}
	else if (argc == 1)
	{
		/*
		 * Assume Locale.setlocal(<category-locale-hash>)
		 * eg; Locale.setlocale(
		 *       Locale::LC_COLLATE => 'en_US'
		 *       Locale::LC_NUMERIC => 'en_GB'
		 *       Locale::LC_CTYPE   => 'en_GB'
		 *       )
		 */
		Check_Type(argv[0], T_HASH);
		retval = rb_hash_new();
		st_foreach(RHASH(argv[0])->tbl, rb_setlocale_i, retval);
	}

	return retval;
}

static VALUE mLangInfo_langinfo(VALUE self, VALUE item_val)
{
	char* retval;

	Check_Type(item_val, T_FIXNUM);

	retval = nl_langinfo(NUM2INT(item_val));

	return rb_tainted_str_new2(retval);
}

void Init_locale()
{
	/* Locale */
	mLocale = rb_define_module("Locale");

	rb_define_const(mLocale, "LC_ALL",      INT2NUM(LC_ALL)      );
	rb_define_const(mLocale, "LC_COLLATE",  INT2NUM(LC_COLLATE)  );
	rb_define_const(mLocale, "LC_CTYPE",    INT2NUM(LC_CTYPE)    );
	rb_define_const(mLocale, "LC_MONETARY", INT2NUM(LC_MONETARY) );
	rb_define_const(mLocale, "LC_MESSAGES", INT2NUM(LC_MESSAGES) );
	rb_define_const(mLocale, "LC_NUMERIC",  INT2NUM(LC_NUMERIC)  );
	rb_define_const(mLocale, "LC_TIME",     INT2NUM(LC_TIME)     );

	rb_define_module_function(mLocale, "setlocale", mLocale_setlocale, -1);

	/* Locale::LangInfo */
	mLocale_LangInfo = rb_define_module_under(mLocale, "LangInfo");
	
	/* LC_TIME */
	rb_define_const(mLocale_LangInfo, "DAY_1", INT2NUM(DAY_1));
	rb_define_const(mLocale_LangInfo, "DAY_2", INT2NUM(DAY_2));
	rb_define_const(mLocale_LangInfo, "DAY_3", INT2NUM(DAY_3));
	rb_define_const(mLocale_LangInfo, "DAY_4", INT2NUM(DAY_4));
	rb_define_const(mLocale_LangInfo, "DAY_5", INT2NUM(DAY_5));
	rb_define_const(mLocale_LangInfo, "DAY_6", INT2NUM(DAY_6));
	rb_define_const(mLocale_LangInfo, "DAY_7", INT2NUM(DAY_7));

	rb_define_const(mLocale_LangInfo, "ABDAY_1", INT2NUM(ABDAY_1));
	rb_define_const(mLocale_LangInfo, "ABDAY_2", INT2NUM(ABDAY_2));
	rb_define_const(mLocale_LangInfo, "ABDAY_3", INT2NUM(ABDAY_3));
	rb_define_const(mLocale_LangInfo, "ABDAY_4", INT2NUM(ABDAY_4));
	rb_define_const(mLocale_LangInfo, "ABDAY_5", INT2NUM(ABDAY_5));
	rb_define_const(mLocale_LangInfo, "ABDAY_6", INT2NUM(ABDAY_6));
	rb_define_const(mLocale_LangInfo, "ABDAY_7", INT2NUM(ABDAY_7));

	rb_define_const(mLocale_LangInfo, "MON_1",  INT2NUM(MON_1));
	rb_define_const(mLocale_LangInfo, "MON_2",  INT2NUM(MON_2));
	rb_define_const(mLocale_LangInfo, "MON_3",  INT2NUM(MON_3));
	rb_define_const(mLocale_LangInfo, "MON_4",  INT2NUM(MON_4));
	rb_define_const(mLocale_LangInfo, "MON_5",  INT2NUM(MON_5));
	rb_define_const(mLocale_LangInfo, "MON_6",  INT2NUM(MON_6));
	rb_define_const(mLocale_LangInfo, "MON_7",  INT2NUM(MON_7));
	rb_define_const(mLocale_LangInfo, "MON_8",  INT2NUM(MON_8));
	rb_define_const(mLocale_LangInfo, "MON_9",  INT2NUM(MON_9));
	rb_define_const(mLocale_LangInfo, "MON_10", INT2NUM(MON_10));
	rb_define_const(mLocale_LangInfo, "MON_11", INT2NUM(MON_11));
	rb_define_const(mLocale_LangInfo, "MON_12", INT2NUM(MON_12));

	rb_define_const(mLocale_LangInfo, "ABMON_1",  INT2NUM(ABMON_1));
	rb_define_const(mLocale_LangInfo, "ABMON_2",  INT2NUM(ABMON_2));
	rb_define_const(mLocale_LangInfo, "ABMON_3",  INT2NUM(ABMON_3));
	rb_define_const(mLocale_LangInfo, "ABMON_4",  INT2NUM(ABMON_4));
	rb_define_const(mLocale_LangInfo, "ABMON_5",  INT2NUM(ABMON_5));
	rb_define_const(mLocale_LangInfo, "ABMON_6",  INT2NUM(ABMON_6));
	rb_define_const(mLocale_LangInfo, "ABMON_7",  INT2NUM(ABMON_7));
	rb_define_const(mLocale_LangInfo, "ABMON_8",  INT2NUM(ABMON_8));
	rb_define_const(mLocale_LangInfo, "ABMON_9",  INT2NUM(ABMON_9));
	rb_define_const(mLocale_LangInfo, "ABMON_10", INT2NUM(ABMON_10));
	rb_define_const(mLocale_LangInfo, "ABMON_11", INT2NUM(ABMON_11));
	rb_define_const(mLocale_LangInfo, "ABMON_12", INT2NUM(ABMON_12));

	rb_define_const(mLocale_LangInfo, "AM_STR", INT2NUM(AM_STR));
	rb_define_const(mLocale_LangInfo, "PM_STR", INT2NUM(PM_STR));

	rb_define_const(mLocale_LangInfo, "D_T_FMT", INT2NUM(D_T_FMT));
	rb_define_const(mLocale_LangInfo, "D_FMT", INT2NUM(D_FMT));
	rb_define_const(mLocale_LangInfo, "T_FMT", INT2NUM(T_FMT));
	rb_define_const(mLocale_LangInfo, "T_FMT_AMPM", INT2NUM(T_FMT_AMPM));

	rb_define_const(mLocale_LangInfo, "ERA", INT2NUM(ERA));
	rb_define_const(mLocale_LangInfo, "ERA_D_FMT", INT2NUM(ERA_D_FMT));
	rb_define_const(mLocale_LangInfo, "ALT_DIGITS", INT2NUM(ALT_DIGITS));
	rb_define_const(mLocale_LangInfo, "ERA_D_T_FMT", INT2NUM(ERA_D_T_FMT));
	rb_define_const(mLocale_LangInfo, "ERA_T_FMT", INT2NUM(ERA_T_FMT));

	/* LC_COLLATE */

	/* LC_CTYPE */
	rb_define_const(mLocale_LangInfo, "CODESET", INT2NUM(CODESET));

	/* LC_MONETARY */
	rb_define_const(mLocale_LangInfo, "INT_CURR_SYMBOL", INT2NUM(INT_CURR_SYMBOL));
	rb_define_const(mLocale_LangInfo, "CURRENCY_SYMBOL", INT2NUM(CURRENCY_SYMBOL));
	rb_define_const(mLocale_LangInfo, "MON_DECIMAL_POINT", INT2NUM(MON_DECIMAL_POINT));
	rb_define_const(mLocale_LangInfo, "MON_THOUSANDS_SEP", INT2NUM(MON_THOUSANDS_SEP));
	rb_define_const(mLocale_LangInfo, "MON_GROUPING", INT2NUM(MON_GROUPING));
	rb_define_const(mLocale_LangInfo, "POSITIVE_SIGN", INT2NUM(POSITIVE_SIGN));
	rb_define_const(mLocale_LangInfo, "NEGATIVE_SIGN", INT2NUM(NEGATIVE_SIGN));
	rb_define_const(mLocale_LangInfo, "INT_FRAC_DIGITS", INT2NUM(INT_FRAC_DIGITS));
	rb_define_const(mLocale_LangInfo, "FRAC_DIGITS", INT2NUM(FRAC_DIGITS));
	rb_define_const(mLocale_LangInfo, "P_CS_PRECEDES", INT2NUM(P_CS_PRECEDES));
	rb_define_const(mLocale_LangInfo, "P_SEP_BY_SPACE", INT2NUM(P_SEP_BY_SPACE));
	rb_define_const(mLocale_LangInfo, "N_CS_PRECEDES", INT2NUM(N_CS_PRECEDES));
	rb_define_const(mLocale_LangInfo, "N_SEP_BY_SPACE", INT2NUM(N_SEP_BY_SPACE));
	rb_define_const(mLocale_LangInfo, "P_SIGN_POSN", INT2NUM(P_SIGN_POSN));
	rb_define_const(mLocale_LangInfo, "N_SIGN_POSN", INT2NUM(N_SIGN_POSN));
	rb_define_const(mLocale_LangInfo, "CRNCYSTR", INT2NUM(CRNCYSTR));
	rb_define_const(mLocale_LangInfo, "INT_P_CS_PRECEDES", INT2NUM(INT_P_CS_PRECEDES));
	rb_define_const(mLocale_LangInfo, "INT_P_SEP_BY_SPACE", INT2NUM(INT_P_SEP_BY_SPACE));
	rb_define_const(mLocale_LangInfo, "INT_N_CS_PRECEDES", INT2NUM(INT_N_CS_PRECEDES));
	rb_define_const(mLocale_LangInfo, "INT_N_SEP_BY_SPACE", INT2NUM(INT_N_SEP_BY_SPACE));
	rb_define_const(mLocale_LangInfo, "INT_P_SIGN_POSN", INT2NUM(INT_P_SIGN_POSN));
	rb_define_const(mLocale_LangInfo, "INT_N_SIGN_POSN", INT2NUM(INT_N_SIGN_POSN));

	/* LC_NUMERIC */
	rb_define_const(mLocale_LangInfo, "DECIMAL_POINT", INT2NUM(DECIMAL_POINT));
	rb_define_const(mLocale_LangInfo, "RADIXCHAR", INT2NUM(RADIXCHAR));
	rb_define_const(mLocale_LangInfo, "THOUSANDS_SEP", INT2NUM(THOUSANDS_SEP));
	rb_define_const(mLocale_LangInfo, "THOUSEP", INT2NUM(THOUSEP));
	rb_define_const(mLocale_LangInfo, "GROUPING", INT2NUM(GROUPING));

	/* LC_MESSAGES */
	rb_define_const(mLocale_LangInfo, "NOEXPR", INT2NUM(NOEXPR));
	rb_define_const(mLocale_LangInfo, "YESEXPR", INT2NUM(YESEXPR));
	rb_define_const(mLocale_LangInfo, "NOSTR", INT2NUM(NOSTR));
	rb_define_const(mLocale_LangInfo, "YESSTR", INT2NUM(YESSTR));

	rb_define_module_function(mLocale_LangInfo, "langinfo", mLangInfo_langinfo, 1);
}
