//////////////////////////////////////////////////////////////////////////////
//
//    KEXIFUTILS.CPP
//
//    Copyright (C) 2002-2004 Renchi Raju <renchi at pooh.tam.uiuc.edu>
//                            Gilles CAULIER <caulier dot gilles at free.fr>
//                            Ralf Hoelzer <kde at ralfhoelzer.com>
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <kdebug.h>
#include <qfile.h>

extern "C" {
#include <libexif/exif-data.h>
#include <libexif/exif-utils.h>
#include <stdlib.h>
}

#include "kexififd.h"
#include "kexifdata.h"
#include "kexifentry.h"

class KExifDataPriv
{
public:
    
    ExifData               *exifData;
    QString                 exifByteOrder;
    QString                 userComment;
    QImage                  thumbnail;
    QPtrList<KExifIfd>      ifdList;
};

KExifData::KExifData()
{
    d = new KExifDataPriv;
    
    d->ifdList.setAutoDelete(true);
    d->exifData      = 0;
    d->exifByteOrder = "";
    d->userComment   = "";
}

KExifData::~KExifData()
{
    d->ifdList.clear();

    if (d->exifData) {
        exif_data_unref(d->exifData);
        d->exifData = 0;
    }

    delete d;
}

bool KExifData::readFromFile(const QString& filename)
{
    if (d->exifData) {
        exif_data_unref(d->exifData);
        d->exifData = 0;
    }

    d->exifData = exif_data_new_from_file(QFile::encodeName(filename));

    if (!d->exifData) 
        return false;

    exif_data_ref(d->exifData);

    ExifByteOrder order;
    order = exif_data_get_byte_order(d->exifData);
    d->exifByteOrder = QString(exif_byte_order_get_name(order));

    // -- Load IFDs ------------------------------------------

    if (EXIF_IFD_COUNT) {

        d->ifdList.clear();

        for (unsigned int i=0; i < EXIF_IFD_COUNT; i++) {

            QString ifdName(exif_ifd_get_name((ExifIfd)i));
            d->ifdList.append(new KExifIfd(ifdName, d->exifData->ifd[i]));
        }
    }

    // -- Load thumbnail -------------------------------------

    if (d->exifData->data) {
        d->thumbnail.loadFromData(d->exifData->data,
                                d->exifData->size);
    }

    return true;
}

bool KExifData::readFromData(char* data, int size)
{
    if (d->exifData) {
        exif_data_unref(d->exifData);
        d->exifData = 0;
    }

    d->exifData = exif_data_new_from_data((const unsigned char*)data, size);

    if (!d->exifData)
    {
        kdDebug() << "Data has no Exif Content" << endl;
        return false;
    }

    exif_data_ref(d->exifData);

    ExifByteOrder order;
    order = exif_data_get_byte_order(d->exifData);
    d->exifByteOrder = QString(exif_byte_order_get_name(order));

    // -- Load IFDs ------------------------------------------

    if (EXIF_IFD_COUNT) {

        d->ifdList.clear();

        for (unsigned int i=0; i < EXIF_IFD_COUNT; i++) {

            QString ifdName(exif_ifd_get_name((ExifIfd)i));
            d->ifdList.append(new KExifIfd(ifdName, d->exifData->ifd[i]));
        }
    }

    // -- Load thumbnail -------------------------------------

    if (d->exifData->data) {
        d->thumbnail.loadFromData(d->exifData->data,
                                  d->exifData->size);
    }

    return true;
}

QImage KExifData::getThumbnail() const
{
    return d->thumbnail;
}

QString KExifData::getUserComment() const
{
   if(d->userComment == "")
   {
       KExifIfd* ifd = 0;
       for (ifd = d->ifdList.first(); ifd; ifd = d->ifdList.next())
       {
         if( ifd->getName() == "EXIF" )
         {
            QPtrListIterator<KExifEntry> entryIterator(ifd->entryList());

            KExifEntry *exifEntry = 0;

            while( (exifEntry = entryIterator.current()) != 0 )
            {
               ++entryIterator;
               if(exifEntry->getName() == "UserComment")
                   d->userComment = exifEntry->getValue();
            }
         }
      }
   }

   return d->userComment;
}

KExifData::ImageOrientation KExifData::getImageOrientation() const
{
   if (d->exifData) {
      //get byte order for later reading
      ExifByteOrder o=exif_data_get_byte_order(d->exifData);
      //retrieve Ifd0
      ExifContent *ifd0=d->exifData->ifd[EXIF_IFD_0];
      //look for entry in ifd0
      ExifEntry *entry=exif_content_get_entry(ifd0, EXIF_TAG_ORIENTATION);

      if (entry && entry->format == EXIF_FORMAT_SHORT && entry->components == 1) {
         //read value
         ExifShort s = exif_get_short(entry->data, o);

         return (KExifData::ImageOrientation)s;
      }
   }

   // if something went wrong, return NORMAL
   return KExifData::NORMAL;
   
}

QDateTime KExifData::getExifDateTime() const
{
   if (d->exifData) {
      //retrieve Ifd0
      ExifContent *ifd0=d->exifData->ifd[EXIF_IFD_0];
      //look for entry in ifd0
      ExifEntry *entry=exif_content_get_entry(ifd0, EXIF_TAG_DATE_TIME);

      if (entry)
      {
#ifdef HAVE_EXIF06
          char val[1024];
          exif_entry_get_value(entry, val, 1023);
          // just in case we don't get a null-terminated string
          val[1024] = '\0'; 
          QString str((const char*)&val);
#else
          QString str(exif_entry_get_value(entry));
#endif
          return QDateTime::fromString(str,Qt::ISODate);
      }
   }

   return QDateTime();
}

QPtrList<KExifIfd> KExifData::ifdList() const
{
    return d->ifdList;
}

void KExifData::saveFile(const QString& filename)
{
    FILE *f;
    f = fopen (QFile::encodeName(filename), "wb");
    fwrite (d->exifData->data, 1, d->exifData->size, f);
    fclose (f);
}
