#ifndef _POSTTHREAD_H_
#define _POSTTHREAD_H_

#include "talk/base/thread.h"

namespace cricket {

class PostThread : public Thread, public MessageHandler {

public:

  // Called by the main thread to start the worker thread
  virtual void Start() {
    EnterExit ee(this);
    refcount_ += 1;
    Thread::Start();
    Post(this, MSG_START);
  }

  // Called by the main thread when the worker is no longer needed
  void Release() {
    EnterExit ee(this);
    Halt();
    refcount_ -= 1;
    if (handler_notify_)
      main_thread_->Clear(handler_notify_);
    handler_notify_ = NULL;
  }

protected:
  
  PostThread(MessageHandler *handler_notify, uint32 id_notify, MessageData *data_notify = NULL) :
    refcount_(1),
    is_finished_(false),
    handler_notify_(handler_notify),
    id_notify_(id_notify),
    data_notify_(data_notify) {
    main_thread_ = ThreadManager::CurrentThread();
  }

  virtual ~PostThread() {
    if (handler_notify_)
      main_thread_->Clear(handler_notify_);
  }

  // This method is executed on the worker thread
  virtual void Run() = 0;
  
  // Called from the main thread to interrupt any long-running work
  // on the worker thread when the main thread Releases.  Short-running
  // threads need not do anything.
  virtual void Halt() {};

  virtual void OnMessage(Message *msg) {
    switch (msg->message_id) {
      case MSG_START:
        Run();
        EnterExit ee(this);
        is_finished_ = true;
        if (handler_notify_)
          main_thread_->Post(handler_notify_, id_notify_, data_notify_);
        refcount_ -= 1;
     }
   }
    
private:
  enum {MSG_START};
  class EnterExit {
    friend class PostThread;
    
    PostThread * t_;
    
    EnterExit(PostThread * t) : t_(t) {
      t_->cs_.Enter();
      t_->refcount_ += 1;
    }
    ~EnterExit() {
      bool d = (0 == (--(t_->refcount_)));
      t_->cs_.Leave();
      if (d)
        delete t_;
    }
  };

  friend class EnterExit;
  
  CriticalSection cs_;
  int refcount_;
  bool is_finished_;
  MessageHandler *handler_notify_;
  uint32 id_notify_;
  MessageData *data_notify_;
  Thread *main_thread_;
  
  bool IsThreadFinished() {
    EnterExit ee(this);
    return is_finished_;
  }

};

} // namespace cricket 

#endif


