/* KeyPool.java -- A set of ephemeral key pairs.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>
   Parts copyright (C) 2001,2002,2003  Free Software Foundation, Inc.

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.math.BigInteger;
import java.security.KeyPair;
import java.security.Security;
import java.util.LinkedList;
import javax.crypto.spec.DHParameterSpec;

import gnu.crypto.hash.HashFactory;
import gnu.crypto.hash.IMessageDigest;
import gnu.crypto.prng.IRandom;
import gnu.crypto.prng.LimitReachedException;
import gnu.crypto.util.Prime;

final class KeyPool extends Thread
{

  // Fields.
  // -------------------------------------------------------------------------

  private static final KeyPool POOL = new KeyPool();

  static final int POOL_SIZE = 5;

  private static final BigInteger ZERO = BigInteger.ZERO;
  private static final BigInteger ONE = BigInteger.ONE;
  private static final BigInteger TWO = BigInteger.valueOf(2L);
  private static final BigInteger E = BigInteger.valueOf(65537L);

  private static final BigInteger DH_G = TWO;

  static
  {
    POOL.setPriority(MIN_PRIORITY);
    POOL.setDaemon(true);
    POOL.start();
  }

  private int dhPoolSize;
  private int rsaPoolSize;
  private final LinkedList dhPool;
  private final LinkedList rsaPool;
  private boolean started;
  private final IRandom prng;

  // Constructor.
  // -------------------------------------------------------------------------

  private KeyPool()
  {
    super(KeyPool.class.getName());
    try
      {
        dhPoolSize =
          Integer.parseInt(Security.getProperty("jessie.keypool.dh.size"));
      }
    catch (Exception x)
      {
        dhPoolSize = POOL_SIZE;
      }
    try
      {
        rsaPoolSize =
          Integer.parseInt(Security.getProperty("jessie.keypool.rsa.size"));
      }
    catch (Exception x)
      {
        rsaPoolSize = POOL_SIZE;
      }
    dhPool = new LinkedList();
    rsaPool = new LinkedList();
    prng = new SynchronizedRandom(EntropyTools.getSeededRandom());
    started = false;
  }

  // Class methods.
  // -------------------------------------------------------------------------

  static KeyPair getDHKeyPair()
  {
    synchronized (POOL.dhPool)
      {
        if (POOL.dhPool.size() < 1)
          {
            return POOL.generateDHKeyPair(defaultParams());
          }
        return (KeyPair) POOL.dhPool.removeFirst();
      }
  }

  static KeyPair getDHKeyPair(DHParameterSpec spec)
  {
    return POOL.generateDHKeyPair(spec);
  }

  static KeyPair getRSAKeyPair()
  {
    synchronized (POOL.rsaPool)
      {
        if (POOL.rsaPool.size() < 1)
          {
            return POOL.generateRSAKeyPair();
          }
        return (KeyPair) POOL.rsaPool.removeFirst();
      }
  }

  private static BigInteger sha(BigInteger x)
  {
    IMessageDigest sha = HashFactory.getInstance("SHA-1");
    byte[] mag = x.toByteArray();
    sha.update(mag, 0, mag.length);
    return new BigInteger(1, sha.digest());
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  public void run()
  {
    if (started)
      throw new IllegalStateException("already started");
    started = true;
    while (true)
      {
        synchronized (dhPool)
          {
            if (dhPool.size() < dhPoolSize)
              dhPool.addLast(generateDHKeyPair(defaultParams()));
          }
        try { sleep(600000); } catch (InterruptedException ie) { }
        synchronized (rsaPool)
          {
            if (rsaPool.size() < rsaPoolSize)
              rsaPool.addLast(generateRSAKeyPair());
          }
        try { sleep(600000); } catch (InterruptedException ie) { }
      }
  }

  private KeyPair generateDHKeyPair(DHParameterSpec spec)
  {
    BigInteger x, y;
    int l = (spec.getL() > 0) ? spec.getL() : spec.getP().bitLength();
    byte[] buf = new byte[l >>> 3];
    do
      {
        nextBytes(buf);
        x = new BigInteger(1, buf);
      }
    while (x.compareTo(spec.getP()) >= 0);
    y = spec.getG().modPow(x, spec.getP());
    return new KeyPair(new JessieDHPublicKey(spec, y),
                       new JessieDHPrivateKey(spec, x));
  }

  private KeyPair generateRSAKeyPair()
  {
    BigInteger p, q, n, d;

    // Simplified version of GNU Crypto's RSAKeyPairGenerator.

    int M = 256;
    BigInteger lower = TWO.pow(255);
    BigInteger upper = TWO.pow(256).subtract(ONE);
    byte[] kb = new byte[32];
    while (true)
      {
        nextBytes(kb);
        p = new BigInteger(1, kb).setBit(0);
        if (p.compareTo(lower) >= 0 && p.compareTo(upper) <= 0 &&
            Prime.isProbablePrime(p) && p.gcd(E).equals(ONE))
          break;
      }

    while (true)
      {
        nextBytes(kb);
        q = new BigInteger(1, kb).setBit(0);
        n = q.multiply(p);
        if (n.bitLength() == 512 && Prime.isProbablePrime(q) &&
            q.gcd(E).equals(ONE))
          break;
      }

    d = E.modInverse(p.subtract(ONE).multiply(q.subtract(ONE)));

    return new KeyPair(new JessieRSAPublicKey(n, E),
                       new JessieRSAPrivateKey(n, d));
  }

  private void nextBytes(byte[] buf)
  {
    try
      {
        prng.nextBytes(buf, 0, buf.length);
      }
    catch (LimitReachedException lre)
      {
        throw new Error(lre);
      }
  }

  private static DHParameterSpec defaultParams()
  {
    BigInteger p = DiffieHellman.GROUP_18;
    String group = Security.getProperty("jessie.keypool.dh.group");
    if (group != null)
      {
        group = group.trim();
        if (group.equals("1"))
          {
            p = DiffieHellman.GROUP_1;
          }
        else if (group.equals("2"))
          {
            p = DiffieHellman.GROUP_2;
          }
        else if (group.equals("5"))
          {
            p = DiffieHellman.GROUP_5;
          }
        else if (group.equals("14"))
          {
            p = DiffieHellman.GROUP_14;
          }
        else if (group.equals("15"))
          {
            p = DiffieHellman.GROUP_15;
          }
        else if (group.equals("16"))
          {
            p = DiffieHellman.GROUP_16;
          }
        else if (group.equals("17"))
          {
            p = DiffieHellman.GROUP_17;
          }
        else if (group.equals("18"))
          {
            p = DiffieHellman.GROUP_18;
          }
      }
    return new DHParameterSpec(p, DH_G, p.bitLength());
  }
}
