/* DigestOutputStream.java -- digesting output stream.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

import gnu.crypto.hash.IMessageDigest;

final class DigestOutputStream extends FilterOutputStream
{

  // Fields.
  // -------------------------------------------------------------------------

  private IMessageDigest md5, sha;
  private boolean digesting;

  // Constructor.
  // -------------------------------------------------------------------------

  DigestOutputStream(OutputStream out, IMessageDigest md5, IMessageDigest sha)
  {
    super(out);
    this.md5 = md5;
    this.sha = sha;
    digesting = true;
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  void setDigesting(boolean digesting)
  {
    this.digesting = digesting;
  }

  public void write(int b) throws IOException
  {
    if (digesting)
      {
        md5.update((byte) b);
        sha.update((byte) b);
      }
    out.write(b);
  }

  public void write(byte[] buf) throws IOException
  {
    write(buf, 0, buf.length);
  }

  public void write(byte[] buf, int off, int len) throws IOException
  {
    if (buf == null)
      {
        throw new NullPointerException();
      }
    if (off < 0 || len < 0 || off+len > buf.length)
      {
        throw new ArrayIndexOutOfBoundsException();
      }
    if (digesting)
      {
        md5.update(buf, off, len);
        sha.update(buf, off, len);
      }
    out.write(buf, off, len);
  }
}
