/* Context.java -- SSLContext implementation.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.File;
import java.io.InputStream;

import java.security.InvalidAlgorithmParameterException;
import java.security.KeyStoreException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.sql.SQLException;

import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContextSpi;
import javax.net.ssl.SSLSessionContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509KeyManager;
import javax.net.ssl.X509TrustManager;

import org.metastatic.jessie.NullManagerParameters;
import org.metastatic.jessie.SRPTrustManager;
import org.metastatic.jessie.StaticTrustAnchors;

/**
 * This is Jessie's implementation of a {@link javax.net.ssl.SSLContext}
 * engine, and is available under the algorithm names ``SSLv3'', ``SSL'',
 * ``TLSv1'', and ``TLS''.
 */
public final class Context extends SSLContextSpi
{

  // Fields.
  // -------------------------------------------------------------------------

  private SessionContext clientSessions;
  private SessionContext serverSessions;
  private X509KeyManager keyManager;
  private X509TrustManager trustManager;
  private SRPTrustManager srpTrustManager;
  private SecureRandom random;

  // Constructor.
  // -------------------------------------------------------------------------

  public Context()
  {
    String codec = Security.getProperty("jessie.clientSessionContext.codec");
    if (codec == null)
      {
        codec = "none";
      }
    if (codec.equalsIgnoreCase("xml"))
      {
        boolean compress = false;
        String gzip = Security.getProperty("jessie.clientSessionContext.xml.compress");
        if (gzip != null)
          {
            compress = gzip.equalsIgnoreCase("true");
          }
        String file = Security.getProperty("jessie.clientSessionContext.xml.file");
        if (file == null)
          {
            throw new IllegalArgumentException("no XML session store specified");
          }
        String pass = Security.getProperty("jessie.clientSessionContext.xml.password");
        if (pass == null)
          {
            pass = "";
          }
        try
          {
            clientSessions =
              new XMLSessionContext(new File(file), pass.toCharArray(), compress);
          }
        catch (Exception ex)
          {
            throw new IllegalArgumentException(ex.toString());
          }
      }
    else if (codec.equalsIgnoreCase("jdbc"))
      {
        String url = Security.getProperty("jessie.clientSessionContext.jdbc.url");
        try
          {
            clientSessions = new JDBCSessionContext(url,
              Security.getProperty("jessie.clientSessionContext.jdbc.user"),
              Security.getProperty("jessie.clientSessionContext.jdbc.password"));
          }
        catch (SQLException sqle)
          {
            throw new IllegalArgumentException(sqle.toString());
          }
      }
    else
      {
        clientSessions = new SessionContext();
      }

    codec = Security.getProperty("jessie.serverSessionContext.codec");
    if (codec == null)
      {
        codec = "null";
      }
    if (codec.equalsIgnoreCase("xml"))
      {
        boolean compress = false;
        String gzip = Security.getProperty("jessie.serverSessionContext.xml.compress");
        if (gzip != null)
          {
            compress = gzip.equalsIgnoreCase("true");
          }
        String file = Security.getProperty("jessie.serverSessionContext.xml.file");
        if (file == null)
          {
            throw new IllegalArgumentException("no XML session store specified");
          }
        String pass = Security.getProperty("jessie.serverSessionContext.xml.password");
        if (pass == null)
          {
            pass = "";
          }
        try
          {
            serverSessions =
              new XMLSessionContext(new File(file), pass.toCharArray(), compress);
          }
        catch (Exception ex)
          {
            throw new IllegalArgumentException(ex.toString());
          }
      }
    else if (codec.equalsIgnoreCase("jdbc"))
      {
        String url = Security.getProperty("jessie.serverSessionContext.jdbc.url");
        try
          {
            serverSessions = new JDBCSessionContext(url,
              Security.getProperty("jessie.serverSessionContext.jdbc.user"),
              Security.getProperty("jessie.serverSessionContext.jdbc.password"));
          }
        catch (SQLException sqle)
          {
            throw new IllegalArgumentException(sqle.toString());
          }
      }
    else
      {
        serverSessions = new SessionContext();
      }
  }

  // Engine methods.
  // -------------------------------------------------------------------------

  protected SSLSessionContext engineGetClientSessionContext()
  {
    return clientSessions;
  }

  protected SSLSessionContext engineGetServerSessionContext()
  {
    return serverSessions;
  }

  protected javax.net.ssl.SSLServerSocketFactory engineGetServerSocketFactory()
  {
    if (keyManager == null || (trustManager == null && srpTrustManager == null)
        || random == null)
      {
        throw new IllegalStateException();
      }
    return new SSLServerSocketFactory(trustManager, srpTrustManager, keyManager,
                                      random, serverSessions);
  }

  protected javax.net.ssl.SSLSocketFactory engineGetSocketFactory()
  {
    if (keyManager == null || trustManager == null || random == null)
      {
        throw new IllegalStateException();
      }
    return new SSLSocketFactory(trustManager, keyManager, random, clientSessions);
  }

  protected void engineInit(KeyManager[] keyManagers,
                            TrustManager[] trustManagers, SecureRandom random)
    throws KeyManagementException
  {
    keyManager = null;
    trustManager = null;
    srpTrustManager = null;
    if (keyManagers != null)
      {
        for (int i = 0; i < keyManagers.length; i++)
          {
            if (keyManagers[i] instanceof X509KeyManager)
              {
                keyManager = (X509KeyManager) keyManagers[i];
                break;
              }
          }
      }
    if (keyManager == null)
      {
        keyManager = defaultKeyManager();
      }
    if (trustManagers != null)
      {
        for (int i = 0; i < trustManagers.length; i++)
          {
            if (trustManagers[i] instanceof X509TrustManager)
              {
                if (trustManager == null)
                  {
                    trustManager = (X509TrustManager) trustManagers[i];
                  }
              }
            else if (trustManagers[i] instanceof SRPTrustManager)
              {
                if (srpTrustManager == null)
                  {
                    srpTrustManager = (SRPTrustManager) trustManagers[i];
                  }
              }
          }
      }
    if (trustManager == null && srpTrustManager == null)
      {
        trustManager = defaultTrustManager();
      }
    if (random != null)
      {
        this.random = random;
      }
    else
      {
        this.random = defaultRandom();
      }
  }

  // Own methods.
  // -------------------------------------------------------------------------

  private X509KeyManager defaultKeyManager() throws KeyManagementException
  {
    KeyManagerFactory fact = null;
    try
      {
        fact = KeyManagerFactory.getInstance("JessieX509", "Jessie");
      }
    catch (NoSuchAlgorithmException nsae)
      {
        throw new KeyManagementException();
      }
    catch (NoSuchProviderException nspe)
      {
        throw new KeyManagementException();
      }
    try
      {
        fact.init(null, null);
        return (X509KeyManager) fact.getKeyManagers()[0];
      }
    catch (NoSuchAlgorithmException nsae) { }
    catch (KeyStoreException kse) { }
    catch (UnrecoverableKeyException uke) { }
    catch (IllegalStateException ise) { }

    try
      {
        fact.init(new NullManagerParameters());
        return (X509KeyManager) fact.getKeyManagers()[0];
      }
    catch (Exception shouldNotHappen)
      {
        throw new Error(shouldNotHappen.toString());
      }
  }

  private X509TrustManager defaultTrustManager() throws KeyManagementException
  {
    try
      {
        TrustManagerFactory fact =
          TrustManagerFactory.getInstance("JessieX509", "Jessie");
        fact.init(StaticTrustAnchors.CA_CERTS);
        return (X509TrustManager) fact.getTrustManagers()[0];
      }
    catch (NoSuchAlgorithmException nsae)
      {
        throw new KeyManagementException(nsae.toString());
      }
    catch (NoSuchProviderException nspe)
      {
        throw new KeyManagementException(nspe.toString());
      }
    catch (InvalidAlgorithmParameterException kse)
      {
        throw new KeyManagementException(kse.toString());
      }
  }

  private SecureRandom defaultRandom() throws KeyManagementException
  {
    String alg = Security.getProperty("jessie.secure.random");
    if (alg == null)
      {
        alg = "SHA1PRNG";
      }
    SecureRandom rand = null;
    try
      {
        rand = SecureRandom.getInstance(alg);
      }
    catch (NoSuchAlgorithmException nsae)
      {
        throw new KeyManagementException(nsae.toString());
      }

    rand.setSeed(EntropyTools.generateFastSeed(20));
    return rand;
  }
}
