/* HttpsURLConnection.java -- HTTPS URL connection.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>
   Parts Copyright (C) 1998,2002  Free Software Foundation, Inc.

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.https;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;

import java.net.URL;

import java.security.Security;
import java.security.cert.X509Certificate;

import java.text.DateFormat;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSocket;

import org.metastatic.jessie.provider.Jessie;

class HttpsURLConnection extends javax.net.ssl.HttpsURLConnection
{

  // Constants and fields.
  // -------------------------------------------------------------------------

  public static final String USER_AGENT = "Jessie/" + Jessie.VERSION;
  public static final int HTTPS_PORT = 443;

  private ArrayList requestProps;
  private ArrayList headers;
  private SSLSocket socket;
  private InputStream in;
  private OutputStream out;
  private ByteArrayOutputStream bufferedOut;
  private boolean requestSent = false;

  // Constructor.
  // -------------------------------------------------------------------------

  protected HttpsURLConnection(URL url) throws IOException
  {
    super(url);
    doOutput = false;
    headers = new ArrayList(10);
  }

  // HTTPS methods.
  // -------------------------------------------------------------------------

  public String getCipherSuite()
  {
    if (!connected)
      throw new IllegalStateException("not yet connected");
    return socket.getSession().getCipherSuite();
  }

  public java.security.cert.Certificate[] getLocalCertificates()
  {
    if (!connected)
      throw new IllegalStateException("not yet connected");
    return socket.getSession().getLocalCertificates();
  }

  public java.security.cert.Certificate[] getPeerCertificates()
    throws SSLPeerUnverifiedException
  {
    if (!connected)
      throw new IllegalStateException("not yet connected");
    return socket.getSession().getPeerCertificates();
  }

  // HttpURLConnection methods.
  // -------------------------------------------------------------------------

  public void disconnect()
  {
    if (!connected)
      return;
    try
      {
        socket.close();
      }
    catch (IOException ioe)
      {
      }
  }

  public boolean usingProxy()
  {
    return false; // XXX
  }

  // URLConnection methods.
  // -------------------------------------------------------------------------

  public synchronized void connect() throws IOException
  {
    String host = url.getHost();
    int port = url.getPort();
    if (port == -1)
      port = HTTPS_PORT;
    socket = (SSLSocket) getSSLSocketFactory().createSocket(host, port);
    String timeout = Security.getProperty("jessie.https.timeout");
    if (timeout != null)
      {
        try { socket.setSoTimeout(Integer.parseInt(timeout)); }
        catch (Exception x) { }
      }
    socket.startHandshake();
    try
      {
        X509Certificate cert = (X509Certificate)
          socket.getSession().getPeerCertificates()[0];
        if (!checkHostname(host, cert) &&
            !hostnameVerifier.verify(host, socket.getSession()))
          throw new SSLPeerUnverifiedException("hostname mismatch");
      }
    catch (SSLPeerUnverifiedException spue)
      {
        if (!hostnameVerifier.verify(host, socket.getSession()))
          throw spue;
      }

    out = socket.getOutputStream();
    connected = true;
  }

  public synchronized InputStream getInputStream() throws IOException
  {
    if (in != null)
      return in;
    if (!connected)
      connect();
    in = socket.getInputStream();
    sendRequest();
    receiveReply();
    return in;
  }

  public synchronized OutputStream getOutputStream() throws IOException
  {
    if (!doOutput)
      throw new IOException("not set up for output");
    if (!method.equals("POST"))
      setRequestMethod("POST");
    if (!connected)
      throw new IOException("not connected");
    if (bufferedOut == null)
      bufferedOut = new ByteArrayOutputStream(256);
    return bufferedOut;
  }

  // The header situatuon is totally FUBAR, so we are doing this ourselves.

  public String getContentEncoding()
  {
    return getHeaderField("Content-Encoding");
  }

  public int getContentLength()
  {
    return getHeaderFieldInt("Content-Length", -1);
  }

  public String getContentType()
  {
    return getHeaderField("Content-Type");
  }

  public long getExpiration()
  {
    return getHeaderFieldDate("Expires", 0L);
  }

  public int getHeaderFieldInt(String header, int def)
  {
    try
      {
        return Integer.parseInt(getHeaderField(header));
      }
    catch (NumberFormatException nfe)
      {
        return def;
      }
  }

  public long getHeaderFieldDate(String header, long def)
  {
    try
      {
        DateFormat f = DateFormat.getDateInstance(DateFormat.LONG);
        f.setLenient(true);
        return f.parse(getHeaderField(header)).getTime();
      }
    catch (Exception e)
      {
        return def;
      }
  }

  public long getIfModifiedSince()
  {
    return getHeaderFieldDate("If-Modified-Since", 0L);
  }

  public Map getHeaderFields()
  {
    HashMap map = new HashMap(headers.size());
    for (int i = 0; i < headers.size(); i++)
      {
        Header h = (Header) headers.get(i);
        if (map.containsKey(h.getKey()))
          {
            ((List) map.get(h.getKey())).add(h.getValue());
          }
        else
          {
            LinkedList l = new LinkedList();
            l.add(h.getValue());
            map.put(h.getKey(), h.getValue());
          }
      }
    for (Iterator it = map.entrySet().iterator(); it.hasNext(); )
      {
        Map.Entry e = (Map.Entry) it.next();
        List l = (List) e.getValue();
        e.setValue(Collections.unmodifiableList(l));
      }
    return Collections.unmodifiableMap(map);
  }

  public String getHeaderField(int n)
  {
    if (!connected)
      throw new IllegalStateException("not connected");
    if (n < 0 || n > headers.size())
      return null;
    return (String) ((Header) headers.get(n)).getValue();
  }

  public String getHeaderField(String key)
  {
    if (!connected)
      throw new IllegalStateException("not connected");
    for (Iterator it = headers.iterator(); it.hasNext(); )
      {
        Header h = (Header) it.next();
        if (((String) h.getKey()).equalsIgnoreCase(key))
          return (String) h.getValue();
      }
    return null;
  }

  public String getHeaderFieldKey(int n)
  {
    if (!connected)
      throw new IllegalStateException("not connected");
    if (n < 0 || n > headers.size())
      return null;
    return (String) ((Header) headers.get(n)).getKey();
  }

  public void addRequestProperty(String name, String value)
  {
    if (requestSent)
      throw new IllegalStateException("already connected");
    if (name == null || value == null)
      throw new NullPointerException();
    if (name.trim().length() == 0 || value.trim().length() == 0)
      throw new IllegalArgumentException();
    if (requestProps == null)
      requestProps = new ArrayList(10);
    for (Iterator it = requestProps.iterator(); it.hasNext(); )
      {
        Header h = (Header) it.next();
        if (((String) h.getKey()).equalsIgnoreCase(name))
          {
            h.setValue(h.getValue() + ", " + value);
            return;
          }
      }
    requestProps.add(new Header(name, value));
  }

  public void setRequestProperty(String name, String value)
  {
    if (requestSent)
      throw new IllegalStateException("already connected");
    if (name == null || value == null)
      throw new NullPointerException();
    if (name.trim().length() == 0 || value.trim().length() == 0)
      throw new IllegalArgumentException();
    if (requestProps == null)
      requestProps = new ArrayList(10);
    for (Iterator it = requestProps.iterator(); it.hasNext(); )
      {
        Header h = (Header) it.next();
        if (((String) h.getKey()).equalsIgnoreCase(name))
          {
            it.remove();
            break;
          }
      }
    requestProps.add(new Header(name, value));
  }

  public String getRequestProperty(String name)
  {
    if (requestSent)
      throw new IllegalStateException("already connected");
    if (name == null)
      return null;
    for (Iterator it = requestProps.iterator(); it.hasNext(); )
      {
        Header h = (Header) it.next();
        if (((String) h.getKey()).equalsIgnoreCase(name))
          return (String) h.getValue();
      }
    return null;
  }

  public Map getRequestProperties()
  {
    if (requestSent)
      throw new IllegalStateException("already connected");
    HashMap map = new HashMap(requestProps.size());
    for (Iterator it = requestProps.iterator(); it.hasNext(); )
      {
        Header h = (Header) it.next();
        map.put(h.getKey(), Collections.singletonList(h.getValue()));
      }
    return Collections.unmodifiableMap(map);
  }

  // Own methods.
  // -------------------------------------------------------------------------

  private boolean checkHostname(String host, X509Certificate cert)
  {
    try
      {
        Collection altNames = cert.getSubjectAlternativeNames();
        Integer two = new Integer(2);
        if (altNames != null)
          {
            for (Iterator it = altNames.iterator(); it.hasNext(); )
              {
                String dnsName = null;
                List name = (List) it.next();
                if (!name.get(0).equals(two))
                  continue;
                dnsName = (String) name.get(1);
                if (host.equalsIgnoreCase(dnsName))
                  return true;
              }
          }
        return false;
      }
    catch (Exception x)
      {
        return false;
      }
  }

  private void sendRequest() throws IOException
  {
    BufferedOutputStream bout = new BufferedOutputStream(out);
    PrintStream httpout = new PrintStream(bout);
    httpout.print(getRequestMethod() + " " + getURL().getFile() + " HTTP/1.1\r\n");
    System.out.println(getRequestMethod() + " " + getURL().getFile() + " HTTP/1.1");

    if (getRequestProperty("Host") == null)
      setRequestProperty("Host", getURL().getHost());
    if (getRequestProperty("Connection") == null)
      setRequestProperty("Connection", "Close");
    if (getRequestProperty("User-Agent") == null)
      setRequestProperty("User-Agent", USER_AGENT);
    else
      setRequestProperty("User-Agent", getRequestProperty("User-Agent") + " "
                         + USER_AGENT);
    if (getRequestProperty("Accept") == null)
      setRequestProperty("Accept", "*/*");
    if (getRequestProperty("Content-type") == null)
      setRequestProperty("Content-type", "application/x-www-form-urlencoded");

    Iterator it = requestProps.iterator();
    while (it.hasNext())
      {
        Header h = (Header) it.next();
        String name = (String) h.getKey();
        String value = (String) h.getValue();
        if (name.length() + value.length() + 2 < 72)
          {
            httpout.print(name + ": " + value + "\r\n");
            System.out.println(name + ": " + value);
          }
        else
          {
            httpout.print(name + ": ");
            int idx = name.length() + 2;
            httpout.print(value.substring(0, idx));
            System.out.print(name + ": " + value.substring(0, idx));
            value = value.substring(idx + 1);
            while (value != null)
              {
                httpout.print("\t");
                idx = Math.min(72, value.length());
                httpout.print(value.substring(0, idx));
                System.out.print(value.substring(0, idx));
                if (idx == value.length())
                  value = null;
                else
                  value = value.substring(idx + 1);
              }
            httpout.print("\r\n");
          }
      }

    if (bufferedOut != null)
      {
        httpout.print("Content-type: application/x-www-form-urlencoded\r\n");
        httpout.print("Content-type: " + bufferedOut.size() + "\r\n");
      }

    httpout.print("\r\n");
    httpout.flush();

    if (bufferedOut != null)
      {
        bufferedOut.writeTo(bout);
        bout.flush();
      }

    requestSent = true;
  }

  private void receiveReply() throws IOException
  {
    String line = readLine();
    System.out.println(line);

    int idx = line.indexOf(" ");
    if (idx < 0 || line.length() < idx + 6)
      throw new IOException("malformed HTTP reply");

    line = line.substring(idx + 1);
    String code = line.substring(0, 3);
    try
      {
        responseCode = Integer.parseInt(code);
      }
    catch (NumberFormatException nfe)
      {
        throw new IOException("malformed HTTP reply");
      }
    responseMessage = line.substring(4);

    String key = null;
    StringBuffer value = new StringBuffer();
    while (true)
      {
        line = readLine();
        System.out.println(line);
        if (line.trim().length() == 0)
          break;

        if (line.startsWith(" ") || line.startsWith("\t"))
          {
            line = line.trim();
            do
              {
                if (line.length() == 1)
                  throw new IOException("malformed header");
                line = line.substring(1);
              }
            while (line.startsWith(" ") || line.startsWith("\t"));
            value.append(' ');
            value.append(line);
          }
        else
          {
            if (key != null)
              {
                headers.add(new Header(key, value.toString()));
                key = null;
                value.setLength(0);
              }
            idx = line.indexOf(":");
            if (idx < 0 || line.length() < idx + 2)
              throw new IOException("malformed header");

            key = line.substring(0, idx);
            line = line.substring(idx + 1);
            while (line.startsWith(" ") || line.startsWith("\t"))
              {
                if (line.length() == 1)
                  throw new IOException("malformed header");
                line = line.substring(1);
              }
            value.append(line);
          }
      }
    if (key != null)
      {
        headers.add(new Header(key, value.toString()));
      }
  }

  private String readLine() throws IOException
  {
    StringBuffer line = new StringBuffer();
    while (true)
      {
        int i = in.read();
        if (i == -1 || i == '\n')
          break;
        if (i != '\r')
          line.append((char) i);
      }
    return line.toString();
  }

  // Inner class.
  // -------------------------------------------------------------------------

  private class Header implements Map.Entry
  {

    // Instance methods.
    // -----------------------------------------------------------------------

    private final String name;
    private String value;

    // Constructor.
    // -----------------------------------------------------------------------

    public Header(String name, String value)
    {
      this.name = name;
      this.value = value;
    }

    // Instance methods.
    // -----------------------------------------------------------------------

    public Object getKey()
    {
      return name;
    }

    public Object getValue()
    {
      return value;
    }

    public boolean equals(Object o)
    {
      return ((Header) o).name.equalsIgnoreCase(name) &&
             ((Header) o).value.equalsIgnoreCase(value);
    }

    public int hashCode()
    {
      return name.hashCode() ^ value.hashCode();
    }

    public Object setValue(Object value)
    {
      String oldval = this.value;
      this.value = (String) value;
      return oldval;
    }
  }
}
