/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.metadata.spi.signature;

import java.lang.reflect.Method;

import org.jboss.reflect.spi.MethodInfo;

/**
 * Method Signature.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class DeclaredMethodSignature extends Signature
{
   /** The declaring class */
   private String declaringClass;
   
   /** The method */
   private Method method;
   
   /**
    * Create a new Signature.
    *
    * @param declaringClass the declaring class
    * @param name the name
    * @param parameters the parameters
    */
   public DeclaredMethodSignature(String declaringClass, String name, String[] parameters)
   {
      super(name, parameters);
      this.declaringClass = declaringClass;
   }

   /**
    * Create a new Signature.
    * 
    * @param declaringClass the declaring class
    * @param name the name
    * @param parameters the parameters
    */
   public DeclaredMethodSignature(String declaringClass, String name, Class<?>... parameters)
   {
      super(name, parameters);
      this.declaringClass = declaringClass;
   }

   /**
    * Create a new Signature.
    * 
    * @param method the method
    */
   public DeclaredMethodSignature(Method method)
   {
      super(method.getName(), (Class<?>[]) null);
      this.method = method;
      this.declaringClass = method.getDeclaringClass().getName();
   }

   /**
    * Create a new Signature.
    * 
    * @param method the method
    */
   public DeclaredMethodSignature(MethodInfo method)
   {
      super(method.getName(), convertParameterTypes(method.getParameterTypes()));
      this.declaringClass = method.getDeclaringClass().getName();
   }

   /**
    * Get the declaringClass.
    * 
    * @return the declaringClass.
    */
   public String getDeclaringClass()
   {
      return declaringClass;
   }

   /**
    * Get the method.
    * 
    * @return the method could be null if not created using a method
    */
   public Method getMethod()
   {
      return method;
   }

   @Override
   protected Class<?>[] getParameterTypes()
   {
      if (method != null)
         return method.getParameterTypes();
      return super.getParameterTypes();
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof DeclaredMethodSignature == false || super.equals(obj) == false)
         return false;

      DeclaredMethodSignature other = (DeclaredMethodSignature) obj;
      return getDeclaringClass().equals(other.getDeclaringClass());
   }

   @Override
   protected void internalToString(StringBuilder builder)
   {
      super.internalToString(builder);
      builder.append(" declaring=").append(declaringClass);
   }
}
