/*
 *   Greebo: An Ant task to manage dependency files.
 *
 *   Copyright (C) 2002 Ozben Evren
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

//
// Heavily modified for use with JBoss
//

package oevren.ant.greebo;

import org.apache.tools.ant.taskdefs.Get;
import org.apache.tools.ant.BuildException;

import java.io.File;

import java.net.URL;
import java.net.MalformedURLException;

import oevren.ant.greebo.FetchDependencyTask.Context;
import org.jboss.tools.buildmagic.task.util.TaskLogger;

/**
 *
 * <p>
 * <pre>
 * $Log: Repository.java,v $
 * Revision 1.1.2.1  2003/03/08 18:07:43  user57
 *  o import of heavily modified Greebo dependency manager task
 *  o modified buildmagic/tasks module to use the new fluff
 *
 * Revision 1.1  2002/11/22 02:41:47  ozbene
 * Initial check-in to CVS
 *
 * </pre>
 *
 * @author   Ozben Evren
 * @version  $Id: Repository.java 21628 2008-08-19 22:06:02Z pgier $
 *
 */
public class Repository 
{
    private URL url;
    private String type;
    private boolean synchronize;

    public static final String FLAT = "flat";
    public static final int FLAT_ID = 0;
    
    public static final String MAVEN = "maven";
    public static final int MAVEN_ID = 1;
    
    public static final String JBOSS = "jboss";
    public static final int JBOSS_ID = 2;

    private Context context;
    private TaskLogger log;
    
    public Repository(Context context) 
    {
        this.context = context;
        this.log = context.getLogger();
        
        type = JBOSS;
        synchronize = false;
    }

    public String toString()
    {
        return getURL().toString();
    }
    
    public URL getURL() {
        return url;
    }

    public void setURL(final URL url) throws MalformedURLException
    {
        if (!url.toString().endsWith("/")) {
            this.url = new URL(url.toString() + "/");
        }
        else {
            this.url = url;
        }
    }

    public String getType() {
        return type;
    }
    
    public int getTypeID() 
    {
        String temp = type.toLowerCase();
        if (temp.equals(FLAT))
            return FLAT_ID;
        
        if (temp.equals(MAVEN))
            return MAVEN_ID;
        
        if (temp.equals(JBOSS))
            return JBOSS_ID;
        
        throw new BuildException("Invalid type: " + type);
    }

    public void setType(final String type) {
        this.type = type;
    }

    public boolean isSynchronize() {
        return synchronize;
    }

    public void setSynchronize(final boolean synchronize) {
        this.synchronize = synchronize;
    }

    public void synchronizeFile(File source, JarDependency dependency)
        throws Exception
    {
        if (url.getProtocol().equals("file")) {
            URL url = dependency.getURL(this);
            log.verbose(("Synchronizing:" + source + "->" + url));
            
            File target = new File(new File(url.getHost()), url.getFile());
            
            try {
                org.jboss.util.file.Files.copy(source, target);
            }
            catch (Exception e) {
                e.printStackTrace();
                throw new BuildException("Could not synchronize download: " + source + " in-repository: " + target);
            }
        }
        
        throw new BuildException("Can only synchronize local file repositories");
    }

    public File getDependency(JarDependency dependency, File dir, Get get) 
        throws Exception 
    {
        URL url = dependency.getURL(this);
        File file = new File(dir, dependency.getFileNameFragment());
       
        // Try to make the dest directory
        file.getParentFile().mkdirs();
        
        // Special handlig for remote repositories when offline
        if (context.isOffline() && !url.getProtocol().equals("file")) {
            // First check if we have a local file
            if (!file.exists()) {
                throw new BuildException("Remote repositories offline; Missing dependency: " + dependency);
            }
        }
        else {
            log.verbose("Getting " + url + "->" + file);
            get.setUseTimestamp(true);
            // get.setVerbose(true);
            get.setSrc(url);
            get.setDest(file);
            get.execute();
        }

        return file;
    }
}
