
/* Command line oriented batch and dialog tool which creates, loads,
   manipulates and burns ISO 9660 filesystem images.

   Copyright 2007-2008 Thomas Schmitt, <scdbackup@gmx.net>

   Provided under GPL version 2.

   This file contains the public option interface of xorriso.
*/

#ifndef Xorriso_includeD
#define Xorriso_includeD yes

/** Opaque handle of the xorriso runtime context */
struct XorrisO; 


/* --------------------- Fundamental Management ------------------- */

/* Create a new xorriso object and tell it the program name to be used
   with messages and for decision of special behavior.
   @param xorriso  returns the newly created XorrisO object 
   @param progname typically argv[0] of main(). Some leafnames of the progname
                   path have special meaning and trigger special behavior:
                    "osirrox"    allows image-to-disk copying: -osirrox "on" 
                    "xorrisofs"  activates permanent mkisofs emulation
                    "genisofs"   alias of "xorrisofs"
                    "mkisofs"    alias of "xorrisofs"
                    "genisoimage"  alias of "xorrisofs"
   @parm flag      unused yet, submit 0
   @return         >0 success , <=0 failure, no object created
*/
int Xorriso_new(struct XorrisO ** xorriso, char *progname, int flag);


/* Make global library initializations.
   This must be done with the first xorriso object that gets created and
   with the first xorriso object that gets created after Xorriso_destroy(,1).
*/ 
int Xorriso_startup_libraries(struct XorrisO *xorriso, int flag);


/* Destroy xorriso object when it is no longer needed.
   @param flag bit0= Make global librariy shutdown.
                     Use only with last xorriso object to be destroyed.
                     
*/
int Xorriso_destroy(struct XorrisO **xorriso, int flag);


/* --------------------- Problem Status and Message API ------------------- */


/** Submit a problem message to the xorriso problem reporting and handling
    system. This will eventually increase problem status rank, which may
    at certain stages in the program be pardoned and reset to 0.
    The pardon is governed by Xorriso_option_abort_on() and by the anger
    of the affected program part. If no pardon has been given, then the problem
    status reaches the caller of option functions.
    Problem status should be inquired by Xorriso_eval_problem_status() and be
    reset before next option execution by Xorriso_set_problem_status().
    The problem status itself does not cause the failure of option functions.
    But in case of failures for other reasons, a remnant overly severe problem
    status can cause overly harsh program reactions.
    @param xorriso    The environment handle
    @param error_code The unique error code of your message.
                      Submit 0 if you do not have reserved error codes within
                      the libburnia project.
    @param msg_text   Not more than 8196 characters of message text.
                      A final newline character gets appended automatically.
    @param os_errno   Eventual errno related to the message. Submit 0 if
                      the message is not related to a operating system error.
    @param severity   One of "ABORT", "FATAL", "SORRY", "WARNING", "HINT",
                      "NOTE", "UPDATE", "DEBUG". Defaults to "FATAL".
    @param flag       Bitfield for control purposes
                        bit0= use pager (as with result)
                        bit1= permission to suppress output
    @return           1 if message was delivered, <=0 if failure
*/
int Xorriso_msgs_submit(struct XorrisO *xorriso,
                        int error_code, char msg_text[], int os_errno,
                        char severity[], int flag);

/** Alternative call interface of Xorriso_msgs_submit with void* instead
    of struct XorrisO*
*/
int Xorriso_msgs_submit_void(void *xorriso,
                        int error_code, char msg_text[], int os_errno,
                        char severity[], int flag);


/** Evaluate an advise whether to abort or whether to go on with option
    processing. This should be called after any option function was processed.
    It updates the problem status by processing the library message queues
    and then it uses this status and the submitted return value ot the
    option function to evaluate the situation.
    @param xorriso    The environment handle
    @param ret        The return value of the previously called option function
    @param flag       bit0= do not issue own event messages
                      bit1= take xorriso->request_to_abort as reason for abort
    @return           Gives the advice:
                        2= pardon was given, go on
                        1= no problem, go on
                        0= function failed but xorriso would not abort, go on
                       <0= do abort
                           -1 = due to xorriso->problem_status
                                or due to ret<0
                           -2 = due to xorriso->request_to_abort
*/
int Xorriso_eval_problem_status(struct XorrisO *xorriso, int ret, int flag);


/** Set the current problem status of the xorriso handle.
    @param xorriso    The environment handle
    @param severity   A severity text. Empty text resets to "No Problem". 
    @param flag       Unused yet. Submit 0.
    @return           <=0 failure (e.g. wrong severity text), 1 success.
*/
int Xorriso_set_problem_status(struct XorrisO *xorriso, char *severity,
                               int flag);


/* The next two functions are part of Xorriso_eval_problem_status().
   You may use them to build an own advisor function or to drain the
   library message queues more frequently.
*/

/** Obtain the current problem status of the xorriso handle.
    @param xorriso    The environment handle
    @param severity   The severity text matching the current problem status
    @param flag       Unused yet. Submit 0.
    @return           The severity rank number. 0= no problem occured.
*/
int Xorriso_get_problem_status(struct XorrisO *xorriso, char severity[80],
                               int flag);


/** Forward any pending messages from the library message queues to the
    xorriso message system which puts out on info channel. This registers
    the severity of the library events like the severity of a message submitted
    via Xorriso_msgs_submit().
    xorriso sets the message queues of the libraries to queuing "ALL".
    So it is essential that they get drained regularly.
    @param xorriso    The environment handle
    @param flag       Unused yet. Submit 0.
    @return           1 on success, <=0 if failure
*/
int Xorriso_process_msg_queues(struct XorrisO *xorriso, int flag);



/* ---------------------------- Options API ------------------------ */
/*    See man 1 xorriso for explanation of the particular options    */
/*
   Before each call to an option function, there should happen:
   Xorriso_set_problem_status() with empty severity text.

   After each call to an option function, there should happen:
   Xorriso_eval_problem_status()
   One should follow its eventual advice to abort.
*/


/* Option -abort_on */
int Xorriso_option_abort_on(struct XorrisO *xorriso, char *severity, int flag);

/* Option -add */
/* @param flag bit0=do not report the added item
               bit1=do not reset pacifier, no final pacifier message
*/
int Xorriso_option_add(struct XorrisO *xorriso, int argc, char **argv,
                       int *idx, int flag);

/* Option -add_plainly "on"|"off" */
int Xorriso_option_add_plainly(struct XorrisO *xorriso, char *mode,
                                   int flag);


/* Option -alter_date, alter_date_r */
/* @param flag bit0=recursive (-alter_date_r)
*/
int Xorriso_option_alter_date(struct XorrisO *xorriso, 
                              char *time_type, char *timestring,
                              int argc, char **argv, int *idx, int flag);

/* Option -ban_stdio_write */
int Xorriso_option_ban_stdio_write(struct XorrisO *xorriso, int flag);

/* Option -blank and -format */
/* @param flag bit0= format rather than blank
   @return <=0 error , 1 success, 2 revoked by -reassure
*/
int Xorriso_option_blank(struct XorrisO *xorriso, char *mode, int flag);

/* Option -boot_image */
int Xorriso_option_boot_image(struct XorrisO *xorriso, char *form,
                              char *treatment, int flag);

/* Option -cd alias -cdi */
int Xorriso_option_cdi(struct XorrisO *xorriso, char *iso_rr_path, int flag);

/* Option -cdx */
int Xorriso_option_cdx(struct XorrisO *xorriso, char *disk_path, int flag);

/* Option -chgrp alias -chgrpi , chgrp_r alias chgrpi */
/* @param flag bit0=recursive (-chgrp_r)
*/
int Xorriso_option_chgrpi(struct XorrisO *xorriso, char *gid,
                          int argc, char **argv, int *idx, int flag);

/* Option -chmod alias -chmodi , -chmod_r alias chmod_ri */
/* @param flag bit0=recursive (-chmod_r)
*/
int Xorriso_option_chmodi(struct XorrisO *xorriso, char *mode,
                          int argc, char **argv, int *idx, int flag);

/* Option -chown alias -chowni , chown_r alias chown_ri */
/* @param flag bit0=recursive (-chown_r)
*/
int Xorriso_option_chowni(struct XorrisO *xorriso, char *uid,
                          int argc, char **argv, int *idx, int flag);

/* Option -close "on"|"off" */
int Xorriso_option_close(struct XorrisO *xorriso, char *mode, int flag);

/* Option -commit */
/* @param flag bit0= leave indrive and outdrive aquired as they were,
                     i.e. do not aquire outdrive as new in-out-drive
               bit1= do not perform eventual -reassure
   @return <=0 error , 1 success, 2 revoked by -reassure
*/
int Xorriso_option_commit(struct XorrisO *xorriso, int flag);

/* Option -commit_eject  */
/* @return <=0 error , 1 success, 2 revoked by -reassure
*/
int Xorriso_option_commit_eject(struct XorrisO *xorriso, char *which, int flag);

/* Option -compare and -compare_r
   @param flag bit0= issue summary message
               bit1= do not reset pacifier, no final pacifier message
               bit2= do not issue pacifier messages at all
               bit3= recursive: -compare_r
*/
int Xorriso_option_compare(struct XorrisO *xorriso, char *disk_path,
                           char *iso_path, int flag);

/* Option -cpr alias -cpri */
int Xorriso_option_cpri( struct XorrisO *xorriso, int argc, char **argv,
                         int *idx, int flag);

/* Options -cpx , -cpax, -cp_rx , -cp_rax */
/* @param flag bit0= recursive (-cp_rx, -cp_rax)
               bit1= full property restore (-cpax, -cp_rax)
*/
int Xorriso_option_cpx(struct XorrisO *xorriso, int argc, char **argv,
                         int *idx, int flag);

/* Option -cut_out */
int Xorriso_option_cut_out(struct XorrisO *xorriso, char *disk_path,
                char *start, char *count, char *iso_rr_path, int flag);

/* Options -dev , -indev, -outdev */
/* @param flag bit0=use as indev , bit1= use as outdev
   @return <=0 error , 1 success, 2 revoked by -reassure
*/
int Xorriso_option_dev(struct XorrisO *xorriso, char *adr, int flag);

/* Option -devices */
/* @return <=0 error , 1 success, 2 revoked by -reassure
*/
int Xorriso_option_devices(struct XorrisO *xorriso, int flag);

/* Option -dialog "on"|"off" */
int Xorriso_option_dialog(struct XorrisO *xorriso, char *mode, int flag);

/* Option -disk_pattern "on"|"ls"|"off" */
int Xorriso_option_disk_pattern(struct XorrisO *xorriso, char *mode, int flag);

/* Option -dummy "on"|"off" */
int Xorriso_option_dummy(struct XorrisO *xorriso, char *mode, int flag);

/* Option -eject */
/* @param flag bit0=do not report toc of eventually remaining drives
*/
int Xorriso_option_eject(struct XorrisO *xorriso, char *which, int flag);

/* Options -end , and -rollback_end */
/* @param flag bit0= discard pending changes 
   @return <=0 error , 1 success, 2 revoked by -reassure
*/
int Xorriso_option_end(struct XorrisO *xorriso, int flag);

/* Option -errfile_log marked|plain  path|-|"" */
int Xorriso_option_errfile_log(struct XorrisO *xorriso,
                               char *mode, char *path, int flag);

/* Option -error_behavior */
int Xorriso_option_error_behavior(struct XorrisO *xorriso, 
                                  char *occasion, char *behavior, int flag);

/* Options -extract , -extract_single */
/* @param flag bit0=do not report the restored item
               bit1=do not reset pacifier, no final pacifier message
               bit5= -extract_single: eventually do not insert directory tree
*/
int Xorriso_option_extract(struct XorrisO *xorriso, char *disk_path,
                       char *iso_path, int flag);

/* Option -follow */
int Xorriso_option_follow(struct XorrisO *xorriso, char *mode, int flag);

/* Option -find alias -findi, and -findx */
/* @param flag bit0= -findx rather than -findi
               bit1= do not reset pacifier, no final pacifier message
                     do not reset find_compare_result
*/ 
int Xorriso_option_find(struct XorrisO *xorriso, int argc, char **argv,
                       int *idx, int flag);

/* Option -fs */
int Xorriso_option_fs(struct XorrisO *xorriso, char *size, int flag);

/* Option -gid */
int Xorriso_option_gid(struct XorrisO *xorriso, char *gid, int flag);

/* Option -grow_blindly */
int Xorriso_option_grow_blindly(struct XorrisO *xorriso, char *msc2, int flag);

/* Option -help and part of -prog_help */
int Xorriso_option_help(struct XorrisO *xorriso, int flag);

/* Option -history */
int Xorriso_option_history(struct XorrisO *xorriso, char *line, int flag);

/* Option -iso_rr_pattern "on"|"ls"|"off" */
int Xorriso_option_iso_rr_pattern(struct XorrisO *xorriso, char *mode,
                                  int flag);

/* Option -joliet "on"|"off" */
int Xorriso_option_joliet(struct XorrisO *xorriso, char *mode, int flag);

/* Option -list_formats */
int Xorriso_option_list_formats(struct XorrisO *xorriso, int flag);

/* Option -load session|track|sbsector value */
/* @param flag bit0= with adr_mode sbsector: adr_value is possibly 16 too high
   @return <=0 error , 1 success, 2 revoked by -reassure
*/
int Xorriso_option_load(struct XorrisO *xorriso, char *adr_mode,
                          char *adr_value, int flag);

/* Option -logfile */
int Xorriso_option_logfile(struct XorrisO *xorriso, char *channel, 
                                                      char *fileadr, int flag);

/* Options -ls  alias -lsi   and -lsl  alias -lsli
       and -lsd alias -lsdi  and -lsdl alias -lsdli
       and -du  alias -dui   and -dus  alias -dusi
   @param flag bit0= long format (-lsl , -du)
               bit1= do not expand patterns but use literally
               bit2= du rather than ls
               bit3= list directories as themselves (ls -d)
*/
int Xorriso_option_lsi(struct XorrisO *xorriso, int argc, char **argv,
                      int *idx, int flag);

/* Options -lsx, -lslx, -lsdx , -lsdlx , -dux , -dusx
   @param flag bit0= long format (-lslx , -dux)
               bit1= do not expand patterns but use literally
               bit2= du rather than ls
               bit3= list directories as themselves (ls -d)
*/
int Xorriso_option_lsx(struct XorrisO *xorriso, int argc, char **argv,
                      int *idx, int flag);

/* Option -map */
/* @param flag bit0=do not report the added item
               bit1=do not reset pacifier, no final pacifier message
*/
int Xorriso_option_map(struct XorrisO *xorriso, char *disk_path,
                       char *iso_path, int flag);

/* Options -map_l , -compare_l , -update_l , -extract_l */
/* @param flag bit8-11= mode 0= -map_l
                             1= -compare_l
                             2= -update_l
                             3= -extract_l
*/
int Xorriso_option_map_l(struct XorrisO *xorriso, int argc, char **argv,
                         int *idx, int flag);

/* Option -mark */
int Xorriso_option_mark(struct XorrisO *xorriso, char *mark, int flag);

/* Option -mkdir alias -mkdiri */
int Xorriso_option_mkdiri(struct XorrisO *xorriso, int argc, char **argv,
                          int *idx, int flag);

/* Option -mv alias -mvi */
int Xorriso_option_mvi(struct XorrisO *xorriso, int argc, char **argv,
                      int *idx, int flag);

/* Option -no_rc */
int Xorriso_option_no_rc(struct XorrisO *xorriso, int flag);

/* Option -not_leaf */
int Xorriso_option_not_leaf(struct XorrisO *xorriso, char *pattern, int flag);

/* Option -not_list */
int Xorriso_option_not_list(struct XorrisO *xorriso, char *adr, int flag);

/* Option -not_paths */
int Xorriso_option_not_paths(struct XorrisO *xorriso, int argc, char **argv,
                             int *idx, int flag);

/* Option -options_from_file*/
/* @return <=0 error , 1 = success , 3 = request to end program run */
int Xorriso_option_options_from_file(struct XorrisO *xorriso, char *adr,
                                     int flag);

/* Option -osirrox "on"|"off" */
int Xorriso_option_osirrox(struct XorrisO *xorriso, char *mode, int flag);

/* Option -overwrite "on"|"nondir"|"off" */
int Xorriso_option_overwrite(struct XorrisO *xorriso, char *mode, int flag);

/* Option -pacifier */
int Xorriso_option_pacifier(struct XorrisO *xorriso, char *style, int flag);

/* Option -padding */
int Xorriso_option_padding(struct XorrisO *xorriso, char *size, int flag);

/* Option -page */
int Xorriso_option_page(struct XorrisO *xorriso, int len, int width, int flag);

/* Option -paste_in */
int Xorriso_option_paste_in(struct XorrisO *xorriso, char *iso_rr_path,
                          char *disk_path, char *start, char *count, int flag);

/* Option -path-list */
int Xorriso_option_path_list(struct XorrisO *xorriso, char *adr, int flag);

/* Option -pathspecs */
int Xorriso_option_pathspecs(struct XorrisO *xorriso, char *mode, int flag);

/* Option -pkt_output */
int Xorriso_option_pkt_output(struct XorrisO *xorriso, char *mode, int flag);

/* Option -print */
int Xorriso_option_print(struct XorrisO *xorriso, char *text, int flag);

/* Option -print_size
   @param flag bit0= report in mkisofs compatible form on real stdout
*/
int Xorriso_option_print_size(struct XorrisO *xorriso, int flag);

/* Option -prog */
int Xorriso_option_prog(struct XorrisO *xorriso, char *name, int flag);

/* Option -prog_help */
int Xorriso_option_prog_help(struct XorrisO *xorriso, char *name, int flag);

/* Option -publisher */
int Xorriso_option_publisher(struct XorrisO *xorriso, char *name, int flag);

/* Option -pwd alias -pwdi */
int Xorriso_option_pwdi(struct XorrisO *xorriso, int flag);

/* Option -pwdx */
int Xorriso_option_pwdx(struct XorrisO *xorriso, int flag);

/* Option -reassure "on"|"tree"|"off" */
int Xorriso_option_reassure(struct XorrisO *xorriso, char *mode, int flag);

/* Option -report_about */
int Xorriso_option_report_about(struct XorrisO *xorriso, char *severity, 
                                int flag);

/* Option -return_with */
int Xorriso_option_return_with(struct XorrisO *xorriso, char *severity,
                               int exit_value, int flag);

/* Option -revoke_exclusions */
int Xorriso_option_revoke_exclusions(struct XorrisO *xorriso, int flag);

/* Options -rm alias -rmi , -rm_r alias -rm_ri , -rmdir alias -rmdiri */
/* @param flag bit0=recursive , bit2= remove empty directory: rmdir */
int Xorriso_option_rmi(struct XorrisO *xorriso, int argc, char **argv,
                       int *idx, int flag);

/* Option -rollback */
/* @param flag bit0= do not -reassure
   @return <=0 error , 1 success, 2 revoked by -reassure
*/
int Xorriso_option_rollback(struct XorrisO *xorriso, int flag);

/* Option -rom_toc_scan */
int Xorriso_option_rom_toc_scan(struct XorrisO *xorriso, char *mode,
                                int flag);

/* Option -session_log */
int Xorriso_option_session_log(struct XorrisO *xorriso, char *path, int flag);

/* Option -speed */
int Xorriso_option_speed(struct XorrisO *xorriso, char *speed, int flag);

/* Option -split_size */
int Xorriso_option_split_size(struct XorrisO *xorriso, char *s, int flag);

/* Option -status */
int Xorriso_option_status(struct XorrisO *xorriso, char *mode, int flag);

/* Option -status_history_max */
int Xorriso_option_status_history_max(struct XorrisO *xorriso, int num1, 
                                      int flag);

/* Option -stream_recording */
int Xorriso_option_stream_recording(struct XorrisO *xorriso, char *mode,
                                    int flag);

/* Option -tell_media_space */
int Xorriso_option_tell_media_space(struct XorrisO *xorriso, int flag);

/* Option -temp_mem_limit */
int Xorriso_option_temp_mem_limit(struct XorrisO *xorriso, char *size, 
                                  int flag);

/* Option -toc */
int Xorriso_option_toc(struct XorrisO *xorriso, int flag);

/* Option -uid */
int Xorriso_option_uid(struct XorrisO *xorriso, char *uid, int flag);

/* Options -update and -update_r
   @param flag bit0= issue summary message
               bit1= do not reset pacifier, no final pacifier message
               bit2= do not issue pacifier messages at all
               bit3= recursive: -update_r
*/ 
int Xorriso_option_update(struct XorrisO *xorriso, char *disk_path,
                          char *iso_path, int flag);

/* Option -use_readline */
int Xorriso_option_use_readline(struct XorrisO *xorriso, char *mode, int flag);

/* Option -version */
int Xorriso_option_version(struct XorrisO *xorriso, int flag);

/* Option -volid */
int Xorriso_option_volid(struct XorrisO *xorriso, char *volid, int flag);


#endif /* Xorriso_includeD */


