/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  adjustment.cc - GtkAdjustment C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "adjustment.h"
#include "private/adjustment_p.h"

using namespace Inti;

/*  Gtk::Adjustment
 */

Gtk::Adjustment::Adjustment(GtkAdjustment *adjustment, bool reference) 
: Object((GtkObject*)adjustment, reference)
{
}

Gtk::Adjustment::Adjustment()
: Object((GtkObject*)AdjustmentClass::create())
{
}

Gtk::Adjustment::Adjustment(double lower, double upper, double step_increment, double page_increment, double page_size)
: Object((GtkObject*)AdjustmentClass::create())
{
	set_range(lower, upper, step_increment, page_increment, page_size);
}

Gtk::Adjustment::Adjustment(double value, double lower, double upper, double step_increment, double page_increment, double page_size)
: Object((GtkObject*)AdjustmentClass::create())
{
	set_value(value);	
//	set_range(lower, upper, step_increment, page_increment, page_size);
  gtk_adjustment()->value = value;
  gtk_adjustment()->lower = lower;
  gtk_adjustment()->upper = upper;
  gtk_adjustment()->step_increment = step_increment;
  gtk_adjustment()->page_increment = page_increment;
  gtk_adjustment()->page_size = page_size;

}

Gtk::Adjustment::~Adjustment()
{
}

GtkAdjustmentClass* 
Gtk::Adjustment::gtk_adjustment_class() const 
{
	return get_class<GtkAdjustmentClass>(); 
}
	
Gtk::Adjustment::operator GtkAdjustment* () const 
{ 
	return this ? gtk_adjustment() : 0; 
}
	
double
Gtk::Adjustment::lower() const
{
	return gtk_adjustment()->lower;
}

double
Gtk::Adjustment::upper() const
{
	return gtk_adjustment()->upper;
}

double
Gtk::Adjustment::step_increment() const
{
	return gtk_adjustment()->step_increment;
}

double
Gtk::Adjustment::page_increment() const
{
	return gtk_adjustment()->page_increment;
}

double
Gtk::Adjustment::page_size() const
{
	return gtk_adjustment()->page_size;
}

double
Gtk::Adjustment::get_value() const
{
	return gtk_adjustment_get_value(gtk_adjustment());
}

void
Gtk::Adjustment::changed()
{
	gtk_adjustment_changed(gtk_adjustment());
}

void
Gtk::Adjustment::value_changed()
{
	gtk_adjustment_value_changed(gtk_adjustment());
}

void
Gtk::Adjustment::clamp_page(double lower, double upper)
{
	gtk_adjustment_clamp_page(gtk_adjustment(), lower, upper);
}

void
Gtk::Adjustment::set_value(double value)
{
	gtk_adjustment_set_value(gtk_adjustment(), value);
}

void
Gtk::Adjustment::set_range(double lower, double upper, double step_increment, double page_increment, double page_size)
{
	GtkAdjustment *adjustment = gtk_adjustment();
	adjustment->lower = MIN(lower, upper);
	adjustment->upper = MAX(lower, upper);
	double range = adjustment->upper - adjustment->lower;
	adjustment->step_increment = MIN(range, step_increment);
	adjustment->page_increment = MIN(range, page_increment);
	adjustment->page_size = MIN(range, page_size);
	set_value(CLAMP(get_value(), adjustment->lower, adjustment->upper));
	changed();
}

/*{
  GtkAdjustment * adj = gtk_adjustment ();

  adj->lower = MIN (lower, upper);
  adj->upper = MAX (lower, upper);
  // Clamp various increments and sizes to be no larger
  // than the range it's possible to move.
  float range = adj->upper - adj->lower;
  adj->step_increment = MIN (range, step_increment);
  adj->page_increment = MIN (range, page_increment);
  adj->page_size      = MIN (range, page_size);

  // Make sure the value is clamped
  set_value (CLAMP (value (), this->lower (), this->upper ()));

  gtk_adjustment_changed (adj);
}*/



/*  Gtk::AdjustmentClass
 */

void
Gtk::AdjustmentClass::init(GtkAdjustmentClass *g_class)
{
	ObjectClass::init((GtkObjectClass*)g_class);
	g_class->changed = &changed_proxy;
	g_class->value_changed = &value_changed_proxy;
}

GType
Gtk::AdjustmentClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_ADJUSTMENT, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::AdjustmentClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::AdjustmentClass::changed_proxy(GtkAdjustment *adjustment)
{
	Adjustment *tmp_adjustment = G::Object::pointer<Adjustment>(adjustment);
	if (tmp_adjustment)
		tmp_adjustment->on_changed();
	else
	{
		GtkAdjustmentClass *g_class = G::TypeInstance::class_peek_parent<GtkAdjustmentClass>(GTK_ADJUSTMENT_GET_CLASS(adjustment));
		if (g_class->changed)
			g_class->changed(adjustment);
	}
}

void
Gtk::AdjustmentClass::value_changed_proxy(GtkAdjustment *adjustment)
{
	Adjustment *tmp_adjustment = G::Object::pointer<Adjustment>(adjustment);
	if (tmp_adjustment)
		tmp_adjustment->on_value_changed();
	else
	{
		GtkAdjustmentClass *g_class = G::TypeInstance::class_peek_parent<GtkAdjustmentClass>(GTK_ADJUSTMENT_GET_CLASS(adjustment));
		if (g_class->value_changed)
			g_class->value_changed(adjustment);
	}
}

/*  Signal handlers
 */

void
Gtk::Adjustment::on_changed()
{
	GtkAdjustmentClass *g_class = class_peek_parent<GtkAdjustmentClass>(gtk_adjustment_class());
	if (g_class->changed)
		g_class->changed(gtk_adjustment());
}

void
Gtk::Adjustment::on_value_changed()
{
	GtkAdjustmentClass *g_class = class_peek_parent<GtkAdjustmentClass>(gtk_adjustment_class());
	if (g_class->value_changed)
		g_class->value_changed(gtk_adjustment());
}

/*  Signals
 */

const Gtk::Adjustment::ChangedSignalType Gtk::Adjustment::changed_signal("changed");

const Gtk::Adjustment::ValueChangedSignalType Gtk::Adjustment::value_changed_signal("value_changed");
